#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_backup_and_restore.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import shutil
import socket
import sys
import tarfile
import time
import re
import base64
import subprocess
import glob
from os import listdir
from os.path import isfile, join

# Third-party libraries
import requests

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.backup_and_restore", "trackme_rest_api_backup_and_restore.log"
)  # Redirect global logging to use the same handler
import logging

logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)
# Redirect global logging to use the same handler
import logging

logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import (
    trackme_getloglevel,
    run_splunk_search,
    trackme_reqinfo,
    trackme_create_report,
    trackme_create_alert,
    trackme_delete_report,
    trackme_create_kvcollection,
    trackme_delete_kvcollection,
    trackme_create_kvtransform,
    trackme_delete_kvtransform,
    trackme_create_macro,
    trackme_delete_macro,
)

# import TrackMe get data libs
from trackme_libs_get_data import (
    get_full_kv_collection,
)

# import Splunk libs
import splunklib.client as client

# import the collections dict
from collections_data import vtenant_account_default

#
# Splunk Cloud certification notes: these functions create and manage archive files for the application in the following directory $SPLUNK/etc/apps/trackme/backup
# There are no options to perform any kind of file manipulations out of this application directory
#


def is_zstd_available():
    """Check if zstd compression is available on the system."""
    try:
        # First check if zstd command exists
        if shutil.which("zstd") is None:
            logger.info("zstd command not found in PATH")
            return False

        # Try to run zstd with version check
        result = subprocess.run(
            ["zstd", "--version"],
            capture_output=True,
            check=True,
            text=True,
            timeout=10,  # Add timeout to prevent hanging
        )
        logger.info(f"zstd is available: {result.stdout.strip()}")
        return True
    except (
        subprocess.CalledProcessError,
        FileNotFoundError,
        subprocess.TimeoutExpired,
    ) as e:
        logger.info(f"zstd is not available: {str(e)}")
        return False


def create_compressed_archive(source_dir, archive_name):
    """Create a compressed archive using zstd if available, otherwise fall back to gzip."""
    # Get the backup directory path from the source_dir
    backup_dir = os.path.dirname(source_dir)

    if is_zstd_available():
        # Use zstd compression
        logger.info(
            f'Creating zstd compressed archive for="{source_dir}" with archive_name="{archive_name}"'
        )
        zst_name = os.path.join(backup_dir, f"{archive_name}.tar.zst")
        try:
            # Create tar file first
            tar_name = os.path.join(backup_dir, f"{archive_name}.tar")
            with tarfile.open(tar_name, mode="w") as archive:
                archive.add(source_dir, arcname="")

            # Compress with zstd
            subprocess.run(["zstd", "-f", tar_name], check=True)

            # Remove the uncompressed tar file
            os.remove(tar_name)

            return zst_name
        except Exception as e:
            logger.warning(
                f"Failed to create zstd archive, falling back to gzip: {str(e)}"
            )
            # Fall back to gzip
            pass

    else:
        logger.info(f"Compression using zstd is not available, falling back to gzip")

    # Fall back to gzip compression
    tgz_name = os.path.join(backup_dir, f"{archive_name}.tgz")
    with tarfile.open(tgz_name, mode="w:gz") as archive:
        archive.add(source_dir, arcname="")

    return tgz_name


def extract_archive(archive_path, extract_dir):
    """Extract an archive file (.tgz or .tar.zst) to the specified directory."""
    logger.info(
        f'Starting archive extraction: archive_path="{archive_path}", extract_dir="{extract_dir}"'
    )

    # Validate input parameters
    if not os.path.exists(archive_path):
        logger.error(f"Archive file does not exist: {archive_path}")
        return False

    if not os.path.isfile(archive_path):
        logger.error(f"Archive path is not a file: {archive_path}")
        return False

    # Check file size
    file_size = os.path.getsize(archive_path)
    logger.info(f"Archive file size: {file_size} bytes")
    if file_size == 0:
        logger.error(f"Archive file is empty: {archive_path}")
        return False

    # Ensure extract directory exists
    os.makedirs(extract_dir, exist_ok=True)

    # Security: Validate extract directory is absolute and safe
    extract_dir = os.path.abspath(extract_dir)
    logger.info(f"Using absolute extract directory: {extract_dir}")

    def safe_extract_member(tar, member, extract_dir):
        """Safely extract a single member from a tar archive, preventing path traversal attacks."""
        # Get the absolute path where this member would be extracted
        member_path = os.path.join(extract_dir, member.name)
        member_path = os.path.abspath(member_path)

        # Security check: ensure the member path is within the extract directory
        # Use os.path.commonpath to properly validate containment
        try:
            common_path = os.path.commonpath([extract_dir, member_path])
            if common_path != extract_dir:
                logger.error(
                    f'SECURITY VIOLATION: member "{member.name}" would extract outside allowed directory. Blocked path traversal attack.'
                )
                return False
        except ValueError:
            # commonpath raises ValueError if paths are on different drives (Windows)
            # or if one path is a prefix of the other but not a proper subdirectory
            logger.error(
                f'SECURITY VIOLATION: member "{member.name}" would extract outside allowed directory. Blocked path traversal attack.'
            )
            return False

        # Extract the member
        try:
            tar.extract(member, extract_dir)
            logger.debug(f"Safely extracted member: {member.name}")
            return True
        except Exception as e:
            logger.error(f'Failed to extract member "{member.name}": {str(e)}')
            return False

    if archive_path.endswith(".tar.zst"):
        # Extract zstd compressed archive
        logger.info(f"Detected zstd compressed archive: {archive_path}")
        if is_zstd_available():
            logger.info("zstd is available, proceeding with zstd extraction")
            try:
                # Create a temporary directory for the intermediate tar file
                import tempfile

                temp_dir = tempfile.mkdtemp(prefix="trackme_extract_")
                logger.info(f"Created temporary directory: {temp_dir}")

                try:
                    # Decompress with zstd first to a temporary location
                    tar_path = os.path.join(
                        temp_dir, os.path.basename(archive_path)[:-4]
                    )  # Remove .zst extension
                    logger.info(
                        f"Decompressing zstd archive to temporary tar file: {tar_path}"
                    )

                    # Use absolute paths and capture output for debugging
                    result = subprocess.run(
                        ["zstd", "-d", os.path.abspath(archive_path), "-o", tar_path],
                        capture_output=True,
                        text=True,
                        check=True,
                        cwd=temp_dir,
                    )
                    logger.info(f"zstd decompression output: {result.stdout}")
                    if result.stderr:
                        logger.info(f"zstd decompression stderr: {result.stderr}")

                    # Verify the decompressed tar file exists and has content
                    if not os.path.exists(tar_path):
                        logger.error(f"Failed to create temporary tar file: {tar_path}")
                        return False

                    tar_size = os.path.getsize(tar_path)
                    logger.info(f"Temporary tar file size: {tar_size} bytes")
                    if tar_size == 0:
                        logger.error(f"Temporary tar file is empty: {tar_path}")
                        return False

                    # Extract the tar file securely
                    logger.info(f"Extracting tar file to: {extract_dir}")
                    with tarfile.open(tar_path, mode="r") as tar:
                        # Security: Extract members one by one with path validation
                        for member in tar.getmembers():
                            if not safe_extract_member(tar, member, extract_dir):
                                logger.error(
                                    f"Failed to safely extract member: {member.name}"
                                )
                                return False

                    logger.info("zstd archive extraction completed successfully")
                    return True
                finally:
                    # Clean up the temporary directory and files
                    logger.info(f"Cleaning up temporary directory: {temp_dir}")
                    shutil.rmtree(temp_dir, ignore_errors=True)
            except Exception as e:
                logger.error(f"Failed to extract zstd archive: {str(e)}")
                return False
        else:
            logger.error("zstd is not available, cannot extract .tar.zst archive")
            return False
    elif archive_path.endswith(".tgz"):
        # Extract gzip compressed archive
        logger.info(f"Detected gzip compressed archive: {archive_path}")
        try:
            with tarfile.open(archive_path, mode="r:gz") as tar:
                # Security: Extract members one by one with path validation
                for member in tar.getmembers():
                    if not safe_extract_member(tar, member, extract_dir):
                        logger.error(f"Failed to safely extract member: {member.name}")
                        return False
            logger.info("gzip archive extraction completed successfully")
            return True
        except Exception as e:
            logger.error(f"Failed to extract gzip archive: {str(e)}")
            return False
    else:
        logger.error(f"Unsupported archive format: {archive_path}")
        return False


def test_splunkd_connectivity(hostname, port, session_key, timeout=5):
    """
    Test connectivity to a Splunk instance by making a simple REST call.

    :param hostname: Target hostname or FQDN
    :param port: Splunkd REST port
    :param session_key: Session key for authentication
    :param timeout: Connection timeout in seconds
    :return: True if connectivity successful, False otherwise
    """
    try:
        target_url = f"https://{hostname}:{port}/services/server/info"
        headers = {
            "Authorization": f"Splunk {session_key}",
            "Content-Type": "application/json",
        }

        response = requests.get(
            target_url, headers=headers, verify=False, timeout=timeout
        )

        # Check if we get a valid response (200 or 401/403 are OK - means server is reachable)
        if response.status_code in [200, 401, 403]:
            logger.debug(
                f"Connectivity test successful for {hostname}:{port} (status: {response.status_code})"
            )
            return True
        else:
            logger.debug(
                f"Connectivity test failed for {hostname}:{port} (status: {response.status_code})"
            )
            return False

    except Exception as e:
        logger.debug(f"Connectivity test failed for {hostname}:{port} - {str(e)}")
        return False


def cleanup_backup_directories(backup_root, exclude_dir=None):
    """Clean up any leftover temporary backup directories from failed attempts."""
    try:
        for item in os.listdir(backup_root):
            item_path = os.path.join(backup_root, item)
            if os.path.isdir(item_path) and item.startswith("trackme-backup-"):
                # Skip the current backup directory if specified
                if exclude_dir and item_path == exclude_dir:
                    continue
                logger.info(f"Cleaning up leftover backup directory: {item_path}")
                shutil.rmtree(item_path, ignore_errors=True)
                logger.info(
                    f"Successfully removed leftover backup directory: {item_path}"
                )
    except Exception as e:
        logger.warning(f"Failed to clean up leftover backup directories: {str(e)}")
        # Don't fail the backup process for cleanup errors


class TrackMeHandlerBackupAndRestore_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerBackupAndRestore_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_backup_and_restore(self, request_info, **kwargs):
        response = {
            "resource_group_name": "backup_and_restore",
            "resource_group_desc": "These endpoints provide backup and restore facilities for the Kvstore collections created and managed in TrackMe, this includes the full scope of active and enabled tenants",
        }

        return {"payload": response, "status": 200}

    # List backup archive files known from the KV, add any local file that the KV wouldn't do about
    def get_backup(self, request_info, **kwargs):
        """
        A simple function to safetly retrieve all records from a KVstore collection with pagination
        """

        def get_full_collection_records(collection):
            collection_records = []
            collection_records_dict = {}
            collection_records_keys = set()

            end = False
            skip_tracker = 0
            while not end:
                process_collection_records = collection.data.query(skip=skip_tracker)
                if process_collection_records:
                    for item in process_collection_records:
                        collection_records.append(item)
                        collection_records_dict[item.get("_key")] = (
                            item  # Add the entire item to the dictionary
                        )
                        collection_records_keys.add(item.get("_key"))
                    skip_tracker += 1000
                else:
                    end = True

            return collection_records, collection_records_dict, collection_records_keys

        describe = False

        # init
        mode = "full"

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            # mode summary
            try:
                mode = resp_dict["mode"]
                # valid mode: full | summary (defaults to full)
                if mode not in ("full", "summary"):
                    return {
                        "payload": f'Invalid mode="{mode}", valid modes are: full | summary',
                        "status": 400,
                    }
            except Exception as e:
                mode = "full"

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        if describe:
            response = {
                "describe": "This endpoint lists all the backup files available on the search head, files are stored in the TrackMe backup directory, it requires a GET call with the following arguments:",
                "resource_desc": "Get the list of backups known to TrackMe",
                "resource_spl_example": '| trackme mode=get url="/services/trackme/v2/backup_and_restore/backup"',
                "options": [
                    {
                        "mode": "(string) OPTIONAL: The output mode, valid values are full | summary, defaults to full",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        else:
            # Get splunkd port
            splunkd_port = request_info.server_rest_port

            # set loglevel
            loglevel = trackme_getloglevel(
                request_info.system_authtoken, request_info.server_rest_port
            )
            logger.setLevel(loglevel)

            # Set backup root dir
            backuproot = os.path.join(splunkhome, "etc", "apps", "trackme", "backup")

            # get local server name
            server_name = socket.gethostname()

            # check backup dir existence
            if not os.path.isdir(backuproot):
                logger.info(
                    f'TrackMe backup process, there are no backup archives available in directory="{backuproot}", instance="{server_name}"'
                )
                return {
                    "payload": {
                        "response": f'TrackMe backup process, there are no backup archives available in directory="{backuproot}", instance="{server_name}"'
                    },
                    "status": 200,
                }

            else:
                # store files in list (support both .tgz and .tar.zst)
                backup_files = [
                    join(backuproot, f)
                    for f in listdir(backuproot)
                    if isfile(join(backuproot, f))
                    and (f.endswith(".tgz") or f.endswith(".tar.zst"))
                ]

                if not backup_files:
                    logger.info(
                        f'TrackMe backup process, there are no backup archives available in directory="{backuproot}", instance="{server_name}"'
                    )
                    return {
                        "payload": {
                            "response": f'TrackMe backup process, there are no backup archives available in directory="{backuproot}", instance="{server_name}"'
                        },
                        "status": 200,
                    }

                else:
                    # Enter the list, verify for that for each archive file we have a corresponding record in the audit collection
                    # if there is no record, then we create a replacement record
                    for backupfile in backup_files:
                        # get the file mtime
                        try:
                            backup_file_mtime = round(os.path.getmtime(backupfile))
                        except Exception as e:
                            backup_file_mtime = None
                            logger.error(
                                f'failure to retrieve the tmime of the tar file archive="{backupfile}" with exception:"{str(e)}"'
                            )

                        # get the file size
                        try:
                            tar_filesize = os.path.getsize(backupfile)
                        except Exception as e:
                            tar_filesize = None
                            logger.error(
                                f'archive="{backupfile}", failure to retrieve the size of the tar file archive="{str(e)}"'
                            )

                        # Store a record in a backup audit collection

                        # Create a message
                        status_message = "discovered"

                        # record / key
                        record = None
                        key = None

                        # Define the KV query
                        query_string = {
                            "$and": [
                                {
                                    "backup_archive": backupfile,
                                    "server_name": server_name,
                                }
                            ]
                        }

                        # backup audit collection
                        collection_name_backup_archives_info = (
                            "kv_trackme_backup_archives_info"
                        )
                        service_backup_archives_info = client.connect(
                            owner="nobody",
                            app="trackme",
                            port=splunkd_port,
                            token=request_info.session_key,
                            timeout=600,
                        )
                        collection_backup_archives_info = (
                            service_backup_archives_info.kvstore[
                                collection_name_backup_archives_info
                            ]
                        )

                        try:
                            kvrecord = collection_backup_archives_info.data.query(
                                query=json.dumps(query_string)
                            )[0]
                            key = kvrecord.get("_key")

                        except Exception as e:
                            key = None

                        # If a record cannot be found, this backup file is not know to TrackMe currently, add a new record
                        if key is None:
                            # attempt to extract and get the backup details

                            # Attempt extraction
                            backup_file_is_valid = False

                            # From TrackMe 2.1.5, we generate a metadata file named as <tarfile>.full.meta and <tarfile>.light.meta
                            # Check if the metadata file exists
                            backup_file_has_metadata = False

                            # check for both metadata files
                            if os.path.isfile(
                                f"{backupfile}.full.meta"
                            ) and os.path.isfile(f"{backupfile}.light.meta"):
                                backup_file_has_metadata = True

                                try:
                                    with open(
                                        f"{backupfile}.full.meta", "r"
                                    ) as read_content:
                                        full_metadata = json.load(read_content)
                                except Exception as e:
                                    full_metadata = None
                                    backup_file_has_metadata = False
                                    logger.error(
                                        f'failed to load the full metadata file="{backupfile}.full.meta" with exception="{str(e)}"'
                                    )

                                try:
                                    with open(
                                        f"{backupfile}.light.meta", "r"
                                    ) as read_content:
                                        light_metadata = json.load(read_content)
                                except Exception as e:
                                    light_metadata = None
                                    backup_file_has_metadata = False
                                    logger.error(
                                        f'failed to load the light metadata file="{backupfile}.light.meta" with exception="{str(e)}"'
                                    )

                            # Set backup dir
                            backupdir = os.path.join(
                                backuproot, os.path.splitext(backupfile)[0]
                            )

                            try:
                                if extract_archive(backupfile, backupdir):
                                    backup_file_is_valid = True
                                else:
                                    backup_file_is_valid = False
                                    response = {
                                        "response": f'The archive name {backupfile} could not be extracted, restore cannot be processed, this backup file will not be considered, exception="file could not be opened successfully"',
                                    }
                                    logger.error(json.dumps(response, indent=2))
                            except Exception as e:
                                backup_file_is_valid = False
                                response = {
                                    "response": f'The archive name {backupfile} could not be extracted, restore cannot be processed, this backup file will not be considered, exception="{str(e)}"',
                                }
                                logger.error(json.dumps(response, indent=2))

                            # get the local server fqdn
                            hostname = socket.gethostname()
                            fqdn = socket.getfqdn()
                            server_fqdn = (
                                fqdn
                                if fqdn != hostname and not fqdn.endswith('.local')
                                else hostname
                            )

                            #
                            # backup file generated with TrackMe 2.1.5 and later
                            #

                            if backup_file_is_valid and backup_file_has_metadata:

                                logger.info(
                                    f'Discovered backup file="{backupfile}" with metadata compatibility mode.'
                                )

                                if full_metadata and light_metadata:

                                    try:
                                        # Insert the record
                                        collection_backup_archives_info.data.insert(
                                            json.dumps(
                                                {
                                                    "mtime": full_metadata.get("mtime"),
                                                    "htime": full_metadata.get("htime"),
                                                    "server_name": server_fqdn,
                                                    "status": json.dumps(
                                                        light_metadata, indent=4
                                                    ),
                                                    "change_type": "backup archive was missing from the info collection and added by automatic discovery (metadata compatibility mode)",
                                                    "backup_archive": full_metadata.get(
                                                        "backup_archive"
                                                    ),
                                                    "size": full_metadata.get("size"),
                                                    "archive_details": json.dumps(
                                                        full_metadata.get(
                                                            "archive_details"
                                                        ),
                                                        indent=4,
                                                    ),
                                                }
                                            )
                                        )

                                    except Exception as e:
                                        logger.error(
                                            f'failed to insert a new KVstore record with exception="{str(e)}"'
                                        )
                                        return {
                                            "payload": {
                                                "response": f'failed to insert a new KVstore record with exception="{str(e)}"'
                                            },
                                            "status": 500,
                                        }

                            #
                            # backup file generated prior to TrackMe 2.1.5
                            #

                            elif backup_file_is_valid:

                                logger.info(
                                    f'Discovered backup file="{backupfile}" with legacy compatibility mode.'
                                )

                                # store the list of json files in a list
                                collections_json_files = [
                                    f
                                    for f in listdir(backupdir)
                                    if isfile(join(backupdir, f))
                                ]

                                # store the list of available collections in the archive
                                collections_available = []

                                # create a dictionnary
                                collections_restore_dict = {}
                                for json_file in collections_json_files:
                                    # strip the extension
                                    collection_name = os.path.splitext(json_file)[0]

                                    # append to the list of available collections for restore
                                    collections_available.append(collection_name)

                                    # get the file size
                                    json_file_size = os.path.getsize(
                                        os.path.join(backupdir, json_file)
                                    )

                                    # get the file mtime
                                    json_file_mtime = round(
                                        os.path.getmtime(
                                            os.path.join(backupdir, json_file)
                                        )
                                    )

                                    # try getting the number of records
                                    try:
                                        with open(
                                            os.path.join(backupdir, json_file), "r"
                                        ) as read_content:
                                            json_file_records = len(
                                                json.load(read_content)
                                            )
                                    except Exception as e:
                                        json_file_records = None

                                    # add to the dict
                                    collections_restore_dict[collection_name] = {
                                        "file": json_file,
                                        "size": json_file_size,
                                        "mtime": json_file_mtime,
                                        "records": json_file_records,
                                    }

                                # remove backup dir
                                try:
                                    shutil.rmtree(backupdir)
                                except OSError as e:
                                    logger.error(
                                        f'failed to purge the extraction temporary directory="{backupdir}" with exception="{str(e)}"'
                                    )

                                try:
                                    # Insert the record
                                    collection_backup_archives_info.data.insert(
                                        json.dumps(
                                            {
                                                "mtime": str(backup_file_mtime),
                                                "htime": str(
                                                    time.strftime(
                                                        "%c",
                                                        time.localtime(
                                                            backup_file_mtime
                                                        ),
                                                    )
                                                ),
                                                "server_name": server_fqdn,
                                                "status": str(status_message),
                                                "change_type": "backup archive was missing from the info collection and added by automatic discovery (legacy archive compatibility mode)",
                                                "backup_archive": str(backupfile),
                                                "size": str(tar_filesize),
                                                "archive_details": json.dumps(
                                                    collections_restore_dict,
                                                    indent=4,
                                                ),
                                            }
                                        )
                                    )

                                except Exception as e:
                                    logger.error(
                                        f'failed to insert a new KVstore record with exception="{str(e)}"'
                                    )
                                    return {
                                        "payload": {
                                            "response": f'failed to insert a new KVstore record with exception="{str(e)}"'
                                        },
                                        "status": 500,
                                    }

                    # Render

                    records = collection_backup_archives_info.data.query()
                    currently_known_archives = []
                    for record in records:
                        currently_known_archives.append(record.get("backup_archive"))

                    logger.info(
                        f'TrackMe get backup files finished successfully, archive_fields="{currently_known_archives}"'
                    )

                    (
                        collection_records,
                        collection_records_dict,
                        collection_records_keys,
                    ) = get_full_collection_records(collection_backup_archives_info)

                    if mode == "full":
                        return {
                            "payload": collection_records,
                            "status": 200,
                        }

                    elif mode == "summary":

                        backup_count = len(collection_records)
                        backup_files = []

                        for record in collection_records:
                            backup_files.append(record.get("backup_archive"))

                        return {
                            "payload": {
                                "backup_count": backup_count,
                                "backup_files": backup_files,
                            },
                            "status": 200,
                        }

    # Take a backup
    def post_backup(self, request_info, **kwargs):

        describe = False
        # Ensure comment is always defined even if no payload is provided
        comment = f"Backup initiated by {request_info.user}, date: {time.strftime('%c')}"

        """
        A simple function to safetly retrieve all records from a KVstore collection with pagination
        """

        def get_full_collection_records(collection):
            collection_records = []
            collection_records_dict = {}
            collection_records_keys = set()

            end = False
            skip_tracker = 0
            while not end:
                process_collection_records = collection.data.query(skip=skip_tracker)
                if process_collection_records:
                    for item in process_collection_records:
                        collection_records.append(item)
                        collection_records_dict[item.get("_key")] = (
                            item  # Add the entire item to the dictionary
                        )
                        collection_records_keys.add(item.get("_key"))
                    skip_tracker += 1000
                else:
                    end = True

            return collection_records, collection_records_dict, collection_records_keys

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            # comment
            try:
                comment = resp_dict["comment"]
                if len(comment) == 0:
                    comment = None
            except Exception as e:
                comment = None
            if not comment:
                comment = f"Backup initiated by {request_info.user}, date: {time.strftime('%c')}"

            # blocklist
            try:
                blocklist = resp_dict["blocklist"]
                if isinstance(blocklist, str):
                    # Convert comma-separated string to list
                    blocklist = [x.strip() for x in blocklist.split(",") if x.strip()]
                elif not isinstance(blocklist, list):
                    blocklist = []
            except Exception as e:
                blocklist = []

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False
            blocklist = []

        if describe:
            response = {
                "describe": "This endpoint performs a backup of all TrackMe collections in a compressed tarball, this file is stored in the TrackMe backup directory, it requires a POST call the following arguments.",
                "resource_desc": "Start TrackMe backup process",
                "resource_spl_example": '| trackme mode=post url="/services/trackme/v2/backup_and_restore/backup"',
                "options": [
                    {
                        "comment": "OPTIONAL: comment to be added to the backup archive",
                        "blocklist": "OPTIONAL: comma-separated list or array of KVstore collection names to exclude from backup",
                    },
                ],
            }

            return {"payload": response, "status": 200}

        else:
            # Get splunkd port
            splunkd_port = request_info.server_rest_port

            # Get service
            service = client.connect(
                owner="nobody",
                app="trackme",
                port=splunkd_port,
                token=request_info.session_key,
                timeout=600,
            )

            # header
            header = {
                "Authorization": "Splunk %s" % request_info.session_key,
                "Content-Type": "application/json",
            }

            # set loglevel
            loglevel = trackme_getloglevel(
                request_info.system_authtoken, request_info.server_rest_port
            )
            logger.setLevel(loglevel)

            # Get trackmeconf
            trackme_conf = trackme_reqinfo(
                request_info.system_authtoken, request_info.server_rest_uri
            )
            trackme_version = trackme_conf.get("trackme_version", "unknown")

            # Set backup root dir
            backuproot = os.path.join(splunkhome, "etc", "apps", "trackme", "backup")

            # Set timestr
            timestr = time.strftime("trackme-backup-%Y%m%d-%H%M%S")

            # Set backup dir
            backupdir = os.path.join(backuproot, str(timestr))

            # get local server name
            server_name = socket.gethostname()

            # Create the backup dir if does not exist
            if not os.path.isdir(backuproot):
                os.mkdir(backuproot)
            if not os.path.isdir(backupdir):
                os.mkdir(backupdir)

            # Clean up any leftover temporary backup directories from previous failed attempts
            cleanup_backup_directories(backuproot, exclude_dir=backupdir)

            # Set up service
            service = client.connect(
                owner="nobody",
                app="trackme",
                port=splunkd_port,
                token=request_info.session_key,
                timeout=600,
            )

            # header
            header = {
                "Authorization": "Splunk %s" % request_info.session_key,
                "Content-Type": "application/json",
            }

            # archive details dictionnary
            archive_details = {}

            # add trackme_version to archive_details
            archive_details["trackme_version"] = trackme_version

            # log start
            logger.info(
                f'TrackMe backup process is initiated, server_name="{server_name}", backup_dir="{backupdir}"'
            )

            ############################################
            # Task 1: Connect to Virtual Tenants KVstore
            ############################################

            # Register the object summary in the vtenant collection
            collection_vtenants_name = "kv_trackme_virtual_tenants"
            collection_vtenants = service.kvstore[collection_vtenants_name]

            # Get records
            vtenants_records, vtenants_collection_keys, vtenants_collection_dict = (
                get_full_kv_collection(collection_vtenants, collection_vtenants_name)
            )

            #################################################################################
            # Task 2: For each Virtual Tenant, retrieve and backup the Virtual Tenant account
            #################################################################################

            # Iterate over the Virtual Tenants
            for vtenant_record in vtenants_records:

                # get the tenant_id
                tenant_id = vtenant_record.get("tenant_id")

                # get the status
                tenant_status = vtenant_record.get("tenant_status")

                # only consider enabled tenants
                if tenant_status == "enabled":

                    # url
                    url = f"{request_info.server_rest_uri}/services/trackme/v2/configuration/admin/get_vtenant_account"

                    try:
                        response = requests.post(
                            url,
                            headers=header,
                            data=json.dumps({"tenant_id": tenant_id}),
                            verify=False,
                            timeout=600,
                        )

                        response.raise_for_status()
                        response_json = response.json()
                        virtual_tenant_account = response_json.get("vtenant_account")

                    except Exception as e:
                        error_msg = f'Failed to retrieve Virtual Tenant account for tenant_id="{tenant_id}", exception="{str(e)}"'
                        logger.error(error_msg)
                        backup_failures.append(error_msg)
                        # Continue with other tenants instead of stopping
                        continue

                    # save the file to our backup directory as tenant_<tenant_id>_vtenant_account.json
                    target = os.path.join(
                        backupdir, f"tenant_{tenant_id}_vtenant_account.json"
                    )

                    try:
                        with open(target, "w") as f:
                            f.write(json.dumps(virtual_tenant_account, indent=1))
                        logger.info(
                            f'Successfully backed up Virtual Tenant account for tenant_id="{tenant_id}"'
                        )
                    except Exception as e:
                        error_msg = f'Exception encountered while trying to backup the Virtual Tenant account for tenant_id="{tenant_id}", exception="{str(e)}"'
                        logger.error(error_msg)
                        backup_failures.append(error_msg)
                        # Continue with other tenants instead of stopping
                        continue

                    # add to the archive details
                    archive_details[f"tenant_{tenant_id}_vtenant_account"] = {
                        "file": f"tenant_{tenant_id}_vtenant_account.json",
                        "size": os.path.getsize(target),
                        "mtime": round(os.path.getmtime(target)),
                    }

            ##########################################
            # Task 3: Knowledge Objects identification
            ##########################################

            # total counters for reporting purposes
            total_count_tenants = 0
            total_count_kos = 0
            total_count_kvstore_collections_definitions = 0
            total_count_transforms_definitions = 0
            total_count_reports = 0
            total_count_alerts = 0
            total_count_macros = 0

            # Iterate over the Virtual Tenants
            for vtenant_record in vtenants_records:

                # get the status
                tenant_status = vtenant_record.get("tenant_status")

                # only consider enabled tenants
                if tenant_status == "enabled":

                    tenant_id = vtenant_record.get("tenant_id")
                    total_count_tenants += 1

                    # tenant counters for reporting purposes
                    tenant_count_kos = 0
                    tenant_count_kvstore_collections_definitions = 0
                    tenant_count_transforms_definitions = 0
                    tenant_count_reports = 0
                    tenant_count_alerts = 0
                    tenant_count_macros = 0

                    # a dictionary to store the list of knowledge objects
                    trackme_knowledge_objects = {}

                    # url
                    url = f"{request_info.server_rest_uri}/services/trackme/v2/configuration/get_tenant_knowledge_objects"

                    try:
                        response = requests.post(
                            url,
                            headers=header,
                            data=json.dumps({"tenant_id": tenant_id}),
                            verify=False,
                            timeout=600,
                        )

                        response.raise_for_status()
                        response_json = response.json()

                    except Exception as e:
                        error_msg = f'Failed to retrieve Knowledge Objects for tenant_id="{tenant_id}", exception="{str(e)}"'
                        logger.error(error_msg)
                        backup_failures.append(error_msg)
                        # Continue with other tenants instead of stopping
                        continue

                    # parse
                    for item in response_json:

                        # increment the counter
                        total_count_kos += 1
                        tenant_count_kos += 1

                        # get the type
                        type = item.get("type")

                        # get the title
                        title = item.get("title")

                        # get the properties
                        properties = item.get("properties", {})

                        # object dict
                        object_dict = {
                            "type": type,
                            "title": title,
                            "properties": properties,
                        }

                        # increment the counters
                        if type == "savedsearches":
                            total_count_reports += 1
                            tenant_count_reports += 1

                            # get the search definition
                            object_dict["definition"] = item.get("definition")

                        elif type == "alerts":
                            total_count_alerts += 1
                            tenant_count_alerts += 1

                            # get the search definition
                            object_dict["definition"] = item.get("definition")

                            # get the alert_properties
                            object_dict["alert_properties"] = item.get(
                                "alert_properties"
                            )

                        elif type == "macros":
                            total_count_macros += 1
                            tenant_count_macros += 1

                            # get the search definition
                            object_dict["definition"] = item.get("definition")

                        elif type == "lookup_definitions":
                            total_count_transforms_definitions += 1
                            tenant_count_transforms_definitions += 1

                            # get collection and fields_list
                            object_dict["collection"] = item.get("collection")
                            object_dict["fields_list"] = item.get("fields_list")

                        elif type == "kvstore_collections":
                            total_count_kvstore_collections_definitions += 1
                            tenant_count_kvstore_collections_definitions += 1

                        # add to the dict
                        trackme_knowledge_objects[title] = object_dict

                    # save the file to our backup directory as knowledge_objects.json
                    target = os.path.join(
                        backupdir, f"tenant_{tenant_id}_knowledge_objects.json"
                    )

                    try:
                        with open(target, "w") as f:
                            f.write(json.dumps(trackme_knowledge_objects, indent=1))
                        logger.info(
                            f'Successfully backed up Knowledge Objects for tenant_id="{tenant_id}"'
                        )
                    except Exception as e:
                        error_msg = f'Exception encountered while trying to backup the knowledge objects for tenant_id="{tenant_id}", exception="{str(e)}"'
                        logger.error(error_msg)
                        backup_failures.append(error_msg)
                        # Continue with other tenants instead of stopping
                        continue

                    # add to the archive details
                    archive_details[f"tenant_{tenant_id}_knowledge_objects"] = {
                        "file": f"tenant_{tenant_id}_knowledge_objects.json",
                        "size": os.path.getsize(target),
                        "mtime": round(os.path.getmtime(target)),
                        "count_knowledge_objects": tenant_count_kos,
                        "count_kvstore_collections": tenant_count_kvstore_collections_definitions,
                        "count_transforms": tenant_count_transforms_definitions,
                        "count_reports": tenant_count_reports,
                        "count_alerts": tenant_count_alerts,
                        "count_macros": tenant_count_macros,
                    }

                    logger.info(
                        f'successfully backed up the knowledge objects, total="{total_count_kos}", kvstore_collections="{total_count_kvstore_collections_definitions}", transforms="{total_count_transforms_definitions}", reports="{total_count_reports}", alerts="{total_count_alerts}", macros="{total_count_macros}"'
                    )

            ####################################
            # Task 4: KVstore collections backup
            ####################################

            # to count the number of non empty collections backed up
            counter_performed = 0

            # to count the number of empty collections
            counter_empty = 0

            # to track backup failures for reporting
            backup_failures = []
            failed_collections = []

            # retrieve the lost of collections to be backed up

            # Spawn a new search
            # Get lagging statistics from live data
            kwargs_search = {
                "app": "trackme",
                "earliest_time": "-5m",
                "latest_time": "now",
                "output_mode": "json",
                "count": 0,
            }
            searchquery = '| rest splunk_server=local "/servicesNS/nobody/trackme/storage/collections/config" | where \'eai:acl.app\'="trackme" AND disabled=0 | table title | stats values(title) as collections | tojson | fields _raw'

            # Get the results and display them using the JSONResultsReader
            try:
                # spawn the search and get the results
                reader = run_splunk_search(
                    service,
                    searchquery,
                    kwargs_search,
                    24,
                    5,
                )

                for item in reader:
                    if isinstance(item, dict):
                        collections_json = json.loads(item.get("_raw"))
                        logger.info(
                            f'successfully retrieved the list of collections to be backed up, collections_json="{collections_json}"'
                        )

            except Exception as e:
                collections_json = None
                logger.error(
                    f'failed to retrieve the list of collections to be handled with exception="{str(e)}"'
                )

            if collections_json:
                # Loop through the collections
                for collection_name in collections_json["collections"]:
                    logger.debug(f'Handling collection="{collection_name}"')

                    # Skip if collection is in blocklist
                    if collection_name in blocklist:
                        logger.info(
                            f'Skipping collection="{collection_name}" as it is in the blocklist'
                        )
                        continue

                    # Always skip stateful charts collections
                    if collection_name.startswith(
                        "kv_trackme_stateful_alerting_charts_tenant_"
                    ):
                        logger.info(
                            f'Skipping collection="{collection_name}" as it is a stateful charts collection'
                        )
                        continue

                    try:
                        collection = service.kvstore[collection_name]

                        # get records
                        (
                            collection_records,
                            collection_records_dict,
                            collection_records_keys,
                        ) = get_full_collection_records(collection)

                        # if empty, increment the counter
                        if len(collection_records) == 0:
                            counter_empty += 1

                        # Backup the collection
                        target = os.path.join(backupdir, f"{collection_name}.json")

                        try:
                            with open(target, "w") as f:
                                f.write(json.dumps(collection_records, indent=1))
                            counter_performed += 1
                            logger.info(
                                f'Successfully backed up collection="{collection_name}" with {len(collection_records)} records'
                            )

                        except Exception as e:
                            error_msg = f'Exception encountered while trying to backup collection="{collection_name}", exception="{str(e)}"'
                            logger.error(error_msg)
                            backup_failures.append(error_msg)
                            failed_collections.append(collection_name)
                            # Continue with other collections instead of stopping

                    except Exception as e:
                        error_msg = f'General exception encountered while trying to access KVstore collection="{collection_name}", exception="{str(e)}"'
                        logger.error(error_msg)
                        backup_failures.append(error_msg)
                        failed_collections.append(collection_name)
                        # Continue with other collections instead of stopping

                # Check if we have any successful backups before creating archive
                if counter_performed == 0 and len(backup_failures) > 0:
                    logger.error(
                        "No collections were successfully backed up. Cleaning up and aborting backup process."
                    )
                    cleanup_backup_directories(backuproot, exclude_dir=backupdir)
                    return {
                        "payload": {
                            "response": f"Backup process failed: No collections were successfully backed up. Failures: {backup_failures}",
                        },
                        "status": 500,
                    }

                # create a compressed archive (zstd preferred, gzip fallback)
                archive_name = os.path.basename(backupdir)
                compressed_archive_path = create_compressed_archive(
                    backupdir, archive_name
                )

                if not compressed_archive_path:
                    logger.error(
                        f'Exception encountered while trying to generate the compressed archive for="{backupdir}"'
                    )
                    return {
                        "payload": {
                            "response": f'Exception encountered while trying to generate the compressed archive for="{backupdir}"',
                        },
                        "status": 500,
                    }

                # Get the archive file size
                try:
                    tar_filesize = os.path.getsize(compressed_archive_path)
                except Exception as e:
                    tar_filesize = None
                    logger.error(
                        f'tenant_id="{compressed_archive_path}", failure to retrieve the size of the compressed archive="{compressed_archive_path}" with exception:"{str(e)}"'
                    )

                # Log backup summary before processing archive details
                logger.info(
                    f"Backup archive created successfully: {compressed_archive_path}"
                )
                logger.info(
                    f"Successfully backed up {counter_performed} collections, {counter_empty} empty collections, {len(backup_failures)} failures"
                )
                if backup_failures:
                    logger.warning(f"Failed collections: {failed_collections}")

                # store the list of json files in a list
                collections_json_files = [
                    f
                    for f in listdir(backupdir)
                    if isfile(join(backupdir, f)) and f.startswith("kv_")
                ]

                # store the list of available collections in the archive
                collections_available = []

                # handle the dictionnary
                for json_file in collections_json_files:

                    # if the file starts by kv_
                    if json_file.startswith("kv_"):

                        if os.path.isfile(os.path.join(backupdir, json_file)):

                            # strip the extension
                            collection_name = os.path.splitext(json_file)[0]

                            # append to the list of available collections for restore
                            collections_available.append(collection_name)

                            # get the file size
                            json_file_size = os.path.getsize(
                                os.path.join(backupdir, json_file)
                            )

                            # get the file mtime
                            json_file_mtime = round(
                                os.path.getmtime(os.path.join(backupdir, json_file))
                            )

                            # try getting the number of records
                            try:
                                with open(
                                    os.path.join(backupdir, json_file), "r"
                                ) as read_content:
                                    json_file_records = len(json.load(read_content))
                            except Exception as e:
                                json_file_records = None

                            # add to the dict
                            archive_details[collection_name] = {
                                "file": json_file,
                                "size": json_file_size,
                                "mtime": json_file_mtime,
                                "records": json_file_records,
                            }

                # remove backup dir
                try:
                    shutil.rmtree(backupdir)
                except OSError as e:
                    logger.error(
                        f'failed to purge the extraction temporary directory="{backupdir}" with exception="{str(e)}"'
                    )

                # Store a record in a backup audit collection

                # backup audit collection
                collection_name_backup_archives_info = "kv_trackme_backup_archives_info"
                service_backup_archives_info = client.connect(
                    owner="nobody",
                    app="trackme",
                    port=splunkd_port,
                    token=request_info.session_key,
                    timeout=600,
                )
                collection_backup_archives_info = service_backup_archives_info.kvstore[
                    collection_name_backup_archives_info
                ]

                # Get time
                current_time = int(round(time.time()))

                # Create a message
                status_message = {
                    "trackme_version": str(trackme_version),
                    "backup_archive": str(compressed_archive_path),
                    "server_name": str(server_name),
                    "comment": str(comment),
                    "kvstore_collections_not_empty": str(counter_performed),
                    "kvstore_collections_empty": str(counter_empty),
                    "kvstore_collections_blocked": len(blocklist),
                    "kvstore_collections_blocklist": blocklist,
                    "knowledge_objects_summary": {
                        "tenants_enabled_count": total_count_tenants,
                        "total": total_count_kos,
                        "kvstore_collections": total_count_kvstore_collections_definitions,
                        "transforms": total_count_transforms_definitions,
                        "reports": total_count_reports,
                        "alerts": total_count_alerts,
                        "macros": total_count_macros,
                    },
                    "backup_summary": {
                        "total_failures": len(backup_failures),
                        "failed_collections": failed_collections,
                        "backup_failures": backup_failures,
                        "partial_backup": len(backup_failures) > 0,
                    },
                }

                response = {
                    "mtime": str(current_time),
                    "htime": str(time.strftime("%c", time.localtime(current_time))),
                    "server_name": str(server_name),
                    "comment": str(comment),
                    "status": status_message,
                    "change_type": "backup archive created",
                    "backup_archive": str(compressed_archive_path),
                    "size": tar_filesize,
                    "archive_details": archive_details,
                }

                kvrecord = {
                    "mtime": str(current_time),
                    "htime": str(time.strftime("%c", time.localtime(current_time))),
                    "server_name": str(server_name),
                    "comment": str(comment),
                    "status": json.dumps(status_message, indent=2),
                    "change_type": "backup archive created",
                    "backup_archive": str(compressed_archive_path),
                    "size": tar_filesize,
                    "archive_details": json.dumps(archive_details, indent=2),
                }

                try:
                    # Insert the record
                    collection_backup_archives_info.data.insert(json.dumps(kvrecord))

                except Exception as e:
                    logger.error(
                        f'Exception encountered while attempting to insert a record in the KVstore collection="{collection_name_backup_archives_info}", exception="{str(e)}"'
                    )
                    return {
                        "payload": {
                            "response": f'Exception encountered while attempting to insert a record in the KVstore collection="{collection_name_backup_archives_info}", exception="{str(e)}"'
                        },
                        "status": 500,
                    }

                # In backup dir, store the status_message in a file named as the backup archive file with .full.metadata extension
                try:
                    with open(f"{compressed_archive_path}.full.meta", "w") as f:
                        f.write(json.dumps(response, indent=2))
                except Exception as e:
                    logger.error(
                        f'Exception encountered while trying to store the metadata file for the backup archive="{compressed_archive_path}", exception="{str(e)}"'
                    )
                    return {
                        "payload": {
                            "response": f'Exception encountered while trying to store the metadata file for the backup archive="{compressed_archive_path}", exception="{str(e)}"'
                        },
                        "status": 500,
                    }

                # In backup dir, store the response in a file named as the backup archive file with .light.metadata extension
                try:
                    with open(f"{compressed_archive_path}.light.meta", "w") as f:
                        f.write(json.dumps(status_message, indent=2))
                except Exception as e:
                    logger.error(
                        f'Exception encountered while trying to store the metadata file for the backup archive="{compressed_archive_path}", exception="{str(e)}"'
                    )
                    return {
                        "payload": {
                            "response": f'Exception encountered while trying to store the metadata file for the backup archive="{compressed_archive_path}", exception="{str(e)}"'
                        },
                        "status": 500,
                    }

                # Log backup summary
                if backup_failures:
                    logger.warning(
                        f"TrackMe backup completed with {len(backup_failures)} failures. Failed collections: {failed_collections}"
                    )
                    for failure in backup_failures:
                        logger.warning(f"Backup failure: {failure}")
                else:
                    logger.info(
                        "TrackMe backup completed successfully with no failures"
                    )

                # Finally render a status message
                logger.info(
                    f'TrackMe backup process is terminated, status="{json.dumps(status_message, indent=4)}"'
                )
                return {"payload": {"response": response}, "status": 200}

            else:

                error_msg = "TrackMe backup process has failed, failed to retrieve the list of KVstore collections to be backed up!"
                logger.error(error_msg)
                # Clean up the current backup directory before returning error
                cleanup_backup_directories(backuproot, exclude_dir=backupdir)
                return {
                    "payload": {
                        "response": error_msg,
                    },
                    "status": 500,
                }

    # Purge older backup archives based on a retention
    def delete_backup(self, request_info, **kwargs):

        def cleanup_timestamped_temp_directories(backup_root_dir):
            """
            Clean up any timestamped temporary directories (backup_temp_ddmmyyyy) found in the backup directory.
            This function handles both local and remote cleanup scenarios.
            """
            try:
                # Look for timestamped temp directories with pattern backup_temp_*
                temp_dir_pattern = os.path.join(backup_root_dir, "backup_temp_*")
                temp_dirs = glob.glob(temp_dir_pattern)
                
                cleaned_dirs = []
                for temp_dir in temp_dirs:
                    try:
                        if os.path.isdir(temp_dir):
                            logger.info(f"Found timestamped temp directory to clean: {temp_dir}")
                            shutil.rmtree(temp_dir, ignore_errors=True)
                            cleaned_dirs.append(temp_dir)
                            logger.info(f"Successfully cleaned timestamped temp directory: {temp_dir}")
                    except Exception as e:
                        logger.warning(f"Failed to clean temp directory {temp_dir}: {str(e)}")
                
                if cleaned_dirs:
                    logger.info(f"Cleaned up {len(cleaned_dirs)} timestamped temp directories: {cleaned_dirs}")
                else:
                    logger.info("No timestamped temp directories found to clean")
                    
                return cleaned_dirs
                
            except Exception as e:
                logger.error(f"Error during timestamped temp directory cleanup: {str(e)}")
                return []

        def purge_kv_record_for_archive(archive_name, archive_path=None):
            """
            Remove the KVstore record(s) that reference the given archive.
            Tries exact path match first (when archive_path is provided), then
            falls back to a regex on the archive_name.
            Returns True if at least one record was deleted, False otherwise.
            """
            removed = False
            try:
                collection_name = "kv_trackme_backup_archives_info"
                service = client.connect(
                    owner="nobody",
                    app="trackme",
                    port=splunkd_port,
                    token=request_info.session_key,
                    timeout=600,
                )
                collection = service.kvstore[collection_name]

                # Exact path match when available
                if archive_path:
                    try:
                        kvrec = collection.data.query(query=json.dumps({"backup_archive": archive_path}))[0]
                        key = kvrec.get("_key")
                        if key:
                            collection.data.delete(json.dumps({"_key": key}))
                            removed = True
                    except Exception:
                        # ignore and try regex fallback below
                        pass

                # Regex fallback on file name
                try:
                    kvrecs = collection.data.query(query=json.dumps({"backup_archive": {"$regex": f".*{archive_name}$"}}))
                    for kvrec in kvrecs:
                        key = kvrec.get("_key")
                        if key:
                            try:
                                collection.data.delete(json.dumps({"_key": key}))
                                removed = True
                            except Exception as e:
                                logger.warning(f"Failed to delete KV record key={key}: {str(e)}")
                except Exception as e:
                    logger.warning(f"KVstore regex cleanup failed for archive '{archive_name}': {str(e)}")
            except Exception as e:
                logger.warning(f"KVstore cleanup failed for archive '{archive_name}': {str(e)}")

            return removed

        describe = False
        retention_days = 30  # default to 30 days of retention if not specified
        force_local = False  # default to False, will be set from request if provided
        archive_name = None  # optional targeted deletion
        target_server_name = None  # optional explicit server to contact remotely

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get local server name
        server_name = socket.gethostname()

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            # Get the retention_days parameter
            try:
                retention_days = resp_dict["retention_days"]

                # convert to an integer
                try:
                    retention_days = int(retention_days)

                except Exception as e:
                    logger.error(
                        f'Invalid retention_days="{retention_days}", expecting an integer'
                    )
                    return {
                        "payload": {
                            "response": f'Invalid retention_days="{retention_days}", expecting an integer, exception="{str(e)}"',
                        },
                        "status": 500,
                    }

            except Exception as e:
                # default to 30 days
                retention_days = 30

            # Optional targeted deletion parameters
            try:
                archive_name = resp_dict.get("archive_name")
            except Exception:
                archive_name = None

            try:
                target_server_name = resp_dict.get("server_name")
            except Exception:
                target_server_name = None

            # Get the force_local parameter
            try:
                force_local = resp_dict.get("force_local", False)
                if force_local:
                    # accept booleans or strings: true/True/1
                    if isinstance(force_local, bool):
                        force_local = force_local
                    elif isinstance(force_local, str):
                        if force_local in ("true", "True", "1"):
                            force_local = True
                        elif force_local in ("false", "False", "0"):
                            force_local = False
                        else:
                            return {
                                "payload": {"error": "force_local must be true or false"},
                                "status": 400,
                            }
                    else:
                        force_local = False

            except Exception as e:
                # default to False
                force_local = False

        else:
            # body is not required in this endpoint
            describe = False

        if describe:
            response = {
                "describe": "This endpoint deletes backups. It supports either: (1) a targeted deletion by archive_name, or (2) a retention-based purge of files older than retention_days.",
                "resource_desc": "Delete TrackMe backup archives",
                "resource_spl_example": "| trackme url=/services/trackme/v2/backup_and_restore/backup mode=delete body=\"{'archive_name': 'trackme-backup-20210205-142635.tgz', 'force_local': 'true'}\" OR | trackme url=/services/trackme/v2/backup_and_restore/backup mode=delete body=\"{'retention_days': '30'}\"",
                "options": [
                    {
                        "retention_days": "(integer) OPTIONAL: the maximal retention for backup archive files in days, if not specified defaults to 30 days",
                    },
                    {
                        "archive_name": "(string) OPTIONAL: Name of the backup archive to delete. If provided, performs a targeted deletion instead of retention purge.",
                    },
                    {
                        "force_local": "(true / false) OPTIONAL: if true, the endpoint will only process local backup archives and will not attempt to delegate to remote servers in a Search Head Cluster context. For targeted deletion with server_name, remote call will force_local=True.",
                    },
                    {
                        "server_name": "(string) OPTIONAL: If provided with archive_name, the endpoint will attempt to call that remote server to delete the archive (with force_local=true on the remote).",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        else:
            # Set backup root dir
            backuproot = os.path.join(splunkhome, "etc", "apps", "trackme", "backup")

            # Clean up any timestamped temporary directories first
            logger.info("Starting cleanup of timestamped temporary directories...")
            cleaned_temp_dirs = cleanup_timestamped_temp_directories(backuproot)

            # If archive_name is provided, perform targeted deletion logic
            if archive_name:
                logger.info(f"delete_backup targeted deletion requested for archive_name='{archive_name}', force_local='{force_local}', server_name='{target_server_name}'")

                # Determine if we should handle locally or remotely
                current_server_name = server_name

                # By default, assume local handling
                handle_local = True
                remote_target_name = None

                # If an explicit server_name is provided, handle locally if it targets this host (short or FQDN)
                if target_server_name:
                    try:
                        local_short = current_server_name.lower()
                        local_fqdn = socket.getfqdn().lower()
                        target_lower = str(target_server_name).lower()
                        target_short = target_lower.split('.', 1)[0]

                        if target_lower in (local_short, local_fqdn) or target_short == local_short:
                            handle_local = True
                            remote_target_name = None
                        else:
                            handle_local = False
                            remote_target_name = target_server_name
                    except Exception:
                        # On any resolution error, fall back to previous behavior
                        if target_server_name != current_server_name:
                            handle_local = False
                            remote_target_name = target_server_name
                else:
                    # If not forced local, try to resolve server_name via KVstore metadata
                    if not force_local:
                        try:
                            collection_name = "kv_trackme_backup_archives_info"
                            service = client.connect(
                                owner="nobody",
                                app="trackme",
                                port=splunkd_port,
                                token=request_info.session_key,
                                timeout=600,
                            )
                            collection = service.kvstore[collection_name]
                            kv_query = {"backup_archive": {"$regex": f".*{archive_name}$"}}
                            kvrecords = collection.data.query(query=json.dumps(kv_query))
                            if len(kvrecords) > 0:
                                backup_server_name = kvrecords[0].get("server_name")
                                if backup_server_name and backup_server_name != current_server_name:
                                    handle_local = False
                                    remote_target_name = backup_server_name
                        except Exception as e:
                            logger.warning(f"KVstore lookup for targeted deletion failed: {str(e)}")

                if handle_local:
                    # Perform local deletion
                    try:
                        archive_path = os.path.join(backuproot, archive_name)
                        if not os.path.exists(archive_path):
                            # File missing locally: purge KVstore record(s) anyway
                            purged = purge_kv_record_for_archive(archive_name, archive_path)
                            msg = f"Archive not found locally ({archive_path}). KVstore record removed={purged}"
                            logger.warning(msg)
                            return {
                                "payload": {"error": msg},
                                "status": 404,
                            }

                        # delete archive file
                        os.remove(archive_path)

                        # delete optional metadata files if present
                        for meta_suffix in [".light.meta", ".full.meta"]:
                            try:
                                meta_path = f"{archive_path}{meta_suffix}"
                                if os.path.exists(meta_path):
                                    os.remove(meta_path)
                            except Exception as e:
                                logger.warning(f"Failed to delete metadata file '{meta_path}': {str(e)}")

                        # purge KVstore record(s)
                        purge_kv_record_for_archive(archive_name, archive_path)

                        response = {
                            "status": f"Successfully deleted backup archive {archive_name} on server {current_server_name}",
                            "local_operation": {
                                "status": "success",
                                "backup_files": [archive_path],
                                "temp_directories_cleaned": len(cleaned_temp_dirs),
                                "temp_directories": cleaned_temp_dirs,
                            },
                            "remote_operations": [],
                        }
                        logger.info(json.dumps(response, indent=4))
                        return {"payload": response, "status": 200}
                    except Exception as e:
                        logger.error(f"Error deleting local archive '{archive_name}': {str(e)}")
                        return {"payload": {"error": str(e)}, "status": 500}

                else:
                    # Perform remote deletion by calling the endpoint on the remote server with force_local=True
                    try:
                        # Determine the best reachable name (short vs FQDN)
                        target_name = remote_target_name
                        if not test_splunkd_connectivity(
                            remote_target_name,
                            request_info.server_rest_port,
                            request_info.session_key,
                        ):
                            remote_server_fqdn = (
                                socket.getfqdn()
                                if remote_target_name == socket.gethostname()
                                else f"{remote_target_name}.{socket.getfqdn().split('.', 1)[1] if '.' in socket.getfqdn() else 'local'}"
                            )
                            logger.info(f"Short hostname failed, trying FQDN: {remote_server_fqdn}")
                            if test_splunkd_connectivity(
                                remote_server_fqdn,
                                request_info.server_rest_port,
                                request_info.session_key,
                            ):
                                target_name = remote_server_fqdn
                            else:
                                return {
                                    "payload": {"error": f"Connectivity failed to remote server {remote_target_name}"},
                                    "status": 500,
                                }

                        target_server_uri = f"https://{target_name}:{request_info.server_rest_port}"
                        target_url = f"{target_server_uri}/services/trackme/v2/backup_and_restore/backup"
                        request_payload = {"archive_name": archive_name, "force_local": True}
                        logger.info(f"Delegating targeted deletion to remote server {target_name} with payload={json.dumps(request_payload)}")

                        response = requests.delete(
                            target_url,
                            json=request_payload,
                            headers={
                                "Authorization": f"Splunk {request_info.session_key}",
                                "Content-Type": "application/json",
                            },
                            verify=False,
                            timeout=300,
                        )

                        if response.status_code == 200:
                            response_data = response.json()
                            remote_info = {
                                "server": target_name,
                                "status": "success",
                                "remote_response": response_data,
                            }
                            overall = {
                                "status": f"Successfully delegated deletion of {archive_name} to remote server {target_name}",
                                "local_operation": {
                                    "status": "delegated",
                                    "backup_files": [],
                                    "temp_directories_cleaned": len(cleaned_temp_dirs),
                                    "temp_directories": cleaned_temp_dirs,
                                },
                                "remote_operations": [remote_info],
                            }
                            logger.info(json.dumps(overall, indent=4))
                            return {"payload": overall, "status": 200}
                        else:
                            logger.error(f"Remote server {target_name} returned error: {response.status_code} - {response.text}")
                            # If remote deletion failed due to 404 (missing file) or connectivity issues (5xx),
                            # still purge the local KVstore record referencing this archive
                            if response.status_code in (404, 502, 503, 504):
                                purge_kv_record_for_archive(archive_name)
                            return {
                                "payload": {
                                    "error": f"Remote deletion failed on {target_name}: HTTP {response.status_code}: {response.text}"
                                },
                                "status": response.status_code,
                            }
                    except Exception as e:
                        logger.error(f"Error during remote targeted deletion to {remote_target_name}: {str(e)}")
                        # On connectivity error, purge KVstore record locally
                        purge_kv_record_for_archive(archive_name)
                        return {"payload": {"error": str(e)}, "status": 500}

            # For reporting purposes (retention-based purge)
            purgedlist = []
            remote_purged_results = []

            # retention_days = 30

            time_in_secs = time.time() - (retention_days * 24 * 60 * 60)
            for root, dirs, files in os.walk(backuproot, topdown=False):
                for file_ in files:
                    full_path = os.path.join(root, file_)
                    stat = os.stat(full_path)

                    if stat.st_mtime <= time_in_secs:
                        if os.path.isdir(full_path):
                            try:
                                os.rmdir(full_path)
                                purgedlist.append(full_path)

                            except Exception as e:
                                logger.error(
                                    f'Failed to delete the backup archive="{full_path}" with exception="{str(e)}"'
                                )
                                return {
                                    "payload": {
                                        "response": f'Failed to delete the backup archive="{full_path}" with exception="{str(e)}"',
                                    },
                                    "status": 500,
                                }

                        else:
                            # Only process backup archive files (.tgz or .tar.zst)
                            if os.path.exists(full_path) and (
                                full_path.endswith(".tgz")
                                or full_path.endswith(".tar.zst")
                            ):
                                # delete
                                os.remove(full_path)
                                # add to list
                                purgedlist.append(full_path)

                                # Purge the record from the KVstore

                                # record / key
                                record = None
                                key = None

                                # Define the KV query
                                query_string = {
                                    "backup_archive": full_path,
                                }

                                # backup audit collection
                                collection_name_backup_archives_info = (
                                    "kv_trackme_backup_archives_info"
                                )
                                service_backup_archives_info = client.connect(
                                    owner="nobody",
                                    app="trackme",
                                    port=splunkd_port,
                                    token=request_info.session_key,
                                    timeout=600,
                                )
                                collection_backup_archives_info = (
                                    service_backup_archives_info.kvstore[
                                        collection_name_backup_archives_info
                                    ]
                                )

                                try:
                                    kvrecord = (
                                        collection_backup_archives_info.data.query(
                                            query=json.dumps(query_string)
                                        )[0]
                                    )
                                    key = kvrecord.get("_key")

                                except Exception as e:
                                    key = None

                                # If a record cannot be found, this backup file is not know to TrackMe currently, add a new record
                                if key is not None:
                                    try:
                                        # Delete the record
                                        collection_backup_archives_info.data.delete(
                                            json.dumps({"_key": key})
                                        )

                                    except Exception as e:
                                        logger.error(
                                            f'Failed to purge the KVstore record in collection="{collection_name_backup_archives_info}" with exception="{str(e)}"'
                                        )
                                        return {
                                            "payload": {
                                                "response": f'Failed to purge the KVstore record in collection="{collection_name_backup_archives_info}" with exception="{str(e)}"',
                                            },
                                            "status": 500,
                                        }

            # Query KVstore to get all unique server names for remote member handling
            # Only do this if force_local is False (default behavior)
            if not force_local:
                collection_name_backup_archives_info = "kv_trackme_backup_archives_info"
                service_backup_archives_info = client.connect(
                    owner="nobody",
                    app="trackme",
                    port=splunkd_port,
                    token=request_info.session_key,
                    timeout=600,
                )
                collection_backup_archives_info = service_backup_archives_info.kvstore[
                    collection_name_backup_archives_info
                ]

                # Get all unique server names from the KVstore
                try:
                    all_records = collection_backup_archives_info.data.query()
                    unique_servers = set()
                    for record in all_records:
                        server_name_from_kv = record.get("server_name")
                        if server_name_from_kv and server_name_from_kv != server_name:
                            unique_servers.add(server_name_from_kv)
                    
                    logger.info(f"Found {len(unique_servers)} remote servers in KVstore: {list(unique_servers)}")
                    
                    # For each remote server, make a call to delete backups with force_local=True
                    for remote_server in unique_servers:
                        try:
                            logger.info(f"Delegating backup deletion to remote server: {remote_server}")
                            
                            # Test connectivity with short hostname first, then FQDN if needed
                            target_server_name = remote_server
                            
                            if not test_splunkd_connectivity(
                                remote_server,
                                request_info.server_rest_port,
                                request_info.session_key,
                            ):
                                # Try with FQDN
                                remote_server_fqdn = (
                                    socket.getfqdn()
                                    if remote_server == socket.gethostname()
                                    else f"{remote_server}.{socket.getfqdn().split('.', 1)[1] if '.' in socket.getfqdn() else 'local'}"
                                )
                                logger.info(
                                    f"Short hostname failed, trying FQDN: {remote_server_fqdn}"
                                )
                                if test_splunkd_connectivity(
                                    remote_server_fqdn,
                                    request_info.server_rest_port,
                                    request_info.session_key,
                                ):
                                    target_server_name = remote_server_fqdn
                                    logger.info(
                                        f"FQDN connectivity successful, using: {target_server_name}"
                                    )
                                else:
                                    logger.warning(
                                        f"Both short hostname and FQDN failed for {remote_server}, skipping remote deletion"
                                    )
                                    remote_purged_results.append({
                                        "server": remote_server,
                                        "status": "failed",
                                        "error": "Connectivity failed"
                                    })
                                    continue
                            else:
                                logger.info(
                                    f"Short hostname connectivity successful, using: {target_server_name}"
                                )

                            # support only https
                            target_server_uri = (
                                f"https://{target_server_name}:{request_info.server_rest_port}"
                            )

                            # Prepare the request payload
                            request_payload = {
                                "retention_days": retention_days,
                                "force_local": True,
                            }

                            target_url = f"{target_server_uri}/services/trackme/v2/backup_and_restore/backup"

                            # Make REST call to target server
                            try:
                                response = requests.delete(
                                    target_url,
                                    json=request_payload,
                                    headers={
                                        "Authorization": f"Splunk {request_info.session_key}",
                                        "Content-Type": "application/json",
                                    },
                                    verify=False,
                                    timeout=300,
                                )

                                if response.status_code == 200:
                                    response_data = response.json()
                                    logger.info(
                                        f"Successfully deleted backups on remote server {remote_server}"
                                    )
                                    
                                    # Extract relevant information from remote response
                                    remote_status = response_data.get("status", "Unknown status")
                                    remote_files_purged = 0
                                    if "local_operation" in response_data:
                                        remote_files_purged = response_data["local_operation"].get("files_purged", 0)
                                    elif "backup_files" in response_data:
                                        remote_files_purged = len(response_data["backup_files"])
                                    
                                    remote_purged_results.append({
                                        "server": remote_server,
                                        "status": "success",
                                        "local_status": remote_status,
                                        "files_purged": remote_files_purged
                                    })
                                else:
                                    logger.error(
                                        f"Remote server {remote_server} returned error: {response.status_code} - {response.text}"
                                    )
                                    remote_purged_results.append({
                                        "server": remote_server,
                                        "status": "failed",
                                        "error": f"HTTP {response.status_code}: {response.text}"
                                    })

                            except requests.exceptions.ConnectionError as e:
                                logger.error(
                                    f"Connection error to remote server {remote_server}: {str(e)}"
                                )
                                remote_purged_results.append({
                                    "server": remote_server,
                                    "status": "failed",
                                    "error": f"Connection error: {str(e)}"
                                })
                            except Exception as e:
                                logger.error(
                                    f"Error making REST call to remote server {remote_server}: {str(e)}"
                                )
                                remote_purged_results.append({
                                    "server": remote_server,
                                    "status": "failed",
                                    "error": f"REST call error: {str(e)}"
                                })

                        except Exception as e:
                            logger.error(
                                f"Error processing remote server {remote_server}: {str(e)}"
                            )
                            remote_purged_results.append({
                                "server": remote_server,
                                "status": "failed",
                                "error": f"Processing error: {str(e)}"
                            })

                except Exception as e:
                    logger.error(
                        f"Failed to query KVstore for remote servers: {str(e)}"
                    )
                    # Continue with local processing even if remote query fails

            if not len(purgedlist) > 2:
                # Determine overall status based on local and remote results
                local_status = f"No local backup files older than {retention_days} days found"
                remote_success_count = sum(1 for result in remote_purged_results if result.get("status") == "success")
                remote_failed_count = len(remote_purged_results) - remote_success_count
                
                if remote_purged_results:
                    if remote_failed_count == 0:
                        overall_status = f"{local_status}. Remote cleanup: {remote_success_count} server(s) processed successfully"
                    elif remote_success_count == 0:
                        overall_status = f"{local_status}. Remote cleanup: {remote_failed_count} server(s) failed"
                    else:
                        overall_status = f"{local_status}. Remote cleanup: {remote_success_count} server(s) succeeded, {remote_failed_count} server(s) failed"
                else:
                    overall_status = local_status

                response = {
                    "status": overall_status,
                    "local_operation": {
                        "status": local_status,
                        "files_purged": 0,
                        "temp_directories_cleaned": len(cleaned_temp_dirs),
                        "temp_directories": cleaned_temp_dirs
                    },
                    "remote_operations": remote_purged_results
                }

                logger.info(json.dumps(response, indent=4))
                return {"payload": response, "status": 200}

            else:
                # For each backup archive, if a record is found in the collection info, remove the record

                # backup audit collection
                collection_name_backup_archives_info = "kv_trackme_backup_archives_info"
                service_backup_archives_info = client.connect(
                    owner="nobody",
                    app="trackme",
                    port=splunkd_port,
                    token=request_info.session_key,
                    timeout=600,
                )
                collection_backup_archives_info = service_backup_archives_info.kvstore[
                    collection_name_backup_archives_info
                ]

                for backupfile in purgedlist:
                    key = None
                    record = None

                    # Define the KV query
                    query_string = {
                        "$and": [
                            {
                                "backup_archive": backupfile,
                                "server_name": server_name,
                            }
                        ]
                    }

                    try:
                        kvrecord = collection_backup_archives_info.data.query(
                            query=json.dumps(query_string)
                        )[0]
                        key = kvrecord.get("_key")

                    except Exception as e:
                        key = None

                    # If a record is found, it shall be purged
                    if key is not None:
                        # Remove the record
                        try:
                            collection_backup_archives_info.data.delete(
                                json.dumps({"_key": key})
                            )

                        except Exception as e:
                            logger.error(
                                f'Failed to purge the KVstore record in collection="{collection_name_backup_archives_info}" with exception="{str(e)}"'
                            )
                            return {
                                "payload": {
                                    "response": f'Failed to purge the KVstore record in collection="{collection_name_backup_archives_info}" with exception="{str(e)}"',
                                },
                                "status": 500,
                            }

                # Determine overall status based on local and remote results
                local_status = f"Successfully purged {len(purgedlist)} local backup files older than {retention_days} days"
                remote_success_count = sum(1 for result in remote_purged_results if result.get("status") == "success")
                remote_failed_count = len(remote_purged_results) - remote_success_count
                
                if remote_purged_results:
                    if remote_failed_count == 0:
                        overall_status = f"{local_status}. Remote cleanup: {remote_success_count} server(s) processed successfully"
                    elif remote_success_count == 0:
                        overall_status = f"{local_status}. Remote cleanup: {remote_failed_count} server(s) failed"
                    else:
                        overall_status = f"{local_status}. Remote cleanup: {remote_success_count} server(s) succeeded, {remote_failed_count} server(s) failed"
                else:
                    overall_status = local_status

                response = {
                    "status": overall_status,
                    "local_operation": {
                        "status": local_status,
                        "files_purged": len(purgedlist),
                        "backup_files": purgedlist,
                        "temp_directories_cleaned": len(cleaned_temp_dirs),
                        "temp_directories": cleaned_temp_dirs
                    },
                    "remote_operations": remote_purged_results
                }

                logger.info(json.dumps(response, indent=4))
                return {"payload": response, "status": 200}

    # Restore collections from a backup archive
    # For Splunk Cloud certification purposes, the archive must be located in the backup directory of the application

    def post_restore(self, request_info, **kwargs):

        def clean_backup_dir(backupdir):
            """
            A simple function to remove the backup dir
            """
            try:
                shutil.rmtree(backupdir)
                return True
            except OSError as e:
                logger.error(
                    f'failed to purge the extraction temporary directory="{backupdir}" with exception="{str(e)}"'
                )

        def restore_kvstore_records(
            service,
            collection_name,
            collections_restore_dict,
            backupdir,
            kvstore_collections_global_records_to_be_restored,
            kvstore_collections_global_records_restored,
            kvstore_collections_restored_warning,
            restore_results_dict,
            kvstore_collections_clean_empty,
        ):
            """
            Restore KVstore records from a backup JSON file, with an option to clean existing records.

            Args:
                service: Splunk service object for KVstore access.
                collection_name: The name of the KVstore collection to restore.
                collections_restore_dict: Dictionary containing metadata of collections.
                backupdir: Path to the backup directory.
                kvstore_collections_global_records_to_be_restored: Global counter for total records to be restored.
                kvstore_collections_global_records_restored: Global counter for successfully restored records.
                kvstore_collections_restored_warning: List of collections with restoration warnings.
                restore_results_dict: Dictionary to store restoration results.
                kvstore_collections_clean_empty: Boolean indicating whether to clean the collection if it was empty in the backup.

            Returns:
                A tuple containing updated global counters, as well as summary dict describing the operation:
                - kvstore_collections_global_records_to_be_restored
                - kvstore_collections_global_records_restored
                - kvstore_collection_restore_summary_dict
            """

            # Check if collection exists in restore dict
            if collection_name not in collections_restore_dict:
                error_msg = f'Collection "{collection_name}" not found in backup data. This may be due to the collection being excluded from backups (e.g., stateful charts collections).'
                logger.warning(error_msg)
                raise KeyError(error_msg)

            # Get collection metadata
            source_json_file = collections_restore_dict[collection_name]["file"]
            source_json_file_size = collections_restore_dict[collection_name]["size"]
            source_json_file_records = collections_restore_dict[collection_name][
                "records"
            ]

            # init dict summary
            kvstore_collection_restore_summary_dict = {
                "collection_name": collection_name,
                "source_json_file": source_json_file,
                "source_json_file_size": source_json_file_size,
                "source_json_file_records": source_json_file_records,
            }

            # Check if this is a stateful charts collection that should be excluded
            if collection_name.startswith(
                "kv_trackme_stateful_alerting_charts_tenant_"
            ):
                info_msg = f'TrackMe restore kvstore records process, skipping restoration of stateful charts collection="{collection_name}" on purpose. KVstore records for stateful charts collections are not restored.'
                logger.info(info_msg)
                tasks_list.append(info_msg)

                # Add results to restore_results_dict with 0 restored records
                restore_results_dict[collection_name] = {
                    "source_json_file": source_json_file,
                    "source_json_file_size": source_json_file_size,
                    "source_json_file_records": source_json_file_records,
                    "restored_records": 0,
                    "skipped": True,
                    "skip_reason": "Stateful charts collection - excluded on purpose",
                }

                # Return with unchanged global counters since we're not restoring any records
                kvstore_collection_restore_summary_dict["restored_records"] = 0
                kvstore_collection_restore_summary_dict["skipped"] = True
                kvstore_collection_restore_summary_dict["skip_reason"] = (
                    "Stateful charts collection - excluded on purpose"
                )

                return (
                    kvstore_collections_global_records_to_be_restored,
                    kvstore_collections_global_records_restored,
                    kvstore_collection_restore_summary_dict,
                )

            # Log start of the process
            logger.info(
                f'TrackMe restore kvstore records process starting, processing restore of collection_name="{collection_name}", '
                f'source_json_file="{source_json_file}", source_json_file_size="{source_json_file_size}", '
                f'source_json_file_records="{source_json_file_records}"'
            )

            # Connect to the KVstore collection
            try:
                collection = service.kvstore[collection_name]
            except Exception as e:
                error_nsg = f'TrackMe restore kvstore records process, failed to connect to KVstore collection="{collection_name}" with exception="{str(e)}"'
                logger.error(error_nsg)
                kvstore_collections_restored_warning.append(collection_name)
                errors_list.append(error_nsg)
                return (
                    kvstore_collections_global_records_to_be_restored,
                    kvstore_collections_global_records_restored,
                    kvstore_collection_restore_summary_dict,
                )

            # Handle kvstore_collections_clean_empty
            if kvstore_collections_clean_empty and source_json_file_records == 0:
                try:
                    collection.data.delete()  # Delete all records in the collection
                    info_nsg = f'TrackMe restore kvstore records process, collection "{collection_name}" was empty in the backup. Existing records have been cleared.'
                    logger.info(info_nsg)
                    tasks_list.append(info_nsg)
                except Exception as e:
                    error_msg = f'TrackMe restore kvstore records process, failed to clear records in collection="{collection_name}" with exception="{str(e)}"'
                    logger.error(error_msg)
                    errors_list.append(error_msg)
                    kvstore_collections_restored_warning.append(collection_name)
                    return (
                        kvstore_collections_global_records_to_be_restored,
                        kvstore_collections_global_records_restored,
                        kvstore_collection_restore_summary_dict,
                    )

            # Load the JSON data
            counter = 0
            try:
                with open(os.path.join(backupdir, source_json_file), "r") as f:
                    data = json.load(f)

                # Update global counters
                kvstore_collections_global_records_to_be_restored += len(data)

                # Process data in chunks
                chunks = [data[i : i + 500] for i in range(0, len(data), 500)]
                for chunk in chunks:
                    try:
                        collection.data.batch_save(*chunk)
                        kvstore_collections_global_records_restored += len(chunk)
                        counter += len(chunk)
                    except Exception as e:
                        error_msg = (
                            f'TrackMe restore kvstore records process, failed to restore records in collection="{collection_name}" with exception="{str(e)}"'
                            f'collection_name="{collection_name}", '
                            f'source_json_file="{source_json_file}", source_json_file_size="{source_json_file_size}", '
                        )
                        logger.error(error_msg)
                        errors_list.append(error_msg)
                        kvstore_collections_restored_warning.append(collection_name)

                # Add results to restore_results_dict
                restore_results_dict[collection_name] = {
                    "source_json_file": source_json_file,
                    "source_json_file_size": source_json_file_size,
                    "source_json_file_records": source_json_file_records,
                    "restored_records": counter,
                }

                # Log success or mismatch in record counts
                if int(source_json_file_records) != int(counter):
                    info_msg = (
                        f"TrackMe restore kvstore records process finished but could not verify the number of restored records "
                        f'to equal the number of records in the source file, collection_name="{collection_name}", '
                        f'source_json_file="{source_json_file}", source_json_file_size="{source_json_file_size}", '
                        f'source_json_file_records="{source_json_file_records}", restored_records="{counter}"'
                    )
                    logger.error(info_msg)
                    errors_list.append(info_msg)
                else:

                    info_msg = (
                        f'TrackMe restore kvstore records process finished successfully, collection_name="{collection_name}", '
                        f'source_json_file="{source_json_file}", source_json_file_size="{source_json_file_size}", '
                        f'source_json_file_records="{source_json_file_records}", restored_records="{counter}"'
                    )
                    logger.info(info_msg)
                    tasks_list.append(info_msg)

            except Exception as e:
                restore_results_dict[collection_name] = {
                    "source_json_file": source_json_file,
                    "source_json_file_size": source_json_file_size,
                    "source_json_file_records": source_json_file_records,
                    "restored_records": 0,
                    "exception": f'TrackMe restore kvstore records process, failed to open JSON file="{os.path.join(backupdir, source_json_file)}" for reading with exception="{str(e)}"',
                }
                kvstore_collections_restored_warning.append(collection_name)
                error_msg = (
                    f'TrackMe restore kvstore records process, filed to open JSON file="{os.path.join(backupdir, source_json_file)}" for reading with exception="{str(e)}"'
                    f'collection_name="{collection_name}", '
                    f'source_json_file="{source_json_file}", source_json_file_size="{source_json_file_size}", '
                    f'source_json_file_records="{source_json_file_records}", restored_records="{counter}"'
                )
                logger.error(error_msg)
                errors_list.append(error_msg)

            # add counters to the summary dict
            kvstore_collection_restore_summary_dict["restored_records"] = counter

            return (
                kvstore_collections_global_records_to_be_restored,
                kvstore_collections_global_records_restored,
                kvstore_collection_restore_summary_dict,
            )

        describe = False

        # init
        dry_run = True

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            if not describe:
                # Get the mandatory backup_archive name
                try:
                    backup_archive = resp_dict["backup_archive"]
                except Exception as e:
                    response = {
                        "response": "The backup_archive name is required in the body of the request"
                    }
                    logger.error(json.dumps(response, indent=2))
                    return {"payload": response, "status": 500}

                # Dry run mode, preview and verify the restore operation, but do not do anything
                # default to True
                try:
                    dry_run = resp_dict["dry_run"]
                    if dry_run in ("True", "true", "1", 1):
                        dry_run = True
                    elif dry_run in ("False", "false", "0", 0):
                        dry_run = False
                    else:
                        return {
                            "payload": {
                                "response": f'Invalid value for dry_run, received="{dry_run}", expecting a boolean value or true / false'
                            },
                            "status": 500,
                        }

                except Exception as e:
                    dry_run = True

                #
                # restore_knowledge_objects (False/True)
                #

                try:
                    restore_knowledge_objects = resp_dict["restore_knowledge_objects"]
                    if restore_knowledge_objects in ("True", "true", "1", 1):
                        restore_knowledge_objects = True
                    else:
                        restore_knowledge_objects = False
                except Exception as e:
                    restore_knowledge_objects = True

                #
                # knowledge_objects_tenants_scope (all / comma separated list of tenant_id)
                #

                try:
                    knowledge_objects_tenants_scope = resp_dict[
                        "knowledge_objects_tenants_scope"
                    ]
                    if knowledge_objects_tenants_scope == "all":
                        knowledge_objects_tenants_scope = "all"
                    else:
                        if not isinstance(knowledge_objects_tenants_scope, list):
                            knowledge_objects_tenants_scope = (
                                knowledge_objects_tenants_scope.split(",")
                            )
                        # for each entry in the list, strip and remove any empty char before or after the list entry
                        knowledge_objects_tenants_scope = [
                            x.strip() for x in knowledge_objects_tenants_scope
                        ]
                except Exception as e:
                    knowledge_objects_tenants_scope = "all"

                #
                # knowledge_objects_lists (all / comma separated list of objects to be restored)
                #

                try:
                    knowledge_objects_lists = resp_dict["knowledge_objects_lists"]
                    if knowledge_objects_lists == "all":
                        knowledge_objects_lists = "all"
                    else:
                        if not isinstance(knowledge_objects_lists, list):
                            knowledge_objects_lists = knowledge_objects_lists.split(",")
                        # for each entry in the list, strip and remove any empty char before or after the list entry
                        knowledge_objects_lists = [
                            x.strip() for x in knowledge_objects_lists
                        ]
                except Exception as e:
                    knowledge_objects_lists = "all"

                #
                # knowledge_objects_replace_existing (False/True) - If the object already exists, it will be replaced
                #

                try:
                    knowledge_objects_replace_existing = resp_dict[
                        "knowledge_objects_replace_existing"
                    ]
                    if knowledge_objects_replace_existing in ("True", "true", "1", 1):
                        knowledge_objects_replace_existing = True
                    else:
                        knowledge_objects_replace_existing = False
                except Exception as e:
                    knowledge_objects_replace_existing = True

                #
                # restore_kvstore_collections (False/True)
                #

                try:
                    restore_kvstore_collections = resp_dict[
                        "restore_kvstore_collections"
                    ]
                    if restore_kvstore_collections in ("True", "true", "1", 1):
                        restore_kvstore_collections = True
                    else:
                        restore_kvstore_collections = False
                except Exception as e:
                    restore_kvstore_collections = True

                #
                # kvstore_collections_scope (all / comma separated list of KVstore collections)
                #

                try:
                    kvstore_collections_scope = resp_dict["kvstore_collections_scope"]
                    if kvstore_collections_scope == "all":
                        kvstore_collections_scope = "all"
                    else:
                        if not isinstance(kvstore_collections_scope, list):
                            kvstore_collections_scope = kvstore_collections_scope.split(
                                ","
                            )
                        # for each entry in the list, strip and remove any empty char before or after the list entry
                        kvstore_collections_scope = [
                            x.strip() for x in kvstore_collections_scope
                        ]
                except Exception as e:
                    kvstore_collections_scope = "all"

                #
                # kvstore_collections_clean_empty (False/True) - If the collection was empty in the backup, restoring will empty any existing record in the collection
                #

                try:
                    kvstore_collections_clean_empty = resp_dict[
                        "kvstore_collections_clean_empty"
                    ]
                    if kvstore_collections_clean_empty in ("True", "true", "1", 1):
                        kvstore_collections_clean_empty = True
                    else:
                        kvstore_collections_clean_empty = False
                except Exception as e:
                    kvstore_collections_clean_empty = True

                #
                # kvstore_collections_restore_non_tenants_collections (False/True) - Restore non-tenants collections
                #

                try:
                    kvstore_collections_restore_non_tenants_collections = resp_dict[
                        "kvstore_collections_restore_non_tenants_collections"
                    ]
                    if kvstore_collections_restore_non_tenants_collections in (
                        "True",
                        "true",
                        "1",
                        1,
                    ):
                        kvstore_collections_restore_non_tenants_collections = True
                    else:
                        kvstore_collections_restore_non_tenants_collections = False
                except Exception as e:
                    kvstore_collections_restore_non_tenants_collections = True

                #
                # restore_virtual_tenant_accounts (False/True) - Restore virtual tenant accounts
                #

                try:
                    restore_virtual_tenant_accounts = resp_dict[
                        "restore_virtual_tenant_accounts"
                    ]
                    if restore_virtual_tenant_accounts in ("True", "true", "1", 1):
                        restore_virtual_tenant_accounts = True
                    else:
                        restore_virtual_tenant_accounts = False

                except Exception as e:
                    restore_virtual_tenant_accounts = True

                #
                # restore_virtual_tenant_main_kvrecord (False/True) - Restore virtual tenant main record
                #

                try:
                    restore_virtual_tenant_main_kvrecord = resp_dict[
                        "restore_virtual_tenant_main_kvrecord"
                    ]
                    if restore_virtual_tenant_main_kvrecord in ("True", "true", "1", 1):
                        restore_virtual_tenant_main_kvrecord = True
                    else:
                        restore_virtual_tenant_main_kvrecord = False

                except Exception as e:
                    restore_virtual_tenant_main_kvrecord = True

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be used to restore TrackMe Knowledge Objects and/or KVstore collections from a backup archive, the archive must be a tarball compressed made available in the backup directory of the application. ($SPLUNK_HOME/etc/apps/trackme/backup) - Restore operations for both knowledge objects and KVstore records are fully based on Splunk API usage, therefore compatible indifferently with Splunk Enterprise and Splunk Cloud, in a standalone or Search Head Cluster context. It requires a POST call with the following arguments:",
                "resource_desc": "Restore TrackMe Knowledge Objects and/or KVstore collections from a backup archive",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/backup_and_restore/restore\" mode=\"post\" body=\"{'backup_archive': 'trackme-backup-20210205-142635.tgz', 'dry_run': 'false', 'target': 'all'}\"",
                "options": [
                    {
                        "dry_run": "(true / false) OPTIONAL: if true, the endpoint will only verify that the archive can be found and successfully extracted, there will be no modifications at all. (default to true)",
                        "restore_virtual_tenant_accounts": "(true / false) OPTIONAL: check and restore the virtual tenant accounts from the submitted archive (default to true)",
                        "restore_virtual_tenant_main_kvrecord": "(true / false) OPTIONAL: check and restore the virtual tenant main record from the submitted archive (default to true)",
                        "restore_knowledge_objects": "(true / false) OPTIONAL: restore the knowledge objects from the submitted archive (default to true) - This requires a backup archive generated with TrackMe 2.1.5 or later",
                        "knowledge_objects_tenants_scope": "(all / comma separated list of tenant_id) OPTIONAL: restore the knowledge objects for all tenants or provide a comma separated list of tenant_id to be restored (defaults to all) - This requires a backup archive generated with TrackMe 2.1.5 or later",
                        "knowledge_objects_lists": "(all / comma separated list of objects to be restored) OPTIONAL: restore all knowledge objects that were backed up in the submitted archive, or provide a comma separated list of objects to be restored (defaults to all) - This requires a backup archive generated with TrackMe 2.1.5 or later",
                        "knowledge_objects_replace_existing": "(true / false) OPTIONAL: if the object already exists, it will be replaced (default to true) - This requires a backup archive generated with TrackMe 2.1.5 or later",
                        "restore_kvstore_collections": "(true / false) OPTIONAL: restore the KVstore collections from the submitted archive (default to true)",
                        "kvstore_collections_scope": "(all / comma separated list of KVstore collections) OPTIONAL: restore all collections that were backed up in the submitted archive, or provide a comma separated list of collections to be restored (defaults to all)",
                        "kvstore_collections_clean_empty": "(true / false) OPTIONAL: if the collection was empty in the backup, restoring will empty any existing record in the collection (default to true)",
                        "kvstore_collections_restore_non_tenants_collections": "(true / false) OPTIONAL: restore non-tenants collections (default to true), non tenants collections are KVstore collections which are tenant specific, such as user preferences, user settings, etc. If set to True, the content of the collection content will be restored from the backup.",
                        "backup_archive": "The archive file to be restored, the tarball compressed file must be located in the backup directory of the trackMe application.",
                        "knowledge_objects_blocklist": "(comma separated list or native list) OPTIONAL: list of knowledge objects that should not be restored",
                        "kvstore_collections_blocklist": "(comma separated list or native list) OPTIONAL: list of KVstore collections that should not be restored",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        else:
            # Get splunkd port
            splunkd_port = request_info.server_rest_port

            # Get service
            service = client.connect(
                owner="nobody",
                app="trackme",
                port=splunkd_port,
                token=request_info.system_authtoken,
                timeout=600,
            )

            # header
            header = {
                "Authorization": "Splunk %s" % request_info.session_key,
                "Content-Type": "application/json",
            }

            # set loglevel
            loglevel = trackme_getloglevel(
                request_info.system_authtoken, request_info.server_rest_port
            )
            logger.setLevel(loglevel)

            # Set backup root dir
            backuproot = os.path.join(splunkhome, "etc", "apps", "trackme", "backup")

            # Set the submitted full path of the archive file
            backupfile = os.path.join(backuproot, backup_archive)

            # Set the full path of the directory for the extraction
            backupdir = os.path.splitext(backupfile)[0]

            # A dict for kvstore records summary
            kvstore_collections_restore_summary_dict = {}

            #
            # knowledge_objects_blocklist (comma separated list or native list)
            #

            try:
                knowledge_objects_blocklist = resp_dict["knowledge_objects_blocklist"]
                if not isinstance(knowledge_objects_blocklist, list):
                    knowledge_objects_blocklist = knowledge_objects_blocklist.split(",")
                # for each entry in the list, strip and remove any empty char before or after the list entry
                knowledge_objects_blocklist = [
                    x.strip() for x in knowledge_objects_blocklist
                ]
            except Exception as e:
                knowledge_objects_blocklist = []

            #
            # kvstore_collections_blocklist (comma separated list or native list)
            #

            try:
                kvstore_collections_blocklist = resp_dict[
                    "kvstore_collections_blocklist"
                ]
                if not isinstance(kvstore_collections_blocklist, list):
                    kvstore_collections_blocklist = kvstore_collections_blocklist.split(
                        ","
                    )
                # for each entry in the list, strip and remove any empty char before or after the list entry
                kvstore_collections_blocklist = [
                    x.strip() for x in kvstore_collections_blocklist
                ]
            except Exception as e:
                kvstore_collections_blocklist = []

            #################
            # Task: log start
            #################

            logger.info(
                f'TrackMe restore process is starting requested by user="{request_info.user}", request="{json.dumps(resp_dict, indent=3)}"'
            )

            # log the value for each argument
            logger.info(
                f"TrackMe restore process, listing arguments summary, "
                f'backup_archive="{backup_archive}", dry_run="{dry_run}", '
                f'restore_virtual_tenant_accounts="{restore_virtual_tenant_accounts}", '
                f'restore_virtual_tenant_main_kvrecord="{restore_virtual_tenant_main_kvrecord}", '
                f'restore_knowledge_objects="{restore_knowledge_objects}", '
                f'knowledge_objects_tenants_scope="{knowledge_objects_tenants_scope}", knowledge_objects_lists="{knowledge_objects_lists}", '
                f'knowledge_objects_replace_existing="{knowledge_objects_replace_existing}", restore_kvstore_collections="{restore_kvstore_collections}", '
                f'kvstore_collections_scope="{kvstore_collections_scope}", kvstore_collections_clean_empty="{kvstore_collections_clean_empty}", '
                f'kvstore_collections_restore_non_tenants_collections="{kvstore_collections_restore_non_tenants_collections}", '
                f'knowledge_objects_blocklist="{knowledge_objects_blocklist}", kvstore_collections_blocklist="{kvstore_collections_blocklist}"'
            )

            ################################################
            # Task: Identify if archive is available locally
            ################################################

            # Assume False
            archive_exists_locally = False

            if os.path.isfile(backupfile):
                archive_exists_locally = True

            if not archive_exists_locally:

                #
                # Export
                #

                # check if we have a KVstore record for this backup archive
                collection_name = "kv_trackme_backup_archives_info"
                collection = service.kvstore[collection_name]

                query_string = {"backup_archive": {"$regex": f".*{backup_archive}$"}}

                try:
                    kvrecords = collection.data.query(query=json.dumps(query_string))
                    kvrecord = kvrecords[0]
                except Exception as e:
                    kvrecords = []
                    kvrecord = None

                # if we do not have a KVrecord, stop and raise a failure, the user needs to export/import the backup archive manually
                if not kvrecord:
                    response = {
                        "response": f"The archive name {backup_archive} could not be found in the KVstore, restore cannot be processed. You can manually export/import the backup archive through the user interface.",
                    }
                    logger.error(json.dumps(response, indent=2))
                    return {"payload": response, "status": 500}

                #
                # remote host handling
                #

                backup_server_name = kvrecord.get("server_name")

                # Query the KVstore collection to get the server_name for this backup archive
                logger.info(
                    f"Backup archive is on different server ({backup_server_name}), delegating export to target server"
                )

                # Determine the best target server name for communication by testing connectivity
                target_server_name = backup_server_name

                # Test connectivity with short hostname first, then FQDN if needed
                logger.info(
                    f"Testing connectivity with short hostname: {backup_server_name}"
                )
                if not test_splunkd_connectivity(
                    backup_server_name,
                    request_info.server_rest_port,
                    request_info.session_key,
                ):
                    # Try with FQDN
                    backup_server_fqdn = (
                        socket.getfqdn()
                        if backup_server_name == socket.gethostname()
                        else f"{backup_server_name}.{socket.getfqdn().split('.', 1)[1] if '.' in socket.getfqdn() else 'local'}"
                    )
                    logger.info(
                        f"Short hostname failed, trying FQDN: {backup_server_fqdn}"
                    )
                    if test_splunkd_connectivity(
                        backup_server_fqdn,
                        request_info.server_rest_port,
                        request_info.session_key,
                    ):
                        target_server_name = backup_server_fqdn
                        logger.info(
                            f"FQDN connectivity successful, using: {target_server_name}"
                        )
                    else:
                        logger.warning(
                            f"Both short hostname and FQDN failed for {backup_server_name}"
                        )
                else:
                    logger.info(
                        f"Short hostname connectivity successful, using: {target_server_name}"
                    )

                # support only https
                target_server_uri = (
                    f"https://{target_server_name}:{request_info.server_rest_port}"
                )

                # Make REST call to target server
                headers = {
                    "Authorization": f"Splunk {request_info.session_key}",
                    "Content-Type": "application/json",
                }

                # Prepare the request payload
                request_payload = {
                    "archive_name": backup_archive,
                    "force_local": True,
                }

                target_url = f"{target_server_uri}/services/trackme/v2/backup_and_restore/export_backup"

                logger.info(f"Making REST call to target server: {target_url}")

                try:
                    response = requests.post(
                        target_url,
                        headers=headers,
                        data=json.dumps(request_payload),
                        verify=False,
                        timeout=600,
                    )

                    if response.status_code == 200:
                        response_data = response.json()
                        logger.info(
                            f"Successfully exported backup from target server {backup_server_name}"
                        )
                        base64_data = response_data.get("archive_base64")

                    else:
                        logger.error(
                            f"Target server {backup_server_name} returned error: {response.status_code} - {response.text}, url={target_url}, request_payload={json.dumps(request_payload)}"
                        )
                        return {
                            "payload": {
                                "error": f"Failed to export from target server {backup_server_name}: {response.text}, url={target_url}, request_payload={json.dumps(request_payload)}"
                            },
                            "status": response.status_code,
                        }

                except requests.exceptions.ConnectionError as e:
                    logger.error(
                        f"Connection error to target server {backup_server_name}: {str(e)}"
                    )
                    return {
                        "payload": {
                            "error": f"Failed to connect to target server {backup_server_name}. Please ensure the server is reachable and TrackMe is installed."
                        },
                        "status": 500,
                    }
                except Exception as e:
                    logger.error(
                        f"Error making REST call to target server {backup_server_name}: {str(e)}, url={target_url}, request_payload={json.dumps(request_payload)}"
                    )
                    return {
                        "payload": {
                            "error": f"Failed to export from target server {backup_server_name}: {str(e)}, url={target_url}, request_payload={json.dumps(request_payload)}"
                        },
                        "status": 500,
                    }

                #
                # Import
                #

                # url
                url = f"{request_info.server_rest_uri}/services/trackme/v2/backup_and_restore/import_backup"
                header = {
                    "Authorization": "Splunk %s" % request_info.session_key,
                    "Content-Type": "application/json",
                }
                response = requests.post(
                    url,
                    headers=header,
                    json={"archive_base64": base64_data},
                    verify=False,
                    timeout=600,
                )

                try:
                    if response.status_code != 200:
                        response = {
                            "response": f"The archive name {backup_archive} could not be imported, restore cannot be processed. You can manually import the backup archive through the user interface.",
                        }
                        logger.error(json.dumps(response, indent=2))
                        return {"payload": response, "status": 500}
                    else:
                        # Archive is now imported and extracted, mark as available locally
                        archive_exists_locally = True
                        logger.info(f"Successfully imported archive {backup_archive} from remote server")
                except Exception as e:
                    response = {
                        "response": f"The archive name {backup_archive} could not be imported, restore cannot be processed. You can manually import the backup archive through the user interface.",
                    }
                    logger.error(json.dumps(response, indent=2))
                    return {"payload": response, "status": 500}

                 # The archive is now imported and available locally, we simply can continue with the local restore process

            ##########################
            # Task: Archive Extraction
            ##########################

            # First, check the backup archive existence (skip if already imported from remote)
            if not archive_exists_locally and not os.path.isfile(backupfile):
                response = {
                    "response": f"The archive name {backupfile} could not be found on the file-system, restore cannot be processed",
                }
                logger.error(json.dumps(response, indent=2))
                return {"payload": response, "status": 500}

            # Attempt extraction using the extract_archive function that supports both .tgz and .tar.zst
            if not extract_archive(backupfile, backupdir):
                response = {
                    "response": f"The archive name {backupfile} could not be extracted, restore cannot be processed, exception='file could not be opened successfully'",
                }
                logger.error(json.dumps(response, indent=2))
                return {"payload": response, "status": 500}

            ######################
            # Task: Check metadata
            ######################

            # default to 2.0.0 (metadata compatibility, generated with TrackMe >= 2.1.5)
            archive_schema_version = "2.0.0"

            # check metadata files
            full_metadata_file = f"{backupfile}.full.meta"
            light_metadata_file = f"{backupfile}.light.meta"

            # From TrackMe 2.1.5, we generate a metadata file named as <tarfile>.full.meta and <tarfile>.light.meta
            # Check if the metadata file exists
            backup_file_has_metadata = False

            # check for both metadata files

            if os.path.isfile(full_metadata_file) and os.path.isfile(
                light_metadata_file
            ):

                backup_file_has_metadata = True

                # full metadata
                try:
                    with open(f"{backupfile}.full.meta", "r") as read_content:
                        full_metadata = json.load(read_content)
                except Exception as e:
                    full_metadata = None
                    backup_file_has_metadata = False
                    logger.error(
                        f'failed to load the full metadata file="{backupfile}.full.meta" with exception="{str(e)}"'
                    )

                # light metadata
                try:
                    with open(f"{backupfile}.light.meta", "r") as read_content:
                        light_metadata = json.load(read_content)
                except Exception as e:
                    light_metadata = None
                    backup_file_has_metadata = False
                    logger.error(
                        f'failed to load the light metadata file="{backupfile}.light.meta" with exception="{str(e)}"'
                    )

                # if we do not have full metadata or light metadata, the archive schema version is 1.0.0 (Only KVstore collections can be restored)
                if not full_metadata or not light_metadata:
                    archive_schema_version = "1.0.0"

            ############################################
            # Task: Parse restorable KVstore collections
            ############################################

            # store the list of KVstore collection json files in a list
            collections_json_files = [
                f
                for f in listdir(backupdir)
                if isfile(join(backupdir, f)) and f.startswith("kv_")
            ]

            # store the list of available collections in the archive
            collections_available = []

            # create a dictionnary
            collections_restore_dict = {}
            for json_file in collections_json_files:
                # strip the extension
                collection_name = os.path.splitext(json_file)[0]

                # append to the list of available collections for restore
                collections_available.append(collection_name)

                # get the file size
                json_file_size = os.path.getsize(os.path.join(backupdir, json_file))

                # get the file mtime
                json_file_mtime = round(
                    os.path.getmtime(os.path.join(backupdir, json_file))
                )

                # try getting the number of records
                try:
                    with open(os.path.join(backupdir, json_file), "r") as read_content:
                        json_file_records = len(json.load(read_content))
                except Exception as e:
                    json_file_records = None

                # add to the dict
                is_empty = False
                if json_file_records == 0:
                    is_empty = True

                collections_restore_dict[collection_name] = {
                    "file": json_file,
                    "size": json_file_size,
                    "mtime": json_file_mtime,
                    "records": json_file_records,
                    "is_empty": is_empty,
                }

            ##############
            # dry_run mode
            ##############

            # if dry run
            if dry_run:

                # init dry_run response
                dry_run_response = {}

                # add response
                response_message = f"Success, the archive {backupfile} could be successfully extracted, consult metadata information added to this response for more insights. You can run this command with dry_run=false to perform the restoration."
                dry_run_response["response"] = response_message

                # if metadata files are available, add the light_metadata as metadata
                if backup_file_has_metadata:
                    dry_run_response["knowledge_objects_summary"] = light_metadata.get(
                        "knowledge_objects_summary"
                    )
                    dry_run_response["metadata"] = light_metadata

                # add collections
                dry_run_response["kvstore_collections_json_files"] = (
                    collections_json_files
                )
                dry_run_response["kvstore_collections_details"] = (
                    collections_restore_dict
                )

                # iterate through knowledge objects files and add to the response_message, as well as the virtual tenant account
                knowledge_objects_json_files = [
                    f
                    for f in listdir(backupdir)
                    if isfile(join(backupdir, f))
                    and f.startswith("tenant_")
                    and f.endswith("_knowledge_objects.json")
                ]

                for knowledge_objects_json_file in knowledge_objects_json_files:

                    # extract the tenant_id
                    tenant_id = knowledge_objects_json_file.split("_")[1]

                    if (
                        knowledge_objects_tenants_scope == "all"
                        or tenant_id in knowledge_objects_tenants_scope
                    ):

                        # open and load the json file
                        try:
                            with open(
                                os.path.join(backupdir, knowledge_objects_json_file),
                                "r",
                            ) as read_content:
                                knowledge_objects_json = json.load(read_content)
                        except Exception as e:
                            clean_backup_dir(backupdir)
                            knowledge_objects_json = None
                            logger.error(
                                f'failed to load the knowledge_objects_json file="{knowledge_objects_json_file}" with exception="{str(e)}"'
                            )

                        # iterate through the knowledge objects lists, add each category of objects to the response
                        # categories: savedsearches, alerts, macros, lookup_definitions, kvstore_collections
                        savedsearches_list = []
                        alerts_list = []
                        macros_list = []
                        lookup_definitions_list = []
                        kvstore_collections_list = []

                        # Iterate through the values of the dictionary
                        for ko in knowledge_objects_json.values():
                            type_ko = ko.get("type")
                            if type_ko == "savedsearches":
                                savedsearches_list.append(ko)
                            elif type_ko == "alerts":
                                alerts_list.append(ko)
                            elif type_ko == "macros":
                                macros_list.append(ko)
                            elif type_ko == "lookup_definitions":
                                lookup_definitions_list.append(ko)
                            elif type_ko == "kvstore_collections":
                                kvstore_collections_list.append(ko)

                        # Add to the response
                        dry_run_response[f"tenant_{tenant_id}_knowledge_objects"] = {
                            "savedsearches": savedsearches_list,
                            "alerts": alerts_list,
                            "macros": macros_list,
                            "lookup_definitions": lookup_definitions_list,
                            "kvstore_collections": kvstore_collections_list,
                        }

                        # check that the virtual account backup file exists, if not log and skip
                        tenant_json_file = f"tenant_{tenant_id}_vtenant_account.json"

                        if not os.path.isfile(
                            os.path.join(backupdir, tenant_json_file)
                        ):
                            clean_backup_dir(backupdir)
                            error_message = f'failed to find the backup file="{tenant_json_file}" for tenant_id="{tenant_id}", the tenant cannot be restored'
                            logger.error(error_message)
                            return {
                                "payload": {
                                    "response": error_message,
                                },
                                "status": 500,
                            }

                        # load the tenant json file
                        try:
                            f = open(os.path.join(backupdir, tenant_json_file), "r")
                            vtenant_account_data = json.loads(f.read())
                            info_message = f'loaded the tenant backup file="{tenant_json_file}" for tenant_id="{tenant_id}"'
                            logger.info(info_message)

                        except Exception as e:
                            clean_backup_dir(backupdir)
                            error_message = f'failed to open json file="{os.path.join(backupdir, tenant_json_file)}" for reading with exception="{str(e)}", cannot restore tenant_id="{tenant_id}"'
                            logger.error(error_message)
                            return {
                                "payload": {
                                    "response": error_message,
                                },
                                "status": 500,
                            }

                        # add to the response
                        dry_run_response[f"tenant_{tenant_id}_vtenant_account"] = (
                            vtenant_account_data
                        )

                # log and render

                # remove backup dir
                clean_backup_dir(backupdir)
                logger.info(response_message)
                return {"payload": dry_run_response, "status": 200}

            else:

                ###################
                # Live restore mode
                ###################

                # create a dict to store the restore results
                restore_results_dict = {}

                # create global counters
                kvstore_collections_global_records_to_be_restored = 0
                kvstore_collections_global_records_restored = 0

                # store collections with failures in a specific list
                kvstore_collections_restored_warning = []

                # Things are serious now, let's restore collection per collection

                # set the list of collections to be restored
                kvstore_collections_to_be_restored = []

                # set the list of collections that were restored
                kvstore_collections_restored = []

                # if kvstore_collections_scope is all, add every collection that can be restored
                if kvstore_collections_scope == "all":
                    # Loop through the collections
                    for collection_name in collections_restore_dict:
                        # Skip stateful charts collections as they are excluded from backups
                        if collection_name.startswith(
                            "kv_trackme_stateful_alerting_charts_tenant_"
                        ):
                            info_msg = f'TrackMe restore kvstore collections process, skipping stateful charts collection="{collection_name}" on purpose. Stateful charts collections are excluded from backups and restores.'
                            logger.info(info_msg)
                            continue
                        # Only add if not in blocklist
                        if collection_name not in kvstore_collections_blocklist:
                            kvstore_collections_to_be_restored.append(collection_name)

                else:
                    # Loop and check if the requested collection is available for restore and not in blocklist
                    for collection_name in kvstore_collections_scope:
                        # Skip stateful charts collections as they are excluded from backups
                        if collection_name.startswith(
                            "kv_trackme_stateful_alerting_charts_tenant_"
                        ):
                            info_msg = f'TrackMe restore kvstore collections process, skipping stateful charts collection="{collection_name}" on purpose. Stateful charts collections are excluded from backups and restores.'
                            logger.info(info_msg)
                            continue

                        if not collection_name in collections_available:
                            # an impossible operaton is requested, we cannot proceed
                            response = {
                                "action": "failure",
                                "response": f'the collection="{collection_name}" requested for restoration is not available in the backup archive file, restore cannot be procceded',
                                "collections_available": collections_available,
                            }

                            return {"payload": response, "status": 500}

                        elif collection_name not in kvstore_collections_blocklist:
                            kvstore_collections_to_be_restored.append(collection_name)

                #########
                # Proceed
                #########

                # init final response_dict
                response_dict = {}

                # init tasks list
                tasks_list = []

                # init errors list
                errors_list = []

                #####################################
                # Task: Restore the Knowledge Objects
                #####################################

                if archive_schema_version == "2.0.0" and restore_knowledge_objects:
                    logger.info(
                        f'Detected archive_schema_version="{archive_schema_version}", Processing restore operations of Knowledge Objects'
                    )

                    # knowledge_objects_json_files: list of json files in the backup archive
                    # each file is named as:
                    # tenant_<tenant_id>_knowledge_objects.json

                    tenants_id_restorable = []
                    tenants_id_available = []

                    knowledge_objects_json_files = [
                        f
                        for f in listdir(backupdir)
                        if isfile(join(backupdir, f))
                        and f.startswith("tenant_")
                        and f.endswith("_knowledge_objects.json")
                    ]

                    for knowledge_objects_json_file in knowledge_objects_json_files:
                        # extract the tenant_id
                        tenant_id = knowledge_objects_json_file.split("_")[1]

                        # add to available
                        tenants_id_available.append(tenant_id)

                        if knowledge_objects_tenants_scope == "all":
                            tenants_id_restorable.append(tenant_id)
                        else:
                            if tenant_id in knowledge_objects_tenants_scope:
                                tenants_id_restorable.append(tenant_id)

                    # if a specific tenant_id was requested, but not found in the backup archive, stop and return an error
                    if knowledge_objects_tenants_scope != "all":
                        for tenant_id in knowledge_objects_tenants_scope:
                            if tenant_id not in tenants_id_restorable:
                                error_message = f'failed to find the backup file for tenant_id="{tenant_id}", the tenant cannot be restored, verify that your input is valid.'
                                logger.error(error_message)
                                return {
                                    "payload": {
                                        "response": error_message,
                                        "tenants_id_available": tenants_id_available,
                                    },
                                    "status": 500,
                                }
                            # only keep in tenants_id_restorable the list of tenants provided in knowledge_objects_tenants_scope
                            tenants_id_restorable = [
                                tenant_id
                                for tenant_id in tenants_id_restorable
                                if tenant_id in knowledge_objects_tenants_scope
                            ]

                    ###############################################
                    # subtask: load the central KVstore from backup
                    ###############################################

                    collection_name_main = "kv_trackme_virtual_tenants"

                    # connect to live KVstore collection
                    try:
                        collection = service.kvstore[collection_name_main]
                    except Exception as e:
                        error_message = f'failed to connect to KVstore collection="{collection_name_main}" with exception="{str(e)}"'
                        logger.error(error_message)
                        return {
                            "payload": {
                                "response": error_message,
                            },
                            "status": 500,
                        }

                    # backup file name
                    source_json_file = collections_restore_dict[collection_name_main][
                        "file"
                    ]

                    # try loading the json data
                    try:
                        f = open(os.path.join(backupdir, source_json_file), "r")
                        vtenants_main_records = json.loads(f.read())

                    except Exception as e:
                        error_message = f'failed to open json file="{os.path.join(backupdir, source_json_file)}" for reading with exception="{str(e)}"'
                        logger.error(error_message)
                        return {
                            "payload": {
                                "response": error_message,
                            },
                            "status": 500,
                        }

                    ###################################################
                    # subtask: restore per tenant the knowledge objects
                    ###################################################

                    # Loop through the restorable tenants, and proceed
                    for tenant_id in tenants_id_restorable:
                        logger.info(
                            f'Processing with restore operations for tenant_id="{tenant_id}"'
                        )

                        #############################################
                        # subtask: restore the Virtual Tenant account
                        #############################################

                        # check that the virtual account backup file exists, if not log and skip
                        tenant_json_file = f"tenant_{tenant_id}_vtenant_account.json"

                        if not os.path.isfile(
                            os.path.join(backupdir, tenant_json_file)
                        ):
                            error_message = f'failed to find the backup file="{tenant_json_file}" for tenant_id="{tenant_id}", the tenant cannot be restored'
                            logger.error(error_message)
                            errors_list.append(error_message)
                            break

                        # load the tenant json file
                        try:
                            f = open(os.path.join(backupdir, tenant_json_file), "r")
                            data = json.loads(f.read())
                            info_message = f'loaded the tenant backup file="{tenant_json_file}" for tenant_id="{tenant_id}"'
                            logger.info(info_message)
                            tasks_list.append(info_message)

                        except Exception as e:
                            error_message = f'failed to open json file="{os.path.join(backupdir, tenant_json_file)}" for reading with exception="{str(e)}", cannot restore tenant_id="{tenant_id}"'
                            logger.error(error_message)
                            errors_list.append(error_message)
                            break

                        #
                        # Proceed
                        #

                        if not restore_virtual_tenant_accounts:
                            info_message = f'restore_virtual_tenant_accounts is set to False, the Virtual Tenant account for tenant_id="{tenant_id}" will not be restored'
                            logger.info(info_message)
                            tasks_list.append(info_message)

                        else:

                            # If the Virtual Account exists, it must be deleted to be restored

                            # Del the vtenant account
                            url = "%s/servicesNS/nobody/trackme/trackme_vtenants/%s" % (
                                request_info.server_rest_uri,
                                tenant_id,
                            )

                            # Retrieve and set the tenant idx, if any failure, logs and use the global index
                            try:
                                response = requests.delete(
                                    url, headers=header, verify=False, timeout=600
                                )
                                if response.status_code not in (200, 201, 204):
                                    # this can be expected if the tenant does not exist, do not consider this as an error
                                    info_message = f'tenant_id="{tenant_id}", delete vtenant account was not required, response.status_code="{response.status_code}"'
                                    logger.info(info_message)
                                    tasks_list.append(info_message)
                                else:
                                    info_message = f'deleted the Virtual Tenant account for tenant_id="{tenant_id}"'
                                    logger.info(info_message)
                                    tasks_list.append(info_message)
                            except Exception as e:
                                # this is an error
                                error_message = f'failed to delete the Virtual Tenant account for tenant_id="{tenant_id}" with exception="{str(e)}"'
                                logger.error(error_message)
                                errors_list.append(error_message)

                            # Remove keys that start with "eai:"
                            keys_to_remove = [
                                k for k in data.keys() if k.startswith("eai:")
                            ]
                            for k in keys_to_remove:
                                del data[k]

                            # also remove disabled
                            if "disabled" in data:
                                del data["disabled"]

                            # in data, add the key name equal to the tenant_id
                            data["name"] = tenant_id

                            # for key value in vtenant_account_default, check if a key is missing from vtenant_data, if so, add it
                            for key, value in vtenant_account_default.items():
                                if key not in data:
                                    data[key] = value

                            # for key value in vetant_data, if the value is null, empty or None, take the value from vtenant_account_default
                            for key, value in data.items():
                                if value is None:
                                    data[key] = vtenant_account_default.get(key)
                                elif isinstance(value, str) and value == "":
                                    data[key] = vtenant_account_default.get(key)

                            # set vtenant_data as any key that is in data and as well listed in default keys from vtenant_account_default
                            vtenant_data = {
                                key: value
                                for key, value in data.items()
                                if key in vtenant_account_default
                            }

                            #
                            # force attempt a delete and ignore errors
                            #

                            # set the url
                            url = "%s/servicesNS/nobody/trackme/trackme_vtenants/%s" % (
                                request_info.server_rest_uri,
                                tenant_id,
                            )

                            # Retrieve and set the tenant idx, if any failure, logs and use the global index
                            try:
                                response = requests.delete(
                                    url, headers=header, verify=False, timeout=600
                                )
                                if response.status_code not in (200, 201, 204):
                                    logger.warning(
                                        f'delete vtenant account has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                                    )
                                else:
                                    logger.info(
                                        f'delete vtenant account was operated successfully, response.status_code="{response.status_code}"'
                                    )
                            except Exception as e:
                                logger.warning(f'delete vtenant account has failed, exception="{str(e)}"')

                            #
                            # attempt to create the Virtual Tenant account, but don't stop in case of error
                            #

                            # Add the vtenant account
                            url = "%s/servicesNS/nobody/trackme/trackme_vtenants" % (
                                request_info.server_rest_uri
                            )

                            # Run the POST call
                            try:

                                response = requests.post(
                                    url,
                                    headers=header,
                                    data=vtenant_data,
                                    verify=False,
                                    timeout=600,
                                )
                                response.raise_for_status()
                                info_message = f'added the Virtual Tenant account for tenant_id="{tenant_id}"'
                                logger.info(info_message)
                                tasks_list.append(info_message)

                            except Exception as e:
                                error_message = f'failed to add the Virtual Tenant account for tenant_id="{tenant_id}" with exception="{str(e)}", vtenant_data="{json.dumps(vtenant_data, indent=2)}"'
                                logger.error(error_message)
                                errors_list.append(error_message)

                        #####################################################
                        # subtask: restore the central record for this tenant
                        #####################################################

                        #
                        # Proceed
                        #

                        if not restore_virtual_tenant_main_kvrecord:

                            info_message = f'restore_virtual_tenant_main_kvrecord is set to False, the main KVstore record for tenant_id="{tenant_id}" will not be restored'
                            logger.info(info_message)
                            tasks_list.append(info_message)

                        else:

                            tenant_id_main_record = None
                            for record in vtenants_main_records:
                                if record["tenant_id"] == tenant_id:
                                    tenant_id_main_record = record
                                    break

                            if not tenant_id_main_record:
                                error_message = f'failed to find the tenant_id="{tenant_id}" in the backup file="{source_json_file}", the tenant cannot be restored'
                                logger.error(error_message)
                                errors_list.append(error_message)
                                break
                            else:
                                logger.info(
                                    f'Found tenant_id="{tenant_id}" in the backup file="{source_json_file}", the tenant can be restored'
                                )

                            # in live KVstore, check if the tenant_id already exists, it it does, replace the record, otherwise create it
                            try:
                                existing_record = collection.data.query(
                                    query=json.dumps({"tenant_id": tenant_id})
                                )[0]
                            except Exception as e:
                                existing_record = None

                            # if the tenant_id already exists, replace the record
                            if existing_record:
                                try:
                                    collection.data.update(
                                        existing_record["_key"],
                                        json.dumps(tenant_id_main_record),
                                    )
                                    tasks_list.append(
                                        f'updated the tenant_id="{tenant_id}" in the KVstore collection="{collection_name_main}"'
                                    )
                                except Exception as e:
                                    error_message = f'failed to update the tenant_id="{tenant_id}" in the KVstore collection="{collection_name_main}" with exception="{str(e)}"'
                                    logger.error(error_message)
                                    return {
                                        "payload": {
                                            "response": error_message,
                                        },
                                        "status": 500,
                                    }
                            else:
                                try:
                                    collection.data.insert(
                                        json.dumps(tenant_id_main_record)
                                    )
                                    tasks_list.append(
                                        f'inserted the tenant_id="{tenant_id}" in the KVstore collection="{collection_name_main}"'
                                    )
                                except Exception as e:
                                    error_message = f'failed to insert the tenant_id="{tenant_id}" in the KVstore collection="{collection_name_main}" with exception="{str(e)}"'
                                    logger.error(error_message)
                                    return {
                                        "payload": {
                                            "response": error_message,
                                        },
                                        "status": 500,
                                    }

                        ##########################################################
                        # task: load knowledge objects backup file for this tenant
                        ##########################################################

                        knowledge_objects_json_file = (
                            f"tenant_{tenant_id}_knowledge_objects.json"
                        )
                        try:
                            f = open(
                                os.path.join(backupdir, knowledge_objects_json_file),
                                "r",
                            )
                            data = json.loads(f.read())
                            info_message = f'loaded the knowledge objects backup file="{knowledge_objects_json_file}" for tenant_id="{tenant_id}"'
                            logger.info(info_message)
                            tasks_list.append(info_message)
                        except Exception as e:
                            error_message = f'failed to open json file="{os.path.join(backupdir, knowledge_objects_json_file)}" for reading with exception="{str(e)}", cannot restore knowledge objects for tenant_id="{tenant_id}"'
                            logger.error(error_message)
                            errors_list.append(error_message)
                            break

                        # init lists and dicts of knowledge objects for this tenant in the backup data
                        kvstore_collections_restorable_list = []
                        kvstore_collections_restorable_dict = {}
                        kvstore_transforms_restorable_list = []
                        kvstore_transforms_restorable_dict = {}
                        macros_restorable_list = []
                        macros_restorable_dict = {}
                        savedsearches_restorable_list = []
                        savedsearches_restorable_dict = {}
                        alerts_restorable_list = []
                        alerts_restorable_dict = {}

                        # Initialize lists to track failed objects for second attempt
                        failed_transforms = []
                        failed_macros = []
                        failed_savedsearches = []
                        failed_alerts = []

                        # first parse the data, get the type of object and the name, add to the appropriate list, log and add to the tasks list
                        for record in data.values():

                            if record["type"] == "kvstore_collections":
                                kvstore_collections_restorable_list.append(
                                    record["title"]
                                )
                                kvstore_collections_restorable_dict[record["title"]] = (
                                    record
                                )
                            if record["type"] == "lookup_definitions":
                                kvstore_transforms_restorable_list.append(
                                    record["title"]
                                )
                                kvstore_transforms_restorable_dict[record["title"]] = (
                                    record
                                )
                            if record["type"] == "macros":
                                macros_restorable_list.append(record["title"])
                                macros_restorable_dict[record["title"]] = record
                            if record["type"] == "savedsearches":
                                savedsearches_restorable_list.append(record["title"])
                                savedsearches_restorable_dict[record["title"]] = record
                            if record["type"] == "alerts":
                                alerts_restorable_list.append(record["title"])
                                alerts_restorable_dict[record["title"]] = record

                        info_message = f'knowledge objects backup file="{knowledge_objects_json_file}" for tenant_id="{tenant_id}" parsed successfully, number of restorable kvstore collections="{len(kvstore_collections_restorable_list)}", number of restorable transforms definitions="{len(kvstore_transforms_restorable_list)}", number of restorable macros="{len(macros_restorable_list)}", number of restorable saved searches="{len(savedsearches_restorable_list)}"'
                        logger.info(info_message)
                        tasks_list.append(info_message)

                        # add to response_dict
                        response_dict[f"knowledge_objects_tenant_id_{tenant_id}"] = {
                            "kvstore_collections": kvstore_collections_restorable_list,
                            "transforms_definitions": kvstore_transforms_restorable_list,
                            "macros": macros_restorable_list,
                            "savedsearches": savedsearches_restorable_list,
                            "alerts": alerts_restorable_list,
                        }

                        ##########################################################
                        # task: restore kvstore collections definitions for tenant
                        ##########################################################

                        # Loop through the restorable collections, and proceed
                        if restore_kvstore_collections:
                            for collection_name in kvstore_collections_restorable_list:
                                logger.info(
                                    f'TrackMe restore process starting, knowledge objects, processing restore of kvstore definition for collection_name="{collection_name}" for tenant_id="{tenant_id}"'
                                )

                                # check if the collection exists already
                                collection_exists = False

                                try:
                                    collection = service.kvstore[collection_name]
                                    collection_exists = True

                                except Exception as e:
                                    pass

                                # if knowledge_objects_replace_existing, systematically attempt to delete the collection
                                if (
                                    collection_exists
                                    and not knowledge_objects_replace_existing
                                ):

                                    logger.info(
                                        f'collection="{collection_name}" already exists and the replace_existing option is set to False, skipping the restore operation'
                                    )
                                    result = {
                                        "object": collection_name,
                                        "object_type": "kvstore collection",
                                        "action": "restore",
                                        "result": "skipped",
                                        "reason": "already exists and replace_existing is set to False",
                                    }
                                    logger.info(json.dumps(result, indent=4))
                                    tasks_list.append(result)
                                    continue

                                else:

                                    if collection_exists:
                                        try:
                                            action = trackme_delete_kvcollection(
                                                request_info.system_authtoken,
                                                request_info.server_rest_uri,
                                                tenant_id,
                                                collection_name,
                                            )
                                            logger.info(
                                                f'collection="{collection_name}" was deleted successfully, response="{action}"'
                                            )
                                            result = {
                                                "object": collection_name,
                                                "object_type": "kvstore collection",
                                                "action": "delete",
                                                "result": "success",
                                            }
                                            tasks_list.append(result)
                                        except Exception as e:
                                            pass  # no need to log or report, the collection might not exist yet

                                try:

                                    ko_acl = {
                                        "owner": kvstore_collections_restorable_dict[
                                            collection_name
                                        ]["properties"]["eai:acl.owner"],
                                        "sharing": kvstore_collections_restorable_dict[
                                            collection_name
                                        ]["properties"]["eai:acl.sharing"],
                                        "perms.write": kvstore_collections_restorable_dict[
                                            collection_name
                                        ][
                                            "properties"
                                        ][
                                            "eai:acl.perms.write"
                                        ],
                                        "perms.read": kvstore_collections_restorable_dict[
                                            collection_name
                                        ][
                                            "properties"
                                        ][
                                            "eai:acl.perms.read"
                                        ],
                                    }

                                    trackme_create_kvcollection(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        collection_name,
                                        ko_acl,
                                    )
                                    result = {
                                        "object": collection_name,
                                        "object_type": "kvstore collection",
                                        "action": "restore",
                                        "result": "success",
                                    }
                                    tasks_list.append(result)
                                except Exception as e:
                                    result = {
                                        "object": collection_name,
                                        "object_type": "kvstore collection",
                                        "action": "restore",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    logger.error(json.dumps(result, indent=4))
                                    errors_list.append(result)

                                # restore the collection records for this collection and add to the list of restored collections
                                try:
                                    (
                                        kvstore_collections_global_records_to_be_restored,
                                        kvstore_collections_global_records_restored,
                                        kvstore_collection_restore_summary_dict,
                                    ) = restore_kvstore_records(
                                        service,
                                        collection_name,
                                        collections_restore_dict,
                                        backupdir,
                                        kvstore_collections_global_records_to_be_restored,
                                        kvstore_collections_global_records_restored,
                                        kvstore_collections_restored_warning,
                                        restore_results_dict,
                                        kvstore_collections_clean_empty,
                                    )
                                except KeyError as e:
                                    # Handle case where collection is not in collections_restore_dict
                                    error_msg = f'Collection "{collection_name}" not found in backup data, skipping restore. This may be due to the collection being excluded from backups (e.g., stateful charts collections).'
                                    logger.warning(error_msg)
                                    result = {
                                        "object": collection_name,
                                        "object_type": "kvstore collection",
                                        "action": "restore",
                                        "result": "skipped",
                                        "reason": "collection not found in backup data",
                                    }
                                    tasks_list.append(result)
                                    continue
                                except Exception as e:
                                    # Handle any other errors during restore
                                    error_msg = f'Failed to restore collection "{collection_name}": {str(e)}'
                                    logger.error(error_msg)
                                    result = {
                                        "object": collection_name,
                                        "object_type": "kvstore collection",
                                        "action": "restore",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    errors_list.append(result)
                                    continue

                                kvstore_collections_restored.append(collection_name)

                                # add to dict
                                kvstore_collections_restore_summary_dict[
                                    collection_name
                                ] = kvstore_collection_restore_summary_dict

                        #########################################################
                        # task: restore kvstore transforms definitions for tenant
                        #########################################################

                        # Loop through the restorable collections transforms, and proceed

                        # if knowledge_objects_lists is not equal to all, filter the list to its value
                        if knowledge_objects_lists != "all":
                            kvstore_transforms_restorable_list = [
                                transform
                                for transform in kvstore_transforms_restorable_list
                                if transform in knowledge_objects_lists
                            ]

                        # Filter out any transforms in the blocklist
                        kvstore_transforms_restorable_list = [
                            transform
                            for transform in kvstore_transforms_restorable_list
                            if transform not in knowledge_objects_blocklist
                        ]

                        for transforms_name in kvstore_transforms_restorable_list:

                            logger.info(
                                f'TrackMe restore process starting, knowledge objects, processing restore of kvstore transforms definition for transforms_name="{transforms_name}" for tenant_id="{tenant_id}"'
                            )

                            # check if the transforms exists already
                            transforms_exists = False

                            try:
                                transforms_object = service.confs["transforms"][
                                    transforms_name
                                ]
                                transforms_exists = True

                            except Exception as e:
                                pass

                            if (
                                transforms_exists
                                and not knowledge_objects_replace_existing
                            ):
                                logger.info(
                                    f'transforms="{transforms_name}" already exists and the replace_existing option is set to False, skipping the restore operation'
                                )
                                result = {
                                    "object": transforms_name,
                                    "object_type": "kvstore transforms",
                                    "action": "restore",
                                    "result": "skipped",
                                    "reason": "already exists and replace_existing is set to False",
                                }
                                logger.info(json.dumps(result, indent=4))
                                tasks_list.append(result)
                                continue

                            else:

                                if transforms_exists:

                                    try:
                                        action = trackme_delete_kvtransform(
                                            request_info.system_authtoken,
                                            request_info.server_rest_uri,
                                            tenant_id,
                                            transforms_name,
                                        )
                                        result = {
                                            "object": transforms_name,
                                            "object_type": "transform",
                                            "action": "delete",
                                            "result": "success",
                                        }
                                        logger.info(json.dumps(result, indent=4))
                                        tasks_list.append(result)

                                    except Exception as e:
                                        result = {
                                            "object": transforms_name,
                                            "object_type": "transform",
                                            "action": "delete",
                                            "result": "failure",
                                            "exception": str(e),
                                        }
                                        logger.error(json.dumps(result, indent=4))
                                        errors_list.append(result)

                                try:

                                    ko_acl = {
                                        "owner": kvstore_transforms_restorable_dict[
                                            transforms_name
                                        ]["properties"]["eai:acl.owner"],
                                        "sharing": kvstore_transforms_restorable_dict[
                                            transforms_name
                                        ]["properties"]["eai:acl.sharing"],
                                        "perms.write": kvstore_transforms_restorable_dict[
                                            transforms_name
                                        ][
                                            "properties"
                                        ][
                                            "eai:acl.perms.write"
                                        ],
                                        "perms.read": kvstore_transforms_restorable_dict[
                                            transforms_name
                                        ][
                                            "properties"
                                        ][
                                            "eai:acl.perms.read"
                                        ],
                                    }

                                    trackme_create_kvtransform(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        transforms_name,
                                        kvstore_transforms_restorable_dict[
                                            transforms_name
                                        ]["fields_list"],
                                        kvstore_transforms_restorable_dict[
                                            transforms_name
                                        ]["collection"],
                                        ko_acl.get("owner"),
                                        ko_acl,
                                    )
                                    result = {
                                        "object": transforms_name,
                                        "object_type": "kvstore transforms",
                                        "action": "restore",
                                        "result": "success",
                                    }
                                    logger.info(json.dumps(result, indent=4))
                                    tasks_list.append(result)

                                except Exception as e:
                                    result = {
                                        "object": transforms_name,
                                        "object_type": "kvstore transforms",
                                        "action": "restore",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    logger.error(json.dumps(result, indent=4))
                                    failed_transforms.append(
                                        {
                                            "name": transforms_name,
                                            "tenant_id": tenant_id,
                                            "transform_fields_list": kvstore_transforms_restorable_dict[
                                                transforms_name
                                            ][
                                                "fields_list"
                                            ],
                                            "transform_collection": kvstore_transforms_restorable_dict[
                                                transforms_name
                                            ][
                                                "collection"
                                            ],
                                            "owner": ko_acl.get("owner"),
                                            "acl": ko_acl,
                                        }
                                    )

                        #############################################
                        # task: restore macros definitions for tenant
                        #############################################

                        # Loop through the restorable macros, and proceed

                        # if knowledge_objects_lists is not equal to all, filter the list to its value
                        if knowledge_objects_lists != "all":
                            macros_restorable_list = [
                                macro
                                for macro in macros_restorable_list
                                if macro in knowledge_objects_lists
                            ]

                        # Filter out any macros in the blocklist
                        macros_restorable_list = [
                            macro
                            for macro in macros_restorable_list
                            if macro not in knowledge_objects_blocklist
                        ]

                        for macro_name in macros_restorable_list:

                            logger.info(
                                f'TrackMe restore process starting, knowledge objects, processing restore of macro definition for macro_name="{macro_name}" for tenant_id="{tenant_id}"'
                            )

                            # check if the macro exists already
                            macro_exists = False

                            try:
                                macro_object = service.confs["macros"][macro_name]
                                macro_exists = True
                            except Exception as e:
                                pass

                            if macro_exists and not knowledge_objects_replace_existing:
                                logger.info(
                                    f'macro="{macro_name}" already exists and the replace_existing option is set to False, skipping the restore operation'
                                )
                                result = {
                                    "object": macro_name,
                                    "object_type": "macro",
                                    "action": "restore",
                                    "result": "skipped",
                                    "reason": "already exists and replace_existing is set to False",
                                }
                                logger.info(json.dumps(result, indent=4))
                                tasks_list.append(result)
                                continue

                            else:

                                if macro_exists:

                                    try:
                                        action = trackme_delete_macro(
                                            request_info.system_authtoken,
                                            request_info.server_rest_uri,
                                            tenant_id,
                                            macro_name,
                                        )
                                        logger.info(
                                            f'macro="{macro_name}" was deleted successfully, response="{action}"'
                                        )
                                        result = {
                                            "object": macro_name,
                                            "object_type": "macro",
                                            "action": "delete",
                                            "result": "success",
                                        }
                                        logger.info(json.dumps(result, indent=4))
                                        tasks_list.append(result)

                                    except Exception as e:
                                        result = {
                                            "object": macro_name,
                                            "object_type": "macro",
                                            "action": "delete",
                                            "result": "failure",
                                            "exception": str(e),
                                        }
                                        logger.error(json.dumps(result, indent=4))
                                        errors_list.append(result)

                                try:

                                    ko_acl = {
                                        "owner": macros_restorable_dict[macro_name][
                                            "properties"
                                        ]["eai:acl.owner"],
                                        "sharing": macros_restorable_dict[macro_name][
                                            "properties"
                                        ]["eai:acl.sharing"],
                                        "perms.write": macros_restorable_dict[
                                            macro_name
                                        ]["properties"]["eai:acl.perms.write"],
                                        "perms.read": macros_restorable_dict[
                                            macro_name
                                        ]["properties"]["eai:acl.perms.read"],
                                    }

                                    trackme_create_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        macros_restorable_dict[macro_name][
                                            "definition"
                                        ],
                                        ko_acl.get("owner"),
                                        ko_acl,
                                    )

                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "restore",
                                        "result": "success",
                                    }
                                    logger.info(json.dumps(result, indent=4))
                                    tasks_list.append(result)

                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "restore",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    logger.error(json.dumps(result, indent=4))
                                    failed_macros.append(
                                        {
                                            "name": macro_name,
                                            "tenant_id": tenant_id,
                                            "macro_definition": macros_restorable_dict[
                                                macro_name
                                            ]["definition"],
                                            "owner": ko_acl.get("owner"),
                                            "acl": ko_acl,
                                        }
                                    )

                        ####################################################
                        # task: restore savedsearches definitions for tenant
                        ####################################################

                        # Save available savedsearches in a main list
                        savedsearches_restorable_list_origin = []
                        for savedsearch in savedsearches_restorable_dict:
                            savedsearches_restorable_list_origin.append(savedsearch)

                        # If knowledge_objects_lists is not equal to "all", filter the list to its value
                        if knowledge_objects_lists != "all":
                            savedsearches_restorable_list = [
                                savedsearch
                                for savedsearch in savedsearches_restorable_list_origin
                                if savedsearch in knowledge_objects_lists
                            ]
                        else:
                            savedsearches_restorable_list = (
                                savedsearches_restorable_list_origin.copy()
                            )

                        # Filter out any saved searches in the blocklist
                        savedsearches_restorable_list = [
                            savedsearch
                            for savedsearch in savedsearches_restorable_list
                            if savedsearch not in knowledge_objects_blocklist
                        ]

                        # Restoring saved searches has some challenges, if the search itself refers to another saved search, we need to
                        # restore this saved search first, and then restore the search that refers to it.

                        # Init a list of restored saved searches
                        savedsearches_restored = []

                        for savedsearch_name in savedsearches_restorable_list:

                            logger.info(
                                f'TrackMe restore process starting, knowledge objects, processing restore of savedsearch definition for savedsearch_name="{savedsearch_name}" for tenant_id="{tenant_id}"'
                            )

                            # First, get the search definition
                            savedsearch_definition = savedsearches_restorable_dict[
                                savedsearch_name
                            ]["definition"]

                            # Use regex to find the parent search
                            parent_search_match = re.search(
                                r"\| savedsearch\s+\"{0,1}([^\"]+)\"{0,1}",
                                savedsearch_definition,
                            )

                            # Extract parent search name if found
                            parent_search_name = (
                                parent_search_match.group(1)
                                if parent_search_match
                                else None
                            )

                            if parent_search_name:
                                # Check if the parent search exists in the original restorable list
                                if (
                                    parent_search_name
                                    in savedsearches_restorable_list_origin
                                ):
                                    # Ensure the parent search is restored first
                                    if parent_search_name not in savedsearches_restored:
                                        logger.info(
                                            f'TrackMe restore process: parent search "{parent_search_name}" detected for savedsearch "{savedsearch_name}". Restoring parent first.'
                                        )
                                        # Recursive restore call for the parent search
                                        if (
                                            parent_search_name
                                            in savedsearches_restorable_list
                                        ):
                                            savedsearches_restorable_list.remove(
                                                parent_search_name
                                            )  # Avoid duplicate processing
                                        savedsearches_restorable_list.insert(
                                            0, parent_search_name
                                        )  # Prioritize restoring the parent search
                                else:
                                    # Log an error and skip the child search if the parent isn't available
                                    error_message = {
                                        "object": savedsearch_name,
                                        "object_type": "savedsearch",
                                        "action": "restore",
                                        "result": "failure",
                                        "reason": f"Parent search '{parent_search_name}' not found in restorable list.",
                                    }
                                    logger.error(json.dumps(error_message, indent=4))
                                    errors_list.append(error_message)
                                    continue  # Skip restoring the current search

                            # Proceed with the restoration logic for the current saved search
                            savedsearch_exists = False
                            try:
                                savedsearch_object = service.saved_searches[
                                    savedsearch_name
                                ]
                                savedsearch_search = savedsearch_object.content[
                                    "search"
                                ]
                                savedsearch_exists = True

                            except Exception:
                                pass

                            if (
                                savedsearch_exists
                                and not knowledge_objects_replace_existing
                            ):
                                logger.info(
                                    f'savedsearch="{savedsearch_name}" already exists and the replace_existing option is set to False, skipping the restore operation'
                                )
                                result = {
                                    "object": savedsearch_name,
                                    "object_type": "savedsearch",
                                    "action": "restore",
                                    "result": "skipped",
                                    "reason": "already exists and replace_existing is set to False",
                                }
                                logger.info(json.dumps(result, indent=4))
                                tasks_list.append(result)
                            else:
                                # Restore logic
                                try:
                                    if savedsearch_exists:
                                        # Delete existing savedsearch if required
                                        action = trackme_delete_report(
                                            request_info.system_authtoken,
                                            request_info.server_rest_uri,
                                            tenant_id,
                                            savedsearch_name,
                                        )
                                        logger.info(
                                            f'savedsearch="{savedsearch_name}" was deleted successfully, response="{action}"'
                                        )

                                    # Prepare properties for restoration
                                    savedsearch_properties = {
                                        "description": savedsearches_restorable_dict[
                                            savedsearch_name
                                        ]["properties"]["description"],
                                        "is_scheduled": savedsearches_restorable_dict[
                                            savedsearch_name
                                        ]["properties"]["is_scheduled"],
                                        "schedule_window": savedsearches_restorable_dict[
                                            savedsearch_name
                                        ][
                                            "properties"
                                        ][
                                            "schedule_window"
                                        ],
                                        "dispatch.earliest_time": savedsearches_restorable_dict[
                                            savedsearch_name
                                        ][
                                            "properties"
                                        ][
                                            "earliest_time"
                                        ],
                                        "dispatch.latest_time": savedsearches_restorable_dict[
                                            savedsearch_name
                                        ][
                                            "properties"
                                        ][
                                            "latest_time"
                                        ],
                                    }

                                    # Only add cron_schedule if it's not None or "None"
                                    cron_schedule = savedsearches_restorable_dict[
                                        savedsearch_name
                                    ]["properties"].get("cron_schedule")
                                    if cron_schedule and cron_schedule not in (
                                        None,
                                        "None",
                                        "null",
                                    ):
                                        savedsearch_properties["cron_schedule"] = (
                                            cron_schedule
                                        )

                                    # if dispatch.sample_ratio is set in the restorable dict, add it to the properties
                                    if (
                                        "dispatch.sample_ratio"
                                        in savedsearches_restorable_dict[
                                            savedsearch_name
                                        ]["properties"]
                                    ):
                                        savedsearch_properties[
                                            "dispatch.sample_ratio"
                                        ] = savedsearches_restorable_dict[
                                            savedsearch_name
                                        ][
                                            "properties"
                                        ][
                                            "dispatch.sample_ratio"
                                        ]

                                    ko_acl = {
                                        "owner": savedsearches_restorable_dict[
                                            savedsearch_name
                                        ]["properties"]["eai:acl.owner"],
                                        "sharing": savedsearches_restorable_dict[
                                            savedsearch_name
                                        ]["properties"]["eai:acl.sharing"],
                                        "perms.write": savedsearches_restorable_dict[
                                            savedsearch_name
                                        ]["properties"]["eai:acl.perms.write"],
                                        "perms.read": savedsearches_restorable_dict[
                                            savedsearch_name
                                        ]["properties"]["eai:acl.perms.read"],
                                    }

                                    # Restore the savedsearch
                                    trackme_create_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        savedsearch_name,
                                        savedsearches_restorable_dict[savedsearch_name][
                                            "definition"
                                        ],
                                        savedsearch_properties,
                                        ko_acl,
                                    )

                                    result = {
                                        "object": savedsearch_name,
                                        "object_type": "savedsearch",
                                        "action": "restore",
                                        "result": "success",
                                    }
                                    logger.info(json.dumps(result, indent=4))
                                    tasks_list.append(result)
                                    savedsearches_restored.append(
                                        savedsearch_name
                                    )  # Mark as restored

                                except Exception as e:
                                    result = {
                                        "object": savedsearch_name,
                                        "object_type": "savedsearch",
                                        "action": "restore",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    logger.error(json.dumps(result, indent=4))
                                    failed_savedsearches.append(
                                        {
                                            "name": savedsearch_name,
                                            "tenant_id": tenant_id,
                                            "savedsearch_definition": savedsearches_restorable_dict[
                                                savedsearch_name
                                            ][
                                                "definition"
                                            ],
                                            "savedsearch_properties": savedsearch_properties,
                                            "owner": ko_acl.get("owner"),
                                            "acl": ko_acl,
                                        }
                                    )

                        #############################################
                        # task: restore alerts definitions for tenant
                        #############################################

                        # Save available alerts in a main list
                        alerts_restorable_list_origin = []
                        for alert in alerts_restorable_dict:
                            alerts_restorable_list_origin.append(alert)

                        # If knowledge_objects_lists is not equal to "all", filter the list to its value
                        if knowledge_objects_lists != "all":
                            alerts_restorable_list = [
                                alert
                                for alert in alerts_restorable_list_origin
                                if alert in knowledge_objects_lists
                            ]
                        else:
                            alerts_restorable_list = (
                                alerts_restorable_list_origin.copy()
                            )

                        # Filter out any alerts in the blocklist
                        alerts_restorable_list = [
                            alert
                            for alert in alerts_restorable_list
                            if alert not in knowledge_objects_blocklist
                        ]

                        # Restoring saved searches has some challenges, if the search itself refers to another saved search, we need to
                        # restore this saved search first, and then restore the search that refers to it.

                        # Init a list of restored saved searches
                        alerts_restored = []

                        for alert_name in alerts_restorable_list:

                            logger.info(
                                f'TrackMe restore process starting, knowledge objects, processing restore of alert definition for alert_name="{alert_name}" for tenant_id="{tenant_id}"'
                            )

                            # First, get the search definition
                            alert_definition = alerts_restorable_dict[alert_name][
                                "definition"
                            ]

                            # Proceed with the restoration logic for the current saved search
                            alert_exists = False
                            try:
                                alert_object = service.saved_searches[alert_name]
                                alert_search = alert_object.content["search"]
                                alert_exists = True

                            except Exception:
                                pass

                            if alert_exists and not knowledge_objects_replace_existing:
                                logger.info(
                                    f'alert="{alert_name}" already exists and the replace_existing option is set to False, skipping the restore operation'
                                )
                                result = {
                                    "object": alert_name,
                                    "object_type": "alert",
                                    "action": "restore",
                                    "result": "skipped",
                                    "reason": "already exists and replace_existing is set to False",
                                }
                                logger.info(json.dumps(result, indent=4))
                                tasks_list.append(result)

                            else:
                                # Restore logic
                                logger.info(
                                    f'alert_name="{alert_name}" does not exist, restoring it'
                                )

                                if alert_exists:
                                    # Delete existing alert if required
                                    try:
                                        action = trackme_delete_report(
                                            request_info.system_authtoken,
                                            request_info.server_rest_uri,
                                            tenant_id,
                                            alert_name,
                                        )
                                        logger.info(
                                            f'alert="{alert_name}" was deleted successfully, response="{action}"'
                                        )
                                    except Exception as e:
                                        pass

                                # Prepare properties for restoration
                                properties = {
                                    "description": alerts_restorable_dict[alert_name][
                                        "properties"
                                    ]["description"],
                                    "is_scheduled": alerts_restorable_dict[alert_name][
                                        "properties"
                                    ]["is_scheduled"],
                                    "schedule_window": alerts_restorable_dict[
                                        alert_name
                                    ]["properties"]["schedule_window"],
                                    "dispatch.earliest_time": alerts_restorable_dict[
                                        alert_name
                                    ]["properties"]["earliest_time"],
                                    "dispatch.latest_time": alerts_restorable_dict[
                                        alert_name
                                    ]["properties"]["latest_time"],
                                }

                                # Only add cron_schedule if it's not None or "None"
                                cron_schedule = alerts_restorable_dict[alert_name][
                                    "properties"
                                ].get("cron_schedule")
                                if cron_schedule and cron_schedule not in (
                                    None,
                                    "None",
                                    "null",
                                ):
                                    properties["cron_schedule"] = cron_schedule

                                # if dispatch.sample_ratio is set in the restorable dict, add it to the properties
                                if (
                                    "dispatch.sample_ratio"
                                    in alerts_restorable_dict[alert_name]["properties"]
                                ):
                                    properties["dispatch.sample_ratio"] = (
                                        alerts_restorable_dict[alert_name][
                                            "properties"
                                        ]["dispatch.sample_ratio"]
                                    )

                                ko_acl = {
                                    "owner": alerts_restorable_dict[alert_name][
                                        "properties"
                                    ]["eai:acl.owner"],
                                    "sharing": alerts_restorable_dict[alert_name][
                                        "properties"
                                    ]["eai:acl.sharing"],
                                    "perms.write": alerts_restorable_dict[alert_name][
                                        "properties"
                                    ]["eai:acl.perms.write"],
                                    "perms.read": alerts_restorable_dict[alert_name][
                                        "properties"
                                    ]["eai:acl.perms.read"],
                                }

                                alert_properties = alerts_restorable_dict[alert_name][
                                    "alert_properties"
                                ]

                                # enabling alert actions:
                                alert_actions = []

                                if (
                                    int(
                                        alert_properties.get(
                                            "action.trackme_stateful_alert", 0
                                        )
                                    )
                                    == 1
                                ):
                                    alert_actions.append("trackme_stateful_alert")

                                if (
                                    int(
                                        alert_properties.get("action.trackme_notable"),
                                        0,
                                    )
                                    == 1
                                ):
                                    alert_actions.append("trackme_notable")

                                if (
                                    int(
                                        alert_properties.get("action.trackme_auto_ack"),
                                        0,
                                    )
                                    == 1
                                ):
                                    alert_actions.append("trackme_auto_ack")

                                if (
                                    int(
                                        alert_properties.get(
                                            "action.trackme_smart_status", 0
                                        )
                                    )
                                    == 1
                                ):
                                    alert_actions.append("trackme_smart_status")

                                # turn the list into a csv string and add to alert_properties
                                alert_properties["actions"] = ",".join(alert_actions)

                                # prevents issues with external alerts, parse alert_properties,
                                # and remove any field which action.<parameter> which does start as:
                                # actions.trackme*

                                # Remove any keys that start with "action." but don't start with "action.trackme"
                                keys_to_remove = [
                                    key
                                    for key in alert_properties.keys()
                                    if key.startswith("action.")
                                    and not key.startswith("action.trackme")
                                ]
                                for key in keys_to_remove:
                                    del alert_properties[key]

                                logger.info(
                                    f'calling trackme_create_alert with properties="{json.dumps(properties, indent=2)}", alert_properties="{json.dumps(alert_properties, indent=2)}"'
                                )

                                try:

                                    # Restore the alert
                                    trackme_create_alert(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        alert_name,
                                        alerts_restorable_dict[alert_name][
                                            "definition"
                                        ],
                                        properties,
                                        alert_properties,
                                        ko_acl,
                                    )

                                    result = {
                                        "object": alert_name,
                                        "object_type": "alert",
                                        "action": "restore",
                                        "result": "success",
                                    }
                                    logger.info(json.dumps(result, indent=4))
                                    tasks_list.append(result)
                                    alerts_restored.append(
                                        alert_name
                                    )  # Mark as restored

                                except Exception as e:
                                    result = {
                                        "object": alert_name,
                                        "object_type": "alert",
                                        "action": "restore",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    logger.error(json.dumps(result, indent=4))
                                    failed_alerts.append(
                                        {
                                            "name": alert_name,
                                            "tenant_id": tenant_id,
                                            "alert_definition": alerts_restorable_dict[
                                                alert_name
                                            ]["definition"],
                                            "properties": properties,
                                            "alert_properties": alert_properties,
                                            "owner": ko_acl.get("owner"),
                                            "acl": ko_acl,
                                        }
                                    )

                        ################################################
                        # task: second attempt to restore failed objects
                        ################################################

                        # Retry failed transforms
                        for failed_transform in failed_transforms:
                            logger.info(
                                f'Second attempt to restore transform="{failed_transform["name"]}" for tenant_id="{failed_transform["tenant_id"]}"'
                            )

                            try:
                                trackme_create_kvtransform(
                                    request_info.system_authtoken,
                                    request_info.server_rest_uri,
                                    tenant_id,
                                    failed_transform["name"],
                                    failed_transform["transform_fields_list"],
                                    failed_transform["transform_collection"],
                                    failed_transform["owner"],
                                    failed_transform["acl"],
                                )

                                result = {
                                    "object": failed_transform["name"],
                                    "object_type": "transform",
                                    "action": "restore",
                                    "result": "success",
                                    "attempt": "second",
                                }
                                logger.info(json.dumps(result, indent=4))
                                tasks_list.append(result)

                            except Exception as e:
                                result = {
                                    "object": failed_transform["name"],
                                    "object_type": "transform",
                                    "action": "restore",
                                    "result": "failure",
                                    "exception": str(e),
                                }
                                logger.error(json.dumps(result, indent=4))
                                errors_list.append(result)

                        # Retry failed macros
                        for failed_macro in failed_macros:
                            logger.info(
                                f'Second attempt to restore macro="{failed_macro["name"]}" for tenant_id="{failed_macro["tenant_id"]}"'
                            )

                            try:
                                trackme_create_macro(
                                    request_info.system_authtoken,
                                    request_info.server_rest_uri,
                                    tenant_id,
                                    failed_macro["name"],
                                    failed_macro["macro_definition"],
                                    failed_macro["owner"],
                                    failed_macro["acl"],
                                )

                                result = {
                                    "object": failed_macro["name"],
                                    "object_type": "macro",
                                    "action": "restore",
                                    "result": "success",
                                    "attempt": "second",
                                }
                                logger.info(json.dumps(result, indent=4))
                                tasks_list.append(result)

                            except Exception as e:
                                result = {
                                    "object": failed_macro["name"],
                                    "object_type": "macro",
                                    "action": "restore",
                                    "result": "failure",
                                    "exception": str(e),
                                }
                                logger.error(json.dumps(result, indent=4))
                                errors_list.append(result)

                        # Retry failed saved searches
                        for failed_savedsearch in failed_savedsearches:
                            logger.info(
                                f'Second attempt to restore savedsearch="{failed_savedsearch["name"]}" for tenant_id="{failed_savedsearch["tenant_id"]}"'
                            )

                            try:
                                trackme_create_report(
                                    request_info.system_authtoken,
                                    request_info.server_rest_uri,
                                    tenant_id,
                                    failed_savedsearch["name"],
                                    failed_savedsearch["savedsearch_definition"],
                                    failed_savedsearch["savedsearch_properties"],
                                    failed_savedsearch["acl"],
                                )

                                result = {
                                    "object": failed_savedsearch["name"],
                                    "object_type": "savedsearch",
                                    "action": "restore",
                                    "result": "success",
                                    "attempt": "second",
                                }
                                logger.info(json.dumps(result, indent=4))
                                tasks_list.append(result)

                            except Exception as e:
                                result = {
                                    "object": failed_savedsearch["name"],
                                    "object_type": "savedsearch",
                                    "action": "restore",
                                    "result": "failure",
                                    "exception": str(e),
                                }
                                logger.error(json.dumps(result, indent=4))
                                errors_list.append(result)

                        # Retry failed alerts
                        for failed_alert in failed_alerts:
                            logger.info(
                                f'Second attempt to restore alert="{failed_alert["name"]}" for tenant_id="{failed_alert["tenant_id"]}", properties="{json.dumps(failed_alert["properties"], indent=2)}", alert_properties="{json.dumps(failed_alert["alert_properties"], indent=2)}"'
                            )

                            try:
                                trackme_create_alert(
                                    request_info.system_authtoken,
                                    request_info.server_rest_uri,
                                    tenant_id,
                                    failed_alert["name"],
                                    failed_alert["alert_definition"],
                                    failed_alert["properties"],
                                    failed_alert["alert_properties"],
                                    failed_alert["acl"],
                                )

                                result = {
                                    "object": failed_alert["name"],
                                    "object_type": "alert",
                                    "action": "restore",
                                    "result": "success",
                                    "attempt": "second",
                                }
                                logger.info(json.dumps(result, indent=4))
                                tasks_list.append(result)

                            except Exception as e:
                                result = {
                                    "object": failed_alert["name"],
                                    "object_type": "alert",
                                    "action": "restore",
                                    "result": "failure",
                                    "exception": str(e),
                                }
                                logger.error(json.dumps(result, indent=4))
                                errors_list.append(result)

                #######################################
                # Task: Restore the KVstore collections
                #######################################

                if (
                    restore_kvstore_collections
                    and kvstore_collections_restore_non_tenants_collections
                ):
                    logger.info(
                        f"TrackMe restore process starting, processing restore of KVstore collections"
                    )

                    # if knowledge_objects_tenants_scope is not all, then we need to filter the collections to be restored
                    # and remove kv_trackme_virtual_tenants and kv_trackme_virtual_tenants_entities_summary
                    if knowledge_objects_tenants_scope != "all":
                        kvstore_collections_to_be_restored = [
                            collection_name
                            for collection_name in kvstore_collections_to_be_restored
                            if collection_name
                            not in [
                                "kv_trackme_virtual_tenants",
                                "kv_trackme_virtual_tenants_entities_summary",
                            ]
                        ]

                    # Loop through the restorable collections, and proceed
                    for collection_name in kvstore_collections_to_be_restored:

                        # Skip if collection is not in the tenant scope
                        if knowledge_objects_tenants_scope != "all":
                            # Extract tenant_id from collection name if it's a tenant-specific collection
                            tenant_id = None
                            if (
                                collection_name.startswith("kv_trackme_")
                                and "_tenant_" in collection_name
                            ):
                                try:
                                    tenant_id = collection_name.split("_tenant_")[
                                        1
                                    ].split("_")[0]
                                except:
                                    pass

                            # Skip if this is a tenant-specific collection and the tenant is not in scope
                            if (
                                tenant_id
                                and tenant_id not in knowledge_objects_tenants_scope
                            ):
                                logger.info(
                                    f'Skipping collection="{collection_name}" as it belongs to tenant="{tenant_id}" which is not in the restore scope'
                                )
                                continue

                        if (
                            collection_name not in kvstore_collections_restored
                            and collection_name != "kv_trackme_backup_archives_info"
                        ):  # restore collections that were not restored with the knowledge objects for the tenant
                            try:
                                (
                                    kvstore_collections_global_records_to_be_restored,
                                    kvstore_collections_global_records_restored,
                                    kvstore_collection_restore_summary_dict,
                                ) = restore_kvstore_records(
                                    service,
                                    collection_name,
                                    collections_restore_dict,
                                    backupdir,
                                    kvstore_collections_global_records_to_be_restored,
                                    kvstore_collections_global_records_restored,
                                    kvstore_collections_restored_warning,
                                    restore_results_dict,
                                    kvstore_collections_clean_empty,
                                )
                            except KeyError as e:
                                # Handle case where collection is not in collections_restore_dict
                                error_msg = f'Collection "{collection_name}" not found in backup data, skipping restore. This may be due to the collection being excluded from backups (e.g., stateful charts collections).'
                                logger.warning(error_msg)
                                continue
                            except Exception as e:
                                # Handle any other errors during restore
                                error_msg = f'Failed to restore collection "{collection_name}": {str(e)}'
                                logger.error(error_msg)
                                continue

                            # add to dict
                            kvstore_collections_restore_summary_dict[
                                collection_name
                            ] = kvstore_collection_restore_summary_dict

                #######################################
                # End: clean up the temporary directory
                #######################################

                clean_backup_dir(backupdir)

            # investigate the results of the kvstore records restoration
            if (
                kvstore_collections_global_records_to_be_restored
                == kvstore_collections_global_records_restored
            ):
                kvstore_records_restore_summary = {
                    "response": "KVstore records restore operation is now complete and no errors were reported",
                    "action": "success",
                    "total_records_to_be_restored": kvstore_collections_global_records_to_be_restored,
                    "total_records_restored": kvstore_collections_global_records_restored,
                    "kvstore_collections_restore_summary_dict": kvstore_collections_restore_summary_dict,
                }

            else:
                kvstore_records_restore_summary = {
                    "response": "KVstore records restore operation is now complete but some errors were reported",
                    "action": "warning",
                    "message": "one or more KVstore collection could not be fully restored",
                    "total_records_to_be_restored": kvstore_collections_global_records_to_be_restored,
                    "total_records_restored": kvstore_collections_global_records_restored,
                    "kvstore_collections_restored_warning": kvstore_collections_restored_warning,
                    "kvstore_collections_restore_summary_dict": kvstore_collections_restore_summary_dict,
                }

            ##########################################
            # task: force clean up any disable tenants
            ##########################################

            # main vtenant collection
            collection_vtenants_name = "kv_trackme_virtual_tenants"
            collection_vtenants = service.kvstore[collection_vtenants_name]

            # Get records
            vtenants_records, vtenants_collection_keys, vtenants_collection_dict = (
                get_full_kv_collection(collection_vtenants, collection_vtenants_name)
            )

            # Iterate over the Virtual Tenants
            for vtenant_record in vtenants_records:

                # get the tenant_id
                tenant_id = vtenant_record.get("tenant_id")

                # get the status
                tenant_status = vtenant_record.get("tenant_status")

                # only consider disabled tenants
                if tenant_status == "disabled":

                    logger.info(
                        f'Detected a disabled tenant_id="{tenant_id}", processing to the forced deletion'
                    )

                    # url
                    url = f"{request_info.server_rest_uri}/services/trackme/v2/vtenants/admin/del_tenant"

                    try:
                        response = requests.delete(
                            url,
                            headers=header,
                            data=json.dumps({"tenant_id": tenant_id, "force": "true"}),
                            verify=False,
                            timeout=600,
                        )

                        response_json = response.json()
                        tasks_list.append(
                            f'Detected a disabled tenant_id="{tenant_id}", forced deletion of tenant was processed, results: {response_json}'
                        )

                    except Exception as e:
                        error_msg = f'forced deletion of Virtual Tenants reported errors, exception="{str(e)}"'
                        logger.error(error_msg)
                        errors_list.append(error_msg)

            ######################
            # task: final response
            ######################

            # define action value, if errors_list is empty, set success, otherwise set warning
            action = "success" if not errors_list else "warning"
            response_dict["action"] = action

            # define response
            response = (
                "restore operation is now complete and no errors were reported, please reload TrackMe"
                if not errors_list
                else "restore operation is now complete but some errors were reported, please review these issues and reload TrackMe"
            )

            # add backupfile name
            response_dict["backupfile"] = backupfile

            # add the archive schema version
            response_dict["archive_schema_version"] = archive_schema_version

            # add the kvstore records restore results
            response_dict["kvstore_records_restore_summary"] = (
                kvstore_records_restore_summary
            )

            # add tasks list
            response_dict["tasks_executed"] = tasks_list

            # add errors list
            response_dict["tasks_errors"] = errors_list

            logger.info(json.dumps(response_dict, indent=4))
            return {"payload": response_dict, "status": 200}

    def post_import_backup(self, request_info, **kwargs):
        """
        Handles the import_backup functionality.
        """
        describe = False

        # Retrieve the payload
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict.get("describe", False)
                if str(describe).lower() == "true":
                    describe = True
            except Exception:
                describe = False

        if describe:
            response = {
                "describe": "This endpoint imports a backup from a Base64-encoded compressed archive file (.tgz or .tar.zst).",
                "resource_desc": "Import a backup archive",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/backup_and_restore/import_backup\" body=\"{'archive_base64': '<base64_string>'}\" (supports both .tgz and .tar.zst)",
                "options": [
                    {
                        "archive_base64": "(string) REQUIRED: Base64 string of the backup archive."
                    },
                ],
            }
            return {"payload": response, "status": 200}

        # Process the import
        try:
            archive_base64 = resp_dict.get("archive_base64")
            if not archive_base64:
                return {
                    "payload": {"error": "archive_base64 is required"},
                    "status": 400,
                }

            # Decode base64 and write to a temporary file
            decoded_data = base64.b64decode(archive_base64)
            temp_dir = os.path.join(splunkhome, "etc", "apps", "trackme", "backup")
            os.makedirs(temp_dir, exist_ok=True)

            # Determine file extension based on content (try to detect format)
            # For now, we'll use a generic name and let the extract function handle it
            temp_file_path = os.path.join(temp_dir, "temp_import.archive")

            with open(temp_file_path, "wb") as temp_file:
                temp_file.write(decoded_data)

            # Try to determine archive type and validate
            archive_type = None
            if temp_file_path.endswith(".tar.zst") or temp_file_path.endswith(".zst"):
                archive_type = "zstd"
            elif temp_file_path.endswith(".tgz") or temp_file_path.endswith(".tar.gz"):
                archive_type = "gzip"
            else:
                # Try to detect by content
                try:
                    # Try zstd first
                    if is_zstd_available():
                        result = subprocess.run(
                            ["zstd", "-t", temp_file_path], capture_output=True
                        )
                        if result.returncode == 0:
                            archive_type = "zstd"
                        else:
                            # Try gzip
                            if tarfile.is_tarfile(temp_file_path):
                                archive_type = "gzip"
                except:
                    # Fall back to gzip check
                    if tarfile.is_tarfile(temp_file_path):
                        archive_type = "gzip"

            if not archive_type:
                os.remove(temp_file_path)
                return {
                    "payload": {"error": "Invalid or unsupported archive format"},
                    "status": 400,
                }

            # Validate the archive content
            if archive_type == "zstd":
                # For zstd, we'll validate during extraction
                # We need to extract to a temporary location to check metadata
                temp_extract_dir = os.path.join(temp_dir, "temp_extract")
                os.makedirs(temp_extract_dir, exist_ok=True)

                # Rename the temp file to have the proper extension for extract_archive
                temp_file_with_ext = os.path.join(temp_dir, "temp_import.tar.zst")
                os.rename(temp_file_path, temp_file_with_ext)

                if not extract_archive(temp_file_with_ext, temp_extract_dir):
                    shutil.rmtree(temp_extract_dir, ignore_errors=True)
                    os.remove(temp_file_with_ext)
                    return {
                        "payload": {"error": "Failed to extract zstd archive"},
                        "status": 400,
                    }

                # Check metadata files
                members = os.listdir(temp_extract_dir)
                full_metadata_found = False
                light_metadata_found = False
                archive_name = None

                for member in members:
                    if member.startswith("trackme-backup-") and member.endswith(
                        "full.meta"
                    ):
                        full_metadata_found = True
                        # extract the archive_name from the full metadata file
                        archive_name = member.split(".full.meta")[0]
                        break

                for member in members:
                    if member.startswith("trackme-backup-") and member.endswith(
                        "light.meta"
                    ):
                        light_metadata_found = True
                        break

                if not full_metadata_found or not light_metadata_found:
                    shutil.rmtree(temp_extract_dir, ignore_errors=True)
                    os.remove(temp_file_with_ext)
                    return {
                        "payload": {
                            "error": f"Missing metadata files, this archive is corrupted, or not a TrackMe backup, or not created with TrackMe 2.1.5 and later (only archive_schema_version 2.0.0 and later can be imported), see available archive content: {members}"
                        },
                        "status": 400,
                    }

                # Clean up temp extraction
                shutil.rmtree(temp_extract_dir, ignore_errors=True)

            elif archive_type == "gzip":
                # Rename the temp file to have the proper extension for tarfile
                temp_file_with_ext = os.path.join(temp_dir, "temp_import.tgz")
                os.rename(temp_file_path, temp_file_with_ext)

                with tarfile.open(temp_file_with_ext, "r:gz") as tar:
                    members = tar.getnames()
                    full_metadata_found = False
                    light_metadata_found = False
                    archive_name = None

                    # check for metadata files
                    for member in members:
                        if member.startswith("trackme-backup-") and member.endswith(
                            "full.meta"
                        ):
                            full_metadata_found = True
                            # extract the archive_name from the full metadata file
                            archive_name = member.split(".full.meta")[0]
                            break

                    for member in members:
                        if member.startswith("trackme-backup-") and member.endswith(
                            "light.meta"
                        ):
                            light_metadata_found = True
                            break

                    if not full_metadata_found or not light_metadata_found:
                        os.remove(temp_file_with_ext)
                        return {
                            "payload": {
                                "error": f"Missing metadata files, this archive is corrupted, or not a TrackMe backup, or not created with TrackMe 2.1.5 and later (only archive_schema_version 2.0.0 and later can be imported), see available archive content: {members}"
                            },
                            "status": 400,
                        }

            # Extract the archive in backupdir
            try:
                backupdir = os.path.join(splunkhome, "etc", "apps", "trackme", "backup")
                os.makedirs(backupdir, exist_ok=True)

                # Use the renamed file with proper extension
                final_temp_file = (
                    temp_file_with_ext
                    if "temp_file_with_ext" in locals()
                    else temp_file_path
                )

                if not extract_archive(final_temp_file, backupdir):
                    return {
                        "payload": {"error": f"Failed to extract the archive"},
                        "status": 500,
                    }

                os.remove(final_temp_file)

            except Exception as e:
                final_temp_file = (
                    temp_file_with_ext
                    if "temp_file_with_ext" in locals()
                    else temp_file_path
                )
                os.remove(final_temp_file)
                return {
                    "payload": {
                        "error": f"Failed to extract the archive, error={str(e)}"
                    },
                    "status": 500,
                }

            # Call get_restore

            # url
            url = f"{request_info.server_rest_uri}/services/trackme/v2/backup_and_restore/backup"

            # header
            header = {
                "Authorization": "Splunk %s" % request_info.session_key,
                "Content-Type": "application/json",
            }

            response = requests.get(
                url,
                headers=header,
                verify=False,
                timeout=600,
            )

            if response.status_code != 200:
                return {
                    "payload": {
                        "error": f"Failed to call get_restore, http.status={response.status_code}, http.response={response.text}"
                    },
                    "status": 500,
                }

            else:
                return {
                    "payload": {"success": "Backup imported successfully"},
                    "status": 200,
                }

        except Exception as e:
            logger.error(f"Error in post_import_backup: {str(e)}")
            return {"payload": {"error": str(e)}, "status": 500}

    def post_export_backup(self, request_info, **kwargs):
        """
        Handles the export_backup functionality.
        """
        describe = False

        # Retrieve the payload
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict.get("describe", False)
                if str(describe).lower() == "true":
                    describe = True
            except Exception:
                describe = False

        if describe:
            response = {
                "describe": "This endpoint exports a backup to a Base64-encoded compressed archive file (.tgz or .tar.zst).",
                "resource_desc": "Export a backup archive",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/backup_and_restore/export_backup\" body=\"{'archive_name': 'trackme-backup-20210205-142635.tgz'}\" (supports both .tgz and .tar.zst)",
                "options": [
                    {
                        "archive_name": "(string) REQUIRED: Name of the backup archive to export.",
                        "force_local": "(true / false) OPTIONAL: if true, the endpoint will assume the backup file is hosted on the local server and will not verify if the backup file is hosted on a different server",
                    },
                ],
            }
            return {"payload": response, "status": 200}

        # get archive_name
        archive_name = resp_dict.get("archive_name")
        if not archive_name:
            return {"payload": {"error": "archive_name is required"}, "status": 400}

        # Get the force_local parameter
        try:
            force_local = resp_dict.get("force_local", False)
            if force_local:
                # accept booleans or strings: true/True/1
                if isinstance(force_local, bool):
                    force_local = force_local
                elif isinstance(force_local, str):
                    if force_local in ("true", "True", "1"):
                        force_local = True
                    elif force_local in ("false", "False", "0"):
                        force_local = False
                    else:
                        return {
                            "payload": {"error": "force_local must be true or false"},
                            "status": 400,
                        }
                else:
                    force_local = False

        except Exception as e:
            # default to False
            force_local = False

        # Get current server name (short hostname as default)
        current_server_name = socket.gethostname()

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # Query the backup archives info collection
        collection_name = "kv_trackme_backup_archives_info"
        collection = service.kvstore[collection_name]

        # Query the KVstore collection in all cases
        query_string = {"backup_archive": {"$regex": f".*{archive_name}$"}}

        try:
            kvrecords = collection.data.query(query=json.dumps(query_string))
            kvrecord = kvrecords[0]
        except Exception as e:
            kvrecords = []
            kvrecord = None

        # handle_local boolean
        handle_local = True

        if not kvrecord or force_local:
            handle_local = True

        else:

            # log kvrecord extracts
            kvrecord_extracts = {
                "_key": kvrecord.get("_key"),
                "htime": kvrecord.get("htime"),
                "server_name": kvrecord.get("server_name"),
                "backup_archive": kvrecord.get("backup_archive"),
                "change_type": kvrecord.get("change_type"),
                "status": kvrecord.get("status"),
                "comment": kvrecord.get("comment"),
            }
            logger.info(
                f"found backup info collection record: {json.dumps(kvrecord_extracts, indent=2)}"
            )

            # Get the server_name from the first matching record
            backup_server_name = kvrecord.get("server_name")

            if backup_server_name != current_server_name:
                logger.info(
                    f"Backup archive is on different server ({backup_server_name}), delegating export to target server"
                )
                handle_local = False

        # process
        if handle_local:

            #
            # local host handling
            #

            logger.info(
                f"Backup archive is on the same server ({current_server_name}), proceeding with local export"
            )

            # Local export (original logic)
            # Locate the archive
            backup_dir = os.path.join(splunkhome, "etc", "apps", "trackme", "backup")
            archive_path = os.path.join(backup_dir, archive_name)

            if not os.path.exists(archive_path):
                return {
                    "payload": {
                        "error": f"Archive not found, file {archive_path} does not exists."
                    },
                    "status": 404,
                }

            try:
                # Create a timestamped temporary directory for the export
                timestamp = time.strftime("%d%m%Y_%H%M%S", time.localtime())
                temp_dir = os.path.join(splunkhome, "etc", "apps", "trackme", "backup")
                temp_export_dir = os.path.join(temp_dir, f"backup_temp_{timestamp}")
                
                # Remove existing temp directory if it exists
                if os.path.exists(temp_export_dir):
                    logger.info(f"Removing existing temporary directory: {temp_export_dir}")
                    shutil.rmtree(temp_export_dir, ignore_errors=True)
                
                # Create the new temporary directory
                os.makedirs(temp_export_dir, exist_ok=True)
                logger.info(f"Created temporary export directory: {temp_export_dir}")

                # Copy the archive and metadata files to temp directory
                shutil.copy2(archive_path, os.path.join(temp_export_dir, archive_name))
                shutil.copy2(
                    f"{archive_path}.light.meta",
                    os.path.join(temp_export_dir, f"{archive_name}.light.meta"),
                )
                shutil.copy2(
                    f"{archive_path}.full.meta",
                    os.path.join(temp_export_dir, f"{archive_name}.full.meta"),
                )

                # Create a tgz archive containing the zst archive and metadata files
                archive_name_base = os.path.splitext(archive_name)[
                    0
                ]  # Remove extension
                if archive_name.endswith(".tar.zst"):
                    archive_name_base = archive_name_base[:-4]  # Remove .tar part

                # Create the tgz archive
                tgz_path = os.path.join(temp_dir, f"{archive_name_base}.tgz")
                with tarfile.open(tgz_path, mode="w:gz") as tgz_archive:
                    tgz_archive.add(temp_export_dir, arcname="")

                # get the size of the tgz archive (MB)
                tgz_size = round(os.path.getsize(tgz_path) / 1024 / 1024, 2)

                # log success
                logger.info(
                    f"Successfully exported backup archive {archive_name} to {tgz_path}, size={tgz_size} MB"
                )

                # Encode the tgz archive in Base64
                with open(tgz_path, "rb") as tgz_file:
                    base64_data = base64.b64encode(tgz_file.read()).decode("utf-8")

                # Clean up temporary files
                os.remove(tgz_path)
                shutil.rmtree(temp_export_dir, ignore_errors=True)
                logger.info(f"Cleaned up temporary export directory: {temp_export_dir}")

                return {
                    "payload": {"archive_base64": base64_data},
                    "status": 200,
                }

            except Exception as e:
                logger.error(f"Error in local export: {str(e)}")
                return {
                    "payload": {"error": f"Failed to export backup: {str(e)}"},
                    "status": 500,
                }

        else:

            #
            # remote host handling
            #

            # Query the KVstore collection to get the server_name for this backup archive
            logger.info(
                f"Backup archive is on different server ({backup_server_name}), delegating export to target server"
            )

            # Determine the best target server name for communication by testing connectivity
            target_server_name = backup_server_name

            # Test connectivity with short hostname first, then FQDN if needed
            logger.info(
                f"Testing connectivity with short hostname: {backup_server_name}"
            )
            if not test_splunkd_connectivity(
                backup_server_name,
                request_info.server_rest_port,
                request_info.session_key,
            ):
                # Try with FQDN
                backup_server_fqdn = (
                    socket.getfqdn()
                    if backup_server_name == socket.gethostname()
                    else f"{backup_server_name}.{socket.getfqdn().split('.', 1)[1] if '.' in socket.getfqdn() else 'local'}"
                )
                logger.info(f"Short hostname failed, trying FQDN: {backup_server_fqdn}")
                if test_splunkd_connectivity(
                    backup_server_fqdn,
                    request_info.server_rest_port,
                    request_info.session_key,
                ):
                    target_server_name = backup_server_fqdn
                    logger.info(
                        f"FQDN connectivity successful, using: {target_server_name}"
                    )
                else:
                    logger.warning(
                        f"Both short hostname and FQDN failed for {backup_server_name}"
                    )
            else:
                logger.info(
                    f"Short hostname connectivity successful, using: {target_server_name}"
                )

            # support only https
            target_server_uri = (
                f"https://{target_server_name}:{request_info.server_rest_port}"
            )

            # Make REST call to target server
            headers = {
                "Authorization": f"Splunk {request_info.session_key}",
                "Content-Type": "application/json",
            }

            # Prepare the request payload
            request_payload = {
                "archive_name": archive_name,
                "force_local": force_local,
            }

            target_url = f"{target_server_uri}/services/trackme/v2/backup_and_restore/export_backup"

            logger.info(f"Making REST call to target server: {target_url}")

            try:
                response = requests.post(
                    target_url,
                    headers=headers,
                    data=json.dumps(request_payload),
                    verify=False,
                    timeout=600,
                )

                if response.status_code == 200:
                    response_data = response.json()
                    logger.info(
                        f"Successfully exported backup from target server {backup_server_name}"
                    )
                    base64_data = response_data.get("archive_base64")
                    return {
                        "payload": {"archive_base64": base64_data},
                        "status": 200,
                    }

                else:
                    logger.error(
                        f"Target server {backup_server_name} returned error: {response.status_code} - {response.text}, url={target_url}, request_payload={json.dumps(request_payload)}"
                    )
                    return {
                        "payload": {
                            "error": f"Failed to export from target server {backup_server_name}: {response.text}, url={target_url}, request_payload={json.dumps(request_payload)}"
                        },
                        "status": response.status_code,
                    }

            except requests.exceptions.ConnectionError as e:
                logger.error(
                    f"Connection error to target server {backup_server_name}: {str(e)}"
                )
                return {
                    "payload": {
                        "error": f"Failed to connect to target server {backup_server_name}. Please ensure the server is reachable and TrackMe is installed."
                    },
                    "status": 500,
                }
            except Exception as e:
                logger.error(
                    f"Error making REST call to target server {backup_server_name}: {str(e)}, url={target_url}, request_payload={json.dumps(request_payload)}"
                )
                return {
                    "payload": {
                        "error": f"Failed to export from target server {backup_server_name}: {str(e)}, url={target_url}, request_payload={json.dumps(request_payload)}"
                    },
                    "status": 500,
                }
