# encoding = utf-8

import os
import sys
import requests
import json
import urllib3
import time
import hashlib
from splunklib.modularinput.event import Event, ET
from splunklib.modularinput.event_writer import EventWriter
from trackme_libs import trackme_reqinfo, trackme_idx_for_tenant
import splunklib.client as client

# Disable urllib3 warnings for SSL
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# set splunkhome
splunkhome = os.environ["SPLUNK_HOME"]

# append libs
sys.path.append(os.path.join(splunkhome, "etc", "apps", "trackme", "lib"))


def process_event(helper, *args, **kwargs):
    helper.log_info("Alert action trackme_smart_status started.")

    # Retrieve the session_key
    helper.log_debug("Get session_key.")
    session_key = helper.session_key

    # Retrieve the server_uri
    helper.log_debug("Get server_uri.")
    server_uri = helper.settings["server_uri"]

    # Get request info and set logging level
    reqinfo = trackme_reqinfo(session_key, server_uri)

    # Get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=session_key,
        timeout=600,
    )

    # Build header and target
    header = {
        "Authorization": f"Splunk {session_key}",
        "Content-Type": "application/json",
    }

    # Get fields info
    tenant_id = helper.get_param("tenant_id")
    helper.log_info(f"tenant_id={tenant_id}")

    object_category = helper.get_param("object_category")
    helper.log_info(f"object_category={object_category}")

    object_name = helper.get_param("object_name")
    helper.log_info(f"object_name={object_name}")

    # Define Meta
    splunk_sourcetype = "trackme:smart_status"
    splunk_host = reqinfo["server_hostname"]

    category_to_source_mapping = {
        "splk-dsm": "ds_smart_status",
        "splk-dhm": "dh_smart_status",
        "splk-mhm": "mh_smart_status",
        "splk-cim": "cim_smart_status",
        "splk-flx": "flx_smart_status",
        "splk-fqm": "fqm_smart_status",
        "splk-wlk": "wlk_smart_status",
    }

    splunk_source = category_to_source_mapping.get(str(object_category))
    endpoint_url = f"/services/trackme/v2/splk_smart_status/{splunk_source}"

    helper.log_info(f"splunk_source={splunk_source}")

    # Target
    target_url = f"{server_uri}/{endpoint_url}"

    # Get the index conf for this tenant
    tenant_idx = trackme_idx_for_tenant(session_key, server_uri, tenant_id)
    tenant_trackme_summary_idx = tenant_idx["trackme_summary_idx"]

    # Last seen action, we use this KVstore to store the last execution of the SmartStatus and prevent from running too many investigations over time
    # notably if the alert was setup incorrectly
    collection_last_seen_activity_name = f"kv_trackme_common_smartstatus_alert_action_last_seen_activity_tenant_{tenant_id}"
    collection_last_seen_activity = service.kvstore[collection_last_seen_activity_name]

    # Define the KV query
    query_string = {
        "object_category": object_category,
        "object": object_name,
    }

    # Initialize kvrecord to None to prevent "referenced before assignment" error
    kvrecord = None

    # Get the current record
    try:
        kvrecord = collection_last_seen_activity.data.query(
            query=json.dumps(query_string)
        )[0]
        key = kvrecord.get("_key")
        last_seen_activity = kvrecord.get("last_seen_activity")

        # calculate time_since_last_activity, last_seen_activity is an epochtime
        time_since_last_activity = time.time() - int(last_seen_activity)

    except Exception as e:
        key = None

    # Smart Status API call

    # proceed boolean
    proceed = True

    if key:
        # If the last action recorded is less than 24 hours old, we skip the action and log an informational message instead
        if time_since_last_activity < 86400:
            helper.log_info(
                f"Skipping SmartStatus action for object_category={object_category}, object={object_name}, time_since_last_activity={time_since_last_activity}, we will not run the action again to prevent from unwanted worklaod in this environment."
            )
            proceed = False
            return 0

    if proceed:

        # action is granted, update or create the kvstore record
        try:
            if key:
                collection_last_seen_activity.data.update(
                    str(key),
                    json.dumps(
                        {
                            "object_category": object_category,
                            "object": object_name,
                            "last_seen_activity": int(time.time()),
                        }
                    ),
                )
            else:
                collection_last_seen_activity.data.insert(
                    json.dumps(
                        {
                            "object_category": object_category,
                            "object": object_name,
                            "last_seen_activity": int(time.time()),
                        }
                    )
                )
        except Exception as e:
            helper.log_error(
                f'Failed to update the KVstore record in collection="{collection_last_seen_activity_name}", exception="{str(e)}'
            )
            return 1

        try:
            record = {
                "tenant_id": tenant_id,
                "object_category": object_category,
                "object": object_name,
            }

            helper.log_info(
                f'Running SmartStatus for record="{json.dumps(record, indent=2)}", target_url="{target_url}"'
            )

            response = requests.post(
                target_url,
                headers=header,
                verify=False,
                data=json.dumps(record),
                timeout=600,
            )
            if response.status_code == 200:
                responseJson = json.loads(response.text)
                helper.log_debug(f"http_response={json.dumps(responseJson)}")
            else:
                helper.log_error(
                    f'TrackMe Smart Status action has failed, http_error_code="{response.status_code}", http_response="{json.dumps(response.text)}"'
                )
                return 1
        except Exception as e:
            helper.log_error(f"TrackMe Smart Status alert action has failed!:{e}")
            return 1

        # Index the response
        responseJson = json.loads(response.text)

        # Add the event_id
        event_id = hashlib.sha256(json.dumps(responseJson).encode()).hexdigest()
        responseJson["event_id"] = event_id

        # Index the audit record
        try:
            target = service.indexes[tenant_trackme_summary_idx]
            target.submit(
                event=json.dumps(responseJson),
                source=str(splunk_source),
                sourcetype=str(splunk_sourcetype),
                host=str(splunk_host),
            )
            helper.log_info(
                f'tenant_id="{tenant_id}", object_category="{object_category}", object="{object_name}", response="SmartStatus event created successfully"'
            )
            return 0
        except Exception as e:
            helper.log_info(
                f'tenant_id="{tenant_id}", object_category="{object_category}", object="{object_name}", response="SmartStatus event creation failure", exception="{e}"'
            )
            return 1
