#!/usr/bin/env python
# coding=utf-8

__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

import os
import sys
import time
import json
import logging
from logging.handlers import RotatingFileHandler
import urllib3

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# set logging
filehandler = RotatingFileHandler(
    "%s/var/log/splunk/trackme_splk_outliers_cim_set_rules.log" % splunkhome,
    mode="a",
    maxBytes=10000000,
    backupCount=1,
)
formatter = logging.Formatter(
    "%(asctime)s %(levelname)s %(filename)s %(funcName)s %(lineno)d %(message)s"
)
logging.Formatter.converter = time.gmtime
filehandler.setFormatter(formatter)
log = logging.getLogger()  # root logger - Good to get it only once.
for hdlr in log.handlers[:]:  # remove the existing file handlers
    if isinstance(hdlr, logging.FileHandler):
        log.removeHandler(hdlr)
log.addHandler(filehandler)  # set the new handler
# set the log level to INFO, DEBUG as the default is ERROR
log.setLevel(logging.INFO)

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# import Splunk lib
from splunklib.searchcommands import (
    dispatch,
    GeneratingCommand,
    Configuration,
    Option,
    validators,
)

# Import trackme libs
from trackme_libs import trackme_reqinfo

# import trackme splk-cim libs
from trackme_libs_splk_cim import trackme_cim_splk_outliers_set_rule


@Configuration(distributed=False)
class SplkOutliersSetRules(GeneratingCommand):
    tenant_id = Option(
        doc="""
        **Syntax:** **tenant_id=****
        **Description:** The value for tenant_id.""",
        require=True,
        validate=validators.Match("tenant_id", r"^.*$"),
    )

    object = Option(
        doc="""
        **Syntax:** **object=****
        **Description:** The value for object.""",
        require=True,
        validate=validators.Match("object", r"^.*$"),
    )

    def generate(self, **kwargs):
        # performance counter
        start = time.time()

        # Get request info and set logging level
        reqinfo = trackme_reqinfo(
            self._metadata.searchinfo.session_key, self._metadata.searchinfo.splunkd_uri
        )
        log.setLevel(reqinfo["logging_level"])

        # Get the session key
        session_key = self._metadata.searchinfo.session_key

        # the KVstore collection and its service object
        collection_name = "kv_trackme_cim_tenant_" + str(self.tenant_id)
        collection = self.service.kvstore[collection_name]

        # Define the KV query
        query_string = {
            "object": self.object,
        }

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        key = None
        try:
            record = collection.data.query(query=json.dumps(query_string))
            key = record[0].get("_key")

        except Exception as e:
            key = None
            raise Exception(
                f'Could not find a KVstore record for the object="{self.object}"'
            )

        # if the record is found:
        if key:
            # load the cim_tracking_rules object
            try:
                cim_tracking_rules = json.loads(record[0].get("cim_tracking_rules"))
            except Exception as e:
                raise Exception(
                    f'Failed to load the cim_tracking_rules for object="{self.object}" with exception="{str(e)}"'
                )

            # load the cim_tracking_results object
            try:
                cim_tracking_results = json.loads(record[0].get("cim_tracking_results"))
            except Exception as e:
                raise Exception(
                    f'Failed to load the cim_tracking_results for object="{self.object}" with exception="{str(e)}"'
                )

            # get the cim_breakby
            try:
                cim_breakby = cim_tracking_rules.get("cim_breakby")
            except Exception as e:
                raise Exception(
                    f'Failed to load the cim_breakby value for object="{self.object}" with exception="{str(e)}"'
                )

            # get the list of cim fields
            try:
                cim_fields_list = cim_tracking_rules.get("cim_fields").split(",")
            except Exception as e:
                raise Exception(
                    f'Failed to load the list of cim fields for object="{self.object}" with exception="{str(e)}"'
                )

            # get the list of cim_entities
            try:
                entities_list = cim_tracking_results.get("cim_entities").split(",")
            except Exception as e:
                raise Exception(
                    f'Failed to load the list of cim entities for object="{self.object}" with exception="{str(e)}"'
                )

            #
            # ML models set rules: per entity / cim_field
            #

            results = []

            for entity in entities_list:
                # loop through the list of CIM fields, check and create ML models if necessary
                for field in cim_fields_list:
                    # check outliers rules, and create if needed
                    try:
                        result = trackme_cim_splk_outliers_set_rule(
                            session_key,
                            reqinfo["server_rest_port"],
                            reqinfo,
                            self.tenant_id,
                            self.object,
                            cim_breakby,
                            entity,
                            field,
                        )
                        results.append(result)

                    except Exception as e:
                        msg = f'tenant_id="{self.tenant_id}", object="{self.object}", cim_field="{field}", Failed to call trackme_cim_splk_outliers_set_rule with exception="{str(e)}"'
                        logging.error(msg)
                        raise Exception(msg)

            logging.info(
                f'tenant_id="{self.tenant_id}", object="{self.object}", results="{results}"'
            )

            yield {
                "_time": time.time(),
                "_raw": {"action": "success", "results": results},
                "action": "success",
                "results": results,
            }

        # Log the run time
        logging.info(
            f"trackmesplkoutlierscimsetrules has terminated, run_time={round(time.time() - start, 3)}"
        )


dispatch(SplkOutliersSetRules, sys.argv, sys.stdin, sys.stdout, __name__)
