#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_sla_policies.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_sla_policies_user",
    "trackme_rest_api_splk_sla_policies_user.log",
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel, trackme_reqinfo

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkSlaPoliciesRead_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkSlaPoliciesRead_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_sla_policies(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_sla_policies",
            "resource_group_desc": "Endpoints related to the management of SLA classes through policies (read only operations)",
        }

        return {"payload": response, "status": 200}

    # get SLA classes definition
    def get_sla_classes_show(self, request_info, **kwargs):

        # Declare
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

        else:
            describe = False

        if describe:
            response = {
                "describe": "This endpoint shows current SLA classes definitions, it requires a GET call:",
                "resource_desc": "Get SLA classes definitions",
                "resource_spl_example": '| trackme mode=get url="/services/trackme/v2/splk_sla_policies/sla_classes_show"',
                "options": [],
            }

            return {"payload": response, "status": 200}

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get trackmeconf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )["trackme_conf"]

        # Get SLA classes conf
        sla_classes = trackme_conf["sla"]["sla_classes"]

        # A list of SLA classes
        sla_classes_list = []

        # try loading the JSON
        try:
            sla_classes = json.loads(sla_classes)
            for sla_class in sla_classes:
                sla_classes_list.append(sla_class)

            # render response
            return {
                "payload": {
                    "response": {
                        "sla_classes": sla_classes_list,
                        "sla_classes_definitions": sla_classes,
                    },
                    "status": 200,
                },
                "status": 200,
            }

        except:
            error_msg = f'Error loading sla_classes JSON, please check the configuration, the JSON is not valid JSON, exception="{str(e)}"'
            logger.error(error_msg)
            return {
                "payload": {
                    "response": f"{error_msg}",
                    "status": 500,
                },
                "status": 500,
            }

    # get SLA classes definition
    def get_sla_classes_show_list(self, request_info, **kwargs):

        # Declare
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

        else:
            describe = False

        if describe:
            response = {
                "describe": "This endpoint returns the list of current SLA classes definitions, it requires a GET call:",
                "resource_desc": "Get SLA classes definitions",
                "resource_spl_example": '| trackme mode=get url="/services/trackme/v2/splk_sla_policies_list/sla_classes_show"',
                "options": [],
            }

            return {"payload": response, "status": 200}

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get trackmeconf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )["trackme_conf"]

        # Get SLA classes conf
        sla_classes = trackme_conf["sla"]["sla_classes"]

        # A list of SLA classes
        sla_classes_list = []

        # try loading the JSON
        try:
            sla_classes = json.loads(sla_classes)
            for sla_class in sla_classes:
                sla_classes_list.append(sla_class)

            # render response
            return {
                "payload": sla_classes_list,
                "status": 200,
            }

        except:
            error_msg = f'Error loading sla_classes JSON, please check the configuration, the JSON is not valid JSON, exception="{str(e)}"'
            logger.error(error_msg)
            return {
                "payload": {
                    "response": f"{error_msg}",
                    "status": 500,
                },
                "status": 500,
            }

    # get all records
    def post_sla_policies_show(self, request_info, **kwargs):

        # Declare
        tenant_id = None
        component = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "tenant_id is required",
                        },
                        "status": 400,
                    }
                try:
                    component = resp_dict["component"]
                    if component not in (
                        "dsm",
                        "dhm",
                        "mhm",
                        "wlk",
                        "flx",
                        "fqm",
                    ):
                        return {
                            "payload": {
                                "action": "failure",
                                "response": f"invalid component {component}, valid options are: dsm/dhm/mhm/wlk/flx/fqm",
                            },
                            "status": 400,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "component is required",
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint retrieves all records for the SLA classes policies collection, it requires a POST call with the following information:",
                "resource_desc": "Get SLA classes policies",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_sla_policies/sla_policies_show\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, valid values are: dsm/dhm/mhm/wlk/flx/fqm",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Data collection
            collection_name = f"kv_trackme_{component}_sla_policies_tenant_{tenant_id}"
            collection = service.kvstore[collection_name]

            return {"payload": collection.data.query(), "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Get model
    def post_sla_policies_by_id(self, request_info, **kwargs):

        # By id
        tenant_id = None
        component = None
        sla_class_policy_id = None

        # query_string to find records
        query_string = None

        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "tenant_id is required",
                        },
                        "status": 400,
                    }
                try:
                    component = resp_dict["component"]
                    if component not in (
                        "dsm",
                        "dhm",
                        "mhm",
                        "wlk",
                        "flx",
                        "fqm",
                    ):
                        return {
                            "payload": {
                                "action": "failure",
                                "response": f"invalid component {component}, valid options are: dsm/dhm/mhm/wlk/flx/fqm",
                            },
                            "status": 400,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "component is required",
                        },
                        "status": 400,
                    }
                try:
                    sla_class_policy_id = resp_dict["sla_class_policy_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "sla_class_policy_id is required",
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint retrieves a tag policy by its id, it requires a GET call with the following data:",
                "resource_desc": "Get a given tag policy",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_sla_policies/sla_policies_by_id\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'sla_class_policy_id': 'pan:traffic'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, valid values are: dsm/dhm/mhm/wlk/flx/fqm",
                        "sla_class_policy_id": "(required) ID of the sla policy",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Define the KV query
        query_string = {
            "sla_class_policy_id": sla_class_policy_id,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            collection_name = f"kv_trackme_{component}_sla_policies_tenant_{tenant_id}"
            collection = service.kvstore[collection_name]

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                return {"payload": collection.data.query_by_id(key), "status": 200}

            else:
                response = {
                    "action": "failure",
                    "response": f'sla_class_policy_id="{sla_class_policy_id}" not found',
                }
                logger.error(json.dumps(response))
                return {"payload": response, "status": 404}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}
