# encoding = utf-8

import os, sys
import requests
import json
import urllib3
import time

# disable urllib3 warnings for SSL
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# set splunkhome
splunkhome = os.environ["SPLUNK_HOME"]

# append libs
sys.path.append(os.path.join(splunkhome, "etc", "apps", "trackme", "lib"))


def process_event(helper, *args, **kwargs):
    helper.log_info("Alert action trackme_auto_ack started.")

    # Retrieve the session_key
    helper.log_debug("Get session_key.")
    session_key = helper.session_key

    # Retrieve the server_uri
    helper.log_debug("Get server_uri.")
    server_uri = helper.settings["server_uri"]

    ack_period = helper.get_param("ack_period")
    helper.log_debug(f"ack_period={ack_period}")

    ack_type = helper.get_param("ack_type")
    helper.log_debug(f"ack_type={ack_type}")

    # build header
    header = {
        "Authorization": f"Splunk {session_key}",
        "Content-Type": "application/json",
    }

    # Loop through the events
    for event in helper.get_events():

        # tenant_id
        tenant_id = event.get("tenant_id")

        # object
        object_value = event.get("object")

        # object_category
        object_category = event.get("object_category")

        # object_state
        object_state = event.get("object_state", "green")

        #
        # Check in_alerting_state
        #

        if object_state in ("red", "orange"):
            in_alerting_state = True
        else:
            in_alerting_state = False

        # skip processing if the obkect not in alerting state
        if not in_alerting_state:
            helper.log_info(
                f"tenant_id={tenant_id}, object_category={object_category}, object={object_value}, object_state={object_state}, Entity is not in alerting state (red or orange), skipping processing"
            )
            continue

        #
        # Check existing Ack
        #

        # Verify if we already have an ack for this object
        # check if the ack is active for the object
        ack_active = False

        # run a POST call against the /services/trackme/v2/ack/get_ack_for_object endpoint
        try:
            ack_response = requests.post(
                f"{server_uri}/services/trackme/v2/ack/get_ack_for_object",
                headers=header,
                data=json.dumps(
                    {
                        "tenant_id": tenant_id,
                        "object_category": object_category,
                        "object_list": object_value,
                    }
                ),
                verify=False,
                timeout=600,
            )
            ack_response.raise_for_status()
            ack_response_json = ack_response.json()
            helper.log_debug(f"Ack response: {json.dumps(ack_response_json, indent=2)}")

            if ack_response_json:
                ack_response_json = ack_response_json[0]
                ack_active_string = ack_response_json.get("ack_state", "inactive")
                ack_mtime = float(ack_response_json.get("ack_mtime", time.time()))
                if ack_active_string == "active":
                    ack_age = time.time() - ack_mtime
                    ack_active = True
                    helper.log_info(
                        f"tenant_id={tenant_id}, object_category={object_category}, object={object_value}, Ack is active, ack_age={ack_age:.2f} seconds, ack_response={json.dumps(ack_response_json, indent=2)}"
                    )
                else:
                    ack_active = False
                    helper.log_info(
                        f"tenant_id={tenant_id}, object_category={object_category}, object={object_value}, Ack is not active"
                    )

        except Exception as e:
            helper.log_error(
                f"Error getting Ack record: exception={str(e)}, tenant_id={tenant_id}, object_category={object_category}, object={object_value}"
            )

        # if ack is active, skip the rest of the processing for this object as we have nothing to do
        if ack_active:
            helper.log_info(
                f"tenant_id={tenant_id}, object_category={object_category}, object={object_value}, Ack is active, skipping processing"
            )
            continue

        #
        # Create new Ack
        #

        # endpoint url
        endpoint_url = "/services/trackme/v2/ack/write/ack_manage"

        # test
        anomaly_reason = event.get("anomaly_reason")
        # amnomaly_reason can be a multivalue field, it will appear as values separated by \n, split it as a list
        anomaly_reason = anomaly_reason.split("\n")

        # build body
        body = {
            "tenant_id": tenant_id,
            "action": "enable",
            "object_category": object_category,
            "object_list": object_value,
            "anomaly_reason": anomaly_reason,
            "ack_period": str(ack_period),
            "ack_type": str(ack_type),
            "ack_source": "auto_ack",
            "update_comment": "alert action auto-acknowledgement",
        }

        helper.log_info(
            f'tenant_id="{tenant_id}", object_category={object_category}, object={object_value}, Running TrackMe auto-acknowledgement, body="{json.dumps(body, indent=1)}"'
        )

        # target url
        target_url = f"{server_uri}/{endpoint_url}"

        # Post to Ack endpoint
        try:
            response = requests.post(
                target_url,
                headers=header,
                verify=False,
                data=json.dumps(body),
                timeout=600,
            )
            if response.status_code == 200:
                helper.log_info(
                    f'tenant="{tenant_id}", object_category={object_category}, object={object_value}, response="{json.dumps(json.loads(response.text))}"'
                )
                return 0
            else:
                helper.log_error(
                    f'tenant_id="{tenant_id}", object_category={object_category}, object={object_value}, TrackMe auto acknowledgement action has failed, http_error_code="{response.status_code}", data="{json.dumps(body, indent=2)}", response="{response.text}"'
                )
        except Exception as e:
            helper.log_error(
                f'tenant_id="{tenant_id}", object_category={object_category}, object={object_value}, TrackMe auto acknowledgement alert action has failed, exception="{e}"'
            )

    #
    # End of processing
    #

    return 0
