#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_identity_cards.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2022-2026, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_identity_cards_power",
    "trackme_rest_api_splk_identity_cards_power.log",
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel, trackme_audit_event

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkIdentityCardsWrite_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkIdentityCardsWrite_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_identity_cards(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_identity_cards/write",
            "resource_group_desc": "Endpoints related to the management of Data identity cards (power operations)",
        }

        return {"payload": response, "status": 200}

    # Add a new card
    def post_identity_cards_add_card(self, request_info, **kwargs):
        # define
        tenant_id = None
        doc_link = None
        doc_note = None
        object_list = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                doc_link = resp_dict["doc_link"]
                # doc_note is optional
                try:
                    doc_note = resp_dict["doc_note"]
                except Exception as e:
                    doc_note = None
                # object_list is optional
                try:
                    object_list = resp_dict["object_list"]
                    # turn as a list
                    object_list = object_list.split(",")
                except Exception as e:
                    object_list = None

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint creates a new identity card that can later on be associated with one or more data sources "
                + "(if the card based on the doc_link does not exist it is created, if the card exists already, the doc_link and doc_note are updated "
                + "and the definition of object is preserved), it requires a POST call with the following data required:",
                "resource_desc": "Create a new identity card to be associated with objects",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_identity_cards/write/identity_cards_add_card\" body=\"{'tenant_id': 'mytenant', 'doc_link': 'https://www.mylink.com/mypage', 'doc_note': 'In case of issue with this feed, consult this documentation'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "doc_link": "href link of the identity card",
                        "doc_note": "OPTIONAL: documentation note",
                        "object_list": "OPTIONAL: a comma separated list of entities to be associated with this identity card",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Retrieve from data
        resp_dict = json.loads(str(request_info.raw_args["payload"]))

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Define the KV query
            query_string = {
                "doc_link": doc_link,
            }

            # collection
            collection_name = "kv_trackme_dsm_knowledge_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None
                record = json.dumps(
                    {"doc_link": str(doc_note), "doc_note": str(doc_note)}
                )

            # Render result

            # Proceed
            try:
                if key:
                    action_desc = "updated"

                    # if a list of object was submitted too, add each of these objects to the current record

                    # get current list of associated objects with that card, if any
                    object_new_list = []
                    try:
                        object_current_list = record[0].get("object")
                    except Exception as e:
                        object_current_list = None

                    # Loop if we have a value
                    if object_current_list:
                        if isinstance(object_current_list, list):
                            for object_value in object_current_list:
                                object_new_list.append(object_value)
                        else:
                            object_new_list.append(record[0].get("object"))

                    # if was requested
                    if object_list:
                        for object_value in object_list:
                            if object_value not in object_new_list:
                                object_new_list.append(object_value)

                    # Update the record
                    collection.data.update(
                        str(key),
                        json.dumps(
                            {
                                "object": record[0].get("object"),
                                "doc_link": str(doc_link),
                                "doc_note": str(doc_note),
                                "object": object_new_list,
                            }
                        ),
                    )

                else:
                    action_desc = "created"
                    # Insert the record
                    collection.data.insert(
                        json.dumps(
                            {
                                "doc_link": str(doc_link),
                                "doc_note": str(doc_note),
                                "object": object_list,
                            }
                        )
                    )

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "update identity card",
                        str(doc_link),
                        "splk-dsm",
                        str(json.dumps(collection.data.query_by_id(key), indent=1)),
                        "The identity card was updated successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'exception while attenpting to register an audit event, exception="{str(e)}"'
                    )

                # render
                response = {
                    "action": "success",
                    "response": f'the doc_link="{doc_link}" was {action_desc} successfully',
                    "record": collection.data.query(query=json.dumps(query_string)),
                }

                # log
                logger.info(json.dumps(response, indent=4))

                return {"payload": response, "status": 200}

            except Exception as e:
                response = {
                    "action": "failure",
                    "response": f'an exception was encountered, exception="{str(e)}"',
                }
                logger.error(json.dumps(response))
                return {"payload": response, "status": 500}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Associate a data source with an identity card
    def post_identity_cards_manage_association(self, request_info, **kwargs):
        """
        | trackme mode=\"post\" url=\"/services/trackme/v2/splk_identity_cards/write/identity_cards_manage_association\" body=\"{'tenant_id': 'mytenant', 'doc_link': 'https://www.mylink.com/mypage', 'action': 'unassociate', 'object_list': 'netscreen:netscreen:firewall,wineventlog:WinEventLog'}\"
        """

        # define
        tenant_id = None
        object_list = None
        doc_link = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                action = resp_dict["action"]
                object_list = resp_dict["object_list"]
                # turns into a list
                object_list = object_list.split(",")
                doc_link = resp_dict["doc_link"]
                try:
                    doc_note = resp_dict["doc_note"]
                except Exception as e:
                    doc_note = None

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint manages the association of objects to an existing identity card, "
                + "it requires a POST call with the following data required:",
                "resource_desc": "Create a new identity card to be associated with objects",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_identity_cards/write/identity_cards_manage_association\" body=\"{'tenant_id': 'mytenant', 'doc_link': 'https://www.mylink.com/mypage', 'action': 'unassociate', 'object_list': 'netscreen:netscreen:firewall,wineventlog:WinEventLog'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "doc_link": "The documentation link to associate these entities with",
                        "doc_note": "OPTIONAL, the documentation note, if action is associate and the identity card does not exist yet, it will be created automatically",
                        "action": "The action to be performed, valid options are: associate | unassociate",
                        "object_list": "Comma separated list of entities to be associated/unassociated with this card",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Retrieve from data
        resp_dict = json.loads(str(request_info.raw_args["payload"]))

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Define the KV query
            query_string = {
                "doc_link": doc_link,
            }

            # collection
            collection_name = "kv_trackme_dsm_knowledge_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            # Get the record
            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result

            # if unassociate but there is no record, then there is nothing we can do
            if not key and action == "unassociate":
                response = {
                    "action": "failure",
                    "response": f'doc_link="{doc_link}" not found',
                }
                logger.error(json.dumps(response))
                return {"payload": response, "status": 404}

            # else if there is no record but action is associate
            elif not key and action == "associate":
                # set new record
                record = {
                    "doc_link": doc_link,
                    "doc_note": doc_note,
                    "object": object_new_list,
                }

                # Update the record
                try:
                    collection.data.insert(json.dumps(record))

                    # render
                    response = {
                        "action": "success",
                        "response": f'A new identity card with doc_link="{doc_link}" was created associated with object_list="{object_list}"',
                        "record": collection.data.query(query=json.dumps(query_string)),
                    }

                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "associate identity card",
                            str(object_value),
                            "splk-dsm",
                            str(json.dumps(collection.data.query_by_id(key), indent=1)),
                            "The identity card was created and associated successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'exception while attenpting to register an audit event, exception="{str(e)}"'
                        )

                except Exception as e:
                    response = {
                        "action": "failure",
                        "response": f'an exception was encountered, exception="{str(e)}"',
                    }
                    logger.error(json.dumps(response))
                    return {"payload": response, "status": 500}

            else:
                # get current list of associated objects with that card, if any
                object_new_list = []
                try:
                    object_current_list = record[0].get("object")
                except Exception as e:
                    object_current_list = None

                # Loop if we have a value
                if object_current_list:
                    if isinstance(object_current_list, list):
                        for object_value in object_current_list:
                            object_new_list.append(object_value)
                    else:
                        object_new_list.append(record[0].get("object"))

                # Loop through the submitted list of object, and act depending on the requested action
                for object_value in object_list:
                    if action == "associate":
                        if object_value not in object_new_list:
                            object_new_list.append(object_value)
                    elif action == "unassociate":
                        if object_value in object_new_list:
                            object_new_list.remove(object_value)

                # set new record
                record = {
                    "doc_link": record[0].get("doc_link"),
                    "doc_note": record[0].get("doc_note"),
                    "object": object_new_list,
                }

                # Update the record
                try:
                    collection.data.update(str(key), record)

                except Exception as e:
                    response = {
                        "action": "failure",
                        "response": f'an exception was encountered, exception="{str(e)}"',
                    }
                    logger.error(json.dumps(response))
                    return {"payload": response, "status": 500}

                # Record an audit change
                for object_value in object_list:
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "associate identity card",
                            str(object_value),
                            "splk-dsm",
                            str(json.dumps(collection.data.query_by_id(key), indent=1)),
                            "The identity card was associated successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'exception while attenpting to register an audit event, exception="{str(e)}"'
                        )

                # render
                if action == "associate":
                    response = {
                        "action": "success",
                        "response": f'the following objects="{object_list}" where successfully associated with the doc_link="{doc_link}"',
                        "record": collection.data.query(query=json.dumps(query_string)),
                    }

                elif action == "unassociate":
                    response = {
                        "action": "success",
                        "response": f'the following objects="{object_list}" where successfully unassociated from the doc_link="{doc_link}"',
                        "record": collection.data.query(query=json.dumps(query_string)),
                    }

                # log
                logger.info(json.dumps(response, indent=4))

                return {"payload": response, "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'general exception, an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # delete identity card
    def post_identity_cards_delete_card(self, request_info, **kwargs):
        """
        | trackme mode=\"post\" url=\"/services/trackme/v2/splk_identity_cards/write/identity_cards_delete_card\" body=\"{'tenant_id': 'mytenant', 'doc_link': 'https://www.mylink.com/mypage'}\"
        """

        # define
        tenant_id = None
        doc_link = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                doc_link = resp_dict["doc_link"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint deletes an idenfity card by the Kvstore key, it requires a DELETE call with the following information:",
                "resource_desc": "Delete an identity card",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_identity_cards/write/identity_cards_delete_card\" body=\"{'tenant_id': 'mytenant', 'doc_link': 'https://www.mylink.com/mypage'}\"",
                "options": [
                    {
                        "key": "KVstore unique identifier for this record",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Define the KV query
            query_string = {
                "doc_link": doc_link,
            }

            # collection
            collection_name = "kv_trackme_dsm_knowledge_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            # Get the record
            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if not key:
                response = {
                    "action": "failure",
                    "response": f'doc_link="{doc_link}" not found',
                }
                logger.error(json.dumps(response))
                return {"payload": response, "status": 404}

            else:
                # delete the record
                try:
                    collection.data.delete(json.dumps({"_key": key}))

                    # Record an audit change
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "delete identity card",
                            str(key),
                            "splk-dsm",
                            str(json.dumps(collection.data.query_by_id(key), indent=1)),
                            "The identity card was deleted successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'exception while attenpting to register an audit event, exception="{str(e)}"'
                        )

                    # render
                    response = {
                        "action": "success",
                        "response": f'record for doc_link="{doc_link}" was successfully deleted',
                        "record": record,
                    }
                    logger.error(json.dumps(response))
                    return {"payload": response, "status": 200}

                except Exception as e:
                    response = {
                        "action": "failure",
                        "response": f'doc_link="{doc_link}" deletion failed with exception="{str(e)}"',
                    }
                    logger.error(json.dumps(response))
                    return {"payload": response, "status": 500}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}
