# encoding = utf-8

import os
import sys
import requests
import json
import time
import re

from splunklib.modularinput.event import Event, ET
from splunklib.modularinput.event_writer import EventWriter

# Disable urlib3 warnings for SSL
import urllib3
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# set splunkhome
splunkhome = os.environ["SPLUNK_HOME"]

# append libs
sys.path.append(os.path.join(splunkhome, "etc", "apps", "trackme", "lib"))

# TrackMe libs
from trackme_libs import trackme_idx_for_tenant


def process_event(helper, *args, **kwargs):
    helper.log_info("Alert action trackme_free_style_rest_call started.")

    session_key = helper.session_key
    server_uri = helper.settings["server_uri"]
    tenant_id = helper.get_param("tenant_id")
    endpoint_url = helper.get_param("endpoint_url")
    http_mode = helper.get_param("http_mode")
    http_body = json.dumps(
        json.loads(helper.get_param("http_body").replace("'", '"'), strict=False),
        indent=1,
    )

    helper.log_info(f"tenant_id={tenant_id}")
    helper.log_info(f"endpoint_url={endpoint_url}")
    helper.log_info(f"http_mode={http_mode}")
    helper.log_info(f"http_body={http_body}")

    header = {
        "Authorization": f"Splunk {session_key}",
        "Content-Type": "application/json",
    }
    target_url = f"{server_uri}/{endpoint_url}"

    tenant_idx = trackme_idx_for_tenant(session_key, server_uri, tenant_id)
    tenant_trackme_summary_idx = tenant_idx["trackme_summary_idx"]

    if http_mode == "get":
        msg = f"Performing HTTP get call to {endpoint_url}"
        if http_body not in ["", "None", None]:
            msg += f" with HTTP body: {http_body}"
        helper.log_info(f"action:={msg}")

        try:
            response = requests.get(
                target_url,
                headers=header,
                verify=False,
                data=(http_body if http_body not in ["", "None", None] else None),
                timeout=300,
            )
            process_response(helper, response, tenant_trackme_summary_idx, endpoint_url)
            return 0

        except Exception as e:
            helper.log_error(f"TrackMe alert action has failed!:{e}")

    elif http_mode == "post":
        helper.log_info(
            f"action:=Performing HTTP post call to {endpoint_url} with HTTP body: {http_body}"
        )
        try:
            response = requests.post(
                target_url,
                headers=header,
                verify=False,
                data=http_body,
                timeout=300,
            )
            process_response(helper, response, tenant_trackme_summary_idx, endpoint_url)
            return 0
        except Exception as e:
            helper.log_error(f"TrackMe alert action has failed!:{e}")

    elif http_mode == "delete":
        helper.log_info(
            f"action:=Performing HTTP delete call to {endpoint_url} with HTTP body: {http_body}"
        )
        try:
            response = requests.delete(
                target_url,
                headers=header,
                verify=False,
                data=http_body,
                timeout=300,
            )
            process_response(helper, response, tenant_trackme_summary_idx, endpoint_url)
            return 0
        except Exception as e:
            helper.log_error(f"TrackMe alert action has failed!:{e}")


def process_response(helper, response, tenant_trackme_summary_idx, endpoint_url):
    if response.status_code == 200:
        # If response is an array containing multiple JSON objects, return as response.text
        if (
            re.search(r"^\[", response.text)
            and re.search(r"\}\,", response.text)
            and re.search(r"\]$", response.text)
        ):
            response_data = response.text
        else:
            try:
                response_data = response.json()
            except ValueError:
                # Response is not JSON, let's parse and make it a JSON answer
                response_data = {"response": response.text.replace('"', r"\"")}

        data = {"_time": time.time(), "_raw": response_data}
        helper.addevent(json.dumps(data), sourcetype="trackme_alert_action")
        helper.writeevents(index=tenant_trackme_summary_idx, source=endpoint_url)
        helper.log_info(
            "TrackMe REST call was successful, the response was indexed in the TrackMe summary index."
        )
    else:
        helper.log_error(
            f"TrackMe alert action has failed, HTTP error code {response.status_code} was returned, server responded with: {response.text}"
        )
