#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_component.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys
import time
import hashlib
import requests
from collections import OrderedDict

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.component_power", "trackme_rest_api_component_power.log"
)
# Redirect logger.module calls


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerComponent_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerComponent_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_component(self, request_info, **kwargs):
        response = {
            "resource_group_name": "component/write",
            "resource_group_desc": "Endpoints specific to TrackMe's components data offload (write operations)",
        }

        return {"payload": response, "status": 200}

    # Update the component summary
    def post_component_summary_update(self, request_info, **kwargs):
        describe = False

        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            if not describe:
                # tenant_id
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {"error": "tenant_id is required"},
                        "status": 500,
                    }

                # component
                try:
                    component = resp_dict["component"]
                    if component not in (
                        "dsm",
                        "dhm",
                        "mhm",
                        "flx",
                        "fqm",
                        "wlk",
                        "cim",
                    ):
                        return {
                            "payload": {"error": "component is invalid"},
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": {"error": "component is required"},
                        "status": 500,
                    }

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint updates the component summary collection which contains high level summary information used in the Virtual Tenant UI for the tenant and component, this endpoint is called automatically by TrackMe to maintain these information, it requires a POST call using data and the following options:",
                "resource_desc": "Update the component summary",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/component/write/component_summary_update\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'component': 'flx'}\"",
                "options": [
                    {
                        "tenant_id": "tenant identifier",
                        "component": "component identifier, valid options are: dsm, dhm, mhm, flx, wlk, cim, fqm",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # summary KVstore collection
        summary_collection_name = f"kv_trackme_virtual_tenants_entities_summary"
        summary_collection = service.kvstore[summary_collection_name]

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # performance counter
        start = time.time()

        def count_records(record, stats):
            """
            Update the stats based on the properties of the record.

            :param record: A dictionary representing a single record.
            :param stats: A dictionary holding the count of various statistics.
            """
            # Increase the total entities count
            if record.get("monitored_state") == "enabled":
                stats["entities"] += 1

            # Check if the object_state is red and increment the appropriate priority counter
            if (
                record.get("object_state") == "red"
                and record.get("monitored_state") == "enabled"
            ):
                priority = record.get("priority")
                if priority == "low":
                    stats["low_red_priority"] += 1
                elif priority == "medium":
                    stats["medium_red_priority"] += 1
                elif priority == "high":
                    stats["high_red_priority"] += 1
                elif priority == "critical":
                    stats["critical_red_priority"] += 1

            # Update the last_exec with the maximum tracker_runtime value
            try:
                tracker_runtime = float(record.get("tracker_runtime", 0))
            except Exception as e:
                tracker_runtime = 0
            if tracker_runtime > stats.get("last_exec", 0):
                stats["last_exec"] = tracker_runtime

        def extended_count_records(record, extended_stats):
            """
            Update the extended_stats based on the properties of the record.

            :param record: A dictionary representing a single record.
            :param extended_stats: A dictionary holding the count of various statistics.
            """
            if record.get("monitored_state") == "enabled":
                extended_stats["count_total"] += 1
            if record.get("monitored_state") == "disabled":
                extended_stats["count_total_disabled"] += 1
            if (
                record.get("object_state") not in ("green", "blue")
                and record.get("monitored_state") == "enabled"
            ):
                extended_stats["count_total_in_alert"] += 1
            if (
                record.get("object_state") == "red"
                and record.get("priority") == "high"
                and record.get("monitored_state") == "enabled"
            ):
                extended_stats["count_total_high_priority_red"] += 1
            if (
                record.get("object_state") == "red"
                and record.get("priority") == "critical"
                and record.get("monitored_state") == "enabled"
            ):
                extended_stats["count_total_critical_priority_red"] += 1
            if record.get("monitored_state") == "enabled":
                if record.get("priority") == "low":
                    extended_stats["count_low_enabled"] += 1
                if record.get("priority") == "medium":
                    extended_stats["count_medium_enabled"] += 1
                if record.get("priority") == "high":
                    extended_stats["count_high_enabled"] += 1
                if record.get("priority") == "critical":
                    extended_stats["count_critical_enabled"] += 1
                if record.get("object_state") == "green":
                    extended_stats["count_green_enabled"] += 1
                if record.get("object_state") == "blue":
                    extended_stats["count_blue_enabled"] += 1
                if record.get("object_state") == "orange":
                    extended_stats["count_orange_enabled"] += 1
                if (
                    record.get("object_state") == "red"
                    and record.get("priority") == "low"
                ):
                    extended_stats["count_red_low_priority_enabled"] += 1
                if (
                    record.get("object_state") == "red"
                    and record.get("priority") == "medium"
                ):
                    extended_stats["count_red_medium_priority_enabled"] += 1
                if (
                    record.get("object_state") == "red"
                    and record.get("priority") == "high"
                ):
                    extended_stats["count_red_high_priority_enabled"] += 1
                if (
                    record.get("object_state") == "red"
                    and record.get("priority") == "critical"
                ):
                    extended_stats["count_red_critical_priority_enabled"] += 1
                if (
                    record.get("object_state") == "red"
                    and record.get("priority") != "high"
                    and record.get("priority") != "critical"
                ):
                    extended_stats["count_red_other_priority_enabled"] += 1
            extended_stats["mtime"] = time.time()
            extended_stats["human_mtime"] = time.strftime(
                "%Y-%m-%d %H:%M:%S", time.gmtime(extended_stats["mtime"])
            )

        # Initialize the stats dictionary with the new structure
        stats = {
            "entities": 0,
            "low_red_priority": 0,
            "medium_red_priority": 0,
            "high_red_priority": 0,
            "critical_red_priority": 0,
            "last_exec": 0,  # Assuming tracker_runtime is an epoch time, initialize with 0
        }

        # extended stats
        extended_stats = {
            "count_total": 0,
            "count_total_disabled": 0,
            "count_total_in_alert": 0,
            "count_total_high_priority_red": 0,
            "count_total_critical_priority_red": 0,
            "count_low_enabled": 0,
            "count_medium_enabled": 0,
            "count_high_enabled": 0,
            "count_critical_enabled": 0,
            "count_blue_enabled": 0,
            "count_orange_enabled": 0,
            "count_green_enabled": 0,
            "count_red_low_priority_enabled": 0,
            "count_red_medium_priority_enabled": 0,
            "count_red_high_priority_enabled": 0,
            "count_red_critical_priority_enabled": 0,
            "count_red_other_priority_enabled": 0,
            "mtime": 0,
            "human_mtime": 0,
        }

        params = {
            "tenant_id": tenant_id,
            "component": component,
            "page": 1,
            "size": 0,
        }

        # Define an header for requests authenticated communications with splunkd
        header = {
            "Authorization": f"Splunk {request_info.system_authtoken}",
            "Content-Type": "application/json",
        }

        # Add the vtenant account
        url = f"{request_info.server_rest_uri}/services/trackme/v2/component/load_component_data"

        # Proceed
        try:

            logger.info(f'calling api url="{url}", params="{params}"')

            response = requests.get(
                url,
                headers=header,
                params=params,
                verify=False,
                timeout=300,
            )

            if response.status_code not in (200, 201, 204):
                msg = f'get component has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                raise Exception(msg)

            else:
                response_json = response.json()
                last_page = response_json.get("last_page", 1)
                data = response_json.get("data", [])

                # add the data to the data_records
                for record in data:
                    # count the records
                    count_records(record, stats)
                    extended_count_records(record, extended_stats)

        except Exception as e:
            msg = f'get component has failed, exception="{str(e)}"'
            logger.error(msg)
            return {"payload": {"response": msg}, "status": 500}

        # Get the summary record
        try:
            vtenant_record = summary_collection.data.query(
                query=json.dumps({"tenant_id": tenant_id})
            )[0]
            vtenant_key = vtenant_record.get("_key")
            logger.debug(
                f'tenant_id="{tenant_id}", vtenant_key="{vtenant_key}", vtenant_report="{json.dumps(vtenant_record)}"'
            )
        except Exception as e:
            vtenant_record = {}
            vtenant_key = None

        # update the summary record
        vtenant_record[f"{component}_entities"] = stats.get("entities")
        vtenant_record[f"{component}_low_red_priority"] = stats.get("low_red_priority")
        vtenant_record[f"{component}_medium_red_priority"] = stats.get(
            "medium_red_priority"
        )
        vtenant_record[f"{component}_high_red_priority"] = stats.get(
            "high_red_priority"
        )
        vtenant_record[f"{component}_critical_red_priority"] = stats.get(
            "critical_red_priority"
        )
        vtenant_record[f"{component}_last_exec"] = stats.get("last_exec")
        vtenant_record[f"{component}_summary_stats"] = json.dumps(stats, indent=2)
        vtenant_record[f"{component}_extended_stats"] = json.dumps(
            extended_stats, indent=2
        )

        try:
            if vtenant_key:
                summary_collection.data.update(
                    str(vtenant_key), json.dumps(vtenant_record)
                )
            else:
                vtenant_record["tenant_id"] = tenant_id
                # add _key as the sha256 of the tenant_id
                vtenant_record["_key"] = hashlib.sha256(
                    f"{tenant_id}".encode()
                ).hexdigest()
                summary_collection.data.insert(json.dumps(vtenant_record))

        except Exception as e:
            error_msg = f'tenant_id="{tenant_id}" cannot be updated or created, exception="{str(e)}"'
            logger.error(error_msg)
            return {
                "payload": {
                    "response": error_msg,
                },
                "status": 500,
            }

        # run_time
        run_time = round((time.time() - start), 3)
        stats["run_time"] = run_time

        # return the response
        logger.info(
            f'context="perf", no_records="{stats.get("entities")}", run_time="{run_time}", tenant_id="{tenant_id}", component="{component}"'
        )

        return {
            "payload": {
                "response": "component summary has been updated",
                "stats": stats,
                "extended_stats": extended_stats,
                "vtenant_summary_record": vtenant_record,
            },
            "status": 200,
        }
