#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_allowlist.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

import os, sys
import json

splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_blocklist_user", "trackme_rest_api_splk_blocklist_user.log"
)
# Redirect logger.module calls


import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel

# import trackme decision maker
from trackme_libs_decisionmaker import convert_epoch_to_datetime

# import splunk
import splunklib.client as client


class TrackMeHandlerSplkBlocklist_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkBlocklist_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_blocklist(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_blocklist",
            "resource_group_desc": "These endpoints provide capabilities to manage blocklists for feeds tracking. (splk-dsm/dhm/mhm, read only operations)",
        }

        return {"payload": response, "status": 200}

    # get all records
    def post_blocklist_show(self, request_info, **kwargs):

        # Declare
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": "tenant_id is required, please provide a valid tenant_id",
                        "status": 500,
                    }

                try:
                    component = resp_dict["component"]
                    if component not in ("dsm", "dhm", "mhm", "flx", "wlk", "fqm"):
                        return {
                            "payload": f'Invalid component="{component}"',
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": "component is required, please provide a valid component",
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint retrieves blocklist records, it requires a POST call with the following information:",
                "resource_desc": "Get blocklist records",
                "resource_spl_example": "| trackme mode=get url=\"/services/trackme/v2/splk_blocklist/blocklist_show\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "component": "The component, valid options are: dsm | dhm | mhm | flx | wlk | fqm",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Data collection
            collection_name = f"kv_trackme_{component}_allowlist_tenant_{tenant_id}"
            collection = service.kvstore[collection_name]

            records = collection.data.query()
            results_records = []
            for item in records:
                mtime = item.get("mtime")
                if mtime:
                    mtime = convert_epoch_to_datetime(mtime)
                else:
                    mtime = "N/A"
                results_records.append(
                    {
                        "_key": item.get("_key"),
                        "object_category": item.get("object_category"),
                        "object": item.get("object"),
                        "action": item.get("action"),
                        "is_rex": item.get("is_rex"),
                        "comment": item.get("comment", ""),
                        "mtime": mtime,
                    }
                )
            return {"payload": results_records, "status": 200}

        except Exception as e:
            error_msg = f'An exception was encountered, exception="{str(e)}"'
            logger.error(error_msg)
            return {"payload": error_msg, "status": 500}
