#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_cim_tracking.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys
import time
import uuid
import threading
from collections import OrderedDict

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_cim_admin", "trackme_rest_api_splk_cim_admin.log"
)
# Redirect logger.module calls


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import (
    trackme_getloglevel,
    trackme_audit_event,
    trackme_create_report,
    trackme_register_tenant_component_summary,
    trackme_delete_tenant_object_summary,
    trackme_reqinfo,
    trackme_send_to_tcm,
    run_splunk_search,
)

# import trackme licensing libs
from trackme_libs_licensing import trackme_check_license

# import trackme splk-cim libs
from trackme_libs_splk_cim import trackme_cim_return_search

# import trackme libs croniter
from trackme_libs_croniter import validate_cron_schedule

# import Splunk lbs
import splunklib.client as client


class TrackMeHandlerSplkCimTracking_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkCimTracking_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_cim(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_cim/admin",
            "resource_group_desc": "Endpoints specific to the splk-cim TrackMe component (Splunk Common Information Model compliance monitoring, admin operations)",
        }

        return {"payload": response, "status": 200}

    def register_component_summary_async(
        self, session_key, splunkd_uri, tenant_id, component
    ):
        try:
            summary_register_response = trackme_register_tenant_component_summary(
                session_key,
                splunkd_uri,
                tenant_id,
                component,
            )
            logger.debug(
                f'function="trackme_register_tenant_component_summary", response="{json.dumps(summary_register_response, indent=2)}"'
            )
        except Exception as e:
            logger.error(
                f'failed to register the component summary with exception="{str(e)}"'
            )

    # Create a new CIM tracker
    def post_cim_tracker(self, request_info, **kwargs):
        """
        | trackme url="/services/trackme/v2/splk_cim/admin/cim_tracker" mode="post" body="{\"tenant_id\": \"mytenant\", \"context\": \"live\", \"object\": \"auth002\", \"account\": \"local\", \"root_constraint\": \"\", \"cim_tracking_rules\": \"<redacted_cim_tracking_rules>\", \"earliest\": \"-65m\", \"latest\": \"-5m\", \"summariesonly\": \"True\", \"cron_schedule\": \"*/5 * * * *\"}"
        """

        # args
        tenant_id = None
        object = None
        tracker_name = None
        account = None
        summariesonly = None
        cim_tracking_rules = None
        priority = None
        cron_schedule = None
        owner = None
        earliest = None
        latest = None
        context = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            # gets args
            if not describe:
                #
                # args
                #

                # tenant
                tenant_id = resp_dict["tenant_id"]

                # object
                object = resp_dict["object"]
                object = object.lower().replace(" ", "_").replace("-", "_")

                # tracker name, convert to lower case and remove spaces
                # include a random UUID
                object_truncated = object.replace(":", "_")[:40]
                tracker_name = f"trackme_cim_{object_truncated}_{uuid.uuid4().hex[:5]}_tenant_{tenant_id}"

                # account, defines if the search is dealing with a local or remote Splunk deployment
                # defaults to local, if provided and differs from local, means the search is remote and used this account
                try:
                    account = resp_dict["account"]
                except Exception as e:
                    account = "local"

                # summariesonly, defaults to false
                try:
                    summariesonly = resp_dict["summariesonly"]
                    if summariesonly in ("f", "false", "False", "FALSE"):
                        summariesonly = "false"
                    elif summariesonly in ("t", "true", "True", "TRUE"):
                        summariesonly = "true"
                    else:
                        summariesonly = "true"
                except Exception as e:
                    summariesonly = "true"

                # cim_tracking_rules
                cim_tracking_rules = resp_dict["cim_tracking_rules"]

                #
                # optional args
                #

                try:
                    cron_schedule = resp_dict["cron_schedule"]
                except Exception as e:
                    cron_schedule = "*/5 * * * *"

                # verify the cron schedule validity, if submitted
                if cron_schedule:
                    try:
                        validate_cron_schedule(cron_schedule)
                    except Exception as e:
                        logger.error(str(e))
                        return {
                            "payload": {
                                "action": "failure",
                                "response": str(e),
                            },
                            "status": 500,
                        }

                try:
                    owner = resp_dict["owner"]
                except Exception as e:
                    owner = None

                try:
                    priority = resp_dict["priority"]
                except Exception as e:
                    priority = "medium"

                # Update comment is optional and used for audit changes
                try:
                    update_comment = resp_dict["update_comment"]
                except Exception as e:
                    update_comment = "API update"

                # earliest and latest for the tracker, if not specified, defaults to -4h / +4h
                try:
                    earliest = resp_dict["earliest"]
                except Exception as e:
                    earliest = "-65m"

                try:
                    latest = resp_dict["latest"]
                except Exception as e:
                    latest = "-5m"

                # context: the context is used for simulation purposes versus live context
                # This is an optional argument, and defaults to live
                # swiching to simulation creates the records in a secondary KVstore that is not used for true alerting purpose
                # As well, the scheduled report will not be created in simulation context
                try:
                    context = resp_dict["context"]
                except Exception as e:
                    context = "live"

                # Optional: burn_test, temporary create the abstract, perform a burn test, report the run time performance, delete and report
                try:
                    burn_test = resp_dict["burn_test"]
                    if burn_test == "True":
                        burn_test = True
                    elif burn_test == "False":
                        burn_test = False
                except Exception as e:
                    burn_test = False

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint allows creating a cim datamodel compliance tracker, it requires a POST call with the following information:",
                "resource_desc": "Create a new cim tracker",
                "resource_spl_example": '| trackme url="/services/trackme/v2/splk_cim/admin/cim_tracker" mode="post" body="{\\"tenant_id\\": \\"mytenant\\", \\"context\\": \\"live\\", \\"object\\": \\"auth002\\", \\"account\\": \\"local\\", \\"root_constraint\\": \\"\\", \\"cim_tracking_rules\\": \\"<redacted_cim_tracking_rules>\\", \\"earliest\\": \\"-65m\\", \\"latest\\": \\"-5m\\", \\"summariesonly\\": \\"True\\", \\"cron_schedule\\": \\"*/5 * * * *\\"}"',
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object": "name of the cim entity, the name of the tracker will be derived from it",
                        "cim_tracking_rules": "the cim tracking rules provided as a JSON object",
                        "priority": "Optional, the priority for this object",
                        "summariesonly": "Optional, the tstats summariesonly mode defaults to true",
                        "owner": "Optional, the Splunk user owning the objects to be created, defaults to the owner set for the tenant",
                        "cron_schedule": "Optional, the cron schedule, defaults to every 5 minutes",
                        "earliest": "Optional, the earliest time value for the tracker, defaults to -4h",
                        "latest": "Optional, the latest time value for the tracker, defaults to +4h",
                        "context": "OPTIONAL: context is used for simulation purposes, defaults to live (valid option: live | simulation)",
                        "burn_test": "Optional, create the abstract report, run a performance test, delete the report and report the performance results, valid options are: True | False (default: False)",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # run creation
        else:
            # Get splunkd port
            splunkd_port = request_info.server_rest_port

            # Get service
            service = client.connect(
                owner="nobody",
                app="trackme",
                port=splunkd_port,
                token=request_info.system_authtoken,
                timeout=300,
            )

            # set loglevel
            loglevel = trackme_getloglevel(
                request_info.system_authtoken, request_info.server_rest_port
            )
            logger.setLevel(loglevel)

            # get TrackMe conf
            trackme_conf = trackme_reqinfo(
                request_info.system_authtoken, request_info.server_rest_uri
            )
            logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

            # TrackMe sharing level
            trackme_default_sharing = trackme_conf["trackme_conf"]["trackme_general"][
                "trackme_default_sharing"
            ]

            # Retrieve the virtual tenant record to access acl
            collection_vtenants_name = "kv_trackme_virtual_tenants"
            collection_vtenants = service.kvstore[collection_vtenants_name]

            # Define the KV query search string
            query_string = {
                "tenant_id": tenant_id,
            }

            # Get the tenant
            try:
                vtenant_record = collection_vtenants.data.query(
                    query=json.dumps(query_string)
                )[0]
                vtenant_key = vtenant_record.get("_key")

            except Exception as e:
                logger.error(
                    f'tenant_id="{tenant_id}", failed to retrieve the tenant record, exception="{str(e)}"'
                )
                return {
                    "payload": f'tenant_id="{tenant_id}", failed to retrieve the tenant record, exception="{str(e)}"',
                    "status": 500,
                }

            # check license state
            try:
                check_license = trackme_check_license(
                    request_info.server_rest_uri, request_info.session_key
                )
                license_is_valid = check_license.get("license_is_valid")
                license_active_cim_trackers = int(
                    check_license.get("license_active_cim_trackers")
                )
                logger.debug(
                    f'function check_license called, response="{json.dumps(check_license, indent=2)}"'
                )

            except Exception as e:
                license_is_valid = 0
                license_active_cim_trackers = 32
                logger.error(f'function check_license exception="{str(e)}"')

            if license_active_cim_trackers >= 32 and license_is_valid != 1:
                # Licensing restrictions reached
                audit_record = {
                    "action": "failure",
                    "change_type": "add new CIM tracker",
                    "tenant_id": str(tenant_id),
                    "result": f"I'm afraid I can't do that, the maximum number of 32 allowed trackers has been reached, there are {license_active_cim_trackers} active trackers currently for this component",
                }

                logger.error(str(audit_record))
                return {"payload": audit_record, "status": 402}

            # verify the owner
            if not owner:
                owner = vtenant_record.get("tenant_owner")

            # check if TCM is enabled in receiver mode
            enable_conf_manager_receiver = int(
                trackme_conf["trackme_conf"]["trackme_general"][
                    "enable_conf_manager_receiver"
                ]
            )

            if enable_conf_manager_receiver == 1:
                try:
                    tcm_response = trackme_send_to_tcm(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        resp_dict,
                        "post",
                        "/services/trackme/v2/splk_cim/admin/cim_tracker",
                    )
                    logger.info(f"trackme_send_to_tcm was successfully executed")
                except Exception as e:
                    logger.error(
                        f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                    )

            #
            # step 1: create the KVstore record
            #

            if context == "live":
                collection_name = "kv_trackme_cim_tenant_" + str(tenant_id)
            elif context == "simulation":
                collection_name = "kv_trackme_cim_simulation_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            # Define the KV query
            query_string = {
                "object": object,
            }

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            try:
                kvrecord = collection.data.query(query=json.dumps(query_string))[0]
                key = kvrecord.get("_key")

            except Exception as e:
                kvrecord = None

            # In live context, we will refuse to continue if a record exists already
            if context == "live":
                if not kvrecord:
                    try:
                        # Insert the record
                        newcord = {
                            "tenant_id": str(tenant_id),
                            "object": str(object),
                            "object_category": "splk-cim",
                            "alias": str(object),
                            "monitored_state": "enabled",
                            "cim_tracking_rules": json.dumps(
                                json.loads(cim_tracking_rules), indent=1
                            ),
                            "object_state": "",
                            "account": str(account),
                            "tracker_name": str(tracker_name),
                            "tracker_runtime": "",
                            "latest_flip_state": "",
                            "latest_flip_time": "",
                            "priority": str(priority),
                        }

                        collection.data.insert(json.dumps(newcord))
                        logger.info(
                            f'record created successfully, record="{json.dumps(newcord, indent=2)}"'
                        )

                    except Exception as e:
                        logger.error(
                            f'tenant_id="{tenant_id}", object="{object}", exception encountered while inserting the KVstore record, exception="{str(e)}"'
                        )
                        return {
                            "payload": f'tenant_id="{tenant_id}", object="{object}", exception encountered while inserting the KVstore record, exception="{str(e)}"',
                            "status": 500,
                        }

                else:
                    logger.error(
                        f'tenant_id="{tenant_id}", object="{object}", a KVstore record exists already for the same entity'
                    )
                    return {
                        "payload": f'tenant_id="{tenant_id}", object="{object}", a KVstore record exists already for the same entity',
                        "status": 500,
                    }

            elif context == "simulation":
                # If the record exists already, attempt to delete
                if kvrecord:
                    # Attempt to delete the record, and stop if failing
                    try:
                        collection.data.delete(json.dumps({"_key": key}))
                        logger.info(f'exising record key="{key}" deleted successfully')

                    except Exception as e:
                        logger.error(
                            f'tenant_id="{tenant_id}", object="{object}", exception encountered while deleteing the KVstore record, exception="{str(e)}"'
                        )
                        return {
                            "payload": f'tenant_id="{tenant_id}", object="{object}", exception encountered while deleteing the KVstore record, exception="{str(e)}"',
                            "status": 500,
                        }

                # Then proceed
                try:
                    # Insert the record
                    newrecord = {
                        "tenant_id": str(tenant_id),
                        "object": str(object),
                        "object_category": "splk-cim",
                        "alias": str(object),
                        "monitored_state": "enabled",
                        "cim_tracking_rules": json.dumps(
                            json.loads(cim_tracking_rules), indent=1
                        ),
                        "object_state": "",
                        "account": str(account),
                        "tracker_name": str(tracker_name),
                        "tracker_runtime": "",
                        "latest_flip_state": "",
                        "latest_flip_time": "",
                        "priority": str(priority),
                    }
                    collection.data.insert(json.dumps(newrecord))
                    logger.info(
                        f'tenant_id="{tenant_id}", object="{object}", KVstore record created successfully, record="{json.dumps(newrecord, indent=0)}"'
                    )

                except Exception as e:
                    logger.error(
                        f'tenant_id="{tenant_id}", object="{object}", exception encountered while attemting to the insert the KVstore record, exception="{str(e)}"'
                    )
                    return {
                        "payload": f'tenant_id="{tenant_id}", object="{object}", exception encountered while attemting to the insert the KVstore record, exception="{str(e)}"',
                        "status": 500,
                    }

            #
            # burn test: execute the abstract report, delete and report the run time performance
            #

            if burn_test:
                burn_test_report = str(tracker_name)
                burn_test_search = (
                    '| trackmecimtrackerexecutor tenant_id="'
                    + str(tenant_id)
                    + '" '
                    + 'object="'
                    + str(object)
                    + '" account="'
                    + str(account)
                    + '" context="simulation" component="splk-cim" summariesonly="'
                    + str(summariesonly)
                    + '" earliest="'
                    + str(earliest)
                    + '" latest="'
                    + str(latest)
                    + '" | spath'
                )

                logger.info(
                    f'tenant_id="{tenant_id}", burn test was requested, starting abstract report burn test now'
                )

                # kwargs
                burn_test_kwargs = {
                    "earliest_time": earliest,
                    "latest_time": latest,
                    "search_mode": "normal",
                    "preview": False,
                    "time_format": "%s",
                    "output_mode": "json",
                    "count": 0,
                }

                burn_test_start_time = time.time()

                # results counter
                burn_test_results_counter = 0

                # run search
                try:
                    reader = run_splunk_search(
                        service,
                        burn_test_search,
                        burn_test_kwargs,
                        24,
                        5,
                    )

                    for item in reader:
                        if isinstance(item, dict):
                            # increment
                            burn_test_results_counter += 1

                    # return
                    burn_test_results_record = {
                        "tenant_id": tenant_id,
                        "run_time": round((time.time() - burn_test_start_time), 3),
                        "results_count": burn_test_results_counter,
                        "report": burn_test_report,
                        "burn_test_success": True,
                    }

                    logger.info(
                        f'tenant_id="{tenant_id}", burn test, results="{json.dumps(burn_test_results_record, indent=2)}"'
                    )
                    return {"payload": burn_test_results_record, "status": 200}

                except Exception as e:
                    # return
                    burn_test_results_record = {
                        "tenant_id": tenant_id,
                        "run_time": round((time.time() - burn_test_start_time), 3),
                        "results_count": burn_test_results_counter,
                        "report": burn_test_report,
                        "burn_test_success": False,
                        "exception": f'search failed with exception="{str(e)}"',
                    }

                    logger.error(json.dumps(burn_test_results_record, indent=2))
                    return {
                        "payload": burn_test_results_record,
                        "status": 200,
                    }

            #
            # step 2: create the tracker report
            #

            if context == "live":
                report_name = str(tracker_name)
                report_search = (
                    '| trackmecimtrackerexecutor tenant_id="'
                    + str(tenant_id)
                    + '" '
                    + 'object="'
                    + str(object)
                    + '" account="'
                    + str(account)
                    + '" component="splk-cim" summariesonly="'
                    + str(summariesonly)
                    + '" earliest="'
                    + str(earliest)
                    + '" latest="'
                    + str(latest)
                    + '" | spath'
                )
                report_properties = {
                    "description": "TrackMe CIM compliance tracker",
                    "is_scheduled": True,
                    "schedule_window": "5",
                    "cron_schedule": str(cron_schedule),
                    "dispatch.earliest_time": str(earliest),
                    "dispatch.latest_time": str(latest),
                }
                report_acl = {
                    "owner": owner,
                    "sharing": trackme_default_sharing,
                    "perms.write": vtenant_record.get("tenant_roles_admin"),
                    "perms.read": f"{vtenant_record.get('tenant_roles_user')},{vtenant_record.get('tenant_roles_power')}",
                }
                tracker_create_report = trackme_create_report(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    report_name,
                    report_search,
                    report_properties,
                    report_acl,
                )

            #
            # step 3: add the cim knowledge objects
            #

            if context == "live":
                # Register the new components in the vtenant collection
                collection_vtenants_name = "kv_trackme_virtual_tenants"
                collection_vtenants = service.kvstore[collection_vtenants_name]

                # Define the KV query search string
                query_string = {
                    "tenant_id": tenant_id,
                }

                # Get the tenant
                try:
                    vtenant_record = collection_vtenants.data.query(
                        query=json.dumps(query_string)
                    )[0]
                    vtenant_key = vtenant_record.get("_key")

                except Exception as e:
                    vtenant_key = None
                    logger.error(
                        f'tenant_id="{tenant_id}", failed to retrieve the tenant record'
                    )

                # We can only proceed with a valid tenant record
                if vtenant_key:
                    # Try to get the current definition
                    try:
                        tenant_cim_objects = vtenant_record.get("tenant_cim_objects")
                        # logger.debug
                        logger.debug(f'tenant_cim_objects="{tenant_cim_objects}"')
                    except Exception as e:
                        tenant_cim_objects = None

                    # add to existing disct
                    if tenant_cim_objects and tenant_cim_objects != "None":
                        logger.debug("vtenant_dict is not empty")
                        vtenant_dict = json.loads(tenant_cim_objects)
                        logger.debug(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )
                        reports = vtenant_dict["reports"]
                        reports.append(str(report_name))
                        vtenant_dict = dict(
                            [
                                ("reports", reports),
                            ]
                        )

                    # empty dict
                    else:
                        logger.debug("vtenant_dict is empty")
                        reports = []
                        reports.append(str(report_name))
                        vtenant_dict = dict(
                            [
                                ("reports", reports),
                            ]
                        )
                        logger.debug(
                            f'creating vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                    try:
                        vtenant_record["tenant_cim_objects"] = json.dumps(
                            vtenant_dict, indent=1
                        )
                        collection_vtenants.data.update(
                            str(vtenant_key), json.dumps(vtenant_record)
                        )

                    except Exception as e:
                        logger.error(
                            f'failure while trying to update the vtenant KVstore record, exception="{str(e)}"'
                        )
                        return {
                            "payload": "Warn: exception encountered: "
                            + str(e)  # Payload of the request.
                        }

            #
            # Step 4: audit
            #

            # get the search string
            cim_tracking_rules = json.loads(cim_tracking_rules)

            try:
                search = trackme_cim_return_search(
                    tracker_name,
                    tenant_id,
                    object,
                    account,
                    json.dumps(cim_tracking_rules),
                    cim_tracking_rules["cim_datamodel_name"],
                    cim_tracking_rules["cim_datamodel_nodename"],
                    cim_tracking_rules["cim_drop_dm_object_name"],
                    cim_tracking_rules["cim_root_constraint"],
                    cim_tracking_rules["cim_fields"],
                    earliest,
                    latest,
                    summariesonly,
                )
                logger.debug(f'search="{search}"')

            except Exception as e:
                logger.error(
                    f'Failed to retrieve the search string from function trackme_cim_return_search with exception="{str(e)}"'
                )
                search = "None"

            audit_record = {
                "object": str(object),
                "account": str(account),
                "tracker_name": str(tracker_name),
                "summariesonly": str(summariesonly),
                "cim_tracking_rules": cim_tracking_rules,
                "priority": str(priority),
                "cron_schedule": str(cron_schedule),
                "owner": str(owner),
                "earliest": str(earliest),
                "latest": str(latest),
                "action": "success",
                "search": search.replace("\n", " "),
            }

            # if live, replace the cron_schedule with the random definition applied by the API
            if context == "live":
                audit_record["cron_schedule"] = tracker_create_report.get(
                    "cron_schedule"
                )

            # Generate an audit record in live context only
            if context == "live":
                try:
                    # audit event
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "add cim tracker",
                        str(object),
                        "splk-cim",
                        json.dumps(audit_record, indent=0),
                        "The new CIM tracker was created successfully",
                        str(update_comment),
                    )

                except Exception as e:
                    logger.error(
                        f'tenant_id="{tenant_id}", object="{object}", exception encountered while attempting to record an audit change, exception="{str(e)}"'
                    )
                    return {
                        "payload": f'tenant_id="{tenant_id}", object="{object}", exception encountered while attempting to record an audit change, exception="{str(e)}"',
                        "status": 500,
                    }

            # final return
            logger.info(json.dumps(audit_record, indent=0))
            return {"payload": audit_record, "status": 200}

    # Remove a CIM tracker and associated objects
    def post_cim_tracker_delete(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint performs the deletion of a CIM tracker and associated objects, it requires a POST call with the following information:",
                "resource_desc": "Delete a cim tracker",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_cim/admin/cim_tracker_delete\" body=\"{'tenant_id': 'mytenant', 'object_list': 'auth001,auth002'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "context": "OPTIONAL: context is used for simulation purposes, defaults to live (valid option: live | simulation)",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # context: the context is used for simulation purposes versus live context
        # This is an optional argument, and defaults to live
        # swiching to simulation creates the records in a secondary KVstore that is not used for true alerting purpose
        # As well, the scheduled report will not be created in simulation context
        try:
            context = resp_dict["context"]
        except Exception as e:
            context = "live"

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        # Data collection
        if context == "live":
            collection_name = "kv_trackme_cim_tenant_" + str(tenant_id)
        elif context == "simulation":
            collection_name = "kv_trackme_cim_simulation_tenant_" + str(tenant_id)
        collection = service.kvstore[collection_name]

        #
        # Outliers collections
        #

        # entity rules collection
        collection_outliers_entity_rules_name = (
            f"kv_trackme_cim_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_entity_rules = service.kvstore[collection_outliers_entity_rules_name]

        # data rules collection
        collection_outliers_entity_data_name = (
            f"kv_trackme_cim_outliers_entity_data_tenant_{tenant_id}"
        )
        collection_outliers_entity_data = service.kvstore[
            collection_outliers_entity_data_name
        ]

        # loop and proceed
        if object_list:
            keys_list = []
            for object in object_list:
                try:
                    kvrecord = collection.data.query(
                        query=json.dumps({"object": object})
                    )[0]
                    key = kvrecord.get("_key")
                    keys_list.append(key)
                except Exception as e:
                    key = None

        for key in keys_list:
            # this operation will be considered to be successful only no failures were encountered
            # any failure encoutered will be added to the record summary for that entity
            sub_failures_count = 0

            kvrecord = collection.data.query(query=json.dumps({"_key": key}))[0]
            object = kvrecord.get("object")

            try:
                # check if TCM is enabled in receiver mode
                enable_conf_manager_receiver = int(
                    trackme_conf["trackme_conf"]["trackme_general"][
                        "enable_conf_manager_receiver"
                    ]
                )

                if enable_conf_manager_receiver == 1:
                    try:
                        tcm_response = trackme_send_to_tcm(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            resp_dict,
                            "post",
                            "/services/trackme/v2/splk_cim/admin/cim_tracker_delete",
                        )
                        logger.info(f"trackme_send_to_tcm was successfully executed")
                    except Exception as e:
                        logger.error(
                            f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                        )

                # Step 1: delete the scheduled report

                if context == "live":
                    # retrieve the name of the tracker, this is the name of the Splunk scheduled report to be removed
                    tracker_name = kvrecord.get("tracker_name")

                    # log debug
                    logger.debug(
                        f'tenant_id="{tenant_id}", object="{object}", tracker successfully retrieved fron the KVstore collection, record="{json.dumps(kvrecord, indent=0)}"'
                    )

                    # ensure this didn't fail, stop otherwise
                    if not tracker_name or len(tracker_name) == 0:
                        logger.error(
                            f'tenant_id="{tenant_id}", object="{object}", failed to retrieve the tracker name from the KVstore collection it might be corrupted, cannot continue.'
                        )

                        # append for summary
                        sub_failures_count += 1
                        result = {
                            "object": object,
                            "action": "delete",
                            "result": "failure",
                            "exception": f'tenant_id="{tenant_id}", object="{object}", failed to retrieve the tracker name from the KVstore collection it might be corrupted, cannot continue.',
                        }
                        records.append(result)

                    else:
                        try:
                            service.saved_searches.delete(str(tracker_name))
                            logger.info(
                                f'tenant_id="{tenant_id}", object="{object}", the tracker was successfully removed, tracker_name="{tracker_name}"'
                            )
                        except Exception as e:
                            logger.error(
                                f'tenant_id="{tenant_id}", object="{object}", failed to remove the tracker, tracker_name="{tracker_name}", exception="{str(e)}"'
                            )

                            sub_failures_count += 1
                            result = {
                                "object": object,
                                "action": "delete",
                                "result": "failure",
                                "exception": f'tenant_id="{tenant_id}", object="{object}", failed to remove the tracker, tracker_name="{tracker_name}", exception="{str(e)}"',
                            }
                            records.append(result)

                # Step 2: delete the KVstore record

                # Remove the record
                try:
                    collection.data.delete(json.dumps({"_key": key}))

                except Exception as e:
                    logger.error(
                        f'tenant_id="{tenant_id}", object="{object}", exception encountered while attempting to delete the KVstore record, exception="{str(e)}"'
                    )
                    sub_failures_count += 1
                    result = {
                        "object": object,
                        "action": "delete",
                        "result": "failure",
                        "exception": f'tenant_id="{tenant_id}", object="{object}", exception encountered while attempting to delete the KVstore record, exception="{str(e)}"',
                    }
                    records.append(result)

                # Step 3: delete the cim knowledge from the tenant

                if context == "live":
                    # Register the new components in the vtenant collection
                    collection_vtenants_name = "kv_trackme_virtual_tenants"
                    collection_vtenants = service.kvstore[collection_vtenants_name]

                    # Define the KV query search string
                    query_string = {
                        "tenant_id": tenant_id,
                    }

                    # Get the tenant
                    try:
                        vtenant_record = collection_vtenants.data.query(
                            query=json.dumps(query_string)
                        )[0]
                        vtenant_key = vtenant_record.get("_key")

                    except Exception as e:
                        vtenant_key = None
                        logger.error(
                            f'tenant_id="{tenant_id}", failed to retrieve the tenant record'
                        )

                    # We can only proceed with a valid tenant record
                    if vtenant_key:
                        # Try to get the current definition
                        try:
                            tenant_cim_objects = vtenant_record.get(
                                "tenant_cim_objects"
                            )
                            # logger.debug
                            logger.debug(f'tenant_cim_objects="{tenant_cim_objects}"')
                        except Exception as e:
                            tenant_cim_objects = None

                        # remove from existing disct
                        if tenant_cim_objects and tenant_cim_objects != "None":
                            logger.debug("vtenant_dict is not empty")
                            vtenant_dict = json.loads(tenant_cim_objects)
                            logger.debug(
                                f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                            )
                            reports = vtenant_dict["reports"]
                            reports.remove(str(tracker_name))
                            vtenant_dict = dict(
                                [
                                    ("reports", reports),
                                ]
                            )

                        # Update the KVstore
                        try:
                            vtenant_record["tenant_cim_objects"] = json.dumps(
                                vtenant_dict, indent=1
                            )
                            collection_vtenants.data.update(
                                str(vtenant_key), json.dumps(vtenant_record)
                            )

                        except Exception as e:
                            logger.error(
                                f'failure while trying to update the vtenant KVstore record, exception="{str(e)}"'
                            )
                            return {
                                "payload": "Warn: exception encountered: "
                                + str(e)  # Payload of the request.
                            }

                # Step 4: purge the register summary object
                if context == "live":
                    try:
                        delete_register_summary = trackme_delete_tenant_object_summary(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            "splk-cim",
                            tracker_name,
                        )
                    except Exception as e:
                        logger.error(
                            f'exception encountered while calling function trackme_delete_tenant_object_summary, exception="{str(e)}"'
                        )

                # Step 5: Purge outliers records, if any
                try:
                    collection_entity_rules.data.delete(json.dumps({"object": object}))
                except Exception as e:
                    pass

                try:
                    collection_outliers_entity_data.data.delete(
                        json.dumps({"object": object})
                    )
                except Exception as e:
                    pass

                # If Live, record an audit change
                if context == "live":
                    # audit event
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "remove cim tracker",
                        str(object),
                        "splk-cim",
                        str(json.dumps(kvrecord, indent=2)),
                        "The CIM tracker and its associated objects were successfully deleted",
                        str(update_comment),
                    )

                logger.info(
                    f'tenant_id="{tenant_id}", object="{object}", The CIM tracker and its associated objects were successfully deleted'
                )

                # Handle the sub operation results
                if sub_failures_count == 0:
                    # increment counter
                    processed_count += 1
                    succcess_count += 1
                    failures_count += 0

                    # append for summary
                    result = {
                        "object": object,
                        "action": "delete",
                        "result": "success",
                        "message": f'tenant_id="{tenant_id}", object="{object}", The CIM tracker and its associated objects were successfully deleted',
                    }
                    records.append(result)

            # raise any exception
            except Exception as e:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                logger.error(
                    f'tenant_id="{tenant_id}", exception encountered, exception="{str(e)}"'
                )

                # append for summary
                result = {
                    "object": object,
                    "action": "delete",
                    "result": "failure",
                    "exception": str(e),
                }

                records.append(result)

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "cim",
            ),
        )
        thread.start()

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            req_summary["action"] = "success"
            return {"payload": req_summary, "status": 200}

        else:
            req_summary["action"] = "failure"
            return {"payload": req_summary, "status": 500}

    # Enable/Disable CIM trackers associated objects
    def post_cim_tracker_enablement(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                action = resp_dict["action"]
                if not action in ("enable", "disable"):
                    return {
                        "payload": "Invalid option for action, valid options are: enable | disable"
                        + str(e),
                        "status": 500,
                    }
                else:
                    if action == "enable":
                        action_value = "enabled"
                    elif action == "disable":
                        action_value = "disabled"

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint disables a CIM tracker and its associated objects, it requires a POST call with the following information:",
                "resource_desc": "Enable/Disable a comma separated list of cim entities",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_cim/admin/cim_tracker_enablement\" body=\"{'tenant_id':'mytenant','object_list':'auth001,auth002','action':'enable'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "action": "The action to be performed, valid options are: enable | disable",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        # Data collection
        collection_name = "kv_trackme_cim_tenant_" + str(tenant_id)
        collection = service.kvstore[collection_name]

        # loop and proceed
        if object_list:
            keys_list = []
            for object in object_list:
                try:
                    kvrecord = collection.data.query(
                        query=json.dumps({"object": object})
                    )[0]
                    key = kvrecord.get("_key")
                    keys_list.append(key)
                except Exception as e:
                    key = None

        for key in keys_list:
            try:
                kvrecord = collection.data.query(query=json.dumps({"_key": key}))[0]
                object = kvrecord.get("object")

                # check if TCM is enabled in receiver mode
                enable_conf_manager_receiver = int(
                    trackme_conf["trackme_conf"]["trackme_general"][
                        "enable_conf_manager_receiver"
                    ]
                )

                if enable_conf_manager_receiver == 1:
                    try:
                        tcm_response = trackme_send_to_tcm(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            resp_dict,
                            "post",
                            "/services/trackme/v2/splk_cim/admin/cim_tracker_enablement",
                        )
                        logger.info(f"trackme_send_to_tcm was successfully executed")
                    except Exception as e:
                        logger.error(
                            f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                        )

                # increment processed counter
                processed_count += 1

                # Step 1: handle the scheduled report

                # retrieve the name of the tracker, this is the name of the Splunk scheduled report to be removed
                tracker_name = kvrecord.get("tracker_name")

                # log debug
                logger.debug(
                    'object="'
                    + str(object)
                    + '", tracker_name successfully retrieved from the KVstore record="'
                    + str(tracker_name)
                    + '"'
                )

                # ensure this didn't fail, stop otherwise
                if not tracker_name or len(tracker_name) == 0:
                    logger.info(
                        "Failed to retrieve the tracker_name from the the KVstore for the object="
                        + str(object)
                        + ", the KVstore record might be corrupted, cannot continue."
                    )

                    # increment failures counter
                    failures_count += 1

                    result = {
                        "object": object,
                        "action": "update",
                        "result": "failure",
                        "exception": f'tenant_id="{tenant_id}", Failed to retrieve the tracker_name from the the KVstore, object="{object}", exception="{str(e)}"',
                    }
                    records.append(result)

                else:
                    try:
                        tracker_update = service.saved_searches[str(tracker_name)]
                        service.post(
                            "%s/%s" % (tracker_update.links["alternate"], str(action))
                        )
                        logger.info(
                            'CIM tracker object="'
                            + str(object)
                            + '", the tracker_name="'
                            + str(tracker_name)
                            + '" was successfully updated.'
                        )

                    except Exception as e:
                        logger.error(
                            'failed to update the tracker_name="'
                            + str(tracker_name)
                            + '" with exception: "'
                            + str(e)
                            + '"'
                        )

                        # increment failures counter
                        failures_count += 1

                        result = {
                            "object": object,
                            "action": "update",
                            "result": "failure",
                            "exception": f'tenant_id="{tenant_id}", failed to update the tracker_name, tracker_name="{tracker_name}", exception="{str(e)}"',
                        }
                        records.append(result)

                # Step 2: update the KVstore record

                try:
                    kvrecord["monitored_state"] = action_value
                    kvrecord["mtime"] = time.time()
                    collection.data.update(str(key), json.dumps(kvrecord))

                    # increment success counter
                    succcess_count += 1

                except Exception as e:
                    logger.error(
                        f'tenant_id="{tenant_id}", exception encountered, exception="{str(e)}"'
                    )

                    # increment failures counter
                    failures_count += 1

                    result = {
                        "object": object,
                        "action": "update",
                        "result": "failure",
                        "exception": f'tenant_id="{tenant_id}", failed to update the KVstore record, object="{object}", exception="{str(e)}"',
                    }
                    records.append(result)

                # append for summary
                result = {
                    "object": object,
                    "action": "update",
                    "result": "success",
                    "message": f'tenant_id="{tenant_id}", The entity was successfully updated',
                }
                records.append(result)

                # audit event
                trackme_audit_event(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    request_info.user,
                    "success",
                    "disable cim tracker",
                    str(object),
                    "splk-cim",
                    str(json.dumps(kvrecord, indent=2)),
                    "The cim entity and its associated objects were successfully disabled",
                    str(update_comment),
                )

                logger.info(
                    'tenant_id="{}", object="{}", action="success", the cim entity and its associated objects were successfully disabled'
                )

            except Exception as e:
                logger.error(
                    f'tenant_id="{tenant_id}", exception encountered, exception="{str(e)}"'
                )

                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                result = {
                    "object": object,
                    "action": "update",
                    "result": "failure",
                    "exception": f'tenant_id="{tenant_id}", failed to update the entity, object="{object}", exception="{str(e)}"',
                }
                records.append(result)

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "cim",
            ),
        )
        thread.start()

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            req_summary["action"] = "success"
            return {"payload": req_summary, "status": 200}

        else:
            req_summary["action"] = "failure"
            return {"payload": req_summary, "status": 500}
