#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_splk_dsm.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

import os, sys
import json
import time
import threading

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_dsm_power", "trackme_rest_api_splk_dsm_power.log"
)
# Redirect logger.module calls


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import (
    trackme_reqinfo,
    trackme_getloglevel,
    trackme_audit_event,
    trackme_register_tenant_component_summary,
    run_splunk_search,
)

# import trackme libs get data
from trackme_libs_get_data import (
    batch_find_records_by_object,
    batch_find_records_by_key,
)

# import trackme libs utils
from trackme_libs_utils import remove_leading_spaces, convert_time_to_seconds

# import trackme libs persistent fields definition
from collections_data import (
    persistent_fields_dsm,
)

# import trackme libs bulk edit
from trackme_libs_bulk_edit import post_bulk_edit, generic_batch_update


# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkDsm_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkDsm_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_dsm(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_dsm/write",
            "resource_group_desc": "Endpoints specific to the splk-dsm TrackMe component (Splunk Data Sources monitoring, power operations)",
        }

        return {"payload": response, "status": 200}

    def register_component_summary_async(
        self, session_key, splunkd_uri, tenant_id, component
    ):
        try:
            summary_register_response = trackme_register_tenant_component_summary(
                session_key,
                splunkd_uri,
                tenant_id,
                component,
            )
            logger.debug(
                f'function="trackme_register_tenant_component_summary", response="{json.dumps(summary_register_response, indent=2)}"'
            )
        except Exception as e:
            logger.error(
                f'failed to register the component summary with exception="{str(e)}"'
            )

    # Bulk edit (to be used from the inline Tabulator)
    def post_ds_bulk_edit(self, request_info, **kwargs):
        """
        This function performs a bulk edit on given json data.
        :param request_info: Contains request related information
        :param kwargs: Other keyword arguments
        :return: Status and payload of the bulk edit operation
        """

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # call the bulkd edit function
        response, http_status = post_bulk_edit(
            self,
            log=logger,
            loglevel=loglevel,
            service=service,
            request_info=request_info,
            component_name="dsm",
            persistent_fields=persistent_fields_dsm,
            collection_name_suffix="dsm",
            endpoint_suffix="dsm",
            **kwargs,
        )

        return {
            "payload": response,
            "status": http_status,
        }

    # Enable/Disable monitoring by object name
    def post_ds_monitoring(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                action = resp_dict["action"]
                if not action in ("enable", "disable"):
                    return {
                        "payload": "Invalid option for action, valid options are: enable | disable",
                        "status": 500,
                    }
                else:
                    if action == "enable":
                        action_value = "enabled"
                    elif action == "disable":
                        action_value = "disabled"

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint enables data monitoring for an existing data source by the data source name (object), it requires a POST call with the following information:",
                "resource_desc": "Enable/Disable monitoring for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_monitoring\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'action': 'disable', 'object_list': 'netscreen:netscreen:firewall,wineventlog:WinEventLog'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "action": "The action to be performed, valid options are: enable | disable",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "dsm",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {"monitored_state": action_value}

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_dsm,
            component="dsm",
            update_comment=update_comment,
            audit_context="update monitoring",
            audit_message="Monitoring state was updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Update priority by object name
    def post_ds_update_priority(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
                else:
                    describe = False
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                priority = resp_dict["priority"]
                if priority not in ("low", "medium", "high", "critical", "pending"):
                    return {
                        "payload": f"Invalid option for priority with priority received: {priority}, valid options are: low | medium | high | critical | pending",
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint defines the priority for an existing data source, it requires a POST call with the following information:",
                "resource_desc": "Update priority for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_update_priority\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'priority': 'high', 'object_list': 'netscreen:netscreen:firewall,wineventlog:WinEventLog'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "priority": "the value for priority, valid options are low / medium / high / critical / pending",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "dsm",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {"priority": priority, "priority_updated": 1}

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_dsm,
            component="dsm",
            update_comment=update_comment,
            audit_context="update priority",
            audit_message="Priority was updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Update lagging policy by object name
    def post_ds_update_lag_policy(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                try:
                    data_lag_alert_kpis = resp_dict[
                        "data_lag_alert_kpis"
                    ]  # all_kpis / lag_ingestion_kpi / lag_event_kpi
                    if not data_lag_alert_kpis in (
                        "all_kpis",
                        "lag_ingestion_kpi",
                        "lag_event_kpi",
                    ):
                        data_lag_alert_kpis = None
                except Exception as e:
                    data_lag_alert_kpis = None

                try:
                    data_max_lag_allowed = resp_dict["data_max_lag_allowed"]
                    data_max_lag_allowed = convert_time_to_seconds(data_max_lag_allowed)
                except Exception as e:
                    data_max_lag_allowed = None

                try:
                    data_max_delay_allowed = resp_dict["data_max_delay_allowed"]
                    data_max_delay_allowed = convert_time_to_seconds(
                        data_max_delay_allowed
                    )
                except Exception as e:
                    data_max_delay_allowed = None

                try:
                    data_override_lagging_class = resp_dict[
                        "data_override_lagging_class"
                    ]  # true / false
                    if not data_override_lagging_class in ("true", "false"):
                        data_override_lagging_class = None
                except Exception as e:
                    data_override_lagging_class = None

                try:
                    allow_adaptive_delay = resp_dict[
                        "allow_adaptive_delay"
                    ]  # true / false
                    if not allow_adaptive_delay in ("true", "false"):
                        allow_adaptive_delay = None
                except Exception as e:
                    allow_adaptive_delay = None

                try:
                    future_tolerance = resp_dict["future_tolerance"]
                except Exception as e:
                    future_tolerance = "system"

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint configures the lagging policy for an existing data source, it requires a POST call with the following information:",
                "resource_desc": "Update lag policies for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_update_lag_policy\" mode=\"post\" body=\"{'tenant_id':'mytenant','object_list':'netscreen:netscreen:firewall','data_max_lag_allowed':'1h','data_max_delay_allowed':'1d','data_lag_alert_kpis':'all_kpis','data_override_lagging_class':'false'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "data_lag_alert_kpis": "OPTIONAL, KPIs policy to be applied, valid options are all_kpis / lag_ingestion_kpi / lag_event_kpi",
                        "data_max_lag_allowed": "OPTIONAL, maximal accepted lagging value in seconds or with unit suffix (h/d/w), e.g. 3600 or '1h' or '1d' or '1w'",
                        "data_max_delay_allowed": "OPTIONAL, maximal accepted delay value in seconds or with unit suffix (h/d/w), e.g. 3600 or '1h' or '1d' or '1w'",
                        "data_override_lagging_class": "OPTIONAL, overrides lagging classes, valid options are true / false",
                        "future_tolerance": "OPTIONAL, the negative value for future tolerance, specify system to rely on the system level setting, disabled to allow data in the future up to 7 days without affecting the status of the entity",
                        "allow_adaptive_delay": "OPTIONAL, allow adaptive delay, valid options are true / false",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "dsm",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {}
        if data_max_lag_allowed:
            update_fields["data_max_lag_allowed"] = data_max_lag_allowed
        if data_max_delay_allowed:
            update_fields["data_max_delay_allowed"] = data_max_delay_allowed
        if data_lag_alert_kpis:
            update_fields["data_lag_alert_kpis"] = data_lag_alert_kpis
        if data_override_lagging_class:
            update_fields["data_override_lagging_class"] = data_override_lagging_class
        if allow_adaptive_delay:
            update_fields["allow_adaptive_delay"] = allow_adaptive_delay
        if future_tolerance:
            update_fields["future_tolerance"] = future_tolerance
        if not update_fields:
            return {
                "payload": {"error": "no valid fields to update"},
                "status": 500,
            }

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_dsm,
            component="dsm",
            update_comment=update_comment,
            audit_context="update monitoring lag policy",
            audit_message="Lag policy was updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Update min dcount host by object name
    def post_ds_update_min_dcount_host(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                min_dcount_host = resp_dict["min_dcount_host"]
                # We need to accept the string any (to disable the fearure) or an integer
                if str(min_dcount_host) not in ("any"):
                    # anything else than "any" or an integer will fail here
                    min_dcount_host = int(min_dcount_host)

                # min_dcount_field is optional, if not provided we will use global_dcount_host
                try:
                    min_dcount_field = resp_dict["min_dcount_field"]
                    if not min_dcount_field in (
                        "avg_dcount_host_5m",
                        "latest_dcount_host_5m",
                        "perc95_dcount_host_5m",
                        "stdev_dcount_host_5m",
                        "global_dcount_host",
                    ):
                        min_dcount_field = "global_dcount_host"
                except Exception as e:
                    min_dcount_field = "global_dcount_host"

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint configures the minimal number of distinct hosts count for an existing data source, it requires a POST call with the following information:",
                "resource_desc": "Update lag policies for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_update_min_dcount_host\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'min_dcount_host': '0', 'min_dcount_field': 'stdev_dcount_host_5m', 'object_list': 'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "min_dcount_host": "minimal accepted number of distinct count hosts, must be an integer or the string any",
                        "min_dcount_field": "The dictinct count metric to be used for this entity, valid options are: avg_dcount_host_5m, latest_dcount_host_5m, perc95_dcount_host_5m, stdev_dcount_host_5m, global_dcount_host",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "dsm",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {
            "min_dcount_host": min_dcount_host,
            "min_dcount_field": min_dcount_field,
        }

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_dsm,
            component="dsm",
            update_comment=update_comment,
            audit_context="update min disctinct host count",
            audit_message="The min disctinct count number of hosts was updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Update monitoring week days by object name
    def post_ds_update_wdays(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                # Week days monitoring can be:
                # manual:all_days / manual:monday-to-friday / manual:monday-to-saturday / [ 0, 1, 2, 3, 4, 5, 6 ] where Sunday is 0
                monitoring_wdays = resp_dict["monitoring_wdays"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint configures the week days monitoring rule for an existing data source, it requires a POST call with the following information:",
                "resource_desc": "Update week days monitoring for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_update_wdays\" mode=\"post\" body=\"{'tenant_id':'mytenant','object_list':'netscreen:netscreen:firewall','data_monitoring_wdays':'manual:1,2,3,4,5'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "monitoring_wdays": "the week days rule, valid options are manual:all_days / manual:monday-to-friday / manual:monday-to-saturday / [ 0, 1, 2, 3, 4, 5, 6 ] where Sunday is 0",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "dsm",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {"data_monitoring_wdays": monitoring_wdays}

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_dsm,
            component="dsm",
            update_comment=update_comment,
            audit_context="update week days monitoring",
            audit_message="Week days monitoring was updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Update monitoring hours ranges by object name
    def post_ds_update_hours_ranges(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                monitoring_hours_ranges = resp_dict["monitoring_hours_ranges"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint configures the week days monitoring rule for an existing data source, it requires a POST call with the following information:",
                "resource_desc": "Update hours of monitoring for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_update_hours_ranges\" mode=\"post\" body=\"{'tenant_id':'mytenant', 'object_list':'netscreen:netscreen:firewall', 'data_monitoring_hours_ranges':'manual:8,9,10,11,12,13,14,15,16,17'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "monitoring_hours_ranges": "the hours ranges rule, valid options are manual:all_ranges / manual:08h-to-20h / [ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 ] where 00h00 to 01h59 is 0",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "dsm",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {"data_monitoring_hours_ranges": monitoring_hours_ranges}

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_dsm,
            component="dsm",
            update_comment=update_comment,
            audit_context="update hours ranges monitoring",
            audit_message="Monitoring hours ranges were updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Remove object entities
    def post_ds_delete(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                deletion_type = resp_dict["deletion_type"]
                if deletion_type not in ("temporary", "permanent"):
                    return {
                        "payload": {
                            "error": "Invalid option for deletion_type, valid options are: tempoary | permanent"
                        },
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint performs a temporary deletion of an existing data source, it requires a POST call with the following information:",
                "resource_desc": "Delete one or more entities, either temporarily or permanently",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_delete\" mode=\"post\" body=\"{'tenant_id':'mytenant', 'deletion_type': 'temporary', 'object_list':'netscreen:netscreen:firewall,wineventlog:WinEventLog'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "deletion_type": "The type of deletion, valid options are: temporary | permanent",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Permanently deleted objects
        collection_perm_deleted_name = (
            f"kv_trackme_common_permanently_deleted_objects_tenant_{tenant_id}"
        )
        collection_perm_deleted = service.kvstore[collection_perm_deleted_name]

        #
        # Outliers collections
        #

        # entity rules collection
        collection_outliers_entity_rules_name = (
            f"kv_trackme_dsm_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_entity_rules = service.kvstore[collection_outliers_entity_rules_name]

        # data rules collection
        collection_outliers_entity_data_name = (
            f"kv_trackme_dsm_outliers_entity_data_tenant_{tenant_id}"
        )
        collection_outliers_entity_data = service.kvstore[
            collection_outliers_entity_data_name
        ]

        #
        # Data sampling collection
        #

        # sampling
        collection_sampling_name = f"kv_trackme_dsm_data_sampling_tenant_{tenant_id}"
        collection_sampling = service.kvstore[collection_sampling_name]

        # Convert comma-separated lists to Python lists if needed
        if isinstance(object_list, str):
            object_list = object_list.split(",")
        if isinstance(keys_list, str):
            keys_list = keys_list.split(",")

        # Determine query method based on input
        if object_list:
            kvrecords_dict, kvrecords = batch_find_records_by_object(
                collection, object_list
            )
        elif keys_list:
            kvrecords_dict, kvrecords = batch_find_records_by_key(collection, keys_list)
        else:
            return {
                "payload": {
                    "error": "either object_list or keys_list must be provided"
                },
                "status": 500,
            }

        # proceed
        for kvrecord in kvrecords:

            key = kvrecord.get("_key")
            object_value = kvrecord.get("object")

            try:
                # Remove the record
                collection.data.delete(json.dumps({"_key": key}))

                if deletion_type == "permanent":
                    # Register a new permanently deleted object
                    collection_perm_deleted.data.insert(
                        json.dumps(
                            {
                                "ctime": str(time.time()),
                                "object": str(object_value),
                                "object_category": "splk-dsm",
                            }
                        )
                    )

                # try deleting outliers records for rules and data, there might be nothing to delete
                try:
                    collection_entity_rules.data.delete(
                        json.dumps({"object": object_value})
                    )
                except Exception as e:
                    pass

                try:
                    collection_outliers_entity_data.data.delete(
                        json.dumps({"object": object_value})
                    )
                except Exception as e:
                    pass

                # try deleting data sampling records, there might be nothing to delete
                try:
                    collection_sampling.data.delete(
                        json.dumps({"object": object_value})
                    )
                except Exception as e:
                    pass

                # Record an audit change
                trackme_audit_event(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    request_info.user,
                    "success",
                    f"delete {deletion_type}",
                    str(object_value),
                    "splk-dsm",
                    str(json.dumps(kvrecord, indent=1)),
                    "Entity was temporarily deleted successfully",
                    str(update_comment),
                )

                # increment counter
                processed_count += 1
                succcess_count += 1
                failures_count += 0

                # append for summary
                result = {
                    "object": object_value,
                    "action": "delete",
                    "result": "success",
                    "message": f'tenant_id="{tenant_id}", The object was successfully deleted',
                }
                records.append(result)

            except Exception as e:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                result = {
                    "object": object_value,
                    "action": "delete",
                    "result": "failure",
                    "exception": f'tenant_id="{tenant_id}", failed to remove the entity, object="{object_value}", exception="{str(e)}"',
                }
                records.append(result)

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "dsm",
            ),
        )
        thread.start()

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Enable/Disable Data Sampling by object name
    def post_ds_manage_data_sampling(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                action = resp_dict["action"]
                if action not in ("enable", "disable", "reset", "run"):
                    return {
                        "payload": 'invalid value for action="{}", valid options are: enable | disable | reset| run',
                        "status": 200,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint allows management of the data sampling feature for an existing data source by the data source name (object), it requires a POST call with the following information:",
                "resource_desc": "Enable/Disable/Reset/Run Data Sampling for one or more entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_manage_data_sampling\" mode=\"post\" body=\"{'tenant_id':'mytenant', 'action': 'reset', 'object_list': 'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "action": "the action to be performed, valid options are: enable | disable | reset | run",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Set the value for data_sample_feature
        if action == "enable":
            data_sample_feature = "enabled"
        elif action == "disable":
            data_sample_feature = "disabled"
        elif action == "reset":
            data_sample_feature = "enabled"
        elif action == "run":
            data_sample_feature = "enabled"

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        # Data collections

        # component
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # sampling
        collection_sampling_name = f"kv_trackme_dsm_data_sampling_tenant_{tenant_id}"
        collection_sampling = service.kvstore[collection_sampling_name]

        # loop and proceed
        if object_list:
            keys_list = []
            for object_value in object_list:
                try:
                    kvrecord = collection.data.query(
                        query=json.dumps({"object": object_value})
                    )[0]
                    key = kvrecord.get("_key")
                    keys_list.append(key)
                except Exception as e:
                    kvrecord = {}
                    key = None

        for key in keys_list:
            try:
                # check if we have a KVrecord already for this object
                query_string = {
                    "$and": [
                        {
                            "_key": key,
                        }
                    ]
                }

                # record from the component
                try:
                    kvrecord = collection.data.query(query=(json.dumps(query_string)))[
                        0
                    ]
                    # object_value
                    object_value = kvrecord.get("object")
                except Exception as e:
                    kvrecord = {}

                # if this entity could not be found
                if not kvrecord:
                    logger.error(
                        f'tenant_id="{tenant_id}", entity not found, object="{object_value}"'
                    )

                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    result = {
                        "object": object_value,
                        "action": action,
                        "result": "failure",
                        "exception": f'tenant_id="{tenant_id}", entity not found, object="{object_value}"',
                    }
                    records.append(result)

                # record from sampling
                try:
                    # try get to get the key
                    kvrecord_sampling = collection_sampling.data.query(
                        query=(json.dumps(query_string))
                    )[0]
                    key_sampling = kvrecord_sampling.get("_key")
                except Exception as e:
                    key_sampling = None
                    kvrecord_sampling = {}

                # if action is enable / disable
                if action in ("enable", "disable"):

                    try:

                        if not kvrecord_sampling:

                            if action == "enable":

                                kvrecord_sampling = {
                                    "_key": key,
                                    "anomaly_reason": "pending",
                                    "current_detected_format": ["pending"],
                                    "data_sample_anomaly_detected": 0,
                                    "data_sample_feature": "pending",
                                    "data_sample_mtime": "pending",
                                    "data_sample_status_colour": "yellow",
                                    "data_sample_status_message": json.dumps(
                                        {
                                            "state": "pending",
                                            "desc": "Data Sampling is pending and was reset for this entity",
                                        }
                                    ),
                                    "data_sampling_nr": trackme_conf["trackme_conf"][
                                        "splk_data_sampling"
                                    ][
                                        "splk_data_sampling_default_sample_record_at_run"
                                    ],
                                    "direction": "none",
                                    "data_sample_feature": "enabled",
                                    "mtime": "pending",
                                    "object": object_value,
                                    "previous_detected_format": ["pending"],
                                }

                            elif action == "disable":

                                kvrecord_sampling = {
                                    "_key": key,
                                    "anomaly_reason": "N/A",
                                    "current_detected_format": ["N/A"],
                                    "data_sample_anomaly_detected": 0,
                                    "data_sample_feature": "N/A",
                                    "data_sample_mtime": "N/A",
                                    "data_sample_status_colour": "N/A",
                                    "data_sample_status_message": json.dumps(
                                        {
                                            "state": "pending",
                                            "desc": "Data Sampling is pending and was reset for this entity",
                                        }
                                    ),
                                    "data_sampling_nr": "N/A",
                                    "direction": "N/A",
                                    "data_sample_feature": "disabled",
                                    "mtime": "N/A",
                                    "object": object_value,
                                    "previous_detected_format": ["N/A"],
                                }

                            # insert the record
                            collection_sampling.data.insert(
                                json.dumps(kvrecord_sampling)
                            )

                        else:

                            # Update the record
                            kvrecord_sampling["data_sample_feature"] = (
                                data_sample_feature
                            )
                            collection_sampling.data.update(
                                key, json.dumps(kvrecord_sampling)
                            )

                        # Record an audit change
                        record = {
                            "object": str(object_value),
                            "data_sample_feature": str(data_sample_feature),
                        }
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            f"{action} data sampling",
                            str(object_value),
                            "splk-dsm",
                            str(json.dumps(record, indent=1)),
                            "Data sampling was managed successfully",
                            str(update_comment),
                        )

                        logger.info(
                            f'tenant_id="{tenant_id}", The object was successfully updated'
                        )

                        # increment counter
                        processed_count += 1
                        succcess_count += 1
                        failures_count += 0

                        # append for summary
                        result = {
                            "object": object_value,
                            "action": action,
                            "result": "success",
                            "message": f'tenant_id="{tenant_id}", The object was successfully updated',
                        }
                        records.append(result)

                    except Exception as e:
                        logger.error(
                            f'tenant_id="{tenant_id}", failed to update the entity, object="{object_value}", exception="{str(e)}"'
                        )

                        # increment counter
                        processed_count += 1
                        succcess_count += 0
                        failures_count += 1

                        result = {
                            "object": object_value,
                            "action": action,
                            "result": "failure",
                            "exception": f'tenant_id="{tenant_id}", failed to update the entity, object="{object_value}", exception="{str(e)}"',
                        }
                        records.append(result)

                # if action is reset
                elif action in ("reset", "run"):
                    try:

                        #
                        # reset
                        #

                        if action == "reset":

                            if (
                                kvrecord_sampling
                            ):  # reset is only possible if the record exists
                                try:
                                    # reset the sampling record
                                    current_data_sampling_nr = kvrecord.get("kvrecord")
                                    kvrecord_sampling = {
                                        "anomaly_reason": "pending",
                                        "current_detected_format": ["pending"],
                                        "data_sample_anomaly_detected": 0,
                                        "data_sample_feature": "pending",
                                        "data_sample_mtime": "pending",
                                        "data_sample_status_colour": "yellow",
                                        "data_sample_status_message": json.dumps(
                                            {
                                                "state": "pending",
                                                "desc": "Data Sampling is pending and was reset for this entity",
                                            }
                                        ),
                                        "data_sampling_nr": current_data_sampling_nr,
                                        "direction": "none",
                                        "data_sample_feature": "enabled",
                                        "mtime": "pending",
                                        "object": object_value,
                                        "previous_detected_format": ["pending"],
                                    }
                                    collection_sampling.data.update(
                                        str(key_sampling), json.dumps(kvrecord_sampling)
                                    )
                                    logger.info(
                                        f'tenant_id="{tenant_id}", object="{object_value}", action="{action}", KVstore sampling record was successfully reset'
                                    )

                                    # audit
                                    trackme_audit_event(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        request_info.user,
                                        "success",
                                        "reset data sampling",
                                        str(object_value),
                                        "splk-dsm",
                                        str(json.dumps(kvrecord_sampling, indent=1)),
                                        "Data sampling was reset successfully",
                                        str(update_comment),
                                    )

                                except Exception as e:
                                    logger.error(
                                        f'tenant_id="{tenant_id}", object="{object_value}", failed to update the KVstore sampling record with exception="{str(e)}"'
                                    )

                                try:
                                    # reset the record
                                    kvrecord["isAnomaly"] = 0
                                    kvrecord["data_sample_lastrun"] = 0

                                    collection.data.update(
                                        str(key), json.dumps(kvrecord)
                                    )
                                    logger.info(
                                        f'tenant_id="{tenant_id}", object="{object_value}", action="{action}", KVstore entity record was successfully reset'
                                    )

                                except Exception as e:
                                    logger.error(
                                        f'tenant_id="{tenant_id}", object="{object_value}", failed to update the entity KVstore record with exception="{str(e)}"'
                                    )

                        #
                        # run
                        #

                        # Define the SPL query
                        kwargs_search = {
                            "app": "trackme",
                            "earliest_time": "-5m",
                            "latest_time": "now",
                            "output_mode": "json",
                            "count": 0,
                        }
                        searchquery = remove_leading_spaces(
                            f"""\
                            | trackmesamplingexecutor tenant_id={tenant_id} object="{object_value}" mode="run_sampling"
                            """
                        )

                        # log debug
                        logger.debug(
                            f'tenant_id="{tenant_id}", object="{object_value}", searchquery="{searchquery}"'
                        )

                        query_results = []
                        # spawn the search and get the results
                        reader = run_splunk_search(
                            service,
                            searchquery,
                            kwargs_search,
                            24,
                            5,
                        )

                        for item in reader:
                            if isinstance(item, dict):
                                query_results.append(item)

                        # increment counter
                        processed_count += 1
                        succcess_count += 1
                        failures_count += 0

                        # append for summary
                        result = {
                            "object": object_value,
                            "action": action,
                            "result": "success",
                            "message": f'tenant_id="{tenant_id}", The action="{action}" was successfully performed',
                            "searchquery": searchquery,
                            "results": query_results,
                        }
                        records.append(result)

                    except Exception as e:
                        logger.error(
                            f'tenant_id="{tenant_id}", failed to perform the action="{action}" for the entity, object="{object_value}", exception="{str(e)}"'
                        )

                        # increment counter
                        processed_count += 1
                        succcess_count += 0
                        failures_count += 1

                        result = {
                            "object": object_value,
                            "action": action,
                            "result": "failure",
                            "exception": f'tenant_id="{tenant_id}", failed to perform the action="{action}" for the entity, object="{object_value}", exception="{str(e)}"',
                        }
                        records.append(result)

            except Exception as e:
                logger.error(
                    f'tenant_id="{tenant_id}", general exception, object="{object_value}", exception="{str(e)}"'
                )

                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                result = {
                    "object": object_value,
                    "action": action,
                    "result": "failure",
                    "exception": f'tenant_id="{tenant_id}", failed to update the entity, object="{object_value}", general exception="{str(e)}"',
                }
                records.append(result)

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Update entity specific data sampling settings
    def post_ds_update_data_sampling_entity_settings(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                # get params
                min_time_btw_iterations_seconds = resp_dict.get(
                    "min_time_btw_iterations_seconds", None
                )
                pct_min_major_inclusive_model_match = resp_dict.get(
                    "pct_min_major_inclusive_model_match", None
                )
                pct_max_exclusive_model_match = resp_dict.get(
                    "pct_max_exclusive_model_match", None
                )
                max_events_per_sampling_iteration = resp_dict.get(
                    "max_events_per_sampling_iteration", None
                )
                relative_time_window_seconds = resp_dict.get(
                    "relative_time_window_seconds", None
                )

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint allows managing data sampling entities settings, it requires a POST call with the following information:",
                "resource_desc": "Update data sampling entities settings for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_update_data_sampling_entity_settings\" mode=\"post\" body=\"{'tenant_id':'mytenant', 'object_list': 'netscreen:netscreen:firewall', 'pct_min_major_inclusive_model_match': 95, 'max_events_per_sampling_iteration': 0, 'max_events_per_sampling_iteration': 10000, 'relative_time_window_seconds': 3600}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "min_time_btw_iterations_seconds": "OPTIONAL: the minimum time in seconds between sampling iterations, ex: 3600",
                        "pct_min_major_inclusive_model_match": "OPTIONAL: the minimum percentage of major inclusive model match per sampling iteration, ex: 95",
                        "pct_max_exclusive_model_match": "OPTIONAL: the maximum percentage of exclusive model match per sampling iteration, ex: 0",
                        "max_events_per_sampling_iteration": "OPTIONAL: the maximum number of events per sampling iteration, ex: 10000",
                        "relative_time_window_seconds": "OPTIONAL: the size in seconds of the time window for the sampling operation, relative to the latest event time know for the entity. This setting is used to calculate the earliest_time when performing the sampling search, for instance 3600 means the search will run against the time window will cover up to 1 hour of events according to the latest event time known for the entity, ex: 3600",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        collection_name = f"kv_trackme_dsm_data_sampling_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "dsm",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {}
        if min_time_btw_iterations_seconds:
            update_fields["min_time_btw_iterations_seconds"] = (
                min_time_btw_iterations_seconds
            )
        if pct_min_major_inclusive_model_match:
            update_fields["pct_min_major_inclusive_model_match"] = (
                pct_min_major_inclusive_model_match
            )
        if pct_max_exclusive_model_match:
            update_fields["pct_max_exclusive_model_match"] = (
                pct_max_exclusive_model_match
            )
        if max_events_per_sampling_iteration:
            update_fields["max_events_per_sampling_iteration"] = (
                max_events_per_sampling_iteration
            )
        if relative_time_window_seconds:
            update_fields["relative_time_window_seconds"] = relative_time_window_seconds

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_dsm,
            component="dsm",
            update_comment=update_comment,
            audit_context="update data sampling entity settings",
            audit_message="Data sampling entity settings were updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Update list of manual tags
    def post_ds_update_manual_tags(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                try:
                    tags_manual = resp_dict["tags_manual"]
                    # if is a list, deduplicate, make it lowercase, sort it and turn as a CSV string
                    if isinstance(tags_manual, list):
                        tags_manual = ",".join(
                            sorted(list(set([x.lower() for x in tags_manual])))
                        )
                    else:
                        # if is a string, split it, deduplicate, make it lowercase, sort it and turn as a CSV string
                        tags_manual = ",".join(
                            sorted(
                                list(set([x.lower() for x in tags_manual.split(",")]))
                            )
                        )

                except Exception as e:
                    return {
                        "payload": {
                            "error": "tags_manual must be provided as a comma seperated list of tags"
                        },
                        "status": 500,
                    }

        else:
            describe = True

        if describe:
            response = {
                "describe": "This endpoint allows defining a comma seperated list of manual tags, it requires a POST call with the following information:",
                "resource_desc": "Define a comma seperated list of tags for one or more entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_update_manual_tags\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'tags_manual': 'mytag1,maytag2,mytag3', 'object_list': 'netscreen:netscreen:firewall,wineventlog:WinEventLog'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "tags_manual": "A comma seperated list of tags to be applied to the entities, to purge all manual tags, send an empty string",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Tags policies collection
        collection_tags_policies_name = f"kv_trackme_dsm_tags_tenant_{tenant_id}"
        collection_tags_policies = service.kvstore[collection_tags_policies_name]

        # loop and proceed
        if object_list:
            keys_list = []
            for object_value in object_list:
                try:
                    kvrecord = collection.data.query(
                        query=json.dumps({"object": object_value})
                    )[0]
                    key = kvrecord.get("_key")
                    keys_list.append(key)
                except Exception as e:
                    key = None

        for key in keys_list:
            try:
                kvrecord = collection.data.query(query=json.dumps({"_key": key}))[0]

                # check if we have tags policies already
                try:
                    kvrecord_tags_policies = collection_tags_policies.data.query(
                        query=json.dumps({"_key": key})
                    )[0]
                except Exception as e:
                    kvrecord_tags_policies = None

                # check if we have tags_auto (list)
                try:
                    tags_auto = kvrecord_tags_policies.get("tags_auto", [])
                except Exception as e:
                    tags_auto = []

                # Update the record
                object_value = kvrecord.get("object")
                tags = kvrecord.get("tags", None)  # get current tags

                # if we have tags, the format is CSV, turn into a list
                if tags:
                    tags = tags.split(",")

                # update the record with ours manual tags
                kvrecord["tags_manual"] = tags_manual

                # make tags_manual_list (list from tags_manual CSV)
                tags_manual_list = tags_manual.split(",")

                # merged them all: define the tags field as the deduplicated, lowercase and sorted list of tags based on the tags_auto and tags_manual_list
                tags = ",".join(
                    sorted(
                        list(
                            set([x.lower() for x in tags_auto + tags_manual_list if x])
                        )
                    )
                )
                # update tags in the kvrecord now
                kvrecord["tags"] = tags
                kvrecord["mtime"] = time.time()
                collection.data.update(str(key), json.dumps(kvrecord))

                # Record an audit change
                trackme_audit_event(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    request_info.user,
                    "success",
                    "update manual tags",
                    str(object_value),
                    "splk-dsm",
                    {"tags_manual": tags_manual_list},
                    "Manual tags list was updated successfully",
                    str(update_comment),
                )

                # increment counter
                processed_count += 1
                succcess_count += 1
                failures_count += 0

                # append for summary
                result = {
                    "object": object_value,
                    "action": "update",
                    "result": "success",
                    "message": f'tenant_id="{tenant_id}", The object was successfully updated',
                }
                records.append(result)

            except Exception as e:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                result = {
                    "object": object_value,
                    "action": "update",
                    "result": "failure",
                    "exception": f'tenant_id="{tenant_id}", failed to update the entity, object="{object_value}", exception="{str(e)}"',
                }
                records.append(result)

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "dsm",
            ),
        )
        thread.start()

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Update SLA class
    def post_ds_update_sla_class(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                try:
                    sla_class = resp_dict["sla_class"]
                except Exception as e:
                    return {
                        "payload": {"error": "sla_class must be provided"},
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint updates the SLA class per entity, it requires a POST call with the following information:",
                "resource_desc": "Update SLA class for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_dsm/write/ds_update_sla_class\" mode=\"post\" body=\"{'tenant_id':'mytenant','object_list':'netscreen:netscreen:firewall','sla_class':'gold'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "sla_class": "(required) The SLA class to be applied to the entities",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "dsm",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {
            "sla_class": sla_class,
        }

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_dsm,
            component="dsm",
            update_comment=update_comment,
            audit_context="update SLA class",
            audit_message="SLA class was updated successfully",
        )

        return {"payload": response, "status": status_code}
