#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_identity_cards.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_identity_cards_user",
    "trackme_rest_api_splk_identity_cards_user.log",
)
# Redirect logger.module calls


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkIdentityCards_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkIdentityCards_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_identity_cards(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_identity_cards",
            "resource_group_desc": "Endpoints related to the management of Data identity cards (read only operations)",
        }

        return {"payload": response, "status": 200}

    # Get the entire data sources collection as a Python array
    def post_identity_cards_collection(self, request_info, **kwargs):
        """
        | trackme mode="post" url="/services/trackme/v2/splk_identity_cards/identity_cards_collection" body="{'tenant_id': 'mytenant'}"
        """

        # define
        tenant_id = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        if describe:
            response = {
                "describe": "This endpoint retrieves the entire Identity Cards collection returned as a JSON array, it requires a GET call with no data required",
                "resource_desc": "Get identity cards for the tenant",
                "resource_spl_example": '| trackme mode="post" url="/services/trackme/v2/splk_identity_cards/identity_cards_collection" body="{\'tenant_id\': \'mytenant\'}"',
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            collection_name = "kv_trackme_dsm_knowledge_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            # Render
            return {"payload": collection.data.query(), "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Get card for an object
    def post_identity_cards_get_card(self, request_info, **kwargs):
        """
        | trackme mode=\"post\" url=\"/services/trackme/v2/splk_identity_cards/identity_cards_get_card\" body=\"{'tenant_id': 'mytenant', 'object': 'wineventlog:WinEventLog'}\"
        """

        # Define
        tenant_id = None
        object_value = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                object_value = resp_dict["object"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint retrieve the identity card linked to a specific object, it requires a GET call with the following information:",
                "resource_desc": "Get identity card for a given object",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_identity_cards/identity_cards_get_card\" body=\"{'tenant_id': 'mytenant', 'object': 'wineventlog:WinEventLog'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "object": "object identifier",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Define the KV query
            query_string = {
                "object": object_value,
            }

            # collection
            collection_name = "kv_trackme_dsm_knowledge_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                return {"payload": collection.data.query_by_id(key), "status": 200}

            else:
                response = {
                    "action": "failure",
                    "response": f'object="{object_value}" not found',
                }
                logger.error(json.dumps(response))
                return {"payload": response, "status": 404}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Get card by doc_link
    def post_identity_cards_get_card_by_doc_link(self, request_info, **kwargs):
        """
        | trackme mode=\"post\" url=\"/services/trackme/v2/splk_identity_cards/identity_cards_get_card_by_doc_link\" body=\"{'tenant_id': 'mytenant', 'doc_link': 'https://www.mylink.com/mypage'}\"
        """

        # Define
        tenant_id = None
        doc_link = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                doc_link = resp_dict["doc_link"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint retrieve the identity card by a dgiven oc_link, it requires a GET call with the following information:",
                "resource_desc": "Get identity card for a given object",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_identity_cards/identity_cards_get_card_by_doc_link\" body=\"{'tenant_id': 'mytenant', 'doc_link': 'https://www.mylink.com/mypage'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "doc_link": "value of the doc link",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Define the KV query
            query_string = {
                "doc_link": doc_link,
            }

            # collection
            collection_name = "kv_trackme_dsm_knowledge_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                return {"payload": collection.data.query_by_id(key), "status": 200}

            else:
                response = {
                    "action": "failure",
                    "response": f'doc_link="{doc_link}" not found',
                }
                logger.error(json.dumps(response))
                return {"payload": response, "status": 404}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}
