#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_splk_dhm.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys
import requests
import time

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_outliers_engine_user",
    "trackme_rest_api_splk_outliers_engine_user.log",
)
# Redirect logger.module calls


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel, run_splunk_search

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkOutliersEngine_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkOutliersEngine_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_outliers_engine(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_outliers_engine",
            "resource_group_desc": "Endpoints related to the management of the Machine Learning Outliers detection (read only operations)",
        }

        return {"payload": response, "status": 200}

    # get Machine Learning models and rules
    def post_outliers_get_rules(self, request_info, **kwargs):

        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "tenant_id is required",
                        },
                        "status": 500,
                    }
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "object is required",
                        },
                        "status": 500,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "component is required",
                        },
                        "status": 500,
                    }

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint retrieves the Machine Learning outliers models and rules, it requires a POST call with the following options:",
                "resource_desc": "Get Machine Learning outliers models and rules",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/outliers_get_rules\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'object': 'myobject'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name, use a * to match all entities",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Define the SPL query
        kwargs_search = {
            "app": "trackme",
            "earliest_time": "-5m",
            "latest_time": "now",
            "output_mode": "json",
            "count": 0,
        }
        searchquery = f'| trackmesplkoutliersgetrules tenant_id="{tenant_id}" component="{component}" object="{object_value}"'

        query_results = []
        try:
            # spawn the search and get the results
            reader = run_splunk_search(
                service,
                searchquery,
                kwargs_search,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    query_results.append(item)
            return {"payload": query_results, "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # get Machine Learning list of models for a given object
    def post_outliers_get_models(self, request_info, **kwargs):

        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "tenant_id is required",
                        },
                        "status": 500,
                    }
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "object is required",
                        },
                        "status": 500,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "component is required",
                        },
                        "status": 500,
                    }

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint retrieves the Machine Learning outliers models, it requires a POST call with the following options:",
                "resource_desc": "Get Machine Learning outliers list of models for a given entity",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/outliers_get_models\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'object': 'myobject'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name, use a * to match all entities",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Define the SPL query
        kwargs_search = {
            "app": "trackme",
            "earliest_time": "-5m",
            "latest_time": "now",
            "output_mode": "json",
            "count": 0,
        }
        searchquery = f'| trackmesplkoutliersgetrules tenant_id="{tenant_id}" component="{component}" object="{object_value}"'

        models_list = []
        try:
            # spawn the search and get the results
            reader = run_splunk_search(
                service,
                searchquery,
                kwargs_search,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    model_id = item.get("model_id")
                    if model_id:
                        models_list.append(model_id)
            return {"payload": {"models": models_list}, "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # get Machine Learning models and rules
    def post_outliers_cim_get_rules(self, request_info, **kwargs):

        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "tenant_id is required",
                        },
                        "status": 500,
                    }
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "object is required",
                        },
                        "status": 500,
                    }
                try:
                    entity_name = resp_dict["entity"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "entity is required",
                        },
                        "status": 500,
                    }
                try:
                    cim_field = resp_dict["cim_field"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "cim_field is required",
                        },
                        "status": 500,
                    }

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint retrieves the Machine Learning outliers models and rules, it requires a POST call with the following options:",
                "resource_desc": "Get Machine Learning outliers models and rules",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/outliers_cim_get_rules\" body=\"{'tenant_id': 'mytenant', 'object': '*', 'entity': '*', 'cim_field':'*'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "object": "(required) object name, use a * to match all objects",
                        "entity": "(required) cim entity name, use a * to match all entities",
                        "cim_field": "(required) cim field, use a * to match all fields",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Define the SPL query
        kwargs_search = {
            "app": "trackme",
            "earliest_time": "-5m",
            "latest_time": "now",
            "output_mode": "json",
            "count": 0,
        }
        searchquery = f'| trackmesplkoutlierscimgetrules tenant_id="{tenant_id}" component="cim" object="{object_value}" entity="{entity_name}" cim_field="{cim_field}"'

        query_results = []
        try:
            # spawn the search and get the results
            reader = run_splunk_search(
                service,
                searchquery,
                kwargs_search,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    query_results.append(item)
            return {"payload": query_results, "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # get Machine Learning list of models for a given cim entity
    def post_outliers_cim_get_models(self, request_info, **kwargs):

        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "tenant_id is required",
                        },
                        "status": 500,
                    }
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "object is required",
                        },
                        "status": 500,
                    }
                try:
                    entity_name = resp_dict["entity"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "entity is required",
                        },
                        "status": 500,
                    }
                try:
                    cim_field = resp_dict["cim_field"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "cim_field is required",
                        },
                        "status": 500,
                    }

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint retrieves the Machine Learning outliers models, it requires a POST call with the following options:",
                "resource_desc": "Get Machine Learning outliers list of models for a given cim entity",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/outliers_cim_get_models\" body=\"{'tenant_id': 'mytenant', 'object': '*', 'entity': '*', 'cim_field':'*'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "object": "(required) object name, use a * to match all objects",
                        "entity": "(required) cim entity name, use a * to match all entities",
                        "cim_field": "(required) cim field, use a * to match all fields",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Define the SPL query
        kwargs_search = {
            "app": "trackme",
            "earliest_time": "-5m",
            "latest_time": "now",
            "output_mode": "json",
            "count": 0,
        }
        searchquery = f'| trackmesplkoutlierscimgetrules tenant_id="{tenant_id}" component="cim" object="{object_value}" entity="{entity_name}" cim_field="{cim_field}"'

        models_list = []
        try:
            # spawn the search and get the results
            reader = run_splunk_search(
                service,
                searchquery,
                kwargs_search,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    model_id = item.get("model_id")
                    if model_id:
                        models_list.append(model_id)
            return {"payload": {"models": models_list}, "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # render ML entity model
    def post_outliers_render_entity_model(self, request_info, **kwargs):
        describe = False

        logger.debug(f"Starting function post_outliers_render_entity_model")

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "tenant_id is required",
                        },
                        "status": 500,
                    }

                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "object is required",
                        },
                        "status": 500,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "component is required",
                        },
                        "status": 500,
                    }

                try:
                    mode = resp_dict["mode"]
                    # valid options are: live, simulation
                    if mode not in ("live", "simulation"):
                        return {
                            "payload": {
                                "response": "mode must be either live or simulation",
                            },
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "mode is required",
                        },
                        "status": 500,
                    }

                try:
                    model_id = resp_dict["model_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "model_id is required",
                        },
                        "status": 500,
                    }

                try:
                    earliest_time = resp_dict["earliest_time"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "earliest_time is required",
                        },
                        "status": 500,
                    }

                try:
                    latest_time = resp_dict["latest_time"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "latest_time is required",
                        },
                        "status": 500,
                    }

        else:
            describe = True

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint renders ML for a given entity, it requires a POST call with the following options:",
                "resource_desc": "Renders ML Outliers for a given entity, this endpoint is used by the backend to render the ML model for a given entity.",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/outliers_render_entity_model\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name",
                        "mode": "(required) rendering mode, valid options: live, simulation",
                        "model_id": "(required) model identifier",
                        "earliest_time": "(required) earliest time",
                        "latest_time": "(required) latest time",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # start time
        start_time = time.time()

        # Outliers rules storage collection
        collection_rules_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_rule = service.kvstore[collection_rules_name]

        # Vtenants storage collection
        vtenants_collection_name = "kv_trackme_virtual_tenants"
        vtenants_collection = service.kvstore[vtenants_collection_name]

        #
        # First, get the full vtenant definition
        #

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        # get
        try:
            vtenant_record = vtenants_collection.data.query(
                query=json.dumps(query_string)
            )
            vtenant_key = vtenant_record[0].get("_key")
        except Exception as e:
            error_msg = (
                f'tenant_id="{tenant_id}" could not be found in the vtenants collection'
            )
            logger.error(error_msg)
            return {
                "payload": {
                    "response": error_msg,
                },
                "status": 500,
            }

        #
        # Get the Outliers rules
        #

        # Define the KV query
        query_string_filter = {
            "object_category": f"splk-{component}",
            "object": object_value,
        }

        query_string = {"$and": [query_string_filter]}

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        key = None

        try:
            records_outliers_rules = collection_rule.data.query(
                query=json.dumps(query_string)
            )
            record_outliers_rules = records_outliers_rules[0]
            key = record_outliers_rules.get("_key")

        except Exception as e:
            key = None

        # if no records
        if not key:
            msg = f'tenant_id="{tenant_id}", component="{component}", object="{object_value}" outliers rules record cannot be found or are not yet available for this entity.'
            logger.error(msg)
            return {
                "payload": {"response": msg},
                "status": 500,
            }

        # log debug
        logger.debug(f'record_outliers_rules="{record_outliers_rules}"')

        # Get the JSON outliers rules object
        entities_outliers = record_outliers_rules.get("entities_outliers")

        # Load as a dict
        try:
            entities_outliers = json.loads(
                record_outliers_rules.get("entities_outliers")
            )
        except Exception as e:
            msg = f'Failed to load entities_outliers with exception="{str(e)}"'

        # log debug
        logger.debug(f'entities_outliers="{entities_outliers}"')

        # Extract as a dict
        entity_outlier_dict = entities_outliers[model_id]

        # log debug
        logger.debug(f'entity_outlier_dict="{entity_outlier_dict}"')

        # Extract the render search
        if mode == "simulation":
            ml_model_render_search = entity_outlier_dict[
                "ml_model_simulation_render_search"
            ]
        elif mode == "live":
            ml_model_render_search = entity_outlier_dict["ml_model_render_search"]
        logger.debug(f'ml_model_simulation_render_search="{ml_model_render_search}"')

        # set kwargs
        kwargs_oneshot = {
            "earliest_time": earliest_time,
            "latest_time": latest_time,
            "search_mode": "normal",
            "preview": False,
            "time_format": "%s",
            "count": 0,
            "output_mode": "json",
        }

        # run search
        search_results = []
        try:
            reader = run_splunk_search(
                service,
                ml_model_render_search,
                kwargs_oneshot,
                24,
                5,
            )

            # loop through the reader results
            for item in reader:
                if isinstance(item, dict):
                    search_results.append(item)

            # return
            logger.info(
                f"function post_outliers_render_entity_model successfully executed in {round(time.time() - start_time, 3)} seconds"
            )
            return {
                "payload": {
                    "search_results": search_results,
                },
                "status": 200,
            }

        except Exception as e:
            return {
                "payload": {
                    "response": f'Failed to run the search with exception="{str(e)}"',
                },
                "status": 500,
            }

    # render ML entity model
    def post_outliers_cim_render_entity_model(self, request_info, **kwargs):
        describe = False

        logger.debug(f"Starting function post_outliers_cim_render_entity_model")

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "tenant_id is required",
                        },
                        "status": 500,
                    }

                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "object is required",
                        },
                        "status": 500,
                    }

                try:
                    entity_name = resp_dict["entity"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "entity is required",
                        },
                        "status": 500,
                    }

                try:
                    cim_field = resp_dict["cim_field"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "cim_field is required",
                        },
                        "status": 500,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "component is required",
                        },
                        "status": 500,
                    }

                try:
                    mode = resp_dict["mode"]
                    # valid options are: live, simulation
                    if mode not in ("live", "simulation"):
                        return {
                            "payload": {
                                "response": "mode must be either live or simulation",
                            },
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "mode is required",
                        },
                        "status": 500,
                    }

                try:
                    model_id = resp_dict["model_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "model_id is required",
                        },
                        "status": 500,
                    }

                try:
                    earliest_time = resp_dict["earliest_time"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "earliest_time is required",
                        },
                        "status": 500,
                    }

                try:
                    latest_time = resp_dict["latest_time"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "latest_time is required",
                        },
                        "status": 500,
                    }

        else:
            describe = True

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint renders ML for a given entity, it requires a POST call with the following options:",
                "resource_desc": "Renders ML Outliers for a given entity, this endpoint is used by the backend to render the ML model for a given entity.",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/outliers_render_entity_model\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name",
                        "entity": "(required) cim entity name",
                        "cim_field": "(required) cim field",
                        "mode": "(required) rendering mode, valid options: live, simulation",
                        "model_id": "(required) model identifier",
                        "earliest_time": "(required) earliest time",
                        "latest_time": "(required) latest time",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # start time
        start_time = time.time()

        # Outliers rules storage collection
        collection_rules_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_rule = service.kvstore[collection_rules_name]

        # Vtenants storage collection
        vtenants_collection_name = "kv_trackme_virtual_tenants"
        vtenants_collection = service.kvstore[vtenants_collection_name]

        #
        # First, get the full vtenant definition
        #

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        # get
        try:
            vtenant_record = vtenants_collection.data.query(
                query=json.dumps(query_string)
            )
            vtenant_key = vtenant_record[0].get("_key")
        except Exception as e:
            error_msg = (
                f'tenant_id="{tenant_id}" could not be found in the vtenants collection'
            )
            logger.error(error_msg)
            return {
                "payload": {
                    "response": error_msg,
                },
                "status": 500,
            }

        #
        # Get the Outliers rules
        #

        # Define the KV query
        query_string_filter = {
            "object_category": f"splk-{component}",
            "object": object_value,
            "entity": entity_name,
            "cim_field": cim_field,
        }

        query_string = {"$and": [query_string_filter]}

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        key = None

        try:
            records_outliers_rules = collection_rule.data.query(
                query=json.dumps(query_string)
            )
            record_outliers_rules = records_outliers_rules[0]
            key = record_outliers_rules.get("_key")

        except Exception as e:
            key = None

        # if no records
        if not key:
            msg = f'tenant_id="{tenant_id}", component="{component}", object="{object_value}" outliers rules record cannot be found or are not yet available for this entity.'
            logger.error(msg)
            return {
                "payload": {"response": msg},
                "status": 500,
            }

        # log debug
        logger.debug(f'record_outliers_rules="{record_outliers_rules}"')

        # Get the JSON outliers rules object
        entities_outliers = record_outliers_rules.get("entities_outliers")

        # Load as a dict
        try:
            entities_outliers = json.loads(
                record_outliers_rules.get("entities_outliers")
            )
        except Exception as e:
            msg = f'Failed to load entities_outliers with exception="{str(e)}"'

        # log debug
        logger.debug(f'entities_outliers="{entities_outliers}"')

        # Extract as a dict
        entity_outlier_dict = entities_outliers[model_id]

        # log debug
        logger.debug(f'entity_outlier_dict="{entity_outlier_dict}"')

        # Extract the render search
        if mode == "simulation":
            ml_model_render_search = entity_outlier_dict[
                "ml_model_simulation_render_search"
            ]
        elif mode == "live":
            ml_model_render_search = entity_outlier_dict["ml_model_render_search"]
        logger.debug(f'ml_model_simulation_render_search="{ml_model_render_search}"')

        # set kwargs
        kwargs_oneshot = {
            "earliest_time": earliest_time,
            "latest_time": latest_time,
            "search_mode": "normal",
            "preview": False,
            "time_format": "%s",
            "count": 0,
            "output_mode": "json",
        }

        # run search
        search_results = []
        try:
            reader = run_splunk_search(
                service,
                ml_model_render_search,
                kwargs_oneshot,
                24,
                5,
            )

            # loop through the reader results
            for item in reader:
                if isinstance(item, dict):
                    search_results.append(item)

            # return
            logger.info(
                f"function post_outliers_cim_render_entity_model successfully executed in {round(time.time() - start_time, 3)} seconds"
            )
            return {
                "payload": {
                    "search_results": search_results,
                },
                "status": 200,
            }

        except Exception as e:
            return {
                "payload": {
                    "response": f'Failed to run the search with exception="{str(e)}"',
                },
                "status": 500,
            }

    # check model
    def post_outliers_check_model(self, request_info, **kwargs):

        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                # get model_id
                try:
                    model_id = resp_dict["model_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "model_id is required",
                        },
                        "status": 500,
                    }

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint checks a specific ML model, it requires a POST call with the following options:",
                "resource_desc": "Check TrackMe Machine Learning model",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/outliers_check_model\" body=\"{'model_id': 'model_178709885414488'}\"",
                "options": [
                    {
                        "model_id": "(required) model identifier",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Define an header for requests authenticated communications with splunkd
        header = {
            "Authorization": "Splunk %s" % request_info.system_authtoken,
            "Content-Type": "application/json",
        }

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # rest_url
        rest_url = f"{request_info.server_rest_uri}/servicesNS/splunk-system-user/trackme/data/lookup-table-files/__mlspl_{model_id}.mlmodel"

        # create an object response_final
        response_final = {}

        # run a get requests, if the model exists, it will return a 200 status code
        try:
            response = requests.get(
                rest_url,
                params={"output_mode": "json", "count": 0},
                headers=header,
                timeout=300,
                verify=False,
            )

            if response.status_code not in (200, 201, 204):
                error_msg = f'Failed to get the model with status_code="{response.status_code}", response.text="{response.text}"'
                response_final["model_exists"] = False
                response_final["rest_url"] = rest_url
                response_final["response"] = error_msg

            else:
                response_final["model_exists"] = True
                response_final["rest_url"] = rest_url
                response_json = response.json()
                # for fields in response_json, add to the response_final
                for key, value in response_json.items():
                    response_final[key] = value

        except Exception as e:
            error_msg = f'Failed to get the model with exception="{str(e)}"'
            response_final["model_exists"] = False
            response_final["response"] = error_msg
            response_final["rest_url"] = rest_url
            logger.error(f"{error_msg}")

        # return the payload
        return {
            "payload": response_final,
            "status": 200,
        }
