#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_tag_policies.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_tag_policies_user",
    "trackme_rest_api_splk_tag_policies_user.log",
)
# Redirect logger.module calls


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkTagPolicies_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkTagPolicies_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_tag_policies(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_tag_policies",
            "resource_group_desc": "Endpoints related to the management of tags (read only operations)",
        }

        return {"payload": response, "status": 200}

    # get all records
    def post_tag_policies_show(self, request_info, **kwargs):
        """
        | trackme mode=post url="/services/trackme/v2/splk_tag_policies/tag_policies_show" body="{'tenant_id': 'mytenant', 'compoennt': 'dsm'}"
        """

        # Declare
        tenant_id = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                if not tenant_id:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "tenant_id is required",
                        },
                        "status": 404,
                    }

                component = resp_dict["component"]
                if not component:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "component is required",
                        },
                        "status": 404,
                    }
                # value must be either: dsm,dhm,mhm,wlk,flx,cim,fqm
                if component not in ("dsm", "dhm", "mhm", "wlk", "flx", "cim", "fqm"):
                    return {
                        "payload": {
                            "response": "The component must be either: dsm,dhm,mhm,wlk,flx,cim,fqm",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint retrieves all records for the tag policies collection, it requires a POST call with the following information:",
                "resource_desc": "Get tags policies",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_tag_policies/tag_policies_show\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "component": "The component identifier",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Data collection
            collection_name = f"kv_trackme_{component}_tags_policies_tenant_{tenant_id}"
            collection = service.kvstore[collection_name]

            return {"payload": collection.data.query(), "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Get model
    def post_tag_policies_by_id(self, request_info, **kwargs):
        """
        | trackme mode=post url=\"/services/trackme/v2/splk_tag_policies/tag_policies_by_id\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'tags_policy_id': 'pan:traffic'}\"
        """

        # By id
        tenant_id = None
        component = None
        tags_policy_id = None

        # query_string to find records
        query_string = None

        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                if not tenant_id:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "tenant_id is required",
                        },
                        "status": 404,
                    }

                component = resp_dict["component"]
                if not component:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "component is required",
                        },
                        "status": 404,
                    }
                # value must be either: dsm,dhm,mhm,wlk,flx,cim,fqm
                if component not in ("dsm", "dhm", "mhm", "wlk", "flx", "cim", "fqm"):
                    return {
                        "payload": {
                            "response": "The component must be either: dsm,dhm,mhm,wlk,flx,cim,fqm",
                            "status": 400,
                        },
                        "status": 400,
                    }

                tags_policy_id = resp_dict["tags_policy_id"]
                if not tags_policy_id:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "tags_policy_id is required",
                        },
                        "status": 404,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint retrieves a tag policy by its id, it requires a GET call with the following data:",
                "resource_desc": "Get a given tag policy",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_tag_policies/tag_policies_by_id\" body=\"{'tenant_id': 'mytenant', 'tags_policy_id': 'pan:traffic'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "component": "The component identifier",
                        "tags_policy_id": "ID of the tags policy",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Define the KV query
        query_string = {
            "tags_policy_id": tags_policy_id,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=300,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            collection_name = f"kv_trackme_{component}_tags_policies_tenant_{tenant_id}"
            collection = service.kvstore[collection_name]

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                return {"payload": collection.data.query_by_id(key), "status": 200}

            else:
                response = {
                    "action": "failure",
                    "response": f'tags_policy_id="{tags_policy_id}" not found',
                }
                logger.error(json.dumps(response))
                return {"payload": response, "status": 404}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}
