#!/usr/bin/env python
# coding=utf-8

__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

import os
import sys
import requests
import time
import json
import re
import threading
import hashlib
import logging
from logging.handlers import RotatingFileHandler
import urllib3

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

splunkhome = os.environ["SPLUNK_HOME"]

# set logging
filehandler = RotatingFileHandler(
    f"{splunkhome}/var/log/splunk/trackme_cimtracker_executor.log",
    mode="a",
    maxBytes=10000000,
    backupCount=1,
)
formatter = logging.Formatter(
    "%(asctime)s %(levelname)s %(filename)s %(funcName)s %(lineno)d %(message)s"
)
logging.Formatter.converter = time.gmtime
filehandler.setFormatter(formatter)
log = logging.getLogger()  # root logger - Good to get it only once.
for hdlr in log.handlers[:]:  # remove the existing file handlers
    if isinstance(hdlr, logging.FileHandler):
        log.removeHandler(hdlr)
log.addHandler(filehandler)  # set the new handler
# set the log level to INFO, DEBUG as the default is ERROR
log.setLevel(logging.INFO)

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

from splunklib.searchcommands import (
    dispatch,
    GeneratingCommand,
    Configuration,
    Option,
    validators,
)

# import trackme libs
from trackme_libs import (
    trackme_reqinfo,
    trackme_idx_for_tenant,
    trackme_gen_state,
    trackme_audit_flip,
    trackme_register_tenant_object_summary,
    trackme_return_tenant_object_summary,
    trackme_register_tenant_component_summary,
    run_splunk_search,
    trackme_handler_events,
)

# import trackme splk-cim libs
from trackme_libs_splk_cim import (
    trackme_cim_splk_outliers_set_rule,
    trackme_cim_splk_outliers_get_status,
    trackme_cim_return_search,
    trackme_cim_fields_gen_metrics,
)

# import trackme libs sla
from trackme_libs_sla import trackme_sla_gen_metrics

# import trackme licensing libs
from trackme_libs_licensing import trackme_check_license


@Configuration(distributed=False)
class SplkCimExecutor(GeneratingCommand):
    tenant_id = Option(
        doc="""
        **Syntax:** **tenant_id=****
        **Description:** The tenant identifier.""",
        require=True,
        default=None,
    )

    component = Option(
        doc="""
        **Syntax:** **component=****
        **Description:** The tracker component name to be executed.""",
        require=True,
        default=None,
    )

    summariesonly = Option(
        doc="""
        **Syntax:** **summariesonly=****
        **Description:** tstats summariesonly mode.""",
        require=False,
        default="True",
        validate=validators.Match(
            "summariesonly", r"^(t|T||true|True|TRUE|f|F|false|False|FALSE)$"
        ),
    )

    account = Option(
        doc="""
        **Syntax:** **account=****
        **Description:** local versus remote search using the given account, local which is the default means the search acts locally.""",
        require=False,
        default="local",
    )

    object = Option(
        doc="""
        **Syntax:** **object=****
        **Description:** The object value.""",
        require=True,
        default=None,
    )

    earliest = Option(
        doc="""
        **Syntax:** **earliest=****
        **Description:** The earliest time quantifier.""",
        require=True,
        default=None,
    )

    latest = Option(
        doc="""
        **Syntax:** **latest=****
        **Description:** The latest time quantifier.""",
        require=True,
        default=None,
    )

    context = Option(
        doc="""
        **Syntax:** **context=****
        **Description:** The context is used for simulation purposes, defaults to live.""",
        require=False,
        default="live",
        validate=validators.Match("context", r"^(live|simulation)$"),
    )

    returnspl = Option(
        doc="""
        **Syntax:** **returnspl=****
        **Description:** Returns the SPL search and exit.""",
        require=False,
        default="false",
        validate=validators.Match(
            "returnspl", r"^(t|T||true|True|TRUE|f|F|false|False|FALSE)$"
        ),
    )

    def register_component_summary_async(
        self, session_key, splunkd_uri, tenant_id, component
    ):
        try:
            summary_register_response = trackme_register_tenant_component_summary(
                session_key,
                splunkd_uri,
                tenant_id,
                component,
            )
            logging.debug(
                f'function="trackme_register_tenant_component_summary", response="{json.dumps(summary_register_response, indent=2)}"'
            )
        except Exception as e:
            logging.error(
                f'failed to register the component summary with exception="{str(e)}"'
            )

    def get_monitoring_wdays(self, monitoring_wdays, monitoring_hours_ranges):
        # get current wday
        current_wday_no = int(time.strftime("%w"))

        # get current hour range
        current_hour_range = int(time.strftime("%H"))

        # get manual wdays
        manual_wdays_pattern = monitoring_wdays.replace("manual:", "").replace(",", "|")
        manual_wdays_regex = re.compile(f"^{manual_wdays_pattern}$")

        # get manual hours ranges
        manual_hours_pattern = monitoring_hours_ranges.replace("manual:", "").replace(
            ",", "|"
        )
        manual_hours_regex = re.compile(f"^({manual_hours_pattern})$")

        # apply wdays rules
        if monitoring_wdays == "auto:all_days" or monitoring_wdays == "manual:all_days":
            isUnderMonitoringDays = True
        elif (
            monitoring_wdays == "auto:monday-to-friday"
            or monitoring_wdays == "manual:monday-to-friday"
        ) and current_wday_no in [
            1,
            2,
            3,
            4,
            5,
        ]:
            isUnderMonitoringDays = True
        elif (
            monitoring_wdays == "auto:monday-to-saturday"
            or monitoring_wdays == "manual:monday-to-saturday"
        ) and current_wday_no in [
            1,
            2,
            3,
            4,
            5,
            6,
        ]:
            isUnderMonitoringDays = True
        elif monitoring_wdays.startswith("manual:") and manual_wdays_regex.match(
            str(current_wday_no)
        ):
            isUnderMonitoringDays = True
        else:
            isUnderMonitoringDays = False

        # apply hours ranges rules
        if (
            monitoring_hours_ranges == "auto:all_ranges"
            or monitoring_hours_ranges == "manual:all_ranges"
        ):
            isUnderMonitoringHours = True
        elif (
            monitoring_hours_ranges == "auto:08h-to-20h"
            or monitoring_hours_ranges == "manual:08h-to-20h"
        ) and current_hour_range in range(8, 19):
            isUnderMonitoringHours = True
        elif monitoring_hours_ranges.startswith("manual:") and manual_hours_regex.match(
            str(current_hour_range)
        ):
            isUnderMonitoringHours = True
        else:
            isUnderMonitoringHours = False

        # generate a field isUnderMonitoringMsg (string) as a summary of the monitoring status
        if isUnderMonitoringDays is True and isUnderMonitoringHours is True:
            isUnderMonitoringMsg = f"This entity is currently under monitoring for days {monitoring_wdays} and hours {monitoring_hours_ranges}"

        elif isUnderMonitoringDays is False and isUnderMonitoringHours is True:
            isUnderMonitoringMsg = f'This entity is not currently under monitoring days conditions, days="{monitoring_wdays}", hours="{monitoring_hours_ranges}"'

        elif isUnderMonitoringDays is True and isUnderMonitoringHours is False:
            isUnderMonitoringMsg = f'This entity is not currently under monitoring hours conditions, days="{monitoring_wdays}", hours="{monitoring_hours_ranges}"'

        else:
            isUnderMonitoringMsg = f'This entity is not currently under monitoring days and hours conditions, days="{monitoring_wdays}", hours="{monitoring_hours_ranges}"'

        # return
        return isUnderMonitoringDays, isUnderMonitoringHours, isUnderMonitoringMsg

    def generate(self, **kwargs):
        if self:
            # performance counter
            start = time.time()

            # Get request info and set logging level
            reqinfo = trackme_reqinfo(
                self._metadata.searchinfo.session_key,
                self._metadata.searchinfo.splunkd_uri,
            )
            log.setLevel(reqinfo["logging_level"])

            # Get configuration and define metadata
            trackme_idx = trackme_idx_for_tenant(
                self._metadata.searchinfo.session_key,
                self._metadata.searchinfo.splunkd_uri,
                self.tenant_id,
            )

            # Get the summary index
            trackme_summary_idx = trackme_idx.get(
                "trackme_summary_idx", "trackme_summary"
            )

            # Get the metric index
            trackme_metrics_idx = trackme_idx.get(
                "trackme_metric_idx", "trackme_metrics"
            )

            # Get the session key
            session_key = self._metadata.searchinfo.session_key

            # build header and target
            header = "Splunk " + str(session_key)

            # check license state
            try:
                check_license = trackme_check_license(
                    reqinfo["server_rest_uri"], session_key
                )
                license_is_valid = check_license.get("license_is_valid")
                logging.debug(
                    f'function check_license called, response="{json.dumps(check_license, indent=2)}"'
                )

            except Exception as e:
                license_is_valid = 0
                logging.error(f'function check_license exception="{str(e)}"')

            # runquery boolean
            runquery = False

            if self.component in ("splk-cim"):
                runquery = True
            else:
                logging.error(
                    f'tenant_id="{self.tenant_id}", The component="{self.component}" is not a valid TrackMe component.'
                )
                raise Exception(
                    f'tenant_id="{self.tenant_id}", The component="{self.component}" is not a valid TrackMe component.'
                )

            if runquery:
                # first, test that we can find this object in the KVstore collection
                if self.context == "live":
                    collection_name = "kv_trackme_cim_tenant_" + str(self.tenant_id)
                elif self.context == "simulation":
                    collection_name = "kv_trackme_cim_simulation_tenant_" + str(
                        self.tenant_id
                    )
                collection = self.service.kvstore[collection_name]

                # Define the KV query
                query_string = {
                    "object": self.object,
                }

                # Get the current record
                # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

                key = None

                try:
                    kvrecord = collection.data.query(query=json.dumps(query_string))[0]
                    key = kvrecord.get("_key")

                except Exception as e:
                    logging.error(
                        f'tenant_id="{self.tenant_id}", object="{self.object}" record could not be found in the KStore collection, cannot proceed.'
                    )
                    raise Exception(
                        f'tenant_id="{self.tenant_id}", object="{self.object}" record could not be found in the KStore collection, cannot proceed.'
                    )

                # ensure the kvrecord has the field object_category with value splk-cim, otherwise add
                if not kvrecord.get("object_category"):
                    kvrecord["object_category"] = "splk-cim"

                # get the tracker name
                tracker_name = kvrecord.get("tracker_name")

                # init priority
                priority = "medium"

                # check restricted components
                if license_is_valid != 1:
                    logging.error(
                        "The requested component is restricted to the Full and Trial edition mode, its execution cannot be accepted"
                    )

                    # Call the component register
                    trackme_register_tenant_object_summary(
                        session_key,
                        self._metadata.searchinfo.splunkd_uri,
                        self.tenant_id,
                        "splk-cim",
                        tracker_name,
                        "failure",
                        time.time(),
                        round(time.time() - start, 3),
                        "The requested component is restricted to the Full and Trial edition mode, its execution cannot be accepted",
                        self.earliest,
                        self.latest,
                    )

                    raise Exception(
                        "The requested component is restricted to the Full and Trial edition mode, its execution cannot be accepted"
                    )

                # sample_json
                """
                {
                    "cim_datamodel_name": "Authentication",
                    "cim_datamodel_nodename": "Authentication",
                    "cim_drop_dm_object_name": "Authentication",
                    "cim_fields": "app,action,src,src_user",
                    "src_user": [{
                        "datamodel_fieldname": "Authentication.src_user"
                    }, {
                        "min_compliant_coverage_percentage": 95
                    }, {
                        "max_unknown_coverage_percentage": 5
                    }, {
                        "regex_validator_rule": ".*"
                    }],
                    "app": [{
                        "datamodel_fieldname": "Authentication.app"
                    }, {
                        "min_compliant_coverage_percentage": 95
                    }, {
                        "max_unknown_coverage_percentage": 5
                    }, {
                        "regex_validator_rule": ".*"
                    }],
                    "action": [{
                        "datamodel_fieldname": "Authentication.action"
                    }, {
                        "min_compliant_coverage_percentage": 95
                    }, {
                        "max_unknown_coverage_percentage": 5
                    }, {
                        "regex_validator_rule": ".*"
                    }],
                    "src": [{
                        "datamodel_fieldname": "Authentication.src"
                    }, {
                        "min_compliant_coverage_percentage": 95
                    }, {
                        "max_unknown_coverage_percentage": 5
                    }, {
                        "regex_validator_rule": ".*"
                    }]	
                }
                """

                target_url = f"{self._metadata.searchinfo.splunkd_uri}/services/trackme/v2/splk_cim/cim_tracker_rules"

                json_data = {
                    "tenant_id": self.tenant_id,
                    "object": self.object,
                    "context": self.context,
                }

                # retrieve the object tracking rules stored in the KVstore, via a rest call to the custom endpoint
                try:
                    response = requests.post(
                        target_url,
                        headers={
                            "Authorization": header,
                            "Content-Type": "application/json",
                        },
                        verify=False,
                        data=json.dumps(json_data, indent=1),
                        timeout=300,
                    )
                    cim_tracking_rules = json.loads(response.text)

                    # log debug
                    logging.debug(
                        f'tenant_id="{self.tenant_id}", object="{self.object}", cim_tracking_rules="{json.dumps(cim_tracking_rules, indent=2)}"'
                    )

                except Exception as e:
                    logging.error(
                        f'tenant_id="{self.tenant_id}", failed to retrieve CIM tracking rules for the object="{self.object}" with exception="{str(e)}"'
                    )
                    raise Exception(
                        f'failed to load the cim_tracking_rules with exception="{str(e)}"'
                    )

                # get the list of CIM fields to be tracked in the object
                cim_fields = cim_tracking_rules["cim_fields"]

                # log debug
                logging.debug(
                    f'tenant_id="{self.tenant_id}", object="{self.object}", list of CIM fields to be tracked for this entity, cim_fields="{cim_fields}"'
                )

                # store as a list
                cim_fields_list = cim_fields.split(",")

                # get the datamodel name
                cim_datamodel_name = cim_tracking_rules["cim_datamodel_name"]

                # log debug
                logging.debug(
                    f'tenant_id="{self.tenant_id}", object="{self.object}", cim_datamodel_name="{cim_datamodel_name}"'
                )

                # get the datamodel name
                cim_datamodel_nodename = cim_tracking_rules["cim_datamodel_nodename"]

                # log debug
                logging.debug(
                    f'tenant_id="{self.tenant_id}", object="{self.object}", cim_datamodel_nodename="{cim_datamodel_nodename}"'
                )

                # get the datamodel search root constraint (which is optional)
                cim_root_constraint = cim_tracking_rules["cim_root_constraint"]

                # log debug
                logging.debug(
                    f'tenant_id="{self.tenant_id}", object="{self.object}", cim_root_constraint="{cim_root_constraint}"'
                )

                # get the drop data name (csv list of object names to be dropped)
                cim_drop_dm_object_name = cim_tracking_rules["cim_drop_dm_object_name"]

                # get the cim breakby
                cim_breakby = cim_tracking_rules["cim_breakby"]

                # log debug
                logging.debug(
                    f'tenant_id="{self.tenant_id}", object="{self.object}", cim_drop_dm_object_name="{cim_drop_dm_object_name}"'
                )

                # get the search string
                try:
                    search = trackme_cim_return_search(
                        tracker_name,
                        self.tenant_id,
                        self.object,
                        self.account,
                        json.dumps(cim_tracking_rules),
                        cim_datamodel_name,
                        cim_datamodel_nodename,
                        cim_drop_dm_object_name,
                        cim_root_constraint,
                        cim_fields,
                        self.earliest,
                        self.latest,
                        self.summariesonly,
                    )
                except Exception as e:
                    logging.error(
                        f'Failed to retrieve the search string from function trackme_cim_return_search with exception="{str(e)}"'
                    )
                    raise Exception(
                        f'Failed to retrieve the search string from function trackme_cim_return_search with exception="{str(e)}"'
                    )

                # log debug
                logging.debug(
                    f'tenant_id="{self.tenant_id}", object="{self.object}", tracker_name="{tracker_name}", search="{search}"'
                )

                # Set the search
                kwargs_oneshot = {
                    "earliest_time": self.earliest,
                    "latest_time": self.latest,
                    "output_mode": "json",
                    "count": 0,
                }

                # if returnspl, let's return the Spl (!) and exit
                if self.returnspl in ("True", "true", "t", "TRUE"):
                    # yield
                    yield_record = {
                        "search": search,
                    }
                    yield {"_time": time.time(), "_raw": yield_record}

                else:
                    # global total count (used for further decisions)
                    global_total_count = 0

                    # local total count (used for local decision regarding the search execution)
                    local_total_count = 0

                    # run the search now
                    try:
                        reader = run_splunk_search(
                            self.service,
                            search,
                            kwargs_oneshot,
                            24,
                            5,
                        )

                        # loop and create our main dict
                        cim_results_maindict = {}

                        for item in reader:
                            if isinstance(item, dict):
                                local_total_count += 1
                                # add the record into a dict
                                # if no break by:
                                if cim_breakby == "none":
                                    cim_results_maindict["global"] = item
                                else:
                                    cim_results_maindict[item.get(cim_breakby)] = item

                        logging.debug(
                            f'tenant_id="{self.tenant_id}", object="{self.object}", cim_results_maindict="{json.dumps(cim_results_maindict, indent=2)}"'
                        )

                    except Exception as e:
                        # Call the component register
                        trackme_register_tenant_object_summary(
                            session_key,
                            self._metadata.searchinfo.splunkd_uri,
                            self.tenant_id,
                            "splk-cim",
                            tracker_name,
                            "failure",
                            time.time(),
                            round(time.time() - start, 3),
                            str(e),
                            self.earliest,
                            self.latest,
                        )
                        msg = f'tenant_id="{self.tenant_id}", object="{self.object}", status="failure", main search failed with exception="{str(e)}"'
                        logging.error(msg)
                        raise Exception(msg)

                    # Investigate the registered component summary
                    # in some cases such as a remote search failure, we would not be able to detect it at the search exec time
                    if self.context != "simulation":
                        # set bool
                        detected_failure_register = False

                        try:
                            last_status_exec = trackme_return_tenant_object_summary(
                                session_key,
                                self._metadata.searchinfo.splunkd_uri,
                                self.tenant_id,
                                "splk-cim",
                                tracker_name,
                            )
                            logging.debug(
                                f'function="trackme_return_tenant_object_summary", tenant_id="{self.tenant_id}", component="splk-cim", report="{self.object}", result="{last_status_exec}"'
                            )

                            if last_status_exec:
                                if isinstance(last_status_exec, dict):
                                    if (
                                        last_status_exec.get("last_status") == "failure"
                                        and local_total_count == 0
                                    ):
                                        logging.error(
                                            f'tenant_id="{self.tenant_id}", object="{self.object}", detected search execution failure from register component, result="{json.dumps(last_status_exec, indent=2)}"'
                                        )
                                        detected_failure_register = True

                        except Exception as e:
                            logging.error(
                                f'could not verify the register object summary, exception="{str(e)}"'
                            )

                        if detected_failure_register:
                            raise Exception(
                                "TrackMe has detected a failure in the search execution, consult the component register status or the logs for more information"
                            )

                    #
                    # Continue exec
                    #

                    try:
                        # build the list of entities
                        entities_list = []

                        # if no break by:
                        if cim_breakby == "none":
                            entities_list.append("global")
                        # otherwise:
                        else:
                            for entity in cim_results_maindict:
                                entities_list.append(entity)

                        #
                        # ML models set rules: per entity / cim_field
                        #
                        for entity in entities_list:
                            # loop through the list of CIM fields, check and create ML models if necessary
                            for field in cim_fields_list:
                                # check outliers rules, and create if needed
                                try:
                                    trackme_cim_splk_outliers_set_rule(
                                        session_key,
                                        reqinfo["server_rest_port"],
                                        reqinfo,
                                        self.tenant_id,
                                        self.object,
                                        cim_breakby,
                                        entity,
                                        field,
                                    )
                                except Exception as e:
                                    logging.error(
                                        f'tenant_id="{self.tenant_id}", object="{self.object}", cim_field="{field}", Failed to call trackme_cim_splk_outliers_set_rule with exception="{str(e)}"'
                                    )

                        # cim field status summary dict
                        cim_fields_status_dict = {}
                        cim_fields_results_dict = {}

                        # empty summary status dict
                        cim_fields_red = {}
                        cim_fields_green = {}

                        #
                        # loop through entities
                        #

                        for entity in entities_list:
                            # get status
                            cim_fields_status_entity_dict = {}
                            cim_fields_results_entity_dict = {}

                            # empty summary status lists
                            cim_fields_entity_red = []
                            cim_fields_entity_green = []

                            #
                            # loop through cim fields in this entity
                            #

                            for field in cim_fields_list:
                                # Store the cim field status in its own dict
                                status_fieldname = field + "_status"
                                status = cim_results_maindict[entity].get(
                                    status_fieldname
                                )
                                dict_status = {
                                    "status": status,
                                }

                                # add to the list depending on the status
                                if status == "green":
                                    cim_fields_entity_green.append(field)
                                else:
                                    cim_fields_entity_red.append(field)

                                logging.debug(
                                    f'tenant_id="{self.tenant_id}", object="{self.object}", cim_fields_entity_green="{cim_fields_entity_green}", cim_fields_entity_red="{cim_fields_entity_red}"'
                                )

                                # append to the dict
                                cim_fields_status_entity_dict[field] = dict_status

                                # get the details
                                # Store the cim field results in its own dict
                                status_fieldname = field + "_status"
                                status = cim_results_maindict[entity].get(
                                    status_fieldname
                                )
                                total_count_fieldname = field + "_total"
                                total_count = cim_results_maindict[entity].get(
                                    total_count_fieldname
                                )
                                count_unknown_fieldname = field + "_count_unknown"
                                count_unknown = cim_results_maindict[entity].get(
                                    count_unknown_fieldname
                                )
                                count_not_unknown_fieldname = (
                                    field + "_count_not_unknown"
                                )
                                count_not_unknown = cim_results_maindict[entity].get(
                                    count_not_unknown_fieldname
                                )
                                count_compliant_fieldname = field + "_count_compliant"
                                count_compliant = cim_results_maindict[entity].get(
                                    count_compliant_fieldname
                                )
                                pct_coverage_unknown_fieldname = (
                                    field + "_pct_coverage_unknown"
                                )
                                pct_coverage_unknown = cim_results_maindict[entity].get(
                                    pct_coverage_unknown_fieldname
                                )
                                pct_coverage_compliant_fieldname = (
                                    field + "_pct_coverage_compliant"
                                )
                                pct_coverage_compliant = cim_results_maindict[
                                    entity
                                ].get(pct_coverage_compliant_fieldname)

                                # Add to the global_total_count
                                global_total_count += int(total_count)

                                # Create the dict summary
                                dict_summary = {
                                    "status": status,
                                    "total_count": total_count,
                                    "count_unknown": count_unknown,
                                    "count_not_unknown": count_not_unknown,
                                    "count_compliant": count_compliant,
                                    "pct_coverage_unknown": pct_coverage_unknown,
                                    "pct_coverage_compliant": pct_coverage_compliant,
                                }

                                # Create the metric summary
                                dict_metrics = dict_summary.copy()
                                # if no break by, add a pseudo grouping by
                                if cim_breakby == "none":
                                    dict_metrics["breakby_field"] = "cim_entity_zone"
                                # otherwise:
                                else:
                                    dict_metrics["breakby_field"] = cim_breakby
                                dict_metrics["breakby_entity"] = entity

                                # log debug
                                logging.debug(
                                    f'tenant_id="{self.tenant_id}", object="{self.object}", field="{field}", cim_fields_results_dict="{dict_summary}"'
                                )

                                # append to the dict
                                cim_fields_results_entity_dict[field] = dict_summary

                                # generate summary events for the object / field
                                if self.context == "live":
                                    try:
                                        trackme_cim_fields_gen_metrics(
                                            session_key,
                                            reqinfo["server_rest_port"],
                                            reqinfo["server_rest_uri"],
                                            self.tenant_id,
                                            self.object,
                                            "splk-cim",
                                            field,
                                            json.dumps(dict_metrics),
                                        )
                                    except Exception as e:
                                        logging.error(
                                            f'tenant_id="{self.tenant_id}", object="{self.object}", failed to stream summary events with exception="{str(e)}"'
                                        )

                            # log status
                            logging.debug(
                                f'tenant_id="{self.tenant_id}", object="{self.object}", cim_fields_entity_green="{cim_fields_entity_green}", cim_fields_entity_red="{cim_fields_entity_red}"'
                            )

                            # add to the final dict
                            cim_fields_status_dict[entity] = (
                                cim_fields_status_entity_dict
                            )
                            cim_fields_results_dict[entity] = (
                                cim_fields_results_entity_dict
                            )
                            cim_fields_red[entity] = cim_fields_entity_red
                            cim_fields_green[entity] = cim_fields_entity_green

                        # log debug
                        logging.debug(
                            f'tenant_id="{self.tenant_id}", object="{self.object}", cim_fields_status_dict="{json.dumps(cim_fields_status_dict, indent=2)}", cim_fields_results_dict="{json.dumps(cim_fields_results_dict, indent=2)}", cim_fields_red="{json.dumps(cim_fields_red, indent=2)}", cim_fields_green="{json.dumps(cim_fields_green, indent=2)}"'
                        )

                        #
                        # Review results and proceed
                        #

                        # define the object_state
                        cim_fields_red_count = 0
                        cim_fields_green_count = 0
                        for entity in cim_fields_red:
                            cim_fields_red_count += len(cim_fields_red[entity])
                        for entity in cim_fields_green:
                            cim_fields_green_count += len(cim_fields_green[entity])

                        # some cim fields came back red
                        if cim_fields_red_count > 0:
                            object_state = "red"
                        # we have no results
                        elif global_total_count == 0:
                            object_state = "red"
                        else:
                            object_state = "green"

                        # build the final response
                        timevent = time.time()
                        runtime = round(float(time.time() - start), 3)
                        anomaly_reason = None
                        # some cim fields came back red
                        if cim_fields_red_count > 0:
                            anomaly_reason = (
                                "compliance_failed: CIM fields compliance not passed."
                            )
                        # we have no results
                        elif global_total_count == 0:
                            anomaly_reason = (
                                "no_results: search did not produce any results."
                            )
                            # set all fields red
                            cim_fields_red["unknown"] = cim_fields_list
                            cim_fields_red_count = len(cim_fields_list)
                        else:
                            anomaly_reason = (
                                "none: CIM fields compliance passed successfully."
                            )
                        cim_tracking_results = {
                            "_time": timevent,
                            "time_human": time.strftime("%c", time.localtime(timevent)),
                            "runtime": runtime,
                            "object": self.object,
                            "object_state": str(object_state),
                            "anomaly_reason": str(anomaly_reason),
                            "cim_fields_red_count": str(cim_fields_red_count),
                            "cim_fields_green_count": str(cim_fields_green_count),
                            "cim_fields_red": cim_fields_red,
                            "cim_fields_green": cim_fields_green,
                            "cim_entities": ",".join(entities_list),
                            "cim_tracking_rules": cim_tracking_rules,
                            "cim_tracking_results": cim_fields_results_dict,
                        }

                        # results to be stored in the KVstore collection
                        cim_tracking_results_summary = {
                            "anomaly_reason": str(anomaly_reason),
                            "cim_fields_red_count": str(cim_fields_red_count),
                            "cim_fields_green_count": str(cim_fields_green_count),
                            "cim_fields_red": cim_fields_red,
                            "cim_fields_green": cim_fields_green,
                            "cim_entities": ",".join(entities_list),
                            "cim_tracking_results": cim_fields_results_dict,
                        }

                        logging.debug(
                            f'tenant_id="{self.tenant_id}", object="{self.object}", cim_tracking_results="{json.dumps(cim_tracking_results, indent=2)}", cim_tracking_results_summary="{json.dumps(cim_tracking_results_summary, indent=2)}"'
                        )

                        #
                        # flipping status & sla metrics
                        #

                        latest_flip_state = kvrecord.get("latest_flip_state")
                        latest_flip_time = kvrecord.get("latest_flip_time")

                        if not latest_flip_state or not latest_flip_time:
                            latest_flip_state = str(object_state)
                            latest_flip_time = time.time()

                            # gen a flip event
                            flip_timestamp = time.strftime(
                                "%d/%m/%Y %H:%M:%S", time.localtime(time.time())
                            )
                            flip_previous_state = "discovered"
                            flip_result = f"{flip_timestamp}, object={self.object} has flipped from previous_state={flip_previous_state} to state={latest_flip_state} with anomaly_reason={anomaly_reason}"

                            flip_record = {
                                "timeStr": flip_timestamp,
                                "tenant_id": self.tenant_id,
                                "keyid": key,
                                "alias": kvrecord.get("alias", self.object),
                                "object": self.object,
                                "object_category": "splk-cim",
                                "object_state": latest_flip_state,
                                "object_previous_state": flip_previous_state,
                                "priority": priority,
                                "latest_flip_time": latest_flip_time,
                                "latest_flip_state": latest_flip_state,
                                "anomaly_reason": str(anomaly_reason),
                                "result": flip_result,
                            }

                            # add event_id
                            flip_record["event_id"] = hashlib.sha256(
                                json.dumps(flip_record).encode()
                            ).hexdigest()

                            if self.context != "simulation":
                                try:
                                    trackme_gen_state(
                                        index=trackme_summary_idx,
                                        sourcetype="trackme:flip",
                                        source="flip_state_change_tracking",
                                        event=flip_record,
                                    )
                                    logging.info(
                                        f'TrackMe flipping event created successfully, tenant_id="{self.tenant_id}", record="{json.dumps(flip_record, indent=1)}"'
                                    )
                                except Exception as e:
                                    logging.error(
                                        f'tenant_id="{self.tenant_id}", object="{self.object}", record="{json.dumps(flip_record, indent=1)}", failed to generate a flipping state event with exception="{e}"'
                                    )

                        elif str(latest_flip_state) != str(object_state):
                            latest_flip_state = str(object_state)
                            latest_flip_time = time.time()

                            # gen a flip event
                            flip_previous_state = str(latest_flip_state)
                            flip_result = (
                                time.strftime(
                                    "%d/%m/%Y %H:%M:%S",
                                    time.localtime(latest_flip_time),
                                )
                                + ", object="
                                + str(self.object)
                                + " has flipped from previous_state="
                                + str(kvrecord.get("latest_flip_state"))
                                + " to state="
                                + str(latest_flip_state)
                            )

                            if self.context != "simulation":
                                # flipping
                                try:
                                    trackme_audit_flip(
                                        session_key,
                                        reqinfo["server_rest_uri"],
                                        str(self.tenant_id),
                                        key,
                                        kvrecord.get("alias", self.object),
                                        str(self.object),
                                        "splk-cim",
                                        priority,
                                        "red",
                                        str(kvrecord.get("latest_flip_state")),
                                        str(latest_flip_time),
                                        str(latest_flip_state),
                                        "Machine Learning outliers anomalies were reported",
                                        str(flip_result),
                                    )
                                except Exception as e:
                                    logging.error(
                                        f'tenant_id="{self.tenant_id}", object="{self.object}", failed to generate a flipping state event with exception="{str(e)}"'
                                    )

                        else:
                            latest_flip_state = latest_flip_state
                            latest_flip_time = latest_flip_time

                        #

                        # priority
                        priority = kvrecord.get("priority")
                        if not priority:
                            priority = "medium"

                        try:
                            # Update the record
                            kvrecord["object_state"] = object_state
                            kvrecord["cim_tracking_results"] = json.dumps(
                                cim_tracking_results_summary, indent=2
                            )
                            kvrecord["tracker_runtime"] = time.time()
                            kvrecord["tracker_last_duration"] = time.time() - start
                            kvrecord["latest_flip_state"] = latest_flip_state
                            kvrecord["latest_flip_time"] = latest_flip_time
                            kvrecord["priority"] = priority
                            kvrecord["anomaly_reason"] = (
                                cim_tracking_results_summary.get("anomaly_reason")
                            )
                            collection.data.update(str(key), json.dumps(kvrecord))

                        except Exception as e:
                            logging.error(
                                f'tenant_id="{self.tenant_id}", object="{self.object}", failure to update the KVstore record collection with exception="{e}"'
                            )
                            # Call the component register
                            trackme_register_tenant_object_summary(
                                session_key,
                                self._metadata.searchinfo.splunkd_uri,
                                self.tenant_id,
                                "splk-cim",
                                tracker_name,
                                "failure",
                                time.time(),
                                round(time.time() - start, 3),
                                str(e),
                                self.earliest,
                                self.latest,
                            )
                            raise Exception(
                                f'tenant_id="{self.tenant_id}", object="{self.object}", failure to update the KVstore record collection with exception="{e}"'
                            )

                        # gen the record
                        gen_state_record = cim_tracking_results
                        gen_state_record["key"] = kvrecord["_key"]
                        gen_state_record["object_category"] = "splk-cim"
                        gen_state_record["tenant_id"] = kvrecord["tenant_id"]

                        try:
                            trackme_gen_state(
                                index=trackme_summary_idx,
                                sourcetype="trackme:state",
                                source=f"current_state_tracking:splk-cim:{self.tenant_id}",
                                event=gen_state_record,
                            )
                            logging.debug(
                                f'TrackMe summary event created successfully, record="{json.dumps(gen_state_record, indent=1)}"'
                            )
                        except Exception as e:
                            logging.error(
                                f'TrackMe summary event creation failure, record="{json.dumps(gen_state_record, indent=1)}", exception="{str(e)}"'
                            )
                            # Call the component register
                            trackme_register_tenant_object_summary(
                                session_key,
                                self._metadata.searchinfo.splunkd_uri,
                                self.tenant_id,
                                "splk-cim",
                                tracker_name,
                                "failure",
                                time.time(),
                                round(time.time() - start, 3),
                                str(e),
                                self.earliest,
                                self.latest,
                            )
                            raise Exception(
                                f'tenant_id="{self.tenant_id}", object="{self.object}", failed to generate a trackme state event with exception="{str(e)}"'
                            )

                        #
                        # END
                        #

                        # log the perf metrics for live only
                        if self.context == "live":
                            # if log level is INFO
                            if reqinfo["logging_level"] == "DEBUG":
                                logging.info(
                                    f'tenant_id="{self.tenant_id}", object="{self.object}", status="success", run_time="{round(time.time() - start, 3)}", report="{tracker_name}", results="{json.dumps(cim_tracking_results, indent=2)}"'
                                )
                            else:
                                logging.info(
                                    f'tenant_id="{self.tenant_id}", object="{self.object}", status="success", enable debug log for more information, run_time="{round(time.time() - start, 3)}", report="{tracker_name}"'
                                )

                        # handle the output depending on the context
                        if self.context == "simulation":
                            yield_record = {}
                            yield_record["_time"] = timevent
                            yield_record["_raw"] = cim_tracking_results
                        else:
                            yield_record = {}
                            yield_record["_time"] = timevent
                            yield_record["time_human"] = time.strftime(
                                "%c", time.localtime(timevent)
                            )
                            yield_record["report"] = str(tracker_name)
                            yield_record["runtime"] = runtime
                            yield_record["result"] = (
                                f'tracker report="{tracker_name}" successfully executed in {runtime} seconds'
                            )
                            yield_record["_raw"] = {
                                "_time": timevent,
                                "time_human": time.strftime(
                                    "%c", time.localtime(timevent)
                                ),
                                "report": str(tracker_name),
                                "runtime": runtime,
                                "result": f'tracker report="{tracker_name}" successfully executed in {runtime} seconds',
                                "search": search,
                            }

                        # if context is live, register the component
                        if self.context == "live":
                            # Call the component register
                            trackme_register_tenant_object_summary(
                                session_key,
                                self._metadata.searchinfo.splunkd_uri,
                                self.tenant_id,
                                "splk-cim",
                                tracker_name,
                                "success",
                                time.time(),
                                round(time.time() - start, 3),
                                "The report was executed successfully",
                                self.earliest,
                                self.latest,
                            )

                            # notification event
                            try:
                                trackme_handler_events(
                                    session_key=self._metadata.searchinfo.session_key,
                                    splunkd_uri=self._metadata.searchinfo.splunkd_uri,
                                    tenant_id=self.tenant_id,
                                    sourcetype="trackme:handler",
                                    source=f"trackme:handler:{self.tenant_id}",
                                    handler_events=[
                                        {
                                            "object": self.object,
                                            "object_id": key,
                                            "object_category": "splk-cim",
                                            "handler": tracker_name,
                                            "handler_message": "Entity was inspected by an hybrid tracker.",
                                            "handler_troubleshoot_search": f"index=_internal sourcetype=trackme:custom_commands:trackmetrackerexecutor tenant_id={self.tenant_id} report={tracker_name}",
                                            "handler_time": time.time(),
                                        }
                                    ],
                                )
                            except Exception as e:
                                logging.error(
                                    f'tenant_id="{self.tenant_id}", component="splk-{self.component}", could not send notification event, exception="{e}"'
                                )

                        # Use threading to do an async call to the register summary without waiting for it to complete
                        thread = threading.Thread(
                            target=self.register_component_summary_async,
                            args=(
                                session_key,
                                self._metadata.searchinfo.splunkd_uri,
                                self.tenant_id,
                                self.component,
                            ),
                        )
                        thread.start()

                        # finally render the output to Splunk
                        yield yield_record

                        ############
                        # No results
                        ############

                        # No results: the search was executed, but does not return any results
                        if not global_total_count > 0 or not cim_tracking_results:
                            # get current flip info
                            record_latest_flip_state = kvrecord.get("latest_flip_state")
                            record_latest_flip_time = kvrecord.get("latest_flip_time")

                            # if the flipping event was not registered yet, generate a new flip event

                            if record_latest_flip_state != "red":
                                new_latest_flip_state = "red"
                                new_latest_flip_time = time.time()
                                new_flip_time_human = time.strftime(
                                    "%d/%m/%Y %H:%M:%S",
                                    time.localtime(new_latest_flip_time),
                                )

                                # gen a flip event
                                flip_previous_state = str(record_latest_flip_state)
                                flip_result = f'{new_flip_time_human}, object="{self.object}" has flipped from previous_state="{record_latest_flip_state}" to state="{new_latest_flip_state}" with anomaly_reason="No results were returned"'

                                if self.context != "simulation":
                                    try:
                                        trackme_audit_flip(
                                            session_key,
                                            reqinfo["server_rest_uri"],
                                            str(self.tenant_id),
                                            key,
                                            kvrecord.get("alias", self.object),
                                            str(self.object),
                                            "splk-cim",
                                            priority,
                                            "red",
                                            str(kvrecord.get("latest_flip_state")),
                                            str(new_latest_flip_time),
                                            str(new_latest_flip_state),
                                            "Machine Learning outliers anomalies were reported",
                                            str(flip_result),
                                        )
                                    except Exception as e:
                                        logging.error(
                                            f'tenant_id="{self.tenant_id}", object="{self.object}", failed to generate a flipping state event with exception="{str(e)}"'
                                        )

                                # replace variables
                                record_latest_flip_state = new_latest_flip_state
                                record_latest_flip_time = new_latest_flip_time

                            try:
                                # Update the record
                                kvrecord["object_state"] = "red"
                                kvrecord["cim_tracking_results"] = json.dumps(
                                    {
                                        "anomaly_reason": "no_results: search did not produce any results."
                                    },
                                    indent=1,
                                )
                                kvrecord["tracker_runtime"] = time.time()
                                kvrecord["tracker_last_duration"] = time.time() - start
                                kvrecord["latest_flip_state"] = latest_flip_state
                                kvrecord["latest_flip_time"] = record_latest_flip_time
                                kvrecord["anomaly_reason"] = (
                                    "no_results: search did not produce any results."
                                )
                                collection.data.update(str(key), json.dumps(kvrecord))

                                # Call the component register
                                trackme_register_tenant_object_summary(
                                    session_key,
                                    self._metadata.searchinfo.splunkd_uri,
                                    self.tenant_id,
                                    "splk-cim",
                                    tracker_name,
                                    "success",
                                    time.time(),
                                    round(time.time() - start, 3),
                                    "The report was executed successfully",
                                    self.earliest,
                                    self.latest,
                                )

                                # call trackme_register_tenant_component_summary
                                thread = threading.Thread(
                                    target=self.register_component_summary_async,
                                    args=(
                                        session_key,
                                        self._metadata.searchinfo.splunkd_uri,
                                        self.tenant_id,
                                        "cim",
                                    ),
                                )
                                thread.start()

                            except Exception as e:
                                logging.error(
                                    f'tenant_id="{self.tenant_id}", object="{self.object}", failure to update the KVstore record collection with exception="{e}"'
                                )
                                # Call the component register
                                trackme_register_tenant_object_summary(
                                    session_key,
                                    self._metadata.searchinfo.splunkd_uri,
                                    self.tenant_id,
                                    "splk-cim",
                                    tracker_name,
                                    "failure",
                                    time.time(),
                                    round(time.time() - start, 3),
                                    str(e),
                                    self.earliest,
                                    self.latest,
                                )
                                raise Exception(
                                    f'tenant_id="{self.tenant_id}", object="{self.object}", failure to update the KVstore record collection with exception="{e}"'
                                )

                        #
                        # Monitoring wdays & hours
                        #

                        # get monitoring wdays and hours ranges

                        # wdays
                        monitoring_wdays = kvrecord.get(
                            "monitoring_wdays", "auto:all_days"
                        )
                        kvrecord["monitoring_wdays"] = monitoring_wdays

                        # hour ranges
                        monitoring_hours_ranges = kvrecord.get(
                            "monitoring_hours_ranges", "auto:all_ranges"
                        )
                        kvrecord["monitoring_hours_ranges"] = monitoring_hours_ranges

                        # call get_monitoring_wdays and define isUnderMonitoringDays, isUnderMonitoringHours, isUnderMonitoringMsg
                        (
                            isUnderMonitoringDays,
                            isUnderMonitoringHours,
                            isUnderMonitoringMsg,
                        ) = self.get_monitoring_wdays(
                            monitoring_wdays, monitoring_hours_ranges
                        )
                        logging.debug(
                            f'tenant_id="{self.tenant_id}", object_value="{self.object}", monitoring_wdays="{monitoring_wdays}", monitoring_hours_ranges="{monitoring_hours_ranges}", isUnderMonitoringDays="{isUnderMonitoringDays}", isUnderMonitoringHours="{isUnderMonitoringHours}", isUnderMonitoringMsg="{isUnderMonitoringMsg}"'
                        )

                        if object_state != "green":
                            if (
                                isUnderMonitoringDays is False
                                or isUnderMonitoringHours is False
                            ):
                                object_state = "orange"
                                kvrecord["object_state"] = object_state

                        # update kvrecord
                        collection.data.update(str(key), json.dumps(kvrecord))

                        ###########################
                        # Machine Learning Outliers
                        ###########################

                        # Verify ML outliers anomaly detection status
                        outlierResults = None
                        try:
                            outlierResults = trackme_cim_splk_outliers_get_status(
                                session_key,
                                reqinfo["server_rest_port"],
                                self.tenant_id,
                                self.object,
                            )
                            logging.debug(
                                f'tenant_id="{self.tenant_id}", object="{self.object}", outlierResults="{json.dumps(outlierResults, indent=4)}"'
                            )

                        except Exception as e:
                            logging.error(
                                f'tenant_id="{self.tenant_id}", object="{self.object}", failed to retrieve outlier anomaly statuses with exception="{str(e)}"e'
                            )

                        # process
                        if (
                            outlierResults
                            and isUnderMonitoringDays
                            and isUnderMonitoringHours
                        ):
                            # get fields
                            isOutlier = int(outlierResults.get("isOutlier"))
                            anomaly_reason = outlierResults.get("anomaly_reason")
                            outlierStatus = outlierResults.get("outlierStatus")
                            isOutlierDesc = outlierResults.get("isOutlierDesc")
                            cim_fields_green_outliers = outlierResults.get(
                                "cim_field_green", {}
                            )
                            cim_fields_red_outliers = outlierResults.get(
                                "cim_field_red", {}
                            )
                            models_processed = outlierResults.get("models_processed")
                            models_in_anomaly = outlierResults.get("models_in_anomaly")
                            anomaly_records = outlierResults.get("anomaly_records")

                            # refresh the record
                            kvrecord = collection.data.query(
                                query=json.dumps(query_string)
                            )[0]

                            # set the object_state
                            if isOutlier == 1:
                                object_state = "red"
                            else:
                                object_state = kvrecord.get("object_state")

                            # run if outliers are detected
                            if isOutlier == 1:
                                # get current flip info
                                record_latest_flip_state = kvrecord.get(
                                    "latest_flip_state"
                                )
                                record_latest_flip_time = kvrecord.get(
                                    "latest_flip_time"
                                )

                                # if the flipping event was not registered yet, generate a new flip event

                                if record_latest_flip_state != "red":
                                    new_latest_flip_state = "red"
                                    new_latest_flip_time = time.time()
                                    new_flip_time_human = time.strftime(
                                        "%d/%m/%Y %H:%M:%S",
                                        time.localtime(new_latest_flip_time),
                                    )

                                    # gen a flip event
                                    flip_previous_state = str(record_latest_flip_state)
                                    flip_result = f'{new_flip_time_human}, object="{self.object}" has flipped from previous_state="{record_latest_flip_state}" to state="{new_latest_flip_state}" with anomaly_reason="Machine Learning outliers anomalies were reported"'

                                    if self.context != "simulation":
                                        try:
                                            trackme_audit_flip(
                                                session_key,
                                                reqinfo["server_rest_uri"],
                                                str(self.tenant_id),
                                                key,
                                                kvrecord.get("alias", self.object),
                                                str(self.object),
                                                "splk-cim",
                                                priority,
                                                "red",
                                                str(kvrecord.get("latest_flip_state")),
                                                str(new_latest_flip_time),
                                                str(new_latest_flip_state),
                                                "Machine Learning outliers anomalies were reported",
                                                str(flip_result),
                                            )
                                        except Exception as e:
                                            logging.error(
                                                f'tenant_id="{self.tenant_id}", object="{self.object}", failed to generate a flipping state event with exception="{str(e)}"'
                                            )

                                    # replace variables
                                    record_latest_flip_state = new_latest_flip_state
                                    record_latest_flip_time = new_latest_flip_time

                                try:
                                    # Update the record
                                    kvrecord["object_state"] = object_state

                                    # for each entry in the cim_fields_red_outliers, add to the cim_fields_red if not already there
                                    try:
                                        for outliers_entity in cim_fields_red_outliers:
                                            entity_cim_fields_red_outliers = (
                                                cim_fields_red_outliers[outliers_entity]
                                            )
                                            for field in entity_cim_fields_red_outliers:
                                                if field not in cim_fields_red[entity]:
                                                    cim_fields_red[entity].append(field)
                                    except Exception as e:
                                        logging.error(
                                            f'tenant_id="{self.tenant_id}", object="{self.object}", cim_fields_red_outliers="{cim_fields_red_outliers}", failed to update the cim_fields_red with outliers with exception="{str(e)}"'
                                        )

                                    # for each entry in the cim_fields_green_outliers, add to the cim_fields_green if not already there
                                    try:
                                        for (
                                            outliers_entity
                                        ) in cim_fields_green_outliers:
                                            entity_cim_fields_green_outliers = (
                                                cim_fields_green_outliers[
                                                    outliers_entity
                                                ]
                                            )
                                            for (
                                                field
                                            ) in entity_cim_fields_green_outliers:
                                                if (
                                                    field
                                                    not in cim_fields_green[entity]
                                                ):
                                                    cim_fields_green[entity].append(
                                                        field
                                                    )
                                    except Exception as e:
                                        logging.error(
                                            f'tenant_id="{self.tenant_id}", object="{self.object}", cim_fields_green_outliers="{cim_fields_green_outliers}", failed to update the cim_fields_green with outliers with exception="{str(e)}"'
                                        )

                                    # count the number of green fields in the cim_fields_green dict
                                    cim_fields_green_count = 0
                                    for entity in cim_fields_green:
                                        cim_fields_green_count += len(
                                            cim_fields_green[entity]
                                        )

                                    # count the number of red fields in the cim_fields_red dict
                                    cim_fields_red_count = 0
                                    for entity in cim_fields_red:
                                        cim_fields_red_count += len(
                                            cim_fields_red[entity]
                                        )

                                    kvrecord["cim_tracking_results"] = json.dumps(
                                        {
                                            "isOutlier": isOutlier,
                                            "anomaly_reason": anomaly_reason,
                                            "outlierStatus": outlierStatus,
                                            "isOutlierDesc": isOutlierDesc,
                                            "cim_fields_green": cim_fields_green,
                                            "cim_fields_green_count": cim_fields_green_count,
                                            "cim_fields_red": cim_fields_red,
                                            "cim_fields_red_count": cim_fields_red_count,
                                            "models_processed": models_processed,
                                            "models_in_anomaly": models_in_anomaly,
                                            "anomaly_records": anomaly_records,
                                        },
                                        indent=1,
                                    )
                                    kvrecord["tracker_runtime"] = time.time()
                                    kvrecord["tracker_last_duration"] = (
                                        time.time() - start
                                    )
                                    kvrecord["priority"] = priority
                                    kvrecord["amomaly_reason"] = anomaly_reason

                                    collection.data.update(
                                        str(key), json.dumps(kvrecord)
                                    )

                                except Exception as e:
                                    logging.error(
                                        f'tenant_id="{self.tenant_id}", object="{self.object}", failed to update the KVstore, exception="{str(e)}"'
                                    )

                        #
                        # sla metrics
                        #

                        if self.context == "live":
                            # call the SLA gen metrics function
                            sla_metrics_gen_start = time.time()

                            if object_state == "green":
                                object_num_state = 1
                            elif object_state == "red":
                                object_num_state = 2
                            elif object_state == "orange":
                                object_num_state = 3
                            elif object_state == "blue":
                                object_num_state = 4
                            else:
                                object_num_state = 5

                            try:
                                sla_metrics = trackme_sla_gen_metrics(
                                    self.tenant_id,
                                    trackme_metrics_idx,
                                    [
                                        {
                                            "tenant_id": self.tenant_id,
                                            "object_id": key,
                                            "object": self.object,
                                            "alias": kvrecord.get("alias"),
                                            "object_category": "splk-cim",
                                            "monitored_state": kvrecord.get(
                                                "monitored_state"
                                            ),
                                            "priority": priority,
                                            "metrics_event": {
                                                "object_state": object_num_state
                                            },
                                        }
                                    ],
                                )
                                logging.debug(
                                    f'context="sla_gen_metrics", tenant_id="{self.tenant_id}", function trackme_sla_gen_metrics success {sla_metrics}, run_time={round(time.time()-sla_metrics_gen_start, 3)}'
                                )
                            except Exception as e:
                                logging.error(
                                    f'context="sla_gen_metrics", tenant_id="{self.tenant_id}", function trackme_sla_gen_metrics failed with exception {str(e)}'
                                )

                    #
                    # End of all
                    #

                    # General exception
                    except Exception as e:
                        logging.error(
                            f'tenant_id="{self.tenant_id}", object="{self.object}", status="failure", main search failed with exception="{str(e)}"'
                        )
                        # Call the component register
                        trackme_register_tenant_object_summary(
                            session_key,
                            self._metadata.searchinfo.splunkd_uri,
                            self.tenant_id,
                            "splk-cim",
                            tracker_name,
                            "failure",
                            time.time(),
                            round(time.time() - start, 3),
                            str(e),
                            self.earliest,
                            self.latest,
                        )


dispatch(SplkCimExecutor, sys.argv, sys.stdin, sys.stdout, __name__)
