#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_cim_tracking.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys
import requests
from collections import OrderedDict

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_cim_user", "trackme_rest_api_splk_cim_user.log"
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel

# import Splunk lbs
import splunklib.client as client


class TrackMeHandlerSplkCimTrackingRead_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkCimTrackingRead_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_cim(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_cim",
            "resource_group_desc": "Endpoints specific to the splk-cim TrackMe component (Splunk Common Information Model compliance monitoring, read-only operations).",
        }

        return {"payload": response, "status": 200}

    # get the component table
    def post_cim_get_table(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "the tenant_id is required",
                        },
                        "status": 500,
                    }

                try:
                    key_id = resp_dict["key_id"]
                    if key_id == "*":
                        key_id = None
                except Exception as e:
                    key_id = None

                try:
                    object = resp_dict["object"]
                    if object == "*":
                        object = None
                except Exception as e:
                    object = None

                # only key_id or object can be specified
                if key_id and object:
                    return {
                        "payload": {
                            "action": "failure",
                            "response": "only key_id or object can be specified, not both",
                        },
                        "status": 500,
                    }

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint retrieves the component table. It requires a POST call with the following options:",
                "resource_desc": "Get the entity table",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_cim/cim_get_table\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'key_id': '*', 'object': '*'}\"",
                "options": [
                    {
                        "tenant_id": "(Required) The tenant identifier",
                        "key_id": "(Optional) The key ID. Do not specify this to match all entities",
                        "object": "(Optional) The entity name. Do not specify this to match all entities",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # url
        url = f"{request_info.server_rest_uri}/services/trackme/v2/component/load_component_data"

        # Define an header for requests authenticated communications with splunkd
        header = {
            "Authorization": f"Splunk {request_info.system_authtoken}",
            "Content-Type": "application/json",
        }

        params = {
            "tenant_id": tenant_id,
            "component": "cim",
            "page": 1,
            "size": 0,
        }

        if key_id:
            params["filter_key"] = key_id
        elif object:
            params["object_key"] = object

        data_records = []

        # Proceed
        try:
            response = requests.get(
                url,
                headers=header,
                params=params,
                verify=False,
                timeout=600,
            )

            if response.status_code not in (200, 201, 204):
                msg = f'get component has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                return {"payload": msg, "status": 500}

            else:
                response_json = response.json()
                data = response_json.get("data", [])

                # add the data to the data_records
                for record in data:
                    data_records.append(record)

                # return
                return {"payload": data_records, "status": 200}

        except Exception as e:
            msg = f'get component has failed, exception="{str(e)}"'
            logger.error(msg)
            return {"payload": msg, "status": 500}

    # get all records
    def post_cim_tracker_show(self, request_info, **kwargs):
        """
        | trackme mode=post url="/services/trackme/v2/splk_cim/cim_tracker_show" body="{'tenant_id': 'mytenant'}"
        """

        # Declare
        tenant_id = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint retrieves all records for the CIM tracker collection. It requires a POST call with the following information:",
                "resource_desc": "Get the CIM entities KVstore collection records",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_cim/cim_tracker_show\" body=\"{'tenant_id': 'mytenant'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Data collection
            collection_name = "kv_trackme_cim_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            # Get all records and apply deduplication
            all_records = collection.data.query()
            seen_objects = set()  # Track seen objects for deduplication
            deduplicated_records = []

            for record in all_records:
                object_name = record.get("object")
                
                # Skip if we've already seen this object
                if object_name in seen_objects:
                    continue
                
                # Add object to seen set
                seen_objects.add(object_name)
                deduplicated_records.append(record)

            return {"payload": deduplicated_records, "status": 200}

        except Exception as e:
            logger.error(f'Warn: exception encountered="{str(e)}"')
            return {"payload": f'Warn: exception encountered="{str(e)}"'}

    # Returns the CIM entity rules
    def post_cim_tracker_rules(self, request_info, **kwargs):
        # By key
        describe = False
        tenant_id = None
        object = None
        context = None

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                object = resp_dict["object"]
                # replace chars as would a creation of a new entity do
                object = object.lower().replace(" ", "_").replace("-", "_")

                # context: the context is used for simulation purposes versus live context
                # This is an optional argument, and defaults to live
                # swiching to simulation creates the records in a secondary KVstore that is not used for true alerting purpose
                # As well, the scheduled report will not be created in simulation context
                try:
                    context = resp_dict["context"]
                except Exception as e:
                    context = "live"

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint retrieves tracking rules for a given object:",
                "resource_desc": "Get cim tracking rules for a given entity",
                "resource_spl_example": "| trackme mode=get url=\"/services/trackme/v2/splk_cim/cim_tracker_rules\" body=\"{'tenant_id': 'mytenant', 'object': 'auth001', 'context': 'live'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object": "name of the cim entity, the name of the tracker will be derived from it",
                        "context": "OPTIONAL: context is used for simulation purposes, defaults to live (valid option: live | simulation)",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        if context == "live":
            collection_name = "kv_trackme_cim_tenant_" + str(tenant_id)
        elif context == "simulation":
            collection_name = "kv_trackme_cim_simulation_tenant_" + str(tenant_id)
        collection = service.kvstore[collection_name]

        # Define the KV query
        query_string = {
            "object": object,
        }

        # Get the record
        try:
            kvrecord = collection.data.query(query=json.dumps(query_string))[0]
        except Exception as e:
            kvrecord = None

        # Render result
        if kvrecord:
            logger.debug(
                f'tenant_id="{tenant_id}", object="{object}", success for record="{kvrecord}"'
            )
            return {"payload": kvrecord.get("cim_tracking_rules"), "status": 200}

        # Render empty result
        else:
            logger.info(
                f'tenant_id="{tenant_id}", resource does not exist, object="{object}"'
            )
            return {"payload": {}, "status": 200}
