#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_lagging_classes.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys
import time

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_lagging_classes_power",
    "trackme_rest_api_splk_lagging_classes_power.log",
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel, trackme_audit_event

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkLaggingClassesWrite_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkLaggingClassesWrite_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_lagging_classes(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_lagging_classes",
            "resource_group_desc": "Endpoints related to the management of lagging classes for splk-feeds components (power operations)",
        }

        return {"payload": response, "status": 200}

    # Add new policy
    def post_lagging_classes_add(self, request_info, **kwargs):

        # Declare
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": "Error: tenant_id is required",
                        "status": 400,
                    }

                try:
                    name = resp_dict["name"]
                except Exception as e:
                    return {
                        "payload": "Error: name is required",
                        "status": 400,
                    }

                try:
                    level = resp_dict["level"]
                except Exception as e:
                    return {
                        "payload": "Error: level is required",
                        "status": 400,
                    }

                try:
                    object_value = resp_dict["object"]
                    if object_value not in ("splk-dsm", "splk-dhm", "all"):
                        return {
                            "payload": "Error: object is required and must be splk-dsm, splk-dhm or all",
                            "status": 400,
                        }
                except Exception as e:
                    return {
                        "payload": "Error: object is required",
                        "status": 400,
                    }

                try:
                    value_lag = int(resp_dict["value_lag"])
                except Exception as e:
                    return {
                        "payload": "Error: value_lag is required and must be an integer",
                        "status": 400,
                    }

                try:
                    value_delay = int(resp_dict["value_delay"])
                except Exception as e:
                    return {
                        "payload": "Error: value_delay is required and must be an integer",
                        "status": 400,
                    }

                # optional: a comment for this entry
                try:
                    comment_value = resp_dict["comment"]
                    if len(comment_value) == 0:
                        comment_value = None
                except Exception as e:
                    comment_value = None

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint creates a new lagging class policy, it requires a POST call with the following data:",
                "resource_desc": "Create a new lagging class policy",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_lagging_classes/write/lagging_classes_add\" body=\"{'tenant_id':'mytenant','lagging_class_type':'events','name':'linux_secure','level':'sourcetype','object':'all','value_lag':'900','value_delay':'3600'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant idenfitier",
                        "name": "name of the lagging class",
                        "level": "which level the lagging class is based on, valid otions are: sourcetype / index / priority",
                        "object": "which type of objects the lagging class is applied to, valid options are: splk-dsm / splk-dhm / all",
                        "value_lag": "the lagging value in seconds, an integer is expected",
                        "value_delay": "the delay value in seconds, an integer is expected",
                        "comment": "OPTIONAL: a comment for this blocklist, the comment will be stored and displayed when accessing records",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Define the KV query
        query_string = {
            "$and": [
                {
                    "name": name,
                    "object": object_value,
                    "level": level,
                }
            ]
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Data collection
            collection_name = "kv_trackme_common_lagging_classes_tenant_" + str(
                tenant_id
            )
            collection = service.kvstore[collection_name]

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if (
                key
                and level in ("sourcetype", "index", "priority")
                and object_value in ("splk-dsm", "splk-dhm", "all")
            ):
                # conflict, the same entity exists already
                logger.error(
                    f'tenant_id="{tenant_id}", conflict the same object exists already, name="{name}"'
                )
                return {
                    "payload": f'tenant_id="{tenant_id}", conflict the same object exists already, name="{name}"',
                    "status": 500,
                }

            elif level in ("sourcetype", "index", "priority") and object_value in (
                "splk-dsm",
                "splk-dhm",
                "all",
            ):
                # This record does not exist yet

                # Perform and audit
                try:

                    record_class = {
                        "name": name,
                        "level": level,
                        "object": object_value,
                        "value_lag": str(value_lag),
                        "value_delay": str(value_delay),
                        "mtime": time.time(),
                    }

                    if comment_value:
                        record_class["comment"] = comment_value

                    # Insert the record
                    collection.data.insert(json.dumps(record_class))

                    # Get record
                    record = json.dumps(
                        collection.data.query(query=json.dumps(query_string)), indent=1
                    )

                    # Audit
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "create lagging class",
                            str(name),
                            "lagging_class",
                            str(record),
                            "The lagging class was created successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                except Exception as e:
                    response = {
                        "action": "failure",
                        "response": f'an exception was encountered, exception="{str(e)}"',
                    }
                    logger.error(json.dumps(response))
                    return {"payload": response, "status": 500}

                logger.info("success for record=" + str(record))
                return {"payload": str(record), "status": 200}

            else:
                logger.error(
                    "Error: data submitted is incorrect, valid options for level are sourcetype / index / priority, valid options for object are splk-dsm / splk-dhm / all"
                )
                return {
                    "payload": "Error: data submitted is incorrect, valid options for level are sourcetype / index / priority, valid options for object are splk-dsm / splk-dhm / all",
                    "status": 400,
                }

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Add new policy
    def post_lagging_classes_metrics_add(self, request_info, **kwargs):

        # Declare
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": "Error: tenant_id is required",
                        "status": 400,
                    }

                try:
                    metric_category = resp_dict["metric_category"]
                except Exception as e:
                    return {
                        "payload": "Error: metric_category is required",
                        "status": 400,
                    }

                try:
                    metric_max_lag_allowed = int(resp_dict["metric_max_lag_allowed"])
                except Exception as e:
                    return {
                        "payload": "Error: metric_max_lag_allowed is required and must be an integer",
                        "status": 400,
                    }

                # optional: a comment for this entry
                try:
                    comment_value = resp_dict["comment"]
                    if len(comment_value) == 0:
                        comment_value = None
                except Exception as e:
                    comment_value = None

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint creates a new tag policy, it requires a POST call with the following data:",
                "resource_desc": "Create a new lagging class policy",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_lagging_classes/write/lagging_classes_metrics_add\" body=\"{'tenant_id':'mytenant','metric_category':'spl','metric_max_lag_allowed':'900'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "metric_category": "name of the metric category",
                        "metric_max_lag_allowed": "the lagging value in seconds, an integer is expected",
                        "comment": "OPTIONAL: a comment for this blocklist, the comment will be stored and displayed when accessing records",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        try:
            # Data collection
            collection_name = "kv_trackme_mhm_lagging_classes_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            # Define the KV query
            query_string = {
                "metric_category": metric_category,
            }

            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                # conflict, the same entity exists already
                logger.error(
                    f'tenant_id="{tenant_id}", conflict the same object exists already, metric_category="{metric_category}"'
                )
                return {
                    "payload": f'tenant_id="{tenant_id}", conflict the same object exists already, metric_category="{metric_category}"',
                    "status": 500,
                }

            else:
                # This record does not exist yet
                new_kvrecord = {
                    "metric_category": metric_category,
                    "metric_max_lag_allowed": metric_max_lag_allowed,
                    "mtime": time.time(),
                }

                if comment_value:
                    new_kvrecord["comment"] = comment_value

                # Perform and audit
                try:
                    # Insert the record
                    collection.data.insert(json.dumps(new_kvrecord))

                    # Audit
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "create metric sla policy",
                            str(metric_category),
                            "metric_sla_policy",
                            new_kvrecord,
                            "The metric SLA policy was created successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                except Exception as e:
                    response = {
                        "action": "failure",
                        "response": f'an exception was encountered, exception="{str(e)}"',
                    }
                    logger.error(json.dumps(response))
                    return {"payload": response, "status": 500}

                response = {
                    "action": "success",
                    "response": "lagging class added successfully",
                    "record": new_kvrecord,
                }
                return {"payload": response, "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Delete records from the collection
    def post_lagging_classes_del(self, request_info, **kwargs):

        # Declare
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": "Error: tenant_id is required",
                        "status": 400,
                    }

                try:
                    lagging_class_type = resp_dict["lagging_class_type"]
                    if lagging_class_type not in ("events, metrics"):
                        return {
                            "payload": f'tenant_id="{tenant_id}", invalid option lagging_class_type="{lagging_class_type}"',
                            "status": 500,
                        }

                except Exception as e:
                    return {
                        "payload": "Error: lagging_class_type is required",
                        "status": 400,
                    }

                try:
                    keys_list = resp_dict["keys_list"]
                    # Handle as a CSV list of keys
                    keys_list = keys_list.split(",")
                except Exception as e:
                    return {
                        "payload": "Error: keys_list is required",
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint deletes lagging classes, it requires a POST call with the following information:",
                "resource_desc": "Delete one or more lagging classes",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_lagging_classes/write/lagging_classes_del\" body=\"{'tenant_id':'mytenant','lagging_class_type':'metrics','component':'splk-mhm','keys_list':'63716c098336d473a8152f30'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "lagging_class_type": "The type of lagging classes, valid options are: events | metrics",
                        "keys_list": "List of record keys seperated by a comma of the records to be deleted from the collection",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Data collection
        if lagging_class_type == "events":
            collection_name = "kv_trackme_common_lagging_classes_tenant_" + str(
                tenant_id
            )
        elif lagging_class_type == "metrics":
            collection_name = "kv_trackme_mhm_lagging_classes_tenant_" + str(tenant_id)
        collection = service.kvstore[collection_name]

        # records summary
        records = []

        # loop
        for item in keys_list:
            # Define the KV query
            query_string = {
                "_key": item,
            }

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only
            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                # This record exists already

                # Store the record for audit purposes
                record = str(json.dumps(collection.data.query_by_id(key), indent=1))

                # Remove and audit
                try:
                    # Remove the record
                    collection.data.delete(json.dumps({"_key": key}))

                    # increment counter
                    processed_count += 1
                    succcess_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "delete lagging class",
                            str(item),
                            "all",
                            str(record),
                            "The lagging class was deleted successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "success",
                        "record": record,
                    }

                    records.append(result)

                    logger.info(json.dumps(result, indent=0))

                except Exception as e:
                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "failure",
                            "delete lagging class",
                            str(item),
                            "all",
                            str(record),
                            str(e),
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "failure",
                        "record": record,
                        "exception": e,
                    }

                    # append to records
                    records.append(result)

                    # log
                    logger.error(json.dumps(result, indent=0))

            else:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                # audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "delete lagging class",
                        str(item),
                        "all",
                        str(record),
                        "HTTP 404 NOT FOUND",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                result = {
                    "action": "delete",
                    "result": "failure",
                    "record": item,
                    "exception": "HTTP 404 NOT FOUND",
                }

                # append to records
                records.append(result)

                # log
                logger.error(json.dumps(result, indent=0))

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Update records
    def post_lagging_classes_update(self, request_info, **kwargs):

        # Declare
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": "Error: tenant_id is required",
                        "status": 400,
                    }

                try:
                    lagging_class_type = resp_dict["lagging_class_type"]
                    if lagging_class_type not in ("events, metrics"):
                        return {
                            "payload": f'tenant_id="{tenant_id}", invalid option lagging_class_type="{lagging_class_type}"',
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": "Error: lagging_class_type is required",
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                    if component not in ("dsm", "dhm", "mhm"):
                        return {
                            "payload": f'Invalid component="{component}"',
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": "Error: component is required",
                        "status": 400,
                    }

                try:
                    records_list = resp_dict["records_list"]
                    records_list = json.loads(records_list)
                except Exception as e:
                    return {
                        "payload": "Error: records_list is required",
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint updates records, it requires a POST call with the following information:",
                "resource_desc": "Update one or more lagging classes",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_lagging_classes/write/post_lagging_classes_update\" body=\"{'tenant_id':'mytenant','lagging_class_type':'metrics','component':'splk-mhm','keys_list':'63716c098336d473a8152f30'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "lagging_class_type": "The type of lagging classes, valid options are: events | metrics",
                        "records_list": "JSON records to be updated",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Data collection
        if lagging_class_type == "events":
            collection_name = "kv_trackme_common_lagging_classes_tenant_" + str(
                tenant_id
            )
        elif lagging_class_type == "metrics":
            collection_name = "kv_trackme_mhm_lagging_classes_tenant_" + str(tenant_id)
        collection = service.kvstore[collection_name]

        # records summary
        records = []

        # loop
        for item in records_list:

            # Define the KV query
            query_string = {
                "_key": item.get("_key"),
            }

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only
            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                # This record exists already

                # Store the record for audit purposes
                record = str(json.dumps(collection.data.query_by_id(key), indent=1))

                # Update and audit
                try:
                    # Update the record

                    if lagging_class_type == "events":
                        collection.data.update(
                            str(key),
                            json.dumps(
                                {
                                    "name": item.get("name"),
                                    "level": item.get("level"),
                                    "object": item.get("object"),
                                    "value_lag": item.get("value_lag"),
                                    "value_delay": item.get("value_delay"),
                                    "comment": item.get("comment", ""),
                                    "mtime": time.time(),
                                }
                            ),
                        )
                    elif lagging_class_type == "metrics":
                        collection.data.update(
                            str(key),
                            json.dumps(
                                {
                                    "metric_category": item.get("metric_category"),
                                    "metric_max_lag_allowed": item.get(
                                        "metric_max_lag_allowed"
                                    ),
                                    "comment": item.get("comment", ""),
                                    "mtime": time.time(),
                                }
                            ),
                        )

                    # increment counter
                    processed_count += 1
                    succcess_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "update lagging classes",
                            str(item),
                            str(component),
                            json.dumps(item, indent=1),
                            "The lagging classs was updated successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "success",
                        "record": record,
                    }

                    records.append(result)

                    logger.info(json.dumps(result, indent=0))

                except Exception as e:
                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "failure",
                            "update lagging classes",
                            str(item),
                            str(component),
                            str(record),
                            str(e),
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "failure",
                        "record": record,
                        "exception": e,
                    }

                    # append to records
                    records.append(result)

                    # log
                    logger.error(json.dumps(result, indent=0))

            else:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                # audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "update lagging classes",
                        str(item),
                        str(component),
                        str(record),
                        "HTTP 404 NOT FOUND",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                result = {
                    "action": "delete",
                    "result": "failure",
                    "record": item,
                    "exception": "HTTP 404 NOT FOUND",
                }

                # append to records
                records.append(result)

                # log
                logger.error(json.dumps(result, indent=0))

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}
