#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_splk_dhm.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import re
import json
import os
import random
import sys
import time
import threading
from datetime import datetime
import hashlib
import requests
import urllib.parse

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_outliers_engine_power",
    "trackme_rest_api_splk_outliers_engine_power.log",
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import (
    trackme_getloglevel,
    trackme_audit_event,
    trackme_reqinfo,
    trackme_idx_for_tenant,
    run_splunk_search,
)

# import trackme libs utils
from trackme_libs_utils import remove_leading_spaces

# import trackme libs mloutliers
from trackme_libs_mloutliers import train_mlmodel, train_cim_mlmodel

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkOutliersEngineWrite_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkOutliersEngineWrite_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_outliers_engine(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_outliers_engine/write",
            "resource_group_desc": "Endpoints related to the management of the Machine Learning Outliers detection (power operations)",
        }

        return {"payload": response, "status": 200}

    # Add a new ML model to a given entity
    def post_outliers_add_model(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {"payload": "component is required", "status": 500}
                try:
                    model_json = resp_dict["model_json"]
                except Exception as e:
                    return {"payload": "model_json is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "resource_desc": "Add a new Machine Learning outliers model",
                "resource_spl_example": '| trackme mode=post url="/services/trackme/v2/splk_outliers_engine/write/outliers_add_model" body="{\'tenant_id\':\'mytenant\',\'component\':\'dsm\',\'object\':\'netscreen:netscreen:firewall\',\'model_json\':\'{\\\\\\"kpi_metric\\\\\\":\\\\\\"splk.feeds.perc95_eventcount_5m\\\\\\",\\\\\\"kpi_span\\\\\\":\\\\\\"10m\\\\\\",\\\\\\"method_calculation\\\\\\":\\\\\\"avg\\\\\\",\\\\\\"density_lowerthreshold\\\\\\":\\\\\\"0.005\\\\\\",\\\\\\"density_upperthreshold\\\\\\":\\\\\\"0.005\\\\\\",\\\\\\"alert_lower_breached\\\\\\":\\\\\\"1\\\\\\",\\\\\\"alert_upper_breached\\\\\\":\\\\\\"1\\\\\\",\\\\\\"period_calculation\\\\\\":\\\\\\"-30d\\\\\\",\\\\\\"time_factor\\\\\\":\\\\\\"%25H\\\\\\",\\\\\\"perc_min_lowerbound_deviation\\\\\\":\\\\\\"5.0\\\\\\",\\\\\\"perc_min_upperbound_deviation\\\\\\":\\\\\\"5.0\\\\\\"}\'}"',
                "describe": "This endpoint adds a new ML model to a given entity, it requires a POST call with the following options:",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "conponent": "(required) The component category",
                        "object": "(required) entity name",
                        "model_json": "(required) The new ML model JSON definition",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Component collection
        collection_component_name = f"kv_trackme_{component}_tenant_{tenant_id}"
        collection_component = service.kvstore[collection_component_name]

        # Data collection_rules
        collection_rules_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_rules = service.kvstore[collection_rules_name]

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Try to load the model JSON definition, if loading fails, stop here
        if not isinstance(model_json, dict):
            try:
                new_model_definition = json.loads(model_json)
                logger.info(
                    f'Successfully loaded model_json="{json.dumps(new_model_definition, indent=4)}"'
                )
            except Exception as e:
                msg = f'Failed to load the model_json="{model_json}" as a properly formatted JSON object with exception="{str(e)}"'
                logger.error(msg)
                return {"payload": msg, "status": 500}
        else:
            new_model_definition = model_json

        # records summary
        records = []

        # Get the component current record
        try:
            # Define the KV query
            query_string = {
                "$and": [
                    {
                        "object_category": f"splk-{component}",
                        "object": object_value,
                    }
                ]
            }
            component_record = collection_component.data.query(
                query=json.dumps(query_string)
            )[0]
            key = component_record.get("_key")

        except Exception as e:
            key = None

        # Get the current entity_rules record, if any

        # init
        entity_rules = {}
        entities_outliers = {}
        entity_has_rules = False

        try:
            # Define the KV query
            query_string = {
                "$and": [
                    {
                        "object_category": f"splk-{component}",
                        "object": object_value,
                    }
                ]
            }

            entity_rules = collection_rules.data.query(query=json.dumps(query_string))[
                0
            ]
        except Exception as e:
            pass

        if entity_rules:
            entity_has_rules = True

        #
        # main
        #

        if (
            not key
        ):  # cannot continue if the object is not found in the component collection
            error_msg = {
                "payload": "object not found",
                "query": query_string,
                "collection": collection_component_name,
                "status": 404,
            }
            logger.error(json.dumps(error_msg))
            return {"payload": error_msg, "status": 404}

        else:
            # Load as a dict
            if entity_rules:
                try:
                    entities_outliers = json.loads(
                        entity_rules.get("entities_outliers")
                    )
                except Exception as e:
                    entities_outliers = {}

            # log debug
            logger.debug(
                f'entities_outliers="{json.dumps(entities_outliers, indent=4)}"'
            )

            # For each expected key, try to retrieve the value
            try:
                kpi_metric = new_model_definition.get("kpi_metric")
                kpi_span = new_model_definition.get("kpi_span")
                method_calculation = new_model_definition.get("method_calculation")
                density_lowerthreshold = new_model_definition.get(
                    "density_lowerthreshold"
                )
                density_upperthreshold = new_model_definition.get(
                    "density_upperthreshold"
                )
                alert_lower_breached = new_model_definition.get("alert_lower_breached")
                alert_upper_breached = new_model_definition.get("alert_upper_breached")
                period_calculation = new_model_definition.get("period_calculation")
                # optional period_calculation_latest
                period_calculation_latest = new_model_definition.get(
                    "period_calculation_latest", "now"
                )
                time_factor = urllib.parse.unquote(
                    new_model_definition.get("time_factor")
                )
                min_value_for_lowerbound_breached = new_model_definition.get(
                    "min_value_for_lowerbound_breached", 0
                )
                min_value_for_upperbound_breached = new_model_definition.get(
                    "min_value_for_upperbound_breached", 0
                )
                static_lower_threshold = new_model_definition.get(
                    "static_lower_threshold", None
                )
                static_upper_threshold = new_model_definition.get(
                    "static_upper_threshold", None
                )
                auto_correct = new_model_definition.get("auto_correct")
                perc_min_lowerbound_deviation = new_model_definition.get(
                    "perc_min_lowerbound_deviation"
                )
                perc_min_upperbound_deviation = new_model_definition.get(
                    "perc_min_upperbound_deviation"
                )

            except Exception as e:
                msg = f'Failed to retrieve expected key from model_json with exception="{str(e)}"'
                logger.error(msg)
                return {"payload": msg, "status": 500}

            # Set the model identifier
            model = f"model_{random.getrandbits(48)}"

            # Set the final model definition
            final_model_definition = {
                "is_disabled": 0,
                "kpi_metric": kpi_metric,
                "kpi_span": kpi_span,
                "method_calculation": method_calculation,
                "density_lowerthreshold": density_lowerthreshold,
                "density_upperthreshold": density_upperthreshold,
                "alert_lower_breached": alert_lower_breached,
                "alert_upper_breached": alert_upper_breached,
                "period_calculation": period_calculation,
                "period_calculation_latest": period_calculation_latest,
                "time_factor": time_factor,
                "auto_correct": auto_correct,
                "perc_min_lowerbound_deviation": perc_min_lowerbound_deviation,
                "perc_min_upperbound_deviation": perc_min_upperbound_deviation,
                "min_value_for_lowerbound_breached": min_value_for_lowerbound_breached,
                "min_value_for_upperbound_breached": min_value_for_upperbound_breached,
                "static_lower_threshold": static_lower_threshold,
                "static_upper_threshold": static_upper_threshold,
                "period_exclusions": [],
                "ml_model_gen_search": "pending",
                "ml_model_render_search": "pending",
                "ml_model_summary_search": "pending",
                "rules_access_search": "pending",
                "ml_model_filename": "pending",
                "ml_model_filesize": "pending",
                "ml_model_lookup_share": "pending",
                "ml_model_lookup_owner": "pending",
                "last_exec": "pending",
            }

            # Add the new model to the dict
            entities_outliers[model] = final_model_definition

            # log debug
            logger.debug(
                f'final model_dict="{json.dumps(entities_outliers, indent=4)}"'
            )

            try:
                # Update the record
                entity_rules["entities_outliers"] = json.dumps(
                    entities_outliers, indent=4
                )
                entity_rules["mtime"] = time.time()

                # Insert or update
                if entity_has_rules:
                    collection_rules.data.update(str(key), json.dumps(entity_rules))
                else:
                    new_kvrecord = {
                        "_key": key,
                        "object_category": f"splk-{component}",
                        "object": object_value,
                        "confidence": "pending",
                        "confidence_reason": "pending",
                        "is_disabled": 0,
                        "mtime": time.time(),
                        "entities_outliers": json.dumps(entities_outliers, indent=4),
                    }
                    collection_rules.data.insert(json.dumps(new_kvrecord))

                # increment counter
                processed_count += 1
                succcess_count += 1
                failures_count += 0

                # append for summary
                result = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "action": "add",
                    "result": "success",
                    "model_definition": {
                        "kpi_metric": kpi_metric,
                        "kpi_span": kpi_span,
                        "method_calculation": method_calculation,
                        "density_lowerthreshold": density_lowerthreshold,
                        "density_upperthreshold": density_upperthreshold,
                        "alert_lower_breached": alert_lower_breached,
                        "alert_upper_breached": alert_upper_breached,
                        "period_calculation": period_calculation,
                        "period_calculation_latest": period_calculation_latest,
                        "time_factor": time_factor,
                        "auto_correct": auto_correct,
                        "perc_min_lowerbound_deviation": perc_min_lowerbound_deviation,
                        "perc_min_upperbound_deviation": perc_min_upperbound_deviation,
                        "min_value_for_lowerbound_breached": min_value_for_lowerbound_breached,
                        "min_value_for_upperbound_breached": min_value_for_upperbound_breached,
                        "static_lower_threshold": static_lower_threshold,
                        "static_upper_threshold": static_upper_threshold,
                    },
                    "message": f'the model="{model}" was successfully added to the outliers rules',
                }
                records.append(result)

            except Exception as e:
                logger.error(
                    f'failed to add the new model to the dictionnary with exception="{str(e)}"'
                )

                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                result = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "action": "add",
                    "result": "failure",
                    "model_definition": {
                        "kpi_metric": kpi_metric,
                        "kpi_span": kpi_span,
                        "method_calculation": method_calculation,
                        "density_lowerthreshold": density_lowerthreshold,
                        "density_upperthreshold": density_upperthreshold,
                        "alert_lower_breached": alert_lower_breached,
                        "alert_upper_breached": alert_upper_breached,
                        "period_calculation": period_calculation,
                        "period_calculation_latest": period_calculation_latest,
                        "time_factor": time_factor,
                        "auto_correct": auto_correct,
                        "perc_min_lowerbound_deviation": perc_min_lowerbound_deviation,
                        "perc_min_upperbound_deviation": perc_min_upperbound_deviation,
                        "min_value_for_lowerbound_breached": min_value_for_lowerbound_breached,
                        "min_value_for_upperbound_breached": min_value_for_upperbound_breached,
                        "static_lower_threshold": static_lower_threshold,
                        "static_upper_threshold": static_upper_threshold,
                    },
                    "exception": f'failed to add the new model to the dictionnary with exception="{str(e)}"',
                }
                records.append(result)

            # log debug
            logger.debug(
                f'final dict, entity_rules="{json.dumps(entities_outliers, indent=4)}"'
            )

            # render HTTP status and summary

            req_summary = {
                "process_count": processed_count,
                "success_count": succcess_count,
                "failures_count": failures_count,
                "records": records,
            }

            if processed_count > 0 and processed_count == succcess_count:
                # log
                logger.info(
                    f'ML model was added successfully, summary="{json.dumps(req_summary, indent=4)}"'
                )

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "add ML models",
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(req_summary, indent=1)),
                        "ML model was added successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": req_summary, "status": 200}

            else:
                # log
                logger.error(
                    f'ML model could not be added, summary="{json.dumps(req_summary, indent=4)}"'
                )

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "add ML models",
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(req_summary, indent=1)),
                        "ML model could not be added",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": req_summary, "status": 500}

    # Add a new ML model to a given entity
    def post_outliers_cim_add_model(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                try:
                    entity_value = resp_dict["entity"]
                except Exception as e:
                    return {"payload": "entity is required", "status": 500}
                try:
                    cim_field = resp_dict["cim_field"]
                except Exception as e:
                    return {"payload": "cim_field is required", "status": 500}
                try:
                    model_json = resp_dict["model_json"]
                except Exception as e:
                    return {"payload": "model_json is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint adds a new ML model to a given entity, it requires a POST call with the following options:",
                "resource_desc": "Add a new Machine Learning outliers model",
                "resource_spl_example": '| trackme mode=post url="/services/trackme/v2/splk_outliers_engine/write/outliers_cim_add_model" body="{\'tenant_id\':\'cim-demo\',\'object\':\'auth001\',\'entity\': \'global\',\'cim_field\': \'signature\',\'model_json\':\'{\\\\\\"entity\\\\\\":\\\\\\"global\\\\\\",\\\\\\"cim_field\\\\\\":\\\\\\"signature\\\\\\",\\\\\\"kpi_metric\\\\\\":\\\\\\"splk.cim.total_count\\\\\\",\\\\\\"kpi_span\\\\\\":\\\\\\"10m\\\\\\",\\\\\\"method_calculation\\\\\\":\\\\\\"avg\\\\\\",\\\\\\"density_lowerthreshold\\\\\\":\\\\\\"0.005\\\\\\",\\\\\\"density_upperthreshold\\\\\\":\\\\\\"0.005\\\\\\",\\\\\\"alert_lower_breached\\\\\\":\\\\\\"1\\\\\\",\\\\\\"alert_upper_breached\\\\\\":\\\\\\"1\\\\\\",\\\\\\"period_calculation\\\\\\":\\\\\\"-30d\\\\\\",\\\\\\"time_factor\\\\\\":\\\\\\"%25H\\\\\\",\\\\\\"perc_min_lowerbound_deviation\\\\\\":\\\\\\"5.0\\\\\\",\\\\\\"perc_min_upperbound_deviation\\\\\\":\\\\\\"5.0\\\\\\"}\'}"',
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "object": "(required) entity name",
                        "entity": "(required) The cim entity, if the cim tracker is not using a custom break by logic, the cim entity is global",
                        "cim_field": "(required) The cim field",
                        "model_json": "(required) The new ML model JSON definition",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # component collection
        collection_component_name = f"kv_trackme_cim_tenant_{tenant_id}"
        collection_component = service.kvstore[collection_component_name]

        # Data collection_rules
        collection_rules_name = (
            f"kv_trackme_cim_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_rules = service.kvstore[collection_rules_name]

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Try to load the model JSON definition, if loading fails, stop here
        if not isinstance(model_json, dict):
            try:
                new_model_definition = json.loads(model_json)
                logger.info(
                    f'Successfully loaded model_json="{json.dumps(new_model_definition, indent=4)}"'
                )
            except Exception as e:
                msg = f'Failed to load the model_json="{model_json}" as a properly formatted JSON object with exception="{str(e)}"'
                logger.error(msg)
                return {"payload": msg, "status": 500}
        else:
            new_model_definition = model_json

        # records summary
        records = []

        # Get the component current record
        try:
            # Define the KV query
            query_string = {
                "$and": [
                    {
                        "object_category": "splk-cim",
                        "object": object_value,
                    }
                ]
            }
            component_record = collection_component.data.query(
                query=json.dumps(query_string)
            )[0]
            key = component_record.get("_key")

        except Exception as e:
            key = None

        # Get the current entity_rules record, if any

        # init
        entity_rules = {}
        entities_outliers = {}
        entities_outliers_key = None
        entity_has_rules = False

        try:
            # Define the KV query
            query_string = {
                "$and": [
                    {
                        "object_category": "splk-cim",
                        "object": object_value,
                        "cim_field": cim_field,
                        "entity": entity_value,
                    }
                ]
            }

            entity_rules = collection_rules.data.query(query=json.dumps(query_string))[
                0
            ]
            entities_outliers_key = entity_rules.get("_key")
        except Exception as e:
            pass

        if entity_rules:
            entity_has_rules = True

        #
        # main
        #

        if (
            not key
        ):  # cannot continue if the object is not found in the component collection
            error_msg = {
                "payload": "object not found",
                "query": query_string,
                "status": 404,
            }
            logger.error(json.dumps(error_msg))
            return {"payload": error_msg, "status": 404}

        else:

            # Load as a dict
            if entity_rules:
                try:
                    entities_outliers = json.loads(
                        entity_rules.get("entities_outliers")
                    )
                except Exception as e:
                    entities_outliers = {}

            # For each expected key, try to retrieve the value
            try:
                kpi_metric = new_model_definition.get("kpi_metric")
                kpi_span = new_model_definition.get("kpi_span")
                method_calculation = new_model_definition.get("method_calculation")
                density_lowerthreshold = new_model_definition.get(
                    "density_lowerthreshold"
                )
                density_upperthreshold = new_model_definition.get(
                    "density_upperthreshold"
                )
                alert_lower_breached = new_model_definition.get("alert_lower_breached")
                alert_upper_breached = new_model_definition.get("alert_upper_breached")
                period_calculation = new_model_definition.get("period_calculation")
                # optional period_calculation_latest
                period_calculation_latest = new_model_definition.get(
                    "period_calculation_latest", "now"
                )
                time_factor = urllib.parse.unquote(
                    new_model_definition.get("time_factor")
                )
                auto_correct = new_model_definition.get("auto_correct")
                perc_min_lowerbound_deviation = new_model_definition.get(
                    "perc_min_lowerbound_deviation"
                )
                perc_min_upperbound_deviation = new_model_definition.get(
                    "perc_min_upperbound_deviation"
                )
                min_value_for_lowerbound_breached = new_model_definition.get(
                    "min_value_for_lowerbound_breached", 0
                )
                min_value_for_upperbound_breached = new_model_definition.get(
                    "min_value_for_upperbound_breached", 0
                )
                static_lower_threshold = new_model_definition.get(
                    "static_lower_threshold", None
                )
                static_upper_threshold = new_model_definition.get(
                    "static_upper_threshold", None
                )

            except Exception as e:
                msg = f'Failed to retrieve expected key from model_json with exception="{str(e)}"'
                logger.error(msg)
                return {"payload": msg, "status": 500}

            # Set the model identifier
            model = f"model_{random.getrandbits(48)}"

            # Set the final model definition
            final_model_definition = {
                "is_disabled": 0,
                "entity": entity_value,
                "cim_field": cim_field,
                "kpi_metric": kpi_metric,
                "kpi_span": kpi_span,
                "method_calculation": method_calculation,
                "density_lowerthreshold": density_lowerthreshold,
                "density_upperthreshold": density_upperthreshold,
                "alert_lower_breached": alert_lower_breached,
                "alert_upper_breached": alert_upper_breached,
                "period_calculation": period_calculation,
                "period_calculation_latest": period_calculation_latest,
                "time_factor": time_factor,
                "auto_correct": auto_correct,
                "perc_min_lowerbound_deviation": perc_min_lowerbound_deviation,
                "perc_min_upperbound_deviation": perc_min_upperbound_deviation,
                "min_value_for_lowerbound_breached": min_value_for_lowerbound_breached,
                "min_value_for_upperbound_breached": min_value_for_upperbound_breached,
                "static_lower_threshold": static_lower_threshold,
                "static_upper_threshold": static_upper_threshold,
                "period_exclusions": [],
                "ml_model_gen_search": "pending",
                "ml_model_render_search": "pending",
                "ml_model_summary_search": "pending",
                "rules_access_search": "pending",
                "ml_model_filename": "pending",
                "ml_model_filesize": "pending",
                "ml_model_lookup_share": "pending",
                "ml_model_lookup_owner": "pending",
                "last_exec": "pending",
            }

            # Add the new model to the dict
            entities_outliers[model] = final_model_definition

            # log debug
            logger.debug(
                f'final model_dict="{json.dumps(entities_outliers, indent=4)}"'
            )

            try:
                # Update the record
                entity_rules["entities_outliers"] = json.dumps(
                    entities_outliers, indent=4
                )
                entity_rules["mtime"] = time.time()

                # Insert or update
                if entity_has_rules:
                    collection_rules.data.update(
                        str(entities_outliers_key), json.dumps(entity_rules)
                    )
                else:
                    new_kvrecord_key = str(random.getrandbits(128))
                    new_kvrecord = {
                        "_key": new_kvrecord_key,
                        "object_category": f"splk-cim",
                        "object": object_value,
                        "cim_field": cim_field,
                        "entity": entity_value,
                        "confidence": "pending",
                        "confidence_reason": "pending",
                        "is_disabled": 0,
                        "mtime": time.time(),
                        "entities_outliers": json.dumps(entities_outliers, indent=4),
                    }
                    collection_rules.data.insert(json.dumps(new_kvrecord))

                # increment counter
                processed_count += 1
                succcess_count += 1
                failures_count += 0

                # append for summary
                result = {
                    "tenant_id": tenant_id,
                    "object_category": "splk-cim",
                    "object": object_value,
                    "entity": entity_value,
                    "cim_field": cim_field,
                    "action": "add",
                    "result": "success",
                    "model_definition": {
                        "entity": entity_value,
                        "cim_field": cim_field,
                        "kpi_metric": kpi_metric,
                        "kpi_span": kpi_span,
                        "method_calculation": method_calculation,
                        "density_lowerthreshold": density_lowerthreshold,
                        "density_upperthreshold": density_upperthreshold,
                        "alert_lower_breached": alert_lower_breached,
                        "alert_upper_breached": alert_upper_breached,
                        "period_calculation": period_calculation,
                        "period_calculation_latest": period_calculation_latest,
                        "time_factor": time_factor,
                        "auto_correct": auto_correct,
                        "perc_min_lowerbound_deviation": perc_min_lowerbound_deviation,
                        "perc_min_upperbound_deviation": perc_min_upperbound_deviation,
                        "min_value_for_lowerbound_breached": min_value_for_lowerbound_breached,
                        "min_value_for_upperbound_breached": min_value_for_upperbound_breached,
                        "static_lower_threshold": static_lower_threshold,
                        "static_upper_threshold": static_upper_threshold,
                    },
                    "message": f'the model="{model}" was successfully added to the outliers rules',
                }
                records.append(result)

            except Exception as e:
                logger.error(
                    f'failed to add the new model to the dictionnary with exception="{str(e)}"'
                )

                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                result = {
                    "tenant_id": tenant_id,
                    "object_category": "splk-cim",
                    "object": object_value,
                    "entity": entity_value,
                    "cim_field": cim_field,
                    "action": "add",
                    "result": "failure",
                    "model_definition": {
                        "entity": entity_value,
                        "cim_field": cim_field,
                        "kpi_metric": kpi_metric,
                        "kpi_span": kpi_span,
                        "method_calculation": method_calculation,
                        "density_lowerthreshold": density_lowerthreshold,
                        "density_upperthreshold": density_upperthreshold,
                        "alert_lower_breached": alert_lower_breached,
                        "alert_upper_breached": alert_upper_breached,
                        "period_calculation": period_calculation,
                        "period_calculation_latest": period_calculation_latest,
                        "time_factor": time_factor,
                        "auto_correct": auto_correct,
                        "perc_min_lowerbound_deviation": perc_min_lowerbound_deviation,
                        "perc_min_upperbound_deviation": perc_min_upperbound_deviation,
                        "min_value_for_lowerbound_breached": min_value_for_lowerbound_breached,
                        "min_value_for_upperbound_breached": min_value_for_upperbound_breached,
                        "static_lower_threshold": static_lower_threshold,
                        "static_upper_threshold": static_upper_threshold,
                    },
                    "exception": f'failed to add the new model to the dictionnary with exception="{str(e)}"',
                }
                records.append(result)

            # log debug
            logger.debug(
                f'final dict, entity_rules="{json.dumps(entities_outliers, indent=4)}"'
            )

            # render HTTP status and summary

            req_summary = {
                "process_count": processed_count,
                "success_count": succcess_count,
                "failures_count": failures_count,
                "records": records,
            }

            if processed_count > 0 and processed_count == succcess_count:
                # log
                logger.info(
                    f'ML model was added successfully, summary="{json.dumps(req_summary, indent=4)}"'
                )

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "add ML models",
                        str(object_value),
                        "splk-cim",
                        str(json.dumps(req_summary, indent=1)),
                        "ML model was added successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": req_summary, "status": 200}

            else:
                # log
                logger.error(
                    f'ML model could not be added, summary="{json.dumps(req_summary, indent=4)}"'
                )

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "add ML models",
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(req_summary, indent=1)),
                        "ML model could not be added",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": req_summary, "status": 500}

    # delete one or more ML models
    def post_outliers_delete_models(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {"payload": "component is required", "status": 500}
                try:
                    models_list = resp_dict["models_list"]
                    # if not a list already, convert to a list from comma separated string
                    if not isinstance(models_list, list):
                        models_list = models_list.split(",")
                except Exception as e:
                    return {"payload": "models_list is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint deletes or more existing ML models, it requires a POST call with the following options:",
                "resource_desc": "Delete a Machine Learning outliers model",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_delete_models\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'object': 'netscreen:netscreen:firewall', 'models_list': 'model_xxxxxxxxxx'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name",
                        "models_list": "(required) Comma separated list of models identifiers to be deleted",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Data collection
        collection_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection = service.kvstore[collection_name]

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        try:
            # Define the KV query
            query_string = {
                "$and": [
                    {
                        "object_category": f"splk-{component}",
                        "object": object_value,
                    }
                ]
            }

            entity_rules = collection.data.query(query=json.dumps(query_string))[0]
            key = entity_rules.get("_key")

        except Exception as e:
            key = None

        # Render result
        if key:
            logger.debug(entity_rules)

            # Load as a dict
            try:
                entities_outliers = json.loads(entity_rules.get("entities_outliers"))
            except Exception as e:
                msg = f'Failed to load entities_outliers with exception="{str(e)}"'
                logger.error(msg)
                return {"payload": msg, "status": 500}

            # log debug
            logger.debug(
                f'entities_outliers="{json.dumps(entities_outliers, indent=4)}"'
            )

            # loop through the models
            for model in models_list:
                try:
                    model_dict = entities_outliers[model]

                    # log debug
                    logger.debug(f'model_dict="{json.dumps(model_dict, indent=4)}"')

                    # delete from the dict
                    del entities_outliers[model]

                    # if the last model from the rules was deleted, Ml is not ready any longer
                    if not len(entities_outliers) > 2:
                        last_exec = "pending"
                    else:
                        last_exec = entity_rules.get("last_exec")

                    # Update the record
                    entity_rules["entities_outliers"] = json.dumps(
                        entities_outliers, indent=4
                    )
                    entity_rules["mtime"] = time.time()
                    entity_rules["last_exec"] = last_exec
                    collection.data.update(str(key), json.dumps(entity_rules))

                    # increment counter
                    processed_count += 1
                    succcess_count += 1
                    failures_count += 0

                    # append for summary
                    result = {
                        "tenant_id": tenant_id,
                        "object_category": f"splk-{component}",
                        "object": object_value,
                        "action": "delete",
                        "result": "success",
                        "message": f'the model="{model}" was successfully deleted from the outliers rules',
                    }
                    records.append(result)

                except Exception as e:
                    logger.error(
                        f'model="{model}" could not be found in the entity_rules="{json.dumps(entities_outliers, indent=4)}"'
                    )

                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    result = {
                        "tenant_id": tenant_id,
                        "object_category": f"splk-{component}",
                        "object": object_value,
                        "action": "delete",
                        "result": "failure",
                        "exception": f'the model="{model}" could not be found in the outliers rules',
                    }
                    records.append(result)

            # log debug
            logger.debug(
                f'final dict, entity_rules="{json.dumps(entities_outliers, indent=4)}"'
            )

            # render HTTP status and summary

            req_summary = {
                "process_count": processed_count,
                "success_count": succcess_count,
                "failures_count": failures_count,
                "records": records,
            }

            if processed_count > 0 and processed_count == succcess_count:
                # log
                logger.info(
                    f'ML models were deleted successfully, summary="{json.dumps(req_summary, indent=4)}"'
                )

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "delete ML models",
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(req_summary, indent=1)),
                        "ML models were deleted successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": req_summary, "status": 200}

            else:
                # log
                logger.error(
                    f'ML models could not be deleted, summary="{json.dumps(req_summary, indent=4)}"'
                )

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "delete ML models",
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(req_summary, indent=1)),
                        "ML models could not be deleted",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": req_summary, "status": 500}

        else:
            msg = f'No record found for this object, query_string="{json.dumps(query_string, indent=2)}", collection="{collection_name}"'
            logger.error(msg)
            return {"payload": {"response": msg}, "status": 404}

    # delete one or more ML models for splk-cim
    def post_outliers_cim_delete_models(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                try:
                    json_dict = json.loads(resp_dict["json_dict"])
                except Exception as e:
                    return {"payload": "json_dict is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint deletes or more existing ML models, it requires a POST call with the following options:",
                "resource_desc": "Delete a Machine Learning outliers model",
                "resource_spl_example": '| trackme mode=post url="/services/trackme/v2/splk_outliers_engine/write/outliers_cim_delete_models" body="{\'tenant_id\':\'cim-demo\',\'object\':\'auth001\',\'json_dict\':\'[{\\\\\\"model_id\\\\\\":\\\\\\"model_150156483858859\\\\\\",\\\\\\"entity\\\\\\":\\\\\\"global\\\\\\",\\\\\\"cim_field\\\\\\":\\\\\\"app\\\\\\"},{\\\\\\"model_id\\\\\\":\\\\\\"model_129056918057674\\\\\\",\\\\\\"entity\\\\\\":\\\\\\"global\\\\\\",\\\\\\"cim_field\\\\\\":\\\\\\"app\\\\\\"}]\'}"',
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "object": "(required) The name of the object",
                        "json_dict": "(required) A list of JSON dictionaries containing for each model to be deleted, the entity, cim_field and model_id",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Data collection
        collection_name = f"kv_trackme_cim_outliers_entity_rules_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Loop
        for p in json_dict:
            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            try:
                # Define the KV query
                query_string = {
                    "$and": [
                        {
                            "object_category": "splk-cim",
                            "object": object_value,
                            "entity": p["entity"],
                            "cim_field": p["cim_field"],
                        }
                    ]
                }

                entity_rules = collection.data.query(query=json.dumps(query_string))[0]
                key = entity_rules.get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                # Load as a dict
                try:
                    entities_outliers = json.loads(
                        entity_rules.get("entities_outliers")
                    )
                except Exception as e:
                    msg = f'Failed to load entities_outliers with exception="{str(e)}"'
                    logger.error(msg)
                    return {"payload": msg, "status": 500}

                # log debug
                logger.debug(
                    f'entities_outliers="{json.dumps(entities_outliers, indent=4)}"'
                )

                # loop through the models
                model = p["model_id"]

                try:
                    model_dict = entities_outliers[model]

                    # log debug
                    logger.debug(f'model_dict="{json.dumps(model_dict, indent=4)}"')

                    # delete from the dict
                    del entities_outliers[model]

                    # if the last model from the rules was deleted, Ml is not ready any longer
                    if not len(entities_outliers) > 2:
                        last_exec = "pending"
                    else:
                        last_exec = entity_rules.get("last_exec")

                    # Update the record
                    entity_rules["entities_outliers"] = json.dumps(
                        entities_outliers, indent=4
                    )
                    entity_rules["mtime"] = time.time()
                    entity_rules["last_exec"] = last_exec
                    collection.data.update(str(key), json.dumps(entity_rules))

                    # increment counter
                    processed_count += 1
                    succcess_count += 1
                    failures_count += 0

                    # append for summary
                    result = {
                        "tenant_id": tenant_id,
                        "object_category": "splk-cim",
                        "object": object_value,
                        "entity": p["entity"],
                        "cim_field": p["cim_field"],
                        "action": "delete",
                        "result": "success",
                        "message": f'the model="{model}" was successfully deleted from the outliers rules',
                    }
                    records.append(result)

                except Exception as e:
                    logger.error(
                        f'model="{model}" could not be found in the entity_rules="{json.dumps(entities_outliers, indent=4)}"'
                    )

                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    result = {
                        "tenant_id": tenant_id,
                        "object_category": "splk-cim",
                        "object": object_value,
                        "entity": p["entity"],
                        "cim_field": p["cim_field"],
                        "action": "delete",
                        "result": "failure",
                        "exception": f'the model="{model}" could not be found in the outliers rules',
                    }
                    records.append(result)

                # log debug
                logger.debug(
                    f'final dict, entity_rules="{json.dumps(entities_outliers, indent=4)}"'
                )

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            # log
            logger.info(
                f'ML models were deleted successfully, summary="{json.dumps(req_summary, indent=4)}"'
            )

            # audit
            try:
                trackme_audit_event(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    request_info.user,
                    "success",
                    "delete ML models",
                    str(object_value),
                    "splk-dsm",
                    str(json.dumps(req_summary, indent=1)),
                    "ML models were deleted successfully",
                    str(update_comment),
                )
            except Exception as e:
                logger.error(
                    f'failed to generate an audit event with exception="{str(e)}"'
                )

            return {"payload": req_summary, "status": 200}

        else:
            # log
            logger.error(
                f'ML models could not be deleted, summary="{json.dumps(req_summary, indent=4)}"'
            )

            # audit
            try:
                trackme_audit_event(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    request_info.user,
                    "failure",
                    "delete ML models",
                    str(object_value),
                    "splk-dsm",
                    str(json.dumps(req_summary, indent=1)),
                    "ML models could not be deleted",
                    str(update_comment),
                )
            except Exception as e:
                logger.error(
                    f'failed to generate an audit event with exception="{str(e)}"'
                )

            return {"payload": req_summary, "status": 500}

    # update one or more ML models
    def post_outliers_update_models(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {"payload": "component is required", "status": 500}
                try:
                    outliers_rules = resp_dict["outliers_rules"]
                except Exception as e:
                    return {"payload": "outliers_rules is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint updates existing ML models rules, it requires a POST call with the following options:",
                "resource_desc": "Update a Machine Learning outliers model",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_update_models\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall','outliers_rules':'<redacted_json_dict>'}",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name",
                        "outliers_rules": "(required) The JSON array object containing the outliers rules",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Data collection
        collection_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection = service.kvstore[collection_name]

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        try:
            # Define the KV query
            query_string = {
                "$and": [
                    {
                        "object_category": f"splk-{component}",
                        "object": object_value,
                    }
                ]
            }

            entity_rules = collection.data.query(query=json.dumps(query_string))[0]
            key = entity_rules.get("_key")

        except Exception as e:
            key = None

        # Render result
        if key:
            logger.debug(entity_rules)

            # Load as a dict
            try:
                entities_outliers = json.loads(entity_rules.get("entities_outliers"))
            except Exception as e:
                msg = f'Failed to load entities_outliers with exception="{str(e)}"'
                logger.error(msg)
                return {"payload": msg, "status": 500}

            # log debug
            logger.debug(
                f'before update, entities_outliers="{json.dumps(entities_outliers, indent=4)}"'
            )

            #
            # Process update
            #

            # load the JSON update as a dict
            logger.debug(f'outliers_rules="{outliers_rules}"')

            for outliers_rule in outliers_rules:
                # log debug
                logger.debug(f'outliers_rule_update="{outliers_rule}"')

                # get the model_id
                model_id = outliers_rule.get("model_id")

                # log debug
                logger.debug(f'Handling model_id="{model_id}"')

                # Update the main dict
                fields_list = [
                    "kpi_metric",
                    "kpi_span",
                    "method_calculation",
                    "period_calculation",
                    "period_calculation_latest",
                    "time_factor",
                    "density_lowerthreshold",
                    "density_upperthreshold",
                    "alert_lower_breached",
                    "alert_upper_breached",
                    "auto_correct",
                    "perc_min_lowerbound_deviation",
                    "perc_min_upperbound_deviation",
                    "min_value_for_lowerbound_breached",
                    "min_value_for_upperbound_breached",
                    "static_lower_threshold",
                    "static_upper_threshold",
                    "algorithm",
                    "boundaries_extraction_macro",
                    "fit_extra_parameters",
                    "apply_extra_parameters",
                    "is_disabled",
                ]
                for field in fields_list:
                    if field in ("time_factor"):
                        entities_outliers[model_id][field] = urllib.parse.unquote(
                            outliers_rule.get(field)
                        )
                    elif field in (
                        "min_value_for_lowerbound_breached",
                        "min_value_for_upperbound_breached",
                    ):
                        entities_outliers[model_id][field] = outliers_rule.get(field, 0)
                    # fields fit_extra_parameters and apply_extra_parameters are optional
                    elif field in ("fit_extra_parameters", "apply_extra_parameters"):
                        if field in outliers_rule:
                            entities_outliers[model_id][field] = outliers_rule.get(
                                field
                            )
                    else:
                        entities_outliers[model_id][field] = outliers_rule.get(field)
            # log debug
            logger.debug(
                f'after update, entities_outliers="{json.dumps(entities_outliers, indent=4)}"'
            )

            # Finally, update the KVstore record
            try:
                # Update the record
                entity_rules["entities_outliers"] = json.dumps(
                    entities_outliers, indent=4
                )
                collection.data.update(str(key), json.dumps(entity_rules))

                # final
                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": "ML models were successfully updated",
                    "failures_count": 0,
                    "entities_outliers": entities_outliers,
                }

                # log
                logger.info(json.dumps(result_record, indent=4))

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "update ML models",
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(result_record, indent=1)),
                        "ML models were updated successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": result_record, "status": 200}

            except Exception as e:
                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": "Failed to update the KVstore record",
                    "failures_count": 1,
                    "exception": str(e),
                }
                logger.error(json.dumps(result_record, indent=4))
                return {"payload": result_record, "status": 500}

        else:
            msg = "No record found for this object"
            logger.error(msg)
            return {"payload": msg, "status": 404}

    # Add an exclusion period to the ML model
    def post_outliers_manage_model_period_exclusion(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                # required in all cases
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {"payload": "component is required", "status": 500}
                try:
                    model_id = resp_dict["model_id"]
                except Exception as e:
                    return {
                        "payload": "model_id is required, use all or * to match all models from this object",
                        "status": 500,
                    }

                # for cim only
                cim_field = resp_dict.get("cim_field", None)
                # if component is cim, cim_field is required
                if component == "cim" and not cim_field:
                    msg = f'component="{component}" requires cim_field to be defined'
                    logger.error(msg)
                    return {"payload": msg, "status": 500}

                action = resp_dict["action"]
                # action value can be add or delete
                if not action in ("add", "delete", "show"):
                    msg = f'action="{action}" is not valid, it must be either "add", "delete" or "show"'
                    logger.error(msg)
                    return {"payload": msg, "status": 500}

                # required for delete, this can be given as a list of comma separated values, check if it is a list and otherwise turn it as a proper list
                period_exclusion_id = resp_dict.get("period_exclusion_id", None)
                if period_exclusion_id:
                    if not isinstance(period_exclusion_id, list):
                        period_exclusion_id = period_exclusion_id.split(",")

                # required for addition
                earliest = resp_dict.get("earliest", None)
                latest = resp_dict.get("latest", None)

                # if action is delete, the period_exclude_id is required
                if action == "delete":
                    if not period_exclusion_id:
                        msg = f'action="{action}" requires period_exclusion_id to be defined'
                        logger.error(msg)
                        return {"payload": msg, "status": 500}

                # if action is add, earliest and latest are required
                if action == "add":
                    if not earliest or not latest:
                        msg = f'action="{action}" requires earliest and latest to be defined'
                        logger.error(msg)
                        return {"payload": msg, "status": 500}
                    # earliest and latest can be provided as epochtime, or date string in the format %Y-%m-%dT%H:%M, check if provided
                    # as epochtime or date string, if dat string attempt to convrt to epochtime using datetime.datetime.strptime
                    try:
                        earliest = int(earliest)
                    except Exception as e:
                        try:
                            logger.debug(
                                f'trying to parse as datetime, earliest="{earliest}"'
                            )
                            earliest_dt = datetime.strptime(
                                str(earliest), "%Y-%m-%dT%H:%M"
                            )
                            earliest = int(round(float(earliest_dt.timestamp())))

                        except Exception as e:
                            msg = f'action="{action}" requires earliest to be defined as epochtime or date string in the format %Y-%m-%dT%H:%M, parsing as date failed with exception={str(e)}'
                            logger.error(msg)
                            return {"payload": msg, "status": 500}
                    try:
                        latest = int(latest)
                    except Exception as e:
                        try:
                            logger.debug(
                                f'trying to parse as datetime, latest="{latest}"'
                            )
                            latest_dt = datetime.strptime(str(latest), "%Y-%m-%dT%H:%M")
                            latest = int(round(float(latest_dt.timestamp())))
                        except Exception as e:
                            msg = f'action="{action}" requires latest to be defined as epochtime or date string in the format %Y-%m-%dT%H:%M, parsing as date failed with exception={str(e)}'
                            logger.error(msg)
                            return {"payload": msg, "status": 500}

                    # also verify that earliest is not lower than latest, the earliest epochtime cannot be before the latest epochtime
                    if not earliest < latest:
                        msg = f'action="{action}" requires earliest to be before latest, earliest="{earliest}", latest="{latest}"'
                        logger.error(msg)
                        return {"payload": msg, "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint adds a period of exclusion for a given ML model, it requires a POST call with the following options:",
                "resource_desc": "Add or delete an exclusion period to a given ML model",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_add_model_period_exclusion\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall', 'model_id': 'The model identifier', 'action': 'add', 'earliest': '<epoch earliest', 'latest': '<epoch latest>'}",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name",
                        "action": "(required) add, delete, show",
                        "model_id": "(required) The model identifier, use all or * to match all models from this object",
                        "earliest": "(required) The earliest time to be excluded (for add), in epochtime or date string in the format %Y-%m-%dT%H:%M",
                        "latest": "(required) The latest time to be excluded (for add), in epochtime or date string in the format %Y-%m-%dT%H:%M",
                        "period_exclusion_id": "(required) The period exclusion identifier, can be provided as a comma separated list of values (for action delete)",
                        "cim_field": "(required) Required for splk-cim only, the cim_field value",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Data collection
        collection_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection = service.kvstore[collection_name]

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        try:
            # Define the KV query
            query_string = {
                "$and": [
                    {
                        "object_category": f"splk-{component}",
                        "object": object_value,
                    }
                ]
            }

            # for cim, add the cim_field to the query
            if component == "cim":
                query_string["$and"].append({"cim_field": cim_field})

            object_rules_definition = collection.data.query(
                query=json.dumps(query_string)
            )[0]
            key = object_rules_definition.get("_key")

        except Exception as e:
            key = None

        # Render result
        if key:
            logger.debug(object_rules_definition)

            # Load as a dict
            try:
                entities_outliers = json.loads(
                    object_rules_definition.get("entities_outliers")
                )
            except Exception as e:
                msg = f'Failed to load entities_outliers with exception="{str(e)}"'
                logger.error(msg)
                return {"payload": msg, "status": 500}

            # log debug
            logger.debug(
                f'before update, entities_outliers="{json.dumps(entities_outliers, indent=4)}"'
            )

            #
            # Process update
            #

            # load the JSON update as a dict
            logger.debug(
                f'entities_outliers="{json.dumps(entities_outliers, indent=2)}"'
            )

            # boolean to check if model_id exists
            model_id_exists = False

            for entity_model_id in entities_outliers:
                logger.debug(f'model_id="{entity_model_id}"')

                entity_rules = entities_outliers[entity_model_id]
                logger.debug(f'entity_rules="{json.dumps(entity_rules, indent=2)}"')

                # if the model_id does not match, break
                if entity_model_id != model_id and not model_id in ("all", "*"):
                    continue
                else:
                    model_id_exists = True

                # log debug
                logger.debug(f'Handling model_id="{entity_model_id}"')

                # Get the current period_exclusions record (list)
                period_exclusions = entity_rules.get("period_exclusions", [])

                # if action is add
                if action == "add":
                    # add our object to the list

                    # period_exclusion_id, generate the sha256 hash of earliest:latest, use this to detect if the period was excluded already
                    period_exclusion_id = hashlib.sha256(
                        f"{earliest}:{latest}".encode()
                    ).hexdigest()

                    # boolean
                    period_exclusion_id_exists = False

                    if len(period_exclusions) > 0:
                        for period_exclusion in period_exclusions:
                            if (
                                period_exclusion["period_exclusion_id"]
                                == period_exclusion_id
                            ):
                                period_exclusion_id_exists = True
                                logger.info(
                                    f'period_exclusion_id="{period_exclusion_id}" already exists, skipping'
                                )
                                continue

                    if not period_exclusion_id_exists:
                        # add the new period
                        period_exclusions.append(
                            {
                                "period_exclusion_id": period_exclusion_id,
                                "earliest": earliest,
                                "earliest_human": time.strftime(
                                    "%c", time.localtime(float(earliest))
                                ),
                                "latest": latest,
                                "latest_human": time.strftime(
                                    "%c", time.localtime(float(latest))
                                ),
                                "ctime": str(round(time.time(), 0)),
                            }
                        )

                # if action is delete, verify the period_exclusion_id exists in the list, if so delete it
                elif action == "delete":
                    # delete the period_exclusion_id from the list
                    for item in period_exclusion_id:
                        logger.debug(f'checking item: "{item}"')
                        # Create a list of items to remove to avoid modifying during iteration
                        items_to_remove = []
                        for period_exclusion in period_exclusions:
                            if period_exclusion["period_exclusion_id"] == item.strip():
                                logger.debug(f'deleting item: "{item}"')
                                items_to_remove.append(period_exclusion)

                        # Remove the collected items
                        for item_to_remove in items_to_remove:
                            period_exclusions.remove(item_to_remove)

                # if action is show, return the period_exclusions list
                elif action == "show":
                    response_list = []
                    # loop through the period_exclusions, add each as a dict to the response_list (model_id, period_exclusion_id, earliest, latest))
                    for period_exclusion in period_exclusions:
                        response_list.append(
                            {
                                "model_id": model_id,
                                "period_exclusion_id": period_exclusion.get(
                                    "period_exclusion_id"
                                ),
                                "earliest": period_exclusion.get("earliest"),
                                "earliest_human": period_exclusion.get(
                                    "earliest_human"
                                ),
                                "latest": period_exclusion.get("latest"),
                                "latest_human": period_exclusion.get("latest_human"),
                                "ctime": period_exclusion.get("ctime"),
                                "ctime_human": time.strftime(
                                    "%c",
                                    time.localtime(
                                        float(period_exclusion.get("ctime"))
                                    ),
                                ),
                            }
                        )
                    return {
                        "payload": response_list,
                        "status": 200,
                    }

                # update the entity_rule record
                entity_rules["period_exclusions"] = period_exclusions

                # update the entities_outliers dict
                entities_outliers[entity_model_id] = entity_rules

                # log
                logger.debug(
                    f'post update entities_outliers="{json.dumps(entities_outliers, indent=2)}"'
                )

            # if model_id does not exist, return a 500 payload
            if not model_id_exists and not model_id in ("all", "*"):
                error_msg = (
                    f'The model_id="{model_id}" does not exist in the outliers rules'
                )
                logger.error(error_msg)
                return {"payload": error_msg, "status": 500}

            # log debug
            logger.debug(
                f'after update, entities_outliers="{json.dumps(entities_outliers, indent=4)}"'
            )

            # update the object record
            object_rules_definition["entities_outliers"] = json.dumps(
                entities_outliers, indent=2
            )

            # Finally, update the KVstore record
            try:
                # Update the record
                collection.data.update(str(key), json.dumps(object_rules_definition))

                # final
                if action == "add":
                    msg = "Exclusion period for ML model was successfully added"
                    msg_audit_title = "add ML model exclusion period"
                elif action == "delete":
                    msg = "Exclusion period for ML model was successfully deleted"
                    msg_audit_title = "delete ML model exclusion period"

                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": msg,
                    "failures_count": 0,
                    "entities_outliers": entities_outliers,
                }

                # log
                logger.info(json.dumps(result_record, indent=4))

                # audit

                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        msg_audit_title,
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(result_record, indent=1)),
                        "ML models were updated successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": result_record, "status": 200}

            except Exception as e:
                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": "Failed to update the KVstore record",
                    "failures_count": 1,
                    "exception": str(e),
                }
                logger.error(json.dumps(result_record, indent=4))
                return {"payload": result_record, "status": 500}

        else:
            msg = "No record found for this object"
            logger.error(msg)
            return {"payload": msg, "status": 404}

    # update one or more ML models, spl-cim has multiple levels of objects (object / entity / cim_field)
    def post_outliers_cim_update_models(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                try:
                    outliers_rules = resp_dict["outliers_rules"]
                except Exception as e:
                    return {"payload": "outliers_rules is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint updates existing ML models rules, it requires a POST call with the following options:",
                "resource_desc": "Update a Machine Learning outliers model",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_cim_update_models\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall','outliers_rules':'<redacted_json_dict>'}",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "object": "(required) entity object name",
                        "outliers_rules": "(required) The JSON array object containing the outliers rules",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Data collection
        collection_name = f"kv_trackme_cim_outliers_entity_rules_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Loop through the dict
        logger.debug(
            f'submitted outliers_rules="{json.dumps(outliers_rules, indent=2)}"'
        )

        #
        # loop in changes
        #

        for p in outliers_rules:
            model_id = p["model_id"]

            # Get the current KVsotre record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            try:
                # Define the KV query
                query_string = {
                    "$and": [
                        {
                            "object_category": "splk-cim",
                            "object": object_value,
                            "entity": p["entity"],
                            "cim_field": p["cim_field"],
                        }
                    ]
                }

                entity_rules = collection.data.query(query=json.dumps(query_string))[0]
                key = entity_rules.get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                logger.debug(entity_rules)

                # Load as a dict
                try:
                    entities_outliers = json.loads(
                        entity_rules.get("entities_outliers")
                    )
                except Exception as e:
                    msg = f'Failed to load entities_outliers with exception="{str(e)}"'
                    logger.error(msg)
                    return {"payload": msg, "status": 500}

                # log debug
                logger.debug(
                    f'before update, entities_outliers="{json.dumps(entities_outliers, indent=4)}"'
                )

                #
                # Process update
                #

                # load the JSON update as a dict
                logger.debug(f'outliers_rules="{json.dumps(outliers_rules, indent=2)}"')

                # update that model

                # log debug
                logger.debug(f'Handling model_id="{p["model_id"]}"')

                # Update the main dict
                fields_list = [
                    "kpi_metric",
                    "kpi_span",
                    "method_calculation",
                    "period_calculation",
                    "period_calculation_latest",
                    "time_factor",
                    "density_lowerthreshold",
                    "density_upperthreshold",
                    "alert_lower_breached",
                    "alert_upper_breached",
                    "auto_correct",
                    "perc_min_lowerbound_deviation",
                    "perc_min_upperbound_deviation",
                    "min_value_for_lowerbound_breached",
                    "min_value_for_upperbound_breached",
                    "static_lower_threshold",
                    "static_upper_threshold",
                    "algorithm",
                    "boundaries_extraction_macro",
                    "fit_extra_parameters",
                    "apply_extra_parameters",
                    "is_disabled",
                ]
                for field in fields_list:
                    if field in ("time_factor"):
                        entities_outliers[model_id][field] = urllib.parse.unquote(
                            p[field]
                        )
                    elif field in (
                        "min_value_for_lowerbound_breached",
                        "min_value_for_upperbound_breached",
                    ):
                        entities_outliers[model_id][field] = p.get(field, 0)

                    # fields fit_extra_parameters and apply_extra_parameters are optional, if not provided, do not update
                    elif field in ("fit_extra_parameters", "apply_extra_parameters"):
                        if p.get(field):
                            entities_outliers[model_id][field] = p[field]
                    else:
                        entities_outliers[model_id][field] = p[field]

                # log debug
                logger.debug(
                    f'after update, entities_outliers="{json.dumps(entities_outliers, indent=4)}"'
                )

                # Finally, update the KVstore record
                try:
                    # Update the record
                    entity_rules["entities_outliers"] = json.dumps(
                        entities_outliers, indent=4
                    )
                    collection.data.update(str(key), json.dumps(entity_rules))

                    # increment counter
                    processed_count += 1
                    succcess_count += 1
                    failures_count += 0

                    # append for summary
                    result = {
                        "object": object_value,
                        "entity": p["entity"],
                        "cim_field": p["cim_field"],
                        "action": "update",
                        "result": "success",
                        "message": f'tenant_id="{tenant_id}", The entity was successfully updated',
                    }
                    records.append(result)

                    # log
                    logger.info(json.dumps(records, indent=4))

                    # audit
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "update ML models",
                            str(object_value),
                            "splk-dsm",
                            str(json.dumps(records, indent=1)),
                            "ML models were updated successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                except Exception as e:
                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    msg = f'tenant_id="{tenant_id}", failed to update the entity, object="{object_value}", exception="{str(e)}"'
                    result = {
                        "object": object_value,
                        "entity": p["entity"],
                        "cim_field": p["cim_field"],
                        "action": "update",
                        "result": "failure",
                        "exception": msg,
                    }
                    records.append(result)
                    logger.error(msg)

                    # audit
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "failure",
                            "update ML models",
                            str(object_value),
                            "splk-dsm",
                            str(json.dumps(records, indent=1)),
                            msg,
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

            else:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                msg = f'tenant_id="{tenant_id}", failed to update the entity, object="{object_value}", exception="ressource not found"'
                result = {
                    "object": object_value,
                    "entity": p["entity"],
                    "cim_field": p["cim_field"],
                    "action": "update",
                    "result": "failure",
                    "exception": f'tenant_id="{tenant_id}", failed to update the entity, object="{object_value}", exception="ressource not found"',
                }
                records.append(result)
                logger.error(msg)

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "update ML models",
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(records, indent=1)),
                        msg,
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            logger.info(
                f'ML models bulk edit was successfull, results="{json.dumps(req_summary, indent=1)}"'
            )
            return {"payload": req_summary, "status": 200}

        else:
            logger.error(
                f'ML models bulk edit has failed, results="{json.dumps(req_summary, indent=1)}"'
            )
            return {"payload": req_summary, "status": 500}

    # reset ML models
    def post_outliers_reset_models(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {"payload": "component is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint resets ML models rules, it requires a POST call with the following options:",
                "resource_desc": "Reset all ML outliers models for a given entity",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_reset_models\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "tenant identifier",
                        "component": "The component category",
                        "object": "entity name",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # entity rules collection
        collection_entity_rules_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_entity_rules = service.kvstore[collection_entity_rules_name]

        # data rules collection
        collection_entity_data_name = (
            f"kv_trackme_{component}_outliers_entity_data_tenant_{tenant_id}"
        )
        collection_entity_data = service.kvstore[collection_entity_data_name]

        # Define the KV query
        query_string = {
            "$and": [
                {
                    "object_category": f"splk-{component}",
                    "object": object_value,
                }
            ]
        }

        # get the entity_rules record
        try:
            kvrecord_entity_rules = collection_entity_rules.data.query(
                query=json.dumps(query_string)
            )[0]
            kvrecord_entity_rules_key = kvrecord_entity_rules.get("_key")
        except Exception as e:
            kvrecord_entity_rules_key = None

        # get the entity_data record
        try:
            kvrecord_entity_data = collection_entity_data.data.query(
                query=json.dumps(query_string)
            )[0]
            kvrecord_entity_data_key = kvrecord_entity_data.get("_key")
        except Exception as e:
            kvrecord_entity_data_key = None

        #
        # proceed
        #

        # first, reset the entity_data record, if any
        kvrecord_entity_data_deleted = False
        if kvrecord_entity_data_key:
            try:
                collection_entity_data.data.delete(
                    json.dumps({"_key": kvrecord_entity_data_key})
                )
                logger.info(
                    f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", collection="{collection_entity_rules_name}", deleted key="{kvrecord_entity_data_key}"'
                )
                kvrecord_entity_data_deleted = True
            except Exception as e:
                logger.error(
                    f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", collection="{collection_entity_rules_name}", failed to deleted key="{kvrecord_entity_data_key}" with exception="{str(e)}"'
                )

        # secondly, purge and re-generate ML models
        kvrecord_entity_rules_deleted = False
        if kvrecord_entity_rules_key:
            #
            # Process reset
            #

            try:
                collection_entity_rules.data.delete(
                    json.dumps({"_key": kvrecord_entity_rules_key})
                )
                logger.info(
                    f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", collection="{collection_entity_data_name}", deleted key="{kvrecord_entity_rules_key}"'
                )
                kvrecord_entity_rules_deleted = True
            except Exception as e:
                logger.error(
                    f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", collection="{collection_entity_data_name}", failed to deleted key="{kvrecord_entity_rules_key}" with exception="{str(e)}"'
                )
                kvrecord_entity_rules_deleted = False

                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": "Failed to delete the KVstore record this entity, cannot proceed to reset",
                    "failures_count": 1,
                    "exception": str(e),
                }
                logger.error(json.dumps(result_record, indent=4))
                return {"payload": result_record, "status": 500}

        else:
            logger.info(
                f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", no KVstore record was found to be deleted'
            )

        # Define the SPL query
        kwargs_search = {
            "app": "trackme",
            "earliest_time": "-5m",
            "latest_time": "now",
            "output_mode": "json",
            "count": 0,
        }

        # Search query reset
        searchquery_reset = remove_leading_spaces(
            f"""\
            | makeresults | head 1
            | eval tenant_id="{tenant_id}", object_category="splk-{component}", object="{object_value}", key="{kvrecord_entity_rules_key}"
            | trackmesplkoutlierssetrules tenant_id="{tenant_id}" component="{component}"
            """
        )

        # log debug
        logger.debug(
            f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", searchquery_reset="{searchquery_reset}"'
        )

        # run search
        reset_results = []
        try:
            # spawn the search and get the results
            reader = run_splunk_search(
                service,
                searchquery_reset,
                kwargs_search,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    reset_results.append(item)

            # final
            result_record = {
                "tenant_id": tenant_id,
                "object_category": f"splk-{component}",
                "object": object_value,
                "results": "ML models were successfully reset",
                "failures_count": 0,
                "reset_results": reset_results,
                "searchquery_reset": searchquery_reset,
                "entity_rules_deleted": kvrecord_entity_rules_deleted,
                "entity_data_deleted": kvrecord_entity_data_deleted,
            }

            # log
            logger.info(json.dumps(result_record, indent=4))

            # audit
            trackme_audit_event(
                request_info.system_authtoken,
                request_info.server_rest_uri,
                tenant_id,
                request_info.user,
                "success",
                "reset ML models",
                str(object_value),
                "splk-dsm",
                str(json.dumps(result_record, indent=1)),
                "ML models were reset successfully",
                str(update_comment),
            )

            return {"payload": result_record, "status": 200}

        except Exception as e:
            # permanent failure
            result_record = {
                "tenant_id": tenant_id,
                "object_category": f"splk-{component}",
                "object": object_value,
                "results": "Failed to reset ML models for this entity",
                "failures_count": 1,
                "entity_rules_deleted": kvrecord_entity_rules_deleted,
                "entity_data_deleted": kvrecord_entity_data_deleted,
                "exception": str(e),
            }
            logger.error(json.dumps(result_record, indent=4))
            return {"payload": result_record, "status": 500}

    # reset and train ML models for cim
    def post_outliers_cim_reset_models(self, request_info, **kwargs):
        describe = False
        tenant_id = None
        object_value = None
        component = "cim"

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint resets ML models rules, it requires a POST call with the following options:",
                "resource_desc": "Reset all ML outliers models for a given entity",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_cim_reset_models\" body=\"{'tenant_id':'cim-demo','object':'auth001'}\"",
                "options": [
                    {
                        "tenant_id": "tenant identifier",
                        "object": "entity name",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # entity rules collection
        collection_entity_rules_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_entity_rules = service.kvstore[collection_entity_rules_name]

        # data rules collection
        collection_entity_data_name = (
            f"kv_trackme_{component}_outliers_entity_data_tenant_{tenant_id}"
        )
        collection_entity_data = service.kvstore[collection_entity_data_name]

        # Define the KV query
        query_string = {
            "$and": [
                {
                    "object_category": f"splk-{component}",
                    "object": object_value,
                }
            ]
        }

        # get the entity_rules record
        try:
            kvrecord_entity_rules = collection_entity_rules.data.query(
                query=json.dumps(query_string)
            )[0]
            kvrecord_entity_rules_key = kvrecord_entity_rules.get("_key")
        except Exception as e:
            kvrecord_entity_rules_key = None

        # get the entity_data record
        try:
            kvrecord_entity_data = collection_entity_data.data.query(
                query=json.dumps(query_string)
            )[0]
            kvrecord_entity_data_key = kvrecord_entity_data.get("_key")
        except Exception as e:
            kvrecord_entity_data_key = None

        #
        # proceed
        #

        # first, reset the entity_data record, if any
        kvrecord_entity_data_deleted = False
        if kvrecord_entity_data_key:
            try:
                collection_entity_data.data.delete(
                    json.dumps({"_key": kvrecord_entity_data_key})
                )
                logger.info(
                    f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", collection="{collection_entity_rules_name}", deleted key="{kvrecord_entity_data_key}"'
                )
                kvrecord_entity_data_deleted = True
            except Exception as e:
                logger.error(
                    f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", collection="{collection_entity_rules_name}", failed to deleted key="{kvrecord_entity_data_key}" with exception="{str(e)}"'
                )

        # secondly, purge and re-generate ML models
        kvrecord_entity_rules_deleted = False
        if kvrecord_entity_rules_key:
            #
            # Process reset
            #

            try:
                collection_entity_rules.data.delete(
                    json.dumps({"_key": kvrecord_entity_rules_key})
                )
                logger.info(
                    f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", collection="{collection_entity_data_name}", deleted key="{kvrecord_entity_rules_key}"'
                )
                kvrecord_entity_rules_deleted = True
            except Exception as e:
                logger.error(
                    f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", collection="{collection_entity_data_name}", failed to deleted key="{kvrecord_entity_rules_key}" with exception="{str(e)}"'
                )
                kvrecord_entity_rules_deleted = False

                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": "Failed to delete the KVstore record this entity, cannot proceed to reset",
                    "failures_count": 1,
                    "exception": str(e),
                }
                logger.error(json.dumps(result_record, indent=4))
                return {"payload": result_record, "status": 500}

        else:
            logger.info(
                f'tenant_id="{tenant_id}", component="splk-cim", object="{object_value}", no KVstore record was found to be deleted'
            )

        # Define the SPL query
        kwargs_search = {
            "app": "trackme",
            "earliest_time": "-5m",
            "latest_time": "now",
            "output_mode": "json",
            "count": 0,
        }

        #
        # reset models
        #

        # Search query reset
        searchquery_reset = f'| trackmesplkoutlierscimsetrules tenant_id="{tenant_id}" object="{object_value}'

        # log debug
        logger.debug(
            f'tenant_id="{tenant_id}", component="splk-cim", object="{object_value}", searchquery_reset="{searchquery_reset}"'
        )

        reset_results = []
        try:
            # spawn the search and get the results
            reader = run_splunk_search(
                service,
                searchquery_reset,
                kwargs_search,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    reset_results.append(item)

            # final
            reset_result_record = {
                "tenant_id": tenant_id,
                "object_category": "splk-cim",
                "object": object_value,
                "results": "ML models were successfully reset",
                "failures_count": 0,
                "reset_results": reset_results,
                "searchquery_reset": searchquery_reset,
                "entity_rules_deleted": kvrecord_entity_rules_deleted,
                "entity_data_deleted": kvrecord_entity_data_deleted,
            }

            # log
            logger.info(json.dumps(reset_result_record, indent=4))

            # audit
            try:
                trackme_audit_event(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    request_info.user,
                    "success",
                    "reset ML models",
                    str(object_value),
                    "splk-dsm",
                    str(json.dumps(reset_result_record, indent=1)),
                    "ML models were reset successfully",
                    str(update_comment),
                )
            except Exception as e:
                logger.error(
                    f'failed to generate an audit event with exception="{str(e)}"'
                )

        except Exception as e:
            reset_result_record = {
                "tenant_id": tenant_id,
                "object_category": "splk-cim",
                "object": object_value,
                "results": "Failed to reset ML models for this entity",
                "failures_count": 1,
                "entity_rules_deleted": kvrecord_entity_rules_deleted,
                "entity_data_deleted": kvrecord_entity_data_deleted,
                "exception": str(e),
            }
            logger.error(json.dumps(reset_result_record, indent=4))

            return {"payload": reset_result_record, "status": 500}

        #
        # Train models
        #

        # Search query train
        searchquery_train = f'| trackmesplkoutlierscimtrain tenant_id="{tenant_id}" component="cim" object="{object_value}" cim_field=*'

        # log debug
        logger.debug(
            f'tenant_id="{tenant_id}", component="cim", object="{object_value}", searchquery_train="{searchquery_train}"'
        )

        train_results = []

        try:
            # spawn the search and get the results
            reader = run_splunk_search(
                service,
                searchquery_train,
                kwargs_search,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    train_results.append(item)

            # final
            train_results_record = {
                "tenant_id": tenant_id,
                "object_category": "splk-cim",
                "object": object_value,
                "results": "ML models were successfully trained",
                "failures_count": 0,
                "train_results": train_results,
                "searchquery_train": searchquery_train,
            }

            # log
            logger.info(json.dumps(train_results_record, indent=4))

            # audit
            try:
                trackme_audit_event(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    request_info.user,
                    "success",
                    "train ML models",
                    str(object_value),
                    "splk-dsm",
                    str(json.dumps(train_results_record, indent=1)),
                    "ML models were trained successfully",
                    str(update_comment),
                )
            except Exception as e:
                logger.error(
                    f'failed to generate an audit event with exception="{str(e)}"'
                )

        except Exception as e:
            train_results_record = {
                "tenant_id": tenant_id,
                "object_category": "splk-cim",
                "object": object_value,
                "results": "Failed to train ML models for this entity",
                "failures_count": 1,
                "exception": str(e),
            }
            logger.error(json.dumps(train_results_record, indent=4))
            return {"payload": train_results_record, "status": 500}

        # return if successful
        return {
            "payload": {
                "tenant_id": tenant_id,
                "object": object_value,
                "result": "success",
                "failures_count": 0,
                "results": "ML models were successfully reset and trained for this entity",
                "reset_results": reset_result_record,
                "train_results": train_results_record,
            },
            "status": 200,
        }

    # train ML models
    def post_outliers_train_models(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {"payload": "component is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint trains ML models rules, it requires a POST call with the following options:",
                "resource_desc": "Train all ML outliers models for a given entity",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_train_models\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Data collection
        collection_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection = service.kvstore[collection_name]

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        try:
            # Define the KV query
            query_string = {
                "$and": [
                    {
                        "object_category": f"splk-{component}",
                        "object": object_value,
                    }
                ]
            }

            entity_rules = collection.data.query(query=json.dumps(query_string))[0]
            key = entity_rules.get("_key")

        except Exception as e:
            key = None

        # Render result
        if key:
            # Define the SPL query
            kwargs_search = {
                "app": "trackme",
                "earliest_time": "-5m",
                "latest_time": "now",
                "output_mode": "json",
                "count": 0,
            }
            searchquery = f'| trackmesplkoutlierstrain tenant_id="{tenant_id}" component="{component}" object="{object_value}"'
            logger.debug(
                f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", searchquery="{searchquery}"'
            )

            query_results = []
            try:
                # spawn the search and get the results
                reader = run_splunk_search(
                    service,
                    searchquery,
                    kwargs_search,
                    24,
                    5,
                )

                for item in reader:
                    if isinstance(item, dict):
                        query_results.append(item)

                # final
                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": "ML models were successfully trained",
                    "failures_count": 0,
                    "query_results": query_results,
                }

                # log
                logger.info(json.dumps(result_record, indent=4))

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "train ML models",
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(result_record, indent=1)),
                        "ML models were train successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": result_record, "status": 200}

            except Exception as e:
                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": "Failed to train ML models for this entity",
                    "failures_count": 1,
                    "exception": str(e),
                }
                logger.error(json.dumps(result_record, indent=4))
                return {"payload": result_record, "status": 500}

    # train ML models
    def post_outliers_cim_train_models(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}
                component = "cim"  # fixed component
                try:
                    cim_field = resp_dict["cim_field"]
                except Exception as e:
                    cim_field = "*"  # default to all fields

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint trains ML models rules, it requires a POST call with the following options:",
                "resource_desc": "Train all ML outliers models for a given entity",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_cim_train_models\" body=\"{'tenant_id':'mytenant','object':'netscreen:netscreen:firewall','cim_field': '*'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "object": "(required) entity name",
                        "cim_field": "(optional) cim field name",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Data collection
        collection_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection = service.kvstore[collection_name]

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        try:
            # Define the KV query
            query_string = {
                "$and": [
                    {
                        "object_category": f"splk-{component}",
                        "object": object_value,
                    }
                ]
            }

            entity_rules = collection.data.query(query=json.dumps(query_string))[0]
            key = entity_rules.get("_key")

        except Exception as e:
            key = None

        # Render result
        if key:
            # Define the SPL query
            kwargs_search = {
                "app": "trackme",
                "earliest_time": "-5m",
                "latest_time": "now",
                "output_mode": "json",
                "count": 0,
            }
            searchquery = f'| trackmesplkoutlierscimtrain tenant_id="{tenant_id}" component="{component}" object="{object_value}" cim_field="{cim_field}"'
            logger.debug(
                f'tenant_id="{tenant_id}", component="{component}", object="{object_value}", searchquery="{searchquery}"'
            )

            query_results = []
            try:
                # spawn the search and get the results
                reader = run_splunk_search(
                    service,
                    searchquery,
                    kwargs_search,
                    24,
                    5,
                )

                for item in reader:
                    if isinstance(item, dict):
                        query_results.append(item)

                # final
                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": "ML models were successfully trained",
                    "failures_count": 0,
                    "query_results": query_results,
                }

                # log
                logger.info(json.dumps(result_record, indent=4))

                # audit
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "train ML models",
                        str(object_value),
                        "splk-dsm",
                        str(json.dumps(result_record, indent=1)),
                        "ML models were train successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": result_record, "status": 200}

            except Exception as e:
                result_record = {
                    "tenant_id": tenant_id,
                    "object_category": f"splk-{component}",
                    "object": object_value,
                    "results": "Failed to train ML models for this entity",
                    "failures_count": 1,
                    "exception": str(e),
                }
                logger.error(json.dumps(result_record, indent=4))
                return {"payload": result_record, "status": 500}

    # train entity ML model
    def post_outliers_train_entity_model(self, request_info, **kwargs):
        describe = False

        logger.debug(f"Starting function post_outliers_train_entity_model")

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "tenant_id is required",
                        },
                        "status": 500,
                    }

                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "object is required",
                        },
                        "status": 500,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "component is required",
                        },
                        "status": 500,
                    }

                try:
                    mode = resp_dict["mode"]
                    # valid options are live, simulation
                    if mode not in ("live", "simulation"):
                        return {
                            "payload": {
                                "response": "mode is invalid, valid options are live, simulation",
                            },
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "mode is required",
                        },
                        "status": 500,
                    }

                try:
                    entity_outlier = resp_dict["entity_outlier"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "entity_outlier is required",
                        },
                        "status": 500,
                    }

                try:
                    entity_outlier_dict = resp_dict["entity_outlier_dict"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "entity_outlier_dict is required",
                        },
                        "status": 500,
                    }

                model_json_def = resp_dict.get("model_json_def", {})
                # if mode is simulation, model_json_def is required
                if mode == "simulation" and not model_json_def:
                    return {
                        "payload": {
                            "response": "model_json_def is required for simulation mode",
                        },
                        "status": 500,
                    }

        else:
            describe = True

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint trains ML for a given entity, it requires a POST call with the following options:",
                "resource_desc": "Programmatically train ML models for a given entity, this endpoints is designed to be called by the backend for the purpose of training ML models",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_train_entity_model\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name",
                        "mode": "(required) train mode, valid options: live, simulation",
                        "entity_outlier": "(required) entity outlier",
                        "entity_outlier_dict": "(required) entity outlier dict",
                        "model_json_def": "(required for simulation only) model json definition",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get request info and set logger.level
        reqinfo = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )

        # trackme_idx_for_tenant
        tenant_indexes = trackme_idx_for_tenant(
            request_info.system_authtoken, request_info.server_rest_uri, tenant_id
        )

        # set the tenant_trackme_metric_idx
        tenant_trackme_metric_idx = tenant_indexes.get(
            "trackme_metric_idx", "trackme_metrics"
        )

        # Outliers rules storage collection
        collection_rules_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_rule = service.kvstore[collection_rules_name]

        # Vtenants storage collection
        vtenants_collection_name = "kv_trackme_virtual_tenants"
        vtenants_collection = service.kvstore[vtenants_collection_name]

        #
        # First, get the full vtenant definition
        #

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        # get
        try:
            vtenant_record = vtenants_collection.data.query(
                query=json.dumps(query_string)
            )
        except Exception as e:
            error_msg = (
                f'tenant_id="{tenant_id}" could not be found in the vtenants collection'
            )
            logger.error(error_msg)
            return {
                "payload": {
                    "response": error_msg,
                },
                "status": 500,
            }

        #
        # Get the Outliers rules
        #

        # Define the KV query
        query_string_filter = {
            "object_category": f"splk-{component}",
            "object": object_value,
        }

        query_string = {"$and": [query_string_filter]}

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        key = None

        try:
            records_outliers_rules = collection_rule.data.query(
                query=json.dumps(query_string)
            )
            record_outliers_rules = records_outliers_rules[0]
            key = record_outliers_rules.get("_key")

        except Exception as e:
            key = None

        # if no records
        if not key:
            msg = f'tenant_id="{tenant_id}", component="{component}", object="{object_value}" outliers rules record cannot be found or are not yet available for this entity.'
            logger.error(msg)
            return {
                "payload": {"response": msg},
                "status": 500,
            }

        # log debug
        logger.debug(f'record_outliers_rules="{record_outliers_rules}"')

        # Get the JSON outliers rules object
        entities_outliers = record_outliers_rules.get("entities_outliers")

        # Load as a dict
        try:
            entities_outliers = json.loads(
                record_outliers_rules.get("entities_outliers")
            )
        except Exception as e:
            msg = f'Failed to load entities_outliers with exception="{str(e)}"'

        # log debug
        logger.debug(f'entities_outliers="{entities_outliers}"')

        # Load the general enablement
        try:
            outliers_is_disabled = int(record_outliers_rules.get("is_disabled"))
            logger.debug(f'is_disabled="{outliers_is_disabled}"')

        except Exception as e:
            msg = f'Failed to extract one or more expected settings from the entity, is this record corrupted? Exception="{str(e)}"'
            logger.error(msg)
            return {
                "payload": {"response": msg},
                "status": 500,
            }

        # proceed
        if outliers_is_disabled == 1:
            msg = f"Outliers detection are disabled at the global level for this entity, nothing to do."
            logger.info(msg)
            return {
                "payload": msg,
                "status": 200,
            }

        else:

            logger.debug(
                f"calling function train_mlmodel with arguments: {tenant_id}, {component}, {object_value}, {key}, {tenant_trackme_metric_idx}, {mode}, {entities_outliers}, {entity_outlier}, {entity_outlier_dict}, {model_json_def}"
            )

            entities_outliers, entity_outlier, entity_outlier_dict = train_mlmodel(
                service,
                request_info.server_rest_uri,
                request_info.system_authtoken,
                request_info.user,
                tenant_id,
                component,
                object_value,
                key,
                tenant_trackme_metric_idx,
                mode,
                entities_outliers,
                entity_outlier,
                entity_outlier_dict,
                model_json_def,
            )

            # temp
            return {
                "payload": {
                    "entities_outliers": entities_outliers,
                    "entity_outlier": entity_outlier,
                    "entity_outlier_dict": entity_outlier_dict,
                },
                "status": 200,
            }

    # train entity ML model
    def post_outliers_cim_train_entity_model(self, request_info, **kwargs):
        describe = False

        logger.debug(f"Starting function post_outliers_cim_train_entity_model")

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "tenant_id is required",
                        },
                        "status": 500,
                    }

                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "object is required",
                        },
                        "status": 500,
                    }

                try:
                    cim_field = resp_dict["cim_field"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "cim_field is required",
                        },
                        "status": 500,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "component is required",
                        },
                        "status": 500,
                    }

                try:
                    mode = resp_dict["mode"]
                    # valid options are live, simulation
                    if mode not in ("live", "simulation"):
                        return {
                            "payload": {
                                "response": "mode is invalid, valid options are live, simulation",
                            },
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "mode is required",
                        },
                        "status": 500,
                    }

                try:
                    entity_outlier = resp_dict["entity_outlier"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "entity_outlier is required",
                        },
                        "status": 500,
                    }

                try:
                    entity_outlier_dict = resp_dict["entity_outlier_dict"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "entity_outlier_dict is required",
                        },
                        "status": 500,
                    }

                model_json_def = resp_dict.get("model_json_def", {})
                # if mode is simulation, model_json_def is required
                if mode == "simulation" and not model_json_def:
                    return {
                        "payload": {
                            "response": "model_json_def is required for simulation mode",
                        },
                        "status": 500,
                    }

        else:
            describe = True

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint trains ML for a given entity, it requires a POST call with the following options:",
                "resource_desc": "Programmatically train ML models for a given entity, this endpoints is designed to be called by the backend for the purpose of training ML models",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_train_entity_model\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category",
                        "object": "(required) entity name",
                        "cim_field": "(required) cim field",
                        "mode": "(required) train mode, valid options: live, simulation",
                        "entity_outlier": "(required) entity outlier",
                        "entity_outlier_dict": "(required) entity outlier dict",
                        "model_json_def": "(required for simulation only) model json definition",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get request info and set logger.level
        reqinfo = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )

        # trackme_idx_for_tenant
        tenant_indexes = trackme_idx_for_tenant(
            request_info.system_authtoken, request_info.server_rest_uri, tenant_id
        )

        # set the tenant_trackme_metric_idx
        tenant_trackme_metric_idx = tenant_indexes.get(
            "trackme_metric_idx", "trackme_metrics"
        )

        # Outliers rules storage collection
        collection_rules_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_rule = service.kvstore[collection_rules_name]

        # Vtenants storage collection
        vtenants_collection_name = "kv_trackme_virtual_tenants"
        vtenants_collection = service.kvstore[vtenants_collection_name]

        #
        # First, get the full vtenant definition
        #

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        # get
        try:
            vtenant_record = vtenants_collection.data.query(
                query=json.dumps(query_string)
            )
        except Exception as e:
            error_msg = (
                f'tenant_id="{tenant_id}" could not be found in the vtenants collection'
            )
            logger.error(error_msg)
            return {
                "payload": {
                    "response": error_msg,
                },
                "status": 500,
            }

        #
        # Get the Outliers rules
        #

        # Define the KV query
        query_string_filter = {
            "object_category": f"splk-{component}",
            "object": object_value,
            "cim_field": cim_field,
        }

        query_string = {"$and": [query_string_filter]}

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        key = None

        try:
            records_outliers_rules = collection_rule.data.query(
                query=json.dumps(query_string)
            )
            record_outliers_rules = records_outliers_rules[0]
            key = record_outliers_rules.get("_key")

        except Exception as e:
            key = None

        # if no records
        if not key:
            msg = f'tenant_id="{tenant_id}", component="{component}", object="{object_value}" outliers rules record cannot be found or are not yet available for this entity.'
            logger.error(msg)
            return {
                "payload": {"response": msg},
                "status": 500,
            }

        # log debug
        logger.debug(f'record_outliers_rules="{record_outliers_rules}"')

        # Get the JSON outliers rules object
        entities_outliers = record_outliers_rules.get("entities_outliers")

        # Load as a dict
        try:
            entities_outliers = json.loads(
                record_outliers_rules.get("entities_outliers")
            )
        except Exception as e:
            msg = f'Failed to load entities_outliers with exception="{str(e)}"'

        # log debug
        logger.debug(f'entities_outliers="{entities_outliers}"')

        # Load the general enablement
        try:
            outliers_is_disabled = int(record_outliers_rules.get("is_disabled"))
            logger.debug(f'is_disabled="{outliers_is_disabled}"')

        except Exception as e:
            msg = f'Failed to extract one or more expected settings from the entity, is this record corrupted? Exception="{str(e)}"'
            logger.error(msg)
            return {
                "payload": {"response": msg},
                "status": 500,
            }

        # proceed
        if outliers_is_disabled == 1:
            msg = f"Outliers detection are disabled at the global level for this entity, nothing to do."
            logger.info(msg)
            return {
                "payload": msg,
                "status": 200,
            }

        else:

            logger.debug(
                f"calling function train_cim_mlmodel with arguments: {tenant_id}, {component}, {object_value}, {key}, {cim_field}, {tenant_trackme_metric_idx}, {mode}, {entities_outliers}, {entity_outlier}, {entity_outlier_dict}, {model_json_def}"
            )

            entities_outliers, entity_outlier, entity_outlier_dict = train_cim_mlmodel(
                service,
                request_info.server_rest_uri,
                request_info.system_authtoken,
                request_info.user,
                tenant_id,
                component,
                object_value,
                key,
                cim_field,
                tenant_trackme_metric_idx,
                mode,
                entities_outliers,
                entity_outlier,
                entity_outlier_dict,
                model_json_def,
            )

            # temp
            return {
                "payload": {
                    "entities_outliers": entities_outliers,
                    "entity_outlier": entity_outlier,
                    "entity_outlier_dict": entity_outlier_dict,
                },
                "status": 200,
            }

    # Force mlmonitor execution for a given object
    def post_outliers_mlmonitor_models(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {"payload": "component is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint runs the execution of the ML monitor backend process, it requires a POST call with the following options:",
                "resource_desc": "Runs Machine Learning Outliers monitor process for a given entity",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_mlmonitor_models\" body=\"{'tenant_id':'mytenant','component':'dsm','object':'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) the component, valid options for this endpoint are: dsm|dhm|flx|fqm|wlk",
                        "object": "(required) the object identifier for the entity to be executed",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # start time
        start_time = time.time()

        # Define the SPL query
        kwargs_search = {
            "app": "trackme",
            "earliest_time": "-5m",
            "latest_time": "now",
            "output_mode": "json",
            "count": 0,
        }
        searchquery = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="{component}" object="{object_value}" force_run="True"'

        query_results = []
        try:
            # spawn the search and get the results
            reader = run_splunk_search(
                service,
                searchquery,
                kwargs_search,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    query_results.append(item)
            if len(query_results) > 0:
                logger.info(
                    f"function post_outliers_mlmonitor_models successfully executed in {round(time.time() - start_time, 3)} seconds"
                )
                return {
                    "payload": {
                        "action": "success",
                        "upstream_query": searchquery,
                        "query_results": query_results,
                    },
                    "status": 200,
                }
            else:
                return {
                    "payload": {
                        "action": "failure",
                        "upstream_query": searchquery,
                        "query_results": "The search was executed successfully, but no results were returned.",
                    },
                    "status": 500,
                }

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Force mlmonitor execution for a given object
    def post_outliers_cim_mlmonitor_models(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}
                try:
                    object_value = resp_dict["object"]
                except Exception as e:
                    return {"payload": "object is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint runs the execution of the ML monitor backend process for a CIM entity, it requires a POST call with the following options:",
                "resource_desc": "Runs Machine Learning Outliers monitor process for a given CIM entity",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_cim_mlmonitor_models\" body=\"{'tenant_id':'mytenant', 'object':'network_traffic:pan_firewall'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "object": "(required) the object identifier for the entity to be executed",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Define the SPL query
        kwargs_search = {
            "app": "trackme",
            "earliest_time": "-5m",
            "latest_time": "now",
            "output_mode": "json",
            "count": 0,
        }
        searchquery = f'| trackmesplkoutlierscimtrackerhelper tenant_id="{tenant_id}" object="{object_value}" force_run="True"'

        query_results = []
        try:
            # spawn the search and get the results
            reader = run_splunk_search(
                service,
                searchquery,
                kwargs_search,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    query_results.append(item)
            if len(query_results) > 0:
                return {
                    "payload": {
                        "action": "success",
                        "upstream_query": searchquery,
                        "query_results": query_results,
                    },
                    "status": 200,
                }
            else:
                return {
                    "payload": {
                        "action": "failure",
                        "upstream_query": searchquery,
                        "query_results": "The search was executed successfully, but no results were returned.",
                    },
                    "status": 500,
                }

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Run Bulk mlmonitor or mltrain
    def post_outliers_bulk_action(self, request_info, **kwargs):
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {"payload": "tenant_id is required", "status": 500}

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                try:
                    component = resp_dict["component"]
                    # valid components are dsm/dhm/flx/fqm/wlk
                    if component not in ("dsm", "dhm", "flx", "fqm", "cim", "wlk"):
                        return {
                            "payload": f"component {component} is invalid, must be either dsm, dhm, flx, fqm or wlk",
                            "status": 500,
                        }
                except Exception as e:
                    return {"payload": "component is required", "status": 500}
                try:
                    action = resp_dict["action"]
                    if action not in (
                        "enable",
                        "disable",
                        "mlmonitor",
                        "mltrain",
                        "reset_status",
                    ):
                        return {
                            "payload": f"action {action} is invalid, must be either enable, disable, mlmonitor, mltrain, or reset_status",
                            "status": 500,
                        }
                except Exception as e:
                    return {"payload": "action is required", "status": 500}

        else:
            # body is not required in this endpoint, if not submitted do not describe the usage
            describe = False

        # if describe is requested, show the usage
        if describe:
            response = {
                "describe": "This endpoint allows performing bulk actions for a given list of entities, it requires a POST call with the following options:",
                "resource_desc": "Run bulk actions for a given list of entities (fire and forget for mlmonitor/mltrain)",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_bulk_action\" body=\"{'tenant_id':'mytenant','component':'dsm','object_list':'entity1,entity2', 'action': 'enable'}\"",
                "options": [
                    {
                        "tenant_id": "(required) tenant identifier",
                        "component": "(required) The component category, valid options: dsm/dhm/flx/fqm/cim/wlk",
                        "object_list": "(required unless using keys_list) comma seperated list of entities",
                        "keys_list": "(required unless using object_list) comma seperated list of keys",
                        "action": "(required) enable, disable, mlmonitor, mltrain, reset_status",
                    }
                ],
                "note": "mlmonitor and mltrain actions are executed in background (fire and forget), while enable/disable/reset_status are executed synchronously",
            }
            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # Define an header for requests authenticated communications with splunkd
        header = {
            "Authorization": "Splunk %s" % request_info.session_key,
            "Content-Type": "application/json",
        }

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # entity main collection
        collection_main_name = f"kv_trackme_{component}_tenant_{tenant_id}"
        collection_main = service.kvstore[collection_main_name]

        # entity rules collection
        collection_rules_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_rules = service.kvstore[collection_rules_name]

        # entity data collection
        collection_data_name = (
            f"kv_trackme_{component}_outliers_entity_data_tenant_{tenant_id}"
        )
        collection_data = service.kvstore[collection_data_name]

        #
        # proceed
        #

        # Retrieve the list object values if keys_list is provided
        if keys_list:
            object_list = []
            for key_value in keys_list:
                kvrecords = collection_main.data.query(
                    query=json.dumps({"_key": key_value})
                )
                for kvrecord in kvrecords:
                    object = kvrecord.get("object", None)
                    if object:
                        object_list.append(object)

        # Define background worker function for mlmonitor and mltrain actions
        def background_action_worker(
            object_value,
            action,
            tenant_id,
            component,
            header,
            request_info,
            update_comment,
        ):
            """Background worker function for mlmonitor and mltrain actions"""
            try:
                if action == "mlmonitor":
                    if component == "cim":
                        rest_url = f"{request_info.server_rest_uri}/services/trackme/v2/splk_outliers_engine/write/outliers_cim_mlmonitor_models"
                    else:
                        rest_url = f"{request_info.server_rest_uri}/services/trackme/v2/splk_outliers_engine/write/outliers_mlmonitor_models"

                elif action == "mltrain":
                    if component == "cim":
                        rest_url = f"{request_info.server_rest_uri}/services/trackme/v2/splk_outliers_engine/write/outliers_cim_train_models"
                    else:
                        rest_url = f"{request_info.server_rest_uri}/services/trackme/v2/splk_outliers_engine/write/outliers_train_models"

                post_data = {
                    "tenant_id": tenant_id,
                    "component": component,
                    "object": object_value,
                }

                response = requests.post(
                    rest_url,
                    headers=header,
                    data=json.dumps(post_data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code == 200:
                    logger.info(
                        f'Background {action} action completed successfully for object="{object_value}"'
                    )
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            f"Oultiers detection bulk action {action}",
                            str(object_value),
                            f"splk-{component}",
                            f"Background {action} action completed successfully",
                            f"The Outliers detection bulk action {action} was performed successfully in background",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )
                else:
                    logger.error(
                        f'Background {action} action failed for object="{object_value}", status_code="{response.status_code}", response="{response.text}"'
                    )
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "failure",
                            f"Oultiers detection bulk action {action}",
                            str(object_value),
                            f"splk-{component}",
                            f"Background {action} action failed",
                            f"The Outliers detection bulk action {action} has failed in background",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

            except Exception as e:
                logger.error(
                    f'Background {action} action failed for object="{object_value}" with exception="{str(e)}"'
                )
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        f"Oultiers detection bulk action {action}",
                        str(object_value),
                        f"splk-{component}",
                        f"Background {action} action failed",
                        f"The Outliers detection bulk action {action} has failed in background with exception: {str(e)}",
                        str(update_comment),
                    )
                except Exception as audit_e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(audit_e)}"'
                    )

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # For mlmonitor and mltrain actions, spawn background threads and return immediately
        if action in ("mlmonitor", "mltrain"):
            logger.info(
                f"Spawning background threads for {action} actions on {len(object_list)} objects"
            )

            # Spawn background threads for each object
            for object_value in object_list:
                thread = threading.Thread(
                    target=background_action_worker,
                    args=(
                        object_value,
                        action,
                        tenant_id,
                        component,
                        header,
                        request_info,
                        update_comment,
                    ),
                    daemon=True,
                )
                thread.start()
                processed_count += 1
                succcess_count += (
                    1  # Count as success since thread was spawned successfully
                )

            # Return immediately for fire and forget behavior
            req_summary = {
                "process_count": processed_count,
                "success_count": succcess_count,
                "failures_count": failures_count,
                "action": action,
                "message": f"Background {action} actions have been spawned for {processed_count} objects. Actions are running in background.",
                "note": "This is a fire and forget operation. Check logs for individual action results.",
            }

            logger.info(
                f"Bulk edit Fire and forget, action={action} actions spawned objects_count={processed_count}"
            )
            return {"payload": req_summary, "status": 200}

        # For synchronous actions (enable, disable, reset_status), process normally
        else:
            logger.info(
                f"Processing synchronous {action} actions for {len(object_list)} objects"
            )

            # Check if collections exist and have data
            try:
                rules_count = len(collection_rules.data.query())
                data_count = len(collection_data.data.query())
                logger.info(
                    f"Collection stats: rules_count={rules_count}, data_count={data_count}"
                )
            except Exception as e:
                logger.error(f"Failed to check collection stats: {str(e)}")

            # Loop through objects for synchronous processing
            for object_value in object_list:
                try:
                    logger.debug(
                        f"Processing object: {object_value} for action: {action}"
                    )
                    #
                    # Enable / Disable, we need to update the value of is_disabled in the rules collection for each matching object
                    #

                    if action in ("enable", "disable"):
                        logger.debug(
                            f"Processing enable/disable action for object: {object_value}"
                        )
                        # Get all matching records
                        kvrecords = collection_rules.data.query(
                            query=json.dumps({"object": object_value})
                        )
                        logger.debug(
                            f"Found {len(kvrecords)} records for object: {object_value}"
                        )

                        if len(kvrecords) == 0:
                            logger.warning(
                                f"No records found for object: {object_value} in collection: {collection_rules_name}"
                            )
                            # Still count as processed but mark as failure
                            processed_count += 1
                            succcess_count += 0
                            failures_count += 1
                            result = {
                                "object": object_value,
                                "action": "update",
                                "result": "failure",
                                "exception": f'No records found for object="{object_value}" in collection="{collection_rules_name}"',
                            }
                            records.append(result)
                            continue

                        for kvrecord in kvrecords:
                            # Update the record
                            current_object_value = kvrecord.get("object")
                            current_key_value = kvrecord.get("_key")
                            if action == "enable":
                                kvrecord["is_disabled"] = 0
                            elif action == "disable":
                                kvrecord["is_disabled"] = 1
                            kvrecord["mtime"] = time.time()

                            # Update the record
                            try:
                                collection_rules.data.update(
                                    current_key_value, json.dumps(kvrecord)
                                )

                                # increment counter
                                processed_count += 1
                                succcess_count += 1
                                failures_count += 0

                                result = {
                                    "object": current_object_value,
                                    "action": "update",
                                    "result": "success",
                                    "message": f'tenant_id="{tenant_id}", object="{current_object_value}" was successfully updated',
                                }
                                records.append(result)

                                try:
                                    trackme_audit_event(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        request_info.user,
                                        "success",
                                        f"Oultiers detection bulk action {action}",
                                        str(object_value),
                                        f"splk-{component}",
                                        str(
                                            json.dumps(
                                                collection_main.data.query(
                                                    query=json.dumps(
                                                        {"object": object_value}
                                                    )
                                                ),
                                                indent=1,
                                            )
                                        ),
                                        f"The Outliers detection bulk action {action} was performed successfully",
                                        str(update_comment),
                                    )
                                except Exception as e:
                                    logger.error(
                                        f'failed to generate an audit event with exception="{str(e)}"'
                                    )

                            except Exception as e:
                                # increment counter
                                processed_count += 1
                                succcess_count += 0
                                failures_count += 1

                                result = {
                                    "object": object_value,
                                    "action": "update",
                                    "result": "failure",
                                    "exception": f'tenant_id="{tenant_id}", object="{current_object_value}" failed to be updated, exception="{str(e)}"',
                                }
                                records.append(result)

                                try:
                                    trackme_audit_event(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        request_info.user,
                                        "failure",
                                        f"Oultiers detection bulk action {action}",
                                        str(object_value),
                                        f"splk-{component}",
                                        str(
                                            json.dumps(
                                                collection_main.data.query(
                                                    query=json.dumps(
                                                        {"object": object_value}
                                                    )
                                                ),
                                                indent=1,
                                            )
                                        ),
                                        f"The Outliers detection bulk action {action} has failed",
                                        str(update_comment),
                                    )
                                except Exception as e:
                                    logger.error(
                                        f'failed to generate an audit event with exception="{str(e)}"'
                                    )

                    #
                    # Reset status
                    #

                    # for reset_status, we need to delete the record in the data collection for each matching object

                    elif action in ("reset_status"):
                        logger.debug(
                            f"Processing reset_status action for object: {object_value}"
                        )
                        # Get all matching records
                        kvrecords = collection_rules.data.query(
                            query=json.dumps({"object": object_value})
                        )
                        logger.debug(
                            f"Found {len(kvrecords)} records for object: {object_value}"
                        )

                        if len(kvrecords) == 0:
                            logger.warning(
                                f"No records found for object: {object_value} in collection: {collection_rules_name}"
                            )
                            # Still count as processed but mark as failure
                            processed_count += 1
                            succcess_count += 0
                            failures_count += 1
                            result = {
                                "object": object_value,
                                "action": "update",
                                "result": "failure",
                                "exception": f'No records found for object="{object_value}" in collection="{collection_rules_name}"',
                            }
                            records.append(result)
                            continue

                        for kvrecord in kvrecords:
                            current_object_value = kvrecord.get("object")
                            current_key_value = kvrecord.get("_key")

                            try:
                                collection_data.data.delete(
                                    json.dumps({"_key": current_key_value})
                                )

                                # increment counter
                                processed_count += 1
                                succcess_count += 1
                                failures_count += 0

                                result = {
                                    "object": current_object_value,
                                    "action": "update",
                                    "result": "success",
                                    "message": f'tenant_id="{tenant_id}", object="{current_object_value}" was successfully updated',
                                }
                                records.append(result)

                                try:
                                    trackme_audit_event(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        request_info.user,
                                        "success",
                                        f"Oultiers detection bulk action {action}",
                                        str(object_value),
                                        f"splk-{component}",
                                        str(
                                            json.dumps(
                                                collection_main.data.query(
                                                    query=json.dumps(
                                                        {"object": object_value}
                                                    )
                                                ),
                                                indent=1,
                                            )
                                        ),
                                        f"The Outliers detection bulk action {action} was performed successfully",
                                        str(update_comment),
                                    )
                                except Exception as e:
                                    logger.error(
                                        f'failed to generate an audit event with exception="{str(e)}"'
                                    )

                            except Exception as e:
                                # increment counter
                                processed_count += 1
                                succcess_count += 0
                                failures_count += 1

                                result = {
                                    "object": object_value,
                                    "action": "update",
                                    "result": "failure",
                                    "exception": f'tenant_id="{tenant_id}", object="{current_object_value}" failed to be updated, exception="{str(e)}"',
                                }
                                records.append(result)

                                try:
                                    trackme_audit_event(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        request_info.user,
                                        "failure",
                                        f"Oultiers detection bulk action {action}",
                                        str(object_value),
                                        f"splk-{component}",
                                        str(
                                            json.dumps(
                                                collection_main.data.query(
                                                    query=json.dumps(
                                                        {"object": object_value}
                                                    )
                                                ),
                                                indent=1,
                                            )
                                        ),
                                        f"The Outliers detection bulk action {action} has failed",
                                        str(update_comment),
                                    )
                                except Exception as e:
                                    logger.error(
                                        f'failed to generate an audit event with exception="{str(e)}"'
                                    )

                except Exception as e:
                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    result = {
                        "object": object_value,
                        "action": "update",
                        "result": "failure",
                        "exception": f'tenant_id="{tenant_id}", object="{object_value}", failed to execute action="{action}", exception="{str(e)}"',
                    }
                    records.append(result)

            logger.info(
                f"Synchronous processing completed: processed={processed_count}, success={succcess_count}, failures={failures_count}"
            )
            # render HTTP status and summary for synchronous actions

            req_summary = {
                "process_count": processed_count,
                "success_count": succcess_count,
                "failures_count": failures_count,
                "records": records,
            }

            if processed_count > 0 and processed_count == succcess_count:
                return {"payload": req_summary, "status": 200}

            else:
                return {"payload": req_summary, "status": 500}

    def post_outliers_bulk_rules_update(self, request_info, **kwargs):
        describe = False

        # Validation rules mapping
        FIELD_VALIDATION_RULES = {
            # kpi_span, span notation such as 10m
            "kpi_span": (
                r"^\d+[m|h|d]$",
                "Must be an integer followed by one of: `m`, `h`, `d`.",
            ),
            # method_calculation, one of stdev, avg, max, min, sum, perc95, latest
            "method_calculation": (
                r"^(stdev|avg|max|min|sum|perc95|latest)$",
                "Must be one of: `stdev`, `avg`, `max`, `min`, `sum`, `perc95`, `latest`.",
            ),
            # period_calculation, now or a relative time notation such as -30d
            "period_calculation": (
                r"^-?\d+[m|h|d]$",
                "Must be an integer followed by one of: `m`, `h`, `d`.",
            ),
            # period_calculation_latest, now or a relative time notation such as -1d
            "period_calculation_latest": (
                r"^(now|-\d+[m|h|d])$",
                "Must be `latest`, `now` or a relative time notation such as `-1d`.",
            ),
            # time_factor, one of %H, %H%M, %w%H, %w%H%M, %w, none
            "time_factor": (
                r"^(%H|%H%M|%w%H|%w%H%M|%w|none)$",
                "Must be one of: `%H`, `%H%M`, `%w%H`, `%w%H%M`, `%w`, `none`.",
            ),
            # density_lowerthreshold, a decimal or integer
            "density_lowerthreshold": (
                r"^[\d|\.]*$",
                "Must be an integer or decimal.",
            ),
            # density_upperthreshold, a decimal or integer
            "density_upperthreshold": (
                r"^[\d|\.]*$",
                "Must be an integer or decimal.",
            ),
            # alert_lower_breached, 0 or 1
            "alert_lower_breached": (r"^(1|0)$", "Must be `1` (True) or `0` (False)."),
            # alert_upper_breached, 0 or 1
            "alert_upper_breached": (r"^(1|0)$", "Must be `1` (True) or `0` (False)."),
            # auto_correct, 1 or 0
            "auto_correct": (r"^(1|0)$", "Must be `1` (True) or `0` (False)."),
            # perc_min_lowerbound_deviation, a decimal or integer
            "perc_min_lowerbound_deviation": (
                r"^[\d|\.]*$",
                "Must be an integer or decimal.",
            ),
            # perc_min_upperbound_deviation, a decimal or integer
            "perc_min_upperbound_deviation": (
                r"^[\d|\.]*$",
                "Must be an integer or decimal.",
            ),
            # min_value_for_lowerbound_breached, a decimal or integer
            "min_value_for_lowerbound_breached": (
                r"^[\d|\.]*$",
                "Must be an integer or decimal.",
            ),
            # min_value_for_upperbound_breached, a decimal or integer
            "min_value_for_upperbound_breached": (
                r"^[\d|\.]*$",
                "Must be an integer or decimal.",
            ),
            # static_lower_threshold, a decimal or integer
            "static_lower_threshold": (
                r"^[\d|\.]*$",
                "Must be an integer or decimal.",
            ),
            # static_upper_threshold, a decimal or integer
            "static_upper_threshold": (
                r"^[\d|\.]*$",
                "Must be an integer or decimal.",
            ),
            # is_disabled, 0 or 1
            "is_disabled": (r"^(1|0)$", "Must be `1` (True) or `0` (False)."),
        }

        def validate_update_fields(update_fields):
            """Validates update fields based on predefined rules."""
            invalid_fields = {}
            for field, value in update_fields.items():
                if field in FIELD_VALIDATION_RULES:
                    pattern, error_message = FIELD_VALIDATION_RULES[field]
                    if not re.match(pattern, str(value)):
                        invalid_fields[field] = error_message

            return invalid_fields

        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception:
            resp_dict = None

        if resp_dict is not None:
            describe = resp_dict.get("describe", False) in ("true", "True")

            if not describe:
                tenant_id = resp_dict.get("tenant_id")
                if not tenant_id:
                    return {"payload": "tenant_id is required", "status": 500}

                object_list = resp_dict.get("object_list", [])
                if isinstance(object_list, str):
                    object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", [])
                if isinstance(keys_list, str):
                    keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": "Either object_list or keys_list must be provided",
                        "status": 500,
                    }

                component = resp_dict.get("component")
                if component not in ("dsm", "dhm", "flx", "fqm", "cim", "wlk"):
                    return {"payload": f"Invalid component {component}", "status": 500}

                update_fields = {
                    k: v
                    for k, v in resp_dict.items()
                    if k not in ("tenant_id", "component", "object_list", "keys_list")
                }

                if not update_fields:
                    return {
                        "payload": "At least one field to update must be provided",
                        "status": 500,
                    }

                # Ensure values are not empty or null
                invalid_fields = [
                    k for k, v in update_fields.items() if v in (None, "")
                ]
                if invalid_fields:
                    return {
                        "payload": f"Invalid values for fields: {', '.join(invalid_fields)}",
                        "status": 500,
                    }

                # Validate input fields
                validation_errors = validate_update_fields(update_fields)
                if validation_errors:
                    return {
                        "payload": f"Invalid values: {validation_errors}",
                        "status": 500,
                    }

                # if we have a field kpi_metric, ensure it does not contain "splk.flx.replaceme" which is specific for
                # splk-flx and preset in the UI, users are required to change this
                if "kpi_metric" in update_fields:
                    if "splk.flx.replaceme" in update_fields["kpi_metric"]:
                        return {
                            "payload": "kpi_metric must be changed from 'splk.flx.replaceme'",
                            "status": 500,
                        }

                # prevents an update from the UI for custom update without care
                # if we have a field called field_name, and/or a value with value equal to field_value, raise an error
                # field_name: field_value is forbidden
                # <something>: field_value is forbidden
                # field_name: <something> is forbidden
                for field, value in update_fields.items():
                    if field in ("field_name", "field_value") or value in (
                        "field_name",
                        "field_value",
                    ):
                        return {
                            "payload": "field_name and field_value are forbidden",
                            "status": 500,
                        }

        else:
            describe = False

        if describe:
            response = {
                "describe": "This endpoint allows performing bulk updates on outlier detection rules for a given list of entities.",
                "resource_desc": "Bulk update rules for outlier detection",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_outliers_engine/write/outliers_bulk_rules_update\" body=\"{'tenant_id':'mytenant','component':'dsm','object_list':'entity1,entity2', 'kpi_span': '10m'}\"",
                "options": [
                    {
                        "tenant_id": "(required) Tenant identifier",
                        "component": "(required) The component category, valid options: dsm/dhm/flx/fqm/cim/wlk",
                        "object_list": "(required unless using keys_list) Comma separated list of entities",
                        "keys_list": "(required unless using object_list) Comma separated list of keys",
                        "update_fields": "(optional) Fields to update within the rules, only specified fields will be modified",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        splunkd_port = request_info.server_rest_port
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        collection_rules_name = (
            f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_rules = service.kvstore[collection_rules_name]

        processed_count = 0
        success_count = 0
        failures_count = 0
        records = []

        if "*" in object_list:
            kvrecords = collection_rules.data.query()
            object_list = [kvrecord.get("object") for kvrecord in kvrecords]

        if keys_list:
            kvrecords = collection_rules.data.query(
                query=json.dumps({"_key": {"$in": keys_list}})
            )
            object_list.extend([kvrecord.get("object") for kvrecord in kvrecords])

        for object_value in object_list:
            try:
                kvrecords = collection_rules.data.query(
                    query=json.dumps({"object": object_value})
                )
                for kvrecord in kvrecords:
                    current_key_value = kvrecord.get("_key")
                    entities_outliers = kvrecord.get("entities_outliers", "{}")

                    if isinstance(entities_outliers, str):
                        try:
                            entities_outliers = json.loads(entities_outliers)
                        except json.JSONDecodeError:
                            failures_count += 1
                            records.append(
                                {
                                    "object": object_value,
                                    "status": "failure",
                                    "error": "Invalid JSON structure in entities_outliers",
                                }
                            )
                            continue

                    for model_key, model_data in entities_outliers.items():
                        if isinstance(model_data, dict):
                            for field, value in update_fields.items():
                                if field in model_data:
                                    model_data[field] = value

                    kvrecord["entities_outliers"] = json.dumps(
                        entities_outliers, indent=2
                    )
                    kvrecord["mtime"] = time.time()

                    try:
                        collection_rules.data.update(
                            current_key_value, json.dumps(kvrecord)
                        )
                        success_count += 1
                        records.append({"object": object_value, "status": "success"})
                    except Exception as e:
                        failures_count += 1
                        records.append(
                            {
                                "object": object_value,
                                "status": "failure",
                                "error": str(e),
                            }
                        )
            except Exception as e:
                failures_count += 1
                records.append(
                    {"object": object_value, "status": "failure", "error": str(e)}
                )

            processed_count += 1

        summary = {
            "processed_count": processed_count,
            "success_count": success_count,
            "failures_count": failures_count,
            "records": records,
        }

        return {"payload": summary, "status": 200 if failures_count == 0 else 500}
