#!/usr/bin/env python
# coding=utf-8

__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

import os
import sys
import json
import logging
import requests
import urllib3
import re
import time
import hashlib

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append lib
sys.path.append(os.path.join(splunkhome, "etc", "apps", "trackme", "lib"))

# import Splunk libs
import splunklib.client as client
import splunklib.results as results

# import trackme libs
from trackme_libs import (
    run_splunk_search,
)

# import trackme libs utils
from trackme_libs_utils import remove_leading_spaces

# import trackme libs
from trackme_libs import get_kv_collection

# import trackme libs get data
from trackme_libs_get_data import get_full_kv_collection

# import the collections dict
from collections_data import (
    collections_dict,
    vtenant_account_default,
    remote_account_default,
)

# logging:
# To avoid overriding logging destination of callers, the libs will not set on purpose any logging definition
# and rely on callers themselves


# Function to format the version number
def trackme_schema_format_version(trackme_version: str) -> int:
    # Split the version into its components
    major, minor, patch = trackme_version.split(".")

    # Ensure the patch version is two digits
    patch = patch.zfill(2)

    # Combine the parts and convert to integer
    schema_version_required = int(major + minor + patch)

    return schema_version_required


# update the schema version to a certain release number
def trackme_schema_get_version(
    reqinfo, tenant_id, schema_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.debug(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_get_version, tenant_id="{tenant_id}"'
    )

    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, trackme_schema_get_version, the vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, trackme_schema_get_version, the vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:
        # try to get the schema version
        try:
            tenant_schema_version = vtenant_record.get("schema_version")
        except Exception as e:
            tenant_schema_version = None

        # update as needed
        if not tenant_schema_version:
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", trackme_schema_get_version, this tenant does not have a schema version defined yet, processing now.'
            )
            return None
        else:
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", trackme_schema_get_version, current schema_version="{tenant_schema_version}"'
            )
            return tenant_schema_version


# update the schema version to a certain release number
def trackme_schema_update_version(
    reqinfo, tenant_id, schema_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.debug(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_update_version, tenant_id="{tenant_id}"'
    )

    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:
        # try to get the schema version
        try:
            tenant_schema_version = vtenant_record.get("schema_version")
        except Exception as e:
            tenant_schema_version = None

        # logging debug
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", current schema_version="{tenant_schema_version}"'
        )

        # update as needed
        if not tenant_schema_version:
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema version check, this tenant does not have a schema version defined yet, processing its update now.'
            )

            try:
                vtenant_record["schema_version"] = schema_version
                vtenant_record["schema_version_mtime"] = time.time()
                collection_vtenants.data.update(
                    str(vtenant_key), json.dumps(vtenant_record)
                )
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema version upgraded successfully, new schema_version="{schema_version}"'
                )
                return {
                    "action": "success",
                    "response": "schema version was successfully upgraded",
                    "schema_version": schema_version,
                }

            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, failure while trying to update the vtenant KVstore record, exception="{str(e)}"'
                )
                raise Exception(
                    f'failure while trying to update the vtenant KVstore record, exception="{str(e)}"'
                )

        elif tenant_schema_version != schema_version:
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema version check, this tenant needs to be upgraded, current_schema_version="{tenant_schema_version}", target_schema_version="{schema_version}", processing its update now.'
            )

            try:
                vtenant_record["schema_version"] = schema_version
                vtenant_record["schema_version_mtime"] = time.time()
                collection_vtenants.data.update(
                    str(vtenant_key), json.dumps(vtenant_record)
                )
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema version upgraded successfully, new schema_version="{schema_version}"'
                )
                return {
                    "action": "success",
                    "response": "schema version was successfully upgraded",
                    "schema_version": schema_version,
                }

            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, failure while trying to update the vtenant KVstore record, exception="{str(e)}"'
                )
                raise Exception(
                    f'failure while trying to update the vtenant KVstore record, exception="{str(e)}"'
                )

        else:
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema version check, this tenant schema is up to date, nothing to do.'
            )
            return {
                "action": "success",
                "schema_version": schema_version,
            }


# get permissions
def get_permissions(vtenant_record):

    # for read permissions, concatenate admin, power and user
    tenant_roles_read_perms = f"{vtenant_record.get('tenant_roles_admin', 'trackme_admin')},{vtenant_record.get('tenant_roles_power', 'trackme_power')},{vtenant_record.get('tenant_roles_user', 'trackme_user')}"

    # for write permissions, concatenate admin, power
    tenant_roles_write_perms = f"{vtenant_record.get('tenant_roles_admin', 'trackme_admin')},{vtenant_record.get('tenant_roles_power', 'trackme_power')}"

    return tenant_roles_read_perms, tenant_roles_write_perms


def update_vtenant_configuration(
    reqinfo, task_name, task_instance_id, tenant_id, updated_vtenant_data=None
):
    """
    Update the configuration of a vtenant by first fetching the current configuration,
    merging with default values, and applying any updates. If a dict of key-value
    pairs is provided, it takes precedence over the defaults.
    Any key from the GET response not in the default config should be kept.

    :param reqinfo: dict containing Splunk session information (e.g., server URI, session key).
    :param task_name: Name of the task for logging purposes.
    :param task_instance_id: ID of the task instance for logging purposes.
    :param tenant_id: ID of the vtenant.
    :param updated_vtenant_data: Optional dict containing key-value pairs to update in the vtenant configuration.
    """

    # endpoint target
    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/maintain_vtenant_account'

    # post data
    post_data = {
        "tenant_id": tenant_id,
        "force_create_missing": True,
    }

    # add updated_vtenant_data if provided
    if updated_vtenant_data:
        post_data["updated_vtenant_data"] = updated_vtenant_data

    try:
        # Get current vtenant account configuration
        response = requests.post(
            url,
            headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
            verify=False,
            data=json.dumps(post_data),
            timeout=600,
        )

        if response.status_code in (200, 201, 204):
            response_json = response.json()
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, successfully call vtenant check and maintain endpoint, response="{json.dumps(response_json, indent=2)}"'
            )
            return True

        else:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, failed to retrieve vtenant configuration, status_code={response.status_code}'
            )
            return False

    except Exception as e:
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, error while fetching vtenant configuration: {str(e)}'
        )
        return False


def update_remote_account_configuration(
    reqinfo, task_name, task_instance_id, tenant_id, default_account_values=None
):
    """
    Update the configuration of any exising remote account, to ensure that the configuration is up to date.

    :param reqinfo: dict containing Splunk session information (e.g., server URI, session key).
    :param task_name: Name of the task for logging purposes.
    :param task_instance_id: ID of the task instance for logging purposes.
    :param tenant_id: ID of the vtenant.
    :param default_account_values: manadatory dict of default values.
    """

    # endpoint target
    url = f'{reqinfo["server_rest_uri"]}/servicesNS/nobody/trackme/trackme_account'

    # current_remote_accounts_dict
    current_remote_accounts_dict = {}

    # first, get the list of remote accounts
    try:
        response = requests.get(
            url,
            headers={
                "Authorization": f'Splunk {reqinfo["session_key"]}',
                "Content-Type": "application/json",
            },
            verify=False,
            params={
                "output_mode": "json",
                "count": -1,
            },
            timeout=600,
        )

        response.raise_for_status()
        response_json = response.json()

        # The list of remote accounts is stored as a list in entry
        remote_accounts = response_json.get("entry", [])

        # iterate through the remote accounts, adding them to the dict, name is the key, then we care about "content" which is a dict of our parameters
        # for this account

        for remote_account in remote_accounts:
            remote_account_name = remote_account.get("name", None)
            remote_account_content = remote_account.get("content", {})

            if remote_account_name and remote_account_content:

                # from remote_account_content, remove the following fields: bearer_token, disabled, eai:acl, eai:appName, eai:userName
                remote_account_content.pop("bearer_token", None)
                remote_account_content.pop("disabled", None)
                remote_account_content.pop("eai:acl", None)
                remote_account_content.pop("eai:appName", None)
                remote_account_content.pop("eai:userName", None)
                # add to the dict
                current_remote_accounts_dict[remote_account_name] = (
                    remote_account_content
                )

    except Exception as e:
        logging.error(
            f'tenant_id="{tenant_id}", task="{task_name}", task_instance_id={task_instance_id}, error while fetching remote account list: {str(e)}'
        )
        return False

    # Second, iterate through our current_remote_accounts_dict, if any of the account is missing key/values from the default_account_values, we will update it
    # running a POST request to the remote account endpoint

    for remote_account_name in current_remote_accounts_dict:
        current_account_config = current_remote_accounts_dict[remote_account_name]

        # check if the account is missing any key/values from the default_account_values
        account_must_be_updated = False
        for key in default_account_values:
            if key not in current_account_config:
                account_must_be_updated = True
                # update the current_account_config with the default value
                current_account_config[key] = default_account_values[key]

                # run a request against /services/trackme/v2/configuration/get_remote_account, body{'account': 'prd1_cm1'} to retrieve the current bearer_token value (field token) and add to the content
                try:
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/get_remote_account'
                    data = {
                        "account": remote_account_name,
                    }
                    response_account_secret = requests.post(
                        url,
                        headers={
                            "Authorization": f'Splunk {reqinfo["session_key"]}',
                            "Content-Type": "application/json",
                        },
                        verify=False,
                        data=json.dumps(data),
                        timeout=600,
                    )
                    response_account_secret.raise_for_status()
                    response_account_secret_json = response_account_secret.json()
                    current_token = response_account_secret_json.get("token", None)

                    if not current_token:
                        logging.error(
                            f'tenant_id="{tenant_id}", task="{task_name}", task_instance_id={task_instance_id}, account={remote_account_name}, error while fetching remote account secret: token not found'
                        )
                        account_must_be_updated = False

                    else:
                        current_account_config["bearer_token"] = current_token

                except Exception as e:
                    logging.error(
                        f'tenant_id="{tenant_id}", task="{task_name}", task_instance_id={task_instance_id}, account={remote_account_name}, error while fetching remote account secret: {str(e)}'
                    )
                    account_must_be_updated = False

        # if the account must be updated, we will run a POST request to the remote account endpoint
        if account_must_be_updated:
            # endpoint target
            url = f'{reqinfo["server_rest_uri"]}/servicesNS/nobody/trackme/trackme_account/{remote_account_name}?output_mode=json'

            try:
                response = requests.post(
                    url,
                    headers={
                        "Authorization": f'Splunk {reqinfo["session_key"]}',
                        "Content-Type": "application/json",
                    },
                    verify=False,
                    params=current_account_config,
                    timeout=600,
                )
                response.raise_for_status()
                logging.info(
                    f'tenant_id="{tenant_id}", task="{task_name}", task_instance_id={task_instance_id}, successfully updated remote account configuration for missing default values, account={remote_account_name}, status={response.status_code}'
                )

            except Exception as e:
                logging.error(
                    f'tenant_id="{tenant_id}", task="{task_name}", task_instance_id={task_instance_id}, error while updating remote account configuration: {str(e)}'
                )
                return False

    return True


# process update version 2.0.9
# changes: in version 2.0.9, we introduced a new Ack Metadata, ack_type, which requires an update of the KV transform


def trackme_schema_upgrade_2009(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2009, tenant_id="{tenant_id}"'
    )

    # dict of collections
    collections_dict = dict(
        [
            (
                "trackme_common_alerts_ack",
                "_key, object, object_category, ack_mtime, ack_expiration, ack_state, ack_type, ack_comment",
            ),
        ]
    )

    # name of the object
    object_name = "trackme_common_alerts_ack"

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:
        transform_name = "trackme_common_alerts_ack_tenant_" + str(tenant_id)
        collection_name = "kv_trackme_common_alerts_ack_tenant_" + str(tenant_id)
        transform_fields = collections_dict[object_name]
        transform_acl = {
            "owner": vtenant_record.get("tenant_owner"),
            "sharing": "app",
            "perms.write": vtenant_record.get("tenant_roles_admin"),
            "perms.read": vtenant_record.get("tenant_roles_user"),
        }

        # delete the transform
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
        data = {
            "tenant_id": tenant_id,
            "transform_name": transform_name,
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
            )

        # create the transform
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
        data = {
            "tenant_id": tenant_id,
            "transform_name": transform_name,
            "transform_fields": transform_fields,
            "collection_name": collection_name,
            "transform_acl": transform_acl,
            "owner": vtenant_record.get("tenant_owner"),
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
            )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 209, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2015(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2015, tenant_id="{tenant_id}"'
    )

    # dict of collections
    collections_dict = dict(
        [
            (
                "trackme_dsm",
                "_key, mtime, search_mode, tenant_id, object_category, alias, data_index, data_last_lag_seen, data_last_ingestion_lag_seen, data_eventcount, data_last_lag_seen_idx, data_first_time_seen, data_last_time_seen, data_last_ingest, data_last_time_seen_idx, data_max_lag_allowed, data_max_delay_allowed, data_lag_alert_kpis, monitored_state, object, data_sourcetype, data_monitoring_wdays, isUnderMonitoringDays, data_monitoring_hours_ranges, isUnderMonitoringHours, data_override_lagging_class, object_state, tracker_runtime, object_previous_state, data_previous_tracker_runtime, dcount_host, min_dcount_host, isAnomaly, data_sample_lastrun, tags, latest_flip_state, latest_flip_time, priority, status_message, anomaly_reason",
            ),
            (
                "trackme_cim",
                "_key, mtime, tenant_id, object, alias, monitored_state, cim_tracking_rules, cim_tracking_results, object_state, account, tracker_name, tracker_runtime, tracker_last_duration, latest_flip_state, latest_flip_time, priority, anomaly_reason",
            ),
            (
                "trackme_flx",
                "_key, mtime, tenant_id, group, object, object_category, object_state, object_description, alias, monitored_state, account, tracker_name, tracker_runtime, status, status_description, metrics, outliers_metrics, monitoring_wdays, isUnderMonitoringDays, monitoring_hours_ranges, isUnderMonitoringHours, latest_flip_state, latest_flip_time, priority, status_message, anomaly_reason",
            ),
            (
                "trackme_dhm",
                "_key, mtime, search_mode, tenant_id, object_category, object, alias, data_index, data_sourcetype, data_last_lag_seen, data_last_ingestion_lag_seen, data_eventcount, data_first_time_seen, data_last_time_seen, data_last_ingest, data_max_lag_allowed, data_max_delay_allowed, data_lag_alert_kpis, monitored_state, data_monitoring_wdays, isUnderMonitoringDays, data_monitoring_hours_ranges, isUnderMonitoringHours, data_override_lagging_class, object_state, tracker_runtime, object_previous_state, data_previous_tracker_runtime, splk_dhm_st_summary, splk_dhm_alerting_policy, latest_flip_state, latest_flip_time, priority, status_message, anomaly_reason",
            ),
            (
                "trackme_mhm",
                "_key, mtime, tenant_id, object_category, object, alias, metric_index, metric_category, metric_details, metric_last_lag_seen, metric_first_time_seen, metric_last_time_seen, metric_max_lag_allowed, monitored_state, metric_monitoring_wdays, metric_override_lagging_class, object_state, tracker_runtime, object_previous_state, metric_previous_tracker_runtime, latest_flip_state, latest_flip_time, priority, status_message, anomaly_reason",
            ),
        ]
    )

    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:
        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append("trackme_dhm")

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            objects_to_process.append("trackme_mhm")

        if vtenant_record.get("tenant_flx_enabled") == 1:
            objects_to_process.append("trackme_flx")

        if vtenant_record.get("tenant_cim_enabled") == 1:
            objects_to_process.append("trackme_cim")

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": vtenant_record.get("tenant_roles_admin"),
                "perms.read": vtenant_record.get("tenant_roles_user"),
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2015, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2016(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2016, tenant_id="{tenant_id}"'
    )

    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:
        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append(
                "trackme_dsm_outliers_mltrain_tracker_tenant_%s" % tenant_id
            )

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append(
                "trackme_dhm_outliers_mltrain_tracker_tenant_%s" % tenant_id
            )

        if vtenant_record.get("tenant_flx_enabled") == 1:
            objects_to_process.append(
                "trackme_flx_outliers_mltrain_tracker_tenant_%s" % tenant_id
            )

        if vtenant_record.get("tenant_cim_enabled") == 1:
            objects_to_process.append(
                "trackme_cim_outliers_mltrain_tracker_tenant_%s" % tenant_id
            )

        for report_name in objects_to_process:
            # create the report
            component = None

            if report_name.startswith("trackme_dsm"):
                component = "dsm"
            elif report_name.startswith("trackme_dhm"):
                component = "dhm"
            elif report_name.startswith("trackme_flx"):
                component = "flx"
            elif report_name.startswith("trackme_cim"):
                component = "cim"

            report_search = (
                '| trackmesplkoutlierstrainhelper tenant_id="%s" component="%s"'
                % (tenant_id, component)
            )
            report_properties = {
                "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                "is_scheduled": True,
                "cron_schedule": "*/60 * * * *",
                "dispatch.earliest_time": "-5m",
                "dispatch.latest_time": "now",
                "schedule_window": "5",
            }

            # for read permissions, concatenate admin and guest
            tenant_roles_read_perms = "%s,%s" % (
                str(vtenant_record.get("tenant_roles_admin")),
                str(vtenant_record.get("tenant_roles_user")),
            )

            # delete the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", delete report has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted report, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the report, report="{report_name}", exception="{str(e)}"'
                )

            # create the report
            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": "app",
                "perms.write": str(vtenant_record.get("tenant_roles_admin")),
                "perms.read": str(tenant_roles_read_perms),
            }
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
                "report_search": report_search,
                "report_properties": report_properties,
                "report_acl": report_acl,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2016, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2020(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # not if target_version > 2064
    if target_version > 2064:
        logging.info(
            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2020, procedure terminated, target_version="{target_version}"'
        )
        return True

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2020, tenant_id="{tenant_id}"'
    )

    components_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:
        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")

        # loop
        for component in components_to_process:
            # for read permissions, concatenate admin and guest
            tenant_roles_read_perms = "%s,%s" % (
                str(vtenant_record.get("tenant_roles_admin")),
                str(vtenant_record.get("tenant_roles_user")),
            )

            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": "app",
                "perms.write": str(vtenant_record.get("tenant_roles_admin")),
                "perms.read": str(tenant_roles_read_perms),
            }

            # create the wrapper
            report_name = (
                f"trackme_{component}_delayed_entities_tracker_tenant_{tenant_id}"
            )
            if component in ("dsm"):
                report_search = f'| trackmesplkfeedsdelayed tenant_id="{tenant_id}" component="{component}" earliest="-24h" latest="+4h" max_runtime_sec="3600" max_ingest_age_sec="86400" min_time_since_inspection_sec="900" dsm_tstats_root_breakby_include_splunk_server=True dsm_tstats_root_breakby_include_host=True'
            elif component in ("dhm"):
                report_search = f'| trackmesplkfeedsdelayed tenant_id="{tenant_id}" component="{component}" earliest="-24h" latest="+4h" max_runtime_sec="3600" max_ingest_age_sec="86400" min_time_since_inspection_sec="900" dhm_tstats_root_breakby_include_splunk_server=True'
            report_properties = {
                "description": "This scheduled report handled delayed entities for the tenant and component",
                "is_scheduled": True,
                "cron_schedule": "*/60 * * * *",
                "dispatch.earliest_time": "-5m",
                "dispatch.latest_time": "now",
                "schedule_window": "5",
            }

            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": "app",
                "perms.write": str(vtenant_record.get("tenant_roles_admin")),
                "perms.read": str(tenant_roles_read_perms),
            }

            # create the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
                "report_search": report_search,
                "report_properties": report_properties,
                "report_acl": report_acl,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2020, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2026(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2026, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:
        # init the wlk component to be disabled
        vtenant_record["tenant_wlk_enabled"] = 0

        # update
        try:
            collection_vtenants.data.update(
                str(vtenant_key), json.dumps(vtenant_record)
            )
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2026, Virtual Tenant updated successfully, result="{json.dumps(vtenant_record, indent=2)}"'
            )

        except Exception as e:
            error_msg = f'task="{task_name}", task_instance_id={task_instance_id}, function trackme_schema_upgrade_2026, failed to update the Virtual Tenant record, exception="{str(e)}"'
            logging.error(error_msg)

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2026, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2034(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2034, tenant_id="{tenant_id}"'
    )
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:
        # first, introduce the tenant_replica boolean and update the tenant record
        vtenant_record["tenant_replica"] = False
        try:
            collection_vtenants.data.update(
                str(vtenant_key), json.dumps(vtenant_record)
            )
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, vtenant record was updated successfully, tenant_replica was set to False'
            )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, vtenant record update has failed, tenant_replica has not been set, exception="{str(e)}"'
            )

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append("trackme_dhm")

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            objects_to_process.append("trackme_mhm")

        if vtenant_record.get("tenant_flx_enabled") == 1:
            objects_to_process.append("trackme_flx")

        if vtenant_record.get("tenant_wlk_enabled") == 1:
            objects_to_process.append("trackme_wlk")

        if vtenant_record.get("tenant_cim_enabled") == 1:
            objects_to_process.append("trackme_cim")

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": vtenant_record.get("tenant_roles_admin"),
                "perms.read": vtenant_record.get("tenant_roles_user"),
            }

            # proceed boolean
            proceed = False

            # check first if the transforms exists, if it does not exist, we do not need to proceed
            try:
                transform__current = service.confs["transform_name"]
                proceed = True
            except Exception as e:
                proceed = False
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", transform does not exist, no need to proceed, transform="{transform_name}"'
                )

            if proceed:

                # delete the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": transform_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

        # log
        logging.info(
            f'task="{task_name}", task_instance_id={task_instance_id}, function trackme_schema_upgrade_2034 terminated, tenant_id="{tenant_id}"'
        )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2034, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2034_least_privileges(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2034_least_privileges, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:
        # get the admin value
        try:
            current_tenant_roles_admin = vtenant_record["tenant_roles_admin"]
        except Exception as e:
            current_tenant_roles_admin = "trackme_admin"

        # set a value
        tenant_roles_power = None
        if current_tenant_roles_admin != "trackme_admin":
            tenant_roles_power = current_tenant_roles_admin
        else:
            tenant_roles_power = "trackme_power"

        # update the record
        vtenant_record["tenant_roles_power"] = tenant_roles_power

        # update
        try:
            collection_vtenants.data.update(
                str(vtenant_key), json.dumps(vtenant_record)
            )
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2034 (least privileges), Virtual Tenant updated successfully, result="{json.dumps(vtenant_record, indent=2)}"'
            )

        except Exception as e:
            error_msg = f'task="{task_name}", task_instance_id={task_instance_id}, function trackme_schema_upgrade_2034_leat_privileges, failed to update the Virtual Tenant record, exception="{str(e)}"'
            logging.error(error_msg)

        # RBAC update

        # Define an header for requests authenticated communications with splunkd
        header = {
            "Authorization": "Splunk %s" % reqinfo["session_key"],
            "Content-Type": "application/json",
        }

        url = "%s/services/trackme/v2/vtenants/admin/update_tenant_rbac" % (
            reqinfo["server_rest_uri"]
        )
        data = {
            "tenant_id": tenant_id,
            "tenant_roles_admin": vtenant_record["tenant_roles_admin"],
            "tenant_roles_power": vtenant_record["tenant_roles_power"],
            "tenant_roles_user": vtenant_record["tenant_roles_user"],
            "tenant_owner": vtenant_record["tenant_owner"],
        }

        # create the account
        try:
            response = requests.post(
                url,
                headers=header,
                data=json.dumps(data, indent=1),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2034 (least privileges), RBAC update has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2034 (least privileges), RBAC update was operated successfully, response="{json.dumps(response.json(), indent=2)}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2034 (least privileges), RBAC update has failed, exception="{str(e)}"'
            )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2034, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2036(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2036, tenant_id="{tenant_id}"'
    )
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        # proceed boolean
        proceed = False

        # First, run a get request to check the vtenant account exists already, if not, create it
        url = f'{reqinfo["server_rest_uri"]}/servicesNS/nobody/trackme/trackme_vtenants/{vtenant_record.get("tenant_id")}'
        try:
            response = requests.get(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                verify=False,
                timeout=600,
            )
            if response.status_code in (200, 201, 204):
                proceed = False
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, vtenant account already exists, tenant_id="{tenant_id}"'
                )
            else:
                proceed = True
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, vtenant account does not exist, tenant_id="{tenant_id}"'
                )
        except Exception as e:
            proceed = True
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, failed to check the vtenant account, exception="{str(e)}"'
            )

        if proceed:

            #
            # Create the vtenant account configuration
            #

            url = f'{reqinfo["server_rest_uri"]}/servicesNS/nobody/trackme/trackme_vtenants'
            data = {
                "name": vtenant_record.get("tenant_id"),
                "description": vtenant_record.get("tenant_desc"),
            }

            # add to data each key value from vtenant_account_default imported
            for key, value in vtenant_account_default.items():
                data[key] = value

            # create the account
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=data,
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, create vtenant account has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, create vtenant account was operated successfully, response.status_code="{response.status_code}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, create vtenant account has failed, exception="{str(e)}"'
                )

        #
        # Decomission of the data sampling obfuscation macro
        #

        if vtenant_record.get("tenant_dsm_enabled") == 1:

            # proceed boolean
            proceed = False

            # get permissions
            tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
                vtenant_record
            )

            # deletion of the sampling tracker
            report_name = f"trackme_dsm_data_sampling_tracker_tenant_{tenant_id}"

            # First, check if the report exists, it is does we have nothing to do
            try:
                report_current = service.saved_searches[report_name]
                proceed = True
            except Exception as e:
                proceed = False
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, report does not exist, nothing to do, report="{report_name}"'
                )

            if proceed:

                # delete the report
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": report_name,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", delete report has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", successfully deleted report, report="{report_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", failed to delete the report, report="{report_name}", exception="{str(e)}"'
                    )

            # attempt to create the new fresh tracker
            # trackme-limited/trackme-report-issues#839: the schema upgrade will fail to create the report as macros originally called do not exist anymore.
            # This restricted version will address this issue.
            report_search = f'| trackmesamplingexecutor tenant_id="{tenant_id}"'
            report_properties = {
                "description": "TrackMe DSM Data Sampling tracker",
                "is_scheduled": True,
                "schedule_window": "5",
                "cron_schedule": "*/20 * * * *",
                "dispatch.earliest_time": "-24h",
                "dispatch.latest_time": "-4h",
            }

            report_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": str(tenant_roles_write_perms),
                "perms.read": str(tenant_roles_read_perms),
            }

            # create the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
                "report_search": report_search,
                "report_properties": report_properties,
                "report_acl": report_acl,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                )

            # decomission the macro
            macro_name = f"trackme_dsm_data_sampling_obfuscation_tenant_{tenant_id}"

            # delete the macro
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_macro'
            data = {
                "tenant_id": tenant_id,
                "macro_name": macro_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", delete macro has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", successfully deleted macrp, macro="{macro_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", failed to delete the report, macro="{macro_name}", exception="{str(e)}"'
                )

        #
        # Replica tenant
        #

        # then, introduce the tenant_replica boolean and update the tenant record
        vtenant_record["tenant_replica"] = False
        try:
            collection_vtenants.data.update(
                str(vtenant_key), json.dumps(vtenant_record)
            )
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, vtenant record was updated successfully, tenant_replica was set to False'
            )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, vtenant record update has failed, tenant_replica has not been set, exception="{str(e)}"'
            )

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append("trackme_dhm")

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            objects_to_process.append("trackme_mhm")

        if vtenant_record.get("tenant_flx_enabled") == 1:
            objects_to_process.append("trackme_flx")

        if vtenant_record.get("tenant_wlk_enabled") == 1:
            objects_to_process.append("trackme_wlk")

        if vtenant_record.get("tenant_cim_enabled") == 1:
            objects_to_process.append("trackme_cim")

        for object_name in objects_to_process:
            transform_name = f"{object_name}_tenant_{tenant_id}"
            collection_name = f"kv_{object_name}_tenant_{tenant_id}"
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": vtenant_record.get("tenant_roles_admin"),
                "perms.read": vtenant_record.get("tenant_roles_user"),
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2036, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2036, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2038(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2038, tenant_id="{tenant_id}"'
    )
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append("trackme_dhm")

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            objects_to_process.append("trackme_mhm")

        if vtenant_record.get("tenant_wlk_enabled") == 1:
            objects_to_process.append("trackme_wlk")

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2038, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2043(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2043, tenant_id="{tenant_id}"'
    )
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm_outliers_entity_rules")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append("trackme_dhm_outliers_entity_rules")

        if vtenant_record.get("tenant_cim_enabled") == 1:
            objects_to_process.append("trackme_cim_outliers_entity_rules")

        if vtenant_record.get("tenant_flx_enabled") == 1:
            objects_to_process.append("trackme_flx_outliers_entity_rules")

        if vtenant_record.get("tenant_wlk_enabled") == 1:
            objects_to_process.append("trackme_wlk_outliers_entity_rules")

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2043, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2044(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2044, tenant_id="{tenant_id}"'
    )
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_flx_enabled") == 1:
            objects_to_process.append("trackme_flx")

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2044, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2045(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2045, tenant_id="{tenant_id}"'
    )

    components_to_process = []
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        # check components and add accordingly
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
            objects_to_process.append("trackme_flx")

        #
        # objects
        #

        for object_name in objects_to_process:

            # get permissions
            tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
                vtenant_record
            )

            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # components
        #

        for component in components_to_process:

            # get permissions
            tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
                vtenant_record
            )

            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the wrapper
            report_name = (
                f"trackme_{component}_inactive_entities_tracker_tenant_{tenant_id}"
            )
            report_search = f'| trackmesplkflxinactiveinspector tenant_id="{tenant_id}" register_component="True" report="{report_name}" max_sec_since_inactivity_before_status_update=3600 max_days_since_inactivity_before_purge=7'
            report_properties = {
                "description": "This scheduled report handles inactive entities for the splk-flx component",
                "is_scheduled": True,
                "cron_schedule": "*/15 * * * *",
                "dispatch.earliest_time": "-5m",
                "dispatch.latest_time": "now",
                "schedule_window": "5",
            }
            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": "app",
                "perms.write": str(vtenant_record.get("tenant_roles_admin")),
                "perms.read": str(tenant_roles_read_perms),
            }

            # create the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
                "report_search": report_search,
                "report_properties": report_properties,
                "report_acl": report_acl,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2045, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2054(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2054, tenant_id="{tenant_id}"'
    )
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm")

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2054, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2064(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2064, tenant_id="{tenant_id}"'
    )

    components_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")

        # Retrieve the exec summary record
        tenant_objects_exec_summary = vtenant_record.get("tenant_objects_exec_summary")

        # load tenant_objects_exec_summary as an object
        try:
            tenant_objects_exec_summary = json.loads(tenant_objects_exec_summary)
        except Exception as e:
            tenant_objects_exec_summary = {}

        # results
        results = []

        # loop
        for component in components_to_process:

            # proceed boolean
            proceed = False

            # create the wrapper
            report_name = (
                f"trackme_{component}_delayed_entities_tracker_tenant_{tenant_id}"
            )

            # only proceed if the report exists
            try:
                report_obj = service.saved_searches[report_name]
                proceed = True
            except Exception as e:
                proceed = False

            if proceed:

                # delete the report
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": report_name,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", delete report has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted report, report="{report_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the report, report="{report_name}", exception="{str(e)}"'
                    )

                # attempt to clear the tenant_objects_exec_summary record from this object
                try:
                    tenant_objects_exec_summary.pop(report_name)
                    vtenant_record["tenant_objects_exec_summary"] = json.dumps(
                        tenant_objects_exec_summary, indent=2
                    )
                    collection_vtenants.data.update(
                        str(vtenant_key), json.dumps(vtenant_record)
                    )
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully cleared the tenant_objects_exec_summary record, report="{report_name}"'
                    )
                    results.append(
                        f'tenant_id="{tenant_id}", successfully cleared the tenant_objects_exec_summary record, report="{report_name}"'
                    )
                except Exception as e:
                    pass

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2064, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2067(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2067, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        # check components and add accordingly
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            process_schema_upgrade = True
        else:
            process_schema_upgrade = False

        if process_schema_upgrade:
            # Define the query
            search = f'| inputlookup trackme_wlk_tenant_{tenant_id} | eval key = _key | where match(object, "\\\\\\\\") | table key, object'

            kwargs_oneshot = {
                "earliest_time": "-5m",
                "latest_time": "now",
                "output_mode": "json",
                "count": 0,
            }

            # A list to store all executions
            results_list = []

            # A dict to store the results
            results_dict = {}

            try:
                reader = run_splunk_search(
                    service,
                    search,
                    kwargs_oneshot,
                    24,
                    5,
                )

                for item in reader:
                    if isinstance(item, dict):
                        results_dict[item["key"]] = item["object"]
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="wlk", detected non ASCII entity to be purged, object="{item["object"]}", key="{item["key"]}"'
                        )

            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="wlk", could not verify the presence of non ASCII entities, search permanently failed, exception="{str(e)}"'
                )

            # if the results_dict is not empty, we need to purge the entities
            if results_dict:
                # Data collection
                collection_name = "kv_trackme_wlk_tenant_" + str(tenant_id)
                collection = service.kvstore[collection_name]

                # loop through the results dict and remove any entity from the KVstore
                for key, object in results_dict.items():
                    try:
                        collection.data.delete(json.dumps({"_key": key}))
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="wlk", successfully purged non ASCII entity, object="{object}", key="{key}"'
                        )
                        results_list.append(
                            {
                                "object": object,
                                "key": key,
                                "result": "successfully purged non ASCII entity",
                            }
                        )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="wlk", failed to purge non ASCII entity, object="{object}", key="{key}", exception="{str(e)}"'
                        )
                        results_list.append(
                            {
                                "object": object,
                                "key": key,
                                "result": "failure to purge non ASCII entity",
                                "exception": str(e),
                            }
                        )

            # log a message if there were no entities to be purged
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="wlk", no non ASCII entities to be purged'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2067, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2070(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2070, tenant_id="{tenant_id}"'
    )
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            objects_to_process.append("trackme_wlk")

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2070, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2071(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2071, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", processing with the schema upgrade'
            )

            # retrieve the list of scheduler reports to be processed
            collection_trackers_name = (
                f"kv_trackme_wlk_hybrid_trackers_tenant_{tenant_id}"
            )
            collection_trackers = service.kvstore[collection_trackers_name]

            trackers_dict = {}

            trackers_to_process = []
            trackers_to_process_kos = []

            # get records from the KVstore
            trackers_records = collection_trackers.data.query()

            for tracker_record in trackers_records:
                tracker_name = tracker_record.get("tracker_name")
                tracker_kos = tracker_record.get("knowledge_objects")

                # if the tracker_name starts by scheduler_
                if re.search("^scheduler_", tracker_name):
                    trackers_to_process.append(tracker_name)
                    trackers_to_process_kos.append(tracker_kos)

                    trackers_dict[tracker_name] = {
                        "knowledge_objects": json.loads(tracker_kos),
                    }

            # loop through the trackers
            for tracker_shortname in trackers_to_process:
                tracker_name = (
                    f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"
                )

                tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_account = tracker_kos["properties"][0]["account"]

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                )

                # in tracker_current_search, replace the sentence:
                # | fields - collection_app, collection_user
                # with: (note that account need to replaced by the actual account name)
                # | fields - collection_app, collection_user
                # ``` Verify the report owner from rest if it is not yet known to TrackMe ```
                # | trackmesplkwlkgetreportowner account="{account}"

                try:
                    tracker_new_search = re.sub(
                        r"\| fields - collection_app, collection_user",
                        f'| fields - collection_app, collection_user\n\n``` Verify the report owner from rest if it is not yet known to TrackMe, this custom command will perform a get Metadata operation attempt if necessary ```\n| trackmesplkwlkgetreportowner account="{tracker_account}"\n',
                        tracker_current_search,
                    )

                    # update the search definition
                    tracker_current.update(search=tracker_new_search)

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, the tracker was updated successfully, new_search="{tracker_new_search}"'
                    )

                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, failed to update the tracker, exception="{str(e)}"'
                    )

        # check components and add accordingly
        objects_to_process = []
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm")

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2071, procedure terminated'
    )
    return True


"""
In this function:
- A bug had affected the definition of outliers rules for wlk, activating lower bound breaches for elapsed which is unwanted
- Enhance the scheduler execution errors detection by taking into account searches in delegated errors
- Update transforms for dsm/dhm for the purposes of allow_adaptive_delay
- Update ML Outliers rules transforms to add the confidence and confidence_reason fields
- Initialize ML Outliers values for confidence and confidence_reason
"""


def trackme_schema_upgrade_2072(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2072, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        #
        # prepare transforms updates
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # main transforms updates
        #

        # check components and add accordingly
        objects_to_process = []
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append("trackme_dhm")

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # ML rules outliers transforms update
        #

        # check components and add accordingly
        objects_to_process = []
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm_outliers_entity_rules")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append("trackme_dhm_outliers_entity_rules")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            objects_to_process.append("trackme_flx_outliers_entity_rules")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            objects_to_process.append("trackme_wlk_outliers_entity_rules")

        for object_name in objects_to_process:
            transform_name = f"{object_name}_tenant_{tenant_id}"
            collection_name = f"kv_{object_name}_tenant_{tenant_id}"
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # ML: initiate confidence and confidence_reason fields for existing tenants
        #

        components_to_process = []
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")

        # Get the value for splk_outliers_min_days_history
        splk_outliers_min_days_history = reqinfo["trackme_conf"][
            "splk_outliers_detection"
        ]["splk_outliers_min_days_history"]

        for component in components_to_process:
            # set kwargs
            kwargs_confidence = {
                "earliest_time": "-90d",
                "latest_time": "now",
                "output_mode": "json",
                "count": 0,
            }

            # define the gen search
            metric_root = None
            if component in ("dsm", "dhm"):
                metric_root = f"trackme.splk.feeds"
            else:
                metric_root = f"trackme.splk.{component}"

            ml_confidence_search = remove_leading_spaces(
                f"""\
                | mstats latest({metric_root}.*) as * where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_category="splk-{component}" object=* by object span=1d
                | stats min(_time) as first_time by object
                | eval metrics_duration=now()-first_time
                | eval days_required = {splk_outliers_min_days_history}
                | eval confidence=if(metrics_duration<(days_required*86400), "low", "normal")
                | eval metrics_duration=tostring(metrics_duration, "duration")
                | eval confidence_reason=if(metrics_duration<(days_required*86400), "ML has insufficient historical metrics to proceed (metrics_duration=" . metrics_duration . ", required=" . days_required + "days)", "ML has sufficient historical metrics to proceed (metrics_duration=" . metrics_duration . ", required=" . days_required . "days)")
                | fields object, confidence, confidence_reason
                | lookup local=t trackme_{component}_outliers_entity_rules_tenant_{tenant_id} object OUTPUT _key as keyid, entities_outliers, is_disabled, last_exec, mtime, object_category | where (isnotnull(keyid) AND isnotnull(confidence))
                | outputlookup append=t key_field=keyid trackme_{component}_outliers_entity_rules_tenant_{tenant_id}
                """
            )

            ml_confidence_update_counter = 0
            ml_confidence_updated_entities = []

            # run search

            # track the search runtime
            start = time.time()

            # proceed
            try:
                reader = run_splunk_search(
                    service,
                    ml_confidence_search,
                    kwargs_confidence,
                    24,
                    5,
                )

                for item in reader:
                    if isinstance(item, dict):
                        ml_confidence_update_counter += 1
                        ml_confidence_updated_entities.append(item.get("object"))

            except Exception as e:
                msg = (
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", '
                    f'search failed with exception="{str(e)}", run_time="{str(time.time() - start)}"'
                )
                logging.error(msg)

            # log
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", ml_confidence_update_counter="{ml_confidence_update_counter}", ml_confidence_updated_entities="{json.dumps(ml_confidence_updated_entities, indent=2)}"'
            )

        #
        # Adaptive delay tracker
        #

        components_to_process = []
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")

        for component in components_to_process:
            #
            # Create the adaptive delay tracker
            #

            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": trackme_default_sharing,
                "perms.write": str(vtenant_record.get("tenant_roles_admin")),
                "perms.read": str(tenant_roles_read_perms),
            }

            # create the wrapper
            report_name = (
                f"trackme_{component}_adaptive_delay_tracker_tenant_{tenant_id}"
            )
            report_search = f'| trackmesplkadaptivedelay tenant_id="{tenant_id}" component="{component}" min_delay_sec=3600 min_historical_metrics_days=7 earliest_time_mstats="-30d" max_runtime=900'
            report_properties = {
                "description": "This scheduled report manages adaptive delay tresholds for TrackMe",
                "is_scheduled": True,
                "cron_schedule": "*/15 * * * *",
                "dispatch.earliest_time": "-5m",
                "dispatch.latest_time": "now",
                "schedule_window": "5",
            }

            # create the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
                "report_search": report_search,
                "report_properties": report_properties,
                "report_acl": report_acl,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                )

        #
        # Workload
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", processing with the schema upgrade'
            )

            #
            # fix outliers rules
            #

            # access the KV
            collection_outliers_rules_name = (
                f"kv_trackme_wlk_outliers_entity_rules_tenant_{tenant_id}"
            )
            collection_outliers_rules = service.kvstore[collection_outliers_rules_name]

            # get all records
            collection_records = []
            collection_records_keys = set()

            end = False
            skip_tracker = 0
            while end == False:
                process_collection_records = collection_outliers_rules.data.query(
                    skip=skip_tracker
                )
                if len(process_collection_records) != 0:
                    for item in process_collection_records:
                        if item.get("_key") not in collection_records_keys:
                            collection_records.append(item)
                            collection_records_keys.add(item.get("_key"))
                    skip_tracker += 5000
                else:
                    end = True

            # loop through the records
            for outlier_record in collection_records:
                entities_outliers = json.loads(outlier_record.get("entities_outliers"))
                for model_id in entities_outliers:
                    kpi_metric = entities_outliers[model_id].get("kpi_metric")
                    if kpi_metric == "splk.wlk.elapsed":
                        entities_outliers[model_id]["alert_lower_breached"] = 0

                # update the record
                outlier_record["entities_outliers"] = json.dumps(
                    entities_outliers, indent=2
                )

                # update the KVstore
                try:
                    collection_outliers_rules.data.update(
                        outlier_record.get("_key"),
                        json.dumps(outlier_record, indent=2),
                    )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the entity rules, exception="{str(e)}"'
                    )

            #
            # enhance scheduler execution errors detection
            #

            # retrieve the list of scheduler reports to be processed
            collection_trackers_name = (
                f"kv_trackme_wlk_hybrid_trackers_tenant_{tenant_id}"
            )
            collection_trackers = service.kvstore[collection_trackers_name]

            trackers_dict = {}

            trackers_to_process = []
            trackers_to_process_kos = []

            # get records from the KVstore
            trackers_records = collection_trackers.data.query()

            for tracker_record in trackers_records:
                tracker_name = tracker_record.get("tracker_name")
                tracker_kos = tracker_record.get("knowledge_objects")

                # if the tracker_name starts by scheduler_
                if re.search("^scheduler_", tracker_name):
                    trackers_to_process.append(tracker_name)
                    trackers_to_process_kos.append(tracker_kos)

                    trackers_dict[tracker_name] = {
                        "knowledge_objects": json.loads(tracker_kos),
                    }

            # loop through the trackers
            for tracker_shortname in trackers_to_process:
                tracker_name = (
                    f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"
                )

                tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_account = tracker_kos["properties"][0]["account"]

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                )

                # in tracker_current_search, replace the sentence:
                # len(errmsg)>0,\"error\"
                # with: (note that account need to replaced by the actual account name)
                # len(errmsg)>0,\"error\" OR status == \"delegated_remote_error\"

                if tracker_account == "local":
                    tracker_new_search = re.sub(
                        r'len\(errmsg\)>0,"error"',
                        r'len(errmsg)>0 OR status == "delegated_remote_error","error"',
                        tracker_current_search,
                    )
                else:
                    tracker_new_search = re.sub(
                        r'len\(errmsg\)>0,\\"error\\"',
                        r'len(errmsg)>0 OR status == \\"delegated_remote_error\\",\\"error\\"',
                        tracker_current_search,
                    )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

            #
            # Update the metadata search to include an explicit call to filters_get_last_updates="host=*"
            #

            # retrieve the list of scheduler reports to be processed
            collection_trackers_name = (
                f"kv_trackme_wlk_hybrid_trackers_tenant_{tenant_id}"
            )
            collection_trackers = service.kvstore[collection_trackers_name]

            trackers_dict = {}

            trackers_to_process = []
            trackers_to_process_kos = []

            # get records from the KVstore
            trackers_records = collection_trackers.data.query()

            for tracker_record in trackers_records:
                tracker_name = tracker_record.get("tracker_name")
                tracker_kos = tracker_record.get("knowledge_objects")

                # if the tracker_name starts by scheduler_
                if re.search("^metadata_", tracker_name):
                    trackers_to_process.append(tracker_name)
                    trackers_to_process_kos.append(tracker_kos)

                    trackers_dict[tracker_name] = {
                        "knowledge_objects": json.loads(tracker_kos),
                    }

            # loop through the trackers
            for tracker_shortname in trackers_to_process:
                tracker_name = (
                    f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"
                )

                tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_account = tracker_kos["properties"][0]["account"]

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                )

                # in tracker_current_search, replace the sentence:
                # max_runtime_sec="900"
                # with:
                # max_runtime_sec="900" filters_get_last_updates="host=*"

                tracker_new_search = re.sub(
                    r'max_runtime_sec="900"',
                    r'max_runtime_sec="900" filters_get_last_updates="host=*"',
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

            #
            # ML Outliers: update the default Outliers rules macro
            #

            macro_name = f"trackme_wlk_set_outliers_metrics_tenant_{tenant_id}"
            macro_current = service.confs["macros"][macro_name]
            macro_current_definition = macro_current.content.get("definition")

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", macro_name="{macro_name}", macro_current_definition="{macro_current_definition}", processing macro update'
            )

            """
            In macro_current_defintion, replace:
            
            eval outliers_metrics="{'elapsed': {'alert_lower_breached': 0, 'alert_upper_breached': 1}}"
            
            by:
            
            eval outliers_metrics="{'elapsed': {'alert_lower_breached': 0, 'alert_upper_breached': 1, 'time_factor': 'none'}}"

            """

            macro_new_definition = "eval outliers_metrics=\"{'elapsed': {'alert_lower_breached': 0, 'alert_upper_breached': 1, 'time_factor': 'none'}}\""

            # update the search definition
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_macro'
            data = {
                "tenant_id": tenant_id,
                "macro_name": macro_name,
                "macro_definition": macro_new_definition,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update macro definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated macro definition, transform="{macro_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the macro definition, transform="{macro_name}", exception="{str(e)}"'
                )

            #
            # ML Outliers: mass update rules for wlk
            #

            # set kwargs
            kwargs_wlk_outliers_rules_update = {
                "earliest_time": "-5m",
                "latest_time": "now",
                "output_mode": "json",
                "count": 0,
            }

            # define the gen search
            wlk_outliers_rules_update_search = remove_leading_spaces(
                f"""\
                | inputlookup trackme_wlk_outliers_entity_rules_tenant_{tenant_id} | eval keyid=_key
                ``` mass update the time_factor ```
                | rex field=entities_outliers mode=sed "s/\\\"time_factor\\\": \\\"%H\\\"/\\\"time_factor\\\": \\\"none\\\"/g"
                ``` force update the confidence and confidence_reason ```
                | eval confidence="low", confidence_reason="upgrade to TrackMe 2.0.72 for splk-wlk requires models to be trained again due to the change of the time_factor field, which will happen automatically as soon as possible."
                | outputlookup append=t key_field=keyid trackme_wlk_outliers_entity_rules_tenant_{tenant_id}
                """
            )

            wlk_outliers_rules_update_counter = 0
            wlk_outliers_rules_updated_entities = []

            # run search

            # track the search runtime
            start = time.time()

            # proceed
            try:
                reader = run_splunk_search(
                    service,
                    wlk_outliers_rules_update_search,
                    kwargs_wlk_outliers_rules_update,
                    24,
                    5,
                )

                for item in reader:
                    if isinstance(item, dict):
                        wlk_outliers_rules_update_counter += 1
                        wlk_outliers_rules_updated_entities.append(item.get("object"))

            except Exception as e:
                msg = (
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", '
                    f'search failed with exception="{str(e)}", run_time="{str(time.time() - start)}"'
                )
                logging.error(msg)

            # log
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", wlk_outliers_rules_update_counter="{wlk_outliers_rules_update_counter}", wlk_outliers_rules_updated_entities="{json.dumps(wlk_outliers_rules_updated_entities, indent=2)}", search="{wlk_outliers_rules_update_search}"'
            )

        #
        # End version 2.0.72
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2072, procedure terminated'
    )
    return True


"""
In this function:
- Update the vtenant accounts to define adaptive_delay
- Create a new KVstore collection for wlk to store last seen activity per type of data
- Support overlap in the searches with deduplication for a safer collection of Wlk metrics
"""


def trackme_schema_upgrade_2075(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2075, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        #
        # Adaptive trackers
        #

        components_to_process = []
        if (
            vtenant_record.get("tenant_dsm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dsm")
        if (
            vtenant_record.get("tenant_dhm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dhm")

        for component in components_to_process:
            #
            # Update the adaptive delay tracker
            #

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", processing with adaptive delay tracker update'
            )

            # report name
            tracker_name = (
                f"trackme_{component}_adaptive_delay_tracker_tenant_{tenant_id}"
            )

            # get the current search definition
            try:
                tracker_current = service.saved_searches[tracker_name]
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                )
                continue

            tracker_current_search = tracker_current.content.get("search")

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
            )

            # add our new parameter
            tracker_new_search = f"{tracker_current_search} max_auto_delay_sec=604800"

            # update the search definition
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": tracker_name,
                "report_search": tracker_new_search,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                )

        #
        # WLK: create new collection & transforms definition
        #

        # check components and add accordingly
        if (
            vtenant_record.get("tenant_wlk_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="wlk", processing with Workload tracker updates'
            )

            transform_name = f"trackme_wlk_last_seen_activity_tenant_{tenant_id}"
            collection_name = f"kv_trackme_wlk_last_seen_activity_tenant_{tenant_id}"
            transform_fields = collections_dict["trackme_wlk_last_seen_activity"]
            definition_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the KVstore collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": definition_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            kvstore_created = False
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                    )
                    kvstore_created = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                )

            #
            # Only continue if successful
            #

            # create the transform
            if kvstore_created:
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": definition_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

                #
                # Trackers updates
                #

                # retrieve the list of scheduler reports to be processed
                collection_trackers_name = (
                    f"kv_trackme_wlk_hybrid_trackers_tenant_{tenant_id}"
                )
                collection_trackers = service.kvstore[collection_trackers_name]

                #
                # enhance scheduler execution to support deduplication
                #

                trackers_dict = {}
                trackers_to_process = []
                trackers_to_process_kos = []

                # get records from the KVstore
                trackers_records = collection_trackers.data.query()

                for tracker_record in trackers_records:
                    tracker_name = tracker_record.get("tracker_name")
                    tracker_kos = tracker_record.get("knowledge_objects")

                    # if the tracker_name starts by scheduler_
                    if re.search("^scheduler_", tracker_name):
                        trackers_to_process.append(tracker_name)
                        trackers_to_process_kos.append(tracker_kos)

                        trackers_dict[tracker_name] = {
                            "knowledge_objects": json.loads(tracker_kos),
                        }

                # loop through the trackers (wrapper)
                for tracker_shortname in trackers_to_process:
                    tracker_name = f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"

                    tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                    # get the current search definition
                    try:
                        tracker_current = service.saved_searches[tracker_name]
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                        )
                        continue

                    tracker_current_search = tracker_current.content.get("search")
                    tracker_account = tracker_kos["properties"][0]["account"]

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                    )

                    # in tracker_current_search, replace the sentence: (if account is not local, double escape the quotes)
                    #  _index_earliest="-5m" _index_latest="now"
                    # with:
                    #  _index_earliest="-20m" _index_latest="now"
                    # | bucket _time span=5m

                    if tracker_account == "local":
                        tracker_new_search = re.sub(
                            r'_index_earliest="-5m" _index_latest="now"',
                            r'_index_earliest="-20m" _index_latest="now" earliest="-20m" latest="now"\n| eval orig_time = _time | bucket _time span=5m',
                            tracker_current_search,
                        )

                    else:
                        tracker_new_search = re.sub(
                            r'_index_earliest=\\"-5m\\" _index_latest=\\"now\\"',
                            r'_index_earliest=\\"-20m\\" _index_latest=\\"now\\" earliest=\\"-20m\\" latest=\\"now\\"\n| eval orig_time = _time | bucket _time span=5m',
                            tracker_current_search,
                        )

                    # only if remote, replace the sentence:
                    #  earliest="-10m" latest="now"
                    # with:
                    #  earliest="-20m" latest="now"

                    if tracker_account != "local":
                        tracker_new_search = re.sub(
                            r'earliest="-10m" latest="now"',
                            r'earliest="-20m" latest="now"',
                            tracker_new_search,
                        )

                    # replace the sentence:
                    # | stats count(eval(status
                    # with:
                    # | stats max(_time) as _time, count(eval(status

                    tracker_new_search = re.sub(
                        r"\| stats count\(eval\(status",
                        r"| stats max(_time) as _time, count(eval(status",
                        tracker_new_search,
                    )

                    # replace the sentence:
                    # by app, user, savedsearch_name
                    # with:
                    # by _time, app, user, savedsearch_name

                    tracker_new_search = re.sub(
                        r"by app, user, savedsearch_name",
                        r"by orig_time, app, user, savedsearch_name",
                        tracker_new_search,
                    )

                    # replace the sentence:
                    # | fields app, user, savedsearch_name, count_completed
                    # with:
                    # | fields last_seen, app, user, savedsearch_name, count_completed

                    tracker_new_search = re.sub(
                        r"\| fields app, user, savedsearch_name, count_completed",
                        r"| fields _time, app, user, savedsearch_name, count_completed",
                        tracker_new_search,
                    )

                    # replace the sentence:
                    # context="live"
                    # with:
                    # context="live" check_last_seen=True check_last_seen_field=last_seen_scheduler

                    tracker_new_search = re.sub(
                        r"context=\"live\"",
                        r"context=live check_last_seen=True check_last_seen_field=last_seen_scheduler",
                        tracker_new_search,
                    )

                    # update the search definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": tracker_name,
                        "report_search": tracker_new_search,
                        "earliest_time": "-20m",
                        "latest_time": "now",
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                        )

                # loop through the trackers (tracker)
                for tracker_shortname in trackers_to_process:
                    tracker_name = f"trackme_wlk_hybrid_{tracker_shortname}_tracker_tenant_{tenant_id}"

                    tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                    # get the current search definition
                    try:
                        tracker_current = service.saved_searches[tracker_name]
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                        )
                        continue

                    tracker_current_search = tracker_current.content.get("search")
                    tracker_account = tracker_kos["properties"][0]["account"]

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                    )

                    # replace the sentence:
                    # earliest="-10m" latest="now"
                    # with:
                    # earliest="-20m" latest="now"

                    tracker_new_search = re.sub(
                        r'earliest="-10m" latest="now"',
                        r'earliest="-20m" latest="now"',
                        tracker_current_search,
                    )

                    # update the search definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": tracker_name,
                        "report_search": tracker_new_search,
                        "earliest_time": "-20m",
                        "latest_time": "now",
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                        )

                #
                # enhance introspection execution to support deduplication
                #

                trackers_dict = {}
                trackers_to_process = []
                trackers_to_process_kos = []

                # get records from the KVstore
                trackers_records = collection_trackers.data.query()

                for tracker_record in trackers_records:
                    tracker_name = tracker_record.get("tracker_name")
                    tracker_kos = tracker_record.get("knowledge_objects")

                    # if the tracker_name starts by scheduler_
                    if re.search("^introspection_", tracker_name):
                        trackers_to_process.append(tracker_name)
                        trackers_to_process_kos.append(tracker_kos)

                        trackers_dict[tracker_name] = {
                            "knowledge_objects": json.loads(tracker_kos),
                        }

                # loop through the trackers (wrapper)
                for tracker_shortname in trackers_to_process:
                    tracker_name = f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"

                    tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                    # get the current search definition
                    try:
                        tracker_current = service.saved_searches[tracker_name]
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                        )
                        continue

                    tracker_current_search = tracker_current.content.get("search")
                    tracker_account = tracker_kos["properties"][0]["account"]

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                    )

                    # in tracker_current_search, replace the sentence: (if account is not local, double escape the quotes)
                    #  _index_earliest="-5m" _index_latest="now"
                    # with:
                    #  _index_earliest="-20m" _index_latest="now"
                    # | bucket _time span=5m

                    if tracker_account == "local":
                        tracker_new_search = re.sub(
                            r'_index_earliest="-5m" _index_latest="now"',
                            r'_index_earliest="-20m" _index_latest="now" earliest="-20m" latest="now"',
                            tracker_current_search,
                        )

                    else:
                        tracker_new_search = re.sub(
                            r'_index_earliest=\\"-5m\\" _index_latest=\\"now\\"',
                            r'_index_earliest=\\"-20m\\" _index_latest=\\"now\\" earliest=\\"-20m\\" latest=\\"now\\"',
                            tracker_current_search,
                        )

                    # only if remote, replace the sentence:
                    #  earliest="-10m" latest="now"
                    # with:
                    #  earliest="-20m" latest="now"

                    if tracker_account != "local":
                        tracker_new_search = re.sub(
                            r'earliest="-10m" latest="now"',
                            r'earliest="-20m" latest="now"',
                            tracker_new_search,
                        )

                    # replace the sentence:
                    # | stats max(_time) as last_time, avg(elapsed) as elapsed, avg(pct_cpu) as pct_cpu
                    # with:
                    # | eval orig_time = _time | bucket orig_time span=5m | stats max(_time) as last_time, avg(elapsed) as elapsed, avg(pct_cpu) as pct_cpu

                    tracker_new_search = re.sub(
                        r"\| stats max\(_time\) as last_time, avg\(elapsed\) as elapsed, avg\(pct_cpu\) as pct_cpu",
                        r"| eval orig_time = _time | bucket orig_time span=5m | stats max(_time) as _time, avg(elapsed) as elapsed, avg(pct_cpu) as pct_cpu",
                        tracker_new_search,
                    )

                    # replace the sentence:
                    # by object, app, savedsearch_name
                    # with:
                    # by _time, object, app, savedsearch_name

                    tracker_new_search = re.sub(
                        r"by object, app, savedsearch_name",
                        r"by orig_time, object, app, savedsearch_name",
                        tracker_new_search,
                    )

                    # replace the sentence:
                    # | fields object, object_id, app, savedsearch_name, last_time, elapsed, pct_cpu, pct_memory, scan_count, type, user, group
                    # with:
                    # | fields object, object_id, app, savedsearch_name, last_seen, elapsed, pct_cpu, pct_memory, scan_count, type, user, group

                    tracker_new_search = re.sub(
                        r"\| fields object, object_id, app, savedsearch_name, last_time, elapsed, pct_cpu, pct_memory, scan_count, type, user, group",
                        r"| fields _time, object, object_id, app, savedsearch_name, elapsed, pct_cpu, pct_memory, scan_count, type, user, group",
                        tracker_new_search,
                    )

                    # replace the sentence:
                    # context="live"
                    # with:
                    # context="live" check_last_seen=True check_last_seen_field=last_seen_introspection

                    tracker_new_search = re.sub(
                        r"context=\"live\"",
                        r"context=live check_last_seen=True check_last_seen_field=last_seen_introspection",
                        tracker_new_search,
                    )

                    # update the search definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": tracker_name,
                        "report_search": tracker_new_search,
                        "earliest_time": "-20m",
                        "latest_time": "now",
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                        )

                # loop through the trackers (tracker)
                for tracker_shortname in trackers_to_process:
                    tracker_name = f"trackme_wlk_hybrid_{tracker_shortname}_tracker_tenant_{tenant_id}"

                    tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                    # get the current search definition
                    try:
                        tracker_current = service.saved_searches[tracker_name]
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                        )
                        continue

                    tracker_current_search = tracker_current.content.get("search")
                    tracker_account = tracker_kos["properties"][0]["account"]

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                    )

                    # replace the sentence:
                    # earliest="-10m" latest="now"
                    # with:
                    # earliest="-20m" latest="now"

                    tracker_new_search = re.sub(
                        r'earliest="-10m" latest="now"',
                        r'earliest="-20m" latest="now"',
                        tracker_current_search,
                    )

                    # update the search definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": tracker_name,
                        "report_search": tracker_new_search,
                        "earliest_time": "-20m",
                        "latest_time": "now",
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                        )

                #
                # enhance notable execution to support deduplication
                #

                trackers_dict = {}
                trackers_to_process = []
                trackers_to_process_kos = []

                # get records from the KVstore
                trackers_records = collection_trackers.data.query()

                for tracker_record in trackers_records:
                    tracker_name = tracker_record.get("tracker_name")
                    tracker_kos = tracker_record.get("knowledge_objects")

                    # if the tracker_name starts by scheduler_
                    if re.search("^notable_", tracker_name):
                        trackers_to_process.append(tracker_name)
                        trackers_to_process_kos.append(tracker_kos)

                        trackers_dict[tracker_name] = {
                            "knowledge_objects": json.loads(tracker_kos),
                        }

                # loop through the trackers (wrapper)
                for tracker_shortname in trackers_to_process:
                    tracker_name = f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"

                    tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                    # get the current search definition
                    try:
                        tracker_current = service.saved_searches[tracker_name]
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                        )
                        continue

                    tracker_current_search = tracker_current.content.get("search")
                    tracker_account = tracker_kos["properties"][0]["account"]

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                    )

                    # in tracker_current_search, replace the sentence: (if account is not local, double escape the quotes)
                    #  _index_earliest="-5m" _index_latest="now"
                    # with:
                    #  _index_earliest="-20m" _index_latest="now"
                    # | bucket _time span=5m

                    if tracker_account == "local":
                        tracker_new_search = re.sub(
                            r'_index_earliest="-5m" _index_latest="now"',
                            r'_index_earliest="-20m" _index_latest="now" earliest="-20m" latest="now"',
                            tracker_current_search,
                        )

                    else:
                        tracker_new_search = re.sub(
                            r'_index_earliest=\\"-5m\\" _index_latest=\\"now\\"',
                            r'_index_earliest=\\"-20m\\" _index_latest=\\"now\\" earliest=\\"-20m\\" latest=\\"now\\"',
                            tracker_current_search,
                        )

                    # only if remote, replace the sentence:
                    #  earliest="-10m" latest="now"
                    # with:
                    #  earliest="-20m" latest="now"

                    if tracker_account != "local":
                        tracker_new_search = re.sub(
                            r'earliest="-10m" latest="now"',
                            r'earliest="-20m" latest="now"',
                            tracker_new_search,
                        )

                    # replace the sentence:
                    # by source
                    # with:
                    # by _time, source span=5m

                    tracker_new_search = re.sub(
                        r"by source",
                        r"by _time, source span=5m",
                        tracker_new_search,
                    )

                    # replace the sentence:
                    # | fields app, user, savedsearch_name, count_ess_notable, group, object, object_id
                    # with:
                    # | fields last_seen, app, user, savedsearch_name, count_ess_notable, group, object, object_id

                    tracker_new_search = re.sub(
                        r"\| fields app, user, savedsearch_name, count_ess_notable, group, object, object_id",
                        r"| fields _time, app, user, savedsearch_name, count_ess_notable, group, object, object_id",
                        tracker_new_search,
                    )

                    # replace the sentence:
                    # context="live"
                    # with:
                    # context="live" check_last_seen=True check_last_seen_field=last_seen_notable

                    tracker_new_search = re.sub(
                        r"context=\"live\"",
                        r"context=live check_last_seen=True check_last_seen_field=last_seen_notable",
                        tracker_new_search,
                    )

                    # update the search definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": tracker_name,
                        "report_search": tracker_new_search,
                        "earliest_time": "-20m",
                        "latest_time": "now",
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                        )

                # loop through the trackers (tracker)
                for tracker_shortname in trackers_to_process:
                    tracker_name = f"trackme_wlk_hybrid_{tracker_shortname}_tracker_tenant_{tenant_id}"

                    tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                    # get the current search definition
                    try:
                        tracker_current = service.saved_searches[tracker_name]
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                        )
                        continue

                    tracker_current_search = tracker_current.content.get("search")
                    tracker_account = tracker_kos["properties"][0]["account"]

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                    )

                    # replace the sentence:
                    # earliest="-10m" latest="now"
                    # with:
                    # earliest="-20m" latest="now"

                    tracker_new_search = re.sub(
                        r'earliest="-10m" latest="now"',
                        r'earliest="-20m" latest="now"',
                        tracker_current_search,
                    )

                    # update the search definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": tracker_name,
                        "report_search": tracker_new_search,
                        "earliest_time": "-20m",
                        "latest_time": "now",
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                        )

                #
                # enhance splunkcloud_svc execution to support deduplication
                #

                trackers_dict = {}
                trackers_to_process = []
                trackers_to_process_kos = []

                # get records from the KVstore
                trackers_records = collection_trackers.data.query()

                for tracker_record in trackers_records:
                    tracker_name = tracker_record.get("tracker_name")
                    tracker_kos = tracker_record.get("knowledge_objects")

                    # if the tracker_name starts by scheduler_
                    if re.search("^splunkcloud_svc_", tracker_name):
                        trackers_to_process.append(tracker_name)
                        trackers_to_process_kos.append(tracker_kos)

                        trackers_dict[tracker_name] = {
                            "knowledge_objects": json.loads(tracker_kos),
                        }

                # loop through the trackers (wrapper)
                for tracker_shortname in trackers_to_process:
                    tracker_name = f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"

                    tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                    # get the current search definition
                    try:
                        tracker_current = service.saved_searches[tracker_name]
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                        )
                        continue

                    tracker_current_search = tracker_current.content.get("search")
                    tracker_account = tracker_kos["properties"][0]["account"]

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                    )

                    # in tracker_current_search, replace the sentence: (if account is not local, double escape the quotes)
                    #  _index_earliest="-15m" _index_latest="now"
                    # with:
                    #  _index_earliest="-3h" _index_latest="now"

                    if tracker_account == "local":
                        tracker_new_search = re.sub(
                            r'_index_earliest="-15m" _index_latest="now"',
                            r'_index_earliest="-3h" _index_latest="now" earliest="--3h" latest="now"',
                            tracker_current_search,
                        )

                    else:
                        tracker_new_search = re.sub(
                            r'_index_earliest=\\"-15m\\" _index_latest=\\"now\\"',
                            r'_index_earliest=\\"-3h\\" _index_latest=\\"now\\" earliest=\\"-3h\\" latest=\\"now\\"',
                            tracker_current_search,
                        )

                    # replace the sentence:
                    # | fields app, group, object, object_id, savedsearch_name, user, svc_usage
                    # with:
                    # | fields _time, app, group, object, object_id, savedsearch_name, user, svc_usage

                    tracker_new_search = re.sub(
                        r"\| fields app, group, object, object_id, savedsearch_name, user, svc_usage",
                        r"| fields _time, app, group, object, object_id, savedsearch_name, user, svc_usage",
                        tracker_new_search,
                    )

                    # replace the sentence:
                    # context="live"
                    # with:
                    # context="live" check_last_seen=True check_last_seen_field=last_seen_splunkcloud_svc

                    tracker_new_search = re.sub(
                        r"context=\"live\"",
                        r"context=live check_last_seen=True check_last_seen_field=last_seen_splunkcloud_svc",
                        tracker_new_search,
                    )

                    # update the search definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": tracker_name,
                        "report_search": tracker_new_search,
                        "earliest_time": "-20m",
                        "latest_time": "now",
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                        )

                # Notes: updating trackers for splunkcloud_svc is not required

            #
            # ML Outliers: mass update rules for wlk
            #

            # set kwargs
            kwargs_wlk_outliers_rules_update = {
                "earliest_time": "-5m",
                "latest_time": "now",
                "output_mode": "json",
                "count": 0,
            }

            # define the gen search
            wlk_outliers_rules_update_search = remove_leading_spaces(
                f"""\
                | inputlookup trackme_wlk_outliers_entity_rules_tenant_{tenant_id} | eval keyid=_key
                ``` mass update the time_factor ```
                | where match(entities_outliers, "%H")
                | rex field=entities_outliers mode=sed "s/\\\"time_factor\\\": \\\"%H\\\"/\\\"time_factor\\\": \\\"none\\\"/g"
                ``` force update the confidence and confidence_reason ```
                | eval confidence="low", confidence_reason="upgrade to TrackMe 2.0.72 for splk-wlk requires models to be trained again due to the change of the time_factor field, which will happen automatically as soon as possible."
                | outputlookup append=t key_field=keyid trackme_wlk_outliers_entity_rules_tenant_{tenant_id}
                """
            )

            wlk_outliers_rules_update_counter = 0
            wlk_outliers_rules_updated_entities = []

            # run search

            # track the search runtime
            start = time.time()

            # proceed
            try:
                reader = run_splunk_search(
                    service,
                    wlk_outliers_rules_update_search,
                    kwargs_wlk_outliers_rules_update,
                    24,
                    5,
                )

                for item in reader:
                    if isinstance(item, dict):
                        wlk_outliers_rules_update_counter += 1
                        wlk_outliers_rules_updated_entities.append(item.get("object"))

            except Exception as e:
                msg = (
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", '
                    f'search failed with exception="{str(e)}", run_time="{str(time.time() - start)}"'
                )
                logging.error(msg)

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2075, procedure terminated'
    )
    return True


def trackme_schema_upgrade_2077(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2077, tenant_id="{tenant_id}"'
    )
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_cim_enabled") == 1:
            objects_to_process.append("trackme_cim")

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2077, procedure terminated'
    )
    return True


"""
In this function:
- Update the adaptive delay report to add the new option max_changes_past_7days
"""


def trackme_schema_upgrade_2078(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2078, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Adaptive trackers
        #

        components_to_process = []
        if (
            vtenant_record.get("tenant_dsm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dsm")
        if (
            vtenant_record.get("tenant_dhm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dhm")

        for component in components_to_process:
            #
            # Update the adaptive delay tracker
            #

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", processing with adaptive delay tracker update'
            )

            # report name
            tracker_name = (
                f"trackme_{component}_adaptive_delay_tracker_tenant_{tenant_id}"
            )

            # get the current search definition
            try:
                tracker_current = service.saved_searches[tracker_name]
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                )
                continue

            tracker_current_search = tracker_current.content.get("search")

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
            )

            # add our new parameter
            tracker_new_search = f"{tracker_current_search} max_changes_past_7days=10"

            # update the search definition
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": tracker_name,
                "report_search": tracker_new_search,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2078, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2078, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2078, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2078, procedure terminated'
    )
    return True


"""
In this function:
- Update the vtenant accounts to define the value for alias
- Update the logical group KVstore collection transforms
"""


def trackme_schema_upgrade_2083(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2083, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # prepare transforms updates
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # transforms updates
        #

        transform_name = f"trackme_common_logical_group_tenant_{tenant_id}"
        collection_name = f"kv_trackme_common_logical_group_tenant_{tenant_id}"
        transform_fields = collections_dict["trackme_common_logical_group"]
        transform_acl = {
            "owner": vtenant_record.get("tenant_owner"),
            "sharing": trackme_default_sharing,
            "perms.write": tenant_roles_write_perms,
            "perms.read": tenant_roles_read_perms,
        }

        # delete the transform
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
        data = {
            "tenant_id": tenant_id,
            "transform_name": transform_name,
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2083, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2083, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2083, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
            )

        # create the transform
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
        data = {
            "tenant_id": tenant_id,
            "transform_name": transform_name,
            "transform_fields": transform_fields,
            "collection_name": collection_name,
            "transform_acl": transform_acl,
            "owner": vtenant_record.get("tenant_owner"),
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2083, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2083, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2083, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
            )

        #
        # END
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2083, procedure terminated'
    )
    return True


"""
In this function:
- Run a RBAC force check and update to ensure permissions of the tenant are correctly set (due to a bug affecting previous updates)
- Create the new dsm tags KVstore collection
- Reflects updates the splk-dsm main kvstore transforms
- Update splk-dsm abstract trackers to include global_dcount_host
- Migrate all ML models to splunk-system-user ownership, remove orphans models
- Ensures that the common trackme_common_replica_trackers Kvcollection and transforms have been created for the tenant
"""


def trackme_schema_upgrade_2084(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2084, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # RBAC force check and update
        #

        tenant_id = vtenant_record.get("tenant_id")
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/vtenants/admin/update_tenant_rbac'
        data = {
            "tenant_id": {tenant_id},
            "tenant_roles_admin": vtenant_record.get("tenant_roles_admin"),
            "tenant_roles_power": vtenant_record.get("tenant_roles_user"),
            "tenant_roles_user": vtenant_record.get("tenant_roles_power"),
            "tenant_owner": vtenant_record.get("tenant_owner"),
        }

        # update the account
        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=data,
                verify=False,
                timeout=1800,  # 30 minutes timeout, on extremely large env, this can take a while
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, RBAC check and update has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, RBAC check and update was operated successfully, response.status_code="{response.status_code}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, RBAC check and update has failed, exception="{str(e)}"'
            )

        #
        # dsm_tags KVstore collection
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:

            # get permissions
            tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
                vtenant_record
            )

            # TrackMe sharing level
            trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
                "trackme_default_sharing"
            ]

            # set
            transform_name = f"trackme_dsm_tags_tenant_{tenant_id}"
            collection_name = f"kv_trackme_dsm_tags_tenant_{tenant_id}"
            transform_fields = collections_dict["trackme_dsm_tags"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the KVstore collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            kvstore_created = False
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                    )
                    kvstore_created = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # continue
            if kvstore_created:

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the tracker report
            report_name = f"trackme_dsm_tags_tracker_tenant_{tenant_id}"
            report_search = f'| trackmesplktags tenant_id="{tenant_id}"'
            report_properties = {
                "description": "This scheduled report applies and maintains tags policies for the splk-dsm component",
                "is_scheduled": True,
                "cron_schedule": "*/15 * * * *",
                "dispatch.earliest_time": "-5m",
                "dispatch.latest_time": "now",
                "schedule_window": "5",
            }

            # create the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
                "report_search": report_search,
                "report_properties": report_properties,
                "report_acl": report_acl,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                )

            # Execute tags update now
            # trackme-limited/trackme-report-issues#840: this operation is not required anymore

        #
        # Update splk-dsm transform
        #

        # check components and add accordingly
        objects_to_process = []
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm")

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # Update splk-dsm abstract trackers to include global_dcount_host
        #

        if vtenant_record.get("tenant_dsm_enabled") == 1:

            # retrieve the list of scheduler reports to be processed
            collection_trackers_name = (
                f"kv_trackme_dsm_hybrid_trackers_tenant_{tenant_id}"
            )
            collection_trackers = service.kvstore[collection_trackers_name]

            trackers_abstract_to_process = []
            trackers_wrapper_to_process = []

            # get records from the KVstore
            trackers_records = collection_trackers.data.query()

            for tracker_record in trackers_records:

                tracker_name = tracker_record.get("tracker_name")
                tracker_kos = json.loads(tracker_record.get("knowledge_objects"))

                tracker_reports = tracker_kos.get("reports")

                for tracker_report in tracker_reports:
                    # if the tracker_name contains the word "_abstract_":
                    if "_abstract_" in tracker_report:
                        trackers_abstract_to_process.append(tracker_report)
                    # if the tracker_name contains the word "_wrapper_":
                    elif "_wrapper_" in tracker_report:
                        trackers_wrapper_to_process.append(tracker_report)

            # loop through abstract reports
            for tracker_name in trackers_abstract_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # in tracker_current_search, replace the sentence:
                #  max(data_last_time_seen) as data_last_time_seen by
                # with:
                #  max(data_last_time_seen) as data_last_time_seen, max(dcount_host) as global_dcount_host by

                tracker_new_search = re.sub(
                    r"max\(data_last_time_seen\) as data_last_time_seen by",
                    r"max(data_last_time_seen) as data_last_time_seen, max(dcount_host) as global_dcount_host by",
                    tracker_current_search,
                )

                # in tracker_current_search, replace the sentence:
                # sum(data_eventcount) as data_eventcount by
                # with:
                # sum(data_eventcount) as data_eventcount, first(global_dcount_host) as global_dcount_host

                tracker_new_search = re.sub(
                    r"sum\(data_eventcount\) as data_eventcount by",
                    r"sum(data_eventcount) as data_eventcount, first(global_dcount_host) as global_dcount_host by",
                    tracker_new_search,
                )

                # in tracker_current_search, replace the sentence:
                # eval dcount_host=round(latest_dcount_host_5m, 2)
                # with:
                # eval dcount_host=round(global_dcount_host, 0)

                tracker_new_search = re.sub(
                    r"eval dcount_host=round\(latest_dcount_host_5m, 2\)",
                    r"eval dcount_host=round(global_dcount_host, 0)",
                    tracker_new_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

            # loop through wrapper reports
            for tracker_name in trackers_wrapper_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]                
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # in tracker_current_search, replace the sentence:
                # metric_name:trackme.splk.feeds.stdev_dcount_host_5m=stdev_dcount_host_5m,
                # with:
                # metric_name:trackme.splk.feeds.stdev_dcount_host_5m=stdev_dcount_host_5m, metric_name:trackme.splk.feeds.global_dcount_host=global_dcount_host,

                tracker_new_search = re.sub(
                    r"metric_name:trackme.splk.feeds.stdev_dcount_host_5m=stdev_dcount_host_5m,",
                    r"metric_name:trackme.splk.feeds.stdev_dcount_host_5m=stdev_dcount_host_5m, metric_name:trackme.splk.feeds.global_dcount_host=global_dcount_host,",
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # Migrate all ML models to splunk-system-user ownership, remove orphans models
        #

        """
        Get all records from an ML rules collection.

        :param collection: The collection to query.
        :return: A list of records, a dictionary of records, a list of keys.
        """

        def get_ml_rules_collection(collection):
            collection_records = []
            collection_records_dict = {}
            count_to_process_list = []

            end = False
            skip_tracker = 0
            while not end:
                process_collection_records = collection.data.query(skip=skip_tracker)
                if process_collection_records:
                    for item in process_collection_records:
                        collection_records.append(item)
                        collection_records_dict[item.get("_key")] = (
                            item  # Add the entire item to the dictionary
                        )
                        count_to_process_list.append(item.get("_key"))
                    skip_tracker += 5000
                else:
                    end = True

            return collection_records, collection_records_dict, count_to_process_list

        """
        Removes an orphan Machine Learning model from the collection.

        :param component: The component name.
        :param rest_url: The REST URL to use.
        :param header: The header to use.
        :param ml_model_lookup_name: The Machine Learning model lookup name.
        :return: True if the model was removed successfully, otherwise False.
        
        """

        def remove_ml_model(component, rest_url, header, ml_model_lookup_name):

            logging.info(
                f'component="{component}", context="inspect_collection:outliers", attempting to delete orphan Machine Learning lookup_name="{ml_model_lookup_name}"'
            )
            try:
                response = requests.delete(
                    rest_url,
                    headers=header,
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (
                    200,
                    201,
                    204,
                ):
                    error_msg = f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", failure to delete ML lookup_name="{ml_model_lookup_name}", url="{rest_url}", response.status_code="{response.status_code}", response.text="{response.text}"'
                    raise Exception(error_msg)
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", action="success", deleted lookup_name="{ml_model_lookup_name}" successfully'
                    )
                    return True

            except Exception as e:
                error_msg = f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", failure to delete ML lookup_name="{ml_model_lookup_name}" with exception="{str(e)}"'
                raise Exception(error_msg)

        """
        Reasign a Machine Learning model to the Splunk system user.

        :param model_id: The model_id to reassign.
        :param rest_url: The REST URL to use.
        :param header: The header to use.

        :return: True if the model was reassigned successfully, otherwise False.

        """

        def reassign_ml_model(model_id, rest_url, header):

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", attempting to re-assign model_id="{model_id}" to splunk-system-user'
            )

            acl_properties = {
                "sharing": "user",
                "owner": "splunk-system-user",
            }

            # proceed boolean
            proceed = False

            # before re-assigning, check if the model exist by running a GET request, if the status code is different from 2**, do not proceed and log an informational message instead
            try:
                response = requests.get(
                    f"{rest_url}",
                    headers=header,
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", model_id="{model_id}" does not exist, it might have been re-assigned in the meantime, skipping re-assignment'
                    )
                    return False
                else:
                    proceed = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", model_id="{model_id}" failed to retrieve model, exception="{str(e)}"'
                )

            if proceed:

                # re-assign post
                try:
                    response = requests.post(
                        f"{rest_url}/acl",
                        headers=header,
                        data=acl_properties,
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (
                        200,
                        201,
                        204,
                    ):
                        error_msg = f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", failure to reassign model_id="{model_id}", url="{rest_url}", response.status_code="{response.status_code}", response.text="{response.text}"'
                        raise Exception(error_msg)
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", action="success", model_id="{model_id}" reassigned successfully'
                        )
                        return True

                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", action="failure", model_id="{model_id}" reassigned failed, exception="{str(e)}"'
                    )
                    raise Exception(str(e))

        # get all vtenants records, this job is not tenant specific
        vtenant_records = collection_vtenants.data.query()

        # A list to store ml_rules_outliers_collections
        ml_rules_outliers_collections = []

        # A dict to ml models definitions
        ml_models_dict = {}

        # A list to store ml models currently configured
        ml_models_list = []

        for vtenant_record in vtenant_records:
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", processing vtenant_record={json.dumps(vtenant_record, indent=2)}'
            )

            # get the tenant_id
            tenant_id = vtenant_record.get("tenant_id")

            # for component in dsm, dhm, cim, flx
            for component in ["dsm", "dhm", "cim", "flx"]:

                # get status
                component_status = vtenant_record.get(f"tenant_{component}_enabled")

                # append the collection
                if component_status == 1:
                    ml_rules_outliers_collections.append(
                        f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}"
                    )

        # for each outliers rules collection
        for ml_rules_collection_name in ml_rules_outliers_collections:

            # connect to the collection service and retrieve the records
            ml_rules_collection = service.kvstore[ml_rules_collection_name]

            # extract ml_rules_tenant_id from the collection name: trackme_<component>_outliers_entity_rules_tenant_<ml_rules_tenant_id>
            ml_rules_tenant_id = ml_rules_collection_name.split("_")[-1]

            # get records
            try:
                ml_rules_records, ml_rules_records_dict, ml_rules_records_count = (
                    get_ml_rules_collection(ml_rules_collection)
                )

                for ml_rules_record in ml_rules_records:

                    # get key
                    ml_rules_record_key = ml_rules_record.get("_key")

                    # get dictionnary entities_outliers from the field entities_outliers
                    entities_outliers = json.loads(
                        ml_rules_record.get("entities_outliers")
                    )

                    # loop trough entities_outliers, the dict key is the model_id
                    for ml_model_entity in entities_outliers:

                        ml_models_dict[ml_model_entity] = {
                            "model_id": ml_model_entity,
                            "collection_name": ml_rules_collection_name,
                            "collection_key": ml_rules_record_key,
                            "tenant_id": ml_rules_tenant_id,
                        }
                        ml_models_list.append(
                            f"__mlspl_{ml_model_entity}.mlmodel"
                        )  # this is the filename

            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", failed to retrieve the records from the collection, collection_name="{ml_rules_collection_name}", exception="{str(e)}"'
                )

            # log
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", {len(ml_models_dict)} ML models were found configured in TrackMe collections, will now start inspecting Splunk existing models.'
            )

        # run the following search to retrieve the list of existing ML models

        # Define the query
        search = f'| rest splunk_server=local timeout=1200 "/servicesNS/nobody/trackme/data/lookup-table-files" | search eai:acl.app="trackme" AND title="__mlspl_model_*.mlmodel" | search [ | `splk_outliers_get_model_files_for_tenant({tenant_id},{component})` | rename ml_model_filename as title | table title | format ] | table title, id'

        kwargs_oneshot = {
            "earliest_time": "-5m",
            "latest_time": "now",
            "output_mode": "json",
            "count": 0,
        }

        # A list to store current ml models (filename)
        ml_models_current_list = []

        # A dict to store the existing models
        ml_models_dict_existing = {}

        try:
            reader = run_splunk_search(
                service,
                search,
                kwargs_oneshot,
                24,
                5,
            )

            for item in reader:
                if isinstance(item, dict):
                    ml_models_current_list.append(
                        item.get("title")
                    )  # this is the model filename
                    ml_models_dict_existing[item.get("title")] = {"id": item.get("id")}

        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", failed to retrieve the list of ML models, exception="{str(e)}"'
            )

        # log the number of currently existing models
        logging.info(
            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, subcontext="mlmodels-migration", {len(ml_models_current_list)} ML models were found in the system, starting orphan models inspection and migration'
        )

        #
        # orphan models purge / reassign
        #

        ml_models_purged_success_count = 0
        ml_models_purged_failures_count = 0
        ml_models_reassigned_success_count = 0
        ml_models_reassigned_failures_count = 0

        header = {
            "Authorization": "Splunk %s" % reqinfo["session_key"],
            "Content-Type": "application/json",
        }

        # for each model in ml_models_current_list, if the model is not in ml_models_list, delete it
        for model_id in ml_models_current_list:
            if model_id != "pending":
                if model_id not in ml_models_list:
                    # remove the model
                    rest_url = ml_models_dict_existing[model_id].get("id")

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="mlmodels-migration", attempting removal of model_id={model_id}"'
                    )

                    try:
                        remove_ml_model("trackme", rest_url, header, model_id)
                        ml_models_purged_success_count += 1
                    except Exception as e:
                        ml_models_purged_failures_count += 1
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="mlmodels-migration", failed to remove the orphan model, model_id="{model_id}", exception="{str(e)}"'
                        )

                elif model_id in ml_models_list:
                    # reassign the model
                    rest_url = ml_models_dict_existing[model_id].get("id")

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="mlmodels-migration", attempting reassignment of model_id={model_id}"'
                    )

                    try:
                        reassigned_model = reassign_ml_model(
                            model_id,
                            rest_url,
                            header,
                        )
                        if reassigned_model:
                            ml_models_reassigned_success_count += 1
                    except Exception as e:
                        ml_models_reassigned_failures_count += 1
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="mlmodels-migration", failed to reassign the model, model_id="{model_id}", exception="{str(e)}"'
                        )

        # log
        logging.info(
            f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="mlmodels-migration", {ml_models_purged_success_count} orphan ML models were removed, {ml_models_purged_failures_count} orphan ML models removals failed, {ml_models_reassigned_success_count} ML models were reassigned to splunk-system-user, {ml_models_reassigned_failures_count} ML models reassignments failed'
        )

        """
        Ensures that the common trackme_common_replica_trackers Kvcollection and transforms have been created for the tenant            
        """

        # proceed boolean
        proceed = False

        # first, check if we have a transforms existing for the tenant, if not we need to proceed, otherwise we have nothing to do

        # transform name
        transform_name = f"trackme_common_replica_trackers_tenant_{tenant_id}"

        # check first if the transforms exists, if it does not exist, we do not need to proceed
        try:
            transform_current = service.confs["transforms"][transform_name]
            proceed = False
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", transform does exist, we do not need to proceed, transform="{transform_name}"'
            )
        except Exception as e:
            proceed = True
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", transform does not exist, we need to proceed, transform="{transform_name}"'
            )

        if proceed:

            # get permissions
            tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
                vtenant_record
            )

            # TrackMe sharing level
            trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
                "trackme_default_sharing"
            ]

            # set
            transform_name = f"trackme_common_replica_trackers_tenant_{tenant_id}"
            collection_name = f"kv_trackme_common_replica_trackers_tenant_{tenant_id}"
            transform_fields = collections_dict["trackme_common_replica_trackers"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the KVstore collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            kvstore_created = False
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                    )
                    kvstore_created = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # continue
            if kvstore_created:

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2084, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

        #
        # END
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2084, procedure terminated'
    )
    return True


"""
In this function:
- Decomission the datagen KVstore
- Address an issue with allowlist for splk-dhm/splk-mhm where the is_rex field was missing in the transforms definition
"""


def trackme_schema_upgrade_2087(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2087, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Decomission datagen KVstore
        #

        transform_name = f"trackme_common_datagen_cache_tenant_{tenant_id}"
        collection_name = f"kv_{transform_name}"

        # delete the transform
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
        data = {
            "tenant_id": tenant_id,
            "transform_name": transform_name,
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
            )

        # delete the KVstore collection
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvcollection'
        data = {
            "tenant_id": tenant_id,
            "collection_name": collection_name,
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", failed to delete the KVstore collection, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", successfully deleted KVstore collection, collection="{collection_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", failed to delete the KVstore collection collection="{collection_name}", exception="{str(e)}"'
            )

        #
        # Update splk-dhm/mhm allowlists transform
        #

        # check components and add accordingly
        objects_to_process = []

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append("dhm")

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            objects_to_process.append("mhm")

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        for component_toprocess in objects_to_process:
            transform_name = (
                f"trackme_{component_toprocess}_allowlist_tenant_{tenant_id}"
            )
            collection_name = f"kv_{transform_name}"
            transform_fields = collections_dict[
                f"trackme_{component_toprocess}_allowlist"
            ]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2087, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # END
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2087, procedure terminated'
    )
    return True


"""
In this function:
- Update main KVstore transforms definitions to include a new field "ctime" (creation time)
- For dsm/dhm/mhm, force an update of blocklists, if any, so we convert wildcards non regex to regex via the API automatically
"""


def trackme_schema_upgrade_2089(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2089, tenant_id="{tenant_id}"'
    )
    objects_to_process = []

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        #
        # Update main KVstore transforms definitions to include a new field "ctime" (creation time)
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            objects_to_process.append("trackme_dsm")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            objects_to_process.append("trackme_dhm")

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            objects_to_process.append("trackme_mhm")

        if vtenant_record.get("tenant_cim_enabled") == 1:
            objects_to_process.append("trackme_cim")

        if vtenant_record.get("tenant_flx_enabled") == 1:
            objects_to_process.append("trackme_flx")

        if vtenant_record.get("tenant_wlk_enabled") == 1:
            objects_to_process.append("trackme_wlk")

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        for object_name in objects_to_process:
            transform_name = "%s_tenant_%s" % (object_name, tenant_id)
            collection_name = "kv_%s_tenant_%s" % (object_name, tenant_id)
            transform_fields = collections_dict[object_name]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": "app",
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # For dsm/dhm/mhm, force an update of blocklists, if any, so we convert wildcards non regex to regex via the API automatically
        #

        components_suffix_list = []

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_suffix_list.append("dsm")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_suffix_list.append("dhm")

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_suffix_list.append("mhm")

        # for each component suffix
        for component_suffix in components_suffix_list:

            # entities KV collection
            blocklist_collection_name = (
                f"kv_trackme_{component_suffix}_allowlist_tenant_{tenant_id}"
            )
            blocklist_collection = service.kvstore[blocklist_collection_name]

            # get records
            blocklist_records, blocklist_collection_keys, blocklist_collection_dict = (
                get_kv_collection(blocklist_collection, blocklist_collection_name)
            )

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", blocklist_records_count="{len(blocklist_records)}"'
            )

            if len(blocklist_records) > 0:

                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/splk_blocklist/write/blocklist_update'
                data = {
                    "tenant_id": f"{tenant_id}",
                    "component": f"{component_suffix}",
                    "records_list": json.dumps(blocklist_records),
                    "update_comment": "TrackMe schema upgrade for version 2.0.89",
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", call to blocklist_update, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully called blocklist_update, component="{component_suffix}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to call to blocklist_update, component="{component_suffix}", exception="{str(e)}"'
                    )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2089, procedure terminated'
    )
    return True


"""
In this function:
- Create the new priority collection for dsm/dhm/mhm/flx/wlk
"""


def trackme_schema_upgrade_2090(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2090, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        #
        # For dsm/dhm/mhm, force an update of blocklists, if any, so we convert wildcards non regex to regex via the API automatically
        #

        components_suffix_list = []

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_suffix_list.append("dsm")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_suffix_list.append("dhm")

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_suffix_list.append("mhm")

        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_suffix_list.append("flx")

        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_suffix_list.append("wlk")

        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_suffix_list.append("cim")

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        # for each component suffix
        for component_suffix in components_suffix_list:

            #
            # priority policies collection
            #

            # set
            transform_name = (
                f"trackme_{component_suffix}_priority_policies_tenant_{tenant_id}"
            )
            collection_name = (
                f"kv_trackme_{component_suffix}_priority_policies_tenant_{tenant_id}"
            )
            transform_fields = collections_dict[
                f"trackme_{component_suffix}_priority_policies"
            ]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the KVstore collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            kvstore_created = False
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                    )
                    kvstore_created = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # continue
            if kvstore_created:

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

            #
            # priority collection
            #

            # set
            transform_name = f"trackme_{component_suffix}_priority_tenant_{tenant_id}"
            collection_name = (
                f"kv_trackme_{component_suffix}_priority_tenant_{tenant_id}"
            )
            transform_fields = collections_dict[f"trackme_{component_suffix}_priority"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the KVstore collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            kvstore_created = False
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                    )
                    kvstore_created = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # continue
            if kvstore_created:

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

            # create the tracker report
            report_name = (
                f"trackme_{component_suffix}_priority_tracker_tenant_{tenant_id}"
            )
            report_search = f'| trackmesplkpriority tenant_id="{tenant_id}" component={component_suffix}'
            report_properties = {
                "description": f"This scheduled report applies and maintains priority policies for the splk-{component_suffix} component",
                "is_scheduled": True,
                "cron_schedule": "*/15 * * * *",
                "dispatch.earliest_time": "-5m",
                "dispatch.latest_time": "now",
                "schedule_window": "5",
            }
            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": "app",
                "perms.write": str(vtenant_record.get("tenant_roles_admin")),
                "perms.read": str(tenant_roles_read_perms),
            }

            # create the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
                "report_search": report_search,
                "report_properties": report_properties,
                "report_acl": report_acl,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2090, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2090, procedure terminated'
    )
    return True


"""
In this function:
- Update main collections to include the sla_class per entity
- Create the new sla collection for dsm/dhm/mhm/flx/wlk
"""


def trackme_schema_upgrade_2091(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2091, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        components_suffix_list = []

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_suffix_list.append("dsm")

        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_suffix_list.append("dhm")

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_suffix_list.append("mhm")

        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_suffix_list.append("flx")

        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_suffix_list.append("wlk")

        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_suffix_list.append("cim")

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        # for each component suffix
        for component_suffix in components_suffix_list:

            #
            # main transform collection
            #

            # set
            transform_name = f"trackme_{component_suffix}_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component_suffix}_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component_suffix}"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            #
            # sla policies collection
            #

            # set
            transform_name = (
                f"trackme_{component_suffix}_sla_policies_tenant_{tenant_id}"
            )
            collection_name = (
                f"kv_trackme_{component_suffix}_sla_policies_tenant_{tenant_id}"
            )
            transform_fields = collections_dict[
                f"trackme_{component_suffix}_sla_policies"
            ]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the KVstore collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            kvstore_created = False
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                    )
                    kvstore_created = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # continue
            if kvstore_created:

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

            #
            # sla collection
            #

            # set
            transform_name = f"trackme_{component_suffix}_sla_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component_suffix}_sla_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component_suffix}_sla"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the KVstore collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            kvstore_created = False
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                    )
                    kvstore_created = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # continue
            if kvstore_created:

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

            # create the tracker report
            report_name = f"trackme_{component_suffix}_sla_tracker_tenant_{tenant_id}"
            report_search = f'| trackmesplkslaclass tenant_id="{tenant_id}" component={component_suffix}'
            report_properties = {
                "description": f"This scheduled report applies and maintains SLA policies for the splk-{component_suffix} component",
                "is_scheduled": True,
                "cron_schedule": "*/15 * * * *",
                "dispatch.earliest_time": "-5m",
                "dispatch.latest_time": "now",
                "schedule_window": "5",
            }
            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": "app",
                "perms.write": str(vtenant_record.get("tenant_roles_admin")),
                "perms.read": str(tenant_roles_read_perms),
            }

            # create the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
                "report_search": report_search,
                "report_properties": report_properties,
                "report_acl": report_acl,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2091, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2091, procedure terminated'
    )
    return True


"""
In this function:
- Update cim main collections to include the object_category
"""


def trackme_schema_upgrade_2093(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2093, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        components_suffix_list = []

        # check components and add accordingly
        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_suffix_list.append("cim")

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        # for each component suffix
        for component_suffix in components_suffix_list:

            #
            # main transform collection
            #

            # set
            transform_name = f"trackme_{component_suffix}_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component_suffix}_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component_suffix}"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2093, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2093, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2093, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2093, procedure terminated'
    )
    return True


"""
In this function:
- Update the Workload main KVstore lookup transforms to include scheduler related fields (issue#631)
"""


def trackme_schema_upgrade_2094(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2094, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        components_suffix_list = []

        # check components and add accordingly
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_suffix_list.append("wlk")

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        # for each component suffix
        for component_suffix in components_suffix_list:

            #
            # main transform collection
            #

            # set
            transform_name = f"trackme_{component_suffix}_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component_suffix}_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component_suffix}"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2094, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2094, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2094, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2094, procedure terminated'
    )
    return True


"""
In this function:
- Update the Virtual Tenant account preferences to include the default_priority parameter
- Create the KVstore collection trackme_common_smartstatus_alert_action_last_seen_activity
"""


def trackme_schema_upgrade_2095(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2095, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Vtenant account preferences
        #

        # get macros
        macros = service.confs["macros"]

        # get the
        try:
            macro_default_priority = macros["trackme_default_priority"].content[
                "definition"
            ]

            # the macro definition will be as:
            """
            eval priority=if(isnull(priority), "medium", priority)
            """

            # use a regex expression to extract the value of the priority level (low, medium, high, critical, pending), if we fail, we should assign medium
            macro_default_priority_level = re.search(
                r'eval\s+priority=if\(isnull\(priority\),\s+"(low|medium|high|critical|pending)",\s+priority\)',
                macro_default_priority,
            ).group(1)

        except Exception as e:
            logging.error(
                f'tenant_id="{tenant_id}", failed to retrieve the current definition for the macro trackme_default_priority, will assign default to medium, exception={str(e)}'
            )
            macro_default_priority_level = "medium"

        # check, valid options are low, medium, high, critical, pending
        if macro_default_priority_level not in [
            "low",
            "medium",
            "high",
            "critical",
            "pending",
        ]:
            logging.error(
                f'tenant_id="{tenant_id}", the macro trackme_default_priority has an invalid value="{macro_default_priority_level}", will assign default to medium'
            )
            macro_default_priority_level = "medium"

        #
        # Update the vtenant account configuration
        #

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Components updates
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        components_suffix_list = []

        # check components and add accordingly
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_suffix_list.append("wlk")

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # main transform collection
        #

        # set
        transform_name = f"trackme_common_smartstatus_alert_action_last_seen_activity_tenant_{tenant_id}"
        collection_name = f"kv_trackme_common_smartstatus_alert_action_last_seen_activity_tenant_{tenant_id}"
        transform_fields = collections_dict[
            f"trackme_common_smartstatus_alert_action_last_seen_activity"
        ]
        ko_acl = {
            "owner": vtenant_record.get("tenant_owner"),
            "sharing": trackme_default_sharing,
            "perms.write": tenant_roles_write_perms,
            "perms.read": tenant_roles_read_perms,
        }

        # create the KVstore collection
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
        data = {
            "tenant_id": tenant_id,
            "collection_name": collection_name,
            "collection_acl": ko_acl,
            "owner": vtenant_record.get("tenant_owner"),
        }

        kvstore_created = False
        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                )
                kvstore_created = True
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
            )

        #
        # Only continue if successful
        #

        if kvstore_created:

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2095, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2095, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2095, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2095, procedure terminated'
    )
    return True


"""
In this function:
- Update the Virtual Tenant account preferences to include pagination_mode and pagination_size parameters
- Uodate the Adaptive delay tracker to include the review_period_no_days parameter
"""


def trackme_schema_upgrade_2096(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2096, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        # get macros
        pagination_mode = reqinfo["trackme_conf"]["trackme_general"]["pagination_mode"]
        pagination_size = reqinfo["trackme_conf"]["trackme_general"]["pagination_size"]

        #
        # Update the vtenant account configuration
        #

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            {"pagination_mode": pagination_mode, "pagination_size": pagination_size},
        )

        #
        # Adaptive delay tracker update
        #

        components_to_process = []
        if (
            vtenant_record.get("tenant_dsm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dsm")
        if (
            vtenant_record.get("tenant_dhm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dhm")

        for component in components_to_process:
            #
            # Update the adaptive delay tracker
            #

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", processing with adaptive delay tracker update'
            )

            # report name
            tracker_name = (
                f"trackme_{component}_adaptive_delay_tracker_tenant_{tenant_id}"
            )

            # get the current search definition
            try:
                tracker_current = service.saved_searches[tracker_name]
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                )
                continue

            tracker_current_search = tracker_current.content.get("search")

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
            )

            # add our new parameter
            tracker_new_search = f"{tracker_current_search} review_period_no_days=30"

            # update the search definition
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": tracker_name,
                "report_search": tracker_new_search,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2096, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2096, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2096, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2096, procedure terminated'
    )
    return True


"""
In this function:
- Update the Virtual Tenant account preferences to include ui_default_timerange parameter
- Update registered TrackMe alerts to call the new macro for applying the maintenance mode
- Update tenant Ack transforms definition
"""


def trackme_schema_upgrade_2097(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2097, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Purge any TrackMe entities containing backslashes in the object value
        #

        components_to_process = []
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")

        for component in components_to_process:

            # Define the query
            search = f'| inputlookup trackme_wlk_tenant_{tenant_id} | eval key = _key | where match(object, "\\\\\\\\") | table key, object'

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", component="{component}", searching for non encoded backslash entities, search="{search}"'
            )

            kwargs_oneshot = {
                "earliest_time": "-5m",
                "latest_time": "now",
                "output_mode": "json",
                "count": 0,
            }

            # A list to store all executions
            results_list = []

            # A dict to store the results
            results_dict = {}

            try:
                reader = run_splunk_search(
                    service,
                    search,
                    kwargs_oneshot,
                    24,
                    5,
                )

                for item in reader:
                    if isinstance(item, dict):
                        results_dict[item["key"]] = item["object"]
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", component="{component}", detected non encoded backslash entity to be purged, object="{item["object"]}", key="{item["key"]}"'
                        )

            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", component="{component}", could not verify the presence of non encoded backslash entities, search permanently failed, exception="{str(e)}"'
                )

            # if the results_dict is not empty, we need to purge the entities
            if results_dict:
                # Data collection
                collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
                collection = service.kvstore[collection_name]

                # loop through the results dict and remove any entity from the KVstore
                for key, object_value in results_dict.items():
                    try:
                        collection.data.delete(json.dumps({"_key": key}))
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", component="{component}", successfully purged non encoded backslash entity, object="{object_value}", key="{key}"'
                        )
                        results_list.append(
                            {
                                "object": object_value,
                                "key": key,
                                "result": "successfully purged non encoded backslash entity",
                            }
                        )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", component="{component}", failed to purge non encoded backslash entity, object="{object_value}", key="{key}", exception="{str(e)}"'
                        )
                        results_list.append(
                            {
                                "object": object_value,
                                "key": key,
                                "result": "failure to purge non encoded backslash entity",
                                "exception": str(e),
                            }
                        )

            # log a message if there were no entities to be purged
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", component="{component}", no non encoded backslash entities to be purged'
                )

        #
        # Update the vtenant account configuration
        #

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update alerts
        #

        # get alert object from field tenant_alert_objects
        alert_objects = vtenant_record.get("tenant_alert_objects", None)

        # check if we have alert objects
        if alert_objects:

            # load the alert objects as a json object
            try:
                alert_objects = json.loads(alert_objects)
            except Exception as e:
                # log alerts to be updated
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", no alerts found for update.'
                )

            # get list of alerts
            alerts_list = alert_objects.get("alerts", [])

            # for each alert
            for alert_name in alerts_list:

                alert_current = None
                try:
                    alert_current = service.saved_searches[alert_name]
                except Exception as e:
                    pass

                if alert_current:
                    alert_current_search = alert_current.content.get("search")
                    alert_current_earliest_time = alert_current.content.get(
                        "dispatch.earliest_time"
                    )
                    alert_current_latest_time = alert_current.content.get(
                        "dispatch.latest_time"
                    )

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", alert_name="{alert_name}", alert_current_search="{alert_current_search}"'
                    )

                    """
                    in alert_current_search, replace the following sentence using regex:

                    appendcols [ | inputlookup trackme_maintenance_mode ] | filldown maintenance_mode | where NOT maintenance_mode="enabled"

                    by:

                    `trackme_apply_maintenance_mode`

                    """

                    alert_new_search = re.sub(
                        r'appendcols \[ \| inputlookup trackme_maintenance_mode \] \| filldown maintenance_mode \| where NOT maintenance_mode="enabled"',
                        r"`trackme_apply_maintenance_mode`",
                        alert_current_search,
                    )

                    # update the alert definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": alert_name,
                        "report_search": alert_new_search,
                        "earliest_time": alert_current_earliest_time,
                        "latest_time": alert_current_latest_time,
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", update alert definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", successfully updated alert definition, alert="{alert_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", failed to update the alert definition, report="{alert_name}", exception="{str(e)}"'
                        )

        #
        # Update Ack transforms definition
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        # set
        transform_name = f"trackme_common_alerts_ack_tenant_{tenant_id}"
        collection_name = f"kv_trackme_common_alerts_ack_tenant_{tenant_id}"
        transform_fields = collections_dict["trackme_common_alerts_ack"]
        transform_acl = {
            "owner": vtenant_record.get("tenant_owner"),
            "sharing": trackme_default_sharing,
            "perms.write": tenant_roles_write_perms,
            "perms.read": tenant_roles_read_perms,
        }

        # delete the transform
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
        data = {
            "tenant_id": tenant_id,
            "transform_name": transform_name,
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
            )

        # create the transform
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
        data = {
            "tenant_id": tenant_id,
            "transform_name": transform_name,
            "transform_fields": transform_fields,
            "collection_name": collection_name,
            "transform_acl": transform_acl,
            "owner": vtenant_record.get("tenant_owner"),
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2097, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
            )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2097, procedure terminated'
    )
    return True


"""
In this function:
 - Update the Virtual Tenant account preferences to include per component Tabulator groupBy
 - Extend the tags feature to all components, in addition with initial dsm only scope
 """


def trackme_schema_upgrade_2098(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2098, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the vtenant account configuration
        #

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Create tags tracker for newly eligible components
        #

        components_to_process = []

        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")
        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_to_process.append("cim")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")

        for component in components_to_process:

            #
            # update main collection transforms
            #

            # set
            transform_name = f"trackme_{component}_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            #
            # Create the tags collection and transforms (note for dsm, this collection already exists)
            #

            if component != "dsm":

                # set
                transform_name = f"trackme_{component}_tags_tenant_{tenant_id}"
                collection_name = f"kv_trackme_{component}_tags_tenant_{tenant_id}"
                transform_fields = collections_dict[f"trackme_{component}_tags"]
                ko_acl = {
                    "owner": vtenant_record.get("tenant_owner"),
                    "sharing": trackme_default_sharing,
                    "perms.write": tenant_roles_write_perms,
                    "perms.read": tenant_roles_read_perms,
                }

                # create the KVstore collection
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
                data = {
                    "tenant_id": tenant_id,
                    "collection_name": collection_name,
                    "collection_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                kvstore_created = False
                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                        )
                        kvstore_created = True
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                    )

                # continue
                if kvstore_created:

                    # create the transform
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                    data = {
                        "tenant_id": tenant_id,
                        "transform_name": transform_name,
                        "transform_fields": transform_fields,
                        "collection_name": collection_name,
                        "transform_acl": ko_acl,
                        "owner": vtenant_record.get("tenant_owner"),
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                        )

            #
            # Create the tags policies collection and transforms (for all components)
            #

            # set
            transform_name = f"trackme_{component}_tags_policies_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_tags_policies_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}_tags"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the KVstore collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            kvstore_created = False
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                    )
                    kvstore_created = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # continue
            if kvstore_created:

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

            #
            # Create the report tracker (not for dsm, as it already exists)
            #

            if component != "dsm":

                report_acl = {
                    "owner": str(vtenant_record.get("tenant_owner")),
                    "sharing": "app",
                    "perms.write": tenant_roles_write_perms,
                    "perms.read": tenant_roles_read_perms,
                }

                # create the tracker report
                report_name = f"trackme_{component}_tags_tracker_tenant_{tenant_id}"
                report_search = (
                    f'| trackmesplktags tenant_id="{tenant_id}" component="{component}"'
                )
                report_properties = {
                    "description": f"This scheduled report applies and maintains tags policies for the splk-{component} component",
                    "is_scheduled": True,
                    "cron_schedule": "*/15 * * * *",
                    "dispatch.earliest_time": "-5m",
                    "dispatch.latest_time": "now",
                    "schedule_window": "5",
                }

                # create the report
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": report_name,
                    "report_search": report_search,
                    "report_properties": report_properties,
                    "report_acl": report_acl,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                    )

                # Execute tags update now
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/splk_tag_policies/write/tag_policies_apply'
                post_data = {
                    "tenant_id": tenant_id,
                    "component": component,
                }

                # update the account
                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(post_data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2098, tags apply operation has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2098, tags apply operation was executed successfully, response.status_code="{response.status_code}", response="{json.dumps(response.json(), indent=2)}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2098, tags apply operation has failed, exception="{str(e)}"'
                    )

            #
            # for dsm only:
            # - migrate existing policies to the new collection
            # - update the tracker report
            # - remove the decomissioned collection and transforms
            #

            if component == "dsm":

                #
                # migrate policies to the new collection
                #

                # Define the query
                search = f"| inputlookup trackme_common_tag_policies_tenant_{tenant_id} | eval key = _key | outputlookup append=t key_field=key trackme_{component}_tags_policies_tenant_{tenant_id}"

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", component="{component}", migrate tags policies, search="{search}"'
                )

                kwargs_oneshot = {
                    "earliest_time": "-5m",
                    "latest_time": "now",
                    "output_mode": "json",
                    "count": 0,
                }

                # a counter
                counter = 0

                try:
                    reader = run_splunk_search(
                        service,
                        search,
                        kwargs_oneshot,
                        24,
                        5,
                    )

                    for item in reader:
                        if isinstance(item, dict):
                            counter += 1

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", component="{component}", migrate tags policies, search successfully operated, records_updated="{counter}"'
                    )

                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", component="{component}", migrate tags policies, search permanently failed, exception="{str(e)}"'
                    )

                #
                # Update the tag tracker
                #

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", processing with tags tracker update'
                )

                # report name
                tracker_name = f"trackme_dsm_tags_tracker_tenant_{tenant_id}"

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # add our new parameter
                tracker_new_search = f'{tracker_current_search} component="dsm"'

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

                #
                # Remove the decomissioned collection and transforms
                #

                # set
                transform_name = f"trackme_common_tag_policies_tenant_{tenant_id}"
                collection_name = f"kv_trackme_common_tag_policies_tenant_{tenant_id}"

                # delete the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

                # delete the KVstore collection
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvcollection'
                data = {
                    "tenant_id": tenant_id,
                    "collection_name": collection_name,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to delete the KVstore collection, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", successfully deleted KVstore collection, collection="{collection_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2098, tenant_id="{tenant_id}", failed to delete the KVstore collection collection="{collection_name}", exception="{str(e)}"'
                    )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2098, procedure terminated'
    )
    return True


"""
In this function:
 - Addresses the transition from md5 hashlib to sha256 for FIPS
 """


def trackme_schema_upgrade_2099(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2099, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # md5 to sha256 migration
        #

        tenant_id = vtenant_record.get("tenant_id")

        components_to_process = []

        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")
        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_to_process.append("cim")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")

        for component in components_to_process:

            #
            # migrate records from md5 to sha256
            #

            if component in ("dsm", "dhm", "flx", "wlk", "mhm"):

                # common collections
                collections_migration_list = [
                    f"kv_trackme_{component}_tenant_{tenant_id}",
                    f"kv_trackme_{component}_priority_tenant_{tenant_id}",
                    f"kv_trackme_{component}_tags_tenant_{tenant_id}",
                    f"kv_trackme_{component}_sla_tenant_{tenant_id}",
                ]

                # Only these components have Outliers KV collections
                if component in ("dsm", "dhm", "flx", "wlk"):
                    collections_migration_list.append(
                        f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}",
                    )
                    collections_migration_list.append(
                        f"kv_trackme_{component}_outliers_entity_data_tenant_{tenant_id}",
                    )

                # splk-dsm specific
                if component in ("dsm"):
                    collections_migration_list.append(
                        f"kv_trackme_{component}_data_sampling_tenant_{tenant_id}"
                    )

            else:
                collections_migration_list = [
                    f"kv_trackme_{component}_tenant_{tenant_id}",
                ]

            logging.info(f'collections_migration_list="{collections_migration_list}"')

            for collection_name in collections_migration_list:

                # entities KV collection
                collection = service.kvstore[collection_name]

                # created_records_key
                created_records_key = []

                # get records
                collection_records, collection_keys, collection_dict = (
                    get_full_kv_collection(collection, collection_name)
                )

                # for each record
                for record in collection_records:

                    object_value = record.get("object")
                    key_value = record.get("_key")

                    # create the new key using sha256
                    new_key_value = hashlib.sha256(
                        object_value.encode("utf-8")
                    ).hexdigest()

                    # only if required
                    if (
                        key_value != new_key_value
                        and new_key_value not in created_records_key
                    ):

                        record_removed = False
                        try:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2099, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", migrating record with object_value="{object_value}", key_value="{key_value}", new_key_value="{new_key_value}", record="{json.dumps(record, indent=2)}"'
                            )
                            # Remove the record
                            collection.data.delete(json.dumps({"_key": key_value}))
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2099, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", successfully removed record with object_value="{object_value}", key_value="{key_value}", new_key_value="{new_key_value}", record="{json.dumps(record, indent=2)}"'
                            )
                            record_removed = True

                        except Exception as e:
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2099, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", migrating record with object_value="{object_value}", failed to remove the record, key_value="{key_value}", exception="{str(e)}"'
                            )

                        if record_removed:
                            # insert a new record with the replaced key
                            record["_key"] = new_key_value
                            try:
                                collection.data.insert(json.dumps(record))
                                logging.info(
                                    f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2099, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", migrating record with object_value="{object_value}", successfully inserted the new record, key_value="{new_key_value}", record="{json.dumps(record, indent=2)}"'
                                )
                                created_records_key.append(new_key_value)
                            except Exception as e:
                                logging.error(
                                    f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2099, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", migrating record with object_value="{object_value}", failed to insert the new record, key_value="{new_key_value}", exception="{str(e)}", record="{json.dumps(record, indent=2)}"'
                                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2099, procedure terminated'
    )
    return True


"""
In this function:
- Update hybrid trackers to include the object_category option when calling the streaming custom command trackmesplkgetflipping
- Update any hybrid tracker that would be using an md5 approach to calculate the expected hash for TrackMe's object and would have been created before the migration to sha256
- Data Sampling events recognition engine v2 migration (delete existing schedule report and re-create with new search)
- The migration of TrackMe 2.0.98 extended tags for all components, but splk-mhm was forgotten
"""


def trackme_schema_upgrade_2100(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2100, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Update hybrid trackers to include the object_category option when calling the streaming custom command trackmesplkgetflipping
        # Update any hybrid tracker that would be using an md5 approach to calculate the expected hash for TrackMe's object and would have been created before the migration to sha256
        #

        # Note: splk-cim is not concerned by this migration

        components_to_process = []

        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")

        for component in components_to_process:

            # retrieve the list of scheduler reports to be processed
            collection_trackers_name = (
                f"kv_trackme_{component}_hybrid_trackers_tenant_{tenant_id}"
            )
            collection_trackers = service.kvstore[collection_trackers_name]

            trackers_wrapper_to_process = []

            # get records from the KVstore
            trackers_records = collection_trackers.data.query()

            for tracker_record in trackers_records:

                tracker_name = tracker_record.get("tracker_name")
                tracker_kos = json.loads(tracker_record.get("knowledge_objects"))

                tracker_reports = tracker_kos.get("reports")

                for tracker_report in tracker_reports:
                    if "_wrapper_" in tracker_report:
                        trackers_wrapper_to_process.append(tracker_report)

            # loop through wrapper reports
            for tracker_name in trackers_wrapper_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # calculate the current sha256 hash of tracker_current_search, we will use this information to define if the tracker needs to be updated
                # after the modifications
                tracker_current_search_hash = hashlib.sha256(
                    tracker_current_search.encode("utf-8")
                ).hexdigest()

                # in tracker_current_search, replace the sentence:
                # trackmesplkgetflipping tenant_id=
                # with:
                # trackmesplkgetflipping object_category=spl-<component> tenant_id=

                tracker_new_search = tracker_current_search.replace(
                    f'trackmesplkgetflipping tenant_id="{tenant_id}"',
                    f'trackmesplkgetflipping object_category="splk-{component}" tenant_id="{tenant_id}"',
                )

                # in tracker_current_search, replace any usage of the md5 function with the sha256 function, this can be detected and replaced with:
                # md5(
                # with:
                # sha256(

                tracker_new_search = tracker_new_search.replace("md5(", "sha256(")

                # calculate the new sha256 hash of tracker_new_search
                tracker_new_search_hash = hashlib.sha256(
                    tracker_new_search.encode("utf-8")
                ).hexdigest()

                # if hashes are the name, generate a logging info message as the report does not need to be updated
                if tracker_current_search_hash == tracker_new_search_hash:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search_hash="{tracker_current_search_hash}", tracker_new_search_hash="{tracker_new_search_hash}", tracker_current_search="{tracker_current_search}", tracker_new_search="{tracker_new_search}", tracker does not need to be updated'
                    )

                else:
                    # update the search definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": tracker_name,
                        "report_search": tracker_new_search,
                        "earliest_time": tracker_current_earliest_time,
                        "latest_time": tracker_current_latest_time,
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                        )

        #
        # Data Sampling events recognition engine v2 migration: delete and re-create the report
        #

        # check components and add accordingly
        if vtenant_record.get("tenant_dsm_enabled") == 1:

            # purge the current KVstore records for sampling

            # Define the query
            search = f"| outputlookup trackme_dsm_data_sampling_tenant_{tenant_id}"

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", purge data sampling collection, search="{search}"'
            )

            kwargs_oneshot = {
                "earliest_time": "-5m",
                "latest_time": "now",
                "output_mode": "json",
                "count": 0,
            }

            try:
                reader = run_splunk_search(
                    service,
                    search,
                    kwargs_oneshot,
                    24,
                    5,
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", data sampling collection was purged.'
                )

            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to execute the data sampling collection purge, exception="{str(e)}"'
                )

            # boolean to act depending on the report deletion
            report_deleted = False

            # report name
            report_name = f"trackme_dsm_data_sampling_tracker_tenant_{tenant_id}"

            # delete the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", delete report has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully deleted report, report="{report_name}"'
                    )
                    report_deleted = True

            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to delete the report, report="{report_name}", exception="{str(e)}"'
                )

            # create the report
            if report_deleted:

                report_search = f'| trackmesamplingexecutor tenant_id="{tenant_id}"'
                report_properties = {
                    "description": "TrackMe DSM Data Sampling tracker",
                    "is_scheduled": True,
                    "schedule_window": "5",
                    "cron_schedule": "*/20 * * * *",
                    "dispatch.earliest_time": "-24h",
                    "dispatch.latest_time": "-4h",
                }
                report_acl = {
                    "owner": str(vtenant_record.get("tenant_owner")),
                    "sharing": "app",
                    "perms.write": tenant_roles_write_perms,
                    "perms.read": tenant_roles_read_perms,
                }

                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": report_name,
                    "report_search": report_search,
                    "report_properties": report_properties,
                    "report_acl": report_acl,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                    )

            #
            # update collection transforms
            #

            # set
            transform_name = f"trackme_dsm_data_sampling_tenant_{tenant_id}"
            collection_name = f"kv_trackme_dsm_data_sampling_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_dsm_data_sampling"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # Create tags tracker for splk-mhm
        #

        components_to_process = []

        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")

        for component in components_to_process:

            #
            # First check if the procedure is required, to do so we will call an endpoint to check if the report exists
            #

            run_procedure = False
            report_name = f"trackme_{component}_tags_tracker_tenant_{tenant_id}"

            # create the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/get_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", the expected report does not exist, procedure is required, report="{report_name}", response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                    run_procedure = True
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully got report definition, procedure is not required, report="{report_name}"'
                    )
                    run_procedure = False
            except Exception as e:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", the expected report does not exist, procedure is required, report="{report_name}", exception="{str(e)}"'
                )
                run_procedure = True

            # proceed if required
            if run_procedure:

                #
                # update main collection transforms
                #

                # set
                transform_name = f"trackme_{component}_tenant_{tenant_id}"
                collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
                transform_fields = collections_dict[f"trackme_{component}"]
                ko_acl = {
                    "owner": vtenant_record.get("tenant_owner"),
                    "sharing": trackme_default_sharing,
                    "perms.write": tenant_roles_write_perms,
                    "perms.read": tenant_roles_read_perms,
                }

                # delete the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

                #
                # Create the tags collection and transforms
                #

                # set
                transform_name = f"trackme_{component}_tags_tenant_{tenant_id}"
                collection_name = f"kv_trackme_{component}_tags_tenant_{tenant_id}"
                transform_fields = collections_dict[f"trackme_{component}_tags"]
                ko_acl = {
                    "owner": vtenant_record.get("tenant_owner"),
                    "sharing": trackme_default_sharing,
                    "perms.write": tenant_roles_write_perms,
                    "perms.read": tenant_roles_read_perms,
                }

                # create the KVstore collection
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
                data = {
                    "tenant_id": tenant_id,
                    "collection_name": collection_name,
                    "collection_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                kvstore_created = False
                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                        )
                        kvstore_created = True
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                    )

                # continue
                if kvstore_created:

                    # create the transform
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                    data = {
                        "tenant_id": tenant_id,
                        "transform_name": transform_name,
                        "transform_fields": transform_fields,
                        "collection_name": collection_name,
                        "transform_acl": ko_acl,
                        "owner": vtenant_record.get("tenant_owner"),
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                        )

                #
                # Create the tags policies collection and transforms (for all components)
                #

                # set
                transform_name = f"trackme_{component}_tags_policies_tenant_{tenant_id}"
                collection_name = (
                    f"kv_trackme_{component}_tags_policies_tenant_{tenant_id}"
                )
                transform_fields = collections_dict[f"trackme_{component}_tags"]
                ko_acl = {
                    "owner": vtenant_record.get("tenant_owner"),
                    "sharing": trackme_default_sharing,
                    "perms.write": tenant_roles_write_perms,
                    "perms.read": tenant_roles_read_perms,
                }

                # create the KVstore collection
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
                data = {
                    "tenant_id": tenant_id,
                    "collection_name": collection_name,
                    "collection_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                kvstore_created = False
                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                        )
                        kvstore_created = True
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                    )

                # continue
                if kvstore_created:

                    # create the transform
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                    data = {
                        "tenant_id": tenant_id,
                        "transform_name": transform_name,
                        "transform_fields": transform_fields,
                        "collection_name": collection_name,
                        "transform_acl": ko_acl,
                        "owner": vtenant_record.get("tenant_owner"),
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                        )

                #
                # Create the report tracker (not for dsm, as it already exists)
                #

                report_acl = {
                    "owner": str(vtenant_record.get("tenant_owner")),
                    "sharing": "app",
                    "perms.write": tenant_roles_write_perms,
                    "perms.read": tenant_roles_read_perms,
                }

                # create the tracker report
                report_name = f"trackme_{component}_tags_tracker_tenant_{tenant_id}"
                report_search = (
                    f'| trackmesplktags tenant_id="{tenant_id}" component="{component}"'
                )
                report_properties = {
                    "description": f"This scheduled report applies and maintains tags policies for the splk-{component} component",
                    "is_scheduled": True,
                    "cron_schedule": "*/15 * * * *",
                    "dispatch.earliest_time": "-5m",
                    "dispatch.latest_time": "now",
                    "schedule_window": "5",
                }

                # create the report
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": report_name,
                    "report_search": report_search,
                    "report_properties": report_properties,
                    "report_acl": report_acl,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2100, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                    )

                # Execute tags update now
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/splk_tag_policies/write/tag_policies_apply'
                post_data = {
                    "tenant_id": tenant_id,
                    "component": component,
                }

                # update the account
                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(post_data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2100, tags apply operation has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2100, tags apply operation was executed successfully, response.status_code="{response.status_code}", response="{json.dumps(response.json(), indent=2)}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2100, tags apply operation has failed, exception="{str(e)}"'
                    )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2100, procedure terminated'
    )
    return True


"""
In this function:
- For Flex Object tenants, update the inactive_entities_tracker to remove an unused option and define increased default value to auto purge.
"""


def trackme_schema_upgrade_2101(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2101, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Adaptive flx inactive entities tracker
        #

        components_to_process = []
        if (
            vtenant_record.get("tenant_flx_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("flx")

        for component in components_to_process:

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", processing with Flex inactive entities tracker update'
            )

            # report name
            tracker_name = f"trackme_flx_inactive_entities_tracker_tenant_{tenant_id}"

            # get the current search definition
            try:
                tracker_current = service.saved_searches[tracker_name]
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                )
                continue

            tracker_current_search = tracker_current.content.get("search")

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
            )

            # add our new parameter
            tracker_new_search = f'| trackmesplkflxinactiveinspector tenant_id="{tenant_id}" register_component="True" report="{tracker_name}" max_days_since_inactivity_before_purge=30'

            # update the search definition
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": tracker_name,
                "report_search": tracker_new_search,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2101, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2101, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2101, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2101, procedure terminated'
    )
    return True


"""
In this function:
- Update tenant allowlist transforms definition
- Update the Virtual Tenant account with the sampling option
"""


def trackme_schema_upgrade_2102(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2102, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update allowlist transforms definition (for feeds components only)
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        # select components
        components_to_process = []
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")

        for component in components_to_process:

            # set
            transform_name = f"trackme_{component}_allowlist_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_allowlist_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}_allowlist"]
            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2102, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2102, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2102, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2102, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2102, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2102, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2102, procedure terminated'
    )
    return True


"""
In this function:
- Create the allowlist collections for splk-flx/wlk
- Update transforms definition for lagging classes
"""


def trackme_schema_upgrade_2104(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2104, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # allowlist extension to splk-flx and splk-wlk
        #

        components_to_process = []

        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")

        for component in components_to_process:

            #
            # Create the allowlist collection and transforms
            #

            # set
            transform_name = f"trackme_{component}_allowlist_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_allowlist_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}_allowlist"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the KVstore collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            kvstore_created = False
            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                    )
                    kvstore_created = True
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # continue
            if kvstore_created:

                # create the transform
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
                data = {
                    "tenant_id": tenant_id,
                    "transform_name": transform_name,
                    "transform_fields": transform_fields,
                    "collection_name": collection_name,
                    "transform_acl": ko_acl,
                    "owner": vtenant_record.get("tenant_owner"),
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                    )

        #
        # lagging class transforms update
        #

        # select components
        components_to_process = []
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")

        for component in components_to_process:

            # set

            if component in ("dsm", "dhm"):
                transform_name = f"trackme_common_lagging_classes_tenant_{tenant_id}"
                collection_name = (
                    f"kv_trackme_common_lagging_classes_tenant_{tenant_id}"
                )
                transform_fields = collections_dict[f"trackme_common_lagging_classes"]

            elif component in ("mhm"):
                transform_name = f"trackme_mhm_lagging_classes_tenant_{tenant_id}"
                collection_name = f"kv_trackme_mhm_lagging_classes_tenant_{tenant_id}"
                transform_fields = collections_dict[f"trackme_mhm_lagging_classes"]

            transform_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": transform_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2104, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2104, procedure terminated'
    )
    return True


"""
In this function:
 - Adresse defect for splk-cim and splk-wlk the transition from md5 hashlib to sha256 for FIPS that was processed in 2.0.99
 """


def trackme_schema_upgrade_2105(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2105, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # md5 to sha256 migration
        #

        tenant_id = vtenant_record.get("tenant_id")

        components_to_process = []

        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")
        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_to_process.append("cim")

        for component in components_to_process:

            #
            # migrate records from md5 to sha256
            #

            # common collections
            collections_migration_list = [
                f"kv_trackme_{component}_tenant_{tenant_id}",
                f"kv_trackme_{component}_priority_tenant_{tenant_id}",
                f"kv_trackme_{component}_tags_tenant_{tenant_id}",
                f"kv_trackme_{component}_sla_tenant_{tenant_id}",
            ]

            # Only these components have Outliers KV collections
            if component in ("cim", "wlk"):
                collections_migration_list.append(
                    f"kv_trackme_{component}_outliers_entity_rules_tenant_{tenant_id}",
                )
                collections_migration_list.append(
                    f"kv_trackme_{component}_outliers_entity_data_tenant_{tenant_id}",
                )

            logging.info(f'collections_migration_list="{collections_migration_list}"')

            for collection_name in collections_migration_list:

                # entities KV collection
                collection = service.kvstore[collection_name]

                # created_records_key
                created_records_key = []

                # get records
                collection_records, collection_keys, collection_dict = (
                    get_full_kv_collection(collection, collection_name)
                )

                # for each record
                for record in collection_records:

                    object_value = record.get("object")
                    key_value = record.get("_key")

                    # create the new key using sha256
                    new_key_value = hashlib.sha256(
                        object_value.encode("utf-8")
                    ).hexdigest()

                    # only if required
                    if (
                        key_value != new_key_value
                        and new_key_value not in created_records_key
                    ):

                        record_removed = False
                        try:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2105, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", migrating record with object_value="{object_value}", key_value="{key_value}", new_key_value="{new_key_value}", record="{json.dumps(record, indent=2)}"'
                            )
                            # Remove the record
                            collection.data.delete(json.dumps({"_key": key_value}))
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2105, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", successfully removed record with object_value="{object_value}", key_value="{key_value}", new_key_value="{new_key_value}", record="{json.dumps(record, indent=2)}"'
                            )
                            record_removed = True

                        except Exception as e:
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2105, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", migrating record with object_value="{object_value}", failed to remove the record, key_value="{key_value}", exception="{str(e)}"'
                            )

                        if record_removed:
                            # insert a new record with the replaced key
                            record["_key"] = new_key_value
                            try:
                                collection.data.insert(json.dumps(record))
                                logging.info(
                                    f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2105, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", migrating record with object_value="{object_value}", successfully inserted the new record, key_value="{new_key_value}", record="{json.dumps(record, indent=2)}"'
                                )
                                created_records_key.append(new_key_value)
                            except Exception as e:
                                logging.error(
                                    f'task="{task_name}", task_instance_id={task_instance_id}, subcontext="records-migration", schema migration 2105, tenant_id="{tenant_id}", component="{component}", collection="{collection_name}", migrating record with object_value="{object_value}", failed to insert the new record, key_value="{new_key_value}", exception="{str(e)}", record="{json.dumps(record, indent=2)}"'
                                )

            #
            # Migrate trackers (only for splk-wlk)
            #

            # check components and add accordingly
            if vtenant_record.get("tenant_wlk_enabled") == 1:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", processing with the schema upgrade'
                )

                # retrieve the list of scheduler reports to be processed
                collection_trackers_name = (
                    f"kv_trackme_wlk_hybrid_trackers_tenant_{tenant_id}"
                )
                collection_trackers = service.kvstore[collection_trackers_name]

                trackers_dict = {}

                trackers_to_process = []
                trackers_to_process_kos = []

                # get records from the KVstore
                trackers_records = collection_trackers.data.query()

                for tracker_record in trackers_records:
                    tracker_name = tracker_record.get("tracker_name")
                    tracker_kos = tracker_record.get("knowledge_objects")
                    trackers_to_process.append(tracker_name)
                    trackers_to_process_kos.append(tracker_kos)
                    trackers_dict[tracker_name] = {
                        "knowledge_objects": json.loads(tracker_kos),
                    }

                # loop through the trackers
                for tracker_shortname in trackers_to_process:
                    tracker_name = f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"

                    tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                    # get the current search definition
                    try:
                        tracker_current = service.saved_searches[tracker_name]
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                        )
                        continue

                    tracker_current_search = tracker_current.content.get("search")
                    tracker_account = tracker_kos["properties"][0]["account"]

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                    )

                    # in tracker_current_search, replace any presence of md5 by sha256

                    try:
                        tracker_new_search = re.sub(
                            r"md5\(([^)]+)\)",
                            r"sha256(\1)",
                            tracker_current_search,
                        )

                        # update the search definition
                        tracker_current.update(search=tracker_new_search)

                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, the tracker was updated successfully, new_search="{tracker_new_search}"'
                        )

                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, failed to update the tracker, exception="{str(e)}"'
                        )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2105, procedure terminated'
    )
    return True


"""
In this function:
- Update maintransforms definition for splk-flx
"""


def trackme_schema_upgrade_2107(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2107, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # update transforms definition for splk-flx
        #

        components_to_process = []

        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")

        for component in components_to_process:

            #
            # update main collection transforms
            #

            # set
            transform_name = f"trackme_{component}_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2107, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2107, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2107, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2107, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2107, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2107, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2107, procedure terminated'
    )
    return True


"""
In this function:
- Update any existing Splunk Remote account preferences to include default values
"""


def trackme_schema_upgrade_2108(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2108, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2108, procedure terminated'
    )
    return True


"""
In this function:
- Update all trackers to remove explict calls to earliest and latest arguments, and add explicit call to alert_no_results
"""


def trackme_schema_upgrade_2109(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2109, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Update tracker main searches to remove explicit earliest/latest, and add explicit alert_no_results
        #

        components_to_process = []

        # for all components except splk-cim
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")

        for component in components_to_process:

            # retrieve the list of scheduler reports to be processed
            collection_trackers_name = (
                f"kv_trackme_{component}_hybrid_trackers_tenant_{tenant_id}"
            )
            collection_trackers = service.kvstore[collection_trackers_name]

            # A list to store trackers to be processed
            trackers_to_process = []

            # get records from the KVstore
            trackers_records = collection_trackers.data.query()

            for tracker_record in trackers_records:

                tracker_name = tracker_record.get("tracker_name")
                tracker_kos = json.loads(tracker_record.get("knowledge_objects"))
                tracker_reports = tracker_kos.get("reports")

                for tracker_report in tracker_reports:
                    # excluse _abstract _wrapper trackers
                    if (
                        not "_abstract_" in tracker_report
                        and not "_wrapper_" in tracker_report
                    ):
                        trackers_to_process.append(tracker_report)

            # loop through abstract reports
            for tracker_name in trackers_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="splk-{component}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # in tracker_current_search, replace the sentence:
                #  earliest=<anything but space> latest=<anything but space>
                # with:
                #  alert_no_results=True

                tracker_new_search = re.sub(
                    r"earliest=[^\s]+ latest=[^\s]+",
                    r"alert_no_results=True",
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2109, tenant_id="{tenant_id}", component="splk-{component}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2109, tenant_id="{tenant_id}", component="splk-{component}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2109, tenant_id="{tenant_id}", component="splk-{component}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2109, procedure terminated'
    )
    return True


"""
In this function:
- Update all components main transforms definition to include priority_reason
- Update all trackers to handle schedule_window inconsistencies
- Create delayed entities inspector collections for dsm and dhm
- Create delayed entities inspector trackers for dsm and dhm
- Create last seen activity collection for flx
- Update the adaptive delay report to add the new option max_sla_percentage
"""


def trackme_schema_upgrade_2110(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2110, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Update main transforms
        #

        components_to_process = []

        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")
        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_to_process.append("cim")

        for component in components_to_process:

            #
            # update main collection transforms
            #

            # set
            transform_name = f"trackme_{component}_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # Update alerts
        #

        try:
            alert_kos = json.loads(vtenant_record.get("tenant_alert_objects"))
            alert_reports = alert_kos.get("alerts")
        except Exception as e:
            alert_reports = []

        # get records from the KVstore
        for alert_name in alert_reports:

            # get the current search definition
            alert_current = None
            try:
                alert_current = service.saved_searches[alert_name]
            except Exception as e:
                alert_current = None

            if alert_current:
                alert_current_search = alert_current.content.get("search")
                alert_current_search = alert_current.content.get("search")
                alert_current_earliest_time = alert_current.content.get(
                    "dispatch.earliest_time"
                )
                alert_current_latest_time = alert_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", alert_name="{alert_name}", alert_current_search="{alert_current_search}"'
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": alert_name,
                    "report_search": alert_current_search,
                    "earliest_time": alert_current_earliest_time,
                    "latest_time": alert_current_latest_time,
                    "schedule_window": "5",
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", update report definition has failed, report="{alert_name}", response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", successfully updated report definition, report="{alert_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to update the report definition, report="{alert_name}", exception="{str(e)}"'
                    )

        #
        # Update trackers
        #

        components_to_process = []

        # for all components except splk-cim
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")
        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_to_process.append("cim")

        for component in components_to_process:

            # A list to store trackers to be processed
            trackers_to_process = []

            try:
                if component == "cim":
                    tracker_kos = json.loads(
                        vtenant_record.get(f"tenant_{component}_objects")
                    )
                else:
                    tracker_kos = json.loads(
                        vtenant_record.get(f"tenant_{component}_hybrid_objects")
                    )
                tracker_reports = tracker_kos.get("reports")
            except Exception as e:
                tracker_reports = []

            for tracker_report in tracker_reports:
                # excluse _abstract _wrapper trackers
                if (
                    not "_abstract_" in tracker_report
                    and not "_wrapper_" in tracker_report
                ):
                    trackers_to_process.append(tracker_report)

            # for splk-dsm only, add trackme_dsm_data_sampling_tracker_tenant_<tenant_id>, trackme_dsm_shared_elastic_tracker_tenant_<tenant_id>
            if component == "dsm":
                trackers_to_process.append(
                    f"trackme_dsm_data_sampling_tracker_tenant_{tenant_id}"
                )
                trackers_to_process.append(
                    f"trackme_dsm_shared_elastic_tracker_tenant_{tenant_id}"
                )

            # loop through abstract reports
            for tracker_name in trackers_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="splk-{component}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_current_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                    "schedule_window": "5",
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", component="splk-{component}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", component="splk-{component}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", component="splk-{component}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # Create Kvstore collections and transforms for delayed entities inspector
        #

        components_to_process = []

        if (
            vtenant_record.get("tenant_dsm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dsm")
        if (
            vtenant_record.get("tenant_dhm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dhm")

        for component in components_to_process:

            # set
            transform_name = (
                f"trackme_{component}_delayed_entities_inspector_tenant_{tenant_id}"
            )
            collection_name = (
                f"kv_trackme_{component}_delayed_entities_inspector_tenant_{tenant_id}"
            )
            transform_fields = collections_dict[
                f"trackme_{component}_delayed_entities_inspector"
            ]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the collection, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", successfully created the collection, collection="{collection_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the transform, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", successfully created the transform, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the transform, transform="{transform_name}", exception="{str(e)}"'
                )

            #
            # Create the adaptive delay tracker
            #

            report_acl = {
                "owner": str(vtenant_record.get("tenant_owner")),
                "sharing": trackme_default_sharing,
                "perms.write": str(vtenant_record.get("tenant_roles_admin")),
                "perms.read": str(tenant_roles_read_perms),
            }

            # create the wrapper
            report_name = f"trackme_{component}_delayed_entities_inspector_tracker_tenant_{tenant_id}"
            report_search = f'| trackmesplkfeedsdelayedinspector tenant_id="{tenant_id}" component="{component}" max_runtime=900'
            report_properties = {
                "description": f"This scheduled report manages delayed entities in the {component} component",
                "is_scheduled": True,
                "cron_schedule": "*/20 * * * *",
                "dispatch.earliest_time": "-5m",
                "dispatch.latest_time": "now",
                "schedule_window": "5",
            }

            # create the report
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": report_name,
                "report_search": report_search,
                "report_properties": report_properties,
                "report_acl": report_acl,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", create report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", successfully created report definition, report="{report_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the report definition, report="{report_name}", exception="{str(e)}"'
                )

        #
        # Create Kvstore collections and transforms for last seen activity in splk-flx
        #

        components_to_process = []

        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")

        for component in components_to_process:

            # set
            transform_name = (
                f"trackme_{component}_last_seen_activity_tenant_{tenant_id}"
            )
            collection_name = (
                f"kv_trackme_{component}_last_seen_activity_tenant_{tenant_id}"
            )
            transform_fields = collections_dict[
                f"trackme_{component}_last_seen_activity"
            ]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the collection, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", successfully created the collection, collection="{collection_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the transform, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", successfully created the transform, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to create the transform, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # Adaptive trackers
        #

        components_to_process = []
        if (
            vtenant_record.get("tenant_dsm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dsm")
        if (
            vtenant_record.get("tenant_dhm_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            components_to_process.append("dhm")

        for component in components_to_process:
            #
            # Update the adaptive delay tracker
            #

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="{component}", processing with adaptive delay tracker update'
            )

            # report name
            tracker_name = (
                f"trackme_{component}_adaptive_delay_tracker_tenant_{tenant_id}"
            )

            # get the current search definition
            try:
                tracker_current = service.saved_searches[tracker_name]
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                )
                continue

            tracker_current_search = tracker_current.content.get("search")

            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
            )

            # add our new parameter
            tracker_new_search = f"{tracker_current_search} max_sla_percentage=90"

            # update the search definition
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
            data = {
                "tenant_id": tenant_id,
                "report_name": tracker_name,
                "report_search": tracker_new_search,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2110, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2110, procedure terminated'
    )
    return True


"""
In this function:
- Update all components main transforms definition
- Update all trackers to report the list of objects it manages, for the purpose of the handler events component and the deprecation of the macro trackme_auto_disablement_period
- Create the new collection and transform for stateful alerting
- Create the new collection and transform for stateful alerting charts
- Create the new collection and transform for SLA notifications
- Update Splunk Cloud Workload SVC trackers to switch to the new index
- Update all Flex Trackers related to Splunk Cloud SVC usage to switch to the new index
"""


def trackme_schema_upgrade_2111(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2111, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Update main transforms
        #

        components_to_process = []

        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")
        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_to_process.append("cim")

        for component in components_to_process:

            #
            # update main collection transforms
            #

            # set
            transform_name = f"trackme_{component}_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # delete the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # Update trackers
        #

        components_to_process = []

        # for all components except splk-cim
        if vtenant_record.get("tenant_dsm_enabled") == 1:
            components_to_process.append("dsm")
        if vtenant_record.get("tenant_dhm_enabled") == 1:
            components_to_process.append("dhm")
        if vtenant_record.get("tenant_mhm_enabled") == 1:
            components_to_process.append("mhm")
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")
        if vtenant_record.get("tenant_wlk_enabled") == 1:
            components_to_process.append("wlk")
        if vtenant_record.get("tenant_cim_enabled") == 1:
            components_to_process.append("cim")

        for component in components_to_process:

            # A list to store trackers to be processed
            trackers_to_process = []

            try:
                if component == "cim":
                    tracker_kos = json.loads(
                        vtenant_record.get(f"tenant_{component}_objects")
                    )
                else:
                    tracker_kos = json.loads(
                        vtenant_record.get(f"tenant_{component}_hybrid_objects")
                    )
                tracker_reports = tracker_kos.get("reports")
            except Exception as e:
                tracker_reports = []

            for tracker_report in tracker_reports:
                # include only _wrapper trackers
                if "_wrapper_" in tracker_report:
                    trackers_to_process.append(tracker_report)

            # loop through abstract reports
            for tracker_name in trackers_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", component="splk-{component}", failed to get the tracker definition, tracker="{tracker_name}", exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="splk-{component}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # in tracker_current_search, replace the sentence:
                # stats count as report_entities_count by tenant_id
                # with:
                # stats count as report_entities_count, values(object) as report_objects_list by tenant_id

                tracker_new_search = re.sub(
                    r"stats count as report_entities_count by tenant_id",
                    r"stats count as report_entities_count, values(object) as report_objects_list by tenant_id",
                    tracker_current_search,
                )

                # deprecation of the macro trackme_auto_disablement_period

                # in tracker_new_search, remove the following, if present:
                # | eval monitored_state=if(metric_last_time_seen<=`trackme_auto_disablement_period`, "disabled", monitored_state)
                # | eval monitored_state=if(metric_last_time_seen<=`trackme_auto_disablement_period`, \"disabled\", monitored_state)
                # | eval monitored_state=if(data_last_time_seen<=`trackme_auto_disablement_period`, "disabled", monitored_state)
                # | eval monitored_state=if(data_last_time_seen<=`trackme_auto_disablement_period`, \"disabled\", monitored_state)
                tracker_new_search = re.sub(
                    r"\| eval monitored_state=if\(metric_last_time_seen<=`trackme_auto_disablement_period`, \"disabled\", monitored_state\)",
                    "",
                    tracker_new_search,
                )
                tracker_new_search = re.sub(
                    r"\| eval monitored_state=if\(data_last_time_seen<=`trackme_auto_disablement_period`, \"disabled\", monitored_state\)",
                    "",
                    tracker_new_search,
                )
                tracker_new_search = re.sub(
                    r"\| eval monitored_state=if\(metric_last_time_seen<=`trackme_auto_disablement_period`, \\\"disabled\\\", monitored_state\)",
                    "",
                    tracker_new_search,
                )
                tracker_new_search = re.sub(
                    r"\| eval monitored_state=if\(data_last_time_seen<=`trackme_auto_disablement_period`, \\\"disabled\\\", monitored_state\)",
                    "",
                    tracker_new_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", component="splk-{component}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", component="splk-{component}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", component="splk-{component}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # Create the new collection and transform for stateful alerting
        #

        # set
        transform_name = f"trackme_stateful_alerting_tenant_{tenant_id}"
        collection_name = f"kv_trackme_stateful_alerting_tenant_{tenant_id}"
        transform_fields = collections_dict[f"trackme_stateful_alerting"]
        ko_acl = {
            "owner": vtenant_record.get("tenant_owner"),
            "sharing": trackme_default_sharing,
            "perms.write": tenant_roles_write_perms,
            "perms.read": tenant_roles_read_perms,
        }

        # create the KVstore collection
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
        data = {
            "tenant_id": tenant_id,
            "collection_name": collection_name,
            "collection_acl": ko_acl,
            "owner": vtenant_record.get("tenant_owner"),
        }

        kvstore_created = False
        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                )
                kvstore_created = True
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
            )

        #
        # Only continue if successful
        #

        if kvstore_created:

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # Create the new collection and transform for stateful alerting charts
        #

        # set
        transform_name = f"trackme_stateful_alerting_charts_tenant_{tenant_id}"
        collection_name = f"kv_trackme_stateful_alerting_charts_tenant_{tenant_id}"
        transform_fields = collections_dict[f"trackme_stateful_alerting_charts"]
        ko_acl = {
            "owner": vtenant_record.get("tenant_owner"),
            "sharing": trackme_default_sharing,
            "perms.write": tenant_roles_write_perms,
            "perms.read": tenant_roles_read_perms,
        }

        # create the KVstore collection
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
        data = {
            "tenant_id": tenant_id,
            "collection_name": collection_name,
            "collection_acl": ko_acl,
            "owner": vtenant_record.get("tenant_owner"),
        }

        kvstore_created = False
        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                )
                kvstore_created = True
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
            )

        #
        # Only continue if successful
        #

        if kvstore_created:

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # Create the new collection and transform for SLA notifications
        #

        # set
        transform_name = f"trackme_{component}_sla_notifications_tenant_{tenant_id}"
        collection_name = f"kv_trackme_{component}_sla_notifications_tenant_{tenant_id}"
        transform_fields = collections_dict[f"trackme_{component}_sla_notifications"]
        ko_acl = {
            "owner": vtenant_record.get("tenant_owner"),
            "sharing": trackme_default_sharing,
            "perms.write": tenant_roles_write_perms,
            "perms.read": tenant_roles_read_perms,
        }

        # create the KVstore collection
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
        data = {
            "tenant_id": tenant_id,
            "collection_name": collection_name,
            "collection_acl": ko_acl,
            "owner": vtenant_record.get("tenant_owner"),
        }

        kvstore_created = False
        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                )
                kvstore_created = True
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
            )

        #
        # Only continue if successful
        #

        if kvstore_created:

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # splk-wlk: Update Workload SVC trackers to switch to the new index
        #

        if (
            vtenant_record.get("tenant_wlk_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", component="wlk", processing with Workload tracker updates'
            )

            #
            # Trackers updates
            #

            # retrieve the list of scheduler reports to be processed
            collection_trackers_name = (
                f"kv_trackme_wlk_hybrid_trackers_tenant_{tenant_id}"
            )
            collection_trackers = service.kvstore[collection_trackers_name]

            trackers_dict = {}
            trackers_to_process = []
            trackers_to_process_kos = []

            # get records from the KVstore
            trackers_records = collection_trackers.data.query()

            for tracker_record in trackers_records:
                tracker_name = tracker_record.get("tracker_name")
                tracker_kos = tracker_record.get("knowledge_objects")

                # if the tracker_name starts by scheduler_
                if re.search("^splunkcloud_", tracker_name):
                    trackers_to_process.append(tracker_name)
                    trackers_to_process_kos.append(tracker_kos)

                    trackers_dict[tracker_name] = {
                        "knowledge_objects": json.loads(tracker_kos),
                    }

            # loop through the trackers (wrapper)
            for tracker_shortname in trackers_to_process:
                tracker_name = (
                    f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"
                )

                tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to retrieve the current search definition, tracker_name="{tracker_name}", exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_account = tracker_kos["properties"][0]["account"]
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                )

                # in tracker_current_search, replace the sentence:
                #  index=summary
                # with:
                #  (index=_cmc_summary OR index=summary)

                tracker_new_search = re.sub(
                    r"index=summary",
                    r"(index=_cmc_summary OR index=summary)",
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # splk-flx: Update Flex Object trackers that would track SVC usage
        #

        if (
            vtenant_record.get("tenant_flx_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", component="flx", processing with Flex Objects tracker updates'
            )

            component = "flx"

            # A list to store trackers to be processed
            trackers_to_process = []

            try:
                tracker_kos = json.loads(
                    vtenant_record.get(f"tenant_{component}_hybrid_objects")
                )
                tracker_reports = tracker_kos.get("reports")
            except Exception as e:
                tracker_reports = []

            for tracker_report in tracker_reports:
                if "_wrapper_" in tracker_report:
                    trackers_to_process.append(tracker_report)

            # loop through abstract reports
            for tracker_name in trackers_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to retrieve the current search definition, tracker_name="{tracker_name}", exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                # get the search definition, if it contains index=summary and splunk-svc or splunk-storage-summary, process, otherwise skip
                if re.search(
                    "index=summary.*splunk-svc", tracker_current_search
                ) or re.search(
                    "index=summary.*splunk-storage-summary", tracker_current_search
                ):
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, processing Flex tracker, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, skipping Flex tracker, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                    )
                    continue

                # in tracker_current_search, replace the sentence:
                #  index=summary
                # with:
                #  (index=_cmc_summary OR index=summary)

                tracker_new_search = re.sub(
                    r"index=summary",
                    r"(index=_cmc_summary OR index=summary)",
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                    "schedule_window": "5",
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", component="splk-{component}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", component="splk-{component}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", component="splk-{component}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # Verify and update the Virtual Tenant account configuration for the delayed entites inspector
        #

        url = f'{reqinfo["server_rest_uri"]}/servicesNS/nobody/trackme/trackme_vtenants/{tenant_id}'
        vtenant_data = {}

        try:
            # Get current vtenant account configuration
            response = requests.get(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                verify=False,
                params={"output_mode": "json"},
                timeout=600,
            )
            if response.status_code in (200, 201, 204):

                logging.info(f"successfully retrieved vtenant configuration")
                vtenant_data_json = response.json()
                vtenant_data_current = vtenant_data_json["entry"][0]["content"]

                # Start with the current configuration and add missing keys from the default config
                # We keep all keys from the current configuration
                for key, value in vtenant_data_current.items():
                    if key in vtenant_account_default:
                        vtenant_data[key] = value

                # before merging with the default config, check if any key is missing
                for key in vtenant_account_default.keys():
                    if key not in vtenant_data:
                        update_is_required = True

                # Merge with default config, only adding missing default keys
                for key, value in vtenant_account_default.items():
                    if key not in vtenant_data:
                        vtenant_data[key] = value

                # Finally, ensures that each key in vtenant_data exists in vtenant_account_default, otherwise drop it
                vtenant_data = {
                    key: value
                    for key, value in vtenant_data.items()
                    if key in vtenant_account_default
                }

            else:
                error_msg = f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to retrieve vtenant configuration, status_code={response.status_code}'
                logging.error(error_msg)

        except Exception as e:
            error_msg = f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to retrieve vtenant configuration, exception={str(e)}'
            logging.error(error_msg)

        # get current values
        if vtenant_data:

            # current values from the vtenant account configuration
            splk_feeds_delayed_inspector_24hours_range_min_sec = int(
                vtenant_data.get("splk_feeds_delayed_inspector_24hours_range_min_sec")
            )
            splk_feeds_delayed_inspector_7days_range_min_sec = int(
                vtenant_data.get("splk_feeds_delayed_inspector_7days_range_min_sec")
            )
            splk_feeds_delayed_inspector_until_disabled_range_min_sec = int(
                vtenant_data.get(
                    "splk_feeds_delayed_inspector_until_disabled_range_min_sec"
                )
            )

            # get system default values
            default_splk_feeds_delayed_inspector_24hours_range_min_sec = int(
                vtenant_account_default.get(
                    "splk_feeds_delayed_inspector_24hours_range_min_sec"
                )
            )
            default_splk_feeds_delayed_inspector_7days_range_min_sec = int(
                vtenant_account_default.get(
                    "splk_feeds_delayed_inspector_7days_range_min_sec"
                )
            )
            default_splk_feeds_delayed_inspector_until_disabled_range_min_sec = int(
                vtenant_account_default.get(
                    "splk_feeds_delayed_inspector_until_disabled_range_min_sec"
                )
            )

            # init vetenant_update_data
            vtenant_update_data = {}

            # for each, if the current value is lower than the default value, update the value
            if (
                splk_feeds_delayed_inspector_24hours_range_min_sec
                < default_splk_feeds_delayed_inspector_24hours_range_min_sec
            ):
                vtenant_update_data[
                    "splk_feeds_delayed_inspector_24hours_range_min_sec"
                ] = default_splk_feeds_delayed_inspector_24hours_range_min_sec
            if (
                splk_feeds_delayed_inspector_7days_range_min_sec
                < default_splk_feeds_delayed_inspector_7days_range_min_sec
            ):
                vtenant_update_data[
                    "splk_feeds_delayed_inspector_7days_range_min_sec"
                ] = default_splk_feeds_delayed_inspector_7days_range_min_sec
            if (
                splk_feeds_delayed_inspector_until_disabled_range_min_sec
                < default_splk_feeds_delayed_inspector_until_disabled_range_min_sec
            ):
                vtenant_update_data[
                    "splk_feeds_delayed_inspector_until_disabled_range_min_sec"
                ] = default_splk_feeds_delayed_inspector_until_disabled_range_min_sec

            # if vtenant_update_data is not empty, update the vtenant account configuration
            if vtenant_update_data:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", processing with vtenant account configuration updates, parameters="{json.dumps(vtenant_update_data, indent=2)}"'
                )
                try:
                    update_vtenant_configuration(
                        reqinfo,
                        task_name,
                        task_instance_id,
                        tenant_id,
                        updated_vtenant_data=vtenant_update_data,
                    )
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", successfully updated vtenant account configuration'
                    )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2111, tenant_id="{tenant_id}", failed to update vtenant account configuration, exception="{str(e)}"'
                    )

        #
        # END
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2111, procedure terminated'
    )
    return True


"""
In this function:
- Update StateFul alerts to include the trackme:state event logic
"""


def trackme_schema_upgrade_2116(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2116, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Update StateFul alerts
        #

        try:
            alert_kos = json.loads(vtenant_record.get("tenant_alert_objects"))
            alert_reports = alert_kos.get("alerts")
        except Exception as e:
            alert_reports = []

        # get records from the KVstore
        for alert_name in alert_reports:

            # get the current search definition
            alert_current = None
            try:
                alert_current = service.saved_searches[alert_name]
            except Exception as e:
                alert_current = None

            # alert_is_stateful boolean
            alert_is_stateful = False

            if alert_current:
                alert_current_search = alert_current.content.get("search")
                alert_current_earliest_time = alert_current.content.get(
                    "dispatch.earliest_time"
                )
                alert_current_latest_time = alert_current.content.get(
                    "dispatch.latest_time"
                )

                # define if the alert is stateful, check in the search definition if we can find:
                # (sourcetype=trackme:flip) and (sourcetype=trackme:sla_breaches) and trackme_notable_idx
                if (
                    re.search(
                        r"(sourcetype=trackme:flip)",
                        alert_current_search,
                    )
                    and re.search(
                        r"(sourcetype=trackme:sla_breaches)",
                        alert_current_search,
                    )
                    and re.search(
                        r"(trackme_notable_idx)",
                        alert_current_search,
                    )
                ):
                    alert_is_stateful = True

                # do not process if the alert is not stateful
                if not alert_is_stateful:
                    continue

                # set the new search definition
                new_search_segment = f'( (`trackme_idx({tenant_id})` (sourcetype=trackme:state) tenant_id="{tenant_id}" object_category="*" object_state!="green") NOT [ | inputlookup trackme_stateful_alerting_tenant_{tenant_id} where (alert_status="opened" OR alert_status="updated") | eval _time=mtime | stats latest(alert_status) as alert_status by object_id | fields object_id | rename object_id as key | format | return $search ] ) OR '
                new_alert_search = f"{new_search_segment} {alert_current_search}"

                # update the search definition
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", alert_name="{alert_name}", alert_current_search="{alert_current_search}"'
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": alert_name,
                    "report_search": new_alert_search,
                    "earliest_time": alert_current_earliest_time,
                    "latest_time": alert_current_latest_time,
                    "schedule_window": "5",
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration {target_version}, tenant_id="{tenant_id}", update report definition has failed, report="{alert_name}", response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration {target_version}, tenant_id="{tenant_id}", successfully updated report definition, report="{alert_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration {target_version}, tenant_id="{tenant_id}", failed to update the report definition, report="{alert_name}", exception="{str(e)}"'
                    )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration {target_version}, procedure terminated'
    )
    return True


"""
In this function:
- Create thresholds collection for flx
- Create disruption queue collection (common)
"""


def trackme_schema_upgrade_2118(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2118, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Create Kvstore collections and transforms for thresholds management in splk-flx
        #

        components_to_process = []

        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")

        for component in components_to_process:

            # set
            transform_name = f"trackme_{component}_thresholds_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_thresholds_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}_thresholds"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2118, tenant_id="{tenant_id}", failed to create the collection, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2118, tenant_id="{tenant_id}", successfully created the collection, collection="{collection_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2118, tenant_id="{tenant_id}", failed to create the collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2118, tenant_id="{tenant_id}", failed to create the transform, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2118, tenant_id="{tenant_id}", successfully created the transform, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2118, tenant_id="{tenant_id}", failed to create the transform, transform="{transform_name}", exception="{str(e)}"'
                )

        #
        # Create the common collection and transform for disruption queue
        #

        # set
        transform_name = f"trackme_common_disruption_queue_tenant_{tenant_id}"
        collection_name = f"kv_trackme_common_disruption_queue_tenant_{tenant_id}"
        transform_fields = collections_dict[f"trackme_common_disruption_queue"]
        ko_acl = {
            "owner": vtenant_record.get("tenant_owner"),
            "sharing": trackme_default_sharing,
            "perms.write": tenant_roles_write_perms,
            "perms.read": tenant_roles_read_perms,
        }

        # create the KVstore collection
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
        data = {
            "tenant_id": tenant_id,
            "collection_name": collection_name,
            "collection_acl": ko_acl,
            "owner": vtenant_record.get("tenant_owner"),
        }

        kvstore_created = False
        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", create KVstore collection has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", successfully KVstore collection, collection="{collection_name}"'
                )
                kvstore_created = True
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", failed to create the KVstore collection, collection="{collection_name}", exception="{str(e)}"'
            )

        #
        # Only continue if successful
        #

        if kvstore_created:

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )
                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2118, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2118, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2118, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2118, procedure terminated'
    )
    return True


"""
In this function:
- Update stateful alerting definition for the Virtual Tenant
"""


def trackme_schema_upgrade_2119(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2119, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # update transforms definition for the Virtual Tenant
        #

        #
        # update main collection transforms
        #

        # set
        transform_name = f"trackme_stateful_alerting_tenant_{tenant_id}"
        collection_name = f"kv_trackme_stateful_alerting_tenant_{tenant_id}"
        transform_fields = collections_dict[f"trackme_stateful_alerting"]
        ko_acl = {
            "owner": vtenant_record.get("tenant_owner"),
            "sharing": trackme_default_sharing,
            "perms.write": tenant_roles_write_perms,
            "perms.read": tenant_roles_read_perms,
        }

        # delete the transform
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/delete_kvtransform'
        data = {
            "tenant_id": tenant_id,
            "transform_name": transform_name,
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2119, tenant_id="{tenant_id}", failed to delete the transform definition, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2119, tenant_id="{tenant_id}", successfully deleted transform definition, transform="{transform_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2119, tenant_id="{tenant_id}", failed to delete the transform definition, transform="{transform_name}", exception="{str(e)}"'
            )

        # create the transform
        url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
        data = {
            "tenant_id": tenant_id,
            "transform_name": transform_name,
            "transform_fields": transform_fields,
            "collection_name": collection_name,
            "transform_acl": ko_acl,
            "owner": vtenant_record.get("tenant_owner"),
        }

        try:
            response = requests.post(
                url,
                headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                data=json.dumps(data),
                verify=False,
                timeout=600,
            )
            if response.status_code not in (200, 201, 204):
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2119, tenant_id="{tenant_id}", create transform definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                )
            else:
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2119, tenant_id="{tenant_id}", successfully created transform definition, transform="{transform_name}"'
                )
        except Exception as e:
            logging.error(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2119, tenant_id="{tenant_id}", failed to create the transform definition, transform="{transform_name}", exception="{str(e)}"'
            )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2119, procedure terminated'
    )
    return True


"""
In this function:
- Update registered TrackMe alerts to call trackme_apply_maintenance_mode_v2 instead of trackme_apply_maintenance_mode
"""


def trackme_schema_upgrade_2121(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2121, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update alerts
        #

        # get alert object from field tenant_alert_objects
        alert_objects = vtenant_record.get("tenant_alert_objects", None)

        # check if we have alert objects
        if alert_objects:

            # load the alert objects as a json object
            try:
                alert_objects = json.loads(alert_objects)
            except Exception as e:
                # log alerts to be updated
                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", no alerts found for update.'
                )

            # get list of alerts
            alerts_list = alert_objects.get("alerts", [])

            # for each alert
            for alert_name in alerts_list:

                alert_current = None
                try:
                    alert_current = service.saved_searches[alert_name]
                except Exception as e:
                    pass

                if alert_current:
                    alert_current_search = alert_current.content.get("search")
                    alert_current_earliest_time = alert_current.content.get(
                        "dispatch.earliest_time"
                    )
                    alert_current_latest_time = alert_current.content.get(
                        "dispatch.latest_time"
                    )

                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", alert_name="{alert_name}", alert_current_search="{alert_current_search}"'
                    )

                    """
                    in alert_current_search, replace the following sentence using regex:

                    `trackme_apply_maintenance_mode`

                    by:

                    `trackme_apply_maintenance_mode_v2`

                    """

                    alert_new_search = re.sub(
                        r"`trackme_apply_maintenance_mode`",
                        r"`trackme_apply_maintenance_mode_v2`",
                        alert_current_search,
                    )

                    # update the alert definition
                    url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                    data = {
                        "tenant_id": tenant_id,
                        "report_name": alert_name,
                        "report_search": alert_new_search,
                        "earliest_time": alert_current_earliest_time,
                        "latest_time": alert_current_latest_time,
                    }

                    try:
                        response = requests.post(
                            url,
                            headers={
                                "Authorization": f'Splunk {reqinfo["session_key"]}'
                            },
                            data=json.dumps(data),
                            verify=False,
                            timeout=600,
                        )
                        if response.status_code not in (200, 201, 204):
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2121, tenant_id="{tenant_id}", update alert definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                            )
                        else:
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2121, tenant_id="{tenant_id}", successfully updated alert definition, alert="{alert_name}"'
                            )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2121, tenant_id="{tenant_id}", failed to update the alert definition, report="{alert_name}", exception="{str(e)}"'
                        )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2121, procedure terminated'
    )
    return True


"""
In this function:
- Update Vtenant central collection for the new splk-fqm components to be set as explicitly disabled in existint tenants
"""


def trackme_schema_upgrade_2122(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2121, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        # init the fqm component to be disabled
        vtenant_record["tenant_fqm_enabled"] = 0

        # update
        try:
            collection_vtenants.data.update(
                str(vtenant_key), json.dumps(vtenant_record)
            )
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2122, Virtual Tenant updated successfully, result="{json.dumps(vtenant_record, indent=2)}"'
            )

        except Exception as e:
            error_msg = f'task="{task_name}", task_instance_id={task_instance_id}, function trackme_schema_upgrade_2122, failed to update the Virtual Tenant record, exception="{str(e)}"'
            logging.error(error_msg)

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2222, procedure terminated'
    )
    return True


"""
In this function:
- Update fqm monitor trackers to include the trackme_fqm_get_description_extended macro
"""

def trackme_schema_upgrade_2123(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2123, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Update trackers
        #

        components_to_process = []

        # for all components except splk-cim
        if vtenant_record.get("tenant_fqm_enabled") == 1:
            components_to_process.append("fqm")

        for component in components_to_process:

            # A list to store trackers to be processed
            trackers_to_process = []

            try:
                tracker_kos = json.loads(
                    vtenant_record.get(f"tenant_{component}_hybrid_objects")
                )
                tracker_reports = tracker_kos.get("reports")
            except Exception as e:
                tracker_reports = []

            for tracker_report in tracker_reports:
                # include only _wrapper trackers
                if "fqm_monitor" in tracker_report and "_wrapper_" in tracker_report:
                    trackers_to_process.append(tracker_report)

            # loop through abstract reports
            for tracker_name in trackers_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="splk-{component}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # in tracker_current_search, replace the sentence:
                # | stats values(description) as description
                # with:
                # | `trackme_fqm_get_description_extended` | stats values(description) as description

                tracker_new_search = re.sub(
                    r"\| stats values\(description\) as description",
                    r"| `trackme_fqm_get_description_extended`\n| stats values(description) as description",
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2123, tenant_id="{tenant_id}", component="splk-{component}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2123, tenant_id="{tenant_id}", component="splk-{component}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2123, tenant_id="{tenant_id}", component="splk-{component}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # END
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2123, procedure terminated'
    )
    return True


"""
In this function:
- Update fqm trackers to include sort 0 _time to force processing on the search heads
"""

def trackme_schema_upgrade_2126(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2126, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Update trackers
        #

        components_to_process = []

        # for all components except splk-cim
        if vtenant_record.get("tenant_fqm_enabled") == 1:
            components_to_process.append("fqm")

        for component in components_to_process:

            # A list to store trackers to be processed
            trackers_to_process = []

            try:
                tracker_kos = json.loads(
                    vtenant_record.get(f"tenant_{component}_hybrid_objects")
                )
                tracker_reports = tracker_kos.get("reports")
            except Exception as e:
                tracker_reports = []

            for tracker_report in tracker_reports:
                # include only _wrapper trackers
                if "_wrapper_" in tracker_report:
                    trackers_to_process.append(tracker_report)

            # loop through abstract reports
            for tracker_name in trackers_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="splk-{component}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # in tracker_current_search, replace:
                # | fields *
                # with:
                # | fields * | sort 0 _time

                tracker_new_search = re.sub(
                    r"\| fields \*",
                    r"| fields * | sort 0 _time",
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2126, tenant_id="{tenant_id}", component="splk-{component}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2126, tenant_id="{tenant_id}", component="splk-{component}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2126, tenant_id="{tenant_id}", component="splk-{component}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # END
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2126, procedure terminated'
    )
    return True


"""
In this function:
- Update fqm monitor wrappers to include the call to trackme_fqm_get_description_extended
"""

def trackme_schema_upgrade_2128(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2128, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Update trackers
        #

        components_to_process = []

        # for all components except splk-cim
        if vtenant_record.get("tenant_fqm_enabled") == 1:
            components_to_process.append("fqm")

        for component in components_to_process:

            # A list to store trackers to be processed
            trackers_to_process = []

            try:
                tracker_kos = json.loads(
                    vtenant_record.get(f"tenant_{component}_hybrid_objects")
                )
                tracker_reports = tracker_kos.get("reports")
            except Exception as e:
                tracker_reports = []

            for tracker_report in tracker_reports:
                # include only _wrapper trackers
                if "_wrapper_" in tracker_report and "_monitor_" in tracker_report:
                    trackers_to_process.append(tracker_report)

            # loop through abstract reports
            for tracker_name in trackers_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", tracker_name="{tracker_name}", failed to get the tracker definition, exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                # do not proceed if the search already calls the macro
                if "trackme_fqm_get_description_extended" in tracker_current_search:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="splk-{component}", tracker_name="{tracker_name}", the search already calls the macro, skipping'
                    )
                    continue

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", component="splk-{component}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                )

                # in tracker_current_search, replace:
                # | stats values(description) as description,
                # with:
                # | `trackme_fqm_get_description_extended`
                # | stats values(description) as description,

                tracker_new_search = re.sub(
                    r"\| stats values\(description\) as description,",
                    r"| `trackme_fqm_get_description_extended`\n| stats values(description) as description,",
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2128, tenant_id="{tenant_id}", component="splk-{component}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2128, tenant_id="{tenant_id}", component="splk-{component}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2128, tenant_id="{tenant_id}", component="splk-{component}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # END
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2128, procedure terminated'
    )
    return True

"""
In this function:
- Fix defect which had led to the duplication of records in the dedicated trackers collection for the flx component
"""

def trackme_schema_upgrade_2130(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2130, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Update trackers
        #

        components_to_process = []

        # for all components except splk-cim
        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")

        for component in components_to_process:

            # Load the dedicated tracker collection
            tracker_collection_name = (
                f"kv_trackme_{component}_hybrid_trackers_tenant_{tenant_id}"
            )
            tracker_collection = service.kvstore[tracker_collection_name]

            # Get existing tracker records from dedicated collection
            existing_tracker_records = tracker_collection.data.query()

            # loop through the records, any record that does not have a value for tracker_id must be removed from the collection
            for tracker_record in existing_tracker_records:
                tracker_key = tracker_record.get("_key")
                tracker_id = tracker_record.get("tracker_id")

                if not tracker_id or len(tracker_id) == 0:
                    try:
                        tracker_collection.data.delete(json.dumps({"_key": tracker_key}))
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2130, tenant_id="{tenant_id}", component="splk-{component}", successfully removed the record {json.dumps(tracker_record, indent=2)} from the KVstore collection {tracker_collection_name}'
                        )
                    except Exception as e:
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2130, tenant_id="{tenant_id}", component="splk-{component}", failed to remove the record {json.dumps(tracker_record, indent=2)} from the KVstore collection {tracker_collection_name}, exception=\"{str(e)}\"'
                        )

            # Get updated tracker records after removing records without tracker_id
            updated_tracker_records = tracker_collection.data.query()
            
            # Group records by tracker_name to identify duplicates
            tracker_name_groups = {}
            for tracker_record in updated_tracker_records:
                tracker_name = tracker_record.get("tracker_name")
                if tracker_name:
                    if tracker_name not in tracker_name_groups:
                        tracker_name_groups[tracker_name] = []
                    tracker_name_groups[tracker_name].append(tracker_record)
            
            # Remove duplicates based on tracker_name, keeping only the first record for each tracker_name
            for tracker_name, records in tracker_name_groups.items():
                if len(records) > 1:
                    # Keep the first record, remove the rest
                    records_to_remove = records[1:]
                    for record_to_remove in records_to_remove:
                        tracker_key = record_to_remove.get("_key")
                        try:
                            tracker_collection.data.delete(json.dumps({"_key": tracker_key}))
                            logging.info(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2130, tenant_id="{tenant_id}", component="splk-{component}", successfully removed duplicate record with tracker_name="{tracker_name}" from the KVstore collection {tracker_collection_name}'
                            )
                        except Exception as e:
                            logging.error(
                                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2130, tenant_id="{tenant_id}", component="splk-{component}", failed to remove duplicate record with tracker_name="{tracker_name}" from the KVstore collection {tracker_collection_name}, exception=\"{str(e)}\"'
                            )

        #
        # END
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2130, procedure terminated'
    )
    return True


"""
In this function:
- Create drilldown searches collection for flx
- Create default metric collection for flx
"""


def trackme_schema_upgrade_2131(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2131, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # Create Kvstore collections and transforms for drilldown searches management in splk-flx
        #

        components_to_process = []

        if vtenant_record.get("tenant_flx_enabled") == 1:
            components_to_process.append("flx")

        for component in components_to_process:

            #
            # Drilldown search collection
            #

            # set
            transform_name = f"trackme_{component}_drilldown_searches_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_drilldown_searches_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}_drilldown_searches"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", failed to create the collection, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", successfully created the collection, collection="{collection_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", failed to create the collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", failed to create the transform, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", successfully created the transform, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", failed to create the transform, transform="{transform_name}", exception="{str(e)}"'
                )

            #
            # Default metric collection
            #

            # set
            transform_name = f"trackme_{component}_default_metric_tenant_{tenant_id}"
            collection_name = f"kv_trackme_{component}_default_metric_tenant_{tenant_id}"
            transform_fields = collections_dict[f"trackme_{component}_default_metric"]
            ko_acl = {
                "owner": vtenant_record.get("tenant_owner"),
                "sharing": trackme_default_sharing,
                "perms.write": tenant_roles_write_perms,
                "perms.read": tenant_roles_read_perms,
            }

            # create the collection
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvcollection'
            data = {
                "tenant_id": tenant_id,
                "collection_name": collection_name,
                "collection_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", failed to create the collection, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", successfully created the collection, collection="{collection_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", failed to create the collection, collection="{collection_name}", exception="{str(e)}"'
                )

            # create the transform
            url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/create_kvtransform'
            data = {
                "tenant_id": tenant_id,
                "transform_name": transform_name,
                "transform_fields": transform_fields,
                "collection_name": collection_name,
                "transform_acl": ko_acl,
                "owner": vtenant_record.get("tenant_owner"),
            }

            try:
                response = requests.post(
                    url,
                    headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                    data=json.dumps(data),
                    verify=False,
                    timeout=600,
                )

                if response.status_code not in (200, 201, 204):
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", failed to create the transform, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", successfully created the transform, transform="{transform_name}"'
                    )
            except Exception as e:
                logging.error(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2131, tenant_id="{tenant_id}", failed to create the transform, transform="{transform_name}", exception="{str(e)}"'
                )

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2131, procedure terminated'
    )
    return True


"""
In this function:
- Update Workload trackers (splk-wlk) to use the source splunk-svc-search-attribution instead of splunk-svc-consumer
- Update Flex trackers configured for SVC tracking to use the source splunk-svc-search-attribution instead of splunk-svc-consumer
"""


def trackme_schema_upgrade_2132(
    reqinfo, tenant_id, source_version, target_version, task_name, task_instance_id
):
    # get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=reqinfo["session_key"],
        timeout=600,
    )

    # Register the object summary in the vtenant collection
    collection_vtenants_name = "kv_trackme_virtual_tenants"
    collection_vtenants = service.kvstore[collection_vtenants_name]

    # Define the KV query search string
    query_string = {
        "tenant_id": tenant_id,
    }

    # log
    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, Starting function trackme_schema_upgrade_2132, tenant_id="{tenant_id}"'
    )

    # get the tenant record
    try:
        vtenant_record = collection_vtenants.data.query(query=json.dumps(query_string))[
            0
        ]
        vtenant_key = vtenant_record.get("_key")
        logging.debug(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was successfully found in the collection, query_string="{query_string}"'
        )

    except Exception as e:
        vtenant_key = None
        logging.error(
            f'task="{task_name}", task_instance_id={task_instance_id}, The vtenant_key was not found in the collection, query_string="{query_string}"'
        )

    if vtenant_key:

        tenant_id = vtenant_record.get("tenant_id")

        #
        # Vtenant account preferences
        #

        update_vtenant_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
        )

        #
        # Update the Splunk Remote account preferences
        #

        update_remote_account_configuration(
            reqinfo,
            task_name,
            task_instance_id,
            tenant_id,
            remote_account_default,
        )

        #
        # Get permissions and sharing levels
        #

        # get permissions
        tenant_roles_read_perms, tenant_roles_write_perms = get_permissions(
            vtenant_record
        )

        # TrackMe sharing level
        trackme_default_sharing = reqinfo["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        #
        # splk-wlk: Update Workload SVC trackers to switch to the new index
        #

        if (
            vtenant_record.get("tenant_wlk_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", component="wlk", processing with Workload tracker updates'
            )

            #
            # Trackers updates
            #

            # retrieve the list of scheduler reports to be processed
            collection_trackers_name = (
                f"kv_trackme_wlk_hybrid_trackers_tenant_{tenant_id}"
            )
            collection_trackers = service.kvstore[collection_trackers_name]

            trackers_dict = {}
            trackers_to_process = []
            trackers_to_process_kos = []

            # get records from the KVstore
            trackers_records = collection_trackers.data.query()

            for tracker_record in trackers_records:
                tracker_name = tracker_record.get("tracker_name")
                tracker_kos = tracker_record.get("knowledge_objects")

                # if the tracker_name starts by scheduler_
                if re.search("^splunkcloud_", tracker_name):
                    trackers_to_process.append(tracker_name)
                    trackers_to_process_kos.append(tracker_kos)

                    trackers_dict[tracker_name] = {
                        "knowledge_objects": json.loads(tracker_kos),
                    }

            # loop through the trackers (wrapper)
            for tracker_shortname in trackers_to_process:
                tracker_name = (
                    f"trackme_wlk_hybrid_{tracker_shortname}_wrapper_tenant_{tenant_id}"
                )

                tracker_kos = trackers_dict[tracker_shortname]["knowledge_objects"]

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", failed to retrieve the current search definition, tracker_name="{tracker_name}", exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_account = tracker_kos["properties"][0]["account"]
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                logging.info(
                    f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", tracker_name="{tracker_name}", account={tracker_account}, tracker_current_search="{tracker_current_search}", tracker_kos="{json.dumps(tracker_kos, indent=2)}"'
                )

                # in tracker_current_search, replace:
                #  splunk-svc-consumer
                # with:
                #  splunk-svc-search-attribution

                tracker_new_search = re.sub(
                    r"splunk-svc-consumer",
                    r"splunk-svc-search-attribution",
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # splk-flx: Update Flex Object trackers that would track SVC usage
        #

        if (
            vtenant_record.get("tenant_flx_enabled") == 1
            and vtenant_record.get("tenant_replica") == 0
        ):
            logging.info(
                f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", component="flx", processing with Flex Objects tracker updates'
            )

            component = "flx"

            # A list to store trackers to be processed
            trackers_to_process = []

            try:
                tracker_kos = json.loads(
                    vtenant_record.get(f"tenant_{component}_hybrid_objects")
                )
                tracker_reports = tracker_kos.get("reports")
            except Exception as e:
                tracker_reports = []

            for tracker_report in tracker_reports:
                if "_wrapper_" in tracker_report:
                    trackers_to_process.append(tracker_report)

            # loop through abstract reports
            for tracker_name in trackers_to_process:

                # get the current search definition
                try:
                    tracker_current = service.saved_searches[tracker_name]
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", failed to retrieve the current search definition, tracker_name="{tracker_name}", exception="{str(e)}"'
                    )
                    continue

                tracker_current_search = tracker_current.content.get("search")
                tracker_current_earliest_time = tracker_current.content.get(
                    "dispatch.earliest_time"
                )
                tracker_current_latest_time = tracker_current.content.get(
                    "dispatch.latest_time"
                )

                # get the search definition, if it contains splunk-svc-consumer, process, otherwise skip
                if re.search(
                    "splunk-svc-consumer", tracker_current_search
                ):
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, processing Flex tracker, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                    )
                else:
                    logging.info(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, skipping Flex tracker, tenant_id="{tenant_id}", tracker_name="{tracker_name}", tracker_current_search="{tracker_current_search}"'
                    )
                    continue

                # in tracker_current_search, replace the sentence:
                #  splunk-svc-consumer
                # with:
                #  splunk-svc-search-attribution

                tracker_new_search = re.sub(
                    r"splunk-svc-consumer",
                    r"splunk-svc-search-attribution",
                    tracker_current_search,
                )

                # update the search definition
                url = f'{reqinfo["server_rest_uri"]}/services/trackme/v2/configuration/admin/update_report'
                data = {
                    "tenant_id": tenant_id,
                    "report_name": tracker_name,
                    "report_search": tracker_new_search,
                    "earliest_time": tracker_current_earliest_time,
                    "latest_time": tracker_current_latest_time,
                    "schedule_window": "5",
                }

                try:
                    response = requests.post(
                        url,
                        headers={"Authorization": f'Splunk {reqinfo["session_key"]}'},
                        data=json.dumps(data),
                        verify=False,
                        timeout=600,
                    )
                    if response.status_code not in (200, 201, 204):
                        logging.error(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", component="splk-{component}", update report definition has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                        )
                    else:
                        logging.info(
                            f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", component="splk-{component}", successfully updated report definition, report="{tracker_name}"'
                        )
                except Exception as e:
                    logging.error(
                        f'task="{task_name}", task_instance_id={task_instance_id}, schema migration 2132, tenant_id="{tenant_id}", component="splk-{component}", failed to update the report definition, report="{tracker_name}", exception="{str(e)}"'
                    )

        #
        # END
        #

    logging.info(
        f'task="{task_name}", task_instance_id={task_instance_id}, tenant_id="{tenant_id}", schema migration 2132, procedure terminated'
    )
    return True
