#!/usr/bin/env python
# coding=utf-8

__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Standard library imports
import os
import sys
import time
import json

# Logging imports
import logging
from logging.handlers import RotatingFileHandler

# Networking imports
import urllib3

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# set logging
filehandler = RotatingFileHandler(
    "%s/var/log/splunk/trackme_extract_splk_cim.log" % splunkhome,
    mode="a",
    maxBytes=10000000,
    backupCount=1,
)
formatter = logging.Formatter(
    "%(asctime)s %(levelname)s %(filename)s %(funcName)s %(lineno)d %(message)s"
)
logging.Formatter.converter = time.gmtime
filehandler.setFormatter(formatter)
log = logging.getLogger()  # root logger - Good to get it only once.
for hdlr in log.handlers[:]:  # remove the existing file handlers
    if isinstance(hdlr, logging.FileHandler):
        log.removeHandler(hdlr)
log.addHandler(filehandler)  # set the new handler
# set the log level to INFO, DEBUG as the default is ERROR
log.setLevel(logging.INFO)

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# import Splunk libs
from splunklib.searchcommands import (
    dispatch,
    StreamingCommand,
    Configuration,
    Option,
    validators,
)

# Import trackme libs
from trackme_libs import trackme_reqinfo


@Configuration(distributed=False)
class TrackMeMergeSplkDhm(StreamingCommand):
    # status will be statically defined as imported

    def stream(self, records):
        # Get request info and set logging level
        reqinfo = trackme_reqinfo(
            self._metadata.searchinfo.session_key, self._metadata.searchinfo.splunkd_uri
        )
        log.setLevel(reqinfo["logging_level"])

        # Loop in the results
        for splrecord in records:
            try:
                # get tenant_id, object
                tenant_id = splrecord["tenant_id"]
                object_name = splrecord["object"]

                # get the results dict
                current_dict = json.loads((splrecord["cim_tracking_results"]))

                # get the cim_tracking_results dict
                cim_tracking_results = current_dict["cim_tracking_results"]

                for cim_field in cim_tracking_results:
                    #
                    # get cim field metrics
                    #

                    # get status and create status_num turned into a numerical value
                    status = cim_tracking_results[cim_field]["status"]
                    if status == "green":
                        status_num = 0
                    elif status == "red":
                        status_num = 1
                    else:
                        status_num = 2

                    # get other metrics
                    total_count = cim_tracking_results[cim_field]["total_count"]
                    count_unknown = cim_tracking_results[cim_field]["count_unknown"]
                    count_not_unknown = cim_tracking_results[cim_field][
                        "count_not_unknown"
                    ]
                    pct_coverage_unknown = cim_tracking_results[cim_field][
                        "pct_coverage_unknown"
                    ]
                    pct_coverage_compliant = cim_tracking_results[cim_field][
                        "pct_coverage_compliant"
                    ]

                    yield {
                        "_time": time.time(),
                        "tenant_id": tenant_id,
                        "object_category": "splk-cim",
                        "object": object_name,
                        "cim_field": cim_field,
                        "trackme.splk.cim.status": status_num,
                        "trackme.splk.cim.total_count": total_count,
                        "trackme.splk.cim.count_unknown": count_unknown,
                        "trackme.splk.cim.count_not_unknown": count_not_unknown,
                        "trackme.splk.cim.pct_coverage_unknown": pct_coverage_unknown,
                        "trackme.splk.cim.pct_coverage_compliant": pct_coverage_compliant,
                        "cim_tracking_results": json.dumps(
                            cim_tracking_results[cim_field], indent=1
                        ),
                    }
                    logging.debug(
                        f'tenant_id="{tenant_id}", object_category="splk-cim", object="{object_name}", cim_field="{cim_field}", success'
                    )

            except Exception as e:
                logging.error(f'An exception was encountered="{str(e)}"')
                raise Exception(f'An exception was encountered="{str(e)}"')


dispatch(TrackMeMergeSplkDhm, sys.argv, sys.stdin, sys.stdout, __name__)
