define([
    'jquery',
    'underscore',
    'api/SplunkVisualizationBase',
    'api/SplunkVisualizationUtils',
    'd3',
    'moment',
    '../contrib/d3-timeline',
    'bootstrap/js/dist/tooltip',
], ($, _, SplunkVisualizationBase, vizUtils, d3) => {
    // Truncates a string to a length, optionally adding a suffix
    const truncate = function truncate(str, maxLength, suffix) {
        const updatedMaxLength = maxLength || 25;
        const updatedSuffix = suffix || '...';
        let updatedStr = str || 'null';
        if (updatedStr.length > updatedMaxLength) {
            updatedStr = updatedStr.substring(0, updatedMaxLength + 1);
            updatedStr += updatedSuffix;
        }
        return updatedStr;
    };

    const TIME_FORMATS = {
        DAYS: '%x',
        MINUTES: '%H:%M',
        SECONDS: '%X',
        SUBSECONDS: '%X %L',
    };

    const isDarkTheme = vizUtils.getCurrentTheme && vizUtils.getCurrentTheme() === 'dark';

    const LEGEND_WIDTH = 200;
    const LEGEND_MARGIN = 40;
    const LEGEND_RECT_SIZE = 18;
    const LEGEND_SPACING = 8;
    const LEGEND_MARGIN_TOP = 30;

    const LABEL_WIDTH = 150;
    const TIMELINE_LABEL_MARGIN = 15;
    const ITEM_HEIGHT = 20;
    const ITEM_MARGIN = 5;
    const CHART_PADDING = 15;

    return SplunkVisualizationBase.extend({
        initialize(...args) {
            SplunkVisualizationBase.prototype.initialize.apply(this, args);

            this.$el = $(this.el);
            this.$el.addClass('splunk-timeline');
            if (isDarkTheme) {
                this.$el.addClass('dark');
            }

            this.compiledTooltipTemplate = _.template(this.tooltipTemplate);
            this.tooltipContainer = $('<div class="splunk-timeline-tooltip"></div>').appendTo('body');
        },

        formatData(data) {
            const { fields } = data;
            const { rows } = data;
            const config = this.getConfig();
            const useColors = vizUtils.normalizeBoolean(this.getEscapedProperty('useColors', config));
            const timeIndex = 0;
            const resourceIndex = 1;
            const durationIndex = useColors ? 3 : 2;

            if (rows.length < 1 && fields.length < 1) {
                return false;
            }

            if (useColors && fields.length < 3) {
                throw new SplunkVisualizationBase.VisualizationError(
                    'Check the Statistics tab. To generate a colorized timeline, the results table must include columns representing these three dimension types: <time>, <resource>, <color>.',
                );
            }

            const groups = {};
            const result = [];
            const colorCategories = {};

            // 0 -> _time
            // 1 -> resourceField
            // 2 -> [colorField | duration]
            // 3 -> [duration]
            // n optional key value pairs
            let minStartingTime = +new Date();
            let maxEndingTime = 0;
            let nonNumericalCategories = false;

            _.each(rows, (row) => {
                const group = row[resourceIndex];
                // if row[2] is a number we assume it's epoch time.
                // epoch time is specified in seconds, so it has to be multiplied by 1000
                const startTime = _.isNaN(+row[timeIndex])
                    ? +vizUtils.parseTimestamp(row[timeIndex])
                    : +row[timeIndex] * 1000;

                if (_.isNaN(startTime)) {
                    throw new SplunkVisualizationBase.VisualizationError(
                        `Invalid time format specified: ${vizUtils.escapeHtml(row[timeIndex])}. ` +
                            `Supported time formats are RFC2822, ISO 8601, and epoch time`,
                    );
                }
                let endTime = false;
                if (!_.isNaN(+row[durationIndex]) && startTime + +row[durationIndex] !== startTime) {
                    endTime = Math.round(startTime + +row[durationIndex]);
                }
                if (!groups[group]) {
                    groups[group] = [];
                }
                const entry = {
                    starting_time: startTime,
                };
                minStartingTime = Math.min(minStartingTime, startTime);
                maxEndingTime = Math.max(maxEndingTime, endTime || startTime);
                if (endTime) {
                    entry.ending_time = Math.round(endTime);
                } else {
                    entry.display = 'circle';
                }
                let category = useColors ? row[2] : 'default_category';
                if (_.isNaN(+category)) {
                    nonNumericalCategories = true;
                } else {
                    category = +category;
                }
                colorCategories[category] = 1;
                entry.resource = row[resourceIndex];
                entry.category = category;
                entry.class = `ccat-${category}`;

                const sliceIndex = useColors ? 4 : 3;
                const rowKeyVals = row.slice(sliceIndex);
                const meta = {};
                _.each(rowKeyVals, (m, i) => {
                    meta[fields[i + sliceIndex].name] = m;
                });
                entry.meta = meta;

                groups[group].push(entry);
            });

            _.each(groups, (group, key) => {
                result.push({
                    label: key,
                    times: group,
                });
            });

            return {
                nonNumericalCategories,
                beginning: minStartingTime,
                ending: maxEndingTime,
                chartData: result,
                colorCategories: _.keys(colorCategories),
                fields,
                indexes: {
                    resourceIndex,
                    timeIndex,
                    durationIndex,
                },
            };
        },

        updateView(data, config) {
            if (!data || data.length < 1) {
                return;
            }
            this.$el.empty();
            this.tooltipContainer.empty();
            this.useDrilldown = this.isEnabledDrilldown(config);

            const { fields } = data;
            const { indexes } = data;
            let width = this.$el.width();
            let height = this.$el.height();

            const useColors = vizUtils.normalizeBoolean(this.getEscapedProperty('useColors', config));
            const colorMode = this.getEscapedProperty('colorMode', config) || 'categorical';
            const numOfBins = this.getEscapedProperty('numOfBins', config) || 6;
            const minColor = this.getEscapedProperty('minColor', config) || '#FFE8E8';
            const maxColor = this.getEscapedProperty('maxColor', config) || '#DA5C5C';
            const axisTimeFormat = TIME_FORMATS[this.getEscapedProperty('axisTimeFormat', config) || 'SECONDS'];
            const tooltipTimeFormat = TIME_FORMATS[this.getEscapedProperty('tooltipTimeFormat', config) || 'SECONDS'];

            let colorScale;
            let chartWidth;
            let categoryScale;

            this.useColors = useColors;

            if (useColors) {
                chartWidth = width - LEGEND_WIDTH - LEGEND_MARGIN;

                if (colorMode === 'categorical') {
                    categoryScale = d3.scale.ordinal().domain(data.colorCategories).range(data.colorCategories);
                    colorScale = d3.scale
                        .ordinal()
                        .domain(data.colorCategories)
                        .range(vizUtils.getColorPalette('splunkCategorical'));
                } else {
                    if (data.nonNumericalCategories) {
                        throw new SplunkVisualizationBase.VisualizationError(
                            'Invalid color field type specified for sequential colorization. ' +
                                'Sequential colorization requires a numerical color field',
                        );
                    }
                    const colorCategories = data.colorCategories.map((item) => parseInt(item, 10));
                    const min = _.min(colorCategories);
                    const max = _.max(colorCategories);
                    const domain = [];
                    const range = [];
                    const interpolateNum = d3.interpolateRound(min, max);
                    const interpolateColor = d3.interpolateHcl(minColor, maxColor); // Rgb, Hcl, Hsl

                    for (let x = 0; x < numOfBins; x += 1) {
                        domain.push(interpolateNum(x / (numOfBins - 1)));
                        range.push(interpolateColor(x / (numOfBins - 1)));
                    }

                    colorScale = d3.scale.ordinal().domain(domain).range(range);

                    const categoryDomain = [];
                    const categoryRange = [];

                    // binning
                    for (let i = 0; i < colorCategories.length; i += 1) {
                        const colorCategory = colorCategories[i];
                        let bin = -1;
                        for (let o = 0; o < domain.length; o += 1) {
                            if (domain[o] <= colorCategory) {
                                bin += 1;
                            }
                        }
                        categoryDomain.push(colorCategory);
                        categoryRange.push(domain[bin]);
                    }
                    categoryScale = d3.scale.ordinal().domain(categoryDomain).range(categoryRange);
                }
            } else {
                chartWidth = width - 25;
                categoryScale = d3.scale.ordinal().domain(data.colorCategories).range(data.colorCategories);
                colorScale = d3.scale.ordinal().domain(data.colorCategories).range(['rgb(30, 147, 198)']);
            }

            const containerEl = d3.select(this.el);

            const that = this;

            const { tooltipContainer } = this;
            const chart = d3
                .timeline()
                .orient('top')
                .showAxisTop()
                .stack()
                .colors((d) => colorScale(categoryScale(d)))
                .colorProperty('category')
                .width(chartWidth)
                // if there is no interval (i.e. only one event) show a 100ms interval
                .ending(data.beginning !== data.ending ? data.ending : data.ending + 100)
                .mouseover((d, i, el) => {
                    el.attr('fill-opacity', 0.6).attr('stroke-width', 2);

                    // mute all elements of another color category
                    containerEl
                        .selectAll('circle, rect:not(.row-green-bar)')
                        .transition(200)
                        .style('opacity', function opacity() {
                            return el.attr('class') === d3.select(this).attr('class') ? 1 : 0.1;
                        });

                    // label alignment and content generation
                    const { tagName } = el.node();
                    let elementX = 0;
                    let elementY = 0;

                    const timeSpanString =
                        d3.time.format(tooltipTimeFormat)(new Date(d.starting_time)) +
                        (d.ending_time ? ` - ${d3.time.format(tooltipTimeFormat)(new Date(d.ending_time))}` : '');

                    const tooltipContent = that.compiledTooltipTemplate({
                        timeSpan: timeSpanString,
                        firstFieldName: vizUtils.escapeHtml(fields[indexes.resourceIndex].name),
                        secondFieldName: useColors ? vizUtils.escapeHtml(fields[2].name) : '',
                        resource: vizUtils.escapeHtml(d.resource),
                        category: useColors ? vizUtils.escapeHtml(d.category) : false,
                        color: useColors ? colorScale(categoryScale(d.category)) : 'white',
                    });

                    if (tagName === 'rect') {
                        elementX = +el.attr('x');
                        elementY = +el.attr('y');
                    } else if (tagName === 'circle') {
                        // eslint-disable-next-line no-unused-vars
                        elementX = +el.attr('cx');
                        elementY = +el.attr('cy');
                    }

                    let placement;
                    if (elementY > height / 2) {
                        placement = 'top';
                    } else {
                        placement = 'bottom';
                    }

                    $(el[0][0])
                        .tooltip({
                            animation: false,
                            title: tooltipContent,
                            html: true,
                            container: tooltipContainer,
                            placement,
                            template:
                                '<div class="tooltip-timeline" role="tooltip"><div class="arrow"></div><div class="tooltip-inner"></div></div>',
                            boundary: this.el.firstChild,
                            // Magic secret advanced config options:
                            // See https://splunk.atlassian.net/browse/SPL-228304?focusedCommentId=10012969
                            popperConfig: {
                                modifiers: {
                                    preventOverflow: {
                                        padding: 0,
                                    }
                                }
                            }
                        })
                        .tooltip('show');
                })
                .mouseout((d, i, el) => {
                    el.attr('fill-opacity', 0.5).attr('stroke-width', 1);
                    containerEl.selectAll('circle, rect').transition(200).style('opacity', 1);

                    $(el).tooltip('dispose');
                })
                .click(this.timelineClick.bind(this))
                .labelFormat((label) => truncate(label, 15))
                .itemHeight(ITEM_HEIGHT)
                .itemMargin(ITEM_MARGIN)
                .labelMargin(TIMELINE_LABEL_MARGIN)
                .tickFormat({
                    format: d3.time.format(axisTimeFormat),
                    tickTime: d3.time.minutes,
                    numTicks: 6,
                    tickSize: 3,
                })
                .background((d, i) => {
                    if (isDarkTheme) {
                        return i % 2 === 0 ? '#2B3033' : '#31373E';
                    }
                    return i % 2 === 0 ? '#F5F5F5' : 'white';
                })
                .fullLengthBackgrounds(true)
                .margin({ left: LABEL_WIDTH, right: 30, top: 30, bottom: 0 });

            width = this.$el.width();
            height = this.$el.height();

            const svgHeight = Math.max(
                // chart height
                (data.chartData.length + 1) * (ITEM_HEIGHT + 2 * ITEM_MARGIN) + 2 * CHART_PADDING,
                // legend height
                (((colorScale.domain() || []).length + 1) * (LEGEND_RECT_SIZE + LEGEND_SPACING) + LEGEND_MARGIN_TOP) *
                    // if there's no legend we don't need to take it into
                    // the chart height calculation
                    (useColors ? 1 : 0),
            );

            const svg = d3
                .select(this.el)
                .append('svg')
                .attr('width', width - 20)
                .attr('height', svgHeight)
                .style('padding', `${CHART_PADDING}px`)
                .datum(data.chartData)
                .call(chart);

            svg.selectAll('circle,rect:not(.row-green-bar)')
                .attr('r', 8)
                .attr('stroke-width', 1)
                .attr('stroke-opacity', 1)
                .attr('stroke', (d) => colorScale(categoryScale(d.category)))
                .attr('fill-opacity', 0.5)
                .attr('class', (d) => `ccat-${colorMode === 'categorical' ? d.category : categoryScale(d.category)}`);

            // eslint-disable-next-line no-unused-vars
            const tooltip = d3.select(this.el).append('div').attr('class', 'modviz-tooltip').style('opacity', 0);

            // add timeline label tooltips (for truncated labels)
            svg.selectAll('.timeline-label')
                .append('title')
                .text((d, i) => d[i].label);

            if (useColors) {
                const legend = svg
                    .selectAll('.legend')
                    .data(colorScale.domain())
                    .enter()
                    .append('g')
                    .attr('class', 'legend')
                    .attr('transform', (d, i) => {
                        const rectHeight = LEGEND_RECT_SIZE + LEGEND_SPACING;
                        const horz = width - LEGEND_WIDTH;
                        const vert = i * rectHeight + LEGEND_MARGIN_TOP;
                        return `translate(${horz},${vert})`;
                    })
                    .on('mouseover', function mouseover() {
                        const currentLegendEl = d3.select(this);
                        currentLegendEl.select('text').style('font-weight', 'bold');
                        const rect = currentLegendEl.select('rect');
                        containerEl
                            .selectAll('circle, rect')
                            .transition(200)
                            .style('opacity', function opacity() {
                                return rect.attr('class') === d3.select(this).attr('class') ? 1 : 0.1;
                            });
                    })
                    .on('mouseout', function mouseout() {
                        d3.select(this).select('text').style('font-weight', 'normal');
                        containerEl.selectAll('circle, rect').transition(250).style('opacity', 1);
                    });
                legend
                    .append('rect')
                    .attr('width', LEGEND_RECT_SIZE)
                    .attr('height', LEGEND_RECT_SIZE)
                    .attr('class', (d) => `ccat-${d}`)
                    .attr('fill', colorScale)
                    .attr('fill-opacity', 0.5)
                    .attr('stroke-width', 1)
                    .style('stroke', colorScale);

                legend
                    .append('text')
                    .attr('x', LEGEND_RECT_SIZE + 2 * LEGEND_SPACING)
                    .attr('y', LEGEND_RECT_SIZE - LEGEND_SPACING + LEGEND_RECT_SIZE / 6)
                    .attr('fill', () => (isDarkTheme ? '#fff' : '#000'))
                    .text((d) => (colorMode === 'categorical' ? '' : '>= ') + truncate(d, 18))
                    .append('title')
                    .text((d) => (colorMode === 'categorical' ? '' : '>= ') + d);
            }

            if (this.useDrilldown) {
                this.$el.addClass('timeline-drilldown');
            } else {
                this.$el.removeClass('timeline-drilldown');
            }
            // eslint-disable-next-line consistent-return
            return this;
        },

        getInitialDataParams() {
            return {
                outputMode: SplunkVisualizationBase.ROW_MAJOR_OUTPUT_MODE,
                count: 10000,
            };
        },

        reflow() {
            this.invalidateUpdateView();
        },

        getEscapedProperty(name, config) {
            const propertyValue = config[this.getPropertyNamespaceInfo().propertyNamespace + name];
            return vizUtils.escapeHtml(propertyValue);
        },

        getConfig() {
            // eslint-disable-next-line no-underscore-dangle
            return this._config;
        },

        timelineClick(d) {
            const { fields } = this.getCurrentData();
            const drilldownDescription = {
                action: SplunkVisualizationBase.FIELD_VALUE_DRILLDOWN,
                data: {},
            };
            drilldownDescription.data[fields[1].name] = d.resource;
            if (this.useColors) {
                drilldownDescription.data[fields[2].name] = d.category;
            }

            const timeField = fields[0];
            // If the starting time is Splunk's event indexed time field, then drilldown
            // to a custom time range.
            // If not, perform a field-value match on the starting time.
            if (timeField.name === '_time') {
                const startingTimeSeconds = d.starting_time / 1000;
                const endingTimeSeconds = d.ending_time / 1000;

                drilldownDescription.earliest = startingTimeSeconds;
                drilldownDescription.latest = endingTimeSeconds + 0.001;
            } else {
                drilldownDescription.data[timeField.name] = d.starting_time;
            }

            this.drilldown(drilldownDescription, d3.event);
        },

        isEnabledDrilldown(config) {
            if (
                config['display.visualizations.custom.drilldown'] &&
                config['display.visualizations.custom.drilldown'] === 'all'
            ) {
                return true;
            }
            return false;
        },

        tooltipTemplate:
            // eslint-disable-next-line no-multi-str
            '\
                <p class="time-span-label"><%= timeSpan %></p>\
                <div class="tooltip-meta">\
                    <p><span><%= firstFieldName %>: </span><span><%= resource %></span></p>\
                    <% if (category) { %>\
                        <p><span><%= secondFieldName %>: </span><span style="color:<%= color %>;"><%= category %></span></p>\
                    <% } %>\
                </div>\
        ',
    });
});
