# encoding = utf-8

import os
import sys
import requests
import json
import urllib3
from urllib.parse import urlparse, urlencode, parse_qs
import time
import hashlib
from trackme_libs import trackme_reqinfo, trackme_idx_for_tenant, trackme_state_event
import splunklib.client as client

import smtplib
import ssl
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.mime.image import MIMEImage
from email.mime.application import MIMEApplication

import base64
from io import BytesIO

# Disable urllib3 warnings for SSL
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# set splunkhome
splunkhome = os.environ["SPLUNK_HOME"]

# append libs
sys.path.append(os.path.join(splunkhome, "etc", "apps", "trackme", "lib"))

from trackme_libs import run_splunk_search

# import trackme libs utils
from trackme_libs_utils import decode_unicode, encode_unicode, remove_leading_spaces, normalize_anomaly_reason

#
# Python 3.9 and higher only - charts generation is not support for older versions
#

# only attempt to import if python version is 3.9 or higher
python_version = os.environ.get("PYTHON_VERSION", "unknown")
python_compatible = False
pygal_available = False
png_conversion_available = False
png_import_error = "unknown"

if sys.version_info >= (3, 9):

    # for further use in the code
    python_compatible = True

    # pygal might fail to load with older versions of Python
    try:
        import pygal
        from pygal.style import DarkStyle

        pygal_available = True
    except (ImportError, SyntaxError) as e:
        pygal = None
        DarkStyle = None
        pygal_available = False

    png_conversion_available = True
    png_import_error = None
    try:
        from svglib.svglib import svg2rlg
        from reportlab.graphics import renderPM
    except (ImportError, SyntaxError) as e:
        png_conversion_available = False
        png_import_error = str(e)


def safe_float(value):
    """
    Function to safely convert a value to a float

    Args:
        value: The value to convert to a float

    Returns:
        The value as a float, or None if the value is not a float
    """

    try:
        return float(value)
    except (TypeError, ValueError):
        return None


def generate_message_id():
    """
    Function to generate a message ID

    Args:
        None

    Returns:
        A message ID
    """

    # Create a random message ID with your domain
    random_hash = hashlib.sha256(str(time.time()).encode()).hexdigest()[:16]
    return random_hash


def generate_drilldown_link(
    parsed_url, drilldown_root_uri, tenant_id, object_category, keyid_value
):
    """
    Function to generate the drilldown link for a given object

    Args:
        drilldown_root_uri: The root URI to build the drilldown link
        object: The object to build the drilldown link for

    Returns:
        The drilldown link
    """

    # Extract the server name (hostname) and server port
    server_protocol = parsed_url.scheme
    server_name = parsed_url.hostname
    server_port = parsed_url.port

    # extract the component suffix as splk-<component>
    component_suffix = object_category.split("-")[-1]

    # Base URL without query parameters
    base_url = f"{server_protocol}://{server_name}"
    if not (server_protocol == "https" and server_port == 443) and not (
        server_protocol == "http" and server_port == 80
    ):
        base_url += f":{server_port}"

    # Path and query parameters
    path = "/app/trackme/TenantHome"
    query_params = {
        "tenant_id": tenant_id,
        "component": component_suffix,
        "keyid": keyid_value,
    }

    # Construct the full URL
    if drilldown_root_uri:
        # Parse the drilldown_root_uri
        parsed_uri = urlparse(drilldown_root_uri)
        base_url = f"{parsed_uri.scheme}://{parsed_uri.netloc}"

        # Preserve the path from the drilldown_root_uri
        root_path = parsed_uri.path.rstrip("/")  # Strip trailing slash for consistency
        full_path = f"{root_path}{path}"  # Combine root path with the specific app path

        # Adjust query parameters if needed
        _, adjusted_query_params = adjust_query_params(
            drilldown_root_uri, query_params.copy()
        )
        drilldown_link = f"{base_url}{full_path}?{urlencode(adjusted_query_params)}"
    else:
        base_url = f"{server_protocol}://{server_name}"
        if not (server_protocol == "https" and server_port == 443) and not (
            server_protocol == "http" and server_port == 80
        ):
            base_url += f":{server_port}"
        full_path = path  # Default to the base path if no drilldown_root_uri provided
        drilldown_link = f"{base_url}{full_path}?{urlencode(query_params)}"

    return drilldown_link


def adjust_query_params(url, query_params):
    """
    Function to adjust the query parameters for the drilldown link

    Args:
        url: The URL to adjust the query parameters for
        query_params: The query parameters to adjust

    Returns:
        The adjusted query parameters
    """

    parsed_url = urlparse(url)
    query_dict = parse_qs(parsed_url.query)
    if "tenant_id" in query_dict:
        query_params.pop("tenant_id", None)
    separator = "&" if parsed_url.query else "?"
    return separator, query_params


def get_priority_from_main_kvstore(helper, collection, object_id):
    """
    Function to get the priority value from the main KVstore collection for a given object_id

    Args:
        collection: The collection to query
        object_id: The object ID to query

    Returns:
        The priority value from the KVstore record, or None if not found or exception occurs
    """

    # Define the KV query
    query_string = {
        "_key": object_id,
    }

    # Get the current record
    try:
        kvrecord = collection.data.query(query=json.dumps(query_string))[0]
        helper.log_debug(
            f"main collection kvrecord for priority: {json.dumps(kvrecord, indent=2)}"
        )
        priority = kvrecord.get("priority")

        if priority:
            helper.log_debug(
                f"Found priority '{priority}' in KVstore for object_id '{object_id}'"
            )
            return priority
        else:
            helper.log_debug(
                f"No priority found in KVstore for object_id '{object_id}'"
            )
            return None

    except Exception as e:
        helper.log_debug(
            f"Exception getting priority from KVstore for object_id '{object_id}': {str(e)}"
        )
        return None


def detect_image_type(b64data):
    """
    Function to detect the type of image from a base64-encoded string
    """

    raw = base64.b64decode(b64data)[:32]
    if raw.startswith(b"\x89PNG\r\n\x1a\n"):
        return "png"
    elif raw.startswith(b"<?xml") or raw.startswith(b"<svg"):
        return "svg"
    else:
        return "unknown"


def build_cid(name):
    """
    Generate a consistent, sanitized Content-ID for inline image references.
    """
    return "".join(c if c.isalnum() else "_" for c in name.strip()) + "@trackme"


def send_trackme_email(
    helper,
    config,
    message_id,
    to_address,
    subject,
    in_reply_to=None,
    reference_chain=None,
    format="html",
    tenant_id=None,
    alias=None,
    object=None,
    object_state=None,
    object_category=None,
    object_id=None,
    priority=None,
    anomaly_reason=None,
    messages=None,
    message_source=None,
    message_source_id=None,
    transition_message=None,
    environment_name=None,
    drilldown_link=None,
    incident_id=None,
    detection_time=None,
    chart_base64_dict=None,
    alert_status=None,
    timerange_charts="24h",
    python_compatible=False,
    charts_option_enabled=False,
):
    """
    Function to send an email

    Args:
        config: The email account settings
        message_id: The message ID
        to_address: The address to send the email to
        subject: The subject of the email
        body: The body of the email

    Notes:
        - For Splunk Cloud vetting purposes, email security is mandatory except for the localhost SMTP server

    """

    # attempt to re-encode the object value
    try:
        object = encode_unicode(object)
    except Exception as e:
        helper.log_error(f"task=send_trackme_email, Error encoding object: {str(e)}")
        pass

    # check target, if not localhost for the local MTA, then security is mandatory, raise an error if not configured
    email_security = config.get("email_security", None)
    email_account = config.get("account", "localhost")
    if not email_security and email_account != "localhost":
        raise RuntimeError("Email security is mandatory for external email delivery")
    elif email_security not in ["ssl", "tls"] and email_account != "localhost":
        raise RuntimeError(
            "Invalid email security configuration, Email security must be ssl or tls"
        )

    # check charts
    chart_base64_dict = chart_base64_dict or {}
    incidents_flipping_charts = []
    performance_charts = []

    # split charts into two sections
    for chart_id, chart in chart_base64_dict.items():
        chart_title = chart.get("chart_title")
        chart_b64 = chart.get("base64")

        if chart_title in ("incidents_events", "flipping_events", "state_events"):
            incidents_flipping_charts.append(
                {"chart_title": chart_title, "chart_b64": chart_b64}
            )
        else:
            performance_charts.append(
                {"chart_title": chart_title, "chart_b64": chart_b64}
            )

    # Outer: alternative (plain + HTML)
    outer = MIMEMultipart("alternative")
    outer["Subject"] = subject
    outer["From"] = config["sender_email"]
    outer["To"] = to_address

    # Generate domain for Message-ID
    domain = (
        config["sender_email"].split("@")[-1]
        if "@" in config["sender_email"]
        else "trackme"
    )
    outer["Message-ID"] = f"<{message_id}@{domain}>"
    if in_reply_to:
        outer["In-Reply-To"] = f"<{in_reply_to}@{domain}>"
    if reference_chain:
        outer["References"] = " ".join(f"<{mid}@{domain}>" for mid in reference_chain)

    # Prepare footer
    footer = config.get(
        "email_footer",
        "This is an automated email, please do not reply directly to this email.",
    )

    # if generate_chart is 1 and python_compatible is False, add to the footer the message:
    if charts_option_enabled and not python_compatible:
        footer += "\n\nEmbedded charts generation is enabled but this Splunk version is not supported, you can disable this message by disabling charts generation in the alert configuration."

    # Construct plain text version
    plain_body = f"""Hello,

The TrackMe entity:

- Incident ID: {incident_id}
- Detection Time: {detection_time}
- Alert Status: {alert_status}
- Environment: {environment_name}
- Tenant: {tenant_id}
- Alias: {alias}
- Object: {object}
- Category: {object_category}
- Object ID: {object_id}
- Priority: {priority}
- Drilldown Link: {drilldown_link}
- Object State: {object_state}
- Anomaly Reason(s): {", ".join(anomaly_reason)}

{transition_message}

{", ".join(messages)}

Regards,
TrackMe Alerting System

{footer}
"""
    part1 = MIMEText(plain_body, "plain")
    outer.attach(part1)

    # Construct HTML version if needed
    if format == "html":
        helper.log_info(f"Sending HTML email to {to_address}")

        icon = (
            "🚨"
            if object_state == "red"
            else (
                "🟠"
                if object_state == "orange"
                else "🔵" if object_state == "blue" else "✅"
            )
        )
        state_color = (
            "#d9534f"
            if object_state == "red"
            else (
                "#f0ad4e"
                if object_state == "orange"
                else "#007bff" if object_state == "blue" else "#5cb85c"
            )
        )
        alert_status_color = (
            "#d9534f"
            if alert_status == "opened"
            else (
                "#f0ad4e"
                if alert_status == "updated"
                else "#5cb85c" if alert_status == "closed" else "#333"
            )
        )

        html_body = f"""
        <html>
        <body style="font-family: Arial, sans-serif; font-size: 14px; color: #333;">
            <div style="border: 1px solid #ccc; padding: 16px; border-radius: 6px; background-color: #fefefe;">
                <h2 style="color: {state_color};">{icon} TrackMe Notification</h2>
                <ul style="list-style-type: none; padding-left: 0;">
                    <li><strong>Environment:</strong> {environment_name}</li>
                    <li><strong>Detection Time:</strong> {detection_time}</li>
                    <li><strong>Alert Status:</strong> <span style="color: {alert_status_color}; font-weight: bold;">{alert_status}</span></li>
                    <li><strong>Tenant:</strong> {tenant_id}</li>
                    <li><strong>Alias:</strong> {alias}</li>
                    <li><strong>Object:</strong> {object}</li>
                    <li><strong>Priority:</strong> {priority}</li>
                    <li><a href="{drilldown_link}">Drilldown Link</a></li>
                </ul>
                <ul style="list-style-type: none; padding-left: 0; margin-top: 16px;">
                    <li><strong>Status:</strong> <span style="color: {state_color}; font-weight: bold;">{object_state}</span></li>
                    <li><strong>Incident ID:</strong> {incident_id}</li>
                </ul>
                <hr style="margin: 20px 0;">
                <div style="padding: 12px; font-family: monospace;">
                    <p style="margin: 0 0 10px 0;"><strong>📋 Detailed Information:</strong></p>
                    <p style="margin-bottom: 10px;">{transition_message}</p>
                    <ul style="list-style-type: disc; padding-left: 20px;">
                        <li><strong>Alias:</strong> {alias}</li>
                        <li><strong>Object:</strong> {object}</li>
                        <li><strong>Object ID:</strong> {object_id}</li>
                        <li><strong>Category:</strong> {object_category}</li>
                        <li><strong>Anomaly Reason(s):</strong>
                            {"<ul style='margin-top: 5px;'>" + "".join(f"<li>{reason}</li>" for reason in anomaly_reason) + "</ul>" if isinstance(anomaly_reason, list) else f"<span>{anomaly_reason}</span>"}
                        </li>
                        <li><strong>Message(s):</strong>
                            {"<ul style='margin-top: 5px;'>" + "".join(f"<li>{message}</li>" for message in messages) + "</ul>" if isinstance(messages, list) else f"<span>{messages}</span>"}
                        </li>
                        <li><strong>Message source:</strong> {message_source}</li>
                        <li><strong>Message source ID:</strong> {message_source_id}</li>
                    </ul>
                </div>
        """

        # Build HTML for Incidents & Flipping section
        if incidents_flipping_charts:
            html_body += f"""
            <hr style=\"margin: 30px 0;\">
            <h3 style=\"margin-bottom: 10px;\">📈 Incidents, Flipping & State Charts ({timerange_charts})</h3>
            """

            for i in range(0, len(incidents_flipping_charts), 3):
                charts_row = incidents_flipping_charts[i : i + 3]
                html_body += """
                <table width=\"100%\" cellspacing=\"0\" cellpadding=\"0\" border=\"0\" style=\"table-layout: fixed; margin-bottom: 20px;\">
                    <tr>
                """
                for idx, chart in enumerate(charts_row):
                    # Use chart title as CID, ensuring it's a valid email CID format
                    cid = build_cid(chart["chart_title"])
                    html_body += f"""
                        <td width="33.33%" align="center" valign="top" style="padding: 5px;">
                            <img src="cid:{cid}" alt="{chart['chart_title']}"
                                style="width: 100%; max-width: 100%; height: auto; border: 1px solid #ccc; border-radius: 8px;" />
                        </td>
                    """
                for _ in range(3 - len(charts_row)):
                    html_body += """
                        <td width=\"33.33%\" style=\"padding: 5px;\">
                            <div style=\"width: 100%; height: 1px; visibility: hidden;\"></div>
                        </td>
                    """
                html_body += "</tr></table>"

        # Build HTML for Performance section
        if performance_charts:
            html_body += f"""
            <hr style=\"margin: 30px 0;\">
            <h3 style=\"margin-bottom: 10px;\">📈 Performance Charts ({timerange_charts})</h3>
            """

            for i in range(0, len(performance_charts), 3):
                charts_row = performance_charts[i : i + 3]
                html_body += """
                <table width=\"100%\" cellspacing=\"0\" cellpadding=\"0\" border=\"0\" style=\"table-layout: fixed; margin-bottom: 20px;\">
                    <tr>
                """
                for idx, chart in enumerate(charts_row):
                    # Use chart title as CID, ensuring it's a valid email CID format
                    cid = build_cid(chart["chart_title"])
                    html_body += f"""
                        <td width="33.33%" align="center" valign="top" style="padding: 5px;">
                            <img src="cid:{cid}" alt="{chart['chart_title']}"
                                style="width: 100%; max-width: 100%; height: auto; border: 1px solid #ccc; border-radius: 8px;" />
                        </td>
                    """
                for _ in range(3 - len(charts_row)):
                    html_body += """
                        <td width=\"33.33%\" style=\"padding: 5px;\">
                            <div style=\"width: 100%; height: 1px; visibility: hidden;\"></div>
                        </td>
                    """
                html_body += "</tr></table>"

        # Close the container and HTML
        html_body += f"""
                <p style="margin-top: 20px; font-size: 12px; color: #888;">{footer}</p>
            </div>
        </body>
        </html>
        """

        part1 = MIMEText(plain_body, "plain")
        part2 = MIMEText(html_body, "html")
        outer.attach(part1)
        outer.attach(part2)

        # Attach images as inline attachments
        all_charts = incidents_flipping_charts + performance_charts
        for chart in all_charts:
            chart_title = chart.get("chart_title", "")
            chart_b64 = chart.get("chart_b64", "")
            if not chart_title or not chart_b64:
                helper.log_error(
                    f"task=charts, Missing chart_title or chart_b64 in chart data"
                )
                continue

            # Create a safe CID from the chart title
            cid = build_cid(chart_title)

            image_type = detect_image_type(chart_b64)
            if image_type == "png":
                img_data = base64.b64decode(chart_b64)
                image = MIMEImage(img_data, _subtype="png")
                image.add_header("Content-ID", f"<{cid}>")
                image.add_header(
                    "Content-Disposition", "inline", filename=f"{chart_title}.png"
                )
                outer.attach(image)
            elif image_type == "svg":
                svg_data = base64.b64decode(chart_b64)
                svg_part = MIMEApplication(svg_data, _subtype="svg+xml")
                svg_part.add_header("Content-ID", f"<{cid}>")
                svg_part.add_header(
                    "Content-Disposition", "inline", filename=f"{chart_title}.svg"
                )
                outer.attach(svg_part)
            else:
                helper.log_error(
                    f"task=charts, Unknown image format {image_type} for chart: {chart_title}"
                )

    # Inner: related (HTML + images)
    related = MIMEMultipart("related")
    part2 = MIMEText(html_body, "html")
    related.attach(part2)
    for idx, chart_record in enumerate(all_charts):
        chart_title = chart_record.get("chart_title", f"chart{idx}")
        chart_b64 = chart_record.get("chart_b64")
        if not chart_b64:
            helper.log_error(
                f"task=charts, Missing chart_b64 data for chart: {chart_title}"
            )
            continue

        cid = build_cid(chart_title)
        image_type = detect_image_type(chart_b64)

        if image_type == "png":
            img_data = base64.b64decode(chart_b64)
            image = MIMEImage(img_data, _subtype="png")
            image.add_header("Content-ID", f"<{cid}>")
            image.add_header(
                "Content-Disposition", "inline", filename=f"{chart_title}.png"
            )
            related.attach(image)
        elif image_type == "svg":
            svg_data = base64.b64decode(chart_b64)
            svg_part = MIMEApplication(svg_data, _subtype="svg+xml")
            svg_part.add_header("Content-ID", f"<{cid}>")
            svg_part.add_header(
                "Content-Disposition", "inline", filename=f"{chart_title}.svg"
            )
            related.attach(svg_part)
        else:
            helper.log_error(
                f"task=charts, Unknown image format {image_type} for chart: {chart_title}"
            )
    outer.attach(related)

    # Extract server & port
    server_parts = config["email_server"].split(":")
    smtp_server = server_parts[0]
    smtp_port = int(server_parts[1])
    context = ssl.create_default_context()

    try:
        if config["email_security"] == "ssl":
            with smtplib.SMTP_SSL(smtp_server, smtp_port, context=context) as server:
                if config.get("email_username") and config.get("email_password"):
                    server.login(config["email_username"], config["email_password"])
                server.send_message(outer)
        else:
            with smtplib.SMTP(smtp_server, smtp_port) as server:
                if config["email_security"] == "tls":
                    server.starttls(context=context)
                if config.get("email_username") and config.get("email_password"):
                    server.login(config["email_username"], config["email_password"])
                server.send_message(outer)

        return message_id

    except Exception as e:
        raise RuntimeError(f"Failed to send email: {e}")


def ingest_stateful_alert_event(
    helper, session_key, server_uri, tenant_id, index, record
):
    """
    Function to ingest a stateful alert event into the TrackMe summary index

    Args:
        session_key: The session key
        server_uri: The server URI
        tenant_id: The tenant ID
        object_category: The object category
        object_value: The object value
    """

    # generate event
    try:
        trackme_state_event(
            session_key=session_key,
            splunkd_uri=server_uri,
            tenant_id=tenant_id,
            index=index,
            sourcetype="trackme:stateful_alerts",
            source=f"trackme_stateful_alerts_{tenant_id}",
            record=record,
        )
        helper.log_info(
            f'tenant_id="{tenant_id}", object_category="{record.get("object_category")}", object="{record.get("object")}", Stateful alert event created successfully, response="{json.dumps(record, indent=2)}"'
        )
        return True

    except Exception as e:
        helper.log_error(
            f'tenant_id="{tenant_id}", object_category="{record.get("object_category")}", object="{record.get("object")}", Stateful alert event creation failure, response="{json.dumps(record, indent=2)}", exception="{str(e)}"'
        )
        raise RuntimeError(f"Failed to ingest stateful alert event: {e}")


def get_keyid_from_main_kvstore(helper, collection, object):
    """
    Function to get the keyid value from the main KVstore collection and for a given object

    Args:
        collection: The collection to query
        object: The object to query

    Returns:
        The keyid value
    """

    # Define the KV query
    query_string = {
        "object": object,
    }

    # Initialize kvrecord to None to prevent "referenced before assignment" error
    kvrecord = None

    # Get the current record
    try:
        kvrecord = collection.data.query(query=json.dumps(query_string))[0]
        key = kvrecord.get("_key")

    except Exception as e:
        key = None

    if kvrecord:
        return kvrecord.get("_key")
    else:
        return None


def get_object_state_from_main_kvstore(helper, collection, object_id):
    """
    Function to get the object_state value from the main KVstore collection and for a given object_id

    Args:
        collection: The collection to query
        object_id: The object ID to query

    Returns:
        The object_state value, anomaly_reason, status_message and monitored_state
    """

    # Define the KV query
    query_string = {
        "_key": object_id,
    }

    # Initialize kvrecord to None to prevent "referenced before assignment" error
    kvrecord = None

    # Get the current record
    try:
        kvrecord = collection.data.query(query=json.dumps(query_string))[0]
        helper.log_debug(f"main collection kvrecord: {json.dumps(kvrecord, indent=2)}")
        key = kvrecord.get("_key")

    except Exception as e:
        key = None

    if kvrecord:

        object_state = kvrecord.get("object_state")
        anomaly_reason = kvrecord.get("anomaly_reason")
        status_message = kvrecord.get("status_message_json")
        monitored_state = kvrecord.get("monitored_state", "enabled")

        # normalize the anomaly_reason
        anomaly_reason = normalize_anomaly_reason(anomaly_reason)

        return object_state, anomaly_reason, status_message, monitored_state
    else:
        return "green", [], None, "enabled"


def get_stateful_record_for_message_id(collection, message_id):
    """
    Function to get the stateful record from the KVstore for a given message_id

    Args:
        collection: The collection to query
        message_id: The message ID to query

    Returns:
        The stateful record
    """

    # Define the KV query
    query_string = {
        "message_id": message_id,
    }

    # Initialize kvrecord to None to prevent "referenced before assignment" error
    kvrecord = None

    # Get the current record
    try:
        kvrecord = collection.data.query(query=json.dumps(query_string))[0]
        key = kvrecord.get("_key")

    except Exception as e:
        key = None

    return kvrecord


def get_stateful_records_for_object_id(helper, collection, object_id):
    """
    Function to get all stateful records for a given object_id

    Args:
        collection: The collection to query
        object_id: The object ID to query

    Returns:
        The last stateful record for the given object_id

    """

    # Define the KV query
    query_string = {
        "object_id": object_id,
    }

    # Get all records
    try:
        kvrecords = collection.data.query(query=json.dumps(query_string))
        helper.log_debug(f"kvrecords: {json.dumps(kvrecords, indent=2)}")
    except Exception as e:
        raise RuntimeError(f"Failed to get stateful records: {e}")
    
    filtered_records = []
    for kvrecord in kvrecords:
        if kvrecord.get("alert_status") != "closed":
            filtered_records.append(kvrecord)

    if filtered_records:
        # select the newer record based on mtime (epochtime)
        filtered_records.sort(key=lambda x: float(x.get("mtime", 0)), reverse=True)
        last_kvrecord = filtered_records[0]
        helper.log_debug(f"last_kvrecord: {json.dumps(last_kvrecord, indent=2)}")

        # Check if there are multiple records and handle duplicates
        if len(filtered_records) > 1:
            # Check if there are any non-closed records (excluding the newest one)
            non_closed_records = []
            for record in filtered_records[1:]:  # Skip the newest record
                alert_status = record.get("alert_status", "opened")
                if alert_status != "closed":
                    non_closed_records.append(record)
            
            # If we found non-closed records, close them
            if non_closed_records:
                helper.log_info(f"Found {len(non_closed_records)} duplicate non-closed records for object_id={object_id}, closing them")
                
                for record in non_closed_records:
                    try:
                        # Update the record to closed status
                        record["alert_status"] = "closed"
                        record["mtime"] = int(time.time())  # Update modification time
                        
                        # Update the record in the collection
                        collection.data.update(record.get("_key"), json.dumps(record))
                        helper.log_info(f"Successfully closed duplicate record with keyid={record.get('_key')} for object_id={object_id}")
                        
                    except Exception as e:
                        helper.log_error(f"Failed to close duplicate record with keyid={record.get('_key')} for object_id={object_id}: {e}")

        # return the last non closed record
        return last_kvrecord

    else:
        return None


def insert_stateful_record(collection, record):
    """
    Function to insert a new stateful record into the KVstore

    Args:
        collection: The collection to insert the record into
        record: The record to insert

    Returns:
        The inserted record

    """

    try:
        collection.data.insert(json.dumps(record))
    except Exception as e:
        raise RuntimeError(f"Failed to insert stateful record: {e}")


def update_stateful_record(collection, record):
    """
    Function to update an existing stateful record in the KVstore

    Args:
        collection: The collection to update the record in
        record: The record to update

    Returns:
        The updated record

    """

    try:
        collection.data.update(record.get("_key"), json.dumps(record))
    except Exception as e:
        raise RuntimeError(f"Failed to update stateful record: {e}")


def flx_get_metrics_catalog_for_object_id(
    helper, service, tenant_id, object_id, timerange_charts="24h"
):
    """
    Function to get the metrics catalog for a given object_id

    Args:
        helper: The helper object
        service: The service object
        tenant_id: The tenant ID
        object_id: The object ID

    Returns:
        The metrics catalog (list)

    """

    # Dynamically discover additional flx metrics
    metrics_catalog = []
    try:
        mcatalog_query = f"""| mcatalog values(metric_name) as metrics where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" metric_name="trackme.splk.flx.*" metric_name!="trackme.splk.flx.status" object_id="{object_id}"
        | mvexpand metrics | rex field=metrics "trackme\\.splk\\.flx\\.(?<metrics>.*)" | stats values(metrics) as metrics """
        search_kwargs = {
            "output_mode": "json",
            "count": 0,
            f"earliest_time": f"-{timerange_charts}@h",
            "latest_time": "now",
        }
        reader = run_splunk_search(service, mcatalog_query, search_kwargs, 24, 5)
        for item in reader:
            if isinstance(item, dict):
                metrics_result = item.get("metrics", [])
                break  # only one result is expected

        # ensure metrics_result is a list
        if metrics_result:
            if not isinstance(metrics_result, list):
                metrics_result = [metrics_result]

        # return the metrics_result
        return metrics_result

    except Exception as e:
        helper.log_error(f"Error retrieving flx metrics for object_id={object_id}: {e}")
        return None


def fqm_get_metrics_catalog_for_object_id(
    helper, service, tenant_id, object_id, timerange_charts="24h"
):
    """
    Function to get the metrics catalog for a given object_id

    Args:
        helper: The helper object
        service: The service object
        tenant_id: The tenant ID
        object_id: The object ID

    Returns:
        The metrics catalog (list)

    """

    # Dynamically discover additional fqm metrics
    metrics_catalog = []
    try:
        mcatalog_query = f"""| mcatalog values(metric_name) as metrics where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" metric_name="trackme.splk.fqm.*" metric_name!="trackme.splk.fqm.status" object_id="{object_id}"
        | mvexpand metrics | rex field=metrics "trackme\\.splk\\.fqm\\.(?<metrics>.*)" | stats values(metrics) as metrics """
        search_kwargs = {
            "output_mode": "json",
            "count": 0,
            f"earliest_time": f"-{timerange_charts}@h",
            "latest_time": "now",
        }
        reader = run_splunk_search(service, mcatalog_query, search_kwargs, 24, 5)
        for item in reader:
            if isinstance(item, dict):
                metrics_result = item.get("metrics", [])
                break  # only one result is expected

        # ensure metrics_result is a list
        if metrics_result:
            if not isinstance(metrics_result, list):
                metrics_result = [metrics_result]

        # return the metrics_result
        return metrics_result

    except Exception as e:
        helper.log_error(f"Error retrieving fqm metrics for object_id={object_id}: {e}")
        return None


def wlk_get_metrics_catalog_for_object_id(
    helper, service, tenant_id, object_id, timerange_charts="24h"
):
    """
    Function to get the metrics catalog for a given object_id

    Args:
        helper: The helper object
        service: The service object
        tenant_id: The tenant ID
        object_id: The object ID

    Returns:
        The metrics catalog (list)

    """

    # Dynamically discover additional wlk metrics
    metrics_catalog = []
    try:
        mcatalog_query = f"""| mcatalog values(metric_name) as metrics where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" metric_name="trackme.splk.wlk.*" object_id="{object_id}"
        | mvexpand metrics | rex field=metrics "trackme\\.splk\\.wlk\\.(?<metrics>.*)" | stats values(metrics) as metrics """
        search_kwargs = {
            "output_mode": "json",
            "count": 0,
            f"earliest_time": f"-{timerange_charts}@h",
            "latest_time": "now",
        }
        reader = run_splunk_search(service, mcatalog_query, search_kwargs, 24, 5)
        for item in reader:
            if isinstance(item, dict):
                metrics_result = item.get("metrics", [])
                break  # only one result is expected

        # ensure metrics_result is a list
        if metrics_result:
            if not isinstance(metrics_result, list):
                metrics_result = [metrics_result]

        # return the metrics_result
        return metrics_result

    except Exception as e:
        helper.log_error(f"Error retrieving wlk metrics for object_id={object_id}: {e}")
        return None


def get_mlmodels_from_kvstore(helper, service, tenant_id, component, object, object_id):
    """
    Function to get the mlmodels from the KVstore for a given object_id
    This function checks if 'ml_outliers_detection' is in the anomaly_reason list
    and if so, retrieves the list of models from the models_summary field

    Args:
        helper: The helper object
        service: The service object
        tenant_id: The tenant ID
        component: The component
        object: The object
        object_id: The object ID

    Returns:
        The list of model IDs from models_summary, or None if not found

    """

    # Initialize kvrecord to None to prevent "referenced before assignment" error
    kvrecord = None

    # Get the current record
    try:

        # connect to the main tenant KVstore collection
        collection_name = (
            f"kv_trackme_{component}_outliers_entity_data_tenant_{tenant_id}"
        )
        collection = service.kvstore[collection_name]

        # Define the KV query
        query_string = {
            "_key": object_id,
        }

        kvrecord = collection.data.query(query=json.dumps(query_string))[0]
        helper.log_debug(f"main collection kvrecord: {json.dumps(kvrecord, indent=2)}")
        key = kvrecord.get("_key")
        models_summary = kvrecord.get("models_summary", {})

        helper.log_debug(f"task=ml_models, models_summary type: {type(models_summary)}")
        helper.log_debug(
            f"task=ml_models, models_summary content: {json.dumps(models_summary, indent=2)}"
        )

        # Handle case where models_summary is a JSON string
        if isinstance(models_summary, str):
            try:
                models_summary = json.loads(models_summary)
                helper.log_debug(
                    f"task=ml_models, Successfully parsed models_summary from JSON string"
                )
            except json.JSONDecodeError as e:
                helper.log_debug(
                    f"task=ml_models, Failed to parse models_summary JSON string: {e}"
                )
                return None

        # if we have models_summary, extract the model IDs (keys of the dict)
        if models_summary and isinstance(models_summary, dict):
            model_ids = list(models_summary.keys())
            helper.log_debug(
                f"task=ml_models, Found model IDs in models_summary: {model_ids}"
            )
            helper.log_debug(
                f"task=ml_models, Number of models found: {len(model_ids)}"
            )
            return model_ids
        else:
            helper.log_debug(
                f"task=ml_models, No models_summary found or it's not a dict. Type: {type(models_summary)}"
            )
            return None

    except Exception as e:
        helper.log_error(
            f"Error retrieving models from KVstore for object_id={object_id}: {e}"
        )
        return None


def generate_chart_base64(
    helper,
    title,
    x_labels,
    series_dict,
    chart_type="line",
    theme="dark",
    object_state_colors=None,
    incidents_state_colors=None,
    force_integer_y_ticks=False,
):
    """
    Function to generate a chart base64

    Args:
        title: The title of the chart
        x_labels: The x labels of the chart
        series_dict: The series dictionary of the chart
        chart_type: The type of the chart

    Returns:
        The chart base64
    """

    if theme == "light":
        base_style = pygal.style.LightStyle
    else:
        base_style = pygal.style.DarkStyle(
            background="#232323",
            plot_background="#232323",
            foreground="#ffffff",
            foreground_strong="#ffffff",
            foreground_subtle="#aaaaaa",
            label_font_size=10,
            major_label_font_size=10,
            legend_font_size=12,
            title_font_size=14,
        )

    if object_state_colors:
        custom_style = pygal.style.Style(
            background=base_style.background,
            plot_background=base_style.plot_background,
            foreground=base_style.foreground,
            foreground_strong=base_style.foreground_strong,
            foreground_subtle=base_style.foreground_subtle,
            label_font_size=base_style.label_font_size,
            major_label_font_size=base_style.major_label_font_size,
            legend_font_size=base_style.legend_font_size,
            title_font_size=base_style.title_font_size,
            colors=object_state_colors,
        )
    elif incidents_state_colors:
        custom_style = pygal.style.Style(
            background=base_style.background,
            plot_background=base_style.plot_background,
            foreground=base_style.foreground,
            foreground_strong=base_style.foreground_strong,
            foreground_subtle=base_style.foreground_subtle,
            label_font_size=base_style.label_font_size,
            major_label_font_size=base_style.major_label_font_size,
            legend_font_size=base_style.legend_font_size,
            title_font_size=base_style.title_font_size,
            colors=incidents_state_colors,
        )
    else:
        custom_style = base_style

    # Detect if this chart should be scaled from 0 to 100 (for percentage metrics)
    percentage_scale = any(
        "pct" in k.lower() or "percent" in k.lower() for k in series_dict
    )

    # Define common chart kwargs
    chart_kwargs = {
        "height": 250,
        "width": 500,
        "show_legend": True,
        "explicit_size": True,
        "x_labels": [],
        "show_x_labels": False,
        "show_minor_x_labels": False,
        "legend_at_bottom": True,
        "style": custom_style,
        "show_dots": False,
    }

    if percentage_scale:
        chart_kwargs["range"] = (0, 100)

    # Choose chart type
    if chart_type == "bar":
        chart = pygal.Bar(**chart_kwargs)
    else:  # fallback to line
        chart = pygal.Line(**chart_kwargs)

    # Now apply the Y-axis fix
    if force_integer_y_ticks:
        try:
            all_y_values = [
                val
                for vals in series_dict.values()
                if isinstance(vals, list)
                for val in vals
                if isinstance(val, (int, float))
            ]
            max_y = max(all_y_values) if all_y_values else 0
            max_y_int = int(max_y) + 1  # Headroom
            chart.range = (0, max_y_int)
            chart.y_labels_major = list(range(0, max_y_int + 1))
            chart.show_minor_y_labels = False
            chart.show_minor_y_guides = False
        except Exception as e:
            helper.log_warn(f"Failed to force integer y ticks: {str(e)}")

    chart.title = title

    if object_state_colors:
        # Force insertion order to match expected color binding (for flipping events)
        fixed_order = ["Green", "Red", "Orange", "Blue"]
        for label in fixed_order:
            values = series_dict.get(label)
            if values is not None:
                chart.add(label, values)
    elif incidents_state_colors:
        # Force insertion order to match expected color binding (for incidents events)
        fixed_order = ["Opened incidents", "Updated incidents", "Closed incidents"]
        for label in fixed_order:
            values = series_dict.get(label)
            if values is not None:
                chart.add(label, values)
    else:
        for label, values in series_dict.items():
            chart.add(label, values)

    svg_data = chart.render()
    return base64.b64encode(svg_data).decode("utf-8")


def generate_chart_base64_png(helper, svg_data):
    """
    Converts SVG string to PNG and encodes as base64.
    """
    try:
        drawing = svg2rlg(BytesIO(svg_data.encode("utf-8")))
        if not drawing:
            helper.log_warn("task=charts, Failed to create reportlab drawing from SVG")
            return None
        png_bytes = renderPM.drawToString(drawing, fmt="PNG")
        if not png_bytes:
            helper.log_warn("task=charts, Failed to render PNG from SVG")
            return None
        png_base64 = base64.b64encode(png_bytes).decode("utf-8")
        return png_base64
    except Exception as e:
        helper.log_warn(f"task=charts, Error converting SVG to PNG: {str(e)}")
        import traceback

        helper.log_warn(f"task=charts, Traceback: {traceback.format_exc()}")
        return None


def generate_chart_base64_final(
    helper,
    title,
    x_labels,
    series_dict,
    chart_type="line",
    theme="dark",
    output_format="png",
    object_state_colors=None,
    incidents_state_colors=None,
    force_integer_y_ticks=False,
):
    """
    Generates a chart in either SVG or PNG base64-encoded format.
    """
    try:
        svg_b64 = generate_chart_base64(
            helper,
            title,
            x_labels,
            series_dict,
            chart_type,
            theme,
            object_state_colors=object_state_colors,
            incidents_state_colors=incidents_state_colors,
            force_integer_y_ticks=force_integer_y_ticks,
        )
        if not svg_b64:
            helper.log_warn(
                f"task=charts, Failed to generate SVG chart for title: {title}"
            )
            return None
        if output_format == "png":
            if not png_conversion_available:
                helper.log_warn(
                    f"task=charts, PNG conversion dependencies missing, falling back to SVG for title: {title}, exception: {png_import_error}"
                )
                return svg_b64
            try:
                svg_data = base64.b64decode(svg_b64).decode("utf-8")
                png_b64 = generate_chart_base64_png(helper, svg_data)
                if not png_b64:
                    helper.log_warn(
                        f"task=charts, Failed to convert SVG to PNG for title: {title}"
                    )
                    return svg_b64
                else:
                    helper.log_debug(
                        f"task=charts, Successfully converted SVG to PNG for title: {title}"
                    )
                return png_b64
            except Exception as e:
                helper.log_warn(
                    f"task=charts, Error converting SVG to PNG for title {title}: {str(e)}. Falling back to SVG."
                )
                return svg_b64
        return svg_b64
    except Exception as e:
        helper.log_warn(
            f"task=charts, Error generating chart for title {title}: {str(e)}"
        )
        return None


def build_incidents_chart(
    helper, search_results, theme_charts=None, timerange_charts="24h"
):
    """
    Function to build incidents chart

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The incidents chart
    """

    try:
        timestamps, opened, updated, closed = [], [], [], []
        for row in search_results:
            timestamps.append(row.get("_time"))
            opened.append(safe_float(row.get("opened")))
            updated.append(safe_float(row.get("updated")))
            closed.append(safe_float(row.get("closed")))

        # Hardcoded series colors: Opened incidents, Updated incidents, Closed incidents
        incidents_state_colors = ["#FFBABA", "#ffd394", "#b6edb6"]

        return generate_chart_base64_final(
            helper,
            f"Incidents Events - Last {timerange_charts}",
            timestamps,
            {
                "Opened incidents": opened,
                "Updated incidents": updated,
                "Closed incidents": closed,
            },
            chart_type="bar",
            theme=theme_charts,
            output_format="png",
            incidents_state_colors=incidents_state_colors,
            force_integer_y_ticks=True,
        )
    except Exception as e:
        helper.log_warn(
            f"task=charts, Failed to generate chart for incidents: {str(e)}"
        )
        return None


def build_flipping_chart(
    helper, search_results, theme_charts=None, timerange_charts="24h"
):
    """
    Function to build flipping events chart with specific per-state colors.

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The flipping chart
    """
    try:
        timestamps, green, red, orange, blue = [], [], [], [], []
        for row in search_results:
            timestamps.append(row.get("_time"))
            green.append(safe_float(row.get("green")))
            red.append(safe_float(row.get("red")))
            orange.append(safe_float(row.get("orange")))
            blue.append(safe_float(row.get("blue")))

        # Hardcoded series colors: Green, Red, Orange, Blue
        state_colors = ["#b6edb6", "#FFBABA", "#ffd394", "#cfebf9"]

        return generate_chart_base64_final(
            helper,
            f"Flipping Events - Last {timerange_charts}",
            timestamps,
            {
                "Green": green,
                "Red": red,
                "Orange": orange,
                "Blue": blue,
            },
            chart_type="bar",
            theme=theme_charts,
            output_format="png",
            object_state_colors=state_colors,
            force_integer_y_ticks=True,
        )
    except Exception as e:
        helper.log_warn(
            f"task=charts, Failed to generate chart for flipping events: {str(e)}"
        )
        return None


def build_state_chart(
    helper, search_results, theme_charts=None, timerange_charts="24h"
):
    """
    Function to build state events chart with specific per-state colors.

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The state chart
    """
    try:
        timestamps, green, red, orange, blue = [], [], [], [], []
        for row in search_results:
            timestamps.append(row.get("_time"))
            green.append(safe_float(row.get("green")))
            red.append(safe_float(row.get("red")))
            orange.append(safe_float(row.get("orange")))
            blue.append(safe_float(row.get("blue")))

        # Hardcoded series colors: Green, Red, Orange, Blue
        state_colors = ["#b6edb6", "#FFBABA", "#ffd394", "#cfebf9"]

        return generate_chart_base64_final(
            helper,
            f"State Events - Last {timerange_charts}",
            timestamps,
            {
                "Green": green,
                "Red": red,
                "Orange": orange,
                "Blue": blue,
            },
            chart_type="bar",
            theme=theme_charts,
            output_format="png",
            object_state_colors=state_colors,
            force_integer_y_ticks=True,
        )
    except Exception as e:
        helper.log_warn(
            f"task=charts, Failed to generate chart for state events: {str(e)}"
        )
        return None


def build_latency_chart(
    helper, search_results, theme_charts=None, timerange_charts="24h"
):
    """
    Function to build a latency chart

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The latency chart
    """

    try:
        timestamps, avg, latest, perc95 = [], [], [], []
        for row in search_results:
            timestamps.append(row.get("_time"))
            avg.append(safe_float(row.get("avg_latency_5m")))
            latest.append(safe_float(row.get("latest_latency_5m")))
            perc95.append(safe_float(row.get("perc95_latency_5m")))
        return generate_chart_base64_final(
            helper,
            f"Latency Metrics - Last {timerange_charts}",
            timestamps,
            {
                "Avg Latency (5m)": avg,
                "Latest Latency (5m)": latest,
                "95th Perc Latency (5m)": perc95,
            },
            theme=theme_charts,
            output_format="png",
        )
    except Exception as e:
        helper.log_warn(f"task=charts, Failed to generate chart for latency: {str(e)}")
        return None


def build_delay_chart(
    helper, search_results, theme_charts=None, timerange_charts="24h"
):
    """
    Function to build a delay chart

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The delay chart
    """

    try:
        timestamps, delays = [], []
        for row in search_results:
            timestamps.append(row.get("_time"))
            delays.append(safe_float(row.get("lag_event_sec")))
        return generate_chart_base64_final(
            helper,
            f"Delay Metrics - Last {timerange_charts}",
            timestamps,
            {
                "Event Lag (sec)": delays,
            },
            theme=theme_charts,
            output_format="png",
        )
    except Exception as e:
        helper.log_warn(f"task=charts, Failed to generate chart for delay: {str(e)}")
        return None


def build_volume_chart(
    helper, search_results, theme_charts=None, timerange_charts="24h"
):
    """
    Function to build a volume chart

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The volume chart
    """

    try:
        timestamps, avg, latest, perc95 = [], [], [], []
        for row in search_results:
            timestamps.append(row.get("_time"))
            avg.append(safe_float(row.get("avg_eventcount_5m")))
            latest.append(safe_float(row.get("latest_eventcount_5m")))
            perc95.append(safe_float(row.get("perc95_eventcount_5m")))
        return generate_chart_base64_final(
            helper,
            f"Volume Metrics - Last {timerange_charts}",
            timestamps,
            {
                "Avg Events (5m)": avg,
                "Latest Events (5m)": latest,
                "95th Perc Events (5m)": perc95,
            },
            theme=theme_charts,
            output_format="png",
        )
    except Exception as e:
        helper.log_warn(f"task=charts, Failed to generate chart for volume: {str(e)}")
        return None


def build_hosts_dcount_chart(
    helper, search_results, theme_charts=None, timerange_charts="24h"
):
    """
    Function to build a hosts dcount chart

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The hosts dcount chart
    """

    try:
        timestamps = []
        global_dcount_host = []
        avg_dcount_host_5m = []
        latest_dcount_host_5m = []
        perc95_dcount_host_5m = []

        for row in search_results:
            timestamps.append(row.get("_time"))
            global_dcount_host.append(safe_float(row.get("global_dcount_host")))
            avg_dcount_host_5m.append(safe_float(row.get("avg_dcount_host_5m")))
            latest_dcount_host_5m.append(safe_float(row.get("latest_dcount_host_5m")))
            perc95_dcount_host_5m.append(safe_float(row.get("perc95_dcount_host_5m")))

        return generate_chart_base64_final(
            helper,
            f"Hosts DCount Metrics - Last {timerange_charts}",
            timestamps,
            {
                "Global DCount (Host)": global_dcount_host,
                "Avg DCount (5m)": avg_dcount_host_5m,
                "Latest DCount (5m)": latest_dcount_host_5m,
                "95th Perc DCount (5m)": perc95_dcount_host_5m,
            },
            theme=theme_charts,
            output_format="png",
        )
    except Exception as e:
        helper.log_warn(
            f"task=charts, Failed to generate chart for hosts_dcount: {str(e)}"
        )
        return None


def build_flx_status_chart(
    helper, search_results, theme_charts=None, timerange_charts="24h"
):
    """
    Function to build a flx status chart

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The flx status chart
    """

    try:
        timestamps = []
        status = []

        for row in search_results:
            timestamps.append(row.get("_time"))
            status.append(safe_float(row.get("status")))

        return generate_chart_base64_final(
            helper,
            f"FLX Status - Last {timerange_charts}",
            timestamps,
            {
                "Status": status,
            },
            theme=theme_charts,
            output_format="png",
        )
    except Exception as e:
        helper.log_warn(
            f"task=charts, Failed to generate chart for flx_status: {str(e)}"
        )
        return None


def build_ml_outliers_chart(
    helper, search_results, theme_charts=None, model_id=None, timerange_charts="24h"
):
    """
    Function to build a ml outliers chart

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The ml outliers chart
    """

    try:
        timestamps = []
        kpi_metric_value = []
        LowerBound = []
        UpperBound = []
        for row in search_results:
            timestamps.append(row.get("_time"))
            kpi_metric_value.append(safe_float(row.get("kpi_metric_value")))
            LowerBound.append(safe_float(row.get("LowerBound")))
            UpperBound.append(safe_float(row.get("UpperBound")))
        return generate_chart_base64_final(
            helper,
            (
                f"Machine Learning Outliers ({model_id}) - Last {timerange_charts}"
                if model_id
                else f"Machine Learning Outliers - Last {timerange_charts}"
            ),
            timestamps,
            {
                "KPI Metric Value": kpi_metric_value,
                "Lower Bound": LowerBound,
                "Upper Bound": UpperBound,
            },
            theme=theme_charts,
            output_format="png",
        )
    except Exception as e:
        helper.log_warn(
            f"task=charts, Failed to generate chart for ml_outliers: {str(e)}"
        )
        return None


def build_data_sampling_anomaly_chart(
    helper, search_results, theme_charts=None, timerange_charts="24h"
):
    """
    Function to build a data sampling anomaly chart

    Args:
        helper: The helper object for logging
        search_results: The search results
        theme_charts: The email charts theme

    Returns:
        The data sampling anomaly chart
    """

    try:
        if not search_results:
            return None

        timestamps = []
        series = {}

        for row in search_results:
            ts = row.get("_time")
            if ts not in timestamps:
                timestamps.append(ts)

            for k, v in row.items():
                if k in ("_time", "_span"):
                    continue
                if k not in series:
                    series[k] = []
                series[k].append(safe_float(v))

        return generate_chart_base64_final(
            helper,
            f"Model Sampling Match % - Last {timerange_charts}",
            timestamps,
            series,
            chart_type="bar",
            theme=theme_charts,
            output_format="png",
        )

    except Exception as e:
        helper.log_warn(
            f"task=charts, Failed to generate chart for data_sampling: {str(e)}"
        )
        return None


def build_requested_charts(
    helper,
    chart_builders,
    chart_types,
    chart_results_map,
    theme_charts=None,
    timerange_charts="24h",
):
    """
    Function to build requested charts

    Args:
        helper: The helper object for logging
        chart_builders: The chart builders
        chart_types: The chart types
        chart_results_map: The chart results map
        theme_charts: The email charts theme

    Returns:
        A dict: {unique_chart_id: {'base64': ..., 'chart_title': ...}}
    """
    chart_dict = {}
    chart_counter = 0
    for chart_type, model_id in chart_types:
        chart_title = None
        if chart_type in ("flx_metric_group", "fqm_metric_group", "wlk_metric_group"):
            for metric_name in model_id:
                metric_data = []
                for row in chart_results_map.get(chart_type, []):
                    if metric_name in row:
                        metric_data.append(
                            {"_time": row.get("_time"), metric_name: row[metric_name]}
                        )
                if metric_data:
                    if chart_type == "flx_metric_group":
                        svg = build_flx_generic_metric_chart(
                            helper,
                            metric_data,
                            metric_name,
                            theme_charts,
                            timerange_charts,
                        )
                    elif chart_type == "fqm_metric_group":
                        svg = build_fqm_generic_metric_chart(
                            helper,
                            metric_data,
                            metric_name,
                            theme_charts,
                            timerange_charts,
                        )
                    elif chart_type == "wlk_metric_group":
                        svg = build_wlk_generic_metric_chart(
                            helper,
                            metric_data,
                            metric_name,
                            theme_charts,
                            timerange_charts,
                        )
                    # not expected to reach here, raise an error
                    else:
                        raise Exception(f"Invalid chart type: {chart_type}")

                    if svg:
                        chart_counter += 1
                        chart_id = f"chart{chart_counter}"
                        chart_dict[chart_id] = {
                            "base64": svg,
                            "chart_title": f"{chart_type}:{metric_name}",
                        }
        else:
            if chart_type in chart_builders:
                builder = chart_builders.get(chart_type)
                if not builder:
                    helper.log_warn(
                        f"[ChartBuilder] No builder found for chart type: {chart_type}"
                    )
                    continue

                # Fix: get the correct key (type:model_id) when model_id is used
                search_key = f"{chart_type}:{model_id}" if model_id else chart_type
                results = chart_results_map.get(search_key, [])

                if results:
                    # Pass model_id only for ml_outliers
                    if chart_type == "ml_outliers":
                        svg = builder(
                            helper,
                            results,
                            theme_charts,
                            model_id=model_id,
                            timerange_charts=timerange_charts,
                        )
                    else:
                        svg = builder(
                            helper,
                            results,
                            theme_charts,
                            timerange_charts=timerange_charts,
                        )

                    if svg:
                        chart_counter += 1
                        chart_id = f"chart{chart_counter}"
                        chart_title = (
                            f"{chart_type}:{model_id}" if model_id else chart_type
                        )
                        chart_dict[chart_id] = {
                            "base64": svg,
                            "chart_title": chart_title,
                        }

    return chart_dict


def build_flx_generic_metric_chart(
    helper, search_results, metric_name="", theme_charts=None, timerange_charts="24h"
):
    """
    Function to build a flx generic metric chart

    Args:
        helper: The helper object
        search_results: The search results
        metric_name: The metric name
        theme_charts: The email charts theme

    Returns:
        The flx generic metric chart
    """

    try:
        if not search_results:
            return None

        if search_results:
            first_row_keys = list(search_results[0].keys())
            helper.log_debug(
                f"[FLX Chart] First row keys for metric '{metric_name}': {first_row_keys}"
            )
        else:
            helper.log_debug(f"[FLX Chart] No results for metric '{metric_name}'")

        timestamps = []
        series = {}
        for row in search_results:
            ts = row.get("_time")
            if ts not in timestamps:
                timestamps.append(ts)
            for k, v in row.items():
                if k in ("_time", "_span"):
                    continue
                if k not in series:
                    series[k] = []
                series[k].append(safe_float(v))
        title = f"FLX Metric: {metric_name} - Last {timerange_charts}"
        # if metric_name contains count, return as bar chart
        if "count" in metric_name:
            return generate_chart_base64_final(
                helper, title, timestamps, series, theme=theme_charts, output_format="png", chart_type="bar"
            )
        else:
            return generate_chart_base64_final(
                helper, title, timestamps, series, theme=theme_charts, output_format="png"
            )
    except Exception as e:
        return None


def build_fqm_generic_metric_chart(
    helper, search_results, metric_name="", theme_charts=None, timerange_charts="24h"
):
    """
    Function to build a fqm generic metric chart

    Args:
        helper: The helper object
        search_results: The search results
        metric_name: The metric name
        theme_charts: The email charts theme

    Returns:
        The fqm generic metric chart
    """

    try:
        if not search_results:
            return None

        if search_results:
            first_row_keys = list(search_results[0].keys())
            helper.log_debug(
                f"[FQM Chart] First row keys for metric '{metric_name}': {first_row_keys}"
            )
        else:
            helper.log_debug(f"[FQM Chart] No results for metric '{metric_name}'")

        timestamps = []
        series = {}
        for row in search_results:
            ts = row.get("_time")
            if ts not in timestamps:
                timestamps.append(ts)
            for k, v in row.items():
                if k in ("_time", "_span"):
                    continue
                if k not in series:
                    series[k] = []
                series[k].append(safe_float(v))
        title = f"FQM Metric: {metric_name} - Last {timerange_charts}"
        # if metric_name contains count, return as bar chart
        if "count" in metric_name:
            return generate_chart_base64_final(
                helper, title, timestamps, series, theme=theme_charts, output_format="png", chart_type="bar"
            )
        else:
            return generate_chart_base64_final(
                helper, title, timestamps, series, theme=theme_charts, output_format="png"
            )
    except Exception as e:
        return None


def build_wlk_generic_metric_chart(
    helper, search_results, metric_name="", theme_charts=None, timerange_charts="24h"
):
    """
    Function to build a wlk generic metric chart

    Args:
        helper: The helper object
        search_results: The search results
        metric_name: The metric name
        theme_charts: The email charts theme

    Returns:
        The wlk generic metric chart
    """

    try:
        if not search_results:
            return None

        if search_results:
            first_row_keys = list(search_results[0].keys())
            helper.log_debug(
                f"[WLK Chart] First row keys for metric '{metric_name}': {first_row_keys}"
            )
        else:
            helper.log_debug(f"[WLK Chart] No results for metric '{metric_name}'")

        timestamps = []
        series = {}
        for row in search_results:
            ts = row.get("_time")
            if ts not in timestamps:
                timestamps.append(ts)
            for k, v in row.items():
                if k in ("_time", "_span"):
                    continue
                if k not in series:
                    series[k] = []
                series[k].append(safe_float(v))
        title = f"WLK Metric: {metric_name} - Last {timerange_charts}"
        # if metric_name contains count, return as bar chart
        if "count" in metric_name:
            return generate_chart_base64_final(
                helper, title, timestamps, series, theme=theme_charts, output_format="png", chart_type="bar"
            )
        else:
            return generate_chart_base64_final(
                helper, title, timestamps, series, theme=theme_charts, output_format="png"
            )
    except Exception as e:
        return None


def get_chart_search(
    chart_type=None,
    tenant_id=None,
    object_category=None,
    object=None,
    keyid=None,
    model_id=None,
    metric_list=None,
):
    """
    Function to get the chart search

    Args:
        chart_type: The chart type
        tenant_id: The tenant ID
        object_category: The object category
        object: The object
        keyid: The keyid
        model_id: The model ID
        metric_list: The metric list

    Returns:
        The chart search
    """

    # get component from object_category (splk-<component>)
    component = object_category.split("-")[1]

    if chart_type == "incidents_events":
        return remove_leading_spaces(
            f"""search (`trackme_idx({tenant_id})` sourcetype=trackme:stateful_alerts tenant_id="{tenant_id}" object_category={object_category} (object="{object}" OR object_id="{keyid}"))
            | timechart minspan=5m bins=1000 count by alert_status"""
        )

    if chart_type == "flipping_events":
        return remove_leading_spaces(
            f"""search `trackme_idx({tenant_id})` source="flip_state_change_tracking" tenant_id={tenant_id} object_category="{object_category}" (object="{object}" OR keyid="{keyid}") | eval separator = "-->" | dedup _time object object_category object_previous_state object_state | table _time object object_category object_previous_state separator object_state result
        | timechart limit=40 useother=t minspan=1h bins=1000 count by object_state"""
        )

    if chart_type == "state_events":
        return remove_leading_spaces(
            f"""| mstats latest(trackme.sla.object_state) as object_state where `trackme_metrics_idx("{tenant_id}")` object_category="{object_category}" tenant_id="{tenant_id}" object_id="{keyid}" by object_category, object span=1m | `trackme_sla_eval_current_state`
            | stats latest(current_state) as current_state by _time
            | timechart minspan=5m bins=1000 count by current_state"""
        )

    if chart_type == "latency":
        return remove_leading_spaces(
            f"""| mstats avg(trackme.splk.feeds.avg_latency_5m) as avg_latency_5m, avg(trackme.splk.feeds.latest_latency_5m) as latest_latency_5m, avg(trackme.splk.feeds.perc95_latency_5m) as perc95_latency_5m where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_category="{object_category}" object="{object}" by object span=5m
        | timechart minspan=5m bins=1000 avg(avg_latency_5m) as avg_latency_5m, avg(latest_latency_5m) as latest_latency_5m, avg(perc95_latency_5m) as perc95_latency_5m"""
        )

    elif chart_type == "delay":
        return remove_leading_spaces(
            f"""| mstats avg(trackme.splk.feeds.lag_event_sec) as lag_event_sec where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_category="{object_category}" object="{object}" by object span=5m
        | timechart minspan=5m bins=1000 avg(lag_event_sec) as lag_event_sec"""
        )

    elif chart_type == "volume":
        return remove_leading_spaces(
            f"""| mstats avg(trackme.splk.feeds.avg_eventcount_5m) as avg_eventcount_5m, avg(trackme.splk.feeds.latest_eventcount_5m) as latest_eventcount_5m, avg(trackme.splk.feeds.perc95_eventcount_5m) as perc95_eventcount_5m where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_category="{object_category}" object="{object}" by object span=5m
        | timechart minspan=5m bins=1000 avg(avg_eventcount_5m) as avg_eventcount_5m, avg(latest_eventcount_5m) as latest_eventcount_5m, avg(perc95_eventcount_5m) as perc95_eventcount_5m"""
        )

    elif chart_type == "hosts_dcount":
        return remove_leading_spaces(
            f"""| mstats avg(trackme.splk.feeds.global_dcount_host) as global_dcount_host, avg(trackme.splk.feeds.avg_dcount_host_5m) as avg_dcount_host_5m, avg(trackme.splk.feeds.latest_dcount_host_5m) as latest_dcount_host_5m, avg(trackme.splk.feeds.perc95_dcount_host_5m) as perc95_dcount_host_5m where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_category="{object_category}" object="{object}" by object span=5m
        | timechart minspan=5m bins=1000 avg(global_dcount_host) as global_dcount_host, avg(avg_dcount_host_5m) as avg_dcount_host_5m, avg(latest_dcount_host_5m) as latest_dcount_host_5m, avg(perc95_dcount_host_5m) as perc95_dcount_host_5m"""
        )

    elif chart_type == "data_sampling_anomaly":
        return remove_leading_spaces(
            f"""| mstats latest(trackme.splk_dsm.sampling.model_pct_match) as model_pct_match where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_category="{object_category}" object="{object}" by model_name span=5m
        | timechart avg(model_pct_match) as model_pct_match by model_name"""
        )

    elif chart_type == "flx_status":
        return remove_leading_spaces(
            f"""| mstats max(trackme.splk.flx.status) as status where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_id="{keyid}" span=5m
        | timechart minspan=5m bins=1000 max(status) as status"""
        )

    elif chart_type == "ml_outliers":
        return remove_leading_spaces(
            f"""| trackmesplkoutliersrender tenant_id="{tenant_id}" component="{component}" object="{object}" model_id="{model_id}"
        | table _time, kpi_metric_value, LowerBound, UpperBound"""
        )

    elif chart_type == "flx_metric_group" and metric_list:
        mstats_parts = [
            f"max(trackme.splk.flx.{metric}) as {metric}" for metric in metric_list
        ]
        mstats_fields = ", ".join(mstats_parts)
        return remove_leading_spaces(
            f"""| mstats {mstats_fields} where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_id="{keyid}" span=5m
        | timechart minspan=5m bins=1000 {', '.join([f'max({m}) as {m}' for m in metric_list])}"""
        )

    elif chart_type == "fqm_metric_group" and metric_list:
        mstats_parts = [
            f"max(trackme.splk.fqm.{metric}) as {metric}" for metric in metric_list
        ]
        mstats_fields = ", ".join(mstats_parts)
        return remove_leading_spaces(
            f"""| mstats {mstats_fields} where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_id="{keyid}" span=5m
        | timechart minspan=5m bins=1000 {', '.join([f'max({m}) as {m}' for m in metric_list])}"""
        )

    elif chart_type == "wlk_metric_group" and metric_list:
        mstats_parts = [
            f"max(trackme.splk.wlk.{metric}) as {metric}" for metric in metric_list
        ]
        mstats_fields = ", ".join(mstats_parts)
        return remove_leading_spaces(
            f"""| mstats {mstats_fields} where `trackme_metrics_idx({tenant_id})` tenant_id="{tenant_id}" object_id="{keyid}" span=5m
        | timechart minspan=5m bins=1000 {', '.join([f'max({m}) as {m}' for m in metric_list])}"""
        )

    return None


def execute_command(helper, service, commands_mode, command, event_data):
    """
    Function to execute a command based on the mode and context

    Args:
        helper: The helper object
        service: The service object
        commands_mode: The commands mode (streaming or generating)
        command: The command to execute
        event_data: The event data to use for field values

    Returns:
        None
    """

    if not command:
        return

    try:
        # Build the search based on mode
        if commands_mode == "streaming":
            # Convert event_data to JSON and encode it in base64
            event_data_json = json.dumps(event_data)
            event_data_b64 = base64.b64encode(event_data_json.encode("utf-8")).decode(
                "utf-8"
            )

            # set the search with base64 encoded data
            search = remove_leading_spaces(
                f"""\
                | trackmeyieldjson json_value_b64="{event_data_b64}"
                | trackmeexpandtokens
                {command}
                """
            )

        else:
            # For generating mode, use the command as is
            search = command

        # Execute the search
        search_kwargs = {
            "output_mode": "json",
            "count": 0,
            "earliest_time": "-5m",
            "latest_time": "now",
        }

        helper.log_info(f"Executing command mode={commands_mode}, search: {search}")
        commands_results_count = 0
        reader = run_splunk_search(service, search, search_kwargs, 24, 5)

        # Process results if needed
        for item in reader:
            if isinstance(item, dict):
                helper.log_debug(f"Command result: {json.dumps(item, indent=2)}")
                commands_results_count += 1

        if commands_results_count > 0:
            helper.log_info(
                f"Command mode={commands_mode} executed successfully, results_count={commands_results_count}"
            )
        else:
            helper.log_error(
                f"Command mode={commands_mode} was executed but no results were returned, command={command}"
            )

    except Exception as e:
        helper.log_error(
            f"Error executing command, mode={commands_mode}, exception={e}"
        )


def create_auto_ack(
    helper,
    server_uri,
    session_key,
    tenant_id,
    object_category,
    object,
    anomaly_reason,
    ack_period,
    ack_type,
):
    """
    Function to create an automated acknowledgment for a given object

    Args:
        helper: The helper object
        server_uri: The server URI
        session_key: The session key
        tenant_id: The tenant ID
        object_category: The object category
        object: The object
        anomaly_reason: The anomaly reason
        ack_period: The acknowledgment period in seconds
        ack_type: The acknowledgment type (sticky or unsticky)

    Returns:
        True if successful, False otherwise
    """

    # Build header
    header = {
        "Authorization": f"Splunk {session_key}",
        "Content-Type": "application/json",
    }

    # endpoint url
    endpoint_url = "/services/trackme/v2/ack/write/ack_manage"

    # normalize the anomaly_reason
    anomaly_reason = normalize_anomaly_reason(anomaly_reason)

    # build body
    body = {
        "tenant_id": tenant_id,
        "action": "enable",
        "object_category": object_category,
        "object_list": encode_unicode(object),
        "anomaly_reason": anomaly_reason,
        "ack_period": str(ack_period),
        "ack_type": str(ack_type),
        "ack_source": "auto_ack",
        "update_comment": "alert action auto-acknowledgement",
    }

    helper.log_info(
        f'task=auto-ack, tenant_id="{tenant_id}", object_category={object_category}, object={object}, Running TrackMe auto-acknowledgement, body="{json.dumps(body, indent=1)}"'
    )

    # target url
    target_url = f"{server_uri}/{endpoint_url}"

    # Post to Ack endpoint
    try:
        response = requests.post(
            target_url,
            headers=header,
            verify=False,
            data=json.dumps(body),
            timeout=600,
        )
        if response.status_code == 200:
            helper.log_info(
                f'task=auto-ack, tenant="{tenant_id}", object_category={object_category}, object={object}, response="{json.dumps(json.loads(response.text))}"'
            )
            return True
        else:
            helper.log_error(
                f'task=auto-ack, tenant_id="{tenant_id}", object_category={object_category}, object={object}, TrackMe auto acknowledgement action has failed, http_error_code="{response.status_code}", data="{json.dumps(body, indent=2)}", response="{response.text}"'
            )
            return False
    except Exception as e:
        helper.log_error(
            f'task=auto-ack, tenant_id="{tenant_id}", object_category={object_category}, object={object}, TrackMe auto acknowledgement alert action has failed, exception="{e}"'
        )
        return False


def process_event(helper, *args, **kwargs):
    """
    Function to process the event

    Args:
        helper: The helper object

    Returns:
        The return code
    """

    helper.log_info("task=init, Alert action trackme_stateful_alert started.")

    # Retrieve the session_key
    helper.log_debug("task=init, Get session_key.")
    session_key = helper.session_key

    # Retrieve the server_uri
    helper.log_debug("task=init, Get server_uri.")
    server_uri = helper.settings["server_uri"]

    # results_link
    helper.log_debug("task=init, Get results_link.")
    results_link = helper.settings["results_link"]

    # from results_link, extract server_name and server_port
    parsed_url = urlparse(results_link)

    # Get request info and set logging level
    reqinfo = trackme_reqinfo(session_key, server_uri)

    # Get service
    service = client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=session_key,
        timeout=600,
    )

    # get search_uri
    search_uri = helper.settings.get("search_uri")

    # Build header and target
    header = {
        "Authorization": f"Splunk {session_key}",
        "Content-Type": "application/json",
    }

    # Maintenance helpers
    def _normalize_tenants_scope(tenants_scope):
        try:
            if isinstance(tenants_scope, list):
                scope_list = tenants_scope
            elif isinstance(tenants_scope, str):
                ts = tenants_scope.strip()
                if ts == "" or ts == "*":
                    scope_list = ["*"]
                else:
                    scope_list = [s.strip() for s in ts.split(",") if s.strip()]
            else:
                scope_list = ["*"]
        except Exception:
            scope_list = ["*"]
        return scope_list

    def get_maintenance_status():
        try:
            endpoint = f"{server_uri}/services/trackme/v2/maintenance/check_global_maintenance_status"
            resp = requests.get(endpoint, headers=header, verify=False, timeout=60)
            resp.raise_for_status()
            body = resp.json()
            body["tenants_scope"] = _normalize_tenants_scope(body.get("tenants_scope", "*"))
            helper.log_info(f"maintenance_check response={json.dumps(body)}")
            return body
        except Exception as e:
            helper.log_error(f"maintenance_check failed exception=\"{str(e)}\"")
            return None

    def tenant_in_scope(tenant_id, tenants_scope):
        scope_list = _normalize_tenants_scope(tenants_scope)
        if "*" in scope_list:
            return True
        return tenant_id in scope_list

    # get the delivery_target
    delivery_target = helper.get_param("delivery_target")

    # get the priority_levels_emails
    priority_levels_emails = helper.get_param("priority_levels_emails")
    if priority_levels_emails == "*":
        priority_levels_emails = ["critical", "high", "medium", "low"]
    else:
        try:
            # turn into a list
            priority_levels_emails = priority_levels_emails.split(",")
            # remove empty strings
            priority_levels_emails = [
                level for level in priority_levels_emails if level
            ]
        except Exception as e:
            priority_levels_emails = []
    helper.log_debug(f"task=init, priority_levels_emails: {priority_levels_emails}")

    # get the priority_levels_commands
    priority_levels_commands = helper.get_param("priority_levels_commands")
    if priority_levels_commands == "*":
        priority_levels_commands = ["critical", "high", "medium", "low"]
    else:
        try:
            # turn into a list
            priority_levels_commands = priority_levels_commands.split(",")
            # remove empty strings
            priority_levels_commands = [
                level for level in priority_levels_commands if level
            ]
        except Exception as e:
            priority_levels_commands = []
    helper.log_debug(f"task=init, priority_levels_commands: {priority_levels_commands}")

    # if commands is enabled, get the mandatory parameters
    if "commands" in delivery_target:

        # perform a REST call to the search_uri
        try:
            response_alert_definition = requests.get(
                f"{server_uri}{search_uri}",
                headers=header,
                verify=False,
                timeout=600,
                params={"output_mode": "json", "count": 0},
            )
            response_alert_definition.raise_for_status()
            response_alert_definition_json = response_alert_definition.json()
            helper.log_debug(
                f"Search definition: {json.dumps(response_alert_definition_json, indent=2)}"
            )

        except Exception as e:
            helper.log_error(
                f"Error getting command definitions: exception={str(e)}, search_uri={search_uri}"
            )
            return 1

        # get commands
        try:
            commands_opened = response_alert_definition_json["entry"][0]["content"][
                "action.trackme_stateful_alert.param.commands_opened"
            ]
        except Exception as e:
            helper.log_error(
                f"Error getting command definitions: exception={str(e)}, search_uri={search_uri}"
            )
            return 1

        try:
            commands_updated = response_alert_definition_json["entry"][0]["content"][
                "action.trackme_stateful_alert.param.commands_updated"
            ]
        except Exception as e:
            helper.log_error(
                f"Error getting command definitions: exception={str(e)}, search_uri={search_uri}"
            )
            return 1

        try:
            commands_closed = response_alert_definition_json["entry"][0]["content"][
                "action.trackme_stateful_alert.param.commands_closed"
            ]
        except Exception as e:
            helper.log_error(
                f"Error getting command definitions: exception={str(e)}, search_uri={search_uri}"
            )
            return 1

        helper.log_debug(f"task=init, Command opened: {commands_opened}")
        helper.log_debug(f"task=init, Command closed: {commands_closed}")
        helper.log_debug(f"task=init, Command updated: {commands_updated}")

    # email account
    email_account = helper.get_param("email_account")

    # Option to include chart in emails (0 or 1)
    generate_charts = int(helper.get_param("generate_charts") or 0)
    if generate_charts == 1:
        charts_option_enabled = True
    else:
        charts_option_enabled = False

    # email charts theme
    theme_charts = helper.get_param("theme_charts")
    if not theme_charts or theme_charts == "" or theme_charts == "null":
        theme_charts = "dark"
    helper.log_debug(f"task=init, theme_charts: {theme_charts}")

    # email charts time window
    timerange_charts = helper.get_param("timerange_charts")
    if not timerange_charts or timerange_charts == "" or timerange_charts == "null":
        timerange_charts = "24h"
    helper.log_debug(f"task=init, timerange_charts: {timerange_charts}")

    # handle charts
    chart_builders = {
        "incidents_events": build_incidents_chart,
        "flipping_events": build_flipping_chart,
        "state_events": build_state_chart,
        "latency": build_latency_chart,
        "delay": build_delay_chart,
        "volume": build_volume_chart,
        "hosts_dcount": build_hosts_dcount_chart,
        "data_sampling_anomaly": build_data_sampling_anomaly_chart,
        "flx_status": build_flx_status_chart,
        "ml_outliers": build_ml_outliers_chart,
        "flx_metric": build_flx_generic_metric_chart,
        "fqm_metric": build_fqm_generic_metric_chart,
        "wlk_metric": build_wlk_generic_metric_chart,
    }

    # get the comma separated list of recipients
    recipients = helper.get_param("email_recipients")
    # turn into a list
    recipients = recipients.split(",")

    # for emails delivery, send update if Ack is active
    try:
        email_send_update_if_ack_active = int(
            helper.get_param("email_send_update_if_ack_active")
        )
    except Exception as e:
        email_send_update_if_ack_active = 1
    helper.log_debug(
        f"task=init, email_send_update_if_ack_active: {email_send_update_if_ack_active}"
    )

    # consider orange as in alerting state (0 for false, 1 for true)
    try:
        consider_orange_as_alerting_state = int(
            helper.get_param("orange_as_alerting_state")
        )
    except Exception as e:
        consider_orange_as_alerting_state = 0
    helper.log_debug(
        f"task=init, consider_orange_as_alerting_state: {consider_orange_as_alerting_state}"
    )

    # Determine alerting states based on configuration
    alerting_states = ["red"]
    if consider_orange_as_alerting_state:
        alerting_states.append("orange")

    # Get environment_name
    environment_name = helper.get_param("environment_name")
    if not environment_name or environment_name == "" or environment_name == "null":
        environment_name = "Splunk"

    # Get drilldown_root_uri
    drilldown_root_uri = helper.get_param("drilldown_root_uri")

    # get email_account_settings
    email_account_settings = {}

    # get commands_mode (common to all events to be processed)
    commands_mode = helper.get_param("commands_mode")

    endpoint = (
        f"{server_uri}/services/trackme/v2/configuration/get_emails_delivery_account"
    )
    try:
        response = requests.post(
            endpoint,
            headers=header,
            json={"account": email_account},
            verify=False,
            timeout=600,
        )
        response.raise_for_status()
        email_account_settings = response.json()
    except Exception as e:
        helper.log_error(
            f"task=init, Error getting email account settings: exception={str(e)}, email_account={email_account}"
        )
        return 1

    # get allowed_email_domains value (optional, comma separated list of allowed email domains)
    allowed_email_domain = email_account_settings.get("allowed_email_domains", None)

    #
    # Automated Acknowledgement
    #

    # auto ack enabled
    auto_ack_enabled = int(helper.get_param("auto_ack_enabled") or 0)
    helper.log_debug(f"task=get-params, auto_ack_enabled: {auto_ack_enabled}")

    # auto ack period
    auto_ack_period = int(helper.get_param("auto_ack_period") or 86400)
    helper.log_debug(f"task=get-params, auto_ack_period: {auto_ack_period}")

    # auto ack type
    auto_ack_type = helper.get_param("auto_ack_type")
    helper.log_debug(f"task=get-params, auto_ack_type: {auto_ack_type}")

    #
    # Main: process upstream events
    #

    # log system info
    helper.log_info(
        f'trackme_state_alert is starting, operating_system="{sys.platform}", python_version="{sys.version}"'
    )

    # Cache maintenance status for the run
    maintenance_info = get_maintenance_status()

    # Loop through the events
    for event in helper.get_events():

        # get tenant_id
        tenant_id = event["tenant_id"]

        # get the trackme_summary_idx for the tenant
        tenant_idx = trackme_idx_for_tenant(session_key, server_uri, tenant_id)
        tenant_trackme_summary_idx = tenant_idx["trackme_summary_idx"]

        # get alias, object, object_category, priority
        alias = event.get("alias", None)

        # get object, we may have to deal with problematic non ASCII chars
        object = decode_unicode(event["object"])

        object_category = event["object_category"]
        priority = event.get("priority", "medium")

        # get component from object_category (splk-<component>)
        component = object_category.split("-")[1]

        # check if the ack is active for the object
        ack_active = False

        # run a POST call against the /services/trackme/v2/ack/get_ack_for_object endpoint
        try:
            ack_response = requests.post(
                f"{server_uri}/services/trackme/v2/ack/get_ack_for_object",
                headers=header,
                data=json.dumps(
                    {
                        "tenant_id": tenant_id,
                        "object_category": object_category,
                        "object_list": object,
                    }
                ),
                verify=False,
                timeout=600,
            )
            ack_response.raise_for_status()
            ack_response_json = ack_response.json()
            helper.log_debug(f"Ack response: {json.dumps(ack_response_json, indent=2)}")

            if ack_response_json:
                ack_response_json = ack_response_json[0]
                ack_active_string = ack_response_json.get("ack_state", "inactive")
                ack_mtime = float(ack_response_json.get("ack_mtime", time.time()))
                if ack_active_string == "active":
                    ack_age = time.time() - ack_mtime
                    ack_active = True
                    helper.log_info(
                        f"tenant_id={tenant_id}, object_category={object_category}, object={object}, Ack is active, ack_age={ack_age:.2f} seconds, ack_response={json.dumps(ack_response_json, indent=2)}"
                    )
                else:
                    ack_active = False
                    helper.log_info(
                        f"tenant_id={tenant_id}, object_category={object_category}, object={object}, Ack is not active"
                    )

        except Exception as e:
            helper.log_error(
                f"Error getting Ack record: exception={str(e)}, tenant_id={tenant_id}, object_category={object_category}, object={object}"
            )

        # connect to the main tenant KVstore collection
        collection_main_name = f"kv_trackme_{component}_tenant_{tenant_id}"
        collection_main = service.kvstore[collection_main_name]

        # connect to the stateful KVstore collection
        collection_stateful_alerting_name = (
            f"kv_trackme_stateful_alerting_tenant_{tenant_id}"
        )
        collection_stateful_alerting = service.kvstore[
            collection_stateful_alerting_name
        ]

        # connect to the stateful KVstore collection for charts storing
        collection_stateful_alerting_charts_name = (
            f"kv_trackme_stateful_alerting_charts_tenant_{tenant_id}"
        )
        collection_stateful_alerting_charts = service.kvstore[
            collection_stateful_alerting_charts_name
        ]

        # get object_id, if not part of the upstream event, get it from the main KVstore
        # try both "keyid" and "key" fields as some events use different field names
        object_id = event.get("keyid", None)
        if not object_id:
            object_id = event.get("key", None)
        if not object_id:
            object_id = get_keyid_from_main_kvstore(helper, collection_main, object)
        if not object_id:
            helper.log_error(
                f"tenant_id={tenant_id}, object={object}, object_category={object_category}, object_id=None, No object_id found for object"
            )
            continue

        # Try to get the most up-to-date priority from the KVstore record
        kvstore_priority = get_priority_from_main_kvstore(
            helper, collection_main, object_id
        )
        if kvstore_priority:
            priority = kvstore_priority
            helper.log_info(
                f"Using priority '{priority}' from KVstore for object_id '{object_id}'"
            )
        else:
            # Fall back to the priority from the upstream event
            priority = event.get("priority", "medium")
            helper.log_info(
                f"Using priority '{priority}' from upstream event for object_id '{object_id}' (KVstore priority not available)"
            )

        # get the drilldown_link
        drilldown_link = generate_drilldown_link(
            parsed_url, drilldown_root_uri, tenant_id, object_category, object_id
        )

        # get the object_state from the main KVstore collection
        (
            collection_object_state,
            collection_anomaly_reason,
            collection_status_message_json,
            collection_monitored_state,
        ) = get_object_state_from_main_kvstore(helper, collection_main, object_id)

        # get object_state, collection_anomaly_reason and status_message_json from the event, if not present fallback to the collection values
        event_object_state = event.get("object_state", collection_object_state)
        event_anomaly_reason = event.get("anomaly_reason", collection_anomaly_reason)
        # normalize the anomaly_reason
        event_anomaly_reason = normalize_anomaly_reason(event_anomaly_reason)
        event_status_message_json = event.get(
            "status_message_json", collection_status_message_json
        )
        event_monitored_state = event.get("monitored_state", collection_monitored_state)

        # merge
        object_state = (
            event_object_state if event_object_state else collection_object_state
        )
        anomaly_reason = (
            event_anomaly_reason if event_anomaly_reason else collection_anomaly_reason
        )
        status_message_json = (
            event_status_message_json
            if event_status_message_json
            else collection_status_message_json
        )

        # for monitored_state, prefer the collection value instead of the event value
        monitored_state = (
            collection_monitored_state
            if collection_monitored_state
            else event_monitored_state
        )

        # cannot process if object_state is None
        if not object_state:
            helper.log_info(
                f'Skipping event: tenant_id={tenant_id}, object={object}, object_category={object_category}, object_id={object_id}, object_state=None, reason="failed to retrieve the object state from the main KVstore"'
            )
            continue
        else:
            helper.log_info(
                f"task=processing, Processing event: tenant_id={tenant_id}, object={object}, object_category={object_category}, object_id={object_id}, object_state={object_state}"
            )

        # get the value of anomaly_reason, and ensure it is a list
        anomaly_reason = event.get("anomaly_reason", None)

        # normalize the anomaly_reason
        anomaly_reason = normalize_anomaly_reason(anomaly_reason)

        # get the stateful record, if any
        stateful_record = get_stateful_records_for_object_id(
            helper, collection_stateful_alerting, object_id
        )
        if stateful_record:
            helper.log_info(f"task=processing, tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}, found an active stateful record for this entity, stateful_record={json.dumps(stateful_record, indent=2)}")
        else:
            helper.log_info(f"task=processing, tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}, no active stateful record found for this entity")

        # if we have a stateful record, get opened_anomaly_reason and updated_anomaly_reason
        if stateful_record:
            opened_anomaly_reason = stateful_record.get("opened_anomaly_reason", [])
            updated_anomaly_reason = stateful_record.get("updated_anomaly_reason", [])
            # normalize the opened_anomaly_reason and updated_anomaly_reason
            opened_anomaly_reason = normalize_anomaly_reason(opened_anomaly_reason)
            updated_anomaly_reason = normalize_anomaly_reason(updated_anomaly_reason)
        else:
            opened_anomaly_reason = []
            updated_anomaly_reason = []

        # merge the content of opened_anomaly_reason and updated_anomaly_reason to anomaly_reason
        anomaly_reason = list(
            set(anomaly_reason + opened_anomaly_reason + updated_anomaly_reason)
        )

        # Determine which chart types to include based on anomaly_reason

        chart_types = []
        model_id = None  # for outliers

        # for dsm/dhm, add latency, delay and volume as the basis
        if component in ("dsm", "dhm"):
            chart_types = [("latency", None), ("delay", None), ("volume", None)]

        elif component == "flx":
            chart_types = [("flx_status", None)]

            # dynamic metrics
            flx_metrics = flx_get_metrics_catalog_for_object_id(
                helper, service, tenant_id, object_id, timerange_charts=timerange_charts
            )
            helper.log_debug(f"task=metrics, flx_metrics: {flx_metrics}")
            if flx_metrics:
                chart_types.append(("flx_metric_group", flx_metrics))  # Pass whole list

        elif component == "fqm":

            # dynamic metrics
            fqm_metrics = fqm_get_metrics_catalog_for_object_id(
                helper, service, tenant_id, object_id, timerange_charts=timerange_charts
            )
            helper.log_debug(f"task=metrics, fqm_metrics: {fqm_metrics}")
            if fqm_metrics:
                chart_types.append(("fqm_metric_group", fqm_metrics))  # Pass whole list

        elif component == "wlk":

            # dynamic metrics
            wlk_metrics = wlk_get_metrics_catalog_for_object_id(
                helper, service, tenant_id, object_id, timerange_charts=timerange_charts
            )
            helper.log_debug(f"task=metrics, wlk_metrics: {wlk_metrics}")
            if wlk_metrics:
                chart_types.append(("wlk_metric_group", wlk_metrics))  # Pass whole list

        # relavant for dsm only:
        if component == "dsm":
            chart_types.append(("hosts_dcount", None))  # always add hosts_dcount

            if "data_sampling_anomaly" in anomaly_reason:
                chart_types.append(("data_sampling_anomaly", None))

        # ML Outliers:
        # - check if 'ml_outliers_detection' is in the anomaly_reason list
        # - if so, retrieve the models from the KVstore models_summary field
        if "ml_outliers_detection" in anomaly_reason:
            helper.log_info(
                f"task=ml_models, Found 'ml_outliers_detection' in anomaly_reason: {anomaly_reason}"
            )
            mlmodels_in_anomaly = get_mlmodels_from_kvstore(
                helper, service, tenant_id, component, object, object_id
            )
            helper.log_info(
                f"task=ml_models, Retrieved mlmodels_in_anomaly: {mlmodels_in_anomaly}"
            )
            if mlmodels_in_anomaly:
                helper.log_info(
                    f"task=ml_models, Adding {len(mlmodels_in_anomaly)} ML outlier charts"
                )
                for model_id in mlmodels_in_anomaly:
                    helper.log_info(
                        f"task=ml_models, Adding chart for model_id: {model_id}"
                    )
                    chart_types.append(("ml_outliers", model_id))
            else:
                helper.log_info(
                    f"task=ml_models, No models found in mlmodels_in_anomaly"
                )
        else:
            helper.log_info(
                f"task=ml_models, 'ml_outliers_detection' not found in anomaly_reason: {anomaly_reason}"
            )

        # always add incidents_events, flipping_events and state_events
        chart_types.append(("incidents_events", None))
        chart_types.append(("flipping_events", None))
        chart_types.append(("state_events", None))

        #
        # stateful record lifecycle:
        # - an alert is closed when the object_state is set to "green"
        # - an alert is opened when the object_state is set to "red"
        # - if the alert is closed and the object_state is not green, this is a new alert and therefore a new message_id
        #

        # alert_new_thread boolean
        alert_new_thread = False

        # Generate a new email-specific message ID
        current_email_id = generate_message_id()

        # if the monitored_state is "disabled", we need to skip the processing
        if monitored_state == "disabled":
            helper.log_info(
                f'Skipping event: tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}, monitored_state={monitored_state}, reason="monitored_state is disabled"'
            )
            continue

        # Maintenance gate: block opening/updating during maintenance; closures allowed
        maintenance_active = False
        try:
            if maintenance_info and maintenance_info.get("maintenance"):
                if tenant_in_scope(tenant_id, maintenance_info.get("tenants_scope", ["*"])):
                    maintenance_active = True
        except Exception:
            maintenance_active = False

        # if the stateful record is not found, we need to create a new thread
        if not stateful_record:
            if maintenance_active and object_state in alerting_states:
                helper.log_info(
                    f"maintenance active, skipping new incident creation: tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}"
                )
                continue
            # Check if ack is active and older than 5 minutes - if yes, skip creating new incident
            if ack_active and ack_age > 300:  # 300 seconds = 5 minutes
                helper.log_info(
                    f'Skipping new incident creation: tenant_id={tenant_id}, object={object}, object_id={object_id}, ack_active=True, ack_age={ack_age:.2f} seconds, reason="Ack is active and older than 5 minutes"'
                )
                continue
            alert_new_thread = True
            incident_id = (
                generate_message_id()
            )  # Generate a stable ID for the lifecycle

            # Create auto ack if enabled and this is a new incident
            if auto_ack_enabled and object_state in alerting_states:
                helper.log_info(
                    f"task=auto-ack, creating auto ack for new incident: tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}"
                )
                create_auto_ack(
                    helper=helper,
                    server_uri=server_uri,
                    session_key=session_key,
                    tenant_id=tenant_id,
                    object_category=object_category,
                    object=object,
                    anomaly_reason=anomaly_reason,
                    ack_period=auto_ack_period,
                    ack_type=auto_ack_type,
                )
        else:
            incident_id = stateful_record.get("incident_id")  # Reuse it if updating

        # check if the event should be skipped
        if stateful_record:

            # get the mtime from stateful_record, of it exists
            mtime = float(stateful_record.get("mtime"))

            # get the event time
            event_time = float(event.get("_time", time.time()))

            # if we have a stateful record, and the event time is not newer than the mtime, we can skip the processing
            if event_time <= mtime:
                helper.log_info(
                    f"Skipping event: tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}, event_time={time.strftime('%c', time.localtime(event_time))}, mtime={time.strftime('%c', time.localtime(mtime))}, reason=\"event is not newer than stateful record last update\""
                )
                continue

            # Skip if less than 60 minutes have passed since the last update, but only for non-closure events
            time_diff_minutes = (event_time - mtime) / 60
            if maintenance_active and object_state in alerting_states:
                helper.log_info(
                    f"maintenance active, skipping incident update: tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}"
                )
                continue
            if time_diff_minutes < 60 and object_state in alerting_states:
                helper.log_info(
                    f"Skipping event: tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}, event_time={time.strftime('%c', time.localtime(event_time))}, mtime={time.strftime('%c', time.localtime(mtime))}, time_diff_minutes={time_diff_minutes:.2f}, reason=\"less than 60 minutes since last update\""
                )
                continue

        # log
        helper.log_info(
            f"task=processing, Processing alert: tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}, alert_new_thread={alert_new_thread}"
        )

        # message_id, if this is a new thread, we need to generate a new message_id
        message_id = None

        if alert_new_thread:
            message_id = current_email_id
            in_reply_to = None
            reference_chain = []
        else:
            message_id = stateful_record.get("message_id")
            in_reply_to = message_id
            reference_chain = stateful_record.get("reference_chain", [message_id])
            if not isinstance(reference_chain, list):
                reference_chain = [message_id]
            reference_chain.append(current_email_id)

        # status_message_json
        try:
            status_message_content = json.loads(status_message_json).get(
                "status_message"
            )
        except:
            status_message_content = []

        # messages (list)
        messages = []

        # message_source and id
        message_source = event.get("sourcetype")
        message_source_id = event.get("event_id")

        # trackme:state provides the message as status_message
        if message_source == "trackme:state":
            messages.append(event["status_message"])

        # trackme:flip provides the message as result
        elif message_source == "trackme:flip":
            messages.append(event["result"])

        # trackme:sla_breaches provides the message as sla_message
        elif message_source == "trackme:sla_breaches":
            messages.append(event["sla_message"])

        # for item in status_message_content, if not in messages, add it
        for item in status_message_content:
            if item not in messages:
                messages.append(item)

        # do no process if no stateful_record and object_state is green/blue
        if not stateful_record and object_state not in alerting_states:
            helper.log_info(
                f"Skipping processing: tenant_id={tenant_id}, object={object}, object_id={object_id}, object_state={object_state}, event_id={event.get('event_id')}, reason=\"no stateful record and non-alerting state\""
            )
            continue

        # set delivery_type as a list depending on the delivery_target
        delivery_type = []
        if delivery_target in (
            "emails_and_ingest",
            "emails_only",
            "emails_commands_and_ingest",
            "commands_and_emails",
        ):
            delivery_type.append("email")
        if delivery_target in (
            "ingest_only",
            "emails_and_ingest",
            "emails_commands_and_ingest",
            "commands_and_ingest",
        ):
            delivery_type.append("ingest")
        if delivery_target in (
            "emails_commands_and_ingest",
            "commands_and_ingest",
            "commands_and_emails",
            "commands_only",
        ):
            delivery_type.append("commands")

        # remove email from delivery type depending on the priority_levels_emails
        if priority_levels_emails and priority not in priority_levels_emails:
            delivery_type.remove("email")
        if priority_levels_commands and priority not in priority_levels_commands:
            delivery_type.remove("commands")

        # init detection_time (human readable format %c %Z)
        detection_time = time.strftime(
            "%c %Z", time.localtime(float(event.get("_time", time.time())))
        )

        # init new_record
        new_record = {}

        # set alert_status
        alert_status = None

        # if necessary, insert the record
        if not stateful_record:

            alert_status = "opened"
            record_key = hashlib.sha256(
                f"{object_id}{time.time()}".encode()
            ).hexdigest()
            new_record = {
                "_key": record_key,
                "tenant_id": tenant_id,
                "object": object,
                "object_category": object_category,
                "object_id": object_id,
                "object_state": object_state,
                "incident_id": incident_id,
                "message_id": current_email_id,
                "messages": messages,
                "reference_chain": reference_chain,
                "delivery_type": delivery_type,
                "alert_status": alert_status,
                "ctime": float(event.get("_time", time.time())),
                "mtime": float(event.get("_time", time.time())),
                "alias": alias,
                "opened_anomaly_reason": anomaly_reason,
            }
            insert_stateful_record(
                collection_stateful_alerting,
                new_record,
            )

        else:

            # handle message
            current_messages = stateful_record.get("messages", [])
            for item in messages:
                if item not in current_messages:
                    current_messages.append(item)

            # if object_state is not green, this is an update
            alert_status = "updated"
            if object_state in alerting_states:
                new_record = {
                    "_key": stateful_record.get("_key"),
                    "tenant_id": tenant_id,
                    "object": object,
                    "object_category": object_category,
                    "object_id": object_id,
                    "object_state": object_state,
                    "incident_id": incident_id,
                    "message_id": current_email_id,
                    "messages": current_messages,
                    "reference_chain": reference_chain,
                    "delivery_type": delivery_type,
                    "alert_status": alert_status,
                    "ctime": float(stateful_record.get("ctime")),
                    "mtime": float(event.get("_time", time.time())),
                    "alias": alias,
                    "opened_anomaly_reason": stateful_record.get(
                        "opened_anomaly_reason", []
                    ),
                    "updated_anomaly_reason": anomaly_reason,
                }
                update_stateful_record(
                    collection_stateful_alerting,
                    new_record,
                )

            else:
                alert_status = "closed"
                new_record = {
                    "_key": stateful_record.get("_key"),
                    "tenant_id": tenant_id,
                    "object": object,
                    "object_category": object_category,
                    "object_id": object_id,
                    "object_state": object_state,
                    "incident_id": incident_id,
                    "message_id": current_email_id,
                    "messages": current_messages,
                    "reference_chain": reference_chain,
                    "delivery_type": delivery_type,
                    "alert_status": alert_status,
                    "ctime": float(stateful_record.get("ctime")),
                    "mtime": float(event.get("_time", time.time())),
                    "alias": alias,
                    "opened_anomaly_reason": stateful_record.get(
                        "opened_anomaly_reason", []
                    ),
                    "updated_anomaly_reason": stateful_record.get(
                        "updated_anomaly_reason", []
                    ),
                }
                update_stateful_record(
                    collection_stateful_alerting,
                    new_record,
                )

        #
        # Ingest stateful alert event into the TrackMe summary index
        #

        # prepare the new_record

        # calculate the event_id as the sha-256 sum of the record
        event_id = hashlib.sha256(json.dumps(new_record).encode()).hexdigest()

        # add the event_id to the record
        new_record["event_id"] = event_id

        # add detection_time to the record
        new_record["detection_time"] = detection_time

        # remove the _key from the record
        new_record.pop("_key", None)

        # include drilldown_link in the record
        new_record["drilldown_link"] = drilldown_link

        # include priority in the record
        new_record["priority"] = priority

        # include message_source and message_source_id in the record
        new_record["message_source"] = message_source
        new_record["message_source_id"] = message_source_id

        if delivery_target in (
            "emails_commands_and_ingest",
            "emails_and_ingest",
            "ingest_only",
            "commands_and_ingest",
        ):

            try:
                ingest_stateful_alert_event(
                    helper,
                    session_key,
                    server_uri,
                    tenant_id,
                    tenant_trackme_summary_idx,
                    new_record,
                )

            except Exception as e:
                helper.log_error(
                    f"task=ingest, Error ingesting stateful alert event: tenant_id={tenant_id}, object={object}, object_id={object_id}, error={e}"
                )

        #
        # Charts generation
        #

        # Only generate charts if email is in delivery type
        if "email" in delivery_type:        

            chart_ids = []
            chart_descriptions = []

            if not pygal_available and python_compatible:
                helper.log_warn(
                    "Pygal is not available and failed to load, likely due to insufficient Python version (required 3.8.x or higher), charts will not be generated"
                )
                generate_charts = False
                chart_base64_dict = []

            elif not python_compatible:
                helper.log_info(
                    f"Python version is less than 3.9, charts generation is not supported, running Python {python_version}"
                )
                generate_charts = False
                chart_base64_dict = []

            if generate_charts:
                chart_results_map = {}
                search_kwargs = {
                    "output_mode": "json",
                    "count": 0,
                    f"earliest_time": f"-{timerange_charts}@h",
                    "latest_time": "now",
                }

                for chart_type, model_or_metrics in chart_types:
                    metric_list = (
                        model_or_metrics if isinstance(model_or_metrics, list) else None
                    )
                    model_id = (
                        model_or_metrics if isinstance(model_or_metrics, str) else None
                    )

                    if chart_type == "incidents_events":
                        chart_descriptions.append("Incidents Events")
                    elif chart_type == "flipping_events":
                        chart_descriptions.append("Flipping Events")
                    elif chart_type == "state_events":
                        chart_descriptions.append("State Events")
                    elif chart_type in (
                        "flx_metric_group",
                        "fqm_metric_group",
                        "wlk_metric_group",
                    ):
                        for metric_name in model_or_metrics:
                            if "flx" in chart_type:
                                chart_descriptions.append(f"FLX Metric: {metric_name}")
                            elif "fqm" in chart_type:
                                chart_descriptions.append(f"FQM Metric: {metric_name}")
                            elif "wlk" in chart_type:
                                chart_descriptions.append(f"WLK Metric: {metric_name}")
                            # not expected to reach here, raise an error
                            else:
                                raise Exception(f"Invalid chart type: {chart_type}")
                    elif chart_type == "ml_outliers":
                        chart_descriptions.append(f"ML Outlier: {model_or_metrics}")
                    elif chart_type == "hosts_dcount":
                        chart_descriptions.append("Hosts DCount")
                    elif chart_type == "latency":
                        chart_descriptions.append("Latency Metrics")
                    elif chart_type == "delay":
                        chart_descriptions.append("Delay Metrics")
                    elif chart_type == "volume":
                        chart_descriptions.append("Volume Metrics")
                    elif chart_type == "data_sampling_anomaly":
                        chart_descriptions.append("Sampling Anomaly %")
                    elif chart_type == "flx_status":
                        chart_descriptions.append("FLX Status")
                    else:
                        chart_descriptions.append(chart_type.title())  # Fallback

                    search_query = get_chart_search(
                        chart_type=chart_type,
                        tenant_id=tenant_id,
                        object_category=object_category,
                        object=object,
                        keyid=object_id,
                        model_id=model_id,
                        metric_list=metric_list,
                    )

                    if not search_query:
                        helper.log_info(
                            f'Skipping chart type: {chart_type}, reason="no search query returned."'
                        )
                        continue

                    key = (
                        chart_type
                        if metric_list
                        else f"{chart_type}:{model_id}" if model_id else chart_type
                    )

                    helper.log_info(
                        f"task=charts, Running search: {search_query}, timerange_charts: {timerange_charts}"
                    )
                    try:
                        reader = run_splunk_search(
                            service, search_query, search_kwargs, 24, 5
                        )
                        chart_results_map[key] = [
                            item for item in reader if isinstance(item, dict)
                        ]
                    except Exception as e:
                        helper.log_error(
                            f"task=charts, Search failed for chart {key}, error: {e}"
                        )

                # Build charts from the gathered results
                chart_base64_dict = build_requested_charts(
                    helper,
                    chart_builders,
                    chart_types,
                    chart_results_map,
                    theme_charts=theme_charts,
                    timerange_charts=timerange_charts,
                )

                # Store charts to KVstore
                if chart_base64_dict:
                    try:

                        idx = 0
                        for chart_id, chart in chart_base64_dict.items():
                            idx += 1
                            chart_b64 = chart["base64"]
                            chart_title = chart["chart_title"]
                            chart_id = hashlib.sha256(
                                f"{incident_id}-{idx}-{time.time()}".encode()
                            ).hexdigest()
                            chart_record = {
                                "_key": chart_id,
                                "object_category": object_category,
                                "object": object,
                                "object_id": object_id,
                                "incident_id": incident_id,
                                "message_id": current_email_id,
                                "ctime": float(time.time()),
                                "chart_id": chart_id,
                                "chart_description": chart_title,
                                "chart_base64": chart_b64,
                            }
                            collection_stateful_alerting_charts.data.insert(
                                json.dumps(chart_record)
                            )
                            chart_ids.append(chart_id)
                        helper.log_info(
                            f"task=charts, {len(chart_base64_dict)} charts stored in KVstore for incident_id={incident_id} with chart_ids={chart_ids}"
                        )
                    except Exception as e:
                        helper.log_error(
                            f"task=charts, Failed to store charts for incident_id={incident_id}: {e}"
                        )

        else:
            # Email not in delivery type, skip chart generation
            helper.log_info(f"task=charts, Email not in delivery type, skipping chart generation, tenant_id={tenant_id}, object={object}, object_id={object_id}")
            chart_ids = []
            chart_descriptions = []            
            chart_base64_dict = []

        #
        # Emails delivery
        #

        if "email" in delivery_type:

            # Loop through all provided recipients
            for recipient in recipients:
                recipient = recipient.strip()
                if not recipient:
                    continue

                # Skip sending email for updated alerts if email_send_update_if_ack_active is 0 and ack_active is True
                # But always send emails for new alerts and incident closures
                if (
                    not alert_new_thread
                    and not email_send_update_if_ack_active
                    and ack_active
                    and object_state in alerting_states
                ):  # Only skip if still in alerting state
                    helper.log_info(
                        f'Skipping email for updated alert: tenant_id={tenant_id}, object={object}, object_id={object_id}, recipient={recipient}, email_send_update_if_ack_active={email_send_update_if_ack_active}, ack_active={ack_active}, reason="email_send_update_if_ack_active is {email_send_update_if_ack_active} and ack_active is {ack_active}"'
                    )
                    continue

                # Build the subject prefix
                subject_prefix = f"[TrackMe Incident-ID: {incident_id}]"

                # Compose subject based on state
                if alert_new_thread:
                    subject_suffix = f" Entity {object} is now in alert"
                elif object_state not in alerting_states:
                    subject_suffix = (
                        f" Entity {object} has received an incident closure update"
                    )
                else:
                    subject_suffix = f" Entity {object} has received an update"

                # Final subject
                subject = f"{subject_prefix}{subject_suffix}"

                # check if the recipient is allowed
                if allowed_email_domain:
                    if not any(
                        domain in recipient
                        for domain in allowed_email_domain.split(",")
                    ):
                        helper.log_warn(
                            f'Skipping recipient: tenant_id={tenant_id}, object={object}, object_id={object_id}, recipient={recipient}, reason="not in allowed email domains"'
                        )
                        continue

                # get email format
                email_format = email_account_settings.get("email_format", "html")

                # transition message
                if object_state in alerting_states:
                    if alert_new_thread:
                        transition_message = (
                            "🚨 The entity has transitioned to an alerting state:"
                            if email_format == "html"
                            else "The entity has transitioned to an alerting state:"
                        )
                    else:
                        transition_message = (
                            "ℹ️ The entity has received an update:"
                            if email_format == "html"
                            else "The entity has received an update:"
                        )
                else:
                    transition_message = (
                        "✅ The entity has received an incident closure update and is now in a non-alerting state:"
                        if email_format == "html"
                        else "The entity has received an incident closure update and is now in a non-alerting state:"
                    )

                #
                # Generate and send email
                #

                helper.log_debug(
                    f"we have {len(chart_base64_dict)} charts to inject into email body"
                )
                helper.log_debug(f"value of chart_base64_dict: {chart_base64_dict}")

                try:
                    msg_id = send_trackme_email(
                        helper=helper,
                        config=email_account_settings,
                        message_id=current_email_id,
                        to_address=recipient,
                        subject=subject,
                        in_reply_to=in_reply_to,
                        reference_chain=reference_chain,
                        format=email_format,
                        tenant_id=tenant_id,
                        alias=alias,
                        object=object,
                        object_state=object_state,
                        object_category=object_category,
                        object_id=object_id,
                        priority=priority,
                        anomaly_reason=anomaly_reason,
                        messages=messages,
                        message_source=message_source,
                        message_source_id=message_source_id,
                        transition_message=transition_message,
                        environment_name=environment_name,
                        drilldown_link=drilldown_link,
                        incident_id=incident_id,
                        detection_time=detection_time,
                        chart_base64_dict=chart_base64_dict,
                        alert_status=alert_status,
                        timerange_charts=timerange_charts,
                        python_compatible=python_compatible,
                        charts_option_enabled=charts_option_enabled,
                    )
                    helper.log_info(
                        f"task=email, Email sent: tenant_id={tenant_id}, object={object}, object_id={object_id}, recipient={recipient}, message_id={msg_id}"
                    )

                except Exception as e:
                    helper.log_error(
                        f"task=email, Error sending email: tenant_id={tenant_id}, object={object}, object_id={object_id}, recipient={recipient}, error={e}"
                    )

        # Execute commands if enabled
        if "commands" in delivery_type:

            helper.log_info(
                f"task=commands, new_record: {json.dumps(new_record, indent=4)}"
            )

            # add to new_record any field in event that is not already in new_record
            for key, value in event.items():
                if key not in new_record:
                    new_record[key] = value

            # Execute appropriate command based on alert status
            if alert_new_thread:
                execute_command(
                    helper, service, commands_mode, commands_opened, new_record
                )
            elif object_state in alerting_states:
                execute_command(
                    helper, service, commands_mode, commands_updated, new_record
                )
            else:
                execute_command(
                    helper, service, commands_mode, commands_closed, new_record
                )

    #
    # End of processing
    #

    return 0
