# encoding = utf-8

import os
import sys
import json
import time
import hashlib
from datetime import datetime
import splunklib.client as client
import urllib3
from urllib.parse import urlparse, urlencode, urlunparse, parse_qs
import requests

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# set splunkhome
splunkhome = os.environ["SPLUNK_HOME"]

# append libs
sys.path.append(os.path.join(splunkhome, "etc", "apps", "trackme", "lib"))

from trackme_libs import trackme_reqinfo, trackme_idx_for_tenant, trackme_state_event

from splunktaucclib.cim_actions import parse_mv


def get_splunk_service(helper, session_key, server_uri):
    reqinfo = trackme_reqinfo(session_key, server_uri)
    return client.connect(
        owner="nobody",
        app="trackme",
        port=reqinfo["server_rest_port"],
        token=session_key,
        timeout=600,
    )


def process_event(helper, *args, **kwargs):
    # Helper functions

    # Adjust query_params function remains unchanged
    def adjust_query_params(url, query_params):
        parsed_url = urlparse(url)
        query_dict = parse_qs(parsed_url.query)
        if "tenant_id" in query_dict:
            query_params.pop("tenant_id", None)
        separator = "&" if parsed_url.query else "?"
        return separator, query_params

    def extract_event_field(event, field, helper, default="Unknown"):
        try:
            # First try to get the single-value field
            value = event.get(field, default)
            # Check if there is a multi-value field and parse it if present
            mv_field = f"__mv_{field}"
            if mv_field in event:
                parsed_value = parse_mv(event[mv_field])
                # If the parsed multi-value field is empty, use the single-value field
                if parsed_value:
                    value = parsed_value
                helper.log_debug(
                    f'Field "{field}" is a multi-value field, parsed value="{parsed_value}"'
                )
            return value
        except KeyError:
            helper.log_warn(f'Failed to extract field "{field}" from record="{event}"')
            return default

    def normalize_tenants_scope(tenants_scope):
        try:
            if isinstance(tenants_scope, list):
                scope_list = tenants_scope
            elif isinstance(tenants_scope, str):
                ts = tenants_scope.strip()
                if ts == "" or ts == "*":
                    scope_list = ["*"]
                else:
                    scope_list = [s.strip() for s in ts.split(",") if s.strip()]
            else:
                scope_list = ["*"]
        except Exception:
            scope_list = ["*"]
        return scope_list

    def get_maintenance_status(helper, server_uri, session_key):
        try:
            header = {
                "Authorization": f"Splunk {session_key}",
                "Content-Type": "application/json",
            }
            endpoint = f"{server_uri}/services/trackme/v2/maintenance/check_global_maintenance_status"
            response = requests.get(endpoint, headers=header, verify=False, timeout=60)
            response.raise_for_status()
            data = response.json()
            data["tenants_scope"] = normalize_tenants_scope(data.get("tenants_scope", "*"))
            helper.log_info(f"maintenance_check response={json.dumps(data)}")
            return data
        except Exception as e:
            helper.log_error(f"maintenance_check failed exception=\"{str(e)}\"")
            return None

    def tenant_in_scope(tenant_id, tenants_scope):
        scope_list = normalize_tenants_scope(tenants_scope)
        if "*" in scope_list:
            return True
        return tenant_id in scope_list

    def create_notable_record(event, drilldown_link, helper):

        # get properties
        properties = {
            k: extract_event_field(event, k, helper)
            for k in event
            if event[k] != "null" and not k.startswith("__")
        }

        # extract anomaly_reason
        anomaly_reason = extract_event_field(event, "anomaly_reason", helper)

        # if anomaly_reason is not already a list, attempt to convert it from a pipe separated string
        if not isinstance(anomaly_reason, list):
            try:
                anomaly_reason = anomaly_reason.split("|") if anomaly_reason else []
            except Exception as e:
                pass

        notable_record = {
            "_time": time.time(),
            "timeStr": str(datetime.now()),
            "tenant_id": event["tenant_id"],
            "alias": event.get("alias"),
            "object": event["object"],
            "object_category": event["object_category"],
            "keyid": event["keyid"],
            "priority": event["priority"],
            "state": extract_event_field(event, "object_state", helper),
            "anomaly_reason": anomaly_reason,
            "status_message": extract_event_field(event, "status_message", helper),
            "drilldown_link": drilldown_link,
            "properties": properties,
        }

        # include score and score_definition in the record if available from event
        # Score is always an integer
        score_raw = event.get("score")
        score_definition = event.get("score_definition")
        
        # Convert score to integer (score is always an integer)
        score = None
        if score_raw is not None:
            try:
                score = int(float(score_raw))  # Convert to float first to handle string "100.0", then to int
            except (ValueError, TypeError):
                helper.log_warning(
                    f"Failed to convert score to integer: score_raw={score_raw}, tenant_id={event.get('tenant_id')}, object={event.get('object')}"
                )
                score = None
        
        if score is not None:
            notable_record["score"] = score
        if score_definition is not None:
            notable_record["score_definition"] = score_definition

        # calculate the event_id as the sha-256 sum of the notable_record
        event_id = hashlib.sha256(json.dumps(notable_record).encode()).hexdigest()

        # add to the record
        notable_record["event_id"] = event_id

        # return record
        return notable_record

    helper.log_info("Alert action trackme_notable started.")

    # session_key
    session_key = helper.session_key

    # server_uri
    server_uri = helper.settings["server_uri"]

    # results_link
    results_link = helper.settings["results_link"]

    # from results_link, extract server_name and server_port
    parsed_url = urlparse(results_link)

    # Extract the server name (hostname) and server port
    server_protocol = parsed_url.scheme
    server_name = parsed_url.hostname
    server_port = parsed_url.port

    # title
    title = helper.get_param("title") or "trackme:notable"

    # drilldown_root_uri
    drilldown_root_uri = helper.get_param("drilldown_root_uri")

    # drilldown_earliest
    drilldown_earliest = helper.get_param("drilldown_earliest") or "-24h"

    # Maintenance status (queried once)
    maintenance_info = get_maintenance_status(helper, server_uri, session_key)

    # Loop through the events
    for event in helper.get_events():

        tenant_id = event["tenant_id"]

        # If maintenance is active and the tenant is in scope, skip this event
        try:
            if maintenance_info and maintenance_info.get("maintenance"):
                if tenant_in_scope(tenant_id, maintenance_info.get("tenants_scope", ["*"])):
                    helper.log_info(
                        f"maintenance active for tenant_id={tenant_id}, skipping trackme_notable event"
                    )
                    continue
        except Exception:
            # Fail-open per event
            pass
        tenant_idx = trackme_idx_for_tenant(session_key, server_uri, tenant_id)
        tenant_trackme_notable_idx = tenant_idx["trackme_notable_idx"]

        # extract the component suffix as splk-<component>
        component_suffix = event["object_category"].split("-")[-1]

        # Base URL without query parameters
        base_url = f"{server_protocol}://{server_name}"
        if not (server_protocol == "https" and server_port == 443) and not (
            server_protocol == "http" and server_port == 80
        ):
            base_url += f":{server_port}"

        # Path and query parameters
        path = "/app/trackme/TenantHome"
        query_params = {
            "tenant_id": tenant_id,
            "component": component_suffix,
            "keyid": event["keyid"],
            "earliest": drilldown_earliest,
        }

        # Construct the full URL
        if drilldown_root_uri:
            # Parse the drilldown_root_uri
            parsed_uri = urlparse(drilldown_root_uri)
            base_url = f"{parsed_uri.scheme}://{parsed_uri.netloc}"

            # Preserve the path from the drilldown_root_uri
            root_path = parsed_uri.path.rstrip(
                "/"
            )  # Strip trailing slash for consistency
            full_path = (
                f"{root_path}{path}"  # Combine root path with the specific app path
            )

            # Adjust query parameters if needed
            _, adjusted_query_params = adjust_query_params(
                drilldown_root_uri, query_params.copy()
            )
            drilldown_link = f"{base_url}{full_path}?{urlencode(adjusted_query_params)}"
        else:
            base_url = f"{server_protocol}://{server_name}"
            if not (server_protocol == "https" and server_port == 443) and not (
                server_protocol == "http" and server_port == 80
            ):
                base_url += f":{server_port}"
            full_path = (
                path  # Default to the base path if no drilldown_root_uri provided
            )
            drilldown_link = f"{base_url}{full_path}?{urlencode(query_params)}"

        # Get notable_record
        notable_record = create_notable_record(event, drilldown_link, helper)

        # generate notable event
        try:
            trackme_state_event(
                session_key=session_key,
                splunkd_uri=server_uri,
                tenant_id=tenant_id,
                index=tenant_trackme_notable_idx,
                sourcetype="trackme:notable",
                source=title,
                record=notable_record,
            )
            helper.log_info(
                f'tenant_id="{notable_record.get("tenant_id")}", object_category="{notable_record.get("object_category")}", object="{notable_record.get("object")}", Notable event created successfully, response="{json.dumps(notable_record, indent=2)}"'
            )
            return 0
        except Exception as e:
            helper.log_error(
                f'tenant_id="{notable_record.get("tenant_id")}", object_category="{notable_record.get("object_category")}", object="{notable_record.get("object")}", Notable event creation failure, response="{json.dumps(notable_record, indent=2)}", exception="{str(e)}"'
            )
            return 1
