#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_splk_flx.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2022-2026, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys
import time
import threading
import hashlib
from collections import OrderedDict

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_flx_power", "trackme_rest_api_splk_flx_power.log"
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import (
    trackme_getloglevel,
    trackme_audit_event,
    trackme_register_tenant_component_summary,
)

# import trackme libs get data
from trackme_libs_get_data import (
    batch_find_records_by_object,
    batch_find_records_by_key,
)

# import trackme libs persistent fields definition
from collections_data import (
    persistent_fields_flx,
)

# import trackme libs bulk edit
from trackme_libs_bulk_edit import post_bulk_edit, generic_batch_update

# import trackme libs utils
from trackme_libs_utils import interpret_boolean

# import trackme libs splk flx
from trackme_libs_splk_flx import normalize_flx_tracker_name

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkFlxTrackingWrite_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkFlxTrackingWrite_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_flx(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_flx/write",
            "resource_group_desc": "Endpoints specific to the splk-flx TrackMe component (Splunk Flex objects tracking, power operations)",
        }

        return {"payload": response, "status": 200}

    def register_component_summary_async(
        self, session_key, splunkd_uri, tenant_id, component
    ):
        try:
            summary_register_response = trackme_register_tenant_component_summary(
                session_key,
                splunkd_uri,
                tenant_id,
                component,
            )
            logger.debug(
                f'function="trackme_register_tenant_component_summary", response="{json.dumps(summary_register_response, indent=2)}"'
            )
        except Exception as e:
            logger.error(
                f'failed to register the component summary with exception="{str(e)}"'
            )

    # Bulk edit (to be used from the inline Tabulator)
    def post_flx_bulk_edit(self, request_info, **kwargs):
        """
        This function performs a bulk edit on given json data.
        :param request_info: Contains request related information
        :param kwargs: Other keyword arguments
        :return: Status and payload of the bulk edit operation
        """

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # call the bulkd edit function
        response, http_status = post_bulk_edit(
            self,
            log=logger,
            loglevel=loglevel,
            service=service,
            request_info=request_info,
            component_name="flx",
            persistent_fields=persistent_fields_flx,
            collection_name_suffix="flx",
            endpoint_suffix="flx",
            **kwargs,
        )

        return {
            "payload": response,
            "status": http_status,
        }

    # Update priority by object name
    def post_flx_update_priority(self, request_info, **kwargs):

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                priority = resp_dict["priority"]
                if priority not in ("low", "medium", "high", "critical", "pending"):
                    return {
                        "payload": f"Invalid option for priority with priority received: {priority}, valid options are: low | medium | high | critical | pending",
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint updates the priority definition for an existing metric host, it requires a POST call with the following information:",
                "resource_desc": "Update priority for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_update_priority\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'priority': 'high', 'object_list': 'Okta:Splunk_TA_okta_identity_cloud:okta_logs'}\"",
                "options": [
                    {
                        "tenant_id": "The tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "priority": "priority value, valid options are low / medium / high / critical / pending",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_flx_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "flx",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {"priority": priority, "priority_updated": 1}

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_flx,
            component="flx",
            update_comment=update_comment,
            audit_context="update priority",
            audit_message="Priority was updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Enable/Disable monitoring by object name
    def post_flx_monitoring(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                action = resp_dict["action"]
                if not action in ("enable", "disable"):
                    return {
                        "payload": f"Invalid option for action, valid options are: enable | disable",
                        "status": 500,
                    }
                else:
                    if action == "enable":
                        action_value = "enabled"
                    elif action == "disable":
                        action_value = "disabled"

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint enables data monitoring for an existing data source by the entity name (object), it requires a POST call with the following information:",
                "resource_desc": "Enable/Disable monitoring for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_monitoring\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'action': 'disable', 'object_list': 'Okta:Splunk_TA_okta_identity_cloud:okta_logs'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "action": "The action to be performed, valid options are: enable | disable",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_flx_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "flx",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {"monitored_state": action_value}

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_flx,
            component="flx",
            update_comment=update_comment,
            audit_context="update monitoring",
            audit_message="Monitoring state was updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Remove entities
    def post_flx_delete(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                deletion_type = resp_dict["deletion_type"]
                if deletion_type not in ("temporary", "permanent"):
                    return {
                        "payload": {
                            "error": "Invalid option for deletion_type, valid options are: tempoary | permanent"
                        },
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint performs a permanent or temporary deletion of entities, it requires a POST call with the following information:",
                "resource_desc": "Delete one or more entities, either temporarily or permanently",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_delete\" mode=\"post\" body=\"{'tenant_id':'mytenant', 'deletion_type': 'temporary', 'object_list':'Okta:Splunk_TA_okta_identity_cloud:okta_logs'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "deletion_type": "The type of deletion, valid options are: temporary | permanent",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_flx_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Permanently deleted objects
        collection_perm_deleted_name = (
            f"kv_trackme_common_permanently_deleted_objects_tenant_{tenant_id}"
        )
        collection_perm_deleted = service.kvstore[collection_perm_deleted_name]

        #
        # Outliers collections
        #

        # entity rules collection
        collection_outliers_entity_rules_name = (
            f"kv_trackme_flx_outliers_entity_rules_tenant_{tenant_id}"
        )
        collection_entity_rules = service.kvstore[collection_outliers_entity_rules_name]

        # data rules collection
        collection_outliers_entity_data_name = (
            f"kv_trackme_flx_outliers_entity_data_tenant_{tenant_id}"
        )
        collection_outliers_entity_data = service.kvstore[
            collection_outliers_entity_data_name
        ]

        # Convert comma-separated lists to Python lists if needed
        if isinstance(object_list, str):
            object_list = object_list.split(",")
        if isinstance(keys_list, str):
            keys_list = keys_list.split(",")

        # Determine query method based on input
        if object_list:
            kvrecords_dict, kvrecords = batch_find_records_by_object(
                collection, object_list
            )
        elif keys_list:
            kvrecords_dict, kvrecords = batch_find_records_by_key(collection, keys_list)
        else:
            return {
                "payload": {
                    "error": "either object_list or keys_list must be provided"
                },
                "status": 500,
            }

        # loop and proceed
        for kvrecord in kvrecords:
            key = kvrecord.get("_key")
            object_value = kvrecord.get("object")

            try:
                # Remove the record
                collection.data.delete(json.dumps({"_key": key}))

                if deletion_type == "permanent":
                    # Register a new permanently deleted object, if not already present
                    try:
                        record_perm_deleted = collection_perm_deleted.data.query(
                            query=json.dumps({"object": object_value, "object_category": "splk-flx"})
                        )[0]
                        pass
                    except Exception as e:
                        record_perm_deleted = None

                    if not record_perm_deleted:
                        try:
                            collection_perm_deleted.data.insert(
                                json.dumps(
                                    {
                                        "ctime": str(time.time()),
                                        "object": str(object_value),
                                        "object_category": "splk-flx",
                                    }
                                )
                            )
                        except Exception as e:
                            logger.error(
                                f'tenant_id="{tenant_id}", failed to register a new permanently deleted object, object="{object_value}", exception="{str(e)}"'
                            )

                # try deleting outliers records for rules and data, there might be nothing to delete
                try:
                    collection_entity_rules.data.delete(
                        json.dumps({"object": object_value})
                    )
                except Exception as e:
                    pass

                try:
                    collection_outliers_entity_data.data.delete(
                        json.dumps({"object": object_value})
                    )
                except Exception as e:
                    pass

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        f"delete {deletion_type}",
                        str(object_value),
                        "splk-flx",
                        str(json.dumps(kvrecord, indent=1)),
                        "Entity was deleted successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                # increment counter
                processed_count += 1
                succcess_count += 1
                failures_count += 0

                # append for summary
                result = {
                    "object": object_value,
                    "action": "delete",
                    "result": "success",
                    "message": f'tenant_id="{tenant_id}", The object was successfully deleted',
                }
                records.append(result)

            except Exception as e:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                result = {
                    "object": object_value,
                    "action": "delete",
                    "result": "failure",
                    "exception": f'tenant_id="{tenant_id}", failed to remove the entity, object="{object_value}", exception="{str(e)}"',
                }
                records.append(result)

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "flx",
            ),
        )
        thread.start()

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Update monitoring week days by object name
    def post_flx_update_wdays(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                # Week days monitoring can be:
                # manual:all_days / manual:monday-to-friday / manual:monday-to-saturday / [ 0, 1, 2, 3, 4, 5, 6 ] where Sunday is 0
                monitoring_wdays = resp_dict["monitoring_wdays"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint configures the week days monitoring rule for an existing data source, it requires a POST call with the following information:",
                "resource_desc": "Update week days monitoring for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_update_wdays\" mode=\"post\" body=\"{'tenant_id':'mytenant','object_list':'netscreen:netscreen:firewall','monitoring_wdays':'manual:1,2,3,4,5'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "monitoring_wdays": "the week days rule, valid options are manual:all_days / manual:monday-to-friday / manual:monday-to-saturday / [ 0, 1, 2, 3, 4, 5, 6 ] where Sunday is 0",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_flx_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "flx",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {"monitoring_wdays": monitoring_wdays}

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_flx,
            component="flx",
            update_comment=update_comment,
            audit_context="update week days monitoring",
            audit_message="Week days monitoring was updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Update monitoring hours ranges by object name
    def post_flx_update_hours_ranges(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                monitoring_hours_ranges = resp_dict["monitoring_hours_ranges"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint configures the week days monitoring rule for an existing data source, it requires a POST call with the following information:",
                "resource_desc": "Update hours of monitoring for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_update_hours_ranges\" mode=\"post\" body=\"{'tenant_id':'mytenant', 'object_list':'netscreen:netscreen:firewall', 'monitoring_hours_ranges':'manual:8,9,10,11,12,13,14,15,16,17'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "monitoring_hours_ranges": "the hours ranges rule, valid options are manual:all_ranges / manual:08h-to-20h / [ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 ] where 00h00 to 01h59 is 0",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_flx_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "flx",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {"monitoring_hours_ranges": monitoring_hours_ranges}

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_flx,
            component="flx",
            update_comment=update_comment,
            audit_context="update hours ranges monitoring",
            audit_message="Monitoring hours ranges were updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Update monitoring time policy and rules by object name
    def post_flx_update_monitoring_time(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                # Get monitoring_time_policy (optional)
                monitoring_time_policy = resp_dict.get("monitoring_time_policy", None)
                
                # Get monitoring_time_rules (optional, takes precedence over policy)
                monitoring_time_rules = resp_dict.get("monitoring_time_rules", None)

                # Validate that at least one is provided
                if not monitoring_time_policy and not monitoring_time_rules:
                    return {
                        "payload": {
                            "error": "either monitoring_time_policy or monitoring_time_rules must be provided"
                        },
                        "status": 500,
                    }

                # Validate monitoring_time_rules format if provided
                if monitoring_time_rules is not None:
                    if isinstance(monitoring_time_rules, str):
                        try:
                            monitoring_time_rules = json.loads(monitoring_time_rules)
                        except Exception as e:
                            return {
                                "payload": {
                                    "error": f"monitoring_time_rules must be a valid JSON object, error: {str(e)}"
                                },
                                "status": 500,
                            }
                    
                    if not isinstance(monitoring_time_rules, dict):
                        return {
                            "payload": {
                                "error": "monitoring_time_rules must be a dictionary with week day keys (0-6) and hour lists as values"
                            },
                            "status": 500,
                        }
                    
                    # Validate each day entry
                    for day_key, hours_list in monitoring_time_rules.items():
                        try:
                            day_int = int(day_key)
                            if day_int < 0 or day_int > 6:
                                return {
                                    "payload": {
                                        "error": f"monitoring_time_rules: day key must be 0-6 (Sunday-Saturday), got {day_int}"
                                    },
                                    "status": 500,
                                }
                        except (ValueError, TypeError):
                            return {
                                "payload": {
                                    "error": f"monitoring_time_rules: day key must be an integer 0-6, got {day_key}"
                                },
                                "status": 500,
                            }
                        
                        if not isinstance(hours_list, list):
                            return {
                                "payload": {
                                    "error": f"monitoring_time_rules: hours for day {day_key} must be a list"
                                },
                                "status": 500,
                            }
                        
                        # Validate hours (0-23, with optional decimals like 0.5, 0.25)
                        for hour in hours_list:
                            try:
                                hour_float = float(hour)
                                if hour_float < 0 or hour_float >= 24:
                                    return {
                                        "payload": {
                                            "error": f"monitoring_time_rules: hours must be between 0-23.99, got {hour_float}"
                                        },
                                        "status": 500,
                                    }
                                # Check for valid decimal increments (0.25, 0.5, 0.75)
                                if hour_float != int(hour_float):
                                    decimal_part = hour_float - int(hour_float)
                                    if decimal_part not in [0.25, 0.5, 0.75]:
                                        return {
                                            "payload": {
                                                "error": f"monitoring_time_rules: decimal hours must be .25, .5, or .75, got {hour_float}"
                                            },
                                            "status": 500,
                                        }
                            except (ValueError, TypeError):
                                return {
                                    "payload": {
                                        "error": f"monitoring_time_rules: hour value must be numeric, got {hour}"
                                    },
                                    "status": 500,
                                }

                # Validate monitoring_time_policy format if provided
                if monitoring_time_policy is not None:
                    valid_policies = [
                        "all_time",
                        "business_days_all_hours",
                        "monday_saturday_all_hours",
                        "business_days_08h_20h",
                        "monday_saturday_08h_20h",
                    ]
                    
                    # Check if it's a string/list of predefined rules
                    if isinstance(monitoring_time_policy, str):
                        if monitoring_time_policy not in valid_policies:
                            # Try to parse as JSON (might be dictionary format)
                            try:
                                monitoring_time_policy = json.loads(monitoring_time_policy)
                            except Exception:
                                return {
                                    "payload": {
                                        "error": f"monitoring_time_policy must be one of {valid_policies} or a valid JSON dictionary"
                                    },
                                    "status": 500,
                                }
                    elif isinstance(monitoring_time_policy, list):
                        # Validate all items in list are valid policies
                        for policy in monitoring_time_policy:
                            if policy not in valid_policies:
                                return {
                                    "payload": {
                                        "error": f"monitoring_time_policy list contains invalid policy: {policy}, valid options are {valid_policies}"
                                    },
                                    "status": 500,
                                }
                    elif isinstance(monitoring_time_policy, dict):
                        # Dictionary format is allowed (same as monitoring_time_rules)
                        pass
                    else:
                        return {
                            "payload": {
                                "error": "monitoring_time_policy must be a string, list of strings, or dictionary"
                            },
                            "status": 500,
                        }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint configures the monitoring time policy and rules for an existing flex object, it requires a POST call with the following information:",
                "resource_desc": "Update monitoring time policy/rules for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_update_monitoring_time\" mode=\"post\" body=\"{'tenant_id':'mytenant','object_list':'group:subgroup:object','monitoring_time_policy':'business_days_08h_20h'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "monitoring_time_policy": "OPTIONAL: predefined policy name (all_time, business_days_all_hours, monday_saturday_all_hours, business_days_08h_20h, monday_saturday_08h_20h) or dictionary format like monitoring_time_rules",
                        "monitoring_time_rules": "OPTIONAL: dictionary with week day keys (0-6 for Sunday-Saturday) and hour lists as values, e.g. {0: [8,9,10], 1: [8,9,10]}. Hours can be 0-23 or decimals like 0.5 (00:30), 0.25 (00:15), 0.75 (00:45). Takes precedence over monitoring_time_policy if both provided.",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_flx_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "flx",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        # monitoring_time_rules takes precedence over monitoring_time_policy
        update_fields = {}
        if monitoring_time_rules is not None:
            # Convert dict keys to strings if they're integers (for JSON serialization)
            if isinstance(monitoring_time_rules, dict):
                monitoring_time_rules = {str(k): v for k, v in monitoring_time_rules.items()}
            update_fields["monitoring_time_rules"] = json.dumps(monitoring_time_rules) if isinstance(monitoring_time_rules, dict) else monitoring_time_rules
        if monitoring_time_policy is not None:
            if isinstance(monitoring_time_policy, dict):
                monitoring_time_policy = {str(k): v for k, v in monitoring_time_policy.items()}
            update_fields["monitoring_time_policy"] = json.dumps(monitoring_time_policy) if isinstance(monitoring_time_policy, (dict, list)) else monitoring_time_policy

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_flx,
            component="flx",
            update_comment=update_comment,
            audit_context="update monitoring time policy/rules",
            audit_message="Monitoring time policy/rules were updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Update list of manual tags
    def post_flx_update_manual_tags(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                try:
                    tags_manual = resp_dict["tags_manual"]
                    # if is a list, deduplicate, make it lowercase, sort it and turn as a CSV string
                    if isinstance(tags_manual, list):
                        tags_manual = ",".join(
                            sorted(list(set([x.lower() for x in tags_manual])))
                        )
                    else:
                        # if is a string, split it, deduplicate, make it lowercase, sort it and turn as a CSV string
                        tags_manual = ",".join(
                            sorted(
                                list(set([x.lower() for x in tags_manual.split(",")]))
                            )
                        )

                except Exception as e:
                    return {
                        "payload": {
                            "error": "tags_manual must be provided as a comma seperated list of tags"
                        },
                        "status": 500,
                    }

        else:
            describe = True

        if describe:
            response = {
                "describe": "This endpoint allows defining a comma seperated list of manual tags, it requires a POST call with the following information:",
                "resource_desc": "Define a comma seperated list of tags for one or more entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_update_manual_tags\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'tags_manual': 'mytag1,maytag2,mytag3', 'object_list': 'netscreen:netscreen:firewall,wineventlog:WinEventLog'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "tags_manual": "A comma seperated list of tags to be applied to the entities, to purge all manual tags, send an empty string",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # records summary
        records = []

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_flx_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Tags policies collection
        collection_tags_policies_name = f"kv_trackme_flx_tags_tenant_{tenant_id}"
        collection_tags_policies = service.kvstore[collection_tags_policies_name]

        # loop and proceed
        if object_list:
            keys_list = []
            for object_value in object_list:
                try:
                    kvrecord = collection.data.query(
                        query=json.dumps({"object": object_value})
                    )[0]
                    key = kvrecord.get("_key")
                    keys_list.append(key)
                except Exception as e:
                    key = None

        for key in keys_list:
            try:
                kvrecord = collection.data.query(query=json.dumps({"_key": key}))[0]

                # check if we have tags policies already
                try:
                    kvrecord_tags_policies = collection_tags_policies.data.query(
                        query=json.dumps({"_key": key})
                    )[0]
                except Exception as e:
                    kvrecord_tags_policies = None

                # check if we have tags_auto (list)
                try:
                    tags_auto = kvrecord_tags_policies.get("tags_auto", [])
                except Exception as e:
                    tags_auto = []

                # Update the record
                object_value = kvrecord.get("object")
                tags = kvrecord.get("tags", None)  # get current tags

                # if we have tags, the format is CSV, turn into a list
                if tags:
                    tags = tags.split(",")

                # update the record with ours manual tags
                kvrecord["tags_manual"] = tags_manual

                # make tags_manual_list (list from tags_manual CSV)
                tags_manual_list = tags_manual.split(",")

                # merged them all: define the tags field as the deduplicated, lowercase and sorted list of tags based on the tags_auto and tags_manual_list
                tags = ",".join(
                    sorted(
                        list(
                            set([x.lower() for x in tags_auto + tags_manual_list if x])
                        )
                    )
                )
                # update tags in the kvrecord now
                kvrecord["tags"] = tags
                kvrecord["mtime"] = time.time()
                collection.data.update(str(key), json.dumps(kvrecord))

                # Record an audit change
                trackme_audit_event(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    request_info.user,
                    "success",
                    "update manual tags",
                    str(object_value),
                    "splk-flx",
                    {"tags_manual": tags_manual_list},
                    "Manual tags list was updated successfully",
                    str(update_comment),
                )

                # increment counter
                processed_count += 1
                succcess_count += 1
                failures_count += 0

                # append for summary
                result = {
                    "object": object_value,
                    "action": "update",
                    "result": "success",
                    "message": f'tenant_id="{tenant_id}", The object was successfully updated',
                }
                records.append(result)

            except Exception as e:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                result = {
                    "object": object_value,
                    "action": "update",
                    "result": "failure",
                    "exception": f'tenant_id="{tenant_id}", failed to update the entity, object="{object_value}", exception="{str(e)}"',
                }
                records.append(result)

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "flx",
            ),
        )
        thread.start()

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Update SLA class
    def post_flx_update_sla_class(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]

                # handle object_list / keys_list
                object_list = resp_dict.get("object_list", None)
                if object_list:
                    if not isinstance(object_list, list):
                        object_list = object_list.split(",")

                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")

                if not object_list and not keys_list:
                    return {
                        "payload": {
                            "error": "either object_list or keys_list must be provided"
                        },
                        "status": 500,
                    }

                try:
                    sla_class = resp_dict["sla_class"]
                except Exception as e:
                    return {
                        "payload": {"error": "sla_class must be provided"},
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint updates the SLA class per entity, it requires a POST call with the following information:",
                "resource_desc": "Update SLA class for a comma separated list of entities",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_update_sla_class\" mode=\"post\" body=\"{'tenant_id':'mytenant','object_list':'netscreen:netscreen:firewall','sla_class':'gold'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "object_list": "List of object entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "keys_list": "List of key entities, provided as a comma separated list of fields, you can provid object_list or keys_list",
                        "sla_class": "(required) The SLA class to be applied to the entities",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_flx_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Prepare the request_info with the necessary data
        update_request_info = {
            "tenant_id": tenant_id,
            "component": "flx",
            "object_list": object_list,
            "keys_list": keys_list,
        }

        # Prepare the update fields
        update_fields = {
            "sla_class": sla_class,
        }

        # Call the generic update function
        response, status_code = generic_batch_update(
            self,
            request_info,
            update_request_info=update_request_info,
            collection=collection,
            update_fields=update_fields,
            persistent_fields=persistent_fields_flx,
            component="flx",
            update_comment=update_comment,
            audit_context="update SLA class",
            audit_message="SLA class was updated successfully",
        )

        return {"payload": response, "status": status_code}

    # Add new policy
    def post_flx_thresholds_add(self, request_info, **kwargs):

        # Declare
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                # handle keys_list
                keys_list = resp_dict.get("keys_list", None)
                if keys_list:
                    if not isinstance(keys_list, list):
                        keys_list = keys_list.split(",")
                else:
                    return {
                        "payload": {"error": "keys_list is required"},
                        "status": 500,
                    }

                # Validate threshold fields
                try:
                    metric_name = resp_dict["metric_name"]
                    value = resp_dict["value"]
                    operator = resp_dict["operator"]
                    condition_true = resp_dict["condition_true"]
                    comment = resp_dict.get("comment", "")
                    # Score is optional, default to 100 if not provided
                    score = resp_dict.get("score", 100)

                    # Validate value is not null or empty, we can accept both numerical values or string referencing a field in the metrics_record
                    if value is None or value == "":
                        return {
                            "payload": {
                                "error": "value must be a numeric value (integer or float)"
                            },
                            "status": 500,
                        }

                    try:
                        # Try to convert to float first
                        value = float(value)
                        # If it's a whole number, convert to int
                        if value.is_integer():
                            value = int(value)
                    except (ValueError, TypeError):
                        return {
                            "payload": {
                                "error": "value must be a numeric value (integer or float)"
                            },
                            "status": 500,
                        }

                    # Validate operator
                    if operator not in ["<", ">", "<=", ">=", "==", "!="]:
                        return {
                            "payload": {
                                "error": "operator must be one of: <, >, <=, >=, ==, !="
                            },
                            "status": 500,
                        }

                    # Validate condition_true (accept 0 or False, 1 or True)
                    try:
                        condition_true = interpret_boolean(condition_true)
                    except ValueError as e:
                        return {
                            "payload": {"error": str(e)},
                            "status": 500,
                        }

                    # Validate score (must be integer between 0 and 100, default 100)
                    try:
                        score = int(score)
                        if score < 0 or score > 100:
                            return {
                                "payload": {
                                    "error": "score must be an integer between 0 and 100"
                                },
                                "status": 500,
                            }
                    except (ValueError, TypeError):
                        return {
                            "payload": {
                                "error": "score must be an integer between 0 and 100"
                            },
                            "status": 500,
                        }

                    # Create threshold object
                    threshold_record = {
                        "metric_name": metric_name,
                        "value": value,
                        "operator": operator,
                        "condition_true": condition_true,
                        "mtime": time.time(),
                        "comment": comment,
                        "score": score,
                    }

                except KeyError as e:
                    return {
                        "payload": {"error": f"Missing required field: {str(e)}"},
                        "status": 500,
                    }
                except Exception as e:
                    return {
                        "payload": {"error": f"Invalid threshold data: {str(e)}"},
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint creates a new threshold or updates an existing threshold for a given object, it requires a POST call with the following data:",
                "resource_desc": "Add or update a threshold",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_flx/write/flx_thresholds_add\" body=\"{'tenant_id': 'mytenant', 'keys_list': 'object1,object2', 'metric_name': 'error_count', 'value': 1000, 'operator': '>', 'condition_true': 1, 'comment': 'Alert on high error count'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "keys_list": "(required) Comma separated list of object keys to apply the threshold to",
                        "metric_name": "(required) Name of the metric to set threshold for",
                        "value": "(required) Numeric threshold value",
                        "operator": "(required) Comparison operator (<, >, <=, >=, ==, !=)",
                        "condition_true": "(required) Condition to be met (0 or 1 or True or False)",
                        "comment": "(optional) Description of the threshold",
                        "score": "(optional) Score value (0-100) to assign when threshold is breached, defaults to 100 if not provided",
                        "update_comment": "(optional) Comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # summary records
        summary_records = []

        # Data collection
        collection_name = f"kv_trackme_flx_thresholds_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        for object_id in keys_list:

            # increment counter
            processed_count += 1

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

            # Define the KV query
            query_string = {
                "object_id": object_id,
                "metric_name": metric_name,
            }

            try:
                record = collection.data.query(query=json.dumps(query_string))
                record_key = record[0].get("_key")
                # Preserve existing score if updating and score not provided, otherwise use default 100
                existing_score = record[0].get("score")
                if existing_score is not None and "score" not in resp_dict:
                    try:
                        threshold_record["score"] = int(existing_score)
                    except (TypeError, ValueError):
                        threshold_record["score"] = 100
                elif "score" not in resp_dict:
                    threshold_record["score"] = 100

            except Exception as e:
                record_key = None
                # For new records, ensure score is set (already set above, but ensure it's there)
                if "score" not in threshold_record:
                    threshold_record["score"] = 100

            # set the object_id
            threshold_record["object_id"] = object_id

            if record_key:
                # Update the record
                try:
                    collection.data.update(
                        str(record_key),
                        json.dumps(threshold_record),
                    )
                    succcess_count += 1
                    summary_records.append(
                        {
                            "action": "update",
                            "result": "success",
                            "record": threshold_record,
                        }
                    )
                except Exception as e:
                    failures_count += 1
                    logger.error(
                        f'failed to update the threshold with exception="{str(e)}"'
                    )
                    summary_records.append(
                        {
                            "action": "update",
                            "result": "failure",
                            "record": threshold_record,
                            "exception": e,
                        }
                    )

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "update threshold",
                        object_id,
                        "splk-flx",
                        collection.data.query(query=json.dumps(query_string)),
                        "The threshold was updated successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

            else:
                # Insert the record
                try:
                    # add a random sha256 hash as the _key
                    threshold_record["_key"] = hashlib.sha256(
                        json.dumps(threshold_record).encode("utf-8")
                    ).hexdigest()
                    collection.data.insert(json.dumps(threshold_record))
                    succcess_count += 1
                    summary_records.append(
                        {
                            "action": "insert",
                            "result": "success",
                            "record": threshold_record,
                        }
                    )
                except Exception as e:
                    failures_count += 1
                    logger.error(
                        f'failed to insert the threshold with exception="{str(e)}"'
                    )
                    summary_records.append(
                        {
                            "action": "insert",
                            "result": "failure",
                            "record": threshold_record,
                            "exception": e,
                        }
                    )

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "add threshold",
                        object_id,
                        "splk-flx",
                        collection.data.query(query=json.dumps(query_string)),
                        "The threshold was added successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": summary_records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Delete records from the collection
    def post_flx_thresholds_del(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        keys_list = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    keys_list = resp_dict["keys_list"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The keys_list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                # if keys_list is a string, turn into a list
                if isinstance(keys_list, str):
                    keys_list = keys_list.split(",")

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint deletes thresholds, it requires a POST call with the following information:",
                "resource_desc": "Delete one or more thresholds",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_flx/write/flx_thresholds_del\" body=\"{'tenant_id': 'mytenant', 'keys_list': 'key1,key2'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "keys_list": "(required) Comma separated list of threshold keys to delete",
                        "update_comment": "(optional) Comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Data collection
        collection_name = f"kv_trackme_flx_thresholds_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # records summary
        records = []

        # loop
        for key in keys_list:
            try:
                # Get the current record
                record = collection.data.query_by_id(key)

                # Get the object_id before deleting the record
                object_id = record.get("object_id")

                # Remove the record
                collection.data.delete(json.dumps({"_key": key}))

                # increment counter
                processed_count += 1
                succcess_count += 1

                # audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "delete threshold",
                        object_id,
                        "splk-flx",
                        record,
                        "The threshold was deleted successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                result = {
                    "action": "delete",
                    "result": "success",
                    "record": record,
                }

                records.append(result)

                logger.info(json.dumps(result, indent=0))

            except Exception as e:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                # audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "delete threshold",
                        object_id,
                        "splk-flx",
                        None,
                        str(e),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                result = {
                    "action": "delete",
                    "result": "failure",
                    "record": key,
                    "exception": str(e),
                }

                # append to records
                records.append(result)

                # log
                logger.error(json.dumps(result, indent=0))

        # render HTTP status and summary
        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}
        else:
            return {"payload": req_summary, "status": 500}

    # Update records
    def post_flx_thresholds_update(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        describe = False
        records_list = None

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    records_list = resp_dict["records_list"]
                    if not isinstance(records_list, list):
                        return {
                            "payload": {
                                "error": f"records_list must be a list of records, received: {type(records_list)}, content: {records_list}"
                            },
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The records_list list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint updates multiple thresholds for flx objects, it requires a POST call with the following information:",
                "resource_desc": "Update multiple thresholds",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_thresholds_update\" mode=\"post\" body=\"{'tenant_id':'mytenant','records_list':[{'key':'key1','metric_name':'error_count','value':1000,'operator':'>','condition_true':'1','comment':'Alert on high error count'}]}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "threshold_records": "List of threshold records to update. Each record must contain:",
                        "records_list": {
                            "_key": "The key of the threshold to update",
                            "object_id": "The object id of the threshold to update",
                            "metric_name": "The name of the metric to set threshold for",
                            "value": "The threshold value (numeric)",
                            "operator": "The comparison operator (<, >, <=, >=, ==, !=)",
                            "condition_true": "The condition to be met (0 or 1 or True or False)",
                            "score": "(optional) The score (0-100) assigned when this threshold is breached. Default is 100.",
                            "comment": "(optional) A comment describing the threshold",
                        },
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Data collection
        collection_name = f"kv_trackme_flx_thresholds_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # records summary
        records = []

        # loop through validated records
        for record in records_list:
            key = None
            current_record = None
            try:
                key = record.get("_key")
                if not key:
                    raise KeyError("'_key' field is required in record")
                # Get the current record
                current_record = collection.data.query_by_id(key)

                # if value or operator are different, update the record and uptime the mtime
                # Get score from current_record, default to 100 if not present
                current_score = current_record.get("score")
                if current_score is None:
                    current_score = 100
                else:
                    try:
                        current_score = int(current_score)
                    except (TypeError, ValueError):
                        current_score = 100
                
                # Get score from record, preserve existing score if not provided
                record_score = record.get("score")
                if record_score is None:
                    # If score not provided, use the current score to preserve existing value
                    record_score = current_score
                else:
                    try:
                        record_score = int(record_score)
                        if record_score < 0 or record_score > 100:
                            return {
                                "payload": {
                                    "error": "score must be an integer between 0 and 100"
                                },
                                "status": 500,
                            }
                    except (TypeError, ValueError):
                        # If invalid, fall back to current score
                        record_score = current_score
                
                if (
                    current_record["value"] != record["value"]
                    or current_record["operator"] != record["operator"]
                    or current_record["condition_true"] != record["condition_true"]
                    or current_record.get("comment", "") != record.get("comment", "")
                    or current_score != record_score
                ):
                    # Ensure score is included in the record
                    record["score"] = record_score
                    # Preserve fields that might not be in the update record
                    if "metric_name" not in record:
                        record["metric_name"] = current_record.get("metric_name")
                    if "object_id" not in record:
                        record["object_id"] = current_record.get("object_id")
                    record["mtime"] = time.time()

                    # in record, convert condition_true to integer (from False/True to 0/1)
                    try:
                        record["condition_true"] = interpret_boolean(
                            record["condition_true"]
                        )
                    except ValueError as e:
                        logger.error(f"Invalid condition_true value: {str(e)}")
                        raise

                    # Update the record
                    collection.data.update(
                        str(key),
                        json.dumps(record),
                    )

                    # increment counter
                    processed_count += 1
                    succcess_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "update threshold",
                            current_record.get("object_id"),
                            "splk-flx",
                            current_record,
                            "The threshold was updated successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "update",
                        "result": "success",
                        "record": current_record,
                    }

                    records.append(result)

                    logger.info(json.dumps(result, indent=0))

            except Exception as e:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                # audit record
                try:
                    object_id = current_record.get("object_id") if current_record else record.get("object_id")
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "update threshold",
                        object_id,
                        "splk-flx",
                        None,
                        str(e),
                        str(update_comment),
                    )
                except Exception as audit_e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(audit_e)}"'
                    )

                result = {
                    "action": "update",
                    "result": "failure",
                    "record": key if key else record,
                    "exception": str(e),
                }

                # append to records
                records.append(result)

                # log
                logger.error(json.dumps(result, indent=0))

        # render HTTP status and summary
        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}
        else:
            return {"payload": req_summary, "status": 500}

    # Bulk update thresholds
    def post_flx_thresholds_update_bulk(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        component = None
        keys_list = None
        record_changes = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                try:
                    keys_list = resp_dict["keys_list"]

                    # if is a string, convert to list from comma separated string
                    if isinstance(keys_list, str):
                        keys_list = keys_list.split(",")

                    # must not be empty
                    if not keys_list or len(keys_list) == 0:
                        return {
                            "payload": {
                                "error": f"keys_list must be a list of keyids, provided as a native list or a comma separated string, received: {type(keys_list)}, content: {keys_list}"
                            },
                            "status": 500,
                        }

                except Exception as e:
                    return {
                        "payload": {
                            "response": "The keys_list list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                try:
                    record_changes = resp_dict["record_changes"]
                    if not isinstance(record_changes, dict):
                        return {
                            "payload": {
                                "error": f"record_changes must be a dictionary, received: {type(record_changes)}, content: {record_changes}"
                            },
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The record_changes dictionary is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint bulk updates thresholds for flx objects, it requires a POST call with the following information:",
                "resource_desc": "Bulk update multiple thresholds",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_thresholds_update_bulk\" mode=\"post\" body=\"{'tenant_id':'mytenant','component':'flx','keys_list':['key1','key2'],'record_changes':{'kpi_metric_name':'error_count','kpi_metric_value':1000,'operator':'>','condition_true':'1'},'update_comment':'Bulk update thresholds'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "component": "Component identifier (e.g., 'flx')",
                        "keys_list": "List of keyids to update",
                        "record_changes": {
                            "kpi_metric_name": "The name of the metric to set threshold for",
                            "kpi_metric_value": "The threshold value (numeric)",
                            "operator": "The comparison operator (<, >, <=, >=, ==, !=)",
                            "condition_true": "The condition to be met (0 or 1 or True or False)",
                        },
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }
            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        success_count = 0
        failures_count = 0

        # Data collection
        collection_name = f"kv_trackme_{component}_thresholds_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # records summary
        records = []

        # loop through keys_list
        for keyid in keys_list:
            logger.debug(f'keyid="{keyid}", searching for records in collection {collection_name}')

            try:
                # Get records in the KVstore that match this object_id (we get a list)
                kvrecords = collection.data.query(query=json.dumps({"object_id": keyid}))
                logger.debug(f'object_id="{keyid}" kvrecords="{json.dumps(kvrecords, indent=2)}"')

            except Exception as e:
                logger.error(f"Failed to retrieve records for object_id {keyid}: {str(e)}")
                return {
                    "payload": f"Failed to retrieve records from the KVstore collection {collection_name} for object_id {keyid}: {str(e)}",
                    "status": 500,
                }
            
            # if not kvrecords, continue, nothing to do.
            if not kvrecords:
                logger.info(f'object_id="{keyid}", no records found, nothing to do.')
                continue

            # loop through the records
            for kvrecord in kvrecords:

                logger.info(f'object_id="{keyid}", processing kvrecord="{json.dumps(kvrecord, indent=2)}"')

                original_record = kvrecord.copy()

                # check if the kpi metric name matches, otherwise skip record
                if kvrecord.get("metric_name") != record_changes["kpi_metric_name"]:
                    continue
                
                # Check if any of the record_changes fields are different
                needs_update = False
                updated_record = kvrecord.copy()

                # Update kpi_metric_value if provided and different
                if "kpi_metric_value" in record_changes and kvrecord.get("value") != record_changes["kpi_metric_value"]:
                    updated_record["value"] = record_changes["kpi_metric_value"]
                    needs_update = True

                # Update operator if provided and different
                if "operator" in record_changes and kvrecord.get("operator") != record_changes["operator"]:
                    updated_record["operator"] = record_changes["operator"]
                    needs_update = True

                # Update condition_true if provided and different
                if "condition_true" in record_changes:
                    new_condition_true = interpret_boolean(record_changes["condition_true"])
                    if interpret_boolean(kvrecord.get("condition_true")) != new_condition_true:
                        updated_record["condition_true"] = new_condition_true
                        needs_update = True

                # Only update if there are changes
                if not needs_update:
                    logger.info(f'object_id="{keyid}", no update is required, original_record="{json.dumps(original_record, indent=2)}"')
                    continue
                
                if needs_update:
                    updated_record["mtime"] = time.time()

                    # Try updating the record
                    try:

                        logger.info(f'object_id="{keyid}", update is required, original_record="{json.dumps(original_record, indent=0)}", updated_record="{json.dumps(updated_record, indent=2)}"')

                        # Update the record
                        collection.data.update(
                            updated_record.get("_key"),
                            json.dumps(updated_record),
                        )
                        # increment counter
                        processed_count += 1
                        success_count += 1
                        # audit record
                        try:
                            trackme_audit_event(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                request_info.user,
                                "success",
                                "bulk update threshold",
                                kvrecord.get("object_id"),
                                f"splk-{component}",
                                kvrecord,
                                "The threshold was bulk updated successfully",
                                str(update_comment),
                            )
                        except Exception as e:
                            logger.error(
                                f'failed to generate an audit event with exception="{str(e)}"'
                            )
                        result = {
                            "action": "bulk_update",
                            "result": "success",
                            "keyid": keyid,
                            "object_id": kvrecord.get("object_id"),
                            "changes": record_changes,
                        }
                        records.append(result)
                        logger.info(json.dumps(result, indent=0))

                    except Exception as e:
                        # increment counter
                        processed_count += 1
                        failures_count += 1
                        # audit record
                        try:
                            trackme_audit_event(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                request_info.user,
                                "failure",
                                "bulk update threshold",
                                keyid,
                                f"splk-{component}",
                                None,
                                str(e),
                                str(update_comment),
                            )
                        except Exception as e:
                            logger.error(
                                f'failed to generate an audit event with exception="{str(e)}"'
                            )
                        result = {
                            "action": "bulk_update",
                            "result": "failure",
                            "keyid": keyid,
                            "exception": str(e),
                        }
                        # append to records
                        records.append(result)
                        # log
                        logger.error(json.dumps(result, indent=0))

                else:
                    # No changes needed
                    processed_count += 1
                    success_count += 1
                    result = {
                        "action": "bulk_update",
                        "result": "no_changes_needed",
                        "keyid": keyid,
                        "object_id": kvrecord.get("object_id"),
                        "changes": record_changes,
                    }
                    records.append(result)
                    logger.info(json.dumps(result, indent=0))

        # render HTTP status and summary
        req_summary = {
            "process_count": processed_count,
            "success_count": success_count,
            "failures_count": failures_count,
            "records": records,
        }
        if processed_count == 0: # nothing to be done, return 200
            logger.info(f'endpoint flx_thresholds_update_bulk, processed_count="{processed_count}", success_count="{success_count}", nothing to be done, returning 200')
        elif processed_count == success_count:
            logger.info(f'endpoint flx_thresholds_update_bulk, processed_count="{processed_count}", success_count="{success_count}", thresholds records updated successfully, returning 200')
            return {"payload": req_summary, "status": 200}
        else:
            logger.info(f'endpoint flx_thresholds_update_bulk, processed_count="{processed_count}", success_count="{success_count}", returning 500')
            return {"payload": req_summary, "status": 500}

    # Update drilldown search definitions for Flex Objects
    def post_flx_update_drilldown_searches(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    drilldown_records = resp_dict["drilldown_records"]
                    if not isinstance(drilldown_records, list):
                        # try loading as a json string, if failed return
                        try:
                            drilldown_records = json.loads(drilldown_records)
                        except Exception as e:
                            return {
                                "payload": {
                                    "error": "drilldown_records must be a list of records or a JSON string containing a list of records"
                                },
                                "status": 500,
                            }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The drilldown_records list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint defines or updates drilldown search definitions for Flex Objects, it requires a POST call with the following information:",
                "resource_desc": "Define or update drilldown search definitions for Flex Objects",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_update_drilldown_searches\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'drilldown_records': [{'tracker_name': 'mytracker', 'drilldown_search': 'index=main | search source=*', 'drilldown_search_earliest': '-24h', 'drilldown_search_latest': 'now'}]}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "drilldown_records": "List of drilldown search records. Each record must contain:",
                        "drilldown_records_fields": {
                            "tracker_name": "The name of the associated tracker",
                            "drilldown_search": "The drilldown search definition",
                            "drilldown_search_earliest": "The drilldown search earliest time",
                            "drilldown_search_latest": "The drilldown search latest time",
                        },
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        success_count = 0
        failures_count = 0

        # records summary
        records = []

        # Data collection
        collection_name = f"kv_trackme_flx_drilldown_searches_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Process each drilldown record
        for drilldown_record in drilldown_records:
            try:
                # Validate required fields
                tracker_name = normalize_flx_tracker_name(tenant_id, drilldown_record.get("tracker_name"))
                drilldown_search = drilldown_record.get("drilldown_search")
                drilldown_search_earliest = drilldown_record.get("drilldown_search_earliest")
                drilldown_search_latest = drilldown_record.get("drilldown_search_latest")

                if not tracker_name:
                    raise ValueError("tracker_name is required")
                if not drilldown_search:
                    raise ValueError("drilldown_search is required")
                if not drilldown_search_earliest:
                    raise ValueError("drilldown_search_earliest is required")
                if not drilldown_search_latest:
                    raise ValueError("drilldown_search_latest is required")

                # Check if record exists for this tracker
                existing_records = collection.data.query(
                    query=json.dumps({"tracker_name": tracker_name})
                )

                # Prepare the record data
                record_data = {
                    "tracker_name": tracker_name,
                    "drilldown_search": drilldown_search,
                    "drilldown_search_earliest": drilldown_search_earliest,
                    "drilldown_search_latest": drilldown_search_latest,
                    "mtime": time.time(),
                }

                if existing_records:
                    # Update existing record
                    existing_record = existing_records[0]
                    record_key = existing_record.get("_key")
                    
                    # Update the record
                    collection.data.update(
                        str(record_key),
                        json.dumps(record_data),
                    )
                    
                    action = "update"
                    logger.info(f'Updated drilldown search for tracker="{tracker_name}"')
                else:
                    # Add new record
                    # Generate a unique key using SHA256 hash
                    record_data["_key"] = hashlib.sha256(
                        json.dumps(record_data).encode("utf-8")
                    ).hexdigest()
                    
                    collection.data.insert(json.dumps(record_data))
                    action = "insert"
                    logger.info(f'Added new drilldown search for tracker="{tracker_name}"')

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        f"{action} drilldown search",
                        tracker_name,
                        "splk-flx",
                        record_data,
                        f"Drilldown search was {action}d successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                # increment counters
                processed_count += 1
                success_count += 1

                # append for summary
                result = {
                    "tracker_name": tracker_name,
                    "action": action,
                    "result": "success",
                    "message": f'Drilldown search was {action}d successfully',
                }
                records.append(result)

            except Exception as e:
                # increment counters
                processed_count += 1
                failures_count += 1

                result = {
                    "tracker_name": drilldown_record.get("tracker_name", "unknown"),
                    "action": "error",
                    "result": "failure",
                    "exception": f'Failed to process drilldown record: {str(e)}',
                }
                records.append(result)
                logger.error(f'Failed to process drilldown record: {str(e)}')

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "flx",
            ),
        )
        thread.start()

        # render HTTP status and summary
        req_summary = {
            "process_count": processed_count,
            "success_count": success_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == success_count:
            return {"payload": req_summary, "status": 200}
        else:
            return {"payload": req_summary, "status": 500}

    # Delete drilldown search definitions for Flex Objects
    def post_flx_delete_drilldown_searches(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    keys_list = resp_dict["keys_list"]
                    if isinstance(keys_list, str):
                        keys_list = keys_list.split(",")
                    elif not isinstance(keys_list, list):
                        return {
                            "payload": {
                                "error": "keys_list must be a list or comma-separated string"
                            },
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The keys_list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint deletes drilldown search definitions for Flex Objects, it requires a POST call with the following information:",
                "resource_desc": "Delete drilldown search definitions for Flex Objects",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_delete_drilldown_searches\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'keys_list': 'key1,key2,key3'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "keys_list": "Comma-separated list of record keys (_key) to delete from the drilldown searches collection",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        success_count = 0
        failures_count = 0

        # records summary
        records = []

        # Data collection
        collection_name = f"kv_trackme_flx_drilldown_searches_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Process each key to delete
        for key in keys_list:
            try:
                # Get the current record before deleting to get tracker_name for audit
                try:
                    record = collection.data.query_by_id(key)
                    tracker_name = record.get("tracker_name", "unknown")
                except Exception as e:
                    # Record doesn't exist, skip it
                    logger.warning(f'Record with key="{key}" not found, skipping deletion')
                    continue

                # Delete the record
                collection.data.delete(json.dumps({"_key": key}))

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "delete drilldown search",
                        tracker_name,
                        "splk-flx",
                        record,
                        "Drilldown search was deleted successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                # increment counters
                processed_count += 1
                success_count += 1

                # append for summary
                result = {
                    "key": key,
                    "tracker_name": tracker_name,
                    "action": "delete",
                    "result": "success",
                    "message": "Drilldown search was deleted successfully",
                }
                records.append(result)
                logger.info(f'Deleted drilldown search for tracker="{tracker_name}" with key="{key}"')

            except Exception as e:
                # increment counters
                processed_count += 1
                failures_count += 1

                result = {
                    "key": key,
                    "action": "delete",
                    "result": "failure",
                    "exception": f'Failed to delete drilldown search: {str(e)}',
                }
                records.append(result)
                logger.error(f'Failed to delete drilldown search with key="{key}": {str(e)}')

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "flx",
            ),
        )
        thread.start()

        # render HTTP status and summary
        req_summary = {
            "process_count": processed_count,
            "success_count": success_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == success_count:
            return {"payload": req_summary, "status": 200}
        else:
            return {"payload": req_summary, "status": 500}

    # Update default metric definitions for Flex Objects
    def post_flx_update_default_metrics(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    default_metric_records = resp_dict["default_metric_records"]
                    if not isinstance(default_metric_records, list):
                        # try loading as a json string, if failed return
                        try:
                            default_metric_records = json.loads(default_metric_records)
                        except Exception as e:
                            return {
                                "payload": {
                                    "error": "default_metric_records must be a list of records or a JSON string containing a list of records"
                                },
                                "status": 500,
                            }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The default_metric_records list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint defines or updates default metric definitions for Flex Objects, it requires a POST call with the following information:",
                "resource_desc": "Define or update default metric definitions for Flex Objects",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_update_default_metrics\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'default_metric_records': [{'tracker_name': 'mytracker', 'metric_name': 'cpu_usage'}]}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "default_metric_records": "List of default metric records. Each record must contain:",
                        "default_metric_records_fields": {
                            "tracker_name": "The name of the associated tracker",
                            "metric_name": "The name of the default metric",
                        },
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        success_count = 0
        failures_count = 0

        # records summary
        records = []

        # Data collection
        collection_name = f"kv_trackme_flx_default_metric_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Process each default metric record
        for default_metric_record in default_metric_records:
            try:
                # Validate required fields
                tracker_name = normalize_flx_tracker_name(tenant_id, default_metric_record.get("tracker_name"))
                metric_name = default_metric_record.get("metric_name")

                if not tracker_name:
                    raise ValueError("tracker_name is required")
                if not metric_name:
                    raise ValueError("metric_name is required")

                # Check if record exists for this tracker
                existing_records = collection.data.query(
                    query=json.dumps({"tracker_name": tracker_name})
                )

                # Prepare the record data
                record_data = {
                    "tracker_name": tracker_name,
                    "metric_name": metric_name,
                    "mtime": time.time(),
                }

                if existing_records:
                    # Update existing record
                    existing_record = existing_records[0]
                    record_key = existing_record.get("_key")
                    
                    # Update the record
                    collection.data.update(
                        str(record_key),
                        json.dumps(record_data),
                    )
                    
                    action = "update"
                    logger.info(f'Updated default metric for tracker="{tracker_name}"')
                else:
                    # Add new record
                    # Generate a unique key using SHA256 hash
                    record_data["_key"] = hashlib.sha256(
                        json.dumps(record_data).encode("utf-8")
                    ).hexdigest()
                    
                    collection.data.insert(json.dumps(record_data))
                    action = "insert"
                    logger.info(f'Added new default metric for tracker="{tracker_name}"')

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        f"{action} default metric",
                        tracker_name,
                        "splk-flx",
                        record_data,
                        f"Default metric was {action}d successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                # increment counters
                processed_count += 1
                success_count += 1

                # append for summary
                result = {
                    "tracker_name": tracker_name,
                    "metric_name": metric_name,
                    "action": action,
                    "result": "success",
                    "message": f'Default metric was {action}d successfully',
                }
                records.append(result)

            except Exception as e:
                # increment counters
                processed_count += 1
                failures_count += 1

                result = {
                    "tracker_name": default_metric_record.get("tracker_name", "unknown"),
                    "action": "error",
                    "result": "failure",
                    "exception": f'Failed to process default metric record: {str(e)}',
                }
                records.append(result)
                logger.error(f'Failed to process default metric record: {str(e)}')

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "flx",
            ),
        )
        thread.start()

        # render HTTP status and summary
        req_summary = {
            "process_count": processed_count,
            "success_count": success_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == success_count:
            return {"payload": req_summary, "status": 200}
        else:
            return {"payload": req_summary, "status": 500}

    # Delete default metric definitions for Flex Objects
    def post_flx_delete_default_metrics(self, request_info, **kwargs):
        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    keys_list = resp_dict["keys_list"]
                    if isinstance(keys_list, str):
                        keys_list = keys_list.split(",")
                    elif not isinstance(keys_list, list):
                        return {
                            "payload": {
                                "error": "keys_list must be a list or comma-separated string"
                            },
                            "status": 500,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The keys_list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint deletes default metric definitions for Flex Objects, it requires a POST call with the following information:",
                "resource_desc": "Delete default metric definitions for Flex Objects",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/splk_flx/write/flx_delete_default_metrics\" mode=\"post\" body=\"{'tenant_id': 'mytenant', 'keys_list': 'key1,key2'}\"",
                "options": [
                    {
                        "tenant_id": "Tenant identifier",
                        "keys_list": "List of keys to delete, provided as a comma separated list of keys",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        success_count = 0
        failures_count = 0

        # records summary
        records = []

        # Data collection
        collection_name = f"kv_trackme_flx_default_metric_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Process each key for deletion
        for key in keys_list:
            try:
                # Get the record before deletion for audit purposes
                try:
                    existing_record = collection.data.query_by_id(key)
                    tracker_name = existing_record.get("tracker_name", "unknown")
                except Exception as e:
                    tracker_name = "unknown"

                # Delete the record
                collection.data.delete(json.dumps({"_key": key}))
                
                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "delete default metric",
                        tracker_name,
                        "splk-flx",
                        {"_key": key, "tracker_name": tracker_name},
                        "Default metric was deleted successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                # increment counters
                processed_count += 1
                success_count += 1

                # append for summary
                result = {
                    "key": key,
                    "tracker_name": tracker_name,
                    "action": "delete",
                    "result": "success",
                    "message": "Default metric was deleted successfully",
                }
                records.append(result)
                logger.info(f'Deleted default metric for tracker="{tracker_name}" with key="{key}"')

            except Exception as e:
                # increment counters
                processed_count += 1
                failures_count += 1

                result = {
                    "key": key,
                    "action": "delete",
                    "result": "failure",
                    "exception": f'Failed to delete default metric: {str(e)}',
                }
                records.append(result)
                logger.error(f'Failed to delete default metric with key="{key}": {str(e)}')

        # call trackme_register_tenant_component_summary
        thread = threading.Thread(
            target=self.register_component_summary_async,
            args=(
                request_info.session_key,
                request_info.server_rest_uri,
                tenant_id,
                "flx",
            ),
        )
        thread.start()

        # render HTTP status and summary
        req_summary = {
            "process_count": processed_count,
            "success_count": success_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == success_count:
            return {"payload": req_summary, "status": 200}
        else:
            return {"payload": req_summary, "status": 500}
