#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_sla_policies.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2022-2026, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys
import re
import time

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_sla_policies_power",
    "trackme_rest_api_splk_sla_policies_power.log",
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import (
    trackme_getloglevel,
    trackme_audit_event,
    get_kv_collection,
    trackme_reqinfo,
)

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkSlaPoliciesWrite_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkSlaPoliciesWrite_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_sla_policies(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_sla_policies/write",
            "resource_group_desc": "Endpoints related to the management of priorities (power operations)",
        }

        return {"payload": response, "status": 200}

    # Add new policy
    def post_sla_policies_add(self, request_info, **kwargs):

        # Declare
        tenant_id = None
        component = None
        sla_policy_id = None
        sla_policy_value = None
        sla_policy_regex = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                    if component not in (
                        "dsm",
                        "dhm",
                        "mhm",
                        "wlk",
                        "flx",
                        "fqm",
                    ):
                        return {
                            "payload": {
                                "action": "failure",
                                "response": f"invalid component {component}, valid options are: dsm/dhm/mhm/wlk/flx/fqm",
                            },
                            "status": 400,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    sla_policy_id = resp_dict["sla_policy_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The sla_policy_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    sla_policy_value = resp_dict["sla_policy_value"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The sla_policy_value is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                # if sla_policy_value is a string, turn into a list and make it lower case, if is a list, return an error as it is not accepted in this context
                if isinstance(sla_policy_value, str):

                    if not len(sla_policy_value) > 0:
                        return {
                            "payload": {
                                "response": "The sla_policy_value should not be empty",
                                "status": 400,
                            },
                            "status": 400,
                        }

                    else:
                        sla_policy_value = sla_policy_value.lower()

                else:
                    return {
                        "payload": {
                            "response": "The sla_policy_value should be a string",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    sla_policy_regex = resp_dict["sla_policy_regex"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The sla_policy_regex is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                # verify that the regex is valid
                try:
                    re.compile(sla_policy_regex)
                except re.error:
                    return {
                        "payload": {
                            "response": "The sla_policy_regex is not a valid regular expression",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint creates a new sla class policy or updates a policy if it exists already, it requires a POST call with the following data:",
                "resource_desc": "Add or update a sla class policy",
                "resource_spl_example": r"| trackme mode=post url=\"/services/trackme/v2/splk_sla_policies/write/sla_policies_add\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'sla_policy_id': 'linux_secure', 'sla_policy_id': 'linux_secure', 'sla_policy_value': 'Linux,OS,CIM', 'sla_policy_regex': '\:linux_secure$'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, valid values are: dsm/dhm/mhm/wlk/flx/fqm",
                        "sla_policy_id": "(required) ID of the sla class policy",
                        "sla_policy_regex": "(required) The regular expression to be used by the sla class policy, special characters should be escaped.",
                        "sla_policy_value": "(required) SLA class to be applied.",
                        "update_comment": "(optional) Comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Define the KV query
        query_string = {
            "sla_policy_id": sla_policy_id,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get trackmeconf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )["trackme_conf"]

        # Get SLA classes conf
        sla_classes = trackme_conf["sla"]["sla_classes"]

        # try loading the JSON
        try:
            sla_classes = json.loads(sla_classes)
            if sla_policy_value not in sla_classes:
                error_msg = f"The SLA class {sla_policy_value} is not available currently in TrackMe's configuration, available classes: {json.dumps(sla_classes, 0)}"
                logger.error(error_msg)
                return {
                    "payload": {
                        "response": f"{error_msg}",
                        "status": 500,
                    },
                    "status": 500,
                }

        except:
            error_msg = f'Error loading sla_classes JSON, please check the configuration, the JSON is not valid JSON, exception="{str(e)}"'
            logger.error(error_msg)
            return {
                "payload": {
                    "response": f"{error_msg}",
                    "status": 500,
                },
                "status": 500,
            }

        # check requested class
        if not sla_policy_value in sla_classes:
            return {
                "payload": {
                    "response": f"The sla class {sla_policy_value} is not available in the configuration, available classes: {json.dumps(sla_classes, 0)}",
                    "status": 400,
                },
                "status": 400,
            }

        # Data collection
        collection_name = f"kv_trackme_{component}_sla_policies_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        try:
            record = collection.data.query(query=json.dumps(query_string))
            key = record[0].get("_key")

        except Exception as e:
            key = None

        # proceed

        # this policy exists already, it will be updated
        if key:
            # set an action_desc
            action_desc = "updated"

            # create a new list
            sla_class_new = []

            # add existing members
            for sla_class in record[0].get("sla_policy_value"):
                sla_class_new.append(sla_class)

            # add sla_class, if not in there already
            for sla_class in sla_policy_value:
                if sla_class not in sla_class_new:
                    sla_class_new.append(sla_class)

        else:
            # set an action_desc
            action_desc = "created"

        # proceed
        try:
            if key:
                # Update the record
                collection.data.update(
                    str(key),
                    json.dumps(
                        {
                            "sla_policy_id": sla_policy_id,
                            "sla_policy_value": sla_policy_value,
                            "sla_policy_regex": sla_policy_regex,
                            "mtime": time.time(),
                        }
                    ),
                )

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "update sla class policy",
                        str(sla_policy_id),
                        f"splk-{component}",
                        collection.data.query(query=json.dumps(query_string)),
                        "The sla class policy was updated successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

            else:
                # Insert the record
                collection.data.insert(
                    json.dumps(
                        {
                            "sla_policy_id": sla_policy_id,
                            "sla_policy_value": sla_policy_value,
                            "sla_policy_regex": sla_policy_regex,
                            "mtime": time.time(),
                        }
                    )
                )

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "add sla class policy",
                        str(sla_policy_id),
                        f"splk-{component}",
                        collection.data.query(query=json.dumps(query_string)),
                        "The sla class policy was added successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

            # render response
            response = {
                "action": "success",
                "action_desc": action_desc,
                "response": f'the sla class policy sla_policy_id="{sla_policy_id}" was {action_desc} successfully',
                "record": collection.data.query(query=json.dumps(query_string)),
            }

            return {"payload": response, "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Delete records from the collection
    def post_sla_policies_del(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        component = None
        sla_policy_id_list = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                    if component not in (
                        "dsm",
                        "dhm",
                        "mhm",
                        "wlk",
                        "flx",
                        "fqm",
                    ):
                        return {
                            "payload": {
                                "action": "failure",
                                "response": f"invalid component {component}, valid options are: dsm/dhm/mhm/wlk/flx/fqm",
                            },
                            "status": 400,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    sla_policy_id_list = resp_dict["sla_policy_id_list"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The sla_policy_id_list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                # if sla_policy_id_list is a string, turn into a list
                if isinstance(sla_policy_id_list, str):
                    sla_policy_id_list = sla_policy_id_list.split(",")

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint deletes sla policies, it requires a POST call with the following information:",
                "resource_desc": "Delete or ore more sla policies",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_sla_policies/write/sla_policies_del\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'sla_policy_id': 'linux_secure,linux_sec'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, valid values are: dsm/dhm/mhm/wlk/flx/fqm",
                        "sla_policy_id_list": "(required) Comma separated list of sla policies",
                        "update_comment": "(optional) Comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Data collection
        collection_name = f"kv_trackme_{component}_sla_policies_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # records summary
        records = []

        # loop
        for item in sla_policy_id_list:
            # Define the KV query
            query_string = {
                "sla_policy_id": item,
            }

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only
            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                # Remove and audit
                try:
                    # Remove the record
                    collection.data.delete(json.dumps({"_key": key}))

                    # increment counter
                    processed_count += 1
                    succcess_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "delete sla class policy",
                            str(item),
                            "all",
                            record,
                            "The lagging class was deleted successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "success",
                        "record": record,
                    }

                    records.append(result)

                    logger.info(json.dumps(result, indent=0))

                except Exception as e:
                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "failure",
                            "delete sla class policy",
                            str(item),
                            "all",
                            record,
                            str(e),
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "failure",
                        "record": record,
                        "exception": e,
                    }

                    # append to records
                    records.append(result)

                    # log
                    logger.error(json.dumps(result, indent=0))

            else:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                # audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "delete sla class policy",
                        str(item),
                        "all",
                        record,
                        "HTTP 404 NOT FOUND",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                result = {
                    "action": "delete",
                    "result": "failure",
                    "record": item,
                    "exception": "HTTP 404 NOT FOUND",
                }

                # append to records
                records.append(result)

                # log
                logger.error(json.dumps(result, indent=0))

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Update records
    def post_sla_policies_update(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        component = None
        records_list = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                    if component not in (
                        "dsm",
                        "dhm",
                        "mhm",
                        "wlk",
                        "flx",
                        "fqm",
                    ):
                        return {
                            "payload": {
                                "action": "failure",
                                "response": f"invalid component {component}, valid options are: dsm/dhm/mhm/wlk/flx/fqm",
                            },
                            "status": 400,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    records_list = resp_dict["records_list"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The records_list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    records_list = json.loads(records_list)
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The records_list is not a valid JSON",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint updates records, it requires a POST call with the following information:",
                "resource_desc": "Update or ore more sla policies",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_sla_policies/write/sla_policies_update\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'records_list': '<redacted_json_records>'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, valid values are: dsm/dhm/mhm/wlk/flx/fqm",
                        "records_list": "(required) JSON records to be updated",
                        "update_comment": "(option) Comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get trackmeconf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )["trackme_conf"]

        # Get SLA classes conf
        sla_classes = trackme_conf["sla"]["sla_classes"]

        # try loading the JSON
        try:
            sla_classes = json.loads(sla_classes)
        except Exception as e:
            error_msg = f'Error loading sla_classes JSON, please check the configuration, the JSON is not valid JSON, exception="{str(e)}"'
            logger.error(error_msg)
            return {
                "payload": {
                    "response": f"{error_msg}",
                    "status": 500,
                },
                "status": 500,
            }

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Data collection
        collection_name = f"kv_trackme_{component}_sla_policies_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # records summary
        records = []

        # debug
        logger.info(f'records_list="{json.dumps(records_list, indent=0)}"')

        # loop
        for item in records_list:
            # debug
            logger.info(f'item="{item}"')

            # check if we have the _key, otherwise search for it
            if item.get("_key"):
                key = item.get("_key")

            else:
                # Define the KV query
                query_string = {
                    "sla_policy_id": item.get("sla_policy_id"),
                }

                # Get the current record
                # Notes: the record is returned as an array, as we search for a specific record, we expect one record only
                try:
                    record = collection.data.query(query=json.dumps(query_string))
                    key = record[0].get("_key")

                except Exception as e:
                    key = None

            # Render result
            if key:
                # This record exists already

                # Store the record for audit purposes
                record = str(json.dumps(collection.data.query_by_id(key), indent=1))

                # Update and audit
                try:
                    # Update the record
                    sla_policy_value = item.get("sla_policy_value")

                    # if sla_policy_value is a string, make it lower case, if is a list, return an error as it is not accepted in this context
                    if isinstance(sla_policy_value, str):

                        if not len(sla_policy_value) > 0:
                            return {
                                "payload": {
                                    "response": "The sla_policy_value should not be empty",
                                    "status": 400,
                                },
                                "status": 400,
                            }

                        else:
                            sla_policy_value = sla_policy_value.lower()
                            if not sla_policy_value in sla_classes:
                                return {
                                    "payload": {
                                        "response": f"The SLA class {sla_policy_value} is not available currently in TrackMe's configuration, available classes: {json.dumps(sla_classes, 0)}",
                                        "status": 400,
                                    },
                                    "status": 400,
                                }

                    else:
                        return {
                            "payload": {
                                "response": "The sla_policy_value should be a string",
                                "status": 400,
                            },
                            "status": 400,
                        }

                    collection.data.update(
                        str(key),
                        json.dumps(
                            {
                                "sla_policy_id": item.get("sla_policy_id"),
                                "sla_policy_value": sla_policy_value,
                                "sla_policy_regex": item.get("sla_policy_regex"),
                                "mtime": time.time(),
                            }
                        ),
                    )

                    # increment counter
                    processed_count += 1
                    succcess_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "update sla policies",
                            str(item),
                            "dsm",
                            record,
                            "The lagging classs was updated successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "success",
                        "record": record,
                    }

                    records.append(result)

                    logger.info(json.dumps(result, indent=0))

                except Exception as e:
                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "failure",
                            "update sla policies",
                            str(item),
                            "dsm",
                            record,
                            str(e),
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "failure",
                        "record": record,
                        "exception": e,
                    }

                    # append to records
                    records.append(result)

                    # log
                    logger.error(json.dumps(result, indent=0))

            else:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                # audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "update sla policies",
                        str(item),
                        "dsm",
                        record,
                        "HTTP 404 NOT FOUND",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                result = {
                    "action": "delete",
                    "result": "failure",
                    "record": item,
                    "exception": "HTTP 404 NOT FOUND",
                }

                # append to records
                records.append(result)

                # log
                logger.error(json.dumps(result, indent=0))

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Simulate sla_class
    def post_sla_policies_simulate(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        component = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                    if component not in (
                        "dsm",
                        "dhm",
                        "mhm",
                        "wlk",
                        "flx",
                        "fqm",
                    ):
                        return {
                            "payload": {
                                "action": "failure",
                                "response": f"invalid component {component}, valid options are: dsm/dhm/mhm/wlk/flx/fqm",
                            },
                            "status": 400,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    regex_value = resp_dict["regex_value"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The argument regex_value is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    sla_class = resp_dict["sla_class"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The argument sla_class is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                # sla_class should be a string
                if isinstance(sla_class, str):
                    if not len(sla_class) > 0:
                        return {
                            "payload": {
                                "response": "The sla_class should not be empty",
                                "status": 400,
                            },
                            "status": 400,
                        }
                    else:
                        sla_class = sla_class.lower()

                else:
                    return {
                        "payload": {
                            "response": "The sla_class should be a string",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint simulates a sla class policy, it requires a POST call with the following information:",
                "resource_desc": "Simulates a sla class policy",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_sla_policies/write/sla_policies_simulate\" body=\"{'tenant_id': 'mytenant', 'regex_value': '^org_eu.*', 'sla_class': 'high'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "regex_value": "(required) The regex to be used for the simulation",
                        "sla_class": "(required) The sla_class value",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get trackmeconf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )["trackme_conf"]

        # Get SLA classes conf
        sla_classes = trackme_conf["sla"]["sla_classes"]

        # Check requested sla class
        if not sla_class in sla_classes:
            return {
                "payload": {
                    "response": f"The SLA class {sla_class} is not available currently in TrackMe's configuration, available classes: {json.dumps(sla_classes, 0)}",
                    "status": 400,
                },
                "status": 400,
            }

        # start
        main_start = time.time()

        #
        # sla policies
        #

        sla_policies_records = [
            {
                "sla_policy_regex": regex_value,
                "sla_policy_value": sla_class,
            }
        ]

        #
        # KV entities
        #

        # entities KV collection
        data_collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
        data_collection = service.kvstore[data_collection_name]

        # get records
        data_records, data_collection_keys, data_collection_dict = get_kv_collection(
            data_collection, data_collection_name
        )

        #
        # Handle policies
        #

        # A policy is composed by the following fields: sla_policy_id, sla_policy_regex, sla_policy_value
        # A record has various fields, in this context we care about: object (the value against the regex is applied) and sla_class_auto (a list to contain all matching sla_class values)

        # Counters and error list
        entities_failures_count = 0
        entities_exceptions_list = []
        entities_matched = []

        # only proceed if we have entities
        if len(data_records) > 0:

            # loop through records and apply policies
            for entity_record in data_records:

                entity_key = entity_record["_key"]
                entity_object = entity_record["object"]

                # apply policies
                if len(sla_policies_records) > 0:
                    for policy_record in sla_policies_records:

                        # check that the regex expression is a valid regex expression
                        # for the purpose of the simulation, we will however return a 200 with a message
                        try:
                            re.compile(regex_value)
                        except re.error:
                            req_summary = {
                                "entities_matched_count": len(entities_matched),
                                "result_summary": "The regex_value is not a valid regular expression, review this expression and try again.",
                                "regex_value": regex_value,
                                "regex_is_valid": "false",
                            }
                            return {"payload": req_summary, "status": 200}

                        try:
                            # apply regex
                            regex = policy_record["sla_policy_regex"]
                            value = policy_record["sla_policy_value"]

                            # make it lower case
                            if isinstance(value, str):
                                value = value.lower()

                            if re.match(regex, entity_record["object"]):
                                logger.info(
                                    f'tenant_id="{tenant_id}", object="{entity_object}", regex has matched this entity, regex="{regex}", value="{value}"'
                                )

                                # append to our list
                                if not entity_object in entities_matched:
                                    entities_matched.append(entity_object)

                            else:
                                logger.info(
                                    f'tenant_id="{tenant_id}", object="{entity_object}", regex has not matched this entity, regex="{regex}", value="{value}"'
                                )

                        except Exception as e:
                            logger.error(
                                f'context="exception", failed to apply policy, exception="{str(e)}"'
                            )

        # set action
        if entities_failures_count == 0:
            action = "success"
        else:
            action = "failure"

        # get run_time
        run_time = round((time.time() - main_start), 3)

        # create a result summary
        if len(entities_matched) > 0:
            result_summary = f"The regex has matched {len(entities_matched)} entities."
        else:
            result_summary = "The regex has not matched any entities, verify your inputs and try again."

        # request summary
        req_summary = {
            "kvstore_collection_entities_count": len(data_records),
            "entities_matched_count": len(entities_matched),
            "entities_matched": entities_matched,
            "result_summary": result_summary,
            "error_messages": entities_exceptions_list,
            "regex_is_valid": "true",
        }

        # render response
        if action == "success":
            logger.info(
                f'sla_class simulation operation has terminated, action="{action}", tenant_id="{tenant_id}", run_time="{run_time}"'
            )
            return {"payload": req_summary, "status": 200}

        else:
            logger.error(
                f'sla_class simulation operation has failed, action="{action}", tenant_id="{tenant_id}", req_summary="{json.dumps(req_summary, indent=2)}"'
            )
            return {"payload": req_summary, "status": 500}

    # Apply sla_class
    def post_sla_policies_apply(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        component = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                    if component not in (
                        "dsm",
                        "dhm",
                        "mhm",
                        "wlk",
                        "flx",
                        "fqm",
                    ):
                        return {
                            "payload": {
                                "action": "failure",
                                "response": f"invalid component {component}, valid options are: dsm/dhm/mhm/wlk/flx/fqm",
                            },
                            "status": 400,
                        }
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint applies sla policies, it requires a POST call with the following information:",
                "resource_desc": "Immediately apply and update sla policies",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_sla_policies/write/sla_policies_apply\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, valid values are: dsm/dhm/mhm/wlk/flx/fqm",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get trackmeconf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )["trackme_conf"]

        # Get SLA classes conf
        sla_classes = trackme_conf["sla"]["sla_classes"]

        # try loading the JSON
        try:
            sla_classes = json.loads(sla_classes)

        except Exception as e:
            return {
                "payload": {
                    "response": "The sla_classes configuration is not a valid JSON",
                    "status": 400,
                },
                "status": 400,
            }

        # start
        main_start = time.time()

        #
        # KV sla policies
        #

        # sla policies KV collection
        sla_policies_collection_name = (
            f"kv_trackme_{component}_sla_policies_tenant_{tenant_id}"
        )
        sla_policies_collection = service.kvstore[sla_policies_collection_name]

        # get records
        (
            sla_policies_records,
            sla_class_collection_keys,
            sla_class_collection_dict,
        ) = get_kv_collection(sla_policies_collection, sla_policies_collection_name)

        #
        # KV entities
        #

        # entities KV collection
        data_collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
        data_collection = service.kvstore[data_collection_name]

        # get records
        data_records, data_collection_keys, data_collection_dict = get_kv_collection(
            data_collection, data_collection_name
        )

        #
        # KV sla_class
        #

        # entities KV collection
        sla_class_collection_name = f"kv_trackme_{component}_sla_tenant_{tenant_id}"
        sla_class_collection = service.kvstore[sla_class_collection_name]

        # get records
        sla_class_records, sla_class_collection_keys, sla_class_collection_dict = (
            get_kv_collection(sla_class_collection, sla_class_collection_name)
        )

        #
        # Handle policies
        #

        # A policy is composed by the following fields: sla_policy_id, sla_policy_regex, sla_policy_value
        # A record has various fields, in this context we care about: object (the value against the regex is applied) and sla_class_auto (a list to contain all matching sla_class values)

        # Counters and error list
        entities_updated_count = 0
        entities_failures_count = 0
        entities_deleted_count = 0
        entities_exceptions_list = []

        updated_records = []  # we will store the updated records here

        # each sla class in sla_classes has a rank field which defines its numerical value, loop through the records and apply the sla_class
        sla_class_rank_dict = {}
        for sla_class in sla_classes:
            sla_class_rank = sla_classes[sla_class].get("rank", 0)
            sla_class_rank_dict[sla_class] = sla_class_rank

        # only proceed if we have entities
        if len(data_records) > 0:

            # loop through records and apply policies
            for entity_record in data_records:

                entity_key = entity_record["_key"]
                entity_object = entity_record["object"]
                sla_class = None
                sla_class_reason = None

                # apply policies
                if len(sla_policies_records) > 0:

                    # Initialize sla_class and reason as None
                    sla_class = None
                    sla_class_reason = None
                    policy_matched = False

                    for policy_record in sla_policies_records:
                        try:
                            regex = policy_record["sla_policy_regex"]
                            value = policy_record["sla_policy_value"].lower()

                            if re.match(regex, entity_record["object"]):
                                logger.info(
                                    f'tenant_id="{tenant_id}", object="{entity_object}", policy="{policy_record["sla_policy_id"]}" has matched this entity, regex="{regex}", value="{value}"'
                                )

                                # Check if a sla_class was already matched
                                if policy_matched:
                                    # Get the numerical value of the matched sla_class
                                    sla_class_matched_num = sla_class_rank_dict.get(
                                        sla_class, -1
                                    )  # Get -1 if none set

                                    # Get the numerical value of the current sla_class
                                    sla_class_current_num = sla_class_rank_dict.get(
                                        value, -1
                                    )

                                    # Update if the new sla_class value is higher
                                    if sla_class_current_num > sla_class_matched_num:
                                        sla_class = value
                                        sla_class_reason = policy_record[
                                            "sla_policy_id"
                                        ]
                                else:
                                    sla_class = value
                                    sla_class_reason = policy_record["sla_policy_id"]
                                    policy_matched = True

                        except Exception as e:
                            logger.error(
                                f'context="exception", failed to apply policy, exception="{str(e)}"'
                            )

                # Add if matched
                if policy_matched:

                    # add to updated_records
                    updated_records.append(
                        {
                            "_key": entity_key,
                            "object": entity_object,
                            "sla_class": sla_class,
                            "sla_class_reason": sla_class_reason,
                            "mtime": time.time(),
                        }
                    )

            # Update records in batches
            chunks = [
                updated_records[i : i + 500]
                for i in range(0, len(updated_records), 500)
            ]
            for chunk in chunks:
                try:
                    sla_class_collection.data.batch_save(*chunk)
                    entities_updated_count += len(chunk)
                except Exception as e:
                    msg = f'KVstore batch save failed with exception="{str(e)}"'
                    logger.error(msg)
                    entities_exceptions_list.append(msg)

            # for record in sla_class_records, if the key of the record does not exist in data_collection_keys, delete it
            for record in sla_class_records:
                if record["_key"] not in data_collection_keys:
                    try:
                        sla_class_collection.data.delete(
                            json.dumps({"_key": record["_key"]})
                        )
                        # counter deleted
                        entities_deleted_count += 1

                    except Exception as e:
                        # counter failure
                        entities_failures_count += 1
                        msg = f'KVstore delete failed for key="{record["_key"]}", exception="{str(e)}"'
                        logger.error(msg)
                        entities_exceptions_list.append(msg)

            # refresh knowledge
            sla_class_records, sla_class_collection_keys, sla_class_collection_dict = (
                get_kv_collection(sla_class_collection, sla_class_collection_name)
            )

            # immediately apply sla_class on the data collection, loop through sla_class_records and apply sla_class to data_records
            updated_data_records = []  # we will store the updated records here

            for sla_class_record in sla_class_records:
                for entity_record in data_records:
                    if entity_record["_key"] == sla_class_record["_key"]:

                        # get the current sla_class value, we will only update the KVstore record if the final sla_class field has changed
                        current_sla_class = entity_record.get("sla_class", None)

                        entity_record["sla_class"] = sla_class_record["sla_class"]

                        # compare current_sla_class (CSV string) and sla_class (CSV string), if they are different, add the record to the updated_data_records list
                        if current_sla_class:
                            if current_sla_class != sla_class_record["sla_class"]:
                                entity_record["mtime"] = time.time()
                                updated_data_records.append(entity_record)

                        else:  # we have no current sla_class, we need to update the record
                            entity_record["mtime"] = time.time()
                            updated_data_records.append(entity_record)

        # set action
        if entities_failures_count == 0:
            action = "success"
        else:
            action = "failure"

        # get run_time
        run_time = round((time.time() - main_start), 3)

        # request summary
        req_summary = {
            "tenant_id": tenant_id,
            "action": action,
            "run_time": run_time,
            "kvstore_lookup_collection": f"trackme_{component}_sla_tenant_{tenant_id}",
            "sla_policies_no_records": len(sla_policies_records),
            "kvstore_collection_entities_count": len(data_records),
            "entities_updated_count": entities_updated_count,
            "entities_failures_count": entities_failures_count,
            "entities_deleted_count": entities_deleted_count,
            "error_messages": entities_exceptions_list,
        }

        # render response
        if action == "success":
            logger.info(
                f'sla_class apply operation has terminated, action="{action}", tenant_id="{tenant_id}", run_time="{run_time}"'
            )
            return {"payload": req_summary, "status": 200}

        else:
            logger.error(
                f'sla_class apply operation has failed, action="{action}", tenant_id="{tenant_id}", req_summary="{json.dumps(req_summary, indent=2)}"'
            )
            return {"payload": req_summary, "status": 500}
