#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_smart_status.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2022-2026, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

import os, sys
import json
import time
import requests

splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_smart_status",
    "trackme_rest_api_splk_smart_status.log",
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# TrackMe REST handler
import trackme_rest_handler

# TrackMe libs
from trackme_libs import trackme_getloglevel

# TrackMe SmartStatus libs
from trackme_libs_smartstatus import (
    smartstatus_flipping_correlation,
    smartstatus_investigations_uc_dsm_future,
    smartstatus_investigations_uc_dhm_future,
    smartstatus_investigations_uc_dsm_latency,
    smartstatus_investigations_uc_dhm_latency,
    smartstatus_investigations_uc_dsm_delay,
    smartstatus_investigations_uc_dhm_delay,
    smartstatus_investigations_uc_hosts_dcount,
    smartstatus_investigations_uc_ml_outliers,
    smartstatus_investigations_uc_events_format_recognition,
    smartstatus_investigations_uc_wlk_execution_errors,
    smartstatus_investigations_uc_wlk_skipping,
    smartstatus_investigations_uc_wlk_orphan,
    smartstatus_investigations_uc_wlk_delayed,
)

# TrackMe splk-feeds libs
from trackme_libs_splk_feeds import (
    splk_dsm_return_entity_info,
    splk_dsm_return_elastic_info,
    splk_dhm_return_entity_info,
)

# Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkSmartStatus_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkSmartStatus_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_smart_status(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_smart_status",
            "resource_group_desc": "Endpoints related to the Smart Status features (automated investigations)",
        }

        return {"payload": response, "status": 200}

    # Get smart status for splk-dsm
    def post_ds_smart_status(self, request_info, **kwargs):
        # define
        tenant_id = None
        object_value = None

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                object_value = resp_dict["object"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoints runs the smart status for a given data source, it requires a POST call with the following options:",
                "resource_desc": "Run Smart Status for a given entity",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_smart_status/ds_smart_status\" body=\"{'tenant_id': 'mytenant', 'object': 'netscreen:netscreen:firewall'}\"",
                "options": [
                    {
                        "tenant_id": "the tenant identifier",
                        "object": "name of the data source",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Define the KV query
        query_string = {
            "object": object_value,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get virtual tenant preferences

        # Proceed
        vtenant_prefs = {}
        indexed_constraint = ""
        try:
            response = requests.post(
                f"{request_info.server_rest_uri}/services/trackme/v2/configuration/components",
                headers={
                    "Authorization": f"Splunk {request_info.system_authtoken}",
                    "Content-Type": "application/json",
                },
                data=json.dumps({"tenant_id": tenant_id}),
                verify=False,
                timeout=600,
            )

            if response.status_code not in (200, 201, 204):
                msg = f'get component has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                logger.error(msg)
                vtenant_prefs = {}

            else:
                vtenant_prefs = response.json()
                indexed_constraint = vtenant_prefs.get("indexed_constraint", "")

        except Exception as e:
            msg = f'get component has failed, exception="{str(e)}"'
            logger.error(msg)
            vtenant_prefs = {}

        # Get the object record
        try:
            collection_name = f"kv_trackme_dsm_tenant_{tenant_id}"
            collection = service.kvstore[collection_name]

            records = collection.data.query(query=json.dumps(query_string))
            record = records[0]
            key = record.get("_key")

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'object="{object_value}" not found',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 404}

        #
        # object is known, proceed
        #

        # set the initial empty smartstatus_response
        smartstatus_response = {}

        # get anomaly_reason
        anomaly_reason = record.get("anomaly_reason")

        # turn into list from pipe delimited string if not already a list
        if not isinstance(anomaly_reason, list):
            anomaly_reason = anomaly_reason.split("|")

        # retrieve flipping correlation
        try:
            flipping_correlation = smartstatus_flipping_correlation(
                request_info, tenant_id, "splk-dsm", object_value
            )
        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered with smartstatus_flipping_correlation, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

        # Get future_tolerance
        try:
            future_tolerance = int(record.get("future_tolerance", -600))
        except Exception as e:
            future_tolerance = -600

        # Get entity info
        elastic_info = splk_dsm_return_elastic_info(
            request_info.session_key,
            request_info.server_rest_port,
            tenant_id,
            object_value,
        )
        entity_info = splk_dsm_return_entity_info(record)

        # Check if is an Elastic Source
        is_elastic = elastic_info.get("is_elastic")
        if is_elastic == 1:
            smartstatus_response["elastic_info"] = elastic_info
        else:
            smartstatus_response["entity_info"] = entity_info

        # Add indexed_constraintto entity_info
        entity_info["indexed_constraint"] = indexed_constraint

        # log debug
        logger.debug(
            f'function ds_smart_status, elastic_info="{json.dumps(elastic_info, indent=2)}"'
        )
        logger.debug(
            f'function ds_smart_status, entity_info="{json.dumps(entity_info, indent=2)}"'
        )

        # Get entity account and search_mode
        account = None
        search_mode = None

        if is_elastic == 1:
            account = elastic_info.get("account")
            search_mode = elastic_info.get("search_mode")
        else:
            account = entity_info.get("account")
            search_mode = entity_info.get("search_mode")

        # if is elastic, update the elastic info to the entity info dict
        if is_elastic == 1:
            entity_info["account"] = account
            entity_info["search_mode"] = search_mode
            entity_info["is_elastic"] = 1
            entity_info["search_constraint"] = elastic_info.get("search_constraint")
            entity_info["elastic_search_mode"] = elastic_info.get("elastic_search_mode")
            entity_info.pop("breakby_key", None)
            entity_info.pop("breakby_value", None)
            entity_info.pop("breakby_statement", None)
        else:
            entity_info["is_elastic"] = 0

        # log debug
        logger.debug(
            f'function ds_smart_status, entity_info="{json.dumps(entity_info, indent=2)}"'
        )

        # add to response
        smartstatus_response["account"] = account
        smartstatus_response["search_mode"] = search_mode

        #################################
        # Greem: happiness is in da house
        #################################

        # if anomaly_reason (which is a list) only contains "none", then there are no anomalies
        if anomaly_reason == ["none"]:
            smartstatus_response["_time"] = time.time()
            smartstatus_response["tenant_id"] = tenant_id
            smartstatus_response["object_category"] = "splk-dsm"
            smartstatus_response["object"] = object_value
            smartstatus_response["alias"] = record.get("alias")
            smartstatus_response["object_state"] = record.get("object_state")
            smartstatus_response["anomaly_reason"] = anomaly_reason
            smartstatus_response["flipping_correlations"] = flipping_correlation
            smartstatus_response["smart_result"] = (
                "The entity is currently in a normal state, therefore further investigations are not required at this stage."
            )

            return {"payload": smartstatus_response, "status": 200}

        ################################################
        # use case investigations: data is in the future
        ################################################

        if "future_over_tolerance" in anomaly_reason:
            logger.info(
                f'performing investigations use case: data in the future, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case search investigates and extracts main information related to the detection of data in the future over the tolerance"

            try:
                uc_results = smartstatus_investigations_uc_dsm_future(
                    request_info, future_tolerance, record, entity_info
                )
                # append to response

                if entity_info.get("search_mode") in ("tstats", "raw", "from"):
                    use_case_desc = "The use case search investigates and extracts the top 100 hosts and 100 sources generating data in the future over the tolerance"
                    search1_results_label = "results_hosts_future"
                    search1_info_label = "search_info_hosts_future"
                    search2_results_label = "results_sources_future"
                    search2_info_label = "search_info_sources_future"
                    search3_results_label = "results_sample_events_future"
                    search3_info_label = "search_info_sample_events_future"

                elif entity_info.get("search_mode") in ("mstats"):
                    use_case_desc = "The use case search investigates and extracts the top 100 hosts and 100 metric categories generating metrics in the future over the tolerance"
                    search1_results_label = "results_hosts_future"
                    search1_info_label = "search_info_hosts_future"
                    search2_results_label = "results_metric_categories_future"
                    search2_info_label = "search_info_metric_categories_future"
                    search3_results_label = "results_sample_metrics_future"
                    search3_info_label = "search_info_sample_metrics_future"

                smartstatus_response["smart_investigations_uc_future"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    search2_results_label: uc_results.get("results_2"),
                    search2_info_label: uc_results.get("search_info_2"),
                    search3_results_label: uc_results.get("results_3"),
                    search3_info_label: uc_results.get("search_info_3"),
                    "smart_function": "smartstatus_investigations_uc_dsm_future",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_dsm_future, exception="{str(e)}"'
                )
                smartstatus_response["smart_investigations_uc_future"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_dsm_future CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_dsm_future",
                    "exception": str(e),
                }

        #########################################
        # use case investigations: entity latency
        #########################################

        if "lag_threshold_breached" in anomaly_reason:
            logger.info(
                f'performing investigations use case: latency, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates and generates main indexing latency statistics for the entity"

            try:
                uc_results = smartstatus_investigations_uc_dsm_latency(
                    request_info, record, entity_info
                )
                # append to response

                if entity_info.get("search_mode") in ("tstats", "raw", "from"):
                    use_case_desc = "The use case investigates and generates main indexing latency statistics for the entity"
                    search1_results_label = "results_latency"
                    search1_info_label = "search_info_latency"
                    search2_results_label = "results_latency_overtime"
                    search2_results_label_sparkline = (
                        "results_latency_overtime_sparkline"
                    )
                    search2_info_label = "search_info_latency_overtime"
                    search3_results_label = "results_sample_events"
                    search3_info_label = "search_info_sample_events"

                smartstatus_response["smartstatus_investigations_uc_dsm_latency"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    search2_results_label: uc_results.get("results_2"),
                    search2_results_label_sparkline: uc_results.get(
                        "results_2_sparkline"
                    ),
                    search2_info_label: uc_results.get("search_info_2"),
                    search3_results_label: uc_results.get("results_3"),
                    search3_info_label: uc_results.get("search_info_3"),
                    "smart_function": "smartstatus_investigations_uc_dsm_latency",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_dsm_latency, exception="{str(e)}"'
                )
                smartstatus_response["smart_investigations_uc_latency"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_dsm_latency CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_dsm_latency",
                    "exception": str(e),
                }

        #######################################
        # use case investigations: entity delay
        #######################################

        if "delay_threshold_breached" in anomaly_reason:
            logger.info(
                f'performing investigations use case: delay, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates and generates delay related statistics for the entity"

            try:
                uc_results = smartstatus_investigations_uc_dsm_delay(
                    request_info, record, entity_info
                )
                # append to response

                if entity_info.get("search_mode") in ("tstats", "raw", "from"):
                    use_case_desc = "The use case investigates data flow interruption (delay breached) for the entity"
                    search1_results_label = "results_delay"
                    search1_info_label = "search_info_delay"
                    search2_results_label = "results_delay_metric_overtime"
                    search2_results_label_sparkline = (
                        "results_delay_metric_overtime_sparkline"
                    )
                    search2_info_label = "search_info_delay_metric_overtime"
                    search3_results_label = "results_flip"
                    search3_info_label = "search_info_flip"

                smartstatus_response["smartstatus_investigations_uc_dsm_delay"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    search2_results_label: uc_results.get("results_2"),
                    search2_results_label_sparkline: uc_results.get(
                        "results_2_sparkline"
                    ),
                    search2_info_label: uc_results.get("search_info_2"),
                    search3_results_label: uc_results.get("results_3"),
                    search3_info_label: uc_results.get("search_info_3"),
                    "smart_function": "smartstatus_investigations_uc_dsm_delay",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_dsm_delay, exception="{str(e)}"'
                )
                smartstatus_response["smart_investigations_uc_delay"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_dsm_delay CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_dsm_delay",
                    "exception": str(e),
                }

        ########################################################
        # use case investigations: distinct host count threshold
        ########################################################

        if "min_hosts_dcount" in anomaly_reason:
            logger.info(
                f'performing investigations use case: distinct host count threshold, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates and generates statistics related to the minimal distinct count of hosts for the entity"

            try:
                uc_results = smartstatus_investigations_uc_hosts_dcount(
                    request_info, record, entity_info
                )
                # append to response

                if entity_info.get("search_mode") in ("tstats", "raw", "from"):
                    use_case_desc = "The use case investigates minimal hosts distinct count threshold breached for the entity"
                    search1_results_label = "results_dcount_hosts"
                    search1_info_label = "search_info_dcount_hosts"
                    search2_results_label = "results_dcount_hosts_metric_overtime"
                    search2_results_label_sparkline = (
                        "results_dcount_hosts_metric_overtime_sparkline"
                    )
                    search2_info_label = "search_info_dcount_hosts_metric_overtime"
                    search3_results_label = "results_flip"
                    search3_info_label = "search_info_flip"

                smartstatus_response["smartstatus_investigations_uc_hosts_dcount"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    search2_results_label: uc_results.get("results_2"),
                    search2_results_label_sparkline: uc_results.get(
                        "results_2_sparkline"
                    ),
                    search2_info_label: uc_results.get("search_info_2"),
                    search3_results_label: uc_results.get("results_3"),
                    search3_info_label: uc_results.get("search_info_3"),
                    "smart_function": "smartstatus_investigations_uc_hosts_dcount",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_hosts_dcount, exception="{str(e)}"'
                )
                smartstatus_response["smartstatus_investigations_uc_hosts_dcount"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_hosts_dcount CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_hosts_dcount",
                    "exception": str(e),
                }

        ##############################################################
        # use case investigations: Machine Learning Outliers Detection
        ##############################################################

        if "ml_outliers_detection" in anomaly_reason:
            logger.info(
                f'performing investigations use case: Machine Learning Outliers, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates ML Outliers for this object by executing the ML monitor process for enabled models"

            try:
                uc_results = smartstatus_investigations_uc_ml_outliers(
                    request_info, record, "dsm"
                )
                # append to response
                use_case_desc = (
                    "The use case investigates ML Outliers detection for the entity"
                )
                search1_results_label = "results_ml_outliers"
                search1_info_label = "search_info_ml_outliers"

                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_ml_outliers, exception="{str(e)}"'
                )
                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_ml_outliers CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                    "exception": str(e),
                }

        ####################################################################
        # use case investigations: Data sampling & events format recognition
        ####################################################################

        if "data_sampling_anomaly" in anomaly_reason:
            logger.info(
                f'performing investigations use case: Data sampling & events format recognition, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates Data sampling for the entity"

            try:
                uc_results = smartstatus_investigations_uc_events_format_recognition(
                    request_info, record, entity_info
                )

                # append to response
                use_case_desc = "The use case investigates Data sampling for the entity"
                search1_results_label = "results_data_sampling"
                search1_info_label = "search_info_data_sampling"

                smartstatus_response[
                    "smartstatus_investigations_uc_events_format_recognition"
                ] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_events_format_recognition",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_events_format_recognition, exception="{str(e)}"'
                )
                smartstatus_response[
                    "smartstatus_investigations_uc_events_format_recognition"
                ] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_events_format_recognition CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_events_format_recognition",
                    "exception": str(e),
                }

        ##########
        # response
        ##########

        smartstatus_response["_time"] = time.time()
        smartstatus_response["tenant_id"] = tenant_id
        smartstatus_response["object_category"] = "splk-dsm"
        smartstatus_response["object"] = object_value
        smartstatus_response["alias"] = record.get("alias")
        smartstatus_response["object_state"] = record.get("object_state")
        smartstatus_response["anomaly_reason"] = anomaly_reason
        smartstatus_response["flipping_correlations"] = flipping_correlation

        return {"payload": smartstatus_response, "status": 200}

    # Get smart status for splk-dhm
    def post_dh_smart_status(self, request_info, **kwargs):
        # define
        tenant_id = None
        object_value = None

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                object_value = resp_dict["object"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoints runs the smart status for a given data source, it requires a POST call with the following options:",
                "resource_desc": "Run Smart Status for a given entity",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_smart_status/dh_smart_status\" body=\"{'tenant_id': 'mytenant', 'object': 'remote|account:lab|firewall.pan.amer.design.node1'}\"",
                "options": [
                    {
                        "tenant_id": "the tenant identifier",
                        "object": "name of the data source",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Define the KV query
        query_string = {
            "object": object_value,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get virtual tenant preferences

        # Proceed
        vtenant_prefs = {}
        indexed_constraint = ""
        try:
            response = requests.post(
                f"{request_info.server_rest_uri}/services/trackme/v2/configuration/components",
                headers={
                    "Authorization": f"Splunk {request_info.system_authtoken}",
                    "Content-Type": "application/json",
                },
                data=json.dumps({"tenant_id": tenant_id}),
                verify=False,
                timeout=600,
            )

            if response.status_code not in (200, 201, 204):
                msg = f'get component has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                logger.error(msg)
                vtenant_prefs = {}

            else:
                vtenant_prefs = response.json()
                indexed_constraint = vtenant_prefs.get("indexed_constraint", "")

        except Exception as e:
            msg = f'get component has failed, exception="{str(e)}"'
            logger.error(msg)
            vtenant_prefs = {}

        # Get the object record
        try:
            collection_name = "kv_" + "trackme_dhm_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            records = collection.data.query(query=json.dumps(query_string))
            record = records[0]
            key = record.get("_key")

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'object="{object_value}" not found',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 404}

        #
        # object is known, proceed
        #

        # Get future_tolerance
        try:
            future_tolerance = int(record.get("future_tolerance", -600))
        except Exception as e:
            future_tolerance = -600

        # Get entity info
        entity_info = splk_dhm_return_entity_info(record)

        # Add indexed_constraintto entity_info
        entity_info["indexed_constraint"] = indexed_constraint

        # set the initial empty smartstatus_response
        smartstatus_response = {}

        # get anomaly_reason
        anomaly_reason = record.get("anomaly_reason")

        # turn into list from pipe delimited string if not already a list
        if not isinstance(anomaly_reason, list):
            anomaly_reason = anomaly_reason.split("|")

        # retrieve flipping correlation
        try:
            flipping_correlation = smartstatus_flipping_correlation(
                request_info, tenant_id, "splk-dhm", object_value
            )
        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered with smartstatus_flipping_correlation, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

        #################################
        # Greem: happiness is in da house
        #################################

        # if anomaly_reason (which is a list) only contains "none", then there are no anomalies
        if anomaly_reason == ["none"]:
            smartstatus_response["_time"] = time.time()
            smartstatus_response["tenant_id"] = tenant_id
            smartstatus_response["object_category"] = "splk-dhm"
            smartstatus_response["object"] = object_value
            smartstatus_response["alias"] = record.get("alias")
            smartstatus_response["object_state"] = record.get("object_state")
            smartstatus_response["anomaly_reason"] = anomaly_reason
            smartstatus_response["flipping_correlations"] = flipping_correlation
            smartstatus_response["smart_result"] = (
                "The entity is currently in a normal state, therefore further investigations are not required at this stage."
            )

            return {"payload": smartstatus_response, "status": 200}

        ################################################
        # use case investigations: data is in the future
        ################################################

        if "future_over_tolerance" in anomaly_reason:
            logger.info(
                f'performing investigations use case: data in the future, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case search investigates and extracts main information related to the detection of data in the future over the tolerance"

            try:
                uc_results = smartstatus_investigations_uc_dhm_future(
                    request_info, future_tolerance, record, entity_info
                )
                # append to response

                use_case_desc = "The use case search investigates and extracts the top 100 sourcetypoes and 100 sources generating data in the future over the tolerance"
                search1_results_label = "results_sourcetypes_future"
                search1_info_label = "search_info_sourcetypes_future"
                search2_results_label = "results_sources_future"
                search2_info_label = "search_info_sources_future"
                search3_results_label = "results_sample_events_future"
                search3_info_label = "search_info_sample_events_future"

                smartstatus_response["smart_investigations_uc_future"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    search2_results_label: uc_results.get("results_2"),
                    search2_info_label: uc_results.get("search_info_2"),
                    search3_results_label: uc_results.get("results_3"),
                    search3_info_label: uc_results.get("search_info_3"),
                    "smart_function": "smartstatus_investigations_uc_dhm_future",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_dhm_future, exception="{str(e)}"'
                )
                smartstatus_response["smart_investigations_uc_future"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_dhm_future CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_dhm_future",
                    "exception": str(e),
                }

        #########################################
        # use case investigations: entity latency
        #########################################

        if "lag_threshold_breached" in anomaly_reason:
            logger.info(
                f'performing investigations use case: latency, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates and generates main indexing latency statistics for the entity"

            try:
                uc_results = smartstatus_investigations_uc_dhm_latency(
                    request_info, record, entity_info
                )
                # append to response

                if entity_info.get("search_mode") in ("tstats", "raw", "from"):
                    use_case_desc = "The use case investigates and generates main indexing latency statistics for the entity"
                    search1_results_label = "results_latency"
                    search1_info_label = "search_info_latency"
                    search2_results_label = "results_latency_overtime"
                    search2_results_label_sparkline = (
                        "results_latency_overtime_sparkline"
                    )
                    search2_info_label = "search_info_latency_overtime"
                    search3_results_label = "results_sample_events"
                    search3_info_label = "search_info_sample_events"

                smartstatus_response["smartstatus_investigations_uc_dhm_latency"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    search2_results_label: uc_results.get("results_2"),
                    search2_results_label_sparkline: uc_results.get(
                        "results_2_sparkline"
                    ),
                    search2_info_label: uc_results.get("search_info_2"),
                    search3_results_label: uc_results.get("results_3"),
                    search3_info_label: uc_results.get("search_info_3"),
                    "smart_function": "smartstatus_investigations_uc_dhm_latency",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_dhm_latency, exception="{str(e)}"'
                )
                smartstatus_response["smart_investigations_uc_latency"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_dsm_latency CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_dhm_latency",
                    "exception": str(e),
                }

        #######################################
        # use case investigations: entity delay
        #######################################

        if "delay_threshold_breached" in anomaly_reason:
            logger.info(
                f'performing investigations use case: delay, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates and generates delay related statistics for the entity"

            try:
                uc_results = smartstatus_investigations_uc_dhm_delay(
                    request_info, record, entity_info
                )
                # append to response

                if entity_info.get("search_mode") in ("tstats", "raw", "from"):
                    use_case_desc = "The use case investigates data flow interruption (delay breached) for the entity"
                    search1_results_label = "results_delay"
                    search1_info_label = "search_info_delay"
                    search2_results_label = "results_delay_metric_overtime"
                    search2_results_label_sparkline = (
                        "results_delay_metric_overtime_sparkline"
                    )
                    search2_info_label = "search_info_delay_metric_overtime"
                    search3_results_label = "results_flip"
                    search3_info_label = "search_info_flip"

                smartstatus_response["smartstatus_investigations_uc_dhm_delay"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    search2_results_label: uc_results.get("results_2"),
                    search2_results_label_sparkline: uc_results.get(
                        "results_2_sparkline"
                    ),
                    search2_info_label: uc_results.get("search_info_2"),
                    search3_results_label: uc_results.get("results_3"),
                    search3_info_label: uc_results.get("search_info_3"),
                    "smart_function": "smartstatus_investigations_uc_dhm_delay",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_dhm_delay, exception="{str(e)}"'
                )
                smartstatus_response["smart_investigations_uc_delay"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_dhm_delay CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_dhm_delay",
                    "exception": str(e),
                }

        ##############################################################
        # use case investigations: Machine Learning Outliers Detection
        ##############################################################

        if "ml_outliers_detection" in anomaly_reason:
            logger.info(
                f'performing investigations use case: Machine Learning Outliers, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates ML Outliers for this object by executing the ML monitor process for enabled models"

            try:
                uc_results = smartstatus_investigations_uc_ml_outliers(
                    request_info, record, "dhm"
                )
                # append to response
                use_case_desc = (
                    "The use case investigates ML Outliers detection for the entity"
                )
                search1_results_label = "results_ml_outliers"
                search1_info_label = "search_info_ml_outliers"

                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_ml_outliers, exception="{str(e)}"'
                )
                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_ml_outliers CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                    "exception": str(e),
                }

        ##########
        # response
        ##########

        smartstatus_response["_time"] = time.time()
        smartstatus_response["tenant_id"] = tenant_id
        smartstatus_response["object_category"] = "splk-dhm"
        smartstatus_response["object"] = object_value
        smartstatus_response["alias"] = record.get("alias")
        smartstatus_response["object_state"] = record.get("object_state")
        smartstatus_response["anomaly_reason"] = anomaly_reason
        smartstatus_response["flipping_correlations"] = flipping_correlation

        return {"payload": smartstatus_response, "status": 200}

    # Get smart status for splk-mhm
    def post_mh_smart_status(self, request_info, **kwargs):
        # define
        tenant_id = None
        object_value = None

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                object_value = resp_dict["object"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoints runs the smart status for a given data source, it requires a POST call with the following options:",
                "resource_desc": "Run Smart Status for a given entity",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_smart_status/mh_smart_status\" body=\"{'tenant_id': 'mytenant', 'object': 'key:env|splunk'}\"",
                "options": [
                    {
                        "tenant_id": "the tenant identifier",
                        "object": "name of the data source",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Define the KV query
        query_string = {
            "object": object_value,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get the object record
        try:
            collection_name = "kv_" + "trackme_mhm_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            records = collection.data.query(query=json.dumps(query_string))
            record = records[0]
            key = record.get("_key")

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'object="{object_value}" not found',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 404}

        #
        # object is known, proceed
        #

        # set the initial empty smartstatus_response
        smartstatus_response = {}

        # get anomaly_reason
        anomaly_reason = record.get("anomaly_reason")

        # turn into list from pipe delimited string if not already a list
        if not isinstance(anomaly_reason, list):
            anomaly_reason = anomaly_reason.split("|")

        # retrieve flipping correlation
        try:
            flipping_correlation = smartstatus_flipping_correlation(
                request_info, tenant_id, "splk-dsm", object_value
            )
        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered with smartstatus_flipping_correlation, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

        ##########
        # response
        ##########

        smartstatus_response["_time"] = time.time()
        smartstatus_response["tenant_id"] = tenant_id
        smartstatus_response["object_category"] = "splk-mhm"
        smartstatus_response["object"] = object_value
        smartstatus_response["alias"] = record.get("alias")
        smartstatus_response["object_state"] = record.get("object_state")
        smartstatus_response["anomaly_reason"] = anomaly_reason
        smartstatus_response["flipping_correlations"] = flipping_correlation

        return {"payload": smartstatus_response, "status": 200}

    # Get smart status for splk-flx
    def post_flx_smart_status(self, request_info, **kwargs):
        # define
        tenant_id = None
        object_value = None

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                object_value = resp_dict["object"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoints runs the smart status for a given data source, it requires a POST call with the following options:",
                "resource_desc": "Run Smart Status for a given entity",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_smart_status/flx_smart_status\" body=\"{'tenant_id': 'mytenant', 'object': 'Okta:Splunk_TA_okta_identity_cloud:okta_logs'}\"",
                "options": [
                    {
                        "tenant_id": "the tenant identifier",
                        "object": "name of the data source",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Define the KV query
        query_string = {
            "object": object_value,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get the object record
        try:
            collection_name = "kv_" + "trackme_flx_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            records = collection.data.query(query=json.dumps(query_string))
            record = records[0]
            key = record.get("_key")

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'object="{object_value}" not found',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 404}

        #
        # object is known, proceed
        #

        # set the initial empty smartstatus_response
        smartstatus_response = {}

        # get anomaly_reason
        anomaly_reason = record.get("anomaly_reason")

        # turn into list from pipe delimited string if not already a list
        if not isinstance(anomaly_reason, list):
            anomaly_reason = anomaly_reason.split("|")

        # retrieve flipping correlation
        try:
            flipping_correlation = smartstatus_flipping_correlation(
                request_info, tenant_id, "splk-flx", object_value
            )
        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered with smartstatus_flipping_correlation, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

        #################################
        # Greem: happiness is in da house
        #################################

        # if anomaly_reason (which is a list) only contains "none", then there are no anomalies
        if anomaly_reason == ["none"]:
            smartstatus_response["_time"] = time.time()
            smartstatus_response["tenant_id"] = tenant_id
            smartstatus_response["object_category"] = "splk-flx"
            smartstatus_response["object"] = object_value
            smartstatus_response["alias"] = record.get("alias")
            smartstatus_response["object_state"] = record.get("object_state")
            smartstatus_response["anomaly_reason"] = anomaly_reason
            smartstatus_response["flipping_correlations"] = flipping_correlation
            smartstatus_response["smart_result"] = (
                "The entity is currently in a normal state, therefore further investigations are not required at this stage."
            )

            return {"payload": smartstatus_response, "status": 200}

        ##############################################################
        # use case investigations: Machine Learning Outliers Detection
        ##############################################################

        if "ml_outliers_detection" in anomaly_reason:
            logger.info(
                f'performing investigations use case: Machine Learning Outliers, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates ML Outliers for this object by executing the ML monitor process for enabled models"

            try:
                uc_results = smartstatus_investigations_uc_ml_outliers(
                    request_info, record, "flx"
                )
                # append to response
                use_case_desc = (
                    "The use case investigates ML Outliers detection for the entity"
                )
                search1_results_label = "results_ml_outliers"
                search1_info_label = "search_info_ml_outliers"

                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_ml_outliers, exception="{str(e)}"'
                )
                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_ml_outliers CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                    "exception": str(e),
                }

        ##########
        # response
        ##########

        smartstatus_response["_time"] = time.time()
        smartstatus_response["tenant_id"] = tenant_id
        smartstatus_response["object_category"] = "splk-flx"
        smartstatus_response["object"] = object_value
        smartstatus_response["alias"] = record.get("alias")
        smartstatus_response["object_state"] = record.get("object_state")
        smartstatus_response["anomaly_reason"] = anomaly_reason
        smartstatus_response["flipping_correlations"] = flipping_correlation

        return {"payload": smartstatus_response, "status": 200}

    # Get smart status for splk-fqm
    def post_fqm_smart_status(self, request_info, **kwargs):
        # define
        tenant_id = None
        object_value = None

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                object_value = resp_dict["object"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoints runs the smart status for a given data source, it requires a POST call with the following options:",
                "resource_desc": "Run Smart Status for a given entity",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_smart_status/fqm_smart_status\" body=\"{'tenant_id': 'mytenant', 'object': 'Okta:Splunk_TA_okta_identity_cloud:okta_logs'}\"",
                "options": [
                    {
                        "tenant_id": "the tenant identifier",
                        "object": "name of the data source",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Define the KV query
        query_string = {
            "object": object_value,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get the object record
        try:
            collection_name = "kv_" + "trackme_fqm_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            records = collection.data.query(query=json.dumps(query_string))
            record = records[0]
            key = record.get("_key")

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'object="{object_value}" not found',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 404}

        #
        # object is known, proceed
        #

        # set the initial empty smartstatus_response
        smartstatus_response = {}

        # get anomaly_reason
        anomaly_reason = record.get("anomaly_reason")

        # turn into list from pipe delimited string if not already a list
        if not isinstance(anomaly_reason, list):
            anomaly_reason = anomaly_reason.split("|")

        # retrieve flipping correlation
        try:
            flipping_correlation = smartstatus_flipping_correlation(
                request_info, tenant_id, "splk-fqm", object_value
            )
        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered with smartstatus_flipping_correlation, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

        #################################
        # Greem: happiness is in da house
        #################################

        # if anomaly_reason (which is a list) only contains "none", then there are no anomalies
        if anomaly_reason == ["none"]:
            smartstatus_response["_time"] = time.time()
            smartstatus_response["tenant_id"] = tenant_id
            smartstatus_response["object_category"] = "splk-fqm"
            smartstatus_response["object"] = object_value
            smartstatus_response["alias"] = record.get("alias")
            smartstatus_response["object_state"] = record.get("object_state")
            smartstatus_response["anomaly_reason"] = anomaly_reason
            smartstatus_response["flipping_correlations"] = flipping_correlation
            smartstatus_response["smart_result"] = (
                "The entity is currently in a normal state, therefore further investigations are not required at this stage."
            )

            return {"payload": smartstatus_response, "status": 200}

        ##############################################################
        # use case investigations: Machine Learning Outliers Detection
        ##############################################################

        if "ml_outliers_detection" in anomaly_reason:
            logger.info(
                f'performing investigations use case: Machine Learning Outliers, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates ML Outliers for this object by executing the ML monitor process for enabled models"

            try:
                uc_results = smartstatus_investigations_uc_ml_outliers(
                    request_info, record, "fqm"
                )
                # append to response
                use_case_desc = (
                    "The use case investigates ML Outliers detection for the entity"
                )
                search1_results_label = "results_ml_outliers"
                search1_info_label = "search_info_ml_outliers"

                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_ml_outliers, exception="{str(e)}"'
                )
                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_ml_outliers CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                    "exception": str(e),
                }

        ##########
        # response
        ##########

        smartstatus_response["_time"] = time.time()
        smartstatus_response["tenant_id"] = tenant_id
        smartstatus_response["object_category"] = "splk-fqm"
        smartstatus_response["object"] = object_value
        smartstatus_response["alias"] = record.get("alias")
        smartstatus_response["object_state"] = record.get("object_state")
        smartstatus_response["anomaly_reason"] = anomaly_reason
        smartstatus_response["flipping_correlations"] = flipping_correlation

        return {"payload": smartstatus_response, "status": 200}

    # Get smart status for splk-wlk
    def post_wlk_smart_status(self, request_info, **kwargs):
        # define
        tenant_id = None
        object_value = None

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                object_value = resp_dict["object"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoints runs the smart status for a given data source, it requires a POST call with the following options:",
                "resource_desc": "Run Smart Status for a given entity",
                "resource_spl_example": "| trackme mode=\"post\" url=\"/services/trackme/v2/splk_smart_status/wlk_smart_status\" body=\"{'tenant_id': 'mytenant', 'object': 'mysearch'}\"",
                "options": [
                    {
                        "tenant_id": "the tenant identifier",
                        "object": "name of the data source",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Define the KV query
        query_string = {
            "object": object_value,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Get the object record
        try:
            collection_name = "kv_" + "trackme_wlk_tenant_" + str(tenant_id)
            collection = service.kvstore[collection_name]

            records = collection.data.query(query=json.dumps(query_string))
            record = records[0]
            key = record.get("_key")

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'object="{object_value}" not found',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 404}

        #
        # object is known, proceed
        #

        # set the initial empty smartstatus_response
        smartstatus_response = {}

        # get anomaly_reason
        anomaly_reason = record.get("anomaly_reason")

        # turn into list from pipe delimited string if not already a list
        if not isinstance(anomaly_reason, list):
            anomaly_reason = anomaly_reason.split("|")

        # retrieve flipping correlation
        try:
            flipping_correlation = smartstatus_flipping_correlation(
                request_info, tenant_id, "splk-wlk", object_value
            )
        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered with smartstatus_flipping_correlation, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

        #################################
        # Greem: happiness is in da house
        #################################

        # if anomaly_reason (which is a list) only contains "none", then there are no anomalies
        if anomaly_reason == ["none"]:
            smartstatus_response["_time"] = time.time()
            smartstatus_response["tenant_id"] = tenant_id
            smartstatus_response["object_category"] = "splk-dsm"
            smartstatus_response["object"] = object_value
            smartstatus_response["alias"] = record.get("alias")
            smartstatus_response["object_state"] = record.get("object_state")
            smartstatus_response["anomaly_reason"] = anomaly_reason
            smartstatus_response["flipping_correlations"] = flipping_correlation
            smartstatus_response["smart_result"] = (
                "The entity is currently in a normal state, therefore further investigations are not required at this stage."
            )

            return {"payload": smartstatus_response, "status": 200}

        ##############################################################
        # wlk specific UCs
        ##############################################################

        if "execution_errors_detected" in anomaly_reason:
            logger.info(
                f'performing investigations use case: execution_errors_detected, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates and retrieves Splunk scheduler execution errors for the entity"

            try:
                uc_results = smartstatus_investigations_uc_wlk_execution_errors(
                    request_info, tenant_id, record
                )
                # append to response

                search1_results_label = "results_execution_errors_detected"
                search1_info_label = "search_info_execution_errors_detected"

                smartstatus_response[
                    "smartstatus_investigations_uc_wlk_execution_errors"
                ] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_wlk_execution_errors",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_wlk_execution_errors, exception="{str(e)}"'
                )
                smartstatus_response[
                    "smartstatus_investigations_uc_wlk_execution_errors"
                ] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_wlk_execution_errors CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_wlk_execution_errors",
                    "exception": str(e),
                }

        if "skipping_searches_detected" in anomaly_reason:
            logger.info(
                f'performing investigations use case: skipping_searches_detected, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates and retrieves Splunk scheduler skipping events for the entity"

            try:
                uc_results = smartstatus_investigations_uc_wlk_skipping(
                    request_info, tenant_id, record
                )
                # append to response

                search1_results_label = "results_skipping_searches_detected"
                search1_info_label = "search_info_skipping_searches_detected"

                smartstatus_response["smartstatus_investigations_uc_wlk_skipping"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_wlk_skipping",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_wlk_skipping, exception="{str(e)}"'
                )
                smartstatus_response["smartstatus_investigations_uc_wlk_skipping"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_wlk_skipping CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_wlk_skipping",
                    "exception": str(e),
                }

        if "orphan_search_detected" in anomaly_reason:
            logger.info(
                f'performing investigations use case: orphan_search_detected, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates and retrieves the orphan status for the entity"

            try:
                uc_results = smartstatus_investigations_uc_wlk_orphan(
                    request_info, tenant_id, record
                )
                # append to response

                search1_results_label = "results_orphan_search_detected"
                search1_info_label = "search_info_orphan_search_detected"

                smartstatus_response["smartstatus_investigations_uc_wlk_orphan"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_wlk_orphan",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_wlk_orphan, exception="{str(e)}"'
                )
                smartstatus_response["smartstatus_investigations_uc_wlk_orphan"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_wlk_orphan CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_wlk_orphan",
                    "exception": str(e),
                }

        if "execution_delayed" in anomaly_reason:
            logger.info(
                f'performing investigations use case: execution_delayed, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates and retrieves metadata and last execution for delayed entities"

            try:
                uc_results = smartstatus_investigations_uc_wlk_delayed(
                    request_info, tenant_id, record
                )
                # append to response

                search1_results_label = "results_execution_delayed"
                search1_info_label = "search_info_execution_delayed"

                smartstatus_response["smartstatus_investigations_uc_wlk_delayed"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_wlk_delayed",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_wlk_delayed, exception="{str(e)}"'
                )
                smartstatus_response["smartstatus_investigations_uc_wlk_delayed"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_wlk_delayed CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_wlk_delayed",
                    "exception": str(e),
                }

        ##############################################################
        # use case investigations: Machine Learning Outliers Detection
        ##############################################################

        if "ml_outliers_detection" in anomaly_reason:
            logger.info(
                f'performing investigations use case: Machine Learning Outliers, tenant_id="{tenant_id}", object="{object_value}"'
            )
            use_case_desc = "The use case investigates ML Outliers for this object by executing the ML monitor process for enabled models"

            try:
                uc_results = smartstatus_investigations_uc_ml_outliers(
                    request_info, record, "dsm"
                )
                # append to response
                use_case_desc = (
                    "The use case investigates ML Outliers detection for the entity"
                )
                search1_results_label = "results_ml_outliers"
                search1_info_label = "search_info_ml_outliers"

                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "success",
                    "use_case_desc": use_case_desc,
                    search1_results_label: uc_results.get("results_1"),
                    search1_info_label: uc_results.get("search_info_1"),
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                }

            except Exception as e:
                logger.error(
                    f'smartstatus, tenant_id="{tenant_id}", object="{object_value}", exception encountered while running uc function smartstatus_investigations_uc_ml_outliers, exception="{str(e)}"'
                )
                smartstatus_response["smartstatus_investigations_uc_ml_outliers"] = {
                    "action": "failure",
                    "use_case_desc": use_case_desc,
                    "response": "an exception was encountered while attempting to run the use case investigation, review REST Api logs: index=_internal sourcetype=trackme:rest_api smartstatus_investigations_uc_ml_outliers CASE(ERROR)",
                    "smart_function": "smartstatus_investigations_uc_ml_outliers",
                    "exception": str(e),
                }

        ##########
        # response
        ##########

        smartstatus_response["_time"] = time.time()
        smartstatus_response["tenant_id"] = tenant_id
        smartstatus_response["object_category"] = "splk-wlk"
        smartstatus_response["object"] = object_value
        smartstatus_response["alias"] = record.get("alias")
        smartstatus_response["object_state"] = record.get("object_state")
        smartstatus_response["anomaly_reason"] = anomaly_reason
        smartstatus_response["flipping_correlations"] = flipping_correlation

        return {"payload": smartstatus_response, "status": 200}
