#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_tag_policies.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2022-2026, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Built-in libraries
import json
import os
import sys
import re
import time

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.splk_tag_policies_power",
    "trackme_rest_api_splk_tag_policies_power.log",
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import trackme_getloglevel, trackme_audit_event, get_kv_collection

# import Splunk libs
import splunklib.client as client


class TrackMeHandlerSplkTagPoliciesWrite_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerSplkTagPoliciesWrite_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_splk_tag_policies(self, request_info, **kwargs):
        response = {
            "resource_group_name": "splk_tag_policies/write",
            "resource_group_desc": "Endpoints related to the management of tags (power operations)",
        }

        return {"payload": response, "status": 200}

    # Add new policy
    def post_tag_policies_add(self, request_info, **kwargs):

        # Declare
        tenant_id = None
        component = None
        tags_policy_id = None
        tags_policy_value = None
        tags_policy_regex = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                # value must be either: dsm,dhm,mhm,wlk,flx,fqm
                if component not in ("dsm", "dhm", "mhm", "wlk", "flx", "fqm"):
                    return {
                        "payload": {
                            "response": "The component must be either: dsm,dhm,mhm,wlk,flx,fqm",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    tags_policy_id = resp_dict["tags_policy_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tags_policy_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    tags_policy_value = resp_dict["tags_policy_value"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tags_policy_value is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                # if tags_policy_value is a string, turn into a list and make it lower case, if is a list, make it lower case
                if isinstance(tags_policy_value, str):
                    tags_policy_value = tags_policy_value.lower()
                    tags_policy_value = tags_policy_value.split(",")
                else:
                    tags_policy_value = [x.lower() for x in tags_policy_value]

                try:
                    tags_policy_regex = resp_dict["tags_policy_regex"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tags_policy_regex is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                # verify that the regex is valid
                try:
                    re.compile(tags_policy_regex)
                except re.error:
                    return {
                        "payload": {
                            "response": "The tags_policy_regex is not a valid regular expression",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint creates a new tag policy or updates a policy if it exists already, it requires a POST call with the following data:",
                "resource_desc": "Add or update a tag policy",
                "resource_spl_example": r"| trackme mode=post url=\"/services/trackme/v2/splk_tag_policies/write/tag_policies_add\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'tags_policy_id': 'linux_secure', 'tags_policy_id': 'linux_secure', 'tags_policy_value': 'Linux,OS,CIM', 'tags_policy_regex': '\:linux_secure$'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, must be either: dsm,dhm,mhm,wlk,flx,fqm",
                        "tags_policy_id": "(required) ID of the tag policy",
                        "tags_policy_regex": "(required) The regular expression to be used by the tags policy, special characters should be escaped.",
                        "tags_policy_value": "(required) A comma separated list of tags",
                        "update_comment": "(optional) Comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Define the KV query
        query_string = {
            "tags_policy_id": tags_policy_id,
        }

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Data collection
        collection_name = f"kv_trackme_{component}_tags_policies_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        try:
            record = collection.data.query(query=json.dumps(query_string))
            key = record[0].get("_key")

        except Exception as e:
            key = None

        # proceed

        # this policy exists already, it will be updated to include any missing tags from the list
        if key:
            # set an action_desc
            action_desc = "updated"

            # create a new list
            tags_new = []

            # add existing members
            for tag in record[0].get("tags_policy_value"):
                tags_new.append(tag)

            # add tags, if not in there already
            for tag in tags_policy_value:
                if tag not in tags_new:
                    tags_new.append(tag)

        else:
            # set an action_desc
            action_desc = "created"

        # proceed
        try:
            if key:
                # Update the record
                collection.data.update(
                    str(key),
                    json.dumps(
                        {
                            "tags_policy_id": tags_policy_id,
                            "tags_policy_value": tags_policy_value,
                            "tags_policy_regex": tags_policy_regex,
                            "mtime": time.time(),
                        }
                    ),
                )

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "update tags policy",
                        str(tags_policy_id),
                        f"splk-{component}",
                        collection.data.query(query=json.dumps(query_string)),
                        "The tag policy was updated successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

            else:
                # Insert the record
                collection.data.insert(
                    json.dumps(
                        {
                            "tags_policy_id": tags_policy_id,
                            "tags_policy_value": tags_policy_value,
                            "tags_policy_regex": tags_policy_regex,
                            "mtime": time.time(),
                        }
                    )
                )

                # Record an audit change
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "add tags policy",
                        str(tags_policy_id),
                        f"splk-{component}",
                        collection.data.query(query=json.dumps(query_string)),
                        "The tag policy was added successfully",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

            # render response
            response = {
                "action": "success",
                "action_desc": action_desc,
                "response": f'the tag policy tags_policy_id="{tags_policy_id}" was {action_desc} successfully',
                "record": collection.data.query(query=json.dumps(query_string)),
            }

            return {"payload": response, "status": 200}

        except Exception as e:
            response = {
                "action": "failure",
                "response": f'an exception was encountered, exception="{str(e)}"',
            }
            logger.error(json.dumps(response))
            return {"payload": response, "status": 500}

    # Delete records from the collection
    def post_tag_policies_del(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        component = None
        tags_policy_id_list = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                # value must be either: dsm,dhm,mhm,wlk,flx,fqm
                if component not in ("dsm", "dhm", "mhm", "wlk", "flx", "fqm"):
                    return {
                        "payload": {
                            "response": "The component must be either: dsm,dhm,mhm,wlk,flx,fqm",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    tags_policy_id_list = resp_dict["tags_policy_id_list"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tags_policy_id_list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                # if tags_policy_id_list is a string, turn into a list
                if isinstance(tags_policy_id_list, str):
                    tags_policy_id_list = tags_policy_id_list.split(",")

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint deletes tag policies, it requires a POST call with the following information:",
                "resource_desc": "Delete or ore more tag policies",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_tag_policies/write/tag_policies_del\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'tags_policy_id': 'linux_secure,linux_sec'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, must be either: dsm,dhm,mhm,wlk,flx,fqm",
                        "tags_policy_id_list": "(required) Comma separated list of tag policies",
                        "update_comment": "(optional) Comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Data collection
        collection_name = f"kv_trackme_{component}_tags_policies_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # records summary
        records = []

        # loop
        for item in tags_policy_id_list:
            # Define the KV query
            query_string = {
                "tags_policy_id": item,
            }

            # Get the current record
            # Notes: the record is returned as an array, as we search for a specific record, we expect one record only
            try:
                record = collection.data.query(query=json.dumps(query_string))
                key = record[0].get("_key")

            except Exception as e:
                key = None

            # Render result
            if key:
                # Remove and audit
                try:
                    # Remove the record
                    collection.data.delete(json.dumps({"_key": key}))

                    # increment counter
                    processed_count += 1
                    succcess_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "delete tag policy",
                            str(item),
                            "all",
                            record,
                            "The lagging class was deleted successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "success",
                        "record": record,
                    }

                    records.append(result)

                    logger.info(json.dumps(result, indent=0))

                except Exception as e:
                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "failure",
                            "delete tag policy",
                            str(item),
                            "all",
                            record,
                            str(e),
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "failure",
                        "record": record,
                        "exception": e,
                    }

                    # append to records
                    records.append(result)

                    # log
                    logger.error(json.dumps(result, indent=0))

            else:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                # audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "delete tag policy",
                        str(item),
                        "all",
                        record,
                        "HTTP 404 NOT FOUND",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                result = {
                    "action": "delete",
                    "result": "failure",
                    "record": item,
                    "exception": "HTTP 404 NOT FOUND",
                }

                # append to records
                records.append(result)

                # log
                logger.error(json.dumps(result, indent=0))

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Update records
    def post_tag_policies_update(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        component = None
        records_list = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                # value must be either: dsm,dhm,mhm,wlk,flx,fqm
                if component not in ("dsm", "dhm", "mhm", "wlk", "flx", "fqm"):
                    return {
                        "payload": {
                            "response": "The component must be either: dsm,dhm,mhm,wlk,flx,fqm",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    records_list = resp_dict["records_list"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The records_list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    records_list = json.loads(records_list)
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The records_list is not a valid JSON",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint updates records, it requires a POST call with the following information:",
                "resource_desc": "Update or ore more tag policies",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_tag_policies/write/tag_policies_update\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'records_list': '<redacted_json_records>'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, must be either: dsm,dhm,mhm,wlk,flx,fqm",
                        "records_list": "(required) JSON records to be updated",
                        "update_comment": "(option) Comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
        except Exception as e:
            update_comment = "API update"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # counters
        processed_count = 0
        succcess_count = 0
        failures_count = 0

        # Data collection
        collection_name = f"kv_trackme_{component}_tags_policies_tenant_{tenant_id}"
        collection = service.kvstore[collection_name]

        # records summary
        records = []

        # debug
        logger.info(f'records_list="{json.dumps(records_list, indent=0)}"')

        # loop
        for item in records_list:
            # debug
            logger.info(f'item="{item}"')

            # check if we have the _key, otherwise search for it
            if item.get("_key"):
                key = item.get("_key")

            else:
                # Define the KV query
                query_string = {
                    "tags_policy_id": item.get("tags_policy_id"),
                }

                # Get the current record
                # Notes: the record is returned as an array, as we search for a specific record, we expect one record only
                try:
                    record = collection.data.query(query=json.dumps(query_string))
                    key = record[0].get("_key")

                except Exception as e:
                    key = None

            # Render result
            if key:
                # This record exists already

                # Store the record for audit purposes
                record = str(json.dumps(collection.data.query_by_id(key), indent=1))

                # Update and audit
                try:
                    # Update the record
                    tags_policy_value = item.get("tags_policy_value")

                    # if tags_policy_value is a string, turn into a list and make it lower case, if is a list, make it lower case
                    if isinstance(tags_policy_value, str):
                        tags_policy_value = tags_policy_value.lower()
                        tags_policy_value = tags_policy_value.split(",")
                    else:
                        tags_policy_value = [x.lower() for x in tags_policy_value]

                    collection.data.update(
                        str(key),
                        json.dumps(
                            {
                                "tags_policy_id": item.get("tags_policy_id"),
                                "tags_policy_value": tags_policy_value,
                                "tags_policy_regex": item.get("tags_policy_regex"),
                                "mtime": time.time(),
                            }
                        ),
                    )

                    # increment counter
                    processed_count += 1
                    succcess_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "update tag policies",
                            str(item),
                            component,
                            record,
                            "The lagging classs was updated successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "success",
                        "record": record,
                    }

                    records.append(result)

                    logger.info(json.dumps(result, indent=0))

                except Exception as e:
                    # increment counter
                    processed_count += 1
                    succcess_count += 0
                    failures_count += 1

                    # audit record
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "failure",
                            "update tag policies",
                            str(item),
                            component,
                            record,
                            str(e),
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

                    result = {
                        "action": "delete",
                        "result": "failure",
                        "record": record,
                        "exception": e,
                    }

                    # append to records
                    records.append(result)

                    # log
                    logger.error(json.dumps(result, indent=0))

            else:
                # increment counter
                processed_count += 1
                succcess_count += 0
                failures_count += 1

                # audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "update tag policies",
                        str(item),
                        component,
                        record,
                        "HTTP 404 NOT FOUND",
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                result = {
                    "action": "delete",
                    "result": "failure",
                    "record": item,
                    "exception": "HTTP 404 NOT FOUND",
                }

                # append to records
                records.append(result)

                # log
                logger.error(json.dumps(result, indent=0))

        # render HTTP status and summary

        req_summary = {
            "process_count": processed_count,
            "success_count": succcess_count,
            "failures_count": failures_count,
            "records": records,
        }

        if processed_count > 0 and processed_count == succcess_count:
            return {"payload": req_summary, "status": 200}

        else:
            return {"payload": req_summary, "status": 500}

    # Simulate tags
    def post_tag_policies_simulate(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        component = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                # value must be either: dsm,dhm,mhm,wlk,flx,fqm
                if component not in ("dsm", "dhm", "mhm", "wlk", "flx", "fqm"):
                    return {
                        "payload": {
                            "response": "The component must be either: dsm,dhm,mhm,wlk,flx,fqm",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    regex_value = resp_dict["regex_value"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The argument regex_value is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    tags_list = resp_dict["tags_list"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The argument tags_list is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                # if tags_list is not a list, turn it into a list from comma separated values
                if isinstance(tags_list, str):
                    tags_list = tags_list.split(",")

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint simulates a tags policy, it requires a POST call with the following information:",
                "resource_desc": "Simulates a tags policy",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_tag_policies/write/tag_policies_simulate\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm', 'regex_value': '^org_eu.*', 'tags_list': 'edr,eu,gdpr'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, must be either: dsm,dhm,mhm,wlk,flx,fqm",
                        "regex_value": "(required) The regex to be used for the simulation",
                        "tags_list": "(required) A comma separated list of tags",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # start
        main_start = time.time()

        #
        # tags policies
        #

        tags_policies_records = [
            {
                "tags_policy_regex": regex_value,
                "tags_policy_value": tags_list,
            }
        ]

        #
        # KV entities
        #

        # entities KV collection
        data_collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
        data_collection = service.kvstore[data_collection_name]

        # get records
        data_records, data_collection_keys, data_collection_dict = get_kv_collection(
            data_collection, data_collection_name
        )

        #
        # Handle policies
        #

        # A policy is composed by the following fields: tags_policy_id, tags_policy_regex, tags_policy_value
        # A record has various fields, in this context we care about: object (the value against the regex is applied) and tags_auto (a list to contain all matching tag values)

        # Counters and error list
        entities_failures_count = 0
        entities_exceptions_list = []
        entities_tags_matched = []
        entities_matched = []

        # only proceed if we have entities
        if len(data_records) > 0:

            # loop through records and apply policies
            for entity_record in data_records:

                entity_key = entity_record["_key"]
                entity_object = entity_record["object"]
                tags_auto = []

                # apply policies
                if len(tags_policies_records) > 0:
                    for policy_record in tags_policies_records:

                        # check that the regex expression is a valid regex expression
                        # for the purpose of the simulation, we will however return a 200 with a message
                        try:
                            re.compile(regex_value)
                        except re.error:
                            req_summary = {
                                "entities_matched_count": len(entities_matched),
                                "result_summary": "The regex_value is not a valid regular expression, review this expression and try again.",
                                "regex_value": regex_value,
                                "regex_is_valid": "false",
                            }
                            return {"payload": req_summary, "status": 200}

                        try:
                            # apply regex
                            regex = policy_record["tags_policy_regex"]
                            values = policy_record["tags_policy_value"]

                            # if values is a string, turn into a list and make it lower case, if is a list, make it lower case
                            if isinstance(values, str):
                                values = values.lower()
                                values = values.split(",")
                            else:
                                values = [x.lower() for x in values]

                            if re.match(regex, entity_record["object"]):
                                logger.info(
                                    f'tenant_id="{tenant_id}", object="{entity_object}", regex has matched this entity, regex="{regex}", values="{values}"'
                                )

                                # append to our list
                                if not entity_object in entities_matched:
                                    entities_matched.append(entity_object)

                                # loop through tags
                                for value in values:
                                    if not value in entities_tags_matched:
                                        entities_tags_matched.append(value)
                            else:
                                logger.info(
                                    f'tenant_id="{tenant_id}", object="{entity_object}", regex has not matched this entity, regex="{regex}", values="{values}"'
                                )

                        except Exception as e:
                            logger.error(
                                f'context="exception", failed to apply policy, exception="{str(e)}"'
                            )

        # set action
        if entities_failures_count == 0:
            action = "success"
        else:
            action = "failure"

        # get run_time
        run_time = round((time.time() - main_start), 3)

        # create a result summary
        if len(entities_matched) > 0:
            result_summary = f"The regex has matched {len(entities_matched)} entities and {len(entities_tags_matched)} tags"
        else:
            result_summary = "The regex has not matched any entities, verify your inputs and try again."

        # request summary
        req_summary = {
            "kvstore_collection_entities_count": len(data_records),
            "entities_matched_count": len(entities_matched),
            "entities_matched": entities_matched,
            "entities_tags_matched_count": len(entities_tags_matched),
            "entities_tags_matched": entities_tags_matched,
            "result_summary": result_summary,
            "error_messages": entities_exceptions_list,
            "regex_is_valid": "true",
        }

        # render response
        if action == "success":
            logger.info(
                f'tags simulation operation has terminated, action="{action}", tenant_id="{tenant_id}", run_time="{run_time}"'
            )
            return {"payload": req_summary, "status": 200}

        else:
            logger.error(
                f'tags simulation operation has failed, action="{action}", tenant_id="{tenant_id}", req_summary="{json.dumps(req_summary, indent=2)}"'
            )
            return {"payload": req_summary, "status": 500}

    # Apply tags
    def post_tag_policies_apply(self, request_info, **kwargs):
        # Declare
        tenant_id = None
        component = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The tenant_id is required",
                            "status": 400,
                        },
                        "status": 400,
                    }

                try:
                    component = resp_dict["component"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "The component is required",
                            "status": 400,
                        },
                        "status": 400,
                    }
                # value must be either: dsm,dhm,mhm,wlk,flx,fqm
                if component not in ("dsm", "dhm", "mhm", "wlk", "flx", "fqm"):
                    return {
                        "payload": {
                            "response": "The component must be either: dsm,dhm,mhm,wlk,flx,fqm",
                            "status": 400,
                        },
                        "status": 400,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint applies tags policies, it requires a POST call with the following information:",
                "resource_desc": "Immediately apply and update tags policies",
                "resource_spl_example": "| trackme mode=post url=\"/services/trackme/v2/splk_tag_policies/write/tag_policies_apply\" body=\"{'tenant_id': 'mytenant', 'component': 'dsm'}\"",
                "options": [
                    {
                        "tenant_id": "(required) The tenant identifier",
                        "component": "(required) The component identifier, must be either: dsm,dhm,mhm,wlk,flx,fqm",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.session_key,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # start
        main_start = time.time()

        #
        # KV tags policies
        #

        # tags policies KV collection
        tags_policies_collection_name = (
            f"kv_trackme_{component}_tags_policies_tenant_{tenant_id}"
        )
        tags_policies_collection = service.kvstore[tags_policies_collection_name]

        # get records
        tags_policies_records, tags_collection_keys, tags_collection_dict = (
            get_kv_collection(tags_policies_collection, tags_policies_collection_name)
        )

        #
        # KV entities
        #

        # entities KV collection
        data_collection_name = f"kv_trackme_{component}_tenant_{tenant_id}"
        data_collection = service.kvstore[data_collection_name]

        # get records
        data_records, data_collection_keys, data_collection_dict = get_kv_collection(
            data_collection, data_collection_name
        )

        #
        # KV tags
        #

        # entities KV collection
        tags_collection_name = f"kv_trackme_{component}_tags_tenant_{tenant_id}"
        tags_collection = service.kvstore[tags_collection_name]

        # get records
        tags_records, tags_collection_keys, tags_collection_dict = get_kv_collection(
            tags_collection, tags_collection_name
        )

        #
        # Handle policies
        #

        # A policy is composed by the following fields: tags_policy_id, tags_policy_regex, tags_policy_value
        # A record has various fields, in this context we care about: object (the value against the regex is applied) and tags_auto (a list to contain all matching tag values)

        # Counters and error list
        entities_updated_count = 0
        entities_failures_count = 0
        entities_deleted_count = 0
        entities_exceptions_list = []

        updated_records = []  # we will store the updated records here

        # only proceed if we have entities
        if len(data_records) > 0:

            # loop through records and apply policies
            for entity_record in data_records:

                entity_key = entity_record["_key"]
                entity_object = entity_record["object"]
                tags_auto = []

                # apply policies
                if len(tags_policies_records) > 0:
                    for policy_record in tags_policies_records:
                        try:
                            # apply regex
                            regex = policy_record["tags_policy_regex"]
                            values = policy_record["tags_policy_value"]

                            # if values is not a list, turn it into a list from comma separated values
                            if isinstance(values, str):
                                values = values.lower()
                                values = values.split(",")
                            else:
                                values = [x.lower() for x in values]

                            if re.match(regex, entity_record["object"]):
                                logger.info(
                                    f'tenant_id="{tenant_id}", object="{entity_object}", policy="{policy_record["tags_policy_id"]}" has matched this entity, regex="{regex}", values="{values}"'
                                )
                                for value in values:
                                    if not value in tags_auto:
                                        tags_auto.append(value)
                            else:
                                logger.info(
                                    f'tenant_id="{tenant_id}", object="{entity_object}", policy="{policy_record["tags_policy_id"]}" has not matched this entity, regex="{regex}", values="{values}"'
                                )

                        except Exception as e:
                            logger.error(
                                f'context="exception", failed to apply policy, exception="{str(e)}"'
                            )

                # add to updated_records
                updated_records.append(
                    {
                        "_key": entity_key,
                        "object": entity_object,
                        "tags_auto": tags_auto,
                    }
                )

            # Update records in batches
            chunks = [
                updated_records[i : i + 500]
                for i in range(0, len(updated_records), 500)
            ]
            for chunk in chunks:
                try:
                    tags_collection.data.batch_save(*chunk)
                    entities_updated_count += len(chunk)
                except Exception as e:
                    msg = f'KVstore batch save failed with exception="{str(e)}"'
                    logger.error(msg)
                    entities_exceptions_list.append(msg)

            # for record in tags_records, if the key of the record does not exist in data_collection_keys, delete it
            for record in tags_records:
                if record["_key"] not in data_collection_keys:
                    try:
                        tags_collection.data.delete(
                            json.dumps({"_key": record["_key"]})
                        )
                        # counter deleted
                        entities_deleted_count += 1

                    except Exception as e:
                        # counter failure
                        entities_failures_count += 1
                        msg = f'KVstore delete failed for key="{record["_key"]}", exception="{str(e)}"'
                        logger.error(msg)
                        entities_exceptions_list.append(msg)

            # refresh knowledge
            tags_records, tags_collection_keys, tags_collection_dict = (
                get_kv_collection(tags_collection, tags_collection_name)
            )

            # immediately apply tags on the data collection, loop through tags_records and apply tags to data_records
            updated_data_records = []  # we will store the updated records here

            for tag_record in tags_records:
                for entity_record in data_records:
                    if entity_record["_key"] == tag_record["_key"]:

                        # get the current tags value, we will only update the KVstore record if the final tags field has changed
                        current_tags = entity_record.get("tags", None)

                        entity_record["tags_auto"] = tag_record["tags_auto"]

                        # get the value of tags_manual (if any), we will need to merge tags_auto and tags_manual into tags (lower case, dedup, sort)
                        tags_manual = entity_record.get(
                            "tags_manual", None
                        )  # tags_manual could be a CSV string or a list
                        if tags_manual:
                            if isinstance(tags_manual, str):
                                tags_manual_list = tags_manual.split(",")
                            elif isinstance(tags_manual, list):
                                tags_manual_list = tags_manual
                            else:
                                tags_manual_list = []
                        else:
                            tags_manual_list = []
                        tags_auto = entity_record.get("tags_auto", [])

                        # merge tags_auto and tags_manual into tags
                        tags = ",".join(
                            sorted(
                                list(
                                    set(
                                        [
                                            x.lower()
                                            for x in tags_auto + tags_manual_list
                                            if x
                                        ]
                                    )
                                )
                            )
                        )
                        entity_record["tags"] = tags

                        # compare current_tags (CSV string) and tags (CSV string), if they are different, add the record to the updated_data_records list
                        if current_tags:
                            if current_tags != tags:
                                entity_record["mtime"] = time.time()
                                updated_data_records.append(entity_record)

                        else:  # we have no current tags, we need to update the record
                            entity_record["mtime"] = time.time()
                            updated_data_records.append(entity_record)

            # update the KVstore with the updated_data_records
            if len(updated_data_records) > 0:

                # batch update/insert
                batch_update_collection_start = time.time()

                # process by chunk
                chunks = [
                    updated_data_records[i : i + 500]
                    for i in range(0, len(updated_data_records), 500)
                ]
                for chunk in chunks:
                    try:
                        data_collection.data.batch_save(*chunk)
                    except Exception as e:
                        logger.error(f'KVstore batch failed with exception="{str(e)}"')
                        entities_failures_count += 1
                        entities_exceptions_list.append(str(e))

                # calculate len(final_records) once
                final_records_len = len(updated_data_records)

                # perf counter for the batch operation
                logger.info(
                    f'context="perf", batch KVstore update terminated, no_records="{final_records_len}", run_time="{round((time.time() - batch_update_collection_start), 3)}"'
                )

        # set action
        if entities_failures_count == 0:
            action = "success"
        else:
            action = "failure"

        # get run_time
        run_time = round((time.time() - main_start), 3)

        # request summary
        req_summary = {
            "tenant_id": tenant_id,
            "action": action,
            "run_time": run_time,
            "kvstore_lookup_collection": f"trackme_{component}_tags_tenant_{tenant_id}",
            "tags_policies_no_records": len(tags_policies_records),
            "kvstore_collection_entities_count": len(data_records),
            "entities_updated_count": entities_updated_count,
            "entities_failures_count": entities_failures_count,
            "entities_deleted_count": entities_deleted_count,
            "error_messages": entities_exceptions_list,
        }

        # render response
        if action == "success":
            logger.info(
                f'tags apply operation has terminated, action="{action}", tenant_id="{tenant_id}", run_time="{run_time}"'
            )
            return {"payload": req_summary, "status": 200}

        else:
            logger.error(
                f'tags apply operation has failed, action="{action}", tenant_id="{tenant_id}", req_summary="{json.dumps(req_summary, indent=2)}"'
            )
            return {"payload": req_summary, "status": 500}
