#!/usr/bin/env python
# coding=utf-8

__name__ = "trackme_rest_handler_vtenant.py"
__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2022-2026, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

import os, sys
import json
import ast
from collections import OrderedDict
import time
import datetime
import hashlib
import requests

splunkhome = os.environ["SPLUNK_HOME"]

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# set logging
from trackme_libs_logging import setup_logger

logger = setup_logger(
    "trackme.rest.vtenants_admin", "trackme_rest_api_vtenants_admin.log"
)
# Redirect global logging to use the same handler
import logging
logging.getLogger().handlers = logger.handlers
logging.getLogger().setLevel(logger.level)


# import rest handler
import trackme_rest_handler

# import trackme libs
from trackme_libs import (
    trackme_getloglevel,
    trackme_reqinfo,
    trackme_audit_event,
    trackme_create_report,
    trackme_delete_report,
    trackme_create_kvcollection,
    trackme_delete_kvcollection,
    trackme_create_kvtransform,
    trackme_create_macro,
    trackme_delete_macro,
    trackme_report_update_enablement,
    trackme_macro_update_enablement,
    trackme_transform_update_enablement,
    trackme_delete_kvtransform,
    trackme_kvcollection_update_enablement,
    trackme_send_to_tcm,
    run_splunk_search,
    get_kv_collection,
    trackme_check_report_exists,
    trackme_check_kvcollection_exists,
    trackme_check_kvtransform_exists,
    trackme_get_version,
)

# import trackme rbac libs
from trackme_libs_rbac import (
    trackme_kvcollection_get_acl,
    trackme_kvcollection_update_acl,
    trackme_macro_get_acl,
    trackme_macro_update_acl,
    trackme_report_get_acl,
    trackme_report_update_acl,
    trackme_transform_get_acl,
    trackme_transform_update_acl,
)

# import trackme libs utils
from trackme_libs_utils import remove_leading_spaces, interpret_boolean, check_tenant_id

# import trackme licensing libs
from trackme_libs_licensing import trackme_check_license

# import trackme libs schema
from trackme_libs_schema import trackme_schema_format_version

# import the collections dict
from collections_data import collections_dict
from collections_data import (
    collections_list_dsm,
    collections_list_flx,
    collections_list_fqm,
    collections_list_dhm,
    collections_list_mhm,
    collections_list_wlk,
    collections_list_common,
    collections_list_main,
)

# import Splunk SDK client
import splunklib.client as client

# import the collections dict
from collections_data import vtenant_account_default


class TrackMeHandlerVtenantsAdmin_v2(trackme_rest_handler.RESTHandler):
    def __init__(self, command_line, command_arg):
        super(TrackMeHandlerVtenantsAdmin_v2, self).__init__(
            command_line, command_arg, logger
        )

    def get_resource_group_desc_vtenants(self, request_info, **kwargs):
        response = {
            "resource_group_name": "vtenants/admin",
            "resource_group_desc": "Endpoints related to the management of TrackMe Virtual Tenants (admin operations)",
        }

        return {"payload": response, "status": 200}

    #
    # Clear exec summary knowledge
    #

    def post_clear_exec_summary(self, request_info, **kwargs):

        # Declare main variables
        tenant_id = None
        run_healthtracker = True

        # Declare booleans
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                # requested tenants
                try:
                    tenants = resp_dict["tenants"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "Error, the tenants list is required to clear the exec summary knowledge, use * or all to target all tenants."
                        },
                        "status": 500,
                    }
                # run the health tracker for each tenant
                try:
                    run_healthtracker = resp_dict["run_healthtracker"]
                    if run_healthtracker in ("false", "False"):
                        run_healthtracker = False
                except Exception as e:
                    run_healthtracker = True

                # Update comment is optional and used for audit changes
                try:
                    update_comment = resp_dict["update_comment"]
                    if update_comment == "Enter your audit message":
                        update_comment = "No comment was provided for this operation"
                except Exception as e:
                    update_comment = "No comment was provided for this operation"

        else:
            describe = True

        if describe:
            response = {
                "describe": "This endpoint clears the current exec summary view, it requires a POST call with the following data:",
                "resource_desc": "Clear Virtual Tenant exec summary knowledge",
                "resource_spl_example": '| trackme url="/services/trackme/v2/vtenants/admin/clear_exec_summary" mode="post" body="{\'tenants\': \'all\'}"',
                "options": [
                    {
                        "tenants": "Comma seperated list of tenants scope, use * or all to target all tenants",
                        "run_healthtracker": "OPTIONAL: run the health tracker after clearing the exec summary knowledge, defaults to True",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # start_time
        start_time = time.time()

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # set service_system
        service_system = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Set collection
        collection_name = "kv_trackme_virtual_tenants"
        collection = service_system.kvstore[collection_name]

        # get records
        collection_records, collection_records_keys, collection_dict = (
            get_kv_collection(collection, collection_name)
        )

        # get active tenants
        active_tenants = []
        for record in collection_dict:
            if collection_dict[record]["tenant_status"] == "enabled":
                active_tenants.append(collection_dict[record]["tenant_id"])

        # requested tenants
        requested_tenants = []

        # if not already a list, turn it into a list from comma separated string
        if not isinstance(tenants, list):
            tenants = tenants.split(",")

        if tenants == ["*"] or tenants == ["all"]:
            for tenant_id in active_tenants:
                requested_tenants.append(tenant_id)

        else:
            for tenant_id in tenants:
                if tenant_id in active_tenants:
                    requested_tenants.append(tenant_id)

        # process

        # processed_collection_dict
        processed_collection_dict = {}

        for tenant_id in requested_tenants:

            for record in collection_dict:
                if tenant_id == collection_dict[record]["tenant_id"]:

                    logger.info(
                        f'Clearing exec summary knowledge for tenant_id="{tenant_id}" requested by user="{request_info.user}"'
                    )

                    # clear tenant_objects_exec_summary
                    collection_dict[record]["tenant_objects_exec_summary"] = ""

                    # update KV
                    try:
                        logger.info(
                            f'tenant_id="{tenant_id}", Attempting to update KVstore record'
                        )
                        collection.data.update(
                            record, json.dumps(collection_dict[record])
                        )
                        logger.info(
                            f'tenant_id="{tenant_id}", Kvstore update successfully completed'
                        )

                    except Exception as e:
                        logger.error(
                            f'tenant_id="{tenant_id}", failed to update the KVstore with exception="{str(e)}"'
                        )
                        return {
                            "payload": {
                                "response": f'tenant_id="{tenant_id}", failed to update the KVstore with exception="{str(e)}"'
                            },
                            "status": 500,
                        }

                    if run_healthtracker:

                        # run the health tracker for the tenant
                        search = (
                            f'| savedsearch "trackme_health_tracker_tenant_{tenant_id}"'
                        )
                        kwargs_oneshot = {
                            "earliest_time": "-5m",
                            "latest_time": "now",
                            "output_mode": "json",
                            "count": 0,
                        }

                        logger.info(
                            f'tenant_id="{tenant_id}", attempting to run the health tracker'
                        )
                        try:
                            reader = run_splunk_search(
                                service_system,
                                search,
                                kwargs_oneshot,
                                24,
                                5,
                            )
                            for item in reader:
                                if isinstance(item, dict):
                                    logger.info(
                                        f'tenant_id="{tenant_id}", health tracker result="{json.dumps(item, indent=2)}'
                                    )

                        except Exception as e:
                            logger.error(
                                f'tenant_id="{tenant_id}", failed to run the health tracker search with exception="{str(e)}"'
                            )
                            return {
                                "payload": {
                                    "response": f'tenant_id="{tenant_id}", failed to run the health tracker, exception="{str(e)}"'
                                },
                                "status": 500,
                            }

                    # add to processed_collection_dict
                    processed_collection_dict[record] = collection_dict[record]

                    # audit event
                    try:
                        trackme_audit_event(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            request_info.user,
                            "success",
                            "clear exec summary knowledge",
                            tenant_id,
                            "virtual_tenants",
                            json.dumps(processed_collection_dict[record]),
                            "The Virtual Tenant exec summary knowledge has been cleared successfully",
                            str(update_comment),
                        )
                    except Exception as e:
                        logger.error(
                            f'failed to generate an audit event with exception="{str(e)}"'
                        )

        # return the response
        run_time = round(time.time() - start_time, 3)

        result_record = {
            "requested_tenants": requested_tenants,
            "user": request_info.user,
            "run_healthtracker": run_healthtracker,
            "results": "exec summary knowledge has been cleared successfully for the selected Virtual Tenants",
            "collection_records": processed_collection_dict,
            "run_time": run_time,
        }

        logger.info(
            f'clear exec summary knowledge has terminated successfully, run_time="{run_time}", user="{request_info.user}", requested_tenants="{json.dumps(requested_tenants, indent=2)}"'
        )

        return {
            "payload": result_record,
            "status": 200,
        }

    #
    # Add new tenant
    #

    def post_add_tenant(self, request_info, **kwargs):
        """
        | trackme url="/services/trackme/v2/vtenants/admin/add_tenant" mode="post" body="{ 'tenant_desc': 'Demo tenant', 'tenant_name': 'mytenant',
        'tenant_roles_admin': 'trackme_admin', 'tenant_roles_power': 'trackme_power', 'tenant_roles_user': 'trackme_user', 'tenant_owner': 'admin', 'tenant_idx_settings': 'global',
        'tenant_dsm_enabled': 'true',
        'tenant_sampling_obfuscation': 'disabled',
        'update_comment': 'Created for the purpose of the documentation.'}"
        """

        # Declare main variables
        tenant_id = None
        tenant_alias = None
        tenant_name = None
        tenant_desc = None
        tenant_owner = "admin"
        tenant_roles_admin = "trackme_admin"
        tenant_roles_user = "trackme_user"
        tenant_roles_power = "trackme_power"
        tenant_idx_settings = None
        query_string = None

        # Declare booleans
        describe = False
        tenant_dsm_enabled = False
        tenant_dhm_enabled = False
        tenant_mhm_enabled = False
        tenant_flx_enabled = False
        tenant_fqm_enabled = False
        tenant_wlk_enabled = False
        tenant_replica = False

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                # get tenant_name, and ensures it complies with various rules
                tenant_name = resp_dict["tenant_name"]

                # ensure tenant_name is a valid identifier
                tenant_name = check_tenant_id(tenant_name)

                # set tenant_id to tenant_name
                tenant_id = tenant_name

                # tenant_alias is optional and will fallback to tenant_id if not specified
                tenant_desc = resp_dict["tenant_desc"]

            # Indexes settings
            try:
                tenant_idx_settings = resp_dict["tenant_idx_settings"]
            except Exception as e:
                tenant_idx_settings = "global"

            # if not global, if not a dict attempt to load as a dict
            if tenant_idx_settings != "global":
                if not isinstance(tenant_idx_settings, dict):

                    tenant_idx_settings_parsed = False

                    # try to load as a JSON object using json.loads
                    try:
                        tenant_idx_settings = json.loads(tenant_idx_settings)
                        tenant_idx_settings_parsed = True
                    except Exception as e:
                        exception_json_loads = True
                        exception_json_loads_error = str(e)

                    if not tenant_idx_settings_parsed:
                        # try to load as a JSON object using ast.literal_eval
                        try:
                            tenant_idx_settings = ast.literal_eval(tenant_idx_settings)
                            tenant_idx_settings_parsed = True
                        except Exception as e:
                            exception_ast_literal_eval = True
                            exception_ast_literal_eval_error = str(e)

                    if not tenant_idx_settings_parsed:
                        if exception_json_loads:
                            error_msg = f'Error, the tenant_idx_settings is not a valid JSON object, you must use either key word global or provide a valid JSON defining the list of indexes for this tenant, we received: {tenant_idx_settings} error="{exception_json_loads_error}"'
                        elif exception_ast_literal_eval:
                            error_msg = f'Error, the tenant_idx_settings is not a valid JSON object, you must use either key word global or provide a valid JSON defining the list of indexes for this tenant, we received: {tenant_idx_settings} error="{exception_ast_literal_eval_error}"'
                        return {
                            "payload": {
                                "response": error_msg,
                            },
                            "status": 500,
                        }

                    else:

                        # ensure we have a valid value (non null, non empty) for each expected index: trackme_summary_idx, trackme_audit_idx,, trackme_notable_idx, trackme_metric_idx
                        expected_indexes = [
                            "trackme_summary_idx",
                            "trackme_audit_idx",
                            "trackme_notable_idx",
                            "trackme_metric_idx",
                        ]
                        for index_name in expected_indexes:
                            if index_name not in tenant_idx_settings:
                                error_msg = f'Error, the tenant_idx_settings is missing the key "{index_name}", you must provide a valid JSON defining the list of indexes for this tenant'
                                return {
                                    "payload": {
                                        "response": error_msg,
                                    },
                                    "status": 500,
                                }

            # RBAC
            try:
                tenant_roles_admin = resp_dict["tenant_roles_admin"]
                if isinstance(tenant_roles_admin, list):
                    tenant_roles_admin = ",".join(resp_dict["tenant_roles_admin"])
            except Exception as e:
                tenant_roles_admin = "trackme_admin"

            try:
                tenant_roles_user = resp_dict["tenant_roles_user"]
                if isinstance(tenant_roles_user, list):
                    tenant_roles_user = ",".join(resp_dict["tenant_roles_user"])
            except Exception as e:
                tenant_roles_user = "trackme_user"

            try:
                tenant_roles_power = resp_dict["tenant_roles_power"]
                if isinstance(tenant_roles_power, list):
                    tenant_roles_power = ",".join(resp_dict["tenant_roles_power"])
            except Exception as e:
                tenant_roles_power = "trackme_power"

            try:
                tenant_owner = resp_dict["tenant_owner"]
            except Exception as e:
                tenant_owner = "admin"

            # for read permissions, concatenate admin, power and user
            tenant_roles_read_perms = (
                f"{tenant_roles_admin},{tenant_roles_power},{tenant_roles_user}"
            )

            # for write permissions, concatenate admin, power
            tenant_roles_write_perms = f"{tenant_roles_admin},{tenant_roles_power}"

            # boolean options
            tenant_replica = interpret_boolean(resp_dict.get("tenant_replica"))
            tenant_dsm_enabled = interpret_boolean(resp_dict.get("tenant_dsm_enabled"))
            tenant_flx_enabled = interpret_boolean(resp_dict.get("tenant_flx_enabled"))
            tenant_fqm_enabled = interpret_boolean(resp_dict.get("tenant_fqm_enabled"))
            tenant_wlk_enabled = interpret_boolean(resp_dict.get("tenant_wlk_enabled"))
            tenant_dhm_enabled = interpret_boolean(resp_dict.get("tenant_dhm_enabled"))
            tenant_mhm_enabled = interpret_boolean(resp_dict.get("tenant_mhm_enabled"))

            # At least one component needs to be enabled
            if not describe:
                if (
                    not tenant_dsm_enabled
                    and not tenant_dhm_enabled
                    and not tenant_mhm_enabled
                    and not tenant_flx_enabled
                    and not tenant_fqm_enabled
                    and not tenant_wlk_enabled
                ):
                    return {
                        "payload": {
                            "response": "Error, at least one component needs to be enabled, a new tenant cannot be created with no components."
                        },
                        "status": 200,
                    }

                #
                # Tenants global options
                #

                # State translation dictionary extended to handle boolean and integer values
                tenant_options_state_mapping = {
                    "enabled": 1,
                    "Enabled": 1,
                    "1": 1,
                    1: 1,  # Ensures integer 1 is also mapped correctly
                    True: 1,
                    "disabled": 0,
                    "Disabled": 0,
                    "0": 0,
                    0: 0,  # Ensures integer 0 is also mapped correctly
                    False: 0,
                }

                # List of settings with their default state values
                tenant_options_settings = {
                    "tenant_sampling_obfuscation": 0,
                    "tenant_adaptive_delay": 1,
                    "tenant_sampling": 1,
                    "tenant_mloutliers": 1,
                    "tenant_mloutliers_allowlist": "dsm,dhm,flx,wlk,fqm",
                    "tenant_adaptive_delay": 1,
                    "tenant_indexed_constraint": "",
                    "tenant_splk_feeds_delayed_inspector_24hours_range_min_sec": trackme_conf[
                        "trackme_conf"
                    ][
                        "splk_general"
                    ][
                        "splk_general_feeds_delayed_inspector_24hours_range_min_sec"
                    ],
                    "tenant_splk_feeds_delayed_inspector_7days_range_min_sec": trackme_conf[
                        "trackme_conf"
                    ][
                        "splk_general"
                    ][
                        "splk_general_feeds_delayed_inspector_7days_range_min_sec"
                    ],
                    "tenant_splk_feeds_delayed_inspector_until_disabled_range_min_sec": trackme_conf[
                        "trackme_conf"
                    ][
                        "splk_general"
                    ][
                        "splk_general_feeds_delayed_inspector_until_disabled_range_min_sec"
                    ],
                    "tenant_splk_feeds_auto_disablement_period": trackme_conf[
                        "trackme_conf"
                    ]["splk_general"]["splk_general_feeds_auto_disablement_period"],
                    "tenant_cmdb_lookup": 1,
                    "tenant_splk_dsm_cmdb_search": "",
                    "tenant_splk_dhm_cmdb_search": "",
                    "tenant_splk_mhm_cmdb_search": "",
                    "tenant_splk_flx_cmdb_search": "",
                    "tenant_splk_fqm_cmdb_search": "",
                    "tenant_splk_wlk_cmdb_search": "",
                    "tenant_default_priority": "medium",
                    "tenant_pagination_mode": trackme_conf["trackme_conf"][
                        "trackme_general"
                    ]["pagination_mode"],
                    "tenant_pagination_size": int(
                        trackme_conf["trackme_conf"]["trackme_general"][
                            "pagination_size"
                        ]
                    ),
                    "tenant_splk_dsm_tabulator_groupby": "data_index",
                    "tenant_splk_dhm_tabulator_groupby": "tenant_id",
                    "tenant_splk_mhm_tabulator_groupby": "tenant_id",
                    "tenant_splk_flx_tabulator_groupby": "group",
                    "tenant_splk_fqm_tabulator_groupby": "metadata_datamodel,metadata_index,metadata_sourcetype",
                    "tenant_splk_wlk_tabulator_groupby": "overgroup",
                    "tenant_default_disruption_min_time_sec": 0,
                    "tenant_default_monitoring_time_policy": "all_time",
                }

                # Process each setting according to the defined rules
                for key, default in tenant_options_settings.items():
                    try:
                        # Get the value from response and map it using the state_mapping with a default fallback
                        value = resp_dict.get(key)
                        # except string based parameters
                        if key in ("tenant_default_priority", "pagination_mode"):
                            tenant_options_settings[key] = value
                        else:
                            tenant_options_settings[key] = (
                                tenant_options_state_mapping.get(value, default)
                            )
                    except Exception as e:
                        tenant_options_settings[key] = default

                # Now settings dictionary has all the processed states
                tenant_sampling_obfuscation = tenant_options_settings[
                    "tenant_sampling_obfuscation"
                ]
                tenant_adaptive_delay = tenant_options_settings["tenant_adaptive_delay"]
                tenant_sampling = tenant_options_settings["tenant_sampling"]
                tenant_mloutliers = tenant_options_settings["tenant_mloutliers"]
                tenant_mloutliers_allowlist = tenant_options_settings[
                    "tenant_mloutliers_allowlist"
                ]
                tenant_adaptive_delay = tenant_options_settings["tenant_adaptive_delay"]
                tenant_indexed_constraint = tenant_options_settings[
                    "tenant_indexed_constraint"
                ]
                tenant_splk_feeds_delayed_inspector_24hours_range_min_sec = (
                    tenant_options_settings[
                        "tenant_splk_feeds_delayed_inspector_24hours_range_min_sec"
                    ]
                )
                tenant_splk_feeds_delayed_inspector_7days_range_min_sec = (
                    tenant_options_settings[
                        "tenant_splk_feeds_delayed_inspector_7days_range_min_sec"
                    ]
                )
                tenant_splk_feeds_delayed_inspector_until_disabled_range_min_sec = tenant_options_settings[
                    "tenant_splk_feeds_delayed_inspector_until_disabled_range_min_sec"
                ]
                tenant_splk_feeds_auto_disablement_period = tenant_options_settings[
                    "tenant_splk_feeds_auto_disablement_period"
                ]
                tenant_cmdb_lookup = tenant_options_settings["tenant_cmdb_lookup"]
                tenant_splk_dsm_cmdb_search = tenant_options_settings[
                    "tenant_splk_dsm_cmdb_search"
                ]
                tenant_splk_dhm_cmdb_search = tenant_options_settings[
                    "tenant_splk_dhm_cmdb_search"
                ]
                tenant_splk_mhm_cmdb_search = tenant_options_settings[
                    "tenant_splk_mhm_cmdb_search"
                ]
                tenant_splk_flx_cmdb_search = tenant_options_settings[
                    "tenant_splk_flx_cmdb_search"
                ]
                tenant_splk_fqm_cmdb_search = tenant_options_settings[
                    "tenant_splk_fqm_cmdb_search"
                ]
                tenant_splk_wlk_cmdb_search = tenant_options_settings[
                    "tenant_splk_wlk_cmdb_search"
                ]
                tenant_default_priority = tenant_options_settings[
                    "tenant_default_priority"
                ]
                tenant_pagination_mode = tenant_options_settings[
                    "tenant_pagination_mode"
                ]
                tenant_pagination_size = tenant_options_settings[
                    "tenant_pagination_size"
                ]
                tenant_splk_dsm_tabulator_groupby = tenant_options_settings[
                    "tenant_splk_dsm_tabulator_groupby"
                ]
                tenant_splk_dhm_tabulator_groupby = tenant_options_settings[
                    "tenant_splk_dhm_tabulator_groupby"
                ]
                tenant_splk_mhm_tabulator_groupby = tenant_options_settings[
                    "tenant_splk_mhm_tabulator_groupby"
                ]
                tenant_splk_flx_tabulator_groupby = tenant_options_settings[
                    "tenant_splk_flx_tabulator_groupby"
                ]
                tenant_splk_fqm_tabulator_groupby = tenant_options_settings[
                    "tenant_splk_fqm_tabulator_groupby"
                ]
                tenant_splk_wlk_tabulator_groupby = tenant_options_settings[
                    "tenant_splk_wlk_tabulator_groupby"
                ]
                tenant_default_disruption_min_time_sec = tenant_options_settings[
                    "tenant_default_disruption_min_time_sec"
                ]
                tenant_default_monitoring_time_policy = tenant_options_settings[
                    "tenant_default_monitoring_time_policy"
                ]

            # if DHM, allows defining the root constraint, defines a default value if unspecified
            if tenant_dhm_enabled:
                # alerting policy
                try:
                    tenant_dhm_alerting_policy = resp_dict["tenant_dhm_alerting_policy"]
                    if tenant_dhm_alerting_policy in (
                        "track_per_host",
                        "track_per_sourcetype",
                    ):
                        tenant_dhm_alerting_policy = str(tenant_dhm_alerting_policy)
                    else:
                        tenant_dhm_alerting_policy = "track_per_host"
                except Exception as e:
                    tenant_dhm_alerting_policy = "track_per_host"

            # if WLK
            if tenant_wlk_enabled:
                # ML outliers at discovery
                try:
                    tenant_wlk_outliers_metrics = resp_dict[
                        "tenant_wlk_outliers_metrics"
                    ]
                except Exception as e:
                    tenant_wlk_outliers_metrics = "none"

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be called to create a new virtual tenant, it requires a POST call with the following information:",
                "resource_desc": "Add a new virtual tenant",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/vtenants/admin/add_tenant\" mode=\"post\" body=\"{ 'tenant_desc': 'Demo tenant', 'tenant_name': 'mytenant', "
                + "'tenant_roles_admin': 'trackme_admin', 'tenant_roles_power': 'trackme_power', 'tenant_roles_user': 'trackme_user', 'tenant_owner': 'admin', 'tenant_idx_settings': 'global', "
                + "'tenant_dsm_enabled': 'true', "
                + "'tenant_sampling_obfuscation': 'disabled', "
                + "'update_comment': 'Created for the purpose of the documentation.'}\"",
                "options": [
                    {
                        "tenant_name": "name of the virtual tenant to be created",
                        "tenant_alias": "OPTIONAL: alias of the virtual tenant to be created, this value is used in the Virtual Tenant UI and can be updated at anytime, if not specified will fallback to the tenant_name",
                        "tenant_desc": "OPTIONAL: description information for this new virtual tenant, if unset this value will be empty and the Virtual Tenant UI will not attempt to show a description for this tenant",
                        "tenant_dsm_enabled": "OPTIONAL: enable or disable the DSM component, accepts True|False, default is False",
                        "tenant_adaptive_delay": "OPTIONAL: for splk-dsm/dhm only - allows adaptive delay to automatically adapt the delay thresholds for this tenant, default is enabled",
                        "tenant_sampling": "OPTIONAL: for splk-dsm only - enables the data sampling feature for this tenant, default is enabled",
                        "tenant_mloutliers": "OPTIONAL: for eligible components (splk-dsm/dhm/flx/fqm/wlk) - enables the ML outliers detection for this tenant, default is enabled",
                        "tenant_mloutliers_allowlist": "OPTIONAL: for eligible components (splk-dsm/dhm/flx/fqm/wlk) - defines the list of components where the ML outliers detection is enabled, default is all components",
                        "tenant_adapaive_delay": "OPTIONAL: for splk-dsm/dhm only - enables the adaptive delay feature for this tenant, default is enabled",
                        "tenant_default_priority": "OPTIONAL: for all components - defines the default priority for the entities discovered, default is medium",
                        "tenant_pagination_mode": "OPTIONAL: for all components - defines the pagination mode for Tabulator, default is local unless a custom value is set at the default level",
                        "tenant_pagination_size": "OPTIONAL: for all components - defines the pagination size for Tabulator, default is 10000 unless a custom value is set at the default level",
                        "tenant_cmdb_lookup": "OPTIONAL: for all components - enables the CMDB lookup feature for this tenant, default is enabled",
                        "tenant_splk_dsm_cmdb_search": "OPTIONAL: for splk-dsm only - defines the CMDB lookup search for this tenant, defaults to empty",
                        "tenant_splk_dhm_cmdb_search": "OPTIONAL: for splk-dhm only - defines the CMDB lookup search for this tenant, defaults to empty",
                        "tenant_splk_mhm_cmdb_search": "OPTIONAL: for splk-mhm only - defines the CMDB lookup search for this tenant, defaults to empty",
                        "tenant_splk_flx_cmdb_search": "OPTIONAL: for splk-flx only - defines the CMDB lookup search for this tenant, defaults to empty",
                        "tenant_splk_fqm_cmdb_search": "OPTIONAL: for splk-fqm only - defines the CMDB lookup search for this tenant, defaults to empty",
                        "tenant_splk_wlk_cmdb_search": "OPTIONAL: for splk-wlk only - defines the CMDB lookup search for this tenant, defaults to empty",
                        "tenant_indexed_constraint": "OPTIONAL: for splk-dsm only - defines the indexed constraint for this tenant, defaults to empty",
                        "tenant_splk_feeds_delayed_inspector_24hours_range_min_sec": "OPTIONAL: for all feeds components - defines the minimum seconds between inspections for the 24 hours range delayed inspector, defaults to the global setting",
                        "tenant_splk_feeds_delayed_inspector_7days_range_min_sec": "OPTIONAL: for all feeds components - defines the minimum seconds between inspections for the 7 days range delayed inspector, defaults to the global setting",
                        "tenant_splk_feeds_delayed_inspector_until_disabled_range_min_sec": "OPTIONAL: for all feeds components - defines the minimum seconds between inspections for the until disabled range delayed inspector, defaults to the global setting",
                        "tenant_splk_feeds_auto_disablement_period": "OPTIONAL: for all feeds components - defines the auto disablement period for the feeds, defaults to the global setting",
                        "tenant_sampling_obfuscation": "OPTIONAL: enable or disable (default) the data sampling obfuscation for this tenant",
                        "tenant_flx_enabled": "OPTIONAL: enable or disable the FLX component, accepts True|False, default is False",
                        "tenant_fqm_enabled": "OPTIONAL: enable or disable the FQM component, accepts True|False, default is False",
                        "tenant_wlk_enabled": "OPTIONAL: enable or disable the WLK component, accepts True|False, default is False",
                        "tenant_dhm_enabled": "OPTIONAL: enable or disable the DHM component, accepts True|False, default is False",
                        "tenant_dhm_enable_outliers": "OPTIONAL: enable (default) or disable outliers detection by default when entities are discovered",
                        "tenant_dhm_alerting_policy": "OPTIONAL: (track_per_host | track_per_sourcetype) define the alerting global policy for this tenant, this can be per host (default) or per sourcetype",
                        "tenant_mhm_enabled": "OPTIONAL: enable or disable the MHM component, accepts True|False, default is False",
                        "tenant_wlk_outliers_metrics": "OPTIONAL: defines the ML outliers preferences at discovery for splk-wlk, defaults to none which disables the feature",
                        "tenant_owner": "OPTIONAL: the user owner of all objects to be created for this tenant, if not specified defaults to the system wide configured owner (nobody by default)",
                        "tenant_roles_admin": "OPTIONAL: a comma separated list of Splunk roles which will be given full control permissions on the tenant objects, defaults to the builtin role trackme_admin",
                        "tenant_roles_power": "OPTIONAL: a comma separated list of Splunk roles which will be given power permissions on the tenant objects, defaults to the builtin role trackme_power",
                        "tenant_roles_user": "OPTIONAL: a comma separated list of Splunk roles which will be given read only permissions on the tenant objects, defaults to the builtin role trackme_user",
                        "tenant_idx_settings": 'OPTIONAL: the indexes definition for the tenant, this can be the keyword "global" to rely on the app configuration level indexes, or a JSON object defining the index for each stanza (defaults to global)',
                        "tenant_replica": "OPTIONAL: Set this tenant as a replica tenant, which means the tenant will be a target for a replica tracker and will not perform active tracking itself, valid options: True|False (default to False)",
                        "tenant_splk_dsm_tabulator_groupby": "OPTIONAL: for splk-dsm only - defines the default groupby for the tabulator view, defaults to data_index",
                        "tenant_splk_dhm_tabulator_groupby": "OPTIONAL: for splk-dhm only - defines the default groupby for the tabulator view, defaults to tenant_id",
                        "tenant_splk_mhm_tabulator_groupby": "OPTIONAL: for splk-mhm only - defines the default groupby for the tabulator view, defaults to tenant_id",
                        "tenant_splk_flx_tabulator_groupby": "OPTIONAL: for splk-flx only - defines the default groupby for the tabulator view, defaults to group",
                        "tenant_splk_fqm_tabulator_groupby": "OPTIONAL: for splk-fqm only - defines the default groupby for the tabulator view, defaults to metadata_datamodel,metadata_index,metadata_sourcetype",
                        "tenant_splk_wlk_tabulator_groupby": "OPTIONAL: for splk-wlk only - defines the default groupby for the tabulator view, defaults to overgroup",
                        "tenant_default_disruption_min_time_sec": "OPTIONAL: for all components - defines the default minimal disruption period in seconds, defaults to 0",
                        "tenant_default_monitoring_time_policy": "OPTIONAL: for all components - defines the default monitoring time policy for this tenant, this is a string value that represents the monitoring time policy, the default is: all_time. (accepts: all_time, business_days_all_hours, monday_saturday_all_hours, business_days_08h_20h, monday_saturday_08h_20h or a JSON dictionary)",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
            if update_comment == "Enter your audit message":
                update_comment = "No comment was provided for this operation"
        except Exception as e:
            update_comment = "No comment was provided for this operation"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        # TrackMe sharing level
        trackme_default_sharing = trackme_conf["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        ################
        # MAIN PROGRAM #
        ################

        # get trackme release
        trackme_version = trackme_get_version(service)
        logger.info(f'running trackme version="{trackme_version}"')

        # set the schema_version_current
        schema_version_current = trackme_schema_format_version(trackme_version)

        # check license state
        try:
            check_license = trackme_check_license(
                request_info.server_rest_uri, request_info.session_key
            )
            license_is_valid = check_license.get("license_is_valid")
            license_subscription_class = check_license.get("license_subscription_class")
            logger.debug(
                f'function check_license called, response="{json.dumps(check_license, indent=2)}"'
            )

        except Exception as e:
            license_is_valid = 0
            license_subscription_class = "free"
            logger.error(f'function check_license exception="{str(e)}"')

        # refuse to operate if the tenant_name is more than 20 chars
        if len(tenant_name) > 20:
            audit_record = {
                "action": "failure",
                "change_type": "add new virtual tenant",
                "object": str(tenant_name),
                "object_category": "virtual_tenants",
                "result": "I'm afraid I can't do that, the maximal number of characters for the tenant name cannot exceed 20 characters.",
            }

            return {"payload": audit_record, "status": 409}

        # tenant_alias is optional and will fallback to tenant_id if not specified
        tenant_alias = resp_dict.get("tenant_alias", tenant_id)

        # Generate the tenant_sha256 from tenant_id
        tenant_sha256 = hashlib.sha256(tenant_id.encode("utf-8")).hexdigest()

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        # Data collection
        collection_name = "kv_trackme_virtual_tenants"
        collection = service.kvstore[collection_name]

        # Get the current number of active tenants for licensing purposes
        query_string = {
            "tenant_status": "enabled",
        }

        try:
            vtenant_active_tenants_records = collection.data.query(
                query=json.dumps(query_string)
            )
        except Exception as e:
            vtenant_active_tenants_records = []

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        try:
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]
            key = vtenant_record.get("_key")

        except Exception as e:
            key = None

        # Proceed
        if key:
            # This record exists already
            audit_record = {
                "action": "failure",
                "change_type": "add new virtual tenant",
                "object": str(tenant_name),
                "object_category": "virtual_tenants",
                "result": "I'm afraid I can't do that, a virtual tenant exists already with the same identifier, run the operation again with a different identifier.",
            }

            logger.error(str(audit_record))
            return {"payload": audit_record, "status": 409}

        # max free tenants licensing restriction reached
        elif len(vtenant_active_tenants_records) >= 2 and license_is_valid != 1:
            # Licensing restrictions reached
            audit_record = {
                "action": "failure",
                "change_type": "add new virtual tenant",
                "object": str(tenant_name),
                "object_category": "virtual_tenants",
                "result": f"I'm afraid I can't do that, this instance is running in Free limited mode edition wich is limited to two active tenants maximum, there are {len(vtenant_active_tenants_records)} active tenants currently",
            }

            logger.error(str(audit_record))
            return {"payload": audit_record, "status": 402}

        # restricted components
        elif license_is_valid != 1 and (
            tenant_flx_enabled or tenant_wlk_enabled
        ):
            # Licensing restrictions reached
            audit_record = {
                "action": "failure",
                "change_type": "add new virtual tenant",
                "object": str(tenant_name),
                "object_category": "virtual_tenants",
                "result": "I'm afraid I can't do that, this instance is running Free limited mode and one or more of the components you have selected are restricted to the Full or Trial edition.",
            }

            logger.error(str(audit_record))
            return {"payload": audit_record, "status": 402}

        # restricted components
        elif (
            license_is_valid == 1
            and (tenant_flx_enabled)
            and license_subscription_class == "free_extended"
        ):
            # Licensing restrictions reached
            audit_record = {
                "action": "failure",
                "change_type": "add new virtual tenant",
                "object": str(tenant_name),
                "object_category": "virtual_tenants",
                "result": "I'm afraid I can't do that, this instance is running Free extended mode and one or more of the components you have selected are restricted to the Full or Trial edition.",
            }

            logger.error(str(audit_record))
            return {"payload": audit_record, "status": 402}

        # reached Free extended mode limitations
        elif (
            license_is_valid == 1
            and license_subscription_class == "free_extended"
            and len(vtenant_active_tenants_records) >= 2
        ):
            # Licensing restrictions reached
            audit_record = {
                "action": "failure",
                "change_type": "add new virtual tenant",
                "object": str(tenant_name),
                "object_category": "virtual_tenants",
                "result": f"I'm afraid I can't do that, this instance is running Free extended Edition wich is limited to two active tenants maximum, there are {len(vtenant_active_tenants_records)} active tenants currently",
            }

            logger.error(str(audit_record))
            return {"payload": audit_record, "status": 402}

        # reached Enterprise mode limitations
        elif (
            license_is_valid == 1
            and license_subscription_class == "enterprise"
            and len(vtenant_active_tenants_records) >= 6
        ):
            # Licensing restrictions reached
            audit_record = {
                "action": "failure",
                "change_type": "add new virtual tenant",
                "object": str(tenant_name),
                "object_category": "virtual_tenants",
                "result": f"I'm afraid I can't do that, this instance is running Enterprise Edition wich is limited to two active tenants maximum, there are {len(vtenant_active_tenants_records)} active tenants currently",
            }

            logger.error(str(audit_record))
            return {"payload": audit_record, "status": 402}

        else:
            # This tenant does not exist yet

            # check if TCM is enabled in receiver mode
            enable_conf_manager_receiver = int(
                trackme_conf["trackme_conf"]["trackme_general"][
                    "enable_conf_manager_receiver"
                ]
            )

            if enable_conf_manager_receiver == 1:
                try:
                    tcm_response = trackme_send_to_tcm(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        resp_dict,
                        "post",
                        "/services/trackme/v2/vtenants/admin/add_tenant",
                    )
                    logger.info(f"trackme_send_to_tcm was successfully executed")
                except Exception as e:
                    logger.error(
                        f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                    )

            # operation counters
            results_record = []

            ################################
            # Data Source Monitoring (DSM) #
            ################################

            # step: Create the DSM virtual tenant knowledge objects if enabled
            if tenant_dsm_enabled:
                for object_name in collections_list_dsm:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # step: create the DSM Data Sampling tracker

                if not tenant_replica:
                    report_name = (
                        f"trackme_dsm_data_sampling_tracker_tenant_{tenant_id}"
                    )
                    report_search = f'| trackmesamplingexecutor tenant_id="{tenant_id}"'
                    report_properties = {
                        "description": "TrackMe DSM Data Sampling tracker",
                        "is_scheduled": True,
                        "schedule_window": "5",
                        "cron_schedule": "*/20 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-24h",
                        "dispatch.latest_time": "-4h",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the DSM Shared Elastic tracker

                    report_name = "trackme_dsm_shared_elastic_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmeelasticexecutor tenant_id="'
                        + str(tenant_id)
                        + '" component="splk-dsm"'
                    )
                    report_properties = {
                        "description": "TrackMe DSM shared elastic tracker",
                        "is_scheduled": True,
                        "schedule_window": "5",
                        "cron_schedule": "*/5 * * * *",
                        "dispatch.earliest_time": "-4h",
                        "dispatch.latest_time": "+4h",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_dsm_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="dsm"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_dsm_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="dsm" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # adaptive delay treshold
                    #

                    report_name = (
                        f"trackme_dsm_adaptive_delay_tracker_tenant_{tenant_id}"
                    )
                    report_search = f'| trackmesplkadaptivedelay tenant_id="{tenant_id}" component="dsm" min_delay_sec=3600 min_historical_metrics_days=7 earliest_time_mstats="-30d" max_runtime=900 max_auto_delay_sec=604800 max_changes_past_7days=10 review_period_no_days=30 max_sla_percentage=90'
                    report_properties = {
                        "description": "This scheduled report manages adaptive delay tresholds for TrackMe",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # delayed entities inspector
                    #

                    report_name = f"trackme_dsm_delayed_entities_inspector_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkfeedsdelayedinspector tenant_id="{tenant_id}" component="dsm" max_runtime=900'
                    report_properties = {
                        "description": "This scheduled report manages delayed entities in the dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_dsm_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="dsm"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_dsm_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=dsm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_dsm_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=dsm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #########################
            # Flex objects tracking #
            #########################

            # step: Create the FLX virtual tenant knowledge objects if enabled
            if tenant_flx_enabled:
                for object_name in collections_list_flx:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                #
                # inactive entities tracker
                #

                if not tenant_replica:
                    report_name = (
                        f"trackme_flx_inactive_entities_tracker_tenant_{tenant_id}"
                    )
                    report_search = f'| trackmesplkflxinactiveinspector tenant_id="{tenant_id}" register_component="True" report="{report_name}" max_days_since_inactivity_before_purge=30'
                    report_properties = {
                        "description": "This scheduled report handles inactive entities for the splk-flx component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }

                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)

                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_flx_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="flx"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_flx_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="flx" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_flx_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="flx"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-flx component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_flx_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=flx'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-flx component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_flx_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=flx'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-flx component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #############################
            # Fields Quality Monitoring #
            #############################

            # step: Create the FQM virtual tenant knowledge objects if enabled
            if tenant_fqm_enabled:
                for object_name in collections_list_fqm:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                #
                # inactive entities tracker
                #

                if not tenant_replica:
                    report_name = (
                        f"trackme_fqm_inactive_entities_tracker_tenant_{tenant_id}"
                    )
                    report_search = f'| trackmesplkfqminactiveinspector tenant_id="{tenant_id}" register_component="True" report="{report_name}" max_days_since_inactivity_before_purge=30'
                    report_properties = {
                        "description": "This scheduled report handles inactive entities for the splk-fqm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }

                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)

                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_fqm_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="fqm"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_fqm_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="fqm" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_fqm_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="fqm"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-fqm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_fqm_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=fqm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-fqm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_fqm_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=fqm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-fqm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #########################
            # Splunk Workload       #
            #########################

            # step: Create the WLK virtual tenant knowledge objects if enabled
            if tenant_wlk_enabled:
                for object_name in collections_list_wlk:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # Step: create the set_status macro
                macro_name = f"trackme_wlk_set_status_tenant_{tenant_id}"
                macro_definition = f"""
                    lookup local=t trackme_wlk_orphan_status_tenant_{tenant_id} object OUTPUT orphan, mtime as orphan_last_check | eval orphan_last_check=case(isnotnull(orphan_last_check), strftime(orphan_last_check, "%c"))
                    | lookup local=t trackme_wlk_versioning_tenant_{tenant_id} object OUTPUT cron_exec_sequence_sec
                    ``` init a status 1```
                    | eval status=1
                    ``` If there are execution errors detected, status=2, we use periods data from 60m to 4h to 24h, the JSON metrics will not contain the metric if it equals to 0 ```
                    ``` Therefore, if a given search generating errors if fixed and has frequent executions, it likely will turn green in the next 60m from the deployment of the fix ```
                    | eval status=case(
                    count_errors_last_60m=0, status,
                    count_errors_last_4h=0, status,
                    count_errors_last_24h=0, status,
                    count_errors_last_60m>0 OR count_errors_last_4h>0 OR count_errors_last_24h>0, 2,
                    1=1, status
                    )
                    ``` If there are skipping searches, define two levels of alerting, less than 5% is 3 (orange), more is 2 (red) ```
                    ``` we base the calculation over the 24 period (suffix last_24h) - this can be customised up to your preferences if you wish to used the additional periods ```
                    | eval status=case(
                    isnum(skipped_pct_last_24h) AND skipped_pct_last_24h>0 AND skipped_pct_last_24h<5, 3, isnum(skipped_pct_last_24h) AND skipped_pct_last_60m>0 AND skipped_pct_last_24h>=5, 2,
                    1=1, status
                    )
                    ``` If we detected the search as an orphan search (not period related) ```
                    | eval status=if(orphan=1, 2, status)
                    ``` Calculate the delta in sequence between now and the last execution compared against the requested cron schedule sequence, add 1h of grace time, detect if the execution has been delayed ```
                    | eval status=if(cron_exec_sequence_sec>0 AND ( now()-last_seen > (cron_exec_sequence_sec + 3600) ), 2, status)
                    ``` Set a brief status description, a more granular description will be provided with the anomaly_reason and status_message fields ```
                    | eval status_description=case(status=1, "normal", status=2, "degraded", status=3, "warning", 1=1, "unknown")
                    """

                macro_acl = {
                    "owner": str(tenant_owner),
                    "sharing": trackme_default_sharing,
                    "perms.write": str(tenant_roles_write_perms),
                    "perms.read": str(tenant_roles_read_perms),
                }
                try:
                    trackme_create_macro(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        macro_name,
                        remove_leading_spaces(macro_definition),
                        tenant_owner,
                        macro_acl,
                    )
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # Step: create the outliers_metrics macro
                macro_name = "trackme_wlk_set_outliers_metrics_tenant_" + str(tenant_id)
                macro_definition = f"""{tenant_wlk_outliers_metrics}"""

                macro_acl = {
                    "owner": str(tenant_owner),
                    "sharing": trackme_default_sharing,
                    "perms.write": str(tenant_roles_write_perms),
                    "perms.read": str(tenant_roles_read_perms),
                }
                try:
                    trackme_create_macro(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        macro_name,
                        macro_definition,
                        tenant_owner,
                        macro_acl,
                    )
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # not for replica tenants
                if not tenant_replica:
                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_wlk_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="wlk"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_wlk_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="wlk" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_wlk_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="wlk"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-wlk component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_wlk_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=wlk'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-wlk component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_wlk_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=wlk'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-wlk component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            ##################
            # DHM
            ##################

            # step: Create the DHM virtual tenant knowledge objects if enabled
            if tenant_dhm_enabled:
                for object_name in collections_list_dhm:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # step: Create dependencies macros

                # alerting policy
                macro_name = (
                    "trackme_dhm_default_splk_dhm_alert_policy"
                    + "_tenant_"
                    + str(tenant_id)
                )
                macro_definition = '"' + str(tenant_dhm_alerting_policy) + '"'
                macro_acl = {
                    "owner": str(tenant_owner),
                    "sharing": trackme_default_sharing,
                    "perms.write": str(tenant_roles_write_perms),
                    "perms.read": str(tenant_roles_read_perms),
                }
                try:
                    trackme_create_macro(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        macro_name,
                        macro_definition,
                        tenant_owner,
                        macro_acl,
                    )
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # not for replica tenants
                if not tenant_replica:
                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_dhm_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="dhm"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_dhm_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="dhm" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # adaptive delay treshold
                    #

                    report_name = (
                        f"trackme_dhm_adaptive_delay_tracker_tenant_{tenant_id}"
                    )
                    report_search = f'| trackmesplkadaptivedelay tenant_id="{tenant_id}" component="dhm" min_delay_sec=3600 min_historical_metrics_days=7 earliest_time_mstats="-30d" max_runtime=900 max_auto_delay_sec=604800 max_changes_past_7days=10 review_period_no_days=30 max_sla_percentage=90'
                    report_properties = {
                        "description": "This scheduled report manages adaptive delay tresholds for TrackMe",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # delayed entities inspector
                    #

                    report_name = f"trackme_dhm_delayed_entities_inspector_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkfeedsdelayedinspector tenant_id="{tenant_id}" component="dhm" max_runtime=900'
                    report_properties = {
                        "description": "This scheduled report manages delayed entities in the dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_dhm_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="dhm"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_dhm_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=dhm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_dhm_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=dhm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            ##################
            # MHM
            ##################

            # step: Create the MHM virtual tenant knowledge objects if enabled
            if tenant_mhm_enabled:
                for object_name in collections_list_mhm:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # not for replica tenants
                if not tenant_replica:

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_mhm_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="mhm"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-mhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_mhm_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=mhm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-mhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_mhm_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=mhm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-mhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #
            # COMMON
            #

            for object_name in collections_list_common:
                # step: create the KVstore collections

                collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                collection_acl = {
                    "owner": str(tenant_owner),
                    "sharing": trackme_default_sharing,
                    "perms.write": str(tenant_roles_write_perms),
                    "perms.read": str(tenant_roles_read_perms),
                }
                try:
                    trackme_create_kvcollection(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        collection_name,
                        collection_acl,
                    )
                    result = {
                        "object": collection_name,
                        "object_type": "kvstore collection",
                        "action": "update rbac",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": collection_name,
                        "object_type": "kvstore collection",
                        "action": "update rbac",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # step: create the KVstore transforms

                transform_name = object_name + "_tenant_" + str(tenant_id)
                transform_fields = collections_dict[object_name]
                transform_acl = {
                    "owner": str(tenant_owner),
                    "sharing": trackme_default_sharing,
                    "perms.write": str(tenant_roles_write_perms),
                    "perms.read": str(tenant_roles_read_perms),
                }
                try:
                    trackme_create_kvtransform(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        transform_name,
                        transform_fields,
                        collection_name,
                        tenant_owner,
                        transform_acl,
                    )
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "create",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "create",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

            #
            # Tenant health tracker
            #

            report_name = "trackme_health_tracker_tenant_" + str(tenant_id)
            report_search = '| trackmetrackerhealth tenant_id="' + str(tenant_id) + '"'
            report_properties = {
                "description": "This scheduled report tracks health status for the tenant",
                "is_scheduled": True,
                "cron_schedule": "*/5 * * * *",
                "dispatch.earliest_time": "-5m",
                "dispatch.latest_time": "now",
                "schedule_window": "5",
            }

            try:
                report_acl = {
                    "owner": str(tenant_owner),
                    "sharing": trackme_default_sharing,
                    "perms.write": str(tenant_roles_write_perms),
                    "perms.read": str(tenant_roles_read_perms),
                }
                trackme_create_report(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    report_name,
                    report_search,
                    report_properties,
                    report_acl,
                )
                result = {
                    "object": report_name,
                    "object_type": "report",
                    "action": "create",
                    "result": "success",
                }
                results_record.append(result)

            except Exception as e:
                result = {
                    "object": report_name,
                    "object_type": "report",
                    "action": "create",
                    "result": "failure",
                    "exception": str(e),
                }
                results_record.append(result)

            #######################
            # END OF KOs CREATION
            #######################

            # Insert the record
            # Build tenant record dict
            tenant_record = {
                "_key": str(tenant_sha256),
                "tenant_name": tenant_name,
                "tenant_id": tenant_id,
                "tenant_desc": tenant_desc,
                "tenant_dsm_enabled": tenant_dsm_enabled,
                "tenant_dhm_enabled": tenant_dhm_enabled,
                "tenant_mhm_enabled": tenant_mhm_enabled,
                "tenant_flx_enabled": tenant_flx_enabled,
                "tenant_fqm_enabled": tenant_fqm_enabled,
                "tenant_wlk_enabled": tenant_wlk_enabled,
                "tenant_owner": tenant_owner,
                "tenant_roles_admin": tenant_roles_admin,
                "tenant_roles_power": tenant_roles_power,
                "tenant_roles_user": tenant_roles_user,
                "tenant_status": "enabled",
                "tenant_idx_settings": json.dumps(
                    tenant_idx_settings, indent=4
                ),
                "tenant_replica": tenant_replica,
            }
            
            # Only set schema_version if it's valid (not 0)
            # If schema_version_current is 0, it means version retrieval failed.
            # In that case, we skip setting schema_version and let the upgrade logic
            # handle it later when version retrieval succeeds.
            if schema_version_current and schema_version_current != 0:
                tenant_record["schema_version"] = schema_version_current
            else:
                logger.warning(
                    f'schema_version_current is {schema_version_current} (version retrieval failed), '
                    f'skipping schema_version in tenant record for tenant_id="{tenant_id}". '
                    f'The upgrade logic will set it later when version retrieval succeeds.'
                )
            
            collection.data.insert(json.dumps(tenant_record))

            # Get record
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]

            # Define an header for requests authenticated communications with splunkd
            header = {
                "Authorization": "Splunk %s" % request_info.system_authtoken,
                "Content-Type": "application/json",
            }

            # Add the vtenant account
            url = "%s/servicesNS/nobody/trackme/trackme_vtenants" % (
                request_info.server_rest_uri
            )
            vtenant_data = {
                "name": tenant_id,
                "alias": tenant_alias,
                "description": tenant_desc,
                "default_priority": tenant_default_priority,
                "ui_default_timerange": "24h",
                "ui_min_object_width": 300,
                "ui_expand_metrics": 0,
                "ui_home_tabs_order": "dsm,flx,dhm,mhm,wlk,fqm,flip,audit,alerts",
                "data_sampling_obfuscation": tenant_sampling_obfuscation,
                "adaptive_delay": tenant_adaptive_delay,
                "sampling": tenant_sampling,
                "mloutliers": tenant_mloutliers,
                "mloutliers_allowlist": tenant_mloutliers_allowlist,
                "indexed_constraint": tenant_indexed_constraint,
                "splk_feeds_delayed_inspector_24hours_range_min_sec": tenant_splk_feeds_delayed_inspector_24hours_range_min_sec,
                "splk_feeds_delayed_inspector_7days_range_min_sec": tenant_splk_feeds_delayed_inspector_7days_range_min_sec,
                "splk_feeds_delayed_inspector_until_disabled_range_min_sec": tenant_splk_feeds_delayed_inspector_until_disabled_range_min_sec,
                "splk_feeds_auto_disablement_period": tenant_splk_feeds_auto_disablement_period,
                "cmdb_lookup": tenant_cmdb_lookup,                
                "pagination_mode": tenant_pagination_mode,
                "pagination_size": tenant_pagination_size,
                "splk_dsm_tabulator_groupby": tenant_splk_dsm_tabulator_groupby,
                "splk_dhm_tabulator_groupby": tenant_splk_dhm_tabulator_groupby,
                "splk_mhm_tabulator_groupby": tenant_splk_mhm_tabulator_groupby,
                "splk_flx_tabulator_groupby": tenant_splk_flx_tabulator_groupby,
                "splk_fqm_tabulator_groupby": tenant_splk_fqm_tabulator_groupby,
                "splk_wlk_tabulator_groupby": tenant_splk_wlk_tabulator_groupby,
                "default_disruption_min_time_sec": tenant_default_disruption_min_time_sec,
                "monitoring_time_policy": tenant_default_monitoring_time_policy,
            }

            # optional keys for CMDB, if the value is set and non empty, adds to the vtenant_data
            if tenant_splk_dsm_cmdb_search and tenant_splk_dsm_cmdb_search != "":
                vtenant_data["splk_dsm_cmdb_search"] = tenant_splk_dsm_cmdb_search
            if tenant_splk_dhm_cmdb_search and tenant_splk_dhm_cmdb_search != "":
                vtenant_data["splk_dhm_cmdb_search"] = tenant_splk_dhm_cmdb_search
            if tenant_splk_mhm_cmdb_search and tenant_splk_mhm_cmdb_search != "":
                vtenant_data["splk_mhm_cmdb_search"] = tenant_splk_mhm_cmdb_search
            if tenant_splk_flx_cmdb_search and tenant_splk_flx_cmdb_search != "":
                vtenant_data["splk_flx_cmdb_search"] = tenant_splk_flx_cmdb_search
            if tenant_splk_fqm_cmdb_search and tenant_splk_fqm_cmdb_search != "":
                vtenant_data["splk_fqm_cmdb_search"] = tenant_splk_fqm_cmdb_search
            if tenant_splk_wlk_cmdb_search and tenant_splk_wlk_cmdb_search != "":
                vtenant_data["splk_wlk_cmdb_search"] = tenant_splk_wlk_cmdb_search

            # Extract and validate impact score configuration fields from request
            # These fields are optional and will use defaults from vtenant_account_default if not provided
            for field_name in vtenant_account_default.keys():
                if field_name.startswith("impact_score_"):
                    if field_name in resp_dict:
                        try:
                            # Validate value is between 0 and 100
                            int_value = int(resp_dict[field_name])
                            if int_value < 0 or int_value > 100:
                                return {
                                    "payload": {
                                        "response": f'Invalid value for {field_name}: {resp_dict[field_name]}. Value must be between 0 and 100.',
                                    },
                                    "status": 400,
                                }
                            vtenant_data[field_name] = str(int_value)
                            logger.debug(
                                f'Added impact score field "{field_name}"="{int_value}" to vtenant_data for tenant_id="{tenant_id}"'
                            )
                        except ValueError:
                            return {
                                "payload": {
                                    "response": f'Invalid value for {field_name}: {resp_dict[field_name]}. Value must be an integer.',
                                },
                                "status": 400,
                            }

            logger.debug(f'before keys and value check, vtenant_data="{json.dumps(vtenant_data, indent=2)}"')

            #
            # protect the system for Outliers and ML training with low customer qualification
            #

            # if tenant_dhm_enabled is the only component enabled, disable the mloutliers
            if tenant_dsm_enabled or tenant_dhm_enabled or tenant_mhm_enabled:
                if not tenant_dsm_enabled:
                    vtenant_data["mloutliers"] = 0

            # for key value in vtenant_account_default, check if a key is missing from vtenant_data, if so, add it
            for key, value in vtenant_account_default.items():
                if key not in vtenant_data:
                    vtenant_data[key] = value

            # for key value in vetant_data, if the value is null, empty or None, take the value from vtenant_account_default
            for key, value in vtenant_data.items():
                if value is None:
                    vtenant_data[key] = vtenant_account_default.get(key)
                elif isinstance(value, str) and value == "":
                    vtenant_data[key] = vtenant_account_default.get(key)

            # Retrieve and set the tenant idx, if any failure, logs and use the global index
            try:
                logger.debug(f'before rest call url="{url}", vtenant_data="{json.dumps(vtenant_data, indent=2)}"')
                response = requests.post(
                    url, headers=header, data=vtenant_data, verify=False, timeout=600
                )
                if response.status_code not in (200, 201, 204):
                    logger.error(
                        f'create vtenant account has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logger.info(
                        f'create vtenant account was operated successfully, response.status_code="{response.status_code}"'
                    )
            except Exception as e:
                logger.error(f'create vtenant account has failed, exception="{str(e)}"')

            # review results record and count failures
            failures_count = 0
            for result_record in results_record:
                try:
                    result_action = result_record.get("result")
                    if result_action == "failure":
                        failures_count += 1
                except Exception as e:
                    logger.error(
                        f'count not find expected action result field in record="{json.dumps(result_record, indent=2)}"'
                    )

            # handle
            if failures_count > 0:
                # Record an audit change
                user = request_info.user
                result = "One or more exceptions were encountered during the creation of the tenant, review the results_details as needed"
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "failure",
                    "change_type": "add new virtual tenant",
                    "object": str(tenant_name),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "create virtual tenant",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 500}

            else:
                # Record an audit change
                user = request_info.user
                result = "All objects for this tenant were successfully created"
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "success",
                    "change_type": "add new virtual tenant",
                    "object": str(tenant_name),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "create virtual tenant",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                # Attempt to execute the health tracker immediately to allow the tenant
                # to be immediately active upon its creation

                savedsearch = "trackme_health_tracker_tenant_" + str(tenant_id)
                search = "| savedsearch " + savedsearch
                kwargs_oneshot = {
                    "earliest_time": "-5m",
                    "latest_time": "now",
                    "output_mode": "json",
                    "count": 0,
                }

                logger.info(
                    f'tenant_id="{tenant_id}", Attempting to execute the health tracker="{savedsearch}"'
                )
                try:
                    reader = run_splunk_search(
                        service,
                        search,
                        kwargs_oneshot,
                        24,
                        5,
                    )

                    for item in reader:
                        if isinstance(item, dict):
                            logger.info(
                                f'tenant_id="{tenant_id}", health tracker="{savedsearch}", execution was successful, results="{json.dumps(json.loads(item.get("_raw")), indent=2)}"'
                            )

                except Exception as e:
                    logger.info(
                        f'tenant_id="{tenant_id}", health tracker="{savedsearch}", execution has failed with exception="{str(e)}"'
                    )

                # Return
                return {"payload": audit_record, "status": 200}

    #
    # Update an existing tenant to add a component
    #

    def post_add_component_tenant(self, request_info, **kwargs):
        # Declare main variables
        tenant_id = None
        component_target = None
        query_string = None
        tenant_replica = False

        # Declare booleans
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                component_target = resp_dict["component_target"]

            # At least one component needs to be enabled
            if not describe:
                if not component_target in (
                    "dsm",
                    "dhm",
                    "mhm",
                    "flx",
                    "wlk",
                    "fqm",
                ):
                    return {
                        "payload": {"response": "Error, invalid component."},
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be called to add a component to an existing virtual tenant, it requires a POST call with the following information:",
                "resource_desc": "Update a virtual tenant",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/vtenants/admin/add_component_tenant\" mode=\"post\" body=\"{ 'tenant_id': 'mytenant', 'component_target': 'dhm' }\""
                + "'update_comment': 'Updated for the purpose of the documentation.'}\"",
                "options": [
                    {
                        "tenant_id": "ID of the virtual tenant to be updated",
                        "component_target": "The component to be enabled, valid options: dsm|dhm|mhm|flx|wlk|fqm",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
            if update_comment == "Enter your audit message":
                update_comment = "No comment was provided for this operation"
        except Exception as e:
            update_comment = "No comment was provided for this operation"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        # TrackMe sharing level
        trackme_default_sharing = trackme_conf["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        # Data collection
        collection_name = "kv_trackme_virtual_tenants"
        collection = service.kvstore[collection_name]

        ################
        # MAIN PROGRAM #
        ################

        # get trackme release
        trackme_version = trackme_get_version(service)
        logger.info(f'running trackme version="{trackme_version}"')

        # check license state
        try:
            check_license = trackme_check_license(
                request_info.server_rest_uri, request_info.session_key
            )
            license_is_valid = check_license.get("license_is_valid")
            license_subscription_class = check_license.get("license_subscription_class")
            logger.debug(
                f'function check_license called, response="{json.dumps(check_license, indent=2)}"'
            )

        except Exception as e:
            license_is_valid = 0
            license_subscription_class = "free"
            logger.error(f'function check_license exception="{str(e)}"')

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        try:
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]
            key = vtenant_record.get("_key")
            tenant_dsm_enabled = int(vtenant_record.get("tenant_dsm_enabled"))
            tenant_flx_enabled = int(vtenant_record.get("tenant_flx_enabled"))
            tenant_fqm_enabled = int(vtenant_record.get("tenant_fqm_enabled"))
            tenant_dhm_enabled = int(vtenant_record.get("tenant_dhm_enabled"))
            tenant_mhm_enabled = int(vtenant_record.get("tenant_mhm_enabled"))
            tenant_wlk_enabled = int(vtenant_record.get("tenant_wlk_enabled"))
            tenant_owner = str(vtenant_record.get("tenant_owner"))
            tenant_roles_admin = str(vtenant_record.get("tenant_roles_admin"))
            tenant_roles_user = str(vtenant_record.get("tenant_roles_user"))
            tenant_roles_power = str(vtenant_record.get("tenant_roles_power"))
            tenant_replica = interpret_boolean(vtenant_record.get("tenant_replica"))

        except Exception as e:
            return {
                "payload": {
                    "response": "Sorry, I am afraid I can't do that, this tenant does not exist yet."
                },
                "status": 500,
            }

        # for read permissions, concatenate admin, power and user
        tenant_roles_read_perms = (
            f"{tenant_roles_admin},{tenant_roles_power},{tenant_roles_user}"
        )

        # for write permissions, concatenate admin, power
        tenant_roles_write_perms = f"{tenant_roles_admin},{tenant_roles_power}"

        # Restricted components
        restricted_components = ["flx", "wlk", "fqm"]
        license_error_messages = {
            0: "I'm afraid I can't do that, this instance is running Free limited mode and one or more of the components you have selected are restricted to the Full or Trial edition.",
            1: "I'm afraid I can't do that, this instance is running Free extended mode and one or more of the components you have selected are restricted to the Full or Trial edition.",
        }

        license_restriction = (license_is_valid != 1) or (
            license_is_valid == 1 and license_subscription_class == "free_extended"
        )

        if license_restriction and component_target in restricted_components:
            audit_record = {
                "action": "failure",
                "change_type": "update virtual tenant",
                "object": str(tenant_id),
                "object_category": "virtual_tenants",
                "result": license_error_messages[license_is_valid],
            }

            logger.error(str(audit_record))
            return {"payload": audit_record, "status": 402}

        else:
            # Verify the operation requested
            component_enabled_map = {
                "dsm": tenant_dsm_enabled,
                "dhm": tenant_dhm_enabled,
                "mhm": tenant_mhm_enabled,
                "flx": tenant_flx_enabled,
                "fqm": tenant_fqm_enabled,
                "wlk": tenant_wlk_enabled,
            }

            if component_enabled_map.get(component_target) == 1:
                audit_record = {
                    "action": "failure",
                    "change_type": "add virtual tenant component",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "result": "I'm afraid I can't do that, the requested component is already enabled for this tenant.",
                }

                logger.error(str(audit_record))
                return {"payload": audit_record, "status": 409}

            # operation counters
            results_record = []

            # check if TCM is enabled in receiver mode
            enable_conf_manager_receiver = int(
                trackme_conf["trackme_conf"]["trackme_general"][
                    "enable_conf_manager_receiver"
                ]
            )

            if enable_conf_manager_receiver == 1:
                try:
                    tcm_response = trackme_send_to_tcm(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        resp_dict,
                        "post",
                        "/services/trackme/v2/vtenants/admin/add_component_tenant",
                    )
                    logger.info(f"trackme_send_to_tcm was successfully executed")
                except Exception as e:
                    logger.error(
                        f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                    )

            ################################
            # Data Source Monitoring (DSM) #
            ################################

            # step: Create the DSM virtual tenant knowledge objects if enabled
            if component_target in "dsm":
                for object_name in collections_list_dsm:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # not for replica tenants
                if not tenant_replica:
                    # step: create the DSM Data Sampling tracker

                    tenant_id_str = str(tenant_id)
                    report_name = (
                        f"trackme_dsm_data_sampling_tracker_tenant_{tenant_id_str}"
                    )

                    report_search = (
                        f'| trackmesamplingexecutor tenant_id="{tenant_id_str}"'
                    )
                    report_properties = {
                        "description": "TrackMe DSM Data Sampling tracker",
                        "is_scheduled": True,
                        "schedule_window": "5",
                        "cron_schedule": "*/20 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-24h",
                        "dispatch.latest_time": "-4h",
                    }

                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the DSM Shared Elastic tracker

                    report_name = "trackme_dsm_shared_elastic_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmeelasticexecutor tenant_id="'
                        + str(tenant_id)
                        + '" component="splk-dsm"'
                    )
                    report_properties = {
                        "description": "TrackMe DSM shared elastic tracker",
                        "is_scheduled": True,
                        "schedule_window": "5",
                        "cron_schedule": "*/5 * * * *",
                        "dispatch.earliest_time": "-4h",
                        "dispatch.latest_time": "+4h",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_dsm_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="dsm"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_dsm_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="dsm" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # adaptive delay treshold
                    #

                    report_name = (
                        f"trackme_dsm_adaptive_delay_tracker_tenant_{tenant_id}"
                    )
                    report_search = f'| trackmesplkadaptivedelay tenant_id="{tenant_id}" component="dsm" min_delay_sec=3600 min_historical_metrics_days=7 earliest_time_mstats="-30d" max_runtime=900 max_auto_delay_sec=604800 max_changes_past_7days=10 review_period_no_days=30 max_sla_percentage=90'
                    report_properties = {
                        "description": "This scheduled report manages adaptive delay tresholds for TrackMe",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # delayed entities inspector
                    #

                    report_name = f"trackme_dsm_delayed_entities_inspector_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkfeedsdelayedinspector tenant_id="{tenant_id}" component="dsm" max_runtime=900'
                    report_properties = {
                        "description": "This scheduled report manages delayed entities in the dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_dsm_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="dsm"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_dsm_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=dsm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_dsm_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=dsm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #########################
            # Flex objects tracking #
            #########################

            # step: Create the FLX virtual tenant knowledge objects if enabled
            if component_target in "flx":
                for object_name in collections_list_flx:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # not for replica tenants
                if not tenant_replica:
                    #
                    # inactive entities tracker
                    #

                    report_name = (
                        f"trackme_flx_inactive_entities_tracker_tenant_{tenant_id}"
                    )
                    report_search = f'| trackmesplkflxinactiveinspector tenant_id="{tenant_id}" register_component="True" report="{report_name}" max_days_since_inactivity_before_purge=30'
                    report_properties = {
                        "description": "This scheduled report handles inactive entities for the splk-flx component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }

                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)

                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_flx_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="flx"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_flx_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="flx" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_flx_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="flx"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-flx component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_flx_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=flx'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-flx component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_flx_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=flx'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-flx component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #############################
            # Fields Quality Monitoring #
            #############################

            # step: Create the FQM virtual tenant knowledge objects if enabled
            if component_target in "fqm":
                for object_name in collections_list_fqm:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # not for replica tenants
                if not tenant_replica:
                    #
                    # inactive entities tracker
                    #

                    report_name = (
                        f"trackme_fqm_inactive_entities_tracker_tenant_{tenant_id}"
                    )
                    report_search = f'| trackmesplkfqminactiveinspector tenant_id="{tenant_id}" register_component="True" report="{report_name}" max_days_since_inactivity_before_purge=30'
                    report_properties = {
                        "description": "This scheduled report handles inactive entities for the splk-fqm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }

                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)

                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_fqm_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="fqm"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_fqm_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="fqm" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_fqm_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="fqm"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-fqm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_fqm_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=fqm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-fqm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_fqm_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=fqm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-fqm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #########################
            # Workload              #
            #########################

            # step: Create the WLK virtual tenant knowledge objects if enabled
            if component_target in "wlk":
                for object_name in collections_list_wlk:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # Workload has some specific objects which are normally created while setting up the tenant

                # Set the outliers to elapsed,
                tenant_wlk_outliers_metrics = "eval outliers_metrics=\"{'elapsed': {'alert_lower_breached': 0, 'alert_upper_breached': 1, 'time_factor': 'none'}}\""

                # Step: create the set_status macro
                macro_name = f"trackme_wlk_set_status_tenant_{tenant_id}"
                macro_definition = f"""
                    lookup local=t trackme_wlk_orphan_status_tenant_{tenant_id} object OUTPUT orphan, mtime as orphan_last_check | eval orphan_last_check=case(isnotnull(orphan_last_check), strftime(orphan_last_check, "%c"))
                    | lookup local=t trackme_wlk_versioning_tenant_{tenant_id} object OUTPUT cron_exec_sequence_sec
                    ``` init a status 1```
                    | eval status=1
                    ``` If there are execution errors detected, status=2, we use periods data from 60m to 4h to 24h, the JSON metrics will not contain the metric if it equals to 0 ```
                    ``` Therefore, if a given search generating errors if fixed and has frequent executions, it likely will turn green in the next 60m from the deployment of the fix ```
                    | eval status=case(
                    count_errors_last_60m=0, status,
                    count_errors_last_4h=0, status,
                    count_errors_last_24h=0, status,
                    count_errors_last_60m>0 OR count_errors_last_4h>0 OR count_errors_last_24h>0, 2,
                    1=1, status
                    )
                    ``` If there are skipping searches, define two levels of alerting, less than 5% is 3 (orange), more is 2 (red) ```
                    ``` we base the calculation over the 24 period (suffix last_24h) - this can be customised up to your preferences if you wish to used the additional periods ```
                    | eval status=case(
                    isnum(skipped_pct_last_24h) AND skipped_pct_last_24h>0 AND skipped_pct_last_24h<5, 3, isnum(skipped_pct_last_24h) AND skipped_pct_last_60m>0 AND skipped_pct_last_24h>=5, 2,
                    1=1, status
                    )
                    ``` If we detected the search as an orphan search (not period related) ```
                    | eval status=if(orphan=1, 2, status)
                    ``` Calculate the delta in sequence between now and the last execution compared against the requested cron schedule sequence, add 1h of grace time, detect if the execution has been delayed ```
                    | eval status=if(cron_exec_sequence_sec>0 AND ( now()-last_seen > (cron_exec_sequence_sec + 3600) ), 2, status)
                    ``` Set a brief status description, a more granular description will be provided with the anomaly_reason and status_message fields ```
                    | eval status_description=case(status=1, "normal", status=2, "degraded", status=3, "warning", 1=1, "unknown")
                    """

                macro_acl = {
                    "owner": str(tenant_owner),
                    "sharing": trackme_default_sharing,
                    "perms.write": str(tenant_roles_write_perms),
                    "perms.read": str(tenant_roles_read_perms),
                }
                try:
                    trackme_create_macro(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        macro_name,
                        macro_definition,
                        tenant_owner,
                        macro_acl,
                    )
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # Step: create the outliers_metrics macro
                macro_name = "trackme_wlk_set_outliers_metrics_tenant_" + str(tenant_id)
                macro_definition = f"""{tenant_wlk_outliers_metrics}"""

                macro_acl = {
                    "owner": str(tenant_owner),
                    "sharing": trackme_default_sharing,
                    "perms.write": str(tenant_roles_write_perms),
                    "perms.read": str(tenant_roles_read_perms),
                }
                try:
                    trackme_create_macro(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        macro_name,
                        macro_definition,
                        tenant_owner,
                        macro_acl,
                    )
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # not for replica tenants
                if not tenant_replica:
                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_wlk_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="wlk"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_wlk_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="wlk" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_wlk_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="wlk"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-wlk component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_wlk_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=wlk'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-wlk component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_wlk_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=wlk'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-wlk component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            ##################
            # DHM
            ##################

            # step: Create the DHM virtual tenant knowledge objects if enabled
            if component_target in "dhm":
                for object_name in collections_list_dhm:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # step: Create dependencies macros

                # alerting policy
                macro_name = (
                    "trackme_dhm_default_splk_dhm_alert_policy"
                    + "_tenant_"
                    + str(tenant_id)
                )
                macro_definition = '"track_per_host"'
                macro_acl = {
                    "owner": str(tenant_owner),
                    "sharing": trackme_default_sharing,
                    "perms.write": str(tenant_roles_write_perms),
                    "perms.read": str(tenant_roles_read_perms),
                }
                try:
                    trackme_create_macro(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        macro_name,
                        macro_definition,
                        tenant_owner,
                        macro_acl,
                    )
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": macro_name,
                        "object_type": "macro",
                        "action": "create",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # not for replica tenants
                if not tenant_replica:
                    #
                    # Splk-Outliers
                    #

                    #
                    # ML training report
                    #

                    report_name = "trackme_dhm_outliers_mltrain_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmesplkoutlierstrainhelper tenant_id="'
                        + str(tenant_id)
                        + '" component="dhm"'
                    )
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # ML monitor report
                    #

                    report_name = (
                        "trackme_dhm_outliers_mlmonitor_tracker_tenant_"
                        + str(tenant_id)
                    )
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="dhm" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # adaptive delay treshold
                    #

                    report_name = (
                        f"trackme_dhm_adaptive_delay_tracker_tenant_{tenant_id}"
                    )
                    report_search = f'| trackmesplkadaptivedelay tenant_id="{tenant_id}" component="dhm" min_delay_sec=3600 min_historical_metrics_days=7 earliest_time_mstats="-30d" max_runtime=900 max_auto_delay_sec=604800 max_changes_past_7days=10 review_period_no_days=30 max_sla_percentage=90'
                    report_properties = {
                        "description": "This scheduled report manages adaptive delay tresholds for TrackMe",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # delayed entities inspector
                    #

                    report_name = f"trackme_dhm_delayed_entities_inspector_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkfeedsdelayedinspector tenant_id="{tenant_id}" component="dhm" max_runtime=900'
                    report_properties = {
                        "description": "This scheduled report manages delayed entities in the dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_dhm_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="dhm"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_dhm_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=dhm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_dhm_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=dhm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            ##################
            # MHM
            ##################

            # step: Create the MHM virtual tenant knowledge objects if enabled
            if component_target in "mhm":
                for object_name in collections_list_mhm:
                    # step: create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            collection_acl,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # step: create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    try:
                        trackme_create_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            transform_fields,
                            collection_name,
                            tenant_owner,
                            transform_acl,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # tags tracker
                    #

                    report_name = f"trackme_mhm_tags_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplktags tenant_id="{tenant_id}" component="mhm"'
                    )
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-mhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # priority tracker
                    #

                    report_name = f"trackme_mhm_priority_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkpriority tenant_id="{tenant_id}" component=mhm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-mhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    #
                    # sla tracker
                    #

                    report_name = f"trackme_mhm_sla_tracker_tenant_{tenant_id}"
                    report_search = (
                        f'| trackmesplkslaclass tenant_id="{tenant_id}" component=mhm'
                    )
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-mhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/15 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    try:
                        report_acl = {
                            "owner": str(tenant_owner),
                            "sharing": trackme_default_sharing,
                            "perms.write": str(tenant_roles_write_perms),
                            "perms.read": str(tenant_roles_read_perms),
                        }
                        trackme_create_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                            report_search,
                            report_properties,
                            report_acl,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "create",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #######################
            # END OF KOs CREATION
            #######################

            # Create a mapping dictionary
            component_target_mapping = {
                "dsm": "tenant_dsm_enabled",
                "dhm": "tenant_dhm_enabled",
                "mhm": "tenant_mhm_enabled",
                "flx": "tenant_flx_enabled",
                "fqm": "tenant_fqm_enabled",
                "wlk": "tenant_wlk_enabled",
            }

            # Update the record
            if component_target in component_target_mapping:
                vtenant_record[component_target_mapping[component_target]] = 1

            # perform
            collection.data.update(str(key), json.dumps(vtenant_record))

            # Get updated record
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]

            # review results record and count failures
            failures_count = 0
            for result_record in results_record:
                try:
                    result_action = result_record.get("result")
                    if result_action == "failure":
                        failures_count += 1
                except Exception as e:
                    logger.error(
                        f'count not find expected action result field in record="{json.dumps(result_record, indent=2)}"'
                    )

            # handle
            if failures_count > 0:
                # Record an audit change
                user = request_info.user
                result = "One or more exceptions were encountered during the creation of the component of this tenant, review the results_details as needed"
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "failure",
                    "change_type": "add virtual tenant component",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "add virtual tenant component",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 500}

            else:
                # Record an audit change
                user = request_info.user
                result = "All objects for this component of the tenant were successfully created"
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "success",
                    "change_type": "add virtual tenant component",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "add virtual tenant component",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                # Attempt to execute the health tracker immediately to allow the tenant
                # to be immediately active upon its creation

                savedsearch = "trackme_health_tracker_tenant_" + str(tenant_id)
                search = "| savedsearch " + savedsearch
                kwargs_oneshot = {
                    "earliest_time": "-5m",
                    "latest_time": "now",
                    "output_mode": "json",
                    "count": 0,
                }

                logger.info(
                    f'tenant_id="{tenant_id}", Attempting to execute the health tracker="{savedsearch}"'
                )
                try:
                    reader = run_splunk_search(
                        service,
                        search,
                        kwargs_oneshot,
                        24,
                        5,
                    )

                    for item in reader:
                        if isinstance(item, dict):
                            logger.info(
                                f'tenant_id="{tenant_id}", health tracker="{savedsearch}", execution was successful, results="{json.dumps(json.loads(item.get("_raw")), indent=2)}"'
                            )

                except Exception as e:
                    logger.info(
                        f'tenant_id="{tenant_id}", health tracker="{savedsearch}", execution has failed with exception="{str(e)}"'
                    )

                # Return
                return {"payload": audit_record, "status": 200}

    #
    # Check and create missing knowledge objects for a component
    #

    def post_check_component_tenant(self, request_info, **kwargs):
        # Declare main variables
        tenant_id = None
        component_target = None
        query_string = None
        tenant_replica = False

        # Declare booleans
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                component_target = resp_dict["component_target"]

            # At least one component needs to be enabled
            if not describe:
                if not component_target in (
                    "dsm",
                    "dhm",
                    "mhm",
                    "flx",
                    "wlk",
                    "fqm",
                ):
                    return {
                        "payload": {"response": "Error, invalid component."},
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be called to check and create missing knowledge objects for a component in an existing virtual tenant, it requires a POST call with the following information:",
                "resource_desc": "Check and create missing knowledge objects for a virtual tenant component",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/vtenants/admin/check_component_tenant\" mode=\"post\" body=\"{ 'tenant_id': 'mytenant', 'component_target': 'dhm' }\""
                + "'update_comment': 'Updated for the purpose of the documentation.'}\"",
                "options": [
                    {
                        "tenant_id": "ID of the virtual tenant to be checked",
                        "component_target": "The component to be checked, valid options: dsm|dhm|mhm|flx|wlk|fqm",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
            if update_comment == "Enter your audit message":
                update_comment = "No comment was provided for this operation"
        except Exception as e:
            update_comment = "No comment was provided for this operation"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        # TrackMe sharing level
        trackme_default_sharing = trackme_conf["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        # Data collection
        collection_name = "kv_trackme_virtual_tenants"
        collection = service.kvstore[collection_name]

        ################
        # MAIN PROGRAM #
        ################

        # get trackme release
        trackme_version = trackme_get_version(service)
        logger.info(f'running trackme version="{trackme_version}"')

        # check license state
        try:
            check_license = trackme_check_license(
                request_info.server_rest_uri, request_info.session_key
            )
            license_is_valid = check_license.get("license_is_valid")
            license_subscription_class = check_license.get("license_subscription_class")
            logger.debug(
                f'function check_license called, response="{json.dumps(check_license, indent=2)}"'
            )

        except Exception as e:
            license_is_valid = 0
            license_subscription_class = "free"
            logger.error(f'function check_license exception="{str(e)}"')

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        try:
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]
            key = vtenant_record.get("_key")
            tenant_dsm_enabled = int(vtenant_record.get("tenant_dsm_enabled"))
            tenant_flx_enabled = int(vtenant_record.get("tenant_flx_enabled"))
            tenant_fqm_enabled = int(vtenant_record.get("tenant_fqm_enabled"))
            tenant_dhm_enabled = int(vtenant_record.get("tenant_dhm_enabled"))
            tenant_mhm_enabled = int(vtenant_record.get("tenant_mhm_enabled"))
            tenant_wlk_enabled = int(vtenant_record.get("tenant_wlk_enabled"))
            tenant_owner = str(vtenant_record.get("tenant_owner"))
            tenant_roles_admin = str(vtenant_record.get("tenant_roles_admin"))
            tenant_roles_user = str(vtenant_record.get("tenant_roles_user"))
            tenant_roles_power = str(vtenant_record.get("tenant_roles_power"))
            tenant_replica = interpret_boolean(vtenant_record.get("tenant_replica"))

        except Exception as e:
            return {
                "payload": {
                    "response": "Sorry, I am afraid I can't do that, this tenant does not exist yet."
                },
                "status": 500,
            }

        # for read permissions, concatenate admin, power and user
        tenant_roles_read_perms = (
            f"{tenant_roles_admin},{tenant_roles_power},{tenant_roles_user}"
        )

        # for write permissions, concatenate admin, power
        tenant_roles_write_perms = f"{tenant_roles_admin},{tenant_roles_power}"

        # Restricted components
        restricted_components = ["flx", "wlk", "fqm"]
        license_error_messages = {
            0: "I'm afraid I can't do that, this instance is running Free limited mode and one or more of the components you have selected are restricted to the Full or Trial edition.",
            1: "I'm afraid I can't do that, this instance is running Free extended mode and one or more of the components you have selected are restricted to the Full or Trial edition.",
        }

        license_restriction = (license_is_valid != 1) or (
            license_is_valid == 1 and license_subscription_class == "free_extended"
        )

        if license_restriction and component_target in restricted_components:
            audit_record = {
                "action": "failure",
                "change_type": "check virtual tenant component",
                "object": str(tenant_id),
                "object_category": "virtual_tenants",
                "result": license_error_messages[license_is_valid],
            }

            logger.error(str(audit_record))
            return {"payload": audit_record, "status": 402}

        else:
            # Verify the operation requested
            component_enabled_map = {
                "dsm": tenant_dsm_enabled,
                "dhm": tenant_dhm_enabled,
                "mhm": tenant_mhm_enabled,
                "flx": tenant_flx_enabled,
                "fqm": tenant_fqm_enabled,
                "wlk": tenant_wlk_enabled,
            }

            if component_enabled_map.get(component_target) != 1:
                audit_record = {
                    "action": "failure",
                    "change_type": "check virtual tenant component",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "result": "I'm afraid I can't do that, the requested component is not enabled for this tenant.",
                }

                logger.error(str(audit_record))
                return {"payload": audit_record, "status": 409}

            # operation counters
            results_record = []

            # check if TCM is enabled in receiver mode
            enable_conf_manager_receiver = int(
                trackme_conf["trackme_conf"]["trackme_general"][
                    "enable_conf_manager_receiver"
                ]
            )

            if enable_conf_manager_receiver == 1:
                try:
                    tcm_response = trackme_send_to_tcm(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        resp_dict,
                        "post",
                        "/services/trackme/v2/vtenants/admin/check_component_tenant",
                    )
                    logger.info(f"trackme_send_to_tcm was successfully executed")
                except Exception as e:
                    logger.error(
                        f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                    )

            ################################
            # Data Source Monitoring (DSM) #
            ################################

            # step: Check and create the DSM virtual tenant knowledge objects if enabled
            if component_target in "dsm":
                for object_name in collections_list_dsm:
                    # step: check and create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    
                    # Check if collection exists
                    if trackme_check_kvcollection_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        collection_name,
                    ):
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the collection
                        try:
                            trackme_create_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                                collection_acl,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    
                    # Check if transform exists
                    if trackme_check_kvtransform_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        transform_name,
                    ):
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the transform
                        try:
                            trackme_create_kvtransform(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                transform_name,
                                transform_fields,
                                collection_name,
                                tenant_owner,
                                transform_acl,
                            )
                            result = {
                                "object": transform_name,
                                "object_type": "transform",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": transform_name,
                                "object_type": "transform",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                # not for replica tenants
                if not tenant_replica:
                    # step: check and create the DSM Data Sampling tracker

                    tenant_id_str = str(tenant_id)
                    report_name = (
                        f"trackme_dsm_data_sampling_tracker_tenant_{tenant_id_str}"
                    )

                    report_search = (
                        f'| trackmesamplingexecutor tenant_id="{tenant_id_str}"'
                    )
                    report_properties = {
                        "description": "TrackMe DSM Data Sampling tracker",
                        "is_scheduled": True,
                        "schedule_window": "5",
                        "cron_schedule": "*/20 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-24h",
                        "dispatch.latest_time": "-4h",
                    }

                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DSM Shared Elastic tracker

                    report_name = "trackme_dsm_shared_elastic_tracker_tenant_" + str(
                        tenant_id
                    )
                    report_search = (
                        '| trackmeelasticexecutor tenant_id="'
                        + str(tenant_id)
                        + '" component="splk-dsm"'
                    )
                    report_properties = {
                        "description": "TrackMe DSM shared elastic tracker",
                        "is_scheduled": True,
                        "schedule_window": "5",
                        "cron_schedule": "*/5 * * * *",
                        "dispatch.earliest_time": "-4h",
                        "dispatch.latest_time": "+4h",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DSM ML training report
                    report_name = "trackme_dsm_outliers_mltrain_tracker_tenant_" + str(tenant_id)
                    report_search = '| trackmesplkoutlierstrainhelper tenant_id="' + str(tenant_id) + '" component="dsm"'
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DSM ML monitor report
                    report_name = "trackme_dsm_outliers_mlmonitor_tracker_tenant_" + str(tenant_id)
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="dsm" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DSM adaptive delay tracker
                    report_name = f"trackme_dsm_adaptive_delay_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkadaptivedelay tenant_id="{tenant_id}" component="dsm" min_delay_sec=3600 min_historical_metrics_days=7 earliest_time_mstats="-30d" max_runtime=900 max_auto_delay_sec=604800 max_changes_past_7days=10 review_period_no_days=30 max_sla_percentage=90'
                    report_properties = {
                        "description": "This scheduled report manages adaptive delay tresholds for TrackMe",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DSM delayed entities inspector
                    report_name = f"trackme_dsm_delayed_entities_inspector_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkfeedsdelayedinspector tenant_id="{tenant_id}" component="dsm" max_runtime=900'
                    report_properties = {
                        "description": "This scheduled report manages delayed entities in the dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DSM tags tracker
                    report_name = f"trackme_dsm_tags_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplktags tenant_id="{tenant_id}" component="dsm"'
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DSM priority tracker
                    report_name = f"trackme_dsm_priority_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkpriority tenant_id="{tenant_id}" component=dsm'
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DSM SLA tracker
                    report_name = f"trackme_dsm_sla_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkslaclass tenant_id="{tenant_id}" component=dsm'
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-dsm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

            ################################
            # Data Health Monitoring (DHM) #
            ################################

            # step: Check and create the DHM virtual tenant knowledge objects if enabled
            if component_target in "dhm":
                for object_name in collections_list_dhm:
                    # step: check and create the KVstore collections

                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    collection_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    
                    # Check if collection exists
                    if trackme_check_kvcollection_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        collection_name,
                    ):
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the collection
                        try:
                            trackme_create_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                                collection_acl,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the KVstore transforms

                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    transform_fields = collections_dict[object_name]
                    transform_acl = {
                        "owner": str(tenant_owner),
                        "sharing": trackme_default_sharing,
                        "perms.write": str(tenant_roles_write_perms),
                        "perms.read": str(tenant_roles_read_perms),
                    }
                    
                    # Check if transform exists
                    if trackme_check_kvtransform_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        transform_name,
                    ):
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the transform
                        try:
                            trackme_create_kvtransform(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                transform_name,
                                transform_fields,
                                collection_name,
                                tenant_owner,
                                transform_acl,
                            )
                            result = {
                                "object": transform_name,
                                "object_type": "transform",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": transform_name,
                                "object_type": "transform",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                # not for replica tenants
                if not tenant_replica:
                    # step: check and create the DHM ML training report
                    report_name = "trackme_dhm_outliers_mltrain_tracker_tenant_" + str(tenant_id)
                    report_search = '| trackmesplkoutlierstrainhelper tenant_id="' + str(tenant_id) + '" component="dhm"'
                    report_properties = {
                        "description": "This scheduled report generate and trains Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "0 22-23,0-6 * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DHM ML monitor report
                    report_name = "trackme_dhm_outliers_mlmonitor_tracker_tenant_" + str(tenant_id)
                    report_search = f'| trackmesplkoutlierstrackerhelper tenant_id="{tenant_id}" component="dhm" allow_auto_train="True"'
                    report_properties = {
                        "description": "This scheduled report monitors Machine Learning models for the tenant",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DHM adaptive delay tracker
                    report_name = f"trackme_dhm_adaptive_delay_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkadaptivedelay tenant_id="{tenant_id}" component="dhm" min_delay_sec=3600 min_historical_metrics_days=7 earliest_time_mstats="-30d" max_runtime=900 max_auto_delay_sec=604800 max_changes_past_7days=10 review_period_no_days=30 max_sla_percentage=90'
                    report_properties = {
                        "description": "This scheduled report manages adaptive delay tresholds for TrackMe",
                        "is_scheduled": True,
                        "cron_schedule": "*/60 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DHM delayed entities inspector
                    report_name = f"trackme_dhm_delayed_entities_inspector_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkfeedsdelayedinspector tenant_id="{tenant_id}" component="dhm" max_runtime=900'
                    report_properties = {
                        "description": "This scheduled report manages delayed entities in the dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DHM tags tracker
                    report_name = f"trackme_dhm_tags_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplktags tenant_id="{tenant_id}" component="dhm"'
                    report_properties = {
                        "description": "This scheduled report applies and maintains tags policies for the splk-dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DHM priority tracker
                    report_name = f"trackme_dhm_priority_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkpriority tenant_id="{tenant_id}" component=dhm'
                    report_properties = {
                        "description": f"This scheduled report applies and maintains priority policies for the splk-dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # step: check and create the DHM SLA tracker
                    report_name = f"trackme_dhm_sla_tracker_tenant_{tenant_id}"
                    report_search = f'| trackmesplkslaclass tenant_id="{tenant_id}" component=dhm'
                    report_properties = {
                        "description": f"This scheduled report applies and maintains sla policies for the splk-dhm component",
                        "is_scheduled": True,
                        "cron_schedule": "*/20 * * * *",
                        "dispatch.earliest_time": "-5m",
                        "dispatch.latest_time": "now",
                        "schedule_window": "5",
                    }
                    
                    # Check if report exists
                    if trackme_check_report_exists(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    ):
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "check",
                            "result": "exists",
                        }
                        results_record.append(result)
                    else:
                        # Create the report
                        try:
                            report_acl = {
                                "owner": str(tenant_owner),
                                "sharing": trackme_default_sharing,
                                "perms.write": str(tenant_roles_write_perms),
                                "perms.read": str(tenant_roles_read_perms),
                            }
                            trackme_create_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                report_search,
                                report_properties,
                                report_acl,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "create",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

            # Note: For brevity, I'm only implementing the DSM, and DHM component checks here.
            # The other components (MHM, FLX, WLK, FQM) would follow the same pattern
            # but with their respective collections and reports.

            #######################
            # END OF KOs CHECKING #
            #######################

            # review results record and count failures
            failures_count = 0
            created_count = 0
            existing_count = 0
            results_created = []
            
            for result_record in results_record:
                try:
                    result_action = result_record.get("result")
                    if result_action == "failure":
                        failures_count += 1
                    elif result_action == "success":
                        created_count += 1
                        # Add to results_created list with additional details
                        created_object = {
                            "object": result_record.get("object"),
                            "object_type": result_record.get("object_type"),
                            "action": result_record.get("action"),
                            "result": result_record.get("result"),
                            "created_at": str(datetime.datetime.now()),
                            "tenant_id": str(tenant_id),
                            "component_target": str(component_target)
                        }
                        results_created.append(created_object)
                    elif result_action == "exists":
                        existing_count += 1
                except Exception as e:
                    logger.error(
                        f'count not find expected action result field in record="{json.dumps(result_record, indent=2)}"'
                    )

            # Record an audit change
            user = request_info.user
            result = f"Check completed: {existing_count} objects already exist, {created_count} objects created, {failures_count} failures"
            audit_record = {
                "_time": str(time.time()),
                "timeStr": str(datetime.datetime.now()),
                "user": str(user),
                "action": "success" if failures_count == 0 else "failure",
                "change_type": "check virtual tenant component",
                "object": str(tenant_id),
                "object_category": "virtual_tenants",
                "object_attrs": vtenant_record,
                "results_summary": str(result),
                "results_details": results_record,
                "results_created": results_created,
                "comment": str(update_comment),
            }

            # index the audit record
            try:
                trackme_audit_event(
                    request_info.system_authtoken,
                    request_info.server_rest_uri,
                    tenant_id,
                    request_info.user,
                    "success" if failures_count == 0 else "failure",
                    "check virtual tenant component",
                    str(tenant_id),
                    "virtual_tenants",
                    json.dumps(vtenant_record),
                    json.dumps(results_record),
                    str(update_comment),
                )
            except Exception as e:
                logger.error(
                    f'tenant_id="{tenant_id}", failed to index audit record, exception="{str(e)}"'
                )

            logger.info(str(audit_record))

            # Return
            return {"payload": audit_record, "status": 200}

    #
    # Update an existing tenant to delete a component
    #

    def post_delete_component_tenant(self, request_info, **kwargs):
        # Declare main variables
        tenant_id = None
        component_target = None
        query_string = None

        # Declare booleans
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                component_target = resp_dict["component_target"]

                # option force
                try:
                    force = resp_dict["force"]
                    if force in ("true", "True"):
                        force = True
                except Exception as e:
                    force = False

            # At least one component needs to be enabled
            if not describe:
                if not component_target in (
                    "dsm",
                    "dhm",
                    "mhm",
                    "flx",
                    "wlk",
                    "fqm",
                ):
                    return {
                        "payload": {"response": "Error, invalid component."},
                        "status": 500,
                    }

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be called to delete a component from an existing virtual tenant, it requires a POST call with the following information:",
                "resource_desc": "Update a virtual tenant",
                "resource_spl_example": "| trackme url=\"/services/trackme/v2/vtenants/admin/delete_component_tenant\" mode=\"post\" body=\"{ 'tenant_id': 'mytenant', 'component_target': 'dhm' }\""
                + "'update_comment': 'Updated for the purpose of the documentation.'}\"",
                "options": [
                    {
                        "tenant_id": "ID of the virtual tenant to be updated",
                        "component_target": "The component to be deleted, valid options: dsm|dhm|mhm|flx|wlk|fqm",
                        "force": "OPTIONAL: (true|false) if true, runs in force mode and attempts to remove any object for the component requested, even if it is disabled already",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
            if update_comment == "Enter your audit message":
                update_comment = "No comment was provided for this operation"
        except Exception as e:
            update_comment = "No comment was provided for this operation"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        # Data collection
        collection_name = "kv_trackme_virtual_tenants"
        collection = service.kvstore[collection_name]

        # Summary data collection
        collection_summary_name = "kv_trackme_virtual_tenants_entities_summary"
        collection_summary = service.kvstore[collection_summary_name]

        ################
        # MAIN PROGRAM #
        ################

        # get trackme release
        trackme_version = trackme_get_version(service)
        logger.info(f'running trackme version="{trackme_version}"')

        # check license state
        try:
            check_license = trackme_check_license(
                request_info.server_rest_uri, request_info.session_key
            )
            license_is_valid = check_license.get("license_is_valid")
            license_subscription_class = check_license.get("license_subscription_class")
            logger.debug(
                f'function check_license called, response="{json.dumps(check_license, indent=2)}"'
            )

        except Exception as e:
            license_is_valid = 0
            license_subscription_class = "free"
            logger.error(f'function check_license exception="{str(e)}"')

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        try:
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]
            key = vtenant_record.get("_key")
            tenant_dsm_enabled = int(vtenant_record.get("tenant_dsm_enabled"))
            tenant_flx_enabled = int(vtenant_record.get("tenant_flx_enabled"))
            tenant_fqm_enabled = int(vtenant_record.get("tenant_fqm_enabled"))
            tenant_dhm_enabled = int(vtenant_record.get("tenant_dhm_enabled"))
            tenant_mhm_enabled = int(vtenant_record.get("tenant_mhm_enabled"))
            tenant_wlk_enabled = int(vtenant_record.get("tenant_wlk_enabled"))
            tenant_owner = str(vtenant_record.get("tenant_owner"))
            tenant_roles_admin = str(vtenant_record.get("tenant_roles_admin"))
            tenant_roles_user = str(vtenant_record.get("tenant_roles_user"))
            tenant_roles_power = str(vtenant_record.get("tenant_roles_power"))
            tenant_replica = interpret_boolean(vtenant_record.get("tenant_replica"))

        except Exception as e:
            return {
                "payload": {
                    "response": "Sorry, I am afraid I can't do that, this tenant does not exist yet."
                },
                "status": 500,
            }

        # if force:
        if force:
            tenant_dsm_enabled = 1
            tenant_flx_enabled = 1
            tenant_fqm_enabled = 1
            tenant_dhm_enabled = 1
            tenant_mhm_enabled = 1
            tenant_wlk_enabled = 1

        # for read permissions, concatenate admin, power and user
        tenant_roles_read_perms = (
            f"{tenant_roles_admin},{tenant_roles_power},{tenant_roles_user}"
        )

        # for write permissions, concatenate admin, power
        tenant_roles_write_perms = f"{tenant_roles_admin},{tenant_roles_power}"

        # Restricted components
        restricted_components = ["flx", "wlk", "fqm"]
        license_error_messages = {
            0: "I'm afraid I can't do that, this instance is running Free limited mode and one or more of the components you have selected are restricted to the Full or Trial edition.",
            1: "I'm afraid I can't do that, this instance is running Free extended mode and one or more of the components you have selected are restricted to the Full or Trial edition.",
        }

        license_restriction = (license_is_valid != 1) or (
            license_is_valid == 1 and license_subscription_class == "free_extended"
        )

        if license_restriction and component_target in restricted_components:
            audit_record = {
                "action": "failure",
                "change_type": "update virtual tenant",
                "object": str(tenant_id),
                "object_category": "virtual_tenants",
                "result": license_error_messages[license_is_valid],
            }

            logger.error(str(audit_record))
            return {"payload": audit_record, "status": 402}

        else:
            # Verify the operation requested
            component_enabled_map = {
                "dsm": tenant_dsm_enabled,
                "dhm": tenant_dhm_enabled,
                "mhm": tenant_mhm_enabled,
                "flx": tenant_flx_enabled,
                "fqm": tenant_fqm_enabled,
                "wlk": tenant_wlk_enabled,
            }

            if component_enabled_map.get(component_target) != 1:
                if not force:
                    audit_record = {
                        "action": "failure",
                        "change_type": "delete virtual tenant component",
                        "object": str(tenant_id),
                        "object_category": "virtual_tenants",
                        "result": "I'm afraid I can't do that, the requested component is already disabled for this tenant.",
                    }

                    logger.error(str(audit_record))
                    return {"payload": audit_record, "status": 409}

            # operation counters
            results_record = []

            # check if TCM is enabled in receiver mode
            enable_conf_manager_receiver = int(
                trackme_conf["trackme_conf"]["trackme_general"][
                    "enable_conf_manager_receiver"
                ]
            )

            if enable_conf_manager_receiver == 1:
                try:
                    tcm_response = trackme_send_to_tcm(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        resp_dict,
                        "post",
                        "/services/trackme/v2/vtenants/admin/delete_component_tenant",
                    )
                    logger.info(f"trackme_send_to_tcm was successfully executed")
                except Exception as e:
                    logger.error(
                        f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                    )

            ################################
            # Data Source Monitoring (DSM) #
            ################################

            # step: Create the DSM virtual tenant knowledge objects if enabled
            if component_target in "dsm":
                # Remove reports

                reports_list = []

                # not for replica tenants
                if not tenant_replica:
                    # add common reports for the component
                    reports_list.append("trackme_dsm_shared_elastic_tracker")
                    reports_list.append("trackme_dsm_data_sampling_tracker")
                    reports_list.append("trackme_dsm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dsm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dsm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dsm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dsm_tags_tracker")
                    reports_list.append("trackme_dsm_priority_tracker")
                    reports_list.append("trackme_dsm_sla_tracker")

                # log
                logger.info(
                    f'tenant_id="{tenant_id}", list of reports to be deleted, reports="{reports_list}"'
                )

                # Loop
                for report in reports_list:
                    report_name = str(report) + "_tenant_" + str(tenant_id)

                    try:
                        action = trackme_delete_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # Remove macros
                def handle_macro_deletion(
                    macro_name,
                    tenant_enabled,
                    system_authtoken,
                    tenant_id,
                    results_record,
                ):
                    if tenant_enabled:
                        try:
                            action = trackme_delete_macro(
                                system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                macro_name,
                            )
                            result = {
                                "object": macro_name,
                                "object_type": "macro",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": macro_name,
                                "object_type": "macro",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                macros_list = []

                components_flags = {
                    "_dsm_": 1,
                }

                for macro in macros_list:
                    macro_name = f"{macro}_tenant_{tenant_id}"
                    for component, tenant_enabled in components_flags.items():
                        if component in macro_name:
                            handle_macro_deletion(
                                macro_name,
                                tenant_enabled,
                                request_info.system_authtoken,
                                tenant_id,
                                results_record,
                            )
                            break

                # Handle related dedicated Elastic Sources reports, if any
                elastic_reports = []
                elastic_collection_name = (
                    "kv_trackme_dsm_elastic_dedicated" + "_tenant_" + str(tenant_id)
                )

                logger.info(
                    f'tenant_id="{tenant_id}", checking for elastic dedicated reports, collection="{elastic_collection_name}"'
                )
                try:
                    elastic_collection = service.kvstore[elastic_collection_name]
                    elastic_records = elastic_collection.data.query()
                    logger.info(f'elastic_records="{str(elastic_records)}"')

                    # Add each reports to our list
                    for elastic_record in elastic_records:
                        elastic_reports.append(elastic_record.get("elastic_report"))
                        elastic_reports.append(elastic_record.get("elastic_wrapper"))
                    logger.info(f'elastic_reports="{str(elastic_reports)}"')

                except Exception as e:
                    logger.info(
                        f'tenant_id="{tenant_id}", there are no dedicated Elastic Source reports to be deleted, collection_name="{collection_name}"'
                    )

                if elastic_reports:
                    for elastic_report in elastic_reports:
                        report_name = elastic_report
                        try:
                            action = trackme_delete_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                # Handle hybrid objects
                tenant_dsm_hybrid_objects = None

                try:
                    tenant_dsm_hybrid_objects = vtenant_record.get(
                        "tenant_dsm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dsm_hybrid_objects = None

                if tenant_dsm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dsm_hybrid_objects="{tenant_dsm_hybrid_objects}"'
                    )

                    if (
                        tenant_dsm_hybrid_objects
                        and tenant_dsm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dsm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # Handle KVstore collections
                for object_name in collections_list_dsm:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #########################
            # Workload              #
            #########################

            # wlk
            if component_target in "wlk":
                # Remove reports
                reports_list = []

                # not for replica tenants
                if not tenant_replica:
                    reports_list.append("trackme_wlk_outliers_mltrain_tracker")
                    reports_list.append("trackme_wlk_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_wlk_tags_tracker")
                    reports_list.append("trackme_wlk_priority_tracker")
                    reports_list.append("trackme_wlk_sla_tracker")

                # log
                logger.info(
                    f'tenant_id="{tenant_id}", list of reports to be deleted, reports="{reports_list}"'
                )

                # Loop
                for report in reports_list:
                    report_name = str(report) + "_tenant_" + str(tenant_id)

                    try:
                        action = trackme_delete_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # Handle hybrid objects
                tenant_wlk_hybrid_objects = None

                try:
                    tenant_wlk_hybrid_objects = vtenant_record.get(
                        "tenant_wlk_hybrid_objects"
                    )

                except Exception as e:
                    tenant_wlk_hybrid_objects = None

                if tenant_wlk_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_wlk_hybrid_objects="{tenant_wlk_hybrid_objects}"'
                    )

                    if (
                        tenant_wlk_hybrid_objects
                        and tenant_wlk_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_wlk_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_wlk:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # Remove macros
                def handle_macro_deletion(
                    macro_name,
                    tenant_enabled,
                    system_authtoken,
                    tenant_id,
                    results_record,
                ):
                    if tenant_enabled:
                        try:
                            action = trackme_delete_macro(
                                system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                macro_name,
                            )
                            result = {
                                "object": macro_name,
                                "object_type": "macro",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": macro_name,
                                "object_type": "macro",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                macros_list = [
                    "trackme_wlk_set_status",
                    "trackme_wlk_set_outliers_metrics",
                ]

                components_flags = {
                    "_wlk_": 1,
                }

                for macro in macros_list:
                    macro_name = f"{macro}_tenant_{tenant_id}"
                    for component, tenant_enabled in components_flags.items():
                        if component in macro_name:
                            handle_macro_deletion(
                                macro_name,
                                tenant_enabled,
                                request_info.system_authtoken,
                                tenant_id,
                                results_record,
                            )
                            break

            #########################
            # Flex Objects          #
            #########################

            # flx
            if component_target in "flx":
                # Remove reports
                reports_list = []

                # not for replica tenants
                if not tenant_replica:
                    reports_list.append("trackme_flx_outliers_mltrain_tracker")
                    reports_list.append("trackme_flx_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_flx_inactive_entities_tracker")
                    reports_list.append("trackme_flx_tags_tracker")
                    reports_list.append("trackme_flx_priority_tracker")
                    reports_list.append("trackme_flx_sla_tracker")

                # log
                logger.info(
                    f'tenant_id="{tenant_id}", list of reports to be deleted, reports="{reports_list}"'
                )

                # Loop
                for report in reports_list:
                    report_name = str(report) + "_tenant_" + str(tenant_id)

                    try:
                        action = trackme_delete_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # Handle hybrid objects
                tenant_flx_hybrid_objects = None

                try:
                    tenant_flx_hybrid_objects = vtenant_record.get(
                        "tenant_flx_hybrid_objects"
                    )

                except Exception as e:
                    tenant_flx_hybrid_objects = None

                if tenant_flx_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_flx_hybrid_objects="{tenant_flx_hybrid_objects}"'
                    )

                    if (
                        tenant_flx_hybrid_objects
                        and tenant_flx_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_flx_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_flx:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #############################
            # Fields Quality Monitoring #
            #############################

            # fqm
            if component_target in "fqm":
                # Remove reports
                reports_list = []

                # not for replica tenants
                if not tenant_replica:
                    reports_list.append("trackme_fqm_outliers_mltrain_tracker")
                    reports_list.append("trackme_fqm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_fqm_inactive_entities_tracker")
                    reports_list.append("trackme_fqm_tags_tracker")
                    reports_list.append("trackme_fqm_priority_tracker")
                    reports_list.append("trackme_fqm_sla_tracker")

                # log
                logger.info(
                    f'tenant_id="{tenant_id}", list of reports to be deleted, reports="{reports_list}"'
                )

                # Loop
                for report in reports_list:
                    report_name = str(report) + "_tenant_" + str(tenant_id)

                    try:
                        action = trackme_delete_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # Handle hybrid objects
                tenant_fqm_hybrid_objects = None

                try:
                    tenant_fqm_hybrid_objects = vtenant_record.get(
                        "tenant_fqm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_fqm_hybrid_objects = None

                if tenant_fqm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_fqm_hybrid_objects="{tenant_fqm_hybrid_objects}"'
                    )

                    if (
                        tenant_fqm_hybrid_objects
                        and tenant_fqm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_fqm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_fqm:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            ##################
            # DHM
            ##################

            # step: Create the DHM virtual tenant knowledge objects if enabled
            if component_target in "dhm":
                # Remove reports
                reports_list = []

                # not for replica tenants
                if not tenant_replica:
                    reports_list.append("trackme_dhm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dhm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dhm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dhm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dhm_tags_tracker")
                    reports_list.append("trackme_dhm_priority_tracker")
                    reports_list.append("trackme_dhm_sla_tracker")

                # log
                logger.info(
                    f'tenant_id="{tenant_id}", list of reports to be deleted, reports="{reports_list}"'
                )

                # Loop
                for report in reports_list:
                    report_name = str(report) + "_tenant_" + str(tenant_id)

                    try:
                        action = trackme_delete_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # Remove macros
                def handle_macro_deletion(
                    macro_name,
                    tenant_enabled,
                    system_authtoken,
                    tenant_id,
                    results_record,
                ):
                    if tenant_enabled:
                        try:
                            action = trackme_delete_macro(
                                system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                macro_name,
                            )
                            result = {
                                "object": macro_name,
                                "object_type": "macro",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": macro_name,
                                "object_type": "macro",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                macros_list = [
                    "trackme_dhm_default_splk_dhm_alert_policy",
                ]

                components_flags = {
                    "_dhm_": 1,
                }

                for macro in macros_list:
                    macro_name = f"{macro}_tenant_{tenant_id}"
                    for component, tenant_enabled in components_flags.items():
                        if component in macro_name:
                            handle_macro_deletion(
                                macro_name,
                                tenant_enabled,
                                request_info.system_authtoken,
                                tenant_id,
                                results_record,
                            )
                            break

                # Handle hybrid objects
                tenant_dhm_hybrid_objects = None

                try:
                    tenant_dhm_hybrid_objects = vtenant_record.get(
                        "tenant_dhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dhm_hybrid_objects = None

                if tenant_dhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dhm_hybrid_objects="{tenant_dhm_hybrid_objects}"'
                    )

                    if (
                        tenant_dhm_hybrid_objects
                        and tenant_dhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_dhm:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            ##################
            # MHM
            ##################

            # step: Create the DHM virtual tenant knowledge objects if enabled
            if component_target in "mhm":
                # Remove reports
                reports_list = []

                # not for replica tenants
                if not tenant_replica:
                    reports_list.append("trackme_mhm_tags_tracker")
                    reports_list.append("trackme_mhm_priority_tracker")
                    reports_list.append("trackme_mhm_sla_tracker")

                # log
                logger.info(
                    f'tenant_id="{tenant_id}", list of reports to be deleted, reports="{reports_list}"'
                )

                # Loop
                for report in reports_list:
                    report_name = str(report) + "_tenant_" + str(tenant_id)

                    try:
                        action = trackme_delete_report(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            report_name,
                        )
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": report_name,
                            "object_type": "report",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # Handle hybrid objects
                tenant_mhm_hybrid_objects = None

                try:
                    tenant_mhm_hybrid_objects = vtenant_record.get(
                        "tenant_mhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_mhm_hybrid_objects = None

                if tenant_mhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_mhm_hybrid_objects="{tenant_mhm_hybrid_objects}"'
                    )

                    if (
                        tenant_mhm_hybrid_objects
                        and tenant_mhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_mhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_mhm:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # delete transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            #################
            # Alerts
            #################

            # Handle alerts
            tenant_alert_objects = None

            try:
                tenant_alert_objects = vtenant_record.get("tenant_alert_objects")

            except Exception as e:
                tenant_alert_objects = None

            if tenant_alert_objects:
                # logger.debug
                logger.debug(f'tenant_alert_objects="{tenant_alert_objects}"')

                if tenant_alert_objects and tenant_alert_objects != "None":
                    vtenant_dict = json.loads(tenant_alert_objects)
                    logger.info(f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"')

                    # get reports
                    alerts = vtenant_dict["alerts"]
                    logger.debug(f'alerts="{alerts}"')

                    # counter
                    reports_deleted_count = 0

                    # Loop through reports
                    for alert_name in alerts:
                        report_name = alert_name

                        # if the report_name contains "splk-{component_target}" in its name, then it is associated with the component_target
                        if "splk-" + component_target in report_name:
                            try:
                                action = trackme_delete_report(
                                    request_info.system_authtoken,
                                    request_info.server_rest_uri,
                                    tenant_id,
                                    report_name,
                                )
                                result = {
                                    "object": report_name,
                                    "object_type": "report",
                                    "action": "delete",
                                    "result": "success",
                                }
                                results_record.append(result)

                                # delete from the list
                                alerts.remove(report_name)

                                # increment
                                reports_deleted_count += 1

                            except Exception as e:
                                result = {
                                    "object": report_name,
                                    "object_type": "report",
                                    "action": "delete",
                                    "result": "failure",
                                    "exception": str(e),
                                }
                                results_record.append(result)

                    # if alerts were deleted, update the Kvstore record now
                    vtenant_dict["alerts"] = alerts

                    # perform
                    try:
                        collection.data.update(str(key), json.dumps(vtenant_record))
                    except Exception as e:
                        logger.error(
                            f'failed to update the vtenant collection record, exception="{str(e)}"'
                        )

            #####################
            # END OF KOs DELETION
            #####################

            # Create a mapping dictionary
            component_target_mapping = {
                "dsm": "tenant_dsm_enabled",
                "dhm": "tenant_dhm_enabled",
                "mhm": "tenant_mhm_enabled",
                "flx": "tenant_flx_enabled",
                "fqm": "tenant_fqm_enabled",
                "wlk": "tenant_wlk_enabled",
            }

            # Create a mapping dictionary for hybrid objects
            component_target_hybrid_objects_mapping = {
                "dsm": "tenant_dsm_hybrid_objects",
                "dhm": "tenant_dhm_hybrid_objects",
                "mhm": "tenant_mhm_hybrid_objects",
                "flx": "tenant_flx_hybrid_objects",
                "fqm": "tenant_fqm_hybrid_objects",
                "wlk": "tenant_wlk_hybrid_objects",
            }

            # Update the record
            if component_target in component_target_mapping:
                vtenant_record[component_target_mapping[component_target]] = 0
                vtenant_record[
                    component_target_hybrid_objects_mapping[component_target]
                ] = ""

            # clear out the tenant_objects_exec_summary, it will be re-generated automatically
            vtenant_record["tenant_objects_exec_summary"] = ""

            # perform
            try:
                collection.data.update(str(key), json.dumps(vtenant_record))
            except Exception as e:
                logger.error(
                    f'failed to update the vtenant collection record, exception="{str(e)}"'
                )

            # clean up the vtenant summary record
            try:
                vtenant_summary_record = collection_summary.data.query(
                    query=json.dumps(query_string)
                )[0]
                summary_key = vtenant_summary_record.get("_key")

            except Exception as e:
                vtenant_summary_record = None
                summary_key = None

            if summary_key:
                try:
                    vtenant_summary_record.pop(f"{component_target}_entities", None)
                    vtenant_summary_record.pop(
                        f"{component_target}_high_red_priority", None
                    )
                    vtenant_summary_record.pop(
                        f"{component_target}_low_red_priority", None
                    )
                    vtenant_summary_record.pop(
                        f"{component_target}_medium_red_priority", None
                    )
                    vtenant_summary_record.pop(f"{component_target}_last_exec", None)
                except Exception as e:
                    logger.error(
                        f'failed to clean up the vtenant record, exception="{str(e)}"'
                    )

            # update
            try:
                collection_summary.data.update(
                    str(summary_key), json.dumps(vtenant_summary_record)
                )
            except Exception as e:
                logger.error(
                    f'failed to update the vtenant collection summary record, exception="{str(e)}"'
                )

            # Get updated record
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]

            # review results record and count failures
            failures_count = 0
            for result_record in results_record:
                try:
                    result_action = result_record.get("result")
                    if result_action == "failure":
                        failures_count += 1
                except Exception as e:
                    logger.error(
                        f'count not find expected action result field in record="{json.dumps(result_record, indent=2)}"'
                    )

            # handle
            if failures_count > 0:
                # Record an audit change
                user = request_info.user
                result = "One or more exceptions were encountered during the update of the tenant, review the results_details as needed"
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "failure",
                    "change_type": "delete virtual tenant component",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "delete virtual tenant component",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 500}

            else:
                # Record an audit change
                user = request_info.user
                result = "All objects for the component of this tenant were successfully deleted"
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "success",
                    "change_type": "delete virtual tenant component",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "delete virtual tenant component",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                # Attempt to execute the health tracker immediately to allow the tenant
                # to be immediately active upon its creation

                savedsearch = "trackme_health_tracker_tenant_" + str(tenant_id)
                search = "| savedsearch " + savedsearch
                kwargs_oneshot = {
                    "earliest_time": "-5m",
                    "latest_time": "now",
                    "output_mode": "json",
                    "count": 0,
                }

                logger.info(
                    f'tenant_id="{tenant_id}", Attempting to execute the health tracker="{savedsearch}"'
                )
                try:
                    reader = run_splunk_search(
                        service,
                        search,
                        kwargs_oneshot,
                        24,
                        5,
                    )

                    for item in reader:
                        if isinstance(item, dict):
                            logger.info(
                                f'tenant_id="{tenant_id}", health tracker="{savedsearch}", execution was successful, results="{json.dumps(json.loads(item.get("_raw")), indent=2)}"'
                            )

                except Exception as e:
                    logger.info(
                        f'tenant_id="{tenant_id}", health tracker="{savedsearch}", execution has failed with exception="{str(e)}"'
                    )

                # Return
                return {"payload": audit_record, "status": 200}

    #
    # Add new tenant in multi-operations mode
    #

    def post_add_tenant_multiops(self, request_info, **kwargs):
        """
        | trackme url="/services/trackme/v2/vtenants/admin/add_tenant_multiops" mode="post"
        """

        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                # mandatory parameters
                try:
                    tenant_dict = resp_dict["tenant_dict"]
                except Exception as e:
                    return {
                        "payload": "tenant_dict is required",
                        "status": 500,
                    }

                # optional
                try:
                    hybrid_objects = resp_dict["hybrid_objects"]
                except Exception as e:
                    hybrid_objects = []

                # optional
                try:
                    execute_trackers = resp_dict["execute_trackers"]
                    if execute_trackers in ("True"):
                        execute_trackers = True
                    elif execute_trackers in ("False"):
                        execute_trackers = False
                    else:
                        execute_trackers = False
                except Exception as e:
                    execute_trackers = False

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint create a new tenant in multi-ops mode, which means that it calls the tenant creation endpoint,"
                + " as well as can perform the creation of hybrid trackers at the time of the tenant creation and execute the trackers, "
                + "it requires a GET call with no data required",
                "resource_desc": "Endpoint wrapper for tenants creation, allows combining the creation of a new tenant with the creation of hybrid trackers and their automatic first execution",
                "resource_spl_example": '| trackme url="/services/trackme/v2/vtenants/admin/add_tenant_multiops" mode="post" '
                + 'body="{"tenant_dict":{"tenant_desc":"Demo","tenant_name":"mytenant","tenant_roles_admin":["trackme_admin"]'
                + ',"tenant_roles_user":["trackme_user"],"tenant_owner":"admin","tenant_idx_settings":"global",'
                + '"tenant_dsm_enabled":true},'
                + '"hybrid_objects":[{"tenant_id":"mytenant","component":"dsm","tracker_name":"local",'
                + '"account":"local","owner":"admin","search_mode":"tstats","earliest_time":"-4h","latest_time":"+4h"'
                + ',"root_constraint":"(index=* OR index=_*) (index=*) (splunk_server=*) (sourcetype!=stash sourcetype!=*too_small sourcetype!=modular_alerts:trackme* sourcetype!=trackme:*)"'
                + ',"breakby_field":"none"}]}"',
                "options": [
                    {
                        "tenant_dict": "A JSON dictionnary containting the tenant definition",
                        "hybrid_objects": "A JSON array of hybrid trackers definitions to be created and executed",
                        "execute_trackers": "Execute trackers after their creation, valid options are: True / False (default)",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Define an header for requests authenticated communications with splunkd
        header = {
            "Authorization": "Splunk %s" % request_info.system_authtoken,
            "Content-Type": "application/json",
        }

        #
        # tenant creation
        #

        # target
        endpoint_url = f"{request_info.server_rest_uri}/services/trackme/v2/vtenants/admin/add_tenant"

        try:
            req_tenant_add_response = requests.post(
                endpoint_url,
                headers=header,
                data=json.dumps(tenant_dict),
                verify=False,
                timeout=600,
            )
            logger.debug(f'response="{req_tenant_add_response.text}"')
            tenant_add_response = req_tenant_add_response.text

            if int(req_tenant_add_response.status_code) != 200:
                logger.error(
                    f'an exception was encountered while attempting to create the new tenant="{json.dumps(tenant_dict, indent=2)}", exception="{tenant_add_response}"'
                )
                return {
                    "payload": {
                        "action": "failure",
                        "response": "An exception was encountered while attempting to create the new tenant",
                        "tenant_dict": tenant_dict,
                        "results": tenant_add_response,
                    },
                    "status": 500,
                }
            else:
                logger.info(
                    f'new tenant was successfully created, tenant_dict="{json.dumps(tenant_dict, indent=2)}", response="{json.dumps(tenant_add_response, indent=2)}"'
                )

        except Exception as e:
            logger.error(
                f'an exception was encountered while attempting to create the new tenant="{json.dumps(tenant_dict, indent=2)}", exception="{str(e)}"'
            )
            return {
                "payload": {
                    "action": "failure",
                    "response": "An exception was encountered while attempting to create the new tenant",
                    "tenant_dict": tenant_dict,
                    "exception": str(e),
                },
                "status": 500,
            }

        #
        # hybrid objects creation
        #

        # store all hybrid objects creation responses in a list
        hybrid_objects_response = []

        # store all hybrid trackers successfully created in a list, we will proceed to their execution in a final phase
        hybrid_objects_trackers = []

        # store all hybrid trackers execution results in a list
        hybrid_objects_execution_results = []

        # endpoint target
        endpoint_url = f"{request_info.server_rest_uri}/services/trackme/v2/splk_hybrid_trackers/admin/hybrid_tracker_create"

        # loop through the hybrid objects to be managed, and proceed
        # If an hybrid object fails to be created, do no exit but report the failure at the end of the whole process

        count_errors_creation = 0
        count_errors_execution = 0
        count_objects = len(hybrid_objects)
        count_processed = 0
        count_executed = 0

        #
        # Loop and create hybrid objects
        #

        for hybrid_object in hybrid_objects:
            logger.info(
                f'proceeding with hybrid object creation, hybrid_object="{json.dumps(hybrid_object, indent=2)}"'
            )

            try:
                req_hybrid_add_response = requests.post(
                    endpoint_url,
                    headers=header,
                    data=json.dumps(hybrid_object),
                    verify=False,
                    timeout=600,
                )
                hybrid_add_response = req_hybrid_add_response.text

                if int(req_hybrid_add_response.status_code) != 200:
                    # increment errors
                    count_errors_creation += 1

                    # log
                    logger.error(
                        f'An exception was encountered while attempting to create the new hybrid_object="{json.dumps(hybrid_object, indent=2)}", hybrid_add_response="{hybrid_add_response}", exception="{req_hybrid_add_response.text}"'
                    )

                    # add to responses
                    hybrid_objects_response.append(
                        {
                            "hybrid_object": hybrid_object,
                            "action": "failure",
                            "status": "An exception was encountered while attempting to create the new hybrid object",
                            "exception": req_hybrid_add_response.text,
                        }
                    )

                else:
                    # we expect a dict
                    if not isinstance(hybrid_add_response, dict):
                        hybrid_add_response = json.loads(hybrid_add_response)

                    logger.info(f'hybrid_add_response="{hybrid_add_response}"')

                    # increment
                    count_processed += 1

                    # log
                    logger.info(
                        f'new hybrid object was successfully created, hybrid_object="{json.dumps(hybrid_object, indent=2)}", response="{json.dumps(hybrid_add_response, indent=2)}"'
                    )

                    # add to responses
                    hybrid_objects_response.append(
                        {
                            "hybrid_object": hybrid_object,
                            "action": "success",
                            "status": "The new hybrid object was processed successfully",
                            "response": hybrid_add_response,
                        }
                    )

                    # add to hybrid objects to executed
                    hybrid_objects_trackers.append(
                        {
                            "tracker_report": hybrid_add_response.get("tracker_report"),
                            "tracker_name": hybrid_add_response.get("tracker_name"),
                            "earliest": hybrid_add_response.get("earliest"),
                            "latest": hybrid_add_response.get("latest"),
                        }
                    )

            except Exception as e:
                # increment errors
                count_errors_creation += 1

                # log
                logger.error(
                    f'An exception was encountered while attempting to create the new hybrid_object="{json.dumps(hybrid_object, indent=2)}", hybrid_add_response="{hybrid_add_response}", exception="{str(e)}"'
                )

                # add to responses
                hybrid_objects_response.append(
                    {
                        "hybrid_object": hybrid_object,
                        "action": "failure",
                        "status": "An exception was encountered while attempting to create the new hybrid object",
                        "exception": str(e),
                    }
                )

        #
        # Loop and execute created hybrid objects
        #

        if execute_trackers:
            for hybrid_object in hybrid_objects_trackers:
                logger.info(
                    f'Attempting to execute hybrid_object="{json.dumps(hybrid_object, indent=2)}"'
                )

                #
                # Splunkd API needs a couple of seconds to refresh while KOs are created
                #

                # set max failed re-attempt
                max_failures_count = 24
                sleep_time = 5
                creation_success = False
                current_failures_count = 0

                while (
                    current_failures_count < max_failures_count and not creation_success
                ):
                    try:
                        newtracker = service.saved_searches[
                            hybrid_object.get("tracker_report")
                        ]
                        logger.info(
                            f'action="success", hybrid tracker was successfully executed, report_name="{hybrid_object.get("tracker_report")}"'
                        )
                        creation_success = True
                        break

                    except Exception as e:
                        # We except this sentence in the exception if the API is not ready yet
                        logger.warning(
                            f'temporary failure, the report is not yet available, will sleep and re-attempt, report report_name="{hybrid_object.get("tracker_report")}"'
                        )
                        time.sleep(sleep_time)
                        current_failures_count += 1

                        if current_failures_count >= max_failures_count:
                            logger.error(
                                f'max attempt reached, failure to create report report_name="{hybrid_object.get("tracker_report")}" with exception:"{str(e)}"'
                            )
                            break

                # set args
                kwargs_search = {
                    "app": "trackme",
                    "earliest_time": hybrid_object.get("earliest"),
                    "latest_time": hybrid_object.get("latest"),
                    "output_mode": "json",
                    "count": 0,
                }

                # set query
                searchquery = "| savedsearch " + hybrid_object.get("tracker_report")

                # execute
                query_results = []
                results_count = 0
                start_time = time.time()

                #
                # Splunkd API needs a couple of seconds to refresh while KOs are created
                #

                creation_success = False

                try:
                    reader = run_splunk_search(
                        service,
                        searchquery,
                        kwargs_search,
                        24,
                        5,
                    )

                    for item in reader:
                        if isinstance(item, dict):
                            query_results.append(item)
                            results_count += 1

                    hybrid_objects_execution_results.append(
                        {
                            "action": "success",
                            "status": "The hybrid object was executed successfully",
                            "hybrid_object": hybrid_object.get("tracker_report"),
                            "searchquery": searchquery,
                            "kwargs": kwargs_search,
                            "results": query_results,
                            "results_count": results_count,
                            "runtime": round(time.time() - start_time, 3),
                        }
                    )

                    # increment
                    count_executed += 1

                    # log
                    logger.info(
                        f'The new hybrid object="{hybrid_object.get("tracker_report")}" was successfully executed'
                    )

                    creation_success = True

                except Exception as e:
                    hybrid_objects_execution_results.append(
                        {
                            "action": "failure",
                            "status": "An exception was encountered while attempting to execute the hybrid tracker",
                            "hybrid_object": hybrid_object.get("tracker_report"),
                            "searchquery": searchquery,
                            "kwargs": kwargs_search,
                            "exception": str(e),
                        }
                    )

                    # log
                    logger.error(
                        f'permanent execution failure, An exception was encountered while attempting to execute the new hybrid object="{json.dumps(hybrid_object, indent=2)}", exception="{str(e)}"'
                    )

        # finally, render response, if the tenant failed to be created, we returned a 500 earlier
        # if we have had errors at creation of hybrid trackers, or errors during their execution, return a 500 with the detailed respoonse

        response_record = {
            "response_tenant_creation": tenant_add_response,
            "response_hybrid_objects_creation": hybrid_objects_response,
            "response_hybrid_objects_execution": hybrid_objects_execution_results,
            "count_hybrid_object_requested": count_objects,
            "count_hybrid_trackers_successfully_created": count_processed,
            "count_hybrid_trackers_creation_failures": count_errors_creation,
            "count_hybrid_trackers_successfully_executed": count_executed,
            "count_hybrid_trackers_execution_failures": count_errors_execution,
        }

        logger.info(
            f'add_tenant_multiops has terminated, response="{json.dumps(response_record, indent=2)}"'
        )

        if count_errors_creation > 0 or count_errors_execution > 0:
            response_record["action"] = "failure"
            return {"payload": response_record, "status": 500}

        else:
            response_record["action"] = "success"
            return {"payload": response_record, "status": 200}

    #
    # Add new tenant in multi-operations mode for Splunk Workload
    #

    def post_add_tenant_wlk_multiops(self, request_info, **kwargs):
        """
        | trackme url="/services/trackme/v2/vtenants/admin/add_tenant_wlk_multiops" mode="post"
        """

        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False
            if not describe:

                # mandatory parameters
                try:
                    tenant_dict = resp_dict["tenant_dict"]
                except Exception as e:
                    return {
                        "payload": "tenant_dict is required",
                        "status": 500,
                    }

                # optional
                try:
                    hybrid_objects = resp_dict["hybrid_objects"]
                except Exception as e:
                    hybrid_objects = []

                # optional
                try:
                    execute_trackers = resp_dict["execute_trackers"]
                    if execute_trackers in ("True"):
                        execute_trackers = True
                    elif execute_trackers in ("False"):
                        execute_trackers = False
                    else:
                        execute_trackers = False
                except Exception as e:
                    execute_trackers = False

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint create a new tenant in multi-ops mode, which means that it calls the tenant creation endpoint,"
                + " as well as can perform the creation of hybrid trackers at the time of the tenant creation and execute the trackers, "
                + "it requires a GET call with no data required",
                "resource_desc": "Endpoint wrapper for tenants creation, allows combining the creation of a new tenant with the creation of hybrid trackers and their automatic first execution",
                "resource_spl_example": '| trackme url="/services/trackme/v2/vtenants/admin/add_tenant_multiops" mode="post" '
                + 'body="{"tenant_dict":{"tenant_desc":"Demo","tenant_name":"mytenant","tenant_roles_admin":["trackme_admin"]'
                + ',"tenant_roles_user":["trackme_user"],"tenant_owner":"admin","tenant_idx_settings":"global",'
                + '"tenant_dsm_enabled":true},'
                + '"hybrid_objects":[{"tenant_id":"mytenant","component":"dsm","tracker_name":"local",'
                + '"account":"local","owner":"admin","search_mode":"tstats","earliest_time":"-4h","latest_time":"+4h"'
                + ',"root_constraint":"(index=* OR index=_*) (index=*) (splunk_server=*) (sourcetype!=stash sourcetype!=*too_small sourcetype!=modular_alerts:trackme* sourcetype!=trackme:*)"'
                + ',"breakby_field":"none"}]}"',
                "options": [
                    {
                        "tenant_dict": "A JSON dictionnary containting the tenant definition",
                        "hybrid_objects": "A JSON array of hybrid trackers definitions to be created and executed",
                        "execute_trackers": "Execute trackers after their creation, valid options are: True / False (default)",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # Get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # Define an header for requests authenticated communications with splunkd
        header = {
            "Authorization": "Splunk %s" % request_info.system_authtoken,
            "Content-Type": "application/json",
        }

        #
        # tenant creation
        #

        # target
        endpoint_url = f"{request_info.server_rest_uri}/services/trackme/v2/vtenants/admin/add_tenant"

        try:
            req_tenant_add_response = requests.post(
                endpoint_url,
                headers=header,
                data=json.dumps(tenant_dict),
                verify=False,
                timeout=600,
            )
            logger.debug(f'response="{req_tenant_add_response.text}"')
            tenant_add_response = req_tenant_add_response.text

            if int(req_tenant_add_response.status_code) != 200:
                logger.error(
                    f'an exception was encountered while attempting to create the new tenant="{json.dumps(tenant_dict, indent=2)}", exception="{tenant_add_response}"'
                )
                return {
                    "payload": {
                        "action": "failure",
                        "response": "An exception was encountered while attempting to create the new tenant",
                        "tenant_dict": tenant_dict,
                        "results": tenant_add_response,
                    },
                    "status": 500,
                }
            else:
                logger.info(
                    f'new tenant was successfully created, tenant_dict="{json.dumps(tenant_dict, indent=2)}", response="{json.dumps(tenant_add_response, indent=2)}"'
                )

        except Exception as e:
            logger.error(
                f'an exception was encountered while attempting to create the new tenant="{json.dumps(tenant_dict, indent=2)}", exception="{str(e)}"'
            )
            return {
                "payload": {
                    "action": "failure",
                    "response": "An exception was encountered while attempting to create the new tenant",
                    "tenant_dict": tenant_dict,
                    "exception": str(e),
                },
                "status": 500,
            }

        #
        # hybrid objects creation
        #

        # store all hybrid objects creation responses in a list
        hybrid_objects_response = []

        # store all hybrid trackers successfully created in a list, we will proceed to their execution in a final phase
        hybrid_objects_trackers = []

        # store all hybrid trackers execution results in a list
        hybrid_objects_execution_results = []

        # endpoint target
        endpoint_url = f"{request_info.server_rest_uri}/services/trackme/v2/splk_wlk/admin/wlk_tracker_create"

        # loop through the hybrid objects to be managed, and proceed
        # If an hybrid object fails to be created, do no exit but report the failure at the end of the whole process

        count_errors_creation = 0
        count_errors_execution = 0
        count_objects = len(hybrid_objects)
        count_processed = 0
        count_executed = 0

        #
        # Loop and create hybrid objects
        #

        for hybrid_object in hybrid_objects:
            logger.info(
                f'proceeding with hybrid object creation, hybrid_object="{json.dumps(hybrid_object, indent=2)}"'
            )

            try:
                req_hybrid_add_response = requests.post(
                    endpoint_url,
                    headers=header,
                    data=json.dumps(hybrid_object),
                    verify=False,
                    timeout=600,
                )
                hybrid_add_response = req_hybrid_add_response.text

                if int(req_hybrid_add_response.status_code) != 200:
                    # increment errors
                    count_errors_creation += 1

                    # log
                    logger.error(
                        f'An exception was encountered while attempting to create the new hybrid_object="{json.dumps(hybrid_object, indent=2)}", hybrid_add_response="{hybrid_add_response}", exception="{req_hybrid_add_response.text}"'
                    )

                    # add to responses
                    hybrid_objects_response.append(
                        {
                            "hybrid_object": hybrid_object,
                            "action": "failure",
                            "status": "An exception was encountered while attempting to create the new hybrid object",
                            "exception": req_hybrid_add_response.text,
                        }
                    )

                else:
                    # we expect a dict
                    if not isinstance(hybrid_add_response, dict):
                        hybrid_add_response = json.loads(hybrid_add_response)

                    logger.info(f'hybrid_add_response="{hybrid_add_response}"')

                    # increment
                    count_processed += 1

                    # log
                    logger.info(
                        f'new hybrid object was successfully created, hybrid_object="{json.dumps(hybrid_object, indent=2)}", response="{json.dumps(hybrid_add_response, indent=2)}"'
                    )

                    # add to responses
                    hybrid_objects_response.append(
                        {
                            "hybrid_object": hybrid_object,
                            "action": "success",
                            "status": "The new hybrid object was processed successfully",
                            "response": hybrid_add_response,
                        }
                    )

                    # add to hybrid objects to executed
                    hybrid_objects_trackers.append(
                        {
                            "tracker_report": hybrid_add_response.get("tracker_report"),
                            "tracker_name": hybrid_add_response.get("tracker_name"),
                            "earliest": hybrid_add_response.get("earliest"),
                            "latest": hybrid_add_response.get("latest"),
                        }
                    )

            except Exception as e:
                # increment errors
                count_errors_creation += 1

                # log
                logger.error(
                    f'An exception was encountered while attempting to create the new hybrid_object="{json.dumps(hybrid_object, indent=2)}", hybrid_add_response="{hybrid_add_response}", exception="{str(e)}"'
                )

                # add to responses
                hybrid_objects_response.append(
                    {
                        "hybrid_object": hybrid_object,
                        "action": "failure",
                        "status": "An exception was encountered while attempting to create the new hybrid object",
                        "exception": str(e),
                    }
                )

        #
        # Loop and execute created hybrid objects
        #

        if execute_trackers:
            for hybrid_object in hybrid_objects_trackers:
                logger.info(
                    f'Attempting to execute hybrid_object="{json.dumps(hybrid_object, indent=2)}"'
                )

                #
                # Splunkd API needs a couple of seconds to refresh while KOs are created
                #

                # set max failed re-attempt
                max_failures_count = 24
                sleep_time = 5
                creation_success = False
                current_failures_count = 0

                while (
                    current_failures_count < max_failures_count and not creation_success
                ):
                    try:
                        newtracker = service.saved_searches[
                            hybrid_object.get("tracker_report")
                        ]
                        logger.info(
                            f'action="success", hybrid tracker was successfully created, report_name="{hybrid_object.get("tracker_report")}"'
                        )
                        creation_success = True
                        break

                    except Exception as e:
                        # We except this sentence in the exception if the API is not ready yet
                        logger.warning(
                            f'temporary failure, the report is not yet available, will sleep and re-attempt, report report_name="{hybrid_object.get("tracker_report")}"'
                        )
                        time.sleep(sleep_time)
                        current_failures_count += 1

                        if current_failures_count >= max_failures_count:
                            logger.error(
                                f'max attempt reached, failure to create report report_name="{hybrid_object.get("tracker_report")}" with exception:"{str(e)}"'
                            )
                            break

                # set args
                kwargs_search = {
                    "app": "trackme",
                    "earliest_time": hybrid_object.get("earliest"),
                    "latest_time": hybrid_object.get("latest"),
                    "output_mode": "json",
                    "count": 0,
                }

                # set query
                searchquery = "| savedsearch " + hybrid_object.get("tracker_report")

                # execute
                query_results = []
                results_count = 0
                start_time = time.time()

                #
                # Splunkd API needs a couple of seconds to refresh while KOs are created
                #

                creation_success = False

                try:
                    reader = run_splunk_search(
                        service,
                        searchquery,
                        kwargs_search,
                        24,
                        5,
                    )

                    for item in reader:
                        if isinstance(item, dict):
                            query_results.append(item)
                            results_count += 1

                    hybrid_objects_execution_results.append(
                        {
                            "action": "success",
                            "status": "The hybrid object was executed successfully",
                            "hybrid_object": hybrid_object.get("tracker_report"),
                            "searchquery": searchquery,
                            "kwargs": kwargs_search,
                            "results": query_results,
                            "results_count": results_count,
                            "runtime": round(time.time() - start_time, 3),
                        }
                    )

                    # increment
                    count_executed += 1

                    # log
                    logger.info(
                        f'The new hybrid object="{hybrid_object.get("tracker_report")}" was successfully executed'
                    )

                    creation_success = True

                except Exception as e:
                    # log
                    logger.error(
                        f'permanent execution failure, An exception was encountered while attempting to execute the new hybrid object="{json.dumps(hybrid_object, indent=2)}", exception="{str(e)}"'
                    )

        # finally, render response, if the tenant failed to be created, we returned a 500 earlier
        # if we have had errors at creation of hybrid trackers, or errors during their execution, return a 500 with the detailed respoonse

        response_record = {
            "response_tenant_creation": tenant_add_response,
            "response_hybrid_objects_creation": hybrid_objects_response,
            "response_hybrid_objects_execution": hybrid_objects_execution_results,
            "count_hybrid_object_requested": count_objects,
            "count_hybrid_trackers_successfully_created": count_processed,
            "count_hybrid_trackers_creation_failures": count_errors_creation,
            "count_hybrid_trackers_successfully_executed": count_executed,
            "count_hybrid_trackers_execution_failures": count_errors_execution,
        }

        logger.info(
            f'add_tenant_multiops has terminated, response="{json.dumps(response_record, indent=2)}"'
        )

        if count_errors_creation > 0 or count_errors_execution > 0:
            response_record["action"] = "failure"
            return {"payload": response_record, "status": 500}

        else:
            response_record["action"] = "success"
            return {"payload": response_record, "status": 200}

    #
    # Delete tenant
    #

    def delete_del_tenant(self, request_info, **kwargs):
        """
        | trackme url=/services/trackme/v2/vtenants/admin/del_tenant mode=delete body="{ 'tenant_id': 'mytenant', 'force': 'true'}"
        """

        # Declare main variables
        tenant_id = None
        query_string = None

        # Declare booleans
        describe = False
        force = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            # option describe
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            # option force
            try:
                force = resp_dict["force"]
                if force in ("true", "True"):
                    force = True
            except Exception as e:
                force = False

            # get tenant_id
            if not describe:
                tenant_id = resp_dict["tenant_id"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be called to delete a virtual tenant, it requires a DELETE call with the following information:",
                "resource_desc": "Delete a virtual tenant",
                "resource_spl_example": "| trackme url=/services/trackme/v2/vtenants/admin/del_tenant mode=delete body=\"{ 'tenant_id': 'mytenant', 'force': 'true'}\"",
                "options": [
                    {
                        "tenant_id": "identifier of the virtual tenant to be deleted",
                        "force": "OPTIONAL: (true|false) if true, runs in force mode and attempts to remove any object linked to the tenant even if it does not exist in the tenants collection",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
            if update_comment == "Enter your audit message":
                update_comment = "No comment was provided for this operation"
        except Exception as e:
            update_comment = "No comment was provided for this operation"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        ################
        # MAIN PROGRAM #
        ################

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        # Data collection
        collection_name = "kv_trackme_virtual_tenants"
        collection = service.kvstore[collection_name]

        # Summary data collection
        collection_summary_name = "kv_trackme_virtual_tenants_entities_summary"
        collection_summary = service.kvstore[collection_summary_name]

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        # default is true (boolean)
        tenant_dsm_enabled = 0
        tenant_dhm_enabled = 0
        tenant_mhm_enabled = 0
        tenant_flx_enabled = 0
        tenant_fqm_enabled = 0
        tenant_wlk_enabled = 0

        # init
        vtenant_record = None

        try:
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]
            key = vtenant_record.get("_key")

        except Exception as e:
            key = None

        # attempt to get components enablement status
        try:
            tenant_dsm_enabled = int(vtenant_record.get("tenant_dsm_enabled"))
        except Exception as e:
            tenant_dsm_enabled = 0
        if force:
            tenant_dsm_enabled = 1

        try:
            tenant_flx_enabled = int(vtenant_record.get("tenant_flx_enabled"))
        except Exception as e:
            tenant_flx_enabled = 0
        if force:
            tenant_flx_enabled = 1

        try:
            tenant_fqm_enabled = int(vtenant_record.get("tenant_fqm_enabled"))
        except Exception as e:
            tenant_fqm_enabled = 0
        if force:
            tenant_fqm_enabled = 1

        try:
            tenant_wlk_enabled = int(vtenant_record.get("tenant_wlk_enabled"))
        except Exception as e:
            tenant_wlk_enabled = 0
        if force:
            tenant_wlk_enabled = 1

        try:
            tenant_dhm_enabled = int(vtenant_record.get("tenant_dhm_enabled"))
        except Exception as e:
            tenant_dhm_enabled = 0
        if force:
            tenant_dhm_enabled = 1

        try:
            tenant_mhm_enabled = int(vtenant_record.get("tenant_mhm_enabled"))
        except Exception as e:
            tenant_mhm_enabled = 0
        if force:
            tenant_mhm_enabled = 1

        try:
            tenant_replica = int(vtenant_record.get("tenant_replica"))
        except Exception as e:
            tenant_replica = 1

        # handle entity found, or force mode
        runop = False
        if key is not None or force:
            runop = True
        else:
            runop = False

        # Render result
        if not runop:
            # This tenant does not exist
            audit_record = {
                "action": "failure",
                "change_type": "delete virtual tenant",
                "object": str(tenant_id),
                "object_category": "virtual_tenants",
                "result": "I'm afraid I can't do that, this tenant identifier could not be found, run the operation again with a different identifier.",
            }

            return {"payload": audit_record, "status": 409}

        else:
            # check if TCM is enabled in receiver mode
            enable_conf_manager_receiver = int(
                trackme_conf["trackme_conf"]["trackme_general"][
                    "enable_conf_manager_receiver"
                ]
            )

            if enable_conf_manager_receiver == 1:
                try:
                    tcm_response = trackme_send_to_tcm(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        resp_dict,
                        "delete",
                        "/services/trackme/v2/vtenants/admin/del_tenant",
                    )
                    logger.info(f"trackme_send_to_tcm was successfully executed")
                except Exception as e:
                    logger.error(
                        f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                    )

            # Define an header for requests authenticated communications with splunkd
            header = {
                "Authorization": "Splunk %s" % request_info.system_authtoken,
                "Content-Type": "application/json",
            }

            # Add the vtenant account
            url = "%s/servicesNS/nobody/trackme/trackme_vtenants/%s" % (
                request_info.server_rest_uri,
                tenant_id,
            )

            # Retrieve and set the tenant idx, if any failure, logs and use the global index
            try:
                response = requests.delete(
                    url, headers=header, verify=False, timeout=600
                )
                if response.status_code not in (200, 201, 204):
                    logger.error(
                        f'delete vtenant account has failed, response.status_code="{response.status_code}", response.text="{response.text}"'
                    )
                else:
                    logger.info(
                        f'delete vtenant account was operated successfully, response.status_code="{response.status_code}"'
                    )
            except Exception as e:
                logger.error(f'delete vtenant account has failed, exception="{str(e)}"')

            # Delete the record from the tenants collection
            if key:
                try:
                    collection.data.delete(json.dumps({"_key": key}))
                except Exception as e:
                    logger.error(
                        f'tenant_id="{tenant_id}", failed to delete the KVstore record from the vtenant collection with exception="{str(e)}"'
                    )

            # Delete the record from the tenants summary collection
            try:
                vtenant_summary_record = collection_summary.data.query(
                    query=json.dumps(query_string)
                )
                summary_key = vtenant_summary_record[0].get("_key")

            except Exception as e:
                vtenant_summary_record = None
                summary_key = None

            if summary_key:
                try:
                    collection_summary.data.delete(json.dumps({"_key": summary_key}))
                except Exception as e:
                    logger.error(
                        f'tenant_id="{tenant_id}", failed to delete the KVstore record from the vtenant summary collection with exception="{str(e)}"'
                    )

            # results_record
            results_record = []

            # DSM
            if tenant_dsm_enabled:
                # not for replica
                if not tenant_replica:
                    # Handle related dedicated Elastic Sources reports, if any
                    elastic_reports = []
                    elastic_collection_name = (
                        "kv_trackme_dsm_elastic_dedicated" + "_tenant_" + str(tenant_id)
                    )

                    logger.info(
                        f'tenant_id="{tenant_id}", checking for elastic dedicated reports, collection="{elastic_collection_name}"'
                    )
                    try:
                        elastic_collection = service.kvstore[elastic_collection_name]
                        elastic_records = elastic_collection.data.query()
                        logger.info(f'elastic_records="{str(elastic_records)}"')

                        # Add each reports to our list
                        for elastic_record in elastic_records:
                            elastic_reports.append(elastic_record.get("elastic_report"))
                            elastic_reports.append(
                                elastic_record.get("elastic_wrapper")
                            )
                        logger.info(f'elastic_reports="{str(elastic_reports)}"')

                    except Exception as e:
                        logger.info(
                            f'tenant_id="{tenant_id}", there are no dedicated Elastic Source reports to be deleted, collection_name="{collection_name}"'
                        )

                    if elastic_reports:
                        for elastic_report in elastic_reports:
                            report_name = elastic_report
                            try:
                                action = trackme_delete_report(
                                    request_info.system_authtoken,
                                    request_info.server_rest_uri,
                                    tenant_id,
                                    report_name,
                                )
                                result = {
                                    "object": report_name,
                                    "object_type": "report",
                                    "action": "delete",
                                    "result": "success",
                                }
                                results_record.append(result)
                            except Exception as e:
                                result = {
                                    "object": report_name,
                                    "object_type": "report",
                                    "action": "delete",
                                    "result": "failure",
                                    "exception": str(e),
                                }
                                results_record.append(result)

                # Handle hybrid objects
                tenant_dsm_hybrid_objects = None

                try:
                    tenant_dsm_hybrid_objects = vtenant_record.get(
                        "tenant_dsm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dsm_hybrid_objects = None

                if tenant_dsm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dsm_hybrid_objects="{tenant_dsm_hybrid_objects}"'
                    )

                    if (
                        tenant_dsm_hybrid_objects
                        and tenant_dsm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dsm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # Handle KVstore collections
                for object_name in collections_list_dsm:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # FLX
            if tenant_flx_enabled:
                # Handle hybrid objects
                tenant_flx_hybrid_objects = None

                try:
                    tenant_flx_hybrid_objects = vtenant_record.get(
                        "tenant_flx_hybrid_objects"
                    )

                except Exception as e:
                    tenant_flx_hybrid_objects = None

                if tenant_flx_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_flx_hybrid_objects="{tenant_flx_hybrid_objects}"'
                    )

                    if (
                        tenant_flx_hybrid_objects
                        and tenant_flx_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_flx_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_flx:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # FQM
            if tenant_fqm_enabled:
                # Handle hybrid objects
                tenant_fqm_hybrid_objects = None

                try:
                    tenant_fqm_hybrid_objects = vtenant_record.get(
                        "tenant_fqm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_fqm_hybrid_objects = None

                if tenant_fqm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_fqm_hybrid_objects="{tenant_fqm_hybrid_objects}"'
                    )

                    if (
                        tenant_fqm_hybrid_objects
                        and tenant_fqm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_fqm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_fqm:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # WLK
            if tenant_wlk_enabled:
                # Handle hybrid objects
                tenant_wlk_hybrid_objects = None

                try:
                    tenant_wlk_hybrid_objects = vtenant_record.get(
                        "tenant_wlk_hybrid_objects"
                    )

                except Exception as e:
                    tenant_wlk_hybrid_objects = None

                if tenant_wlk_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_wlk_hybrid_objects="{tenant_wlk_hybrid_objects}"'
                    )

                    if (
                        tenant_wlk_hybrid_objects
                        and tenant_wlk_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_wlk_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_wlk:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # DHM
            if tenant_dhm_enabled:
                # Handle hybrid objects
                tenant_dhm_hybrid_objects = None

                try:
                    tenant_dhm_hybrid_objects = vtenant_record.get(
                        "tenant_dhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dhm_hybrid_objects = None

                if tenant_dhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dhm_hybrid_objects="{tenant_dhm_hybrid_objects}"'
                    )

                    if (
                        tenant_dhm_hybrid_objects
                        and tenant_dhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_dhm:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    if collection_name in service.kvstore:
                        try:
                            action = trackme_delete_kvcollection(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                collection_name,
                            )
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": collection_name,
                                "object_type": "kvstore collection",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                    # delete transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # MHM
            if tenant_mhm_enabled:
                # Handle hybrid objects
                tenant_mhm_hybrid_objects = None

                try:
                    tenant_mhm_hybrid_objects = vtenant_record.get(
                        "tenant_mhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_mhm_hybrid_objects = None

                if tenant_mhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_mhm_hybrid_objects="{tenant_mhm_hybrid_objects}"'
                    )

                    if (
                        tenant_mhm_hybrid_objects
                        and tenant_mhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_mhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_delete_report(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_delete_macro(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "delete",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_mhm:
                    # del KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # delete transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_delete_kvtransform(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # COMMON

            # replica trackers

            # Handle replica objects
            tenant_replica_objects = None

            try:
                tenant_replica_objects = vtenant_record.get("tenant_replica_objects")

            except Exception as e:
                tenant_replica_objects = None

            if tenant_replica_objects:
                # logger.debug
                logger.debug(f'tenant_replica_objects="{tenant_replica_objects}"')

                if tenant_replica_objects and tenant_replica_objects != "None":
                    vtenant_dict = json.loads(tenant_replica_objects)
                    logger.info(f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"')

                    # get macros and reports
                    try:
                        replica_macros = vtenant_dict["macros"]
                        logger.debug(f'replica_macros="{replica_macros}"')
                    except Exception as e:
                        replica_macros = None

                    try:
                        replica_reports = vtenant_dict["reports"]
                        logger.debug(f'replica_reports="{replica_reports}"')
                    except Exception as e:
                        hybrid_reports = None

                    if replica_reports:
                        # Loop through reports
                        for replica_report in replica_reports:
                            report_name = replica_report
                            try:
                                action = trackme_delete_report(
                                    request_info.system_authtoken,
                                    request_info.server_rest_uri,
                                    tenant_id,
                                    report_name,
                                )
                                result = {
                                    "object": report_name,
                                    "object_type": "report",
                                    "action": "delete",
                                    "result": "success",
                                }
                                results_record.append(result)
                            except Exception as e:
                                result = {
                                    "object": report_name,
                                    "object_type": "report",
                                    "action": "delete",
                                    "result": "failure",
                                    "exception": str(e),
                                }
                                results_record.append(result)

                    if replica_macros:
                        # Loop through macros
                        for replica_macro in replica_macros:
                            macro_name = replica_macro
                            try:
                                action = trackme_delete_macro(
                                    request_info.system_authtoken,
                                    request_info.server_rest_uri,
                                    tenant_id,
                                    macro_name,
                                )
                                result = {
                                    "object": macro_name,
                                    "object_type": "macro",
                                    "action": "delete",
                                    "result": "success",
                                }
                                results_record.append(result)
                            except Exception as e:
                                result = {
                                    "object": macro_name,
                                    "object_type": "macro",
                                    "action": "delete",
                                    "result": "failure",
                                    "exception": str(e),
                                }
                                results_record.append(result)

            # objects
            for object_name in collections_list_common:
                # del KV
                collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                if collection_name in service.kvstore:
                    try:
                        action = trackme_delete_kvcollection(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # delete transform
                transform_name = object_name + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_delete_kvtransform(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        transform_name,
                    )
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "delete",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "delete",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

            # Remove reports

            reports_list = []

            # not for replica
            if not tenant_replica or force:
                # add common reports for the component
                if tenant_dsm_enabled:
                    reports_list.append("trackme_dsm_shared_elastic_tracker")
                    reports_list.append("trackme_dsm_data_sampling_tracker")
                    reports_list.append("trackme_dsm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dsm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dsm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dsm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dsm_tags_tracker")
                    reports_list.append("trackme_dsm_priority_tracker")
                    reports_list.append("trackme_dsm_sla_tracker")

                # add common reports for the component
                if tenant_dhm_enabled:
                    reports_list.append("trackme_dhm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dhm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dhm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dhm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dhm_tags_tracker")
                    reports_list.append("trackme_dhm_priority_tracker")
                    reports_list.append("trackme_dhm_sla_tracker")

                # add common reports for the component
                # none for this one if tracker creation was not requested

                # add reports conditionally
                if tenant_flx_enabled:
                    reports_list.append("trackme_flx_outliers_mltrain_tracker")
                    reports_list.append("trackme_flx_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_flx_inactive_entities_tracker")
                    reports_list.append("trackme_flx_tags_tracker")
                    reports_list.append("trackme_flx_priority_tracker")
                    reports_list.append("trackme_flx_sla_tracker")

                # add reports conditionally
                if tenant_fqm_enabled:
                    reports_list.append("trackme_fqm_outliers_mltrain_tracker")
                    reports_list.append("trackme_fqm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_fqm_inactive_entities_tracker")
                    reports_list.append("trackme_fqm_tags_tracker")
                    reports_list.append("trackme_fqm_priority_tracker")
                    reports_list.append("trackme_fqm_sla_tracker")

                # add reports conditionally
                if tenant_wlk_enabled:
                    reports_list.append("trackme_wlk_outliers_mltrain_tracker")
                    reports_list.append("trackme_wlk_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_wlk_tags_tracker")
                    reports_list.append("trackme_wlk_priority_tracker")
                    reports_list.append("trackme_wlk_sla_tracker")

                # add reports conditionally
                if tenant_mhm_enabled:
                    reports_list.append("trackme_mhm_tags_tracker")
                    reports_list.append("trackme_mhm_priority_tracker")
                    reports_list.append("trackme_mhm_sla_tracker")

            # include the tenant health tracker
            reports_list.append("trackme_health_tracker")

            # log
            logger.info(
                f'tenant_id="{tenant_id}", list of reports to be deleted, reports="{reports_list}"'
            )

            # Loop
            for report in reports_list:
                report_name = str(report) + "_tenant_" + str(tenant_id)

                try:
                    action = trackme_delete_report(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                    )
                    result = {
                        "object": report_name,
                        "object_type": "report",
                        "action": "delete",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": report_name,
                        "object_type": "report",
                        "action": "delete",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

            # Remove macros
            def handle_macro_deletion(
                macro_name, tenant_enabled, system_authtoken, tenant_id, results_record
            ):
                if tenant_enabled:
                    try:
                        action = trackme_delete_macro(
                            system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            macro_name,
                        )
                        result = {
                            "object": macro_name,
                            "object_type": "macro",
                            "action": "delete",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": macro_name,
                            "object_type": "macro",
                            "action": "delete",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            macros_list = [
                "trackme_dhm_default_splk_dhm_alert_policy",
                "trackme_wlk_set_status",
                "trackme_wlk_set_outliers_metrics",
            ]

            components_flags = {
                "_dsm_": tenant_dsm_enabled,
                "_dhm_": tenant_dhm_enabled,
                "_mhm_": tenant_mhm_enabled,
                "_flx_": tenant_flx_enabled,
                "_fqm_": tenant_fqm_enabled,
                "_wlk_": tenant_wlk_enabled,
            }

            for macro in macros_list:
                macro_name = f"{macro}_tenant_{tenant_id}"
                for component, tenant_enabled in components_flags.items():
                    if component in macro_name:
                        handle_macro_deletion(
                            macro_name,
                            tenant_enabled,
                            request_info.system_authtoken,
                            tenant_id,
                            results_record,
                        )
                        break

            #
            # Alerts
            #

            # Handle alerts
            tenant_alert_objects = None

            try:
                tenant_alert_objects = vtenant_record.get("tenant_alert_objects")

            except Exception as e:
                tenant_alert_objects = None

            if tenant_alert_objects:
                # logger.debug
                logger.debug(f'tenant_alert_objects="{tenant_alert_objects}"')

                if tenant_alert_objects and tenant_alert_objects != "None":
                    vtenant_dict = json.loads(tenant_alert_objects)
                    logger.info(f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"')

                    # get reports
                    alerts = vtenant_dict["alerts"]
                    logger.debug(f'alerts="{alerts}"')

                    # Loop through reports
                    for alert_name in alerts:
                        report_name = alert_name
                        try:
                            action = trackme_delete_report(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "delete",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "delete",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

            #######################
            # END OF KOs REMOVAL
            #######################

            # Record an audit change
            user = request_info.user

            # review results record and count failures
            failures_count = 0
            for result_record in results_record:
                try:
                    result_action = result_record.get("result")
                    if result_action == "failure":
                        failures_count += 1
                except Exception as e:
                    logger.error(
                        f'count not find expected action result field in record="{json.dumps(result_record, indent=2)}"'
                    )

            # handle
            if failures_count > 0 and not force:
                result = (
                    "An exception was encountered while attempting to remove one or "
                    "more objects from this tenant, review the audit events for more information"
                )
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "failure",
                    "change_type": "delete virtual tenant",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "delete virtual tenant",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 500}

            else:
                result = "All objects from this tenant were successfully removed"
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "success",
                    "change_type": "delete virtual tenant",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "delete virtual tenant",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 200}

    #
    # Enable tenant
    #

    def post_enable_tenant(self, request_info, **kwargs):
        """
        | trackme url=/services/trackme/v2/vtenants/admin/enable_tenant mode=post body="{ 'tenant_id': 'mytenant'}"
        """

        # Declare main variables
        tenant_id = None
        query_string = None

        # Declare booleans
        describe = False
        force = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            # option describe
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            # option force
            try:
                force = resp_dict["force"]
                if force in ("true", "True"):
                    force = True
            except Exception as e:
                force = False

            # get tenant_id
            if not describe:
                tenant_id = resp_dict["tenant_id"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be called to enable a virtual tenant, it requires a POST call with the following information:",
                "resource_desc": "Enable a virtual tenant",
                "resource_spl_example": "| trackme url=/services/trackme/v2/vtenants/admin/enable_tenant mode=post body=\"{ 'tenant_id': 'mytenant'}\"",
                "options": [
                    {
                        "tenant_id": "identifier of the virtual tenant to be enabled",
                        "force": "OPTIONAL: (true|false) if true, runs in force mode and attempts to handle any object linked to the tenant even if it does not exist in the tenants collection",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
            if update_comment == "Enter your audit message":
                update_comment = "No comment was provided for this operation"
        except Exception as e:
            update_comment = "No comment was provided for this operation"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        ################
        # MAIN PROGRAM #
        ################

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        # Data collection
        collection_name = "kv_trackme_virtual_tenants"
        collection = service.kvstore[collection_name]

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        # default is true (boolean)
        tenant_dsm_enabled = 0
        tenant_dhm_enabled = 0
        tenant_mhm_enabled = 0
        tenant_flx_enabled = 0
        tenant_fqm_enabled = 0
        tenant_wlk_enabled = 0

        # RBAC Metadata
        tenant_owner = "admin"
        tenant_roles_admin = "trackme_admin"
        tenant_roles_user = "trackme_user"
        tenant_roles_power = "trackme_power"

        try:
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]
            key = vtenant_record.get("_key")

            try:
                tenant_dsm_enabled = int(vtenant_record.get("tenant_dsm_enabled"))
            except Exception as e:
                tenant_dsm_enabled = 0

            try:
                tenant_flx_enabled = int(vtenant_record.get("tenant_flx_enabled"))
            except Exception as e:
                tenant_flx_enabled = 0

            try:
                tenant_fqm_enabled = int(vtenant_record.get("tenant_fqm_enabled"))
            except Exception as e:
                tenant_fqm_enabled = 0

            try:
                tenant_wlk_enabled = int(vtenant_record.get("tenant_wlk_enabled"))
            except Exception as e:
                tenant_wlk_enabled = 0

            try:
                tenant_dhm_enabled = int(vtenant_record.get("tenant_dhm_enabled"))
            except Exception as e:
                tenant_dhm_enabled = 0

            try:
                tenant_mhm_enabled = int(vtenant_record.get("tenant_mhm_enabled"))
            except Exception as e:
                tenant_mhm_enabled = 0

            try:
                tenant_replica = int(vtenant_record.get("tenant_replica"))
            except Exception as e:
                tenant_replica = 1

            tenant_owner = str(vtenant_record.get("tenant_owner"))
            tenant_roles_admin = str(vtenant_record.get("tenant_roles_admin"))
            tenant_roles_user = str(vtenant_record.get("tenant_roles_user"))
            tenant_roles_power = str(vtenant_record.get("tenant_roles_power"))
            tenant_dsm_hybrid_objects = str(
                vtenant_record.get("tenant_dsm_hybrid_objects")
            )

        except Exception as e:
            key = None

        # for read permissions, concatenate admin, power and user
        tenant_roles_read_perms = (
            f"{tenant_roles_admin},{tenant_roles_power},{tenant_roles_user}"
        )

        # for write permissions, concatenate admin, power
        tenant_roles_write_perms = f"{tenant_roles_admin},{tenant_roles_power}"

        # handle entity found, or force mode
        runop = False
        if key is not None or force:
            runop = True
        else:
            runop = False

        # Render result
        if not runop:
            # This tenant does not exist
            audit_record = {
                "action": "failure",
                "change_type": "enable virtual tenant",
                "object": str(tenant_id),
                "object_category": "virtual_tenants",
                "result": "I'm afraid I can't do that, this tenant identifier could not be found, run the operation again with a different identifier.",
            }

            return {"payload": audit_record, "status": 409}

        else:
            # Update the record from the tenants collection
            vtenant_record["tenant_status"] = "enabled"
            collection.data.update(str(key), json.dumps(vtenant_record))

            # check if TCM is enabled in receiver mode
            enable_conf_manager_receiver = int(
                trackme_conf["trackme_conf"]["trackme_general"][
                    "enable_conf_manager_receiver"
                ]
            )

            if enable_conf_manager_receiver == 1:
                try:
                    tcm_response = trackme_send_to_tcm(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        resp_dict,
                        "post",
                        "/services/trackme/v2/vtenants/admin/enable_tenant",
                    )
                    logger.info(f"trackme_send_to_tcm was successfully executed")
                except Exception as e:
                    logger.error(
                        f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                    )

            # results_record
            results_record = []

            # DSM
            if tenant_dsm_enabled:
                # Handle related dedicated Elastic Sources reports, if any
                elastic_reports = []
                elastic_collection_name = (
                    "kv_trackme_dsm_elastic_dedicated" + "_tenant_" + str(tenant_id)
                )
                elastic_collection = service.kvstore[elastic_collection_name]

                logger.info(
                    f'tenant_id="{tenant_id}", checking for elastic dedicated reports, collection="{elastic_collection_name}"'
                )
                try:
                    elastic_records = elastic_collection.data.query()
                    logger.info(f'elastic_records="{str(elastic_records)}"')

                    # Add each report to our list
                    for elastic_record in elastic_records:
                        elastic_reports.append(elastic_record.get("elastic_report"))
                        elastic_reports.append(elastic_record.get("elastic_wrapper"))
                    logger.info(f'elastic_reports="{str(elastic_reports)}"')

                except Exception as e:
                    logger.info(
                        f'tenant_id="{tenant_id}", there are no dedicated Elastic Source reports to be updated, collection_name="{collection_name}"'
                    )

                if elastic_reports:
                    for elastic_report in elastic_reports:
                        report_name = elastic_report
                        try:
                            action = trackme_report_update_enablement(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                "enable",
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "enable",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "enable",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                # Handle hybrid objects
                tenant_dsm_hybrid_objects = None

                try:
                    tenant_dsm_hybrid_objects = vtenant_record.get(
                        "tenant_dsm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dsm_hybrid_objects = None

                if tenant_dsm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dsm_hybrid_objects="{tenant_dsm_hybrid_objects}"'
                    )

                    if (
                        tenant_dsm_hybrid_objects
                        and tenant_dsm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dsm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # Handle collections
                for object_name in collections_list_dsm:
                    # enable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "enable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # enable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "enable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # FLX
            if tenant_flx_enabled:
                # Handle hybrid objects
                tenant_flx_hybrid_objects = None

                try:
                    tenant_flx_hybrid_objects = vtenant_record.get(
                        "tenant_flx_hybrid_objects"
                    )

                except Exception as e:
                    tenant_flx_hybrid_objects = None

                if tenant_flx_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_flx_hybrid_objects="{tenant_flx_hybrid_objects}"'
                    )

                    if (
                        tenant_flx_hybrid_objects
                        and tenant_flx_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_flx_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # Handle collections and other knowledge objects
                for object_name in collections_list_flx:
                    # enable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "enable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # enable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "enable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # FQM
            if tenant_fqm_enabled:
                # Handle hybrid objects
                tenant_fqm_hybrid_objects = None

                try:
                    tenant_fqm_hybrid_objects = vtenant_record.get(
                        "tenant_fqm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_fqm_hybrid_objects = None

                if tenant_fqm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_fqm_hybrid_objects="{tenant_fqm_hybrid_objects}"'
                    )

                    if (
                        tenant_fqm_hybrid_objects
                        and tenant_fqm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_fqm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # Handle collections and other knowledge objects
                for object_name in collections_list_fqm:
                    # enable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "enable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # enable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "enable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # WLK
            if tenant_wlk_enabled:
                # Handle hybrid objects
                tenant_wlk_hybrid_objects = None

                try:
                    tenant_wlk_hybrid_objects = vtenant_record.get(
                        "tenant_wlk_hybrid_objects"
                    )

                except Exception as e:
                    tenant_wlk_hybrid_objects = None

                if tenant_wlk_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_wlk_hybrid_objects="{tenant_wlk_hybrid_objects}"'
                    )

                    if (
                        tenant_wlk_hybrid_objects
                        and tenant_wlk_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_wlk_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # Handle collections and other knowledge objects
                for object_name in collections_list_wlk:
                    # enable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "enable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # enable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "enable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # DHM
            if tenant_dhm_enabled:
                # Handle hybrid objects
                tenant_dhm_hybrid_objects = None

                try:
                    tenant_dhm_hybrid_objects = vtenant_record.get(
                        "tenant_dhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dhm_hybrid_objects = None

                if tenant_dhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dhm_hybrid_objects="{tenant_dhm_hybrid_objects}"'
                    )

                    if (
                        tenant_dhm_hybrid_objects
                        and tenant_dhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_dhm:
                    # enable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "enable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # enable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "enable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # MHM
            if tenant_mhm_enabled:
                # Handle hybrid objects
                tenant_mhm_hybrid_objects = None

                try:
                    tenant_mhm_hybrid_objects = vtenant_record.get(
                        "tenant_mhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_mhm_hybrid_objects = None

                if tenant_mhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_mhm_hybrid_objects="{tenant_mhm_hybrid_objects}"'
                    )

                    if (
                        tenant_mhm_hybrid_objects
                        and tenant_mhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_mhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "enable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "enable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_mhm:
                    # enable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "enable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "enable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # enable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "enable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # COMMON
            for object_name in collections_list_common:
                # enable KV
                collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_kvcollection_update_enablement(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        collection_name,
                        "enable",
                    )
                    result = {
                        "object": collection_name,
                        "object_type": "kvstore collection",
                        "action": "enable",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": collection_name,
                        "object_type": "kvstore collection",
                        "action": "enable",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # enable transform
                transform_name = object_name + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_transform_update_enablement(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        transform_name,
                        "enable",
                    )
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "disable",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "disable",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

            # enable reports

            reports_list = []

            # not for replica
            if not tenant_replica:
                # add common reports for the component
                if tenant_dsm_enabled:
                    reports_list.append("trackme_dsm_shared_elastic_tracker")
                    reports_list.append("trackme_dsm_data_sampling_tracker")
                    reports_list.append("trackme_dsm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dsm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dsm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dsm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dsm_tags_tracker")
                    reports_list.append("trackme_dsm_priority_tracker")
                    reports_list.append("trackme_dsm_sla_tracker")

                # add common reports for the component
                if tenant_dhm_enabled:
                    reports_list.append("trackme_dhm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dhm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dhm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dhm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dhm_tags_tracker")
                    reports_list.append("trackme_dhm_priority_tracker")
                    reports_list.append("trackme_dhm_sla_tracker")

                # add common reports for the component
                # none for this one if tracker creation was not requested

                # add reports conditionally
                if tenant_flx_enabled:
                    reports_list.append("trackme_flx_outliers_mltrain_tracker")
                    reports_list.append("trackme_flx_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_flx_inactive_entities_tracker")
                    reports_list.append("trackme_flx_tags_tracker")
                    reports_list.append("trackme_flx_priority_tracker")
                    reports_list.append("trackme_flx_sla_tracker")

                # add reports conditionally
                if tenant_fqm_enabled:
                    reports_list.append("trackme_fqm_outliers_mltrain_tracker")
                    reports_list.append("trackme_fqm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_fqm_inactive_entities_tracker")
                    reports_list.append("trackme_fqm_tags_tracker")
                    reports_list.append("trackme_fqm_priority_tracker")
                    reports_list.append("trackme_fqm_sla_tracker")

                # add reports conditionally
                if tenant_wlk_enabled:
                    reports_list.append("trackme_wlk_outliers_mltrain_tracker")
                    reports_list.append("trackme_wlk_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_wlk_tags_tracker")
                    reports_list.append("trackme_wlk_priority_tracker")
                    reports_list.append("trackme_wlk_sla_tracker")

                # add reports conditionally
                if tenant_mhm_enabled:
                    reports_list.append("trackme_mhm_tags_tracker")
                    reports_list.append("trackme_mhm_priority_tracker")
                    reports_list.append("trackme_mhm_sla_tracker")

            # include the tenant health tracker
            reports_list.append("trackme_health_tracker")

            for report in reports_list:
                report_name = str(report) + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_report_update_enablement(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                        "enable",
                    )
                    result = {
                        "object": report_name,
                        "object_type": "report",
                        "action": "enable",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": report_name,
                        "object_type": "report",
                        "action": "enable",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

            # enable macros
            def handle_macro_enablement(
                macro_name, tenant_enabled, system_authtoken, tenant_id, results_record
            ):
                if tenant_enabled:
                    try:
                        action = trackme_macro_update_enablement(
                            system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            macro_name,
                            "enable",
                        )
                        result = {
                            "object": macro_name,
                            "object_type": "macro",
                            "action": "enable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": macro_name,
                            "object_type": "macro",
                            "action": "enable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            macros_list = [
                "trackme_dhm_default_splk_dhm_alert_policy",
                "trackme_wlk_set_status",
                "trackme_wlk_set_outliers_metrics",
            ]

            components_flags = {
                "_dsm_": tenant_dsm_enabled,
                "_dhm_": tenant_dhm_enabled,
                "_mhm_": tenant_mhm_enabled,
                "_flx_": tenant_flx_enabled,
                "_fqm_": tenant_fqm_enabled,
                "_wlk_": tenant_wlk_enabled,
            }

            for macro in macros_list:
                macro_name = f"{macro}_tenant_{tenant_id}"
                for component, tenant_enabled in components_flags.items():
                    if component in macro_name:
                        handle_macro_enablement(
                            macro_name,
                            tenant_enabled,
                            request_info.system_authtoken,
                            tenant_id,
                            results_record,
                        )
                        break

            #
            # Alerts
            #

            # Handle alerts
            tenant_alert_objects = None

            try:
                tenant_alert_objects = vtenant_record.get("tenant_alert_objects")

            except Exception as e:
                tenant_alert_objects = None

            if tenant_alert_objects:
                # logger.debug
                logger.debug(f'tenant_alert_objects="{tenant_alert_objects}"')

                if tenant_alert_objects and tenant_alert_objects != "None":
                    vtenant_dict = json.loads(tenant_alert_objects)
                    logger.info(f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"')

                    # get reports
                    alerts = vtenant_dict["alerts"]
                    logger.debug(f'alerts="{alerts}"')

                    # Loop through reports
                    for alert_name in alerts:
                        report_name = alert_name
                        try:
                            action = trackme_report_update_enablement(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                "enable",
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "enable",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "enable",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

            ############################
            # END OF KOs ENABLEMENT
            ############################

            # Record an audit change
            user = request_info.user

            # review results record and count failures
            failures_count = 0
            for result_record in results_record:
                try:
                    result_action = result_record.get("result")
                    if result_action == "failure":
                        failures_count += 1
                except Exception as e:
                    logger.error(
                        f'count not find expected action result field in record="{json.dumps(result_record, indent=2)}"'
                    )

            # handle
            if failures_count > 0:
                result = (
                    "An exception was encountered while attempting to enable one or "
                    "more objects from this tenant, review the audit events for more information"
                )
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "failure",
                    "change_type": "enable virtual tenant",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "enable virtual tenant",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 500}

            else:
                result = "All objects from this tenant were successfully enabled"
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "success",
                    "change_type": "enable virtual tenant",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "enable virtual tenant",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 200}

    #
    # Disable tenant
    #

    def post_disable_tenant(self, request_info, **kwargs):
        """
        | trackme url=/services/trackme/v2/vtenants/admin/disable_tenant mode=post body="{ 'tenant_id': 'mytenant'}"
        """

        # Declare main variables
        tenant_id = None
        query_string = None

        # Declare booleans
        describe = False
        force = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            # option describe
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            # option force
            try:
                force = resp_dict["force"]
                if force in ("true", "True"):
                    force = True
            except Exception as e:
                force = False

            # get tenant_id
            if not describe:
                tenant_id = resp_dict["tenant_id"]

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be called to disable a virtual tenant, it requires a POST call with the following information:",
                "resource_desc": "Disable a virtual tenant",
                "resource_spl_example": "| trackme url=/services/trackme/v2/vtenants/admin/disable_tenant mode=post body=\"{ 'tenant_id': 'mytenant'}\"",
                "options": [
                    {
                        "tenant_id": "identifier of the virtual tenant to be disabled",
                        "force": "OPTIONAL: (true|false) if true, runs in force mode and attempts to handle any object linked to the tenant even if it does not exist in the tenants collection",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
            if update_comment == "Enter your audit message":
                update_comment = "No comment was provided for this operation"
        except Exception as e:
            update_comment = "No comment was provided for this operation"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        ################
        # MAIN PROGRAM #
        ################

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        # Data collection
        collection_name = "kv_trackme_virtual_tenants"
        collection = service.kvstore[collection_name]

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        # default is False
        tenant_dsm_enabled = 0
        tenant_dhm_enabled = 0
        tenant_mhm_enabled = 0
        tenant_flx_enabled = 0
        tenant_fqm_enabled = 0
        tenant_wlk_enabled = 0

        # RBAC Metadata
        tenant_owner = "admin"
        tenant_roles_admin = "trackme_admin"
        tenant_roles_user = "trackme_user"
        tenant_roles_power = "trackme_power"

        try:
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]
            key = vtenant_record.get("_key")

            try:
                tenant_dsm_enabled = int(vtenant_record.get("tenant_dsm_enabled"))
            except Exception as e:
                tenant_dsm_enabled = 0

            try:
                tenant_flx_enabled = int(vtenant_record.get("tenant_flx_enabled"))
            except Exception as e:
                tenant_flx_enabled = 0

            try:
                tenant_fqm_enabled = int(vtenant_record.get("tenant_fqm_enabled"))
            except Exception as e:
                tenant_fqm_enabled = 0

            try:
                tenant_wlk_enabled = int(vtenant_record.get("tenant_wlk_enabled"))
            except Exception as e:
                tenant_wlk_enabled = 0

            try:
                tenant_dhm_enabled = int(vtenant_record.get("tenant_dhm_enabled"))
            except Exception as e:
                tenant_dhm_enabled = 0

            try:
                tenant_mhm_enabled = int(vtenant_record.get("tenant_mhm_enabled"))
            except Exception as e:
                tenant_mhm_enabled = 0

            try:
                tenant_replica = int(vtenant_record.get("tenant_replica"))
            except Exception as e:
                tenant_replica = 1

            tenant_owner = str(vtenant_record.get("tenant_owner"))
            tenant_roles_admin = str(vtenant_record.get("tenant_roles_admin"))
            tenant_roles_user = str(vtenant_record.get("tenant_roles_user"))
            tenant_roles_power = str(vtenant_record.get("tenant_roles_power"))
            tenant_dsm_hybrid_objects = str(
                vtenant_record.get("tenant_dsm_hybrid_objects")
            )

        except Exception as e:
            logger.error(
                f'tenant_id="{tenant_id}" was not found in collection="{collection_name}"'
            )
            key = None

        # for read permissions, concatenate admin, power and user
        tenant_roles_read_perms = (
            f"{tenant_roles_admin},{tenant_roles_power},{tenant_roles_user}"
        )

        # for write permissions, concatenate admin, power
        tenant_roles_write_perms = f"{tenant_roles_admin},{tenant_roles_power}"

        # handle entity found, or force mode
        runop = False
        if key is not None or force:
            runop = True
        else:
            runop = False

        # Render result
        if not runop:
            # This tenant does not exist
            audit_record = {
                "action": "failure",
                "change_type": "disable virtual tenant",
                "object": str(tenant_id),
                "object_category": "virtual_tenants",
                "result": "I'm afraid I can't do that, this tenant identifier could not be found, run the operation again with a different identifier.",
            }

            return {"payload": audit_record, "status": 409}

        else:
            # Update the record from the tenants collection
            vtenant_record["tenant_status"] = "disabled"
            collection.data.update(str(key), json.dumps(vtenant_record))

            # check if TCM is enabled in receiver mode
            enable_conf_manager_receiver = int(
                trackme_conf["trackme_conf"]["trackme_general"][
                    "enable_conf_manager_receiver"
                ]
            )

            if enable_conf_manager_receiver == 1:
                try:
                    tcm_response = trackme_send_to_tcm(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        resp_dict,
                        "post",
                        "/services/trackme/v2/vtenants/admin/disable_tenant",
                    )
                    logger.info(f"trackme_send_to_tcm was successfully executed")
                except Exception as e:
                    logger.error(
                        f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                    )

            # results_record
            results_record = []

            # DSM
            if tenant_dsm_enabled:
                # Handle related dedicated Elastic Sources reports, if any
                elastic_reports = []
                elastic_collection_name = (
                    "kv_trackme_dsm_elastic_dedicated" + "_tenant_" + str(tenant_id)
                )
                elastic_collection = service.kvstore[elastic_collection_name]

                logger.info(
                    f'tenant_id="{tenant_id}", checking for elastic dedicated reports, collection="{elastic_collection_name}"'
                )
                try:
                    elastic_records = elastic_collection.data.query()
                    logger.info(f'elastic_records="{str(elastic_records)}"')

                    # Add each report to our list
                    for elastic_record in elastic_records:
                        elastic_reports.append(elastic_record.get("elastic_report"))
                        elastic_reports.append(elastic_record.get("elastic_wrapper"))
                    logger.info(f'elastic_reports="{str(elastic_reports)}"')

                except Exception as e:
                    logger.info(
                        f'tenant_id="{tenant_id}", there are no dedicated Elastic Source reports to be updated, collection_name="{collection_name}"'
                    )

                if elastic_reports:
                    for elastic_report in elastic_reports:
                        report_name = elastic_report
                        try:
                            action = trackme_report_update_enablement(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                "disable",
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "disable",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "disable",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                # Handle hybrid objects
                tenant_dsm_hybrid_objects = None

                try:
                    tenant_dsm_hybrid_objects = vtenant_record.get(
                        "tenant_dsm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dsm_hybrid_objects = None

                if tenant_dsm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dsm_hybrid_objects="{tenant_dsm_hybrid_objects}"'
                    )

                    if (
                        tenant_dsm_hybrid_objects
                        and tenant_dsm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dsm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # Handle collections
                for object_name in collections_list_dsm:
                    # disable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "disable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # disable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "disable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # FLX
            if tenant_flx_enabled:
                # Handle hybrid objects
                tenant_flx_hybrid_objects = None

                try:
                    tenant_flx_hybrid_objects = vtenant_record.get(
                        "tenant_flx_hybrid_objects"
                    )

                except Exception as e:
                    tenant_flx_hybrid_objects = None

                if tenant_flx_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_flx_hybrid_objects="{tenant_flx_hybrid_objects}"'
                    )

                    if (
                        tenant_flx_hybrid_objects
                        and tenant_flx_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_flx_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_flx:
                    # disable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "disable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # disable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "disable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # FQM
            if tenant_fqm_enabled:
                # Handle hybrid objects
                tenant_fqm_hybrid_objects = None

                try:
                    tenant_fqm_hybrid_objects = vtenant_record.get(
                        "tenant_fqm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_fqm_hybrid_objects = None

                if tenant_fqm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_fqm_hybrid_objects="{tenant_fqm_hybrid_objects}"'
                    )

                    if (
                        tenant_fqm_hybrid_objects
                        and tenant_fqm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_fqm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_fqm:
                    # disable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "disable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # disable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "disable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # WLK
            if tenant_wlk_enabled:
                # Handle hybrid objects
                tenant_wlk_hybrid_objects = None

                try:
                    tenant_wlk_hybrid_objects = vtenant_record.get(
                        "tenant_wlk_hybrid_objects"
                    )

                except Exception as e:
                    tenant_wlk_hybrid_objects = None

                if tenant_wlk_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_wlk_hybrid_objects="{tenant_wlk_hybrid_objects}"'
                    )

                    if (
                        tenant_wlk_hybrid_objects
                        and tenant_wlk_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_wlk_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_wlk:
                    # disable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "disable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # disable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "disable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # DHM
            if tenant_dhm_enabled:
                # Handle hybrid objects
                tenant_dhm_hybrid_objects = None

                try:
                    tenant_dhm_hybrid_objects = vtenant_record.get(
                        "tenant_dhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dhm_hybrid_objects = None

                if tenant_dhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dhm_hybrid_objects="{tenant_dhm_hybrid_objects}"'
                    )

                    if (
                        tenant_dhm_hybrid_objects
                        and tenant_dhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_dhm:
                    # disable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "disable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # disable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "disable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # MHM
            if tenant_mhm_enabled:
                # Handle hybrid objects
                tenant_mhm_hybrid_objects = None

                try:
                    tenant_mhm_hybrid_objects = vtenant_record.get(
                        "tenant_mhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_mhm_hybrid_objects = None

                if tenant_mhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_mhm_hybrid_objects="{tenant_mhm_hybrid_objects}"'
                    )

                    if (
                        tenant_mhm_hybrid_objects
                        and tenant_mhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_mhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_enablement(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        "disable",
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "disable",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_mhm:
                    # disable KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            "disable",
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # disable transform
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_enablement(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            "disable",
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # COMMON
            for object_name in collections_list_common:
                # disable KV
                collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_kvcollection_update_enablement(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        collection_name,
                        "disable",
                    )
                    result = {
                        "object": collection_name,
                        "object_type": "kvstore collection",
                        "action": "disable",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": collection_name,
                        "object_type": "kvstore collection",
                        "action": "disable",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # disable transform
                transform_name = object_name + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_transform_update_enablement(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        transform_name,
                        "disable",
                    )
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "disable",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "disable",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

            # Disable reports

            reports_list = []

            # not for replica
            if not tenant_replica:
                # add common reports for the component
                if tenant_dsm_enabled:
                    reports_list.append("trackme_dsm_shared_elastic_tracker")
                    reports_list.append("trackme_dsm_data_sampling_tracker")
                    reports_list.append("trackme_dsm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dsm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dsm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dsm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dsm_tags_tracker")
                    reports_list.append("trackme_dsm_priority_tracker")
                    reports_list.append("trackme_dsm_sla_tracker")

                # add common reports for the component
                if tenant_dhm_enabled:
                    reports_list.append("trackme_dhm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dhm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dhm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dhm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dhm_tags_tracker")
                    reports_list.append("trackme_dhm_priority_tracker")
                    reports_list.append("trackme_dhm_sla_tracker")

                # add common reports for the component
                # none for this one if tracker creation was not requested

                # add reports conditionally
                if tenant_flx_enabled:
                    reports_list.append("trackme_flx_outliers_mltrain_tracker")
                    reports_list.append("trackme_flx_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_flx_inactive_entities_tracker")
                    reports_list.append("trackme_flx_tags_tracker")
                    reports_list.append("trackme_flx_priority_tracker")
                    reports_list.append("trackme_flx_sla_tracker")

                # add reports conditionally
                if tenant_fqm_enabled:
                    reports_list.append("trackme_fqm_outliers_mltrain_tracker")
                    reports_list.append("trackme_fqm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_fqm_inactive_entities_tracker")
                    reports_list.append("trackme_fqm_tags_tracker")
                    reports_list.append("trackme_fqm_priority_tracker")
                    reports_list.append("trackme_fqm_sla_tracker")

                # add reports conditionally
                if tenant_wlk_enabled:
                    reports_list.append("trackme_wlk_outliers_mltrain_tracker")
                    reports_list.append("trackme_wlk_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_wlk_tags_tracker")
                    reports_list.append("trackme_wlk_priority_tracker")
                    reports_list.append("trackme_wlk_sla_tracker")

                # add reports conditionally
                if tenant_mhm_enabled:
                    reports_list.append("trackme_mhm_tags_tracker")
                    reports_list.append("trackme_mhm_priority_tracker")
                    reports_list.append("trackme_mhm_sla_tracker")

            # include the tenant health tracker
            reports_list.append("trackme_health_tracker")

            for report in reports_list:
                report_name = str(report) + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_report_update_enablement(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                        "disable",
                    )
                    result = {
                        "object": report_name,
                        "object_type": "report",
                        "action": "disable",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": report_name,
                        "object_type": "report",
                        "action": "disable",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

            # disable macros
            def handle_macro_disablement(
                macro_name, tenant_enabled, system_authtoken, tenant_id, results_record
            ):
                if tenant_enabled:
                    try:
                        action = trackme_macro_update_enablement(
                            system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            macro_name,
                            "disable",
                        )
                        result = {
                            "object": macro_name,
                            "object_type": "macro",
                            "action": "disable",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": macro_name,
                            "object_type": "macro",
                            "action": "disable",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            macros_list = [
                "trackme_dhm_default_splk_dhm_alert_policy",
                "trackme_wlk_set_status",
                "trackme_wlk_set_outliers_metrics",
            ]

            components_flags = {
                "_dsm_": tenant_dsm_enabled,
                "_dhm_": tenant_dhm_enabled,
                "_mhm_": tenant_mhm_enabled,
                "_flx_": tenant_flx_enabled,
                "_fqm_": tenant_fqm_enabled,
                "_wlk_": tenant_wlk_enabled,
            }

            for macro in macros_list:
                macro_name = f"{macro}_tenant_{tenant_id}"
                for component, tenant_enabled in components_flags.items():
                    if component in macro_name:
                        handle_macro_disablement(
                            macro_name,
                            tenant_enabled,
                            request_info.system_authtoken,
                            tenant_id,
                            results_record,
                        )
                        break

            #
            # Alerts
            #

            # Handle alerts
            tenant_alert_objects = None

            try:
                tenant_alert_objects = vtenant_record.get("tenant_alert_objects")

            except Exception as e:
                tenant_alert_objects = None

            if tenant_alert_objects:
                # logger.debug
                logger.debug(f'tenant_alert_objects="{tenant_alert_objects}"')

                if tenant_alert_objects and tenant_alert_objects != "None":
                    vtenant_dict = json.loads(tenant_alert_objects)
                    logger.info(f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"')

                    # get reports
                    alerts = vtenant_dict["alerts"]
                    logger.debug(f'alerts="{alerts}"')

                    # Loop through reports
                    for alert_name in alerts:
                        report_name = alert_name
                        try:
                            action = trackme_report_update_enablement(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                "disable",
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "disable",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "disable",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

            ############################
            # END OF KOs DISABLEMENENT
            ############################

            # Record an audit change
            user = request_info.user

            # review results record and count failures
            failures_count = 0
            for result_record in results_record:
                try:
                    result_action = result_record.get("result")
                    if result_action == "failure":
                        failures_count += 1
                except Exception as e:
                    logger.error(
                        f'count not find expected action result field in record="{json.dumps(result_record, indent=2)}"'
                    )

            # handle
            if failures_count > 0:
                result = (
                    "An exception was encountered while attempting to disable one or "
                    "more objects from this tenant, review the audit events for more information"
                )
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "failure",
                    "change_type": "disable virtual tenant",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "delete virtual tenant",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 200}

            else:
                result = "All objects from this tenant were successfully disabled"
                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "success",
                    "change_type": "disable virtual tenant",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "success",
                        "delete virtual tenant",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 200}

    #
    # Update tenant RBAC and ownership
    #

    def post_update_tenant_rbac(self, request_info, **kwargs):
        """
        | trackme url=/services/trackme/v2/vtenants/admin/update_tenant_rbac mode=post body="{ 'tenant_id': 'mytenant', 'tenant_roles_admin': 'emea_admin', 'tenant_roles_power': 'emea_power', 'tenant_roles_user': 'emea_user', 'tenant_owner': 'svc-trackme'}"
        """

        # Declare main variables
        tenant_id = None
        query_string = None

        # Declare booleans
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            # option describe
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            # get tenant_id
            if not describe:
                tenant_id = resp_dict["tenant_id"]
                tenant_owner = resp_dict["tenant_owner"]
                tenant_roles_admin = resp_dict["tenant_roles_admin"]
                if isinstance(tenant_roles_admin, list):
                    tenant_roles_admin = ",".join(resp_dict["tenant_roles_admin"])
                tenant_roles_user = resp_dict["tenant_roles_user"]
                if isinstance(tenant_roles_user, list):
                    tenant_roles_user = ",".join(resp_dict["tenant_roles_user"])
                tenant_roles_power = resp_dict["tenant_roles_power"]
                if isinstance(tenant_roles_power, list):
                    tenant_roles_power = ",".join(resp_dict["tenant_roles_power"])

        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be called to updated RBAC on a virtual tenant, it requires a POST call with the following information:",
                "resource_desc": "Update RBAC for a virtual tenant",
                "resource_spl_example": "| trackme url=/services/trackme/v2/vtenants/admin/update_tenant_rbac mode=post body=\"{ 'tenant_id': 'mytenant', 'tenant_roles_admin': 'emea_admin', 'tenant_roles_power': 'emea_power', 'tenant_roles_user': 'emea_user', 'tenant_owner': 'svc-trackme'}\"",
                "options": [
                    {
                        "tenant_id": "identifier of the virtual tenant to be enabled",
                        "tenant_owner": "OPTIONAL: the user owner of all objects to be created for this tenant, if not specified defaults to the system wide configured owner (nobody by default)",
                        "tenant_roles_admin": "OPTIONAL: a comma separated list of Splunk roles which will be given full control permissions on the tenant objects, defaults to the builtin role trackme_admin",
                        "tenant_roles_power": "OPTIONAL: a comma separated list of Splunk roles which will be given power permissions on the tenant objects, defaults to the builtin role trackme_power",
                        "tenant_roles_user": "OPTIONAL: a comma separated list of Splunk roles which will be given read only permissions on the tenant objects, defaults to the builtin role trackme_user",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict["update_comment"]
            if update_comment == "Enter your audit message":
                update_comment = "No comment was provided for this operation"
        except Exception as e:
            update_comment = "No comment was provided for this operation"

        # Get splunkd port
        splunkd_port = request_info.server_rest_port

        # get service
        service = client.connect(
            owner="nobody",
            app="trackme",
            port=splunkd_port,
            token=request_info.system_authtoken,
            timeout=600,
        )

        # set loglevel
        loglevel = trackme_getloglevel(
            request_info.system_authtoken, request_info.server_rest_port
        )
        logger.setLevel(loglevel)

        # get TrackMe conf
        trackme_conf = trackme_reqinfo(
            request_info.system_authtoken, request_info.server_rest_uri
        )
        logger.debug(f'trackme_conf="{json.dumps(trackme_conf, indent=2)}"')

        # TrackMe sharing level
        trackme_default_sharing = trackme_conf["trackme_conf"]["trackme_general"][
            "trackme_default_sharing"
        ]

        ################
        # MAIN PROGRAM #
        ################

        # Get the current record
        # Notes: the record is returned as an array, as we search for a specific record, we expect one record only

        # Data collection
        collection_name = "kv_trackme_virtual_tenants"
        collection = service.kvstore[collection_name]

        # Define the KV query search string
        query_string = {
            "tenant_id": tenant_id,
        }

        # default is False
        tenant_dsm_enabled = 0
        tenant_flx_enabled = 0
        tenant_dhm_enabled = 0
        tenant_mhm_enabled = 0
        tenant_flx_enabled = 0
        tenant_fqm_enabled = 0

        try:
            vtenant_record = collection.data.query(query=json.dumps(query_string))[0]
            key = vtenant_record.get("_key")

            try:
                tenant_dsm_enabled = int(vtenant_record.get("tenant_dsm_enabled"))
            except Exception as e:
                tenant_dsm_enabled = 0

            try:
                tenant_flx_enabled = int(vtenant_record.get("tenant_flx_enabled"))
            except Exception as e:
                tenant_flx_enabled = 0

            try:
                tenant_fqm_enabled = int(vtenant_record.get("tenant_fqm_enabled"))
            except Exception as e:
                tenant_fqm_enabled = 0

            try:
                tenant_wlk_enabled = int(vtenant_record.get("tenant_wlk_enabled"))
            except Exception as e:
                tenant_wlk_enabled = 0

            try:
                tenant_dhm_enabled = int(vtenant_record.get("tenant_dhm_enabled"))
            except Exception as e:
                tenant_dhm_enabled = 0

            try:
                tenant_mhm_enabled = int(vtenant_record.get("tenant_mhm_enabled"))
            except Exception as e:
                tenant_mhm_enabled = 0

            try:
                tenant_replica = int(vtenant_record.get("tenant_replica"))
            except Exception as e:
                tenant_replica = 1

        except Exception as e:
            key = None

        # for read permissions, concatenate admin, power and user
        tenant_roles_read_perms = (
            f"{tenant_roles_admin},{tenant_roles_power},{tenant_roles_user}"
        )

        # for write permissions, concatenate admin, power
        tenant_roles_write_perms = f"{tenant_roles_admin},{tenant_roles_power}"

        # handle entity found
        runop = False
        if key is not None:
            runop = True
        else:
            runop = False

        # Render result
        if not runop:
            # This tenant does not exist
            audit_record = {
                "action": "failure",
                "change_type": "update virtual tenant RBAC",
                "object": str(tenant_id),
                "object_category": "virtual_tenants",
                "result": "I'm afraid I can't do that, this tenant identifier could not be found, run the operation again with a different identifier.",
            }

            return {"payload": audit_record, "status": 409}

        else:
            # check if TCM is enabled in receiver mode
            enable_conf_manager_receiver = int(
                trackme_conf["trackme_conf"]["trackme_general"][
                    "enable_conf_manager_receiver"
                ]
            )

            if enable_conf_manager_receiver == 1:
                try:
                    tcm_response = trackme_send_to_tcm(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        resp_dict,
                        "post",
                        "/services/trackme/v2/vtenants/admin/update_tenant_rbac",
                    )
                    logger.info(f"trackme_send_to_tcm was successfully executed")
                except Exception as e:
                    logger.error(
                        f'trackme_send_to_tcm has failed with exception="{str(e)}"'
                    )

            # results_record
            results_record = []

            # Args
            kwargs = {
                "sharing": trackme_default_sharing,
                "owner": str(tenant_owner),
                "perms.write": str(tenant_roles_write_perms),
                "perms.read": str(tenant_roles_read_perms),
            }

            #
            # main collections and transforms:
            # - ensure that requested roles in the tenant are present in the ACL for read permissions for KVstore and transforms
            #

            for object_name in collections_list_main:

                kv_acl_read_perms = []
                kv_acl_write_perms = []
                kv_acl_owner = None
                kv_acl_sharing = None
                kv_acl_update_is_required = False

                transforms_acl_read_perms = []
                transforms_acl_write_perms = []
                transforms_acl_owner = None
                transforms_acl_sharing = None
                transforms_acl_update_is_required = False

                # get KV acl
                collection_name = f"kv_{object_name}"
                try:
                    kv_acl = trackme_kvcollection_get_acl(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        collection_name,
                    )

                    # Extract kv_acl_read_perms
                    kv_acl_read_perms = kv_acl["entry"][0]["acl"]["perms"]["read"]
                    kv_acl_write_perms = kv_acl["entry"][0]["acl"]["perms"]["write"]
                    kv_acl_owner = kv_acl["entry"][0]["acl"]["owner"]
                    kv_acl_sharing = kv_acl["entry"][0]["acl"]["sharing"]
                    logger.debug(
                        f"collection_name={collection_name}, kv_acl_read_perms={kv_acl_read_perms}"
                    )

                except Exception as e:
                    kv_acl = None
                    kv_acl_read_perms = []
                    logger.error(
                        f'failed to get KVstore current ACL, collection_name="{collection_name}", exception="{str(e)}"'
                    )

                # check the KV acl, each role defined in the tenant_roles_read_perms should be present, if it is not, add it to the list
                if kv_acl_read_perms:
                    # requested perms read list
                    requested_perms_read_list = tenant_roles_read_perms.split(",")
                    for requested_role in requested_perms_read_list:
                        if (
                            requested_role not in kv_acl_read_perms
                            and requested_role
                            not in (
                                "admin",
                                "trackme_admin",
                                "trackme_power",
                                "trackme_user",
                            )
                        ):
                            kv_acl_update_is_required = True
                            # append to the list
                            kv_acl_read_perms.append(requested_role)

                # Update KV acl if necessary
                if kv_acl_update_is_required:
                    logger.info(
                        f'RBAC update is required for main KV collection, collection_name="{collection_name}", kv_acl_update_is_required="{kv_acl_update_is_required}", kv_acl_read_perms="{kv_acl_read_perms}"'
                    )

                    try:
                        action = trackme_kvcollection_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            {
                                "sharing": kv_acl_sharing,
                                "owner": kv_acl_owner,
                                "perms.write": ",".join(kv_acl_write_perms),
                                "perms.read": ",".join(kv_acl_read_perms),
                            },
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                # get transforms ACL
                transform_name = f"{object_name}"
                try:
                    transform_acl = trackme_transform_get_acl(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        transform_name,
                    )

                    # Extract transforms_acl_read_perms
                    transforms_acl_read_perms = transform_acl["entry"][0]["acl"][
                        "perms"
                    ]["read"]
                    transforms_acl_write_perms = transform_acl["entry"][0]["acl"][
                        "perms"
                    ]["write"]
                    transforms_acl_owner = transform_acl["entry"][0]["acl"]["owner"]
                    transforms_acl_sharing = transform_acl["entry"][0]["acl"]["sharing"]
                    logger.debug(
                        f"transform_name={transform_name}, transforms_acl_read_perms={transforms_acl_read_perms}"
                    )

                except Exception as e:
                    transform_acl = None
                    transforms_acl_read_perms = []
                    logger.error(
                        f'failed to get transform current ACL, transform_name="{transform_name}", exception="{str(e)}"'
                    )

                # check the transforms acl, each role defined in the tenant_roles_read_perms should be present, if it is not, add it to the list
                if transforms_acl_read_perms:
                    # requested perms read list
                    requested_perms_read_list = tenant_roles_read_perms.split(",")
                    for requested_role in requested_perms_read_list:
                        if (
                            requested_role not in transforms_acl_read_perms
                            and requested_role
                            not in (
                                "admin",
                                "trackme_admin",
                                "trackme_power",
                                "trackme_user",
                            )
                        ):
                            transforms_acl_update_is_required = True
                            # append to the list
                            transforms_acl_read_perms.append(requested_role)

                # Update transforms acl if necessary
                if transforms_acl_update_is_required:
                    logger.info(
                        f'RBAC update is required for main KV transforms, transform_name="{transform_name}", transforms_acl_update_is_required="{transforms_acl_update_is_required}", transforms_acl_read_perms="{transforms_acl_read_perms}"'
                    )

                    try:
                        action = trackme_transform_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            {
                                "sharing": transforms_acl_sharing,
                                "owner": transforms_acl_owner,
                                "perms.write": ",".join(transforms_acl_write_perms),
                                "perms.read": ",".join(transforms_acl_read_perms),
                            },
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # DSM
            if tenant_dsm_enabled:
                # Handle related dedicated Elastic Sources reports, if any
                elastic_reports = []
                elastic_collection_name = (
                    "kv_trackme_dsm_elastic_dedicated" + "_tenant_" + str(tenant_id)
                )
                elastic_collection = service.kvstore[elastic_collection_name]

                logger.info(
                    f'tenant_id="{tenant_id}", checking for elastic dedicated reports, collection="{elastic_collection_name}"'
                )
                try:
                    elastic_records = elastic_collection.data.query()
                    logger.info(f'elastic_records="{str(elastic_records)}"')

                    # Add each report to our list
                    for elastic_record in elastic_records:
                        elastic_reports.append(elastic_record.get("elastic_report"))
                        elastic_reports.append(elastic_record.get("elastic_wrapper"))
                    logger.info(f'elastic_reports="{str(elastic_reports)}"')

                except Exception as e:
                    logger.info(
                        f'tenant_id="{tenant_id}", there are no dedicated Elastic Source reports to be updated, collection_name="{elastic_collection_name}"'
                    )

                if elastic_reports:
                    for elastic_report in elastic_reports:
                        report_name = elastic_report
                        try:
                            action = trackme_report_update_acl(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                kwargs,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "update rbac",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "update rbac",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

                # Handle hybrid objects
                tenant_dsm_hybrid_objects = None

                try:
                    tenant_dsm_hybrid_objects = vtenant_record.get(
                        "tenant_dsm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dsm_hybrid_objects = None

                if tenant_dsm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dsm_hybrid_objects="{tenant_dsm_hybrid_objects}"'
                    )

                    if (
                        tenant_dsm_hybrid_objects
                        and tenant_dsm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dsm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # Handle collections
                for object_name in collections_list_dsm:
                    # update KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            kwargs,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # update transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            kwargs,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # FLX
            if tenant_flx_enabled:
                # Handle hybrid objects
                tenant_flx_hybrid_objects = None

                try:
                    tenant_flx_hybrid_objects = vtenant_record.get(
                        "tenant_flx_hybrid_objects"
                    )

                except Exception as e:
                    tenant_flx_hybrid_objects = None

                if tenant_flx_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_flx_hybrid_objects="{tenant_flx_hybrid_objects}"'
                    )

                    if (
                        tenant_flx_hybrid_objects
                        and tenant_flx_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_flx_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_flx:
                    # update KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            kwargs,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # update transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            kwargs,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # FQM
            if tenant_fqm_enabled:
                # Handle hybrid objects
                tenant_fqm_hybrid_objects = None

                try:
                    tenant_fqm_hybrid_objects = vtenant_record.get(
                        "tenant_fqm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_fqm_hybrid_objects = None

                if tenant_fqm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_fqm_hybrid_objects="{tenant_fqm_hybrid_objects}"'
                    )

                    if (
                        tenant_fqm_hybrid_objects
                        and tenant_fqm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_fqm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_fqm:
                    # update KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            kwargs,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # update transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            kwargs,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # WLK
            if tenant_wlk_enabled:
                # Handle hybrid objects
                tenant_wlk_hybrid_objects = None

                try:
                    tenant_wlk_hybrid_objects = vtenant_record.get(
                        "tenant_wlk_hybrid_objects"
                    )

                except Exception as e:
                    tenant_wlk_hybrid_objects = None

                if tenant_wlk_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_wlk_hybrid_objects="{tenant_wlk_hybrid_objects}"'
                    )

                    if (
                        tenant_wlk_hybrid_objects
                        and tenant_wlk_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_wlk_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                # handle collections and other knowledge objects
                for object_name in collections_list_wlk:
                    # update KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            kwargs,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # update transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            kwargs,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # DHM
            if tenant_dhm_enabled:
                # Handle hybrid objects
                tenant_dhm_hybrid_objects = None

                try:
                    tenant_dhm_hybrid_objects = vtenant_record.get(
                        "tenant_dhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_dhm_hybrid_objects = None

                if tenant_dhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_dhm_hybrid_objects="{tenant_dhm_hybrid_objects}"'
                    )

                    if (
                        tenant_dhm_hybrid_objects
                        and tenant_dhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_dhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_dhm:
                    # update KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            kwargs,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # update transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            kwargs,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # MHM
            if tenant_mhm_enabled:
                # Handle hybrid objects
                tenant_mhm_hybrid_objects = None

                try:
                    tenant_mhm_hybrid_objects = vtenant_record.get(
                        "tenant_mhm_hybrid_objects"
                    )

                except Exception as e:
                    tenant_mhm_hybrid_objects = None

                if tenant_mhm_hybrid_objects:
                    # logger.debug
                    logger.debug(
                        f'tenant_mhm_hybrid_objects="{tenant_mhm_hybrid_objects}"'
                    )

                    if (
                        tenant_mhm_hybrid_objects
                        and tenant_mhm_hybrid_objects != "None"
                    ):
                        vtenant_dict = json.loads(tenant_mhm_hybrid_objects)
                        logger.info(
                            f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"'
                        )

                        # get macros and reports
                        try:
                            hybrid_macros = vtenant_dict["macros"]
                            logger.debug(f'hybrid_macros="{hybrid_macros}"')
                        except Exception as e:
                            hybrid_macros = None

                        try:
                            hybrid_reports = vtenant_dict["reports"]
                            logger.debug(f'hybrid_reports="{hybrid_reports}"')
                        except Exception as e:
                            hybrid_reports = None

                        if hybrid_reports:
                            # Loop through reports
                            for hybrid_report in hybrid_reports:
                                report_name = hybrid_report
                                try:
                                    action = trackme_report_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        report_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": report_name,
                                        "object_type": "report",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                        if hybrid_macros:
                            # Loop through macros
                            for hybrid_macro in hybrid_macros:
                                macro_name = hybrid_macro
                                try:
                                    action = trackme_macro_update_acl(
                                        request_info.system_authtoken,
                                        request_info.server_rest_uri,
                                        tenant_id,
                                        macro_name,
                                        kwargs,
                                    )
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "success",
                                    }
                                    results_record.append(result)
                                except Exception as e:
                                    result = {
                                        "object": macro_name,
                                        "object_type": "macro",
                                        "action": "update rbac",
                                        "result": "failure",
                                        "exception": str(e),
                                    }
                                    results_record.append(result)

                for object_name in collections_list_mhm:
                    # update KV
                    collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_kvcollection_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            collection_name,
                            kwargs,
                        )
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": collection_name,
                            "object_type": "kvstore collection",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

                    # update transforms
                    transform_name = object_name + "_tenant_" + str(tenant_id)
                    try:
                        action = trackme_transform_update_acl(
                            request_info.system_authtoken,
                            request_info.server_rest_uri,
                            tenant_id,
                            transform_name,
                            kwargs,
                        )
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": transform_name,
                            "object_type": "transform",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            # COMMON
            for object_name in collections_list_common:
                # enable KV
                collection_name = "kv_" + object_name + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_kvcollection_update_acl(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        collection_name,
                        kwargs,
                    )
                    result = {
                        "object": collection_name,
                        "object_type": "kvstore collection",
                        "action": "update rbac",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": collection_name,
                        "object_type": "kvstore collection",
                        "action": "update rbac",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

                # update transforms
                transform_name = object_name + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_transform_update_acl(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        transform_name,
                        kwargs,
                    )
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "update rbac",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": transform_name,
                        "object_type": "transform",
                        "action": "update rbac",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

            # enable reports

            reports_list = []

            # not for replica
            if not tenant_replica:
                # add common reports for the component
                if tenant_dsm_enabled:
                    reports_list.append("trackme_dsm_shared_elastic_tracker")
                    reports_list.append("trackme_dsm_data_sampling_tracker")
                    reports_list.append("trackme_dsm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dsm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dsm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dsm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dsm_tags_tracker")
                    reports_list.append("trackme_dsm_priority_tracker")
                    reports_list.append("trackme_dsm_sla_tracker")

                # add common reports for the component
                if tenant_dhm_enabled:
                    reports_list.append("trackme_dhm_outliers_mltrain_tracker")
                    reports_list.append("trackme_dhm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_dhm_adaptive_delay_tracker")
                    reports_list.append(
                        "trackme_dhm_delayed_entities_inspector_tracker"
                    )
                    reports_list.append("trackme_dhm_tags_tracker")
                    reports_list.append("trackme_dhm_priority_tracker")
                    reports_list.append("trackme_dhm_sla_tracker")

                # add common reports for the component
                # none for this one if tracker creation was not requested

                # add reports conditionally
                if tenant_flx_enabled:
                    reports_list.append("trackme_flx_outliers_mltrain_tracker")
                    reports_list.append("trackme_flx_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_flx_inactive_entities_tracker")
                    reports_list.append("trackme_flx_tags_tracker")
                    reports_list.append("trackme_flx_priority_tracker")
                    reports_list.append("trackme_flx_sla_tracker")

                # add reports conditionally
                if tenant_fqm_enabled:
                    reports_list.append("trackme_fqm_outliers_mltrain_tracker")
                    reports_list.append("trackme_fqm_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_fqm_inactive_entities_tracker")
                    reports_list.append("trackme_fqm_tags_tracker")
                    reports_list.append("trackme_fqm_priority_tracker")
                    reports_list.append("trackme_fqm_sla_tracker")

                # add reports conditionally
                if tenant_wlk_enabled:
                    reports_list.append("trackme_wlk_outliers_mltrain_tracker")
                    reports_list.append("trackme_wlk_outliers_mlmonitor_tracker")
                    reports_list.append("trackme_wlk_tags_tracker")
                    reports_list.append("trackme_wlk_priority_tracker")
                    reports_list.append("trackme_wlk_sla_tracker")

                # add reports conditionally
                if tenant_mhm_enabled:
                    reports_list.append("trackme_mhm_tags_tracker")
                    reports_list.append("trackme_mhm_priority_tracker")
                    reports_list.append("trackme_mhm_sla_tracker")

            # include the tenant health tracker
            reports_list.append("trackme_health_tracker")

            # Handler the owner (cannot be performed via splunklib)
            kwargs = {
                "sharing": trackme_default_sharing,
                "owner": str(tenant_owner),
                "perms.write": str(tenant_roles_write_perms),
                "perms.read": str(tenant_roles_read_perms),
            }

            for report in reports_list:
                report_name = str(report) + "_tenant_" + str(tenant_id)
                try:
                    action = trackme_report_update_acl(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        report_name,
                        kwargs,
                    )
                    result = {
                        "object": report_name,
                        "object_type": "report",
                        "action": "update rbac",
                        "result": "success",
                    }
                    results_record.append(result)
                except Exception as e:
                    result = {
                        "object": report_name,
                        "object_type": "report",
                        "action": "update rbac",
                        "result": "failure",
                        "exception": str(e),
                    }
                    results_record.append(result)

            # update macros acl
            def handle_macro_update_acl(
                macro_name,
                splunkd_uri,
                tenant_enabled,
                system_authtoken,
                tenant_id,
                kwargs,
                results_record,
            ):
                if tenant_enabled:
                    try:
                        action = trackme_macro_update_acl(
                            system_authtoken, splunkd_uri, tenant_id, macro_name, kwargs
                        )
                        result = {
                            "object": macro_name,
                            "object_type": "macro",
                            "action": "update rbac",
                            "result": "success",
                        }
                        results_record.append(result)
                    except Exception as e:
                        result = {
                            "object": macro_name,
                            "object_type": "macro",
                            "action": "update rbac",
                            "result": "failure",
                            "exception": str(e),
                        }
                        results_record.append(result)

            macros_list = [
                "trackme_dhm_default_splk_dhm_alert_policy",
                "trackme_wlk_set_status",
                "trackme_wlk_set_outliers_metrics",
            ]

            components_flags = {
                "_dsm_": tenant_dsm_enabled,
                "_dhm_": tenant_dhm_enabled,
                "_mhm_": tenant_mhm_enabled,
                "_flx_": tenant_flx_enabled,
                "_fqm_": tenant_fqm_enabled,
                "_wlk_": tenant_wlk_enabled,
            }

            for macro in macros_list:
                macro_name = f"{macro}_tenant_{tenant_id}"
                for component, tenant_enabled in components_flags.items():
                    if component in macro_name:
                        handle_macro_update_acl(
                            macro_name,
                            request_info.server_rest_uri,
                            tenant_enabled,
                            request_info.system_authtoken,
                            tenant_id,
                            kwargs,
                            results_record,
                        )
                        break

            #
            # Alerts
            #

            # Handle alerts
            tenant_alert_objects = None

            # Handler the owner (cannot be performed via splunklib)
            kwargs = {
                "sharing": trackme_default_sharing,
                "owner": str(tenant_owner),
                "perms.write": str(tenant_roles_write_perms),
                "perms.read": str(tenant_roles_read_perms),
            }

            try:
                tenant_alert_objects = vtenant_record.get("tenant_alert_objects")

            except Exception as e:
                tenant_alert_objects = None

            if tenant_alert_objects:
                # logger.debug
                logger.debug(f'tenant_alert_objects="{tenant_alert_objects}"')

                if tenant_alert_objects and tenant_alert_objects != "None":
                    vtenant_dict = json.loads(tenant_alert_objects)
                    logger.info(f'vtenant_dict="{json.dumps(vtenant_dict, indent=1)}"')

                    # get reports
                    alerts = vtenant_dict["alerts"]
                    logger.debug(f'alerts="{alerts}"')

                    # Loop through reports
                    for alert_name in alerts:
                        report_name = alert_name
                        try:
                            action = trackme_report_update_acl(
                                request_info.system_authtoken,
                                request_info.server_rest_uri,
                                tenant_id,
                                report_name,
                                kwargs,
                            )
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "update rbac",
                                "result": "success",
                            }
                            results_record.append(result)
                        except Exception as e:
                            result = {
                                "object": report_name,
                                "object_type": "report",
                                "action": "update rbac",
                                "result": "failure",
                                "exception": str(e),
                            }
                            results_record.append(result)

            ###############################
            # END OF KOs RBAC REASSIGNEMENT
            ###############################

            # Record an audit change
            user = request_info.user

            # review results record and count failures
            failures_count = 0
            for result_record in results_record:
                try:
                    result_action = result_record.get("result")
                    if result_action == "failure":
                        failures_count += 1
                except Exception as e:
                    logger.error(
                        f'count not find expected action result field in record="{json.dumps(result_record, indent=2)}"'
                    )

            # handle
            if failures_count > 0:
                orig_rbac = {
                    "tenant_owner": vtenant_record.get("tenant_owner"),
                    "tenant_roles_admin": vtenant_record.get("tenant_roles_admin"),
                    "tenant_roles_power": vtenant_record.get("tenant_roles_power"),
                    "tenant_roles_user": vtenant_record.get("tenant_roles_user"),
                }

                dest_rbac = {
                    "tenant_owner": tenant_owner,
                    "tenant_roles_admin": tenant_roles_admin,
                    "tenant_roles_power": tenant_roles_power,
                    "tenant_roles_user": tenant_roles_user,
                }

                result = (
                    "An exception was encountered while attemting to update one or"
                    "more objects from this tenant, review the audit events for more information. "
                    f"RBAC policy change requested from: {json.dumps(orig_rbac, indent=2)} to {json.dumps(dest_rbac, indent=2)}"
                )

                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "failure",
                    "change_type": "enable virtual tenant",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                # index the audit record
                try:
                    trackme_audit_event(
                        request_info.system_authtoken,
                        request_info.server_rest_uri,
                        tenant_id,
                        request_info.user,
                        "failure",
                        "update RBAC virtual tenant",
                        str(tenant_id),
                        "virtual_tenants",
                        json.dumps(vtenant_record),
                        json.dumps(results_record),
                        str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 200}

            else:
                orig_rbac = {
                    "tenant_owner": vtenant_record.get("tenant_owner"),
                    "tenant_roles_admin": vtenant_record.get("tenant_roles_admin"),
                    "tenant_roles_power": vtenant_record.get("tenant_roles_power"),
                    "tenant_roles_user": vtenant_record.get("tenant_roles_user"),
                }

                dest_rbac = {
                    "tenant_owner": tenant_owner,
                    "tenant_roles_admin": tenant_roles_admin,
                    "tenant_roles_power": tenant_roles_power,
                    "tenant_roles_user": tenant_roles_user,
                }

                result = (
                    "All objects from this tenant were successfully updated, "
                    + f"RBAC policy change requested from: {json.dumps(orig_rbac, indent=2)} to {json.dumps(dest_rbac, indent=2)}"
                )

                audit_record = {
                    "_time": str(time.time()),
                    "timeStr": str(datetime.datetime.now()),
                    "user": str(user),
                    "action": "success",
                    "change_type": "update RBAC virtual tenant",
                    "object": str(tenant_id),
                    "object_category": "virtual_tenants",
                    "object_attrs": vtenant_record,
                    "results_summary": str(result),
                    "results_details": results_record,
                    "comment": str(update_comment),
                }

                vtenant_record["tenant_owner"] = tenant_owner
                vtenant_record["tenant_roles_admin"] = tenant_roles_admin
                vtenant_record["tenant_roles_power"] = tenant_roles_power
                vtenant_record["tenant_roles_user"] = tenant_roles_user
                collection.data.update(str(key), json.dumps(vtenant_record))

                # index the audit record

                # trackme_audit_event(session_key, splunkd_uri, tenant_id, user, action, change_type, object_name, object_category, object_attrs, result, comment):

                try:
                    trackme_audit_event(
                        session_key=request_info.system_authtoken,
                        splunkd_uri=request_info.server_rest_uri,
                        tenant_id=tenant_id,
                        user=request_info.user,
                        action=str(action),
                        change_type="update RBAC virtual tenant",
                        object_name=str(tenant_id),
                        object_category="virtual_tenants",
                        object_attrs=json.dumps(vtenant_record),
                        result=json.dumps(results_record),
                        comment=str(update_comment),
                    )
                except Exception as e:
                    logger.error(
                        f'failed to generate an audit event with exception="{str(e)}"'
                    )

                return {"payload": audit_record, "status": 200}

    #
    # Update tenant score configuration
    #

    def post_update_tenant_score_config(self, request_info, **kwargs):
        """
        | trackme url=/services/trackme/v2/vtenants/admin/update_tenant_score_config mode=post body="{ 'tenant_id': 'mytenant', 'impact_score_outliers_default': 50, 'impact_score_dsm_delay_threshold_breach': 80 }"
        """

        # Declare main variables
        tenant_id = None
        describe = False

        # Retrieve from data
        try:
            resp_dict = json.loads(str(request_info.raw_args["payload"]))
        except Exception as e:
            resp_dict = None

        if resp_dict is not None:
            # option describe
            try:
                describe = resp_dict["describe"]
                if describe in ("true", "True"):
                    describe = True
            except Exception as e:
                describe = False

            # get tenant_id
            if not describe:
                try:
                    tenant_id = resp_dict["tenant_id"]
                except Exception as e:
                    return {
                        "payload": {
                            "response": "tenant_id is required",
                        },
                        "status": 400,
                    }
        else:
            # body is required in this endpoint, if not submitted describe the usage
            describe = True

        if describe:
            response = {
                "describe": "This endpoint can be called to update impact score configuration on a virtual tenant, it requires a POST call with the following information:",
                "resource_desc": "Update impact score configuration for a virtual tenant",
                "resource_spl_example": "| trackme url=/services/trackme/v2/vtenants/admin/update_tenant_score_config mode=post body=\"{ 'tenant_id': 'mytenant', 'impact_score_outliers_default': 50, 'impact_score_dsm_delay_threshold_breach': 80 }\"",
                "options": [
                    {
                        "tenant_id": "identifier of the virtual tenant to update",
                        "impact_score_outliers_default": "OPTIONAL: default impact score for outliers (0-100)",
                        "impact_score_dsm_data_sampling_anomaly": "OPTIONAL: impact score for DSM data sampling anomaly (0-100)",
                        "impact_score_dsm_delay_threshold_breach": "OPTIONAL: impact score for DSM delay threshold breach (0-100)",
                        "impact_score_dsm_latency_threshold_breach": "OPTIONAL: impact score for DSM latency threshold breach (0-100)",
                        "impact_score_dsm_min_hosts_dcount_breach": "OPTIONAL: impact score for DSM min hosts dcount breach (0-100)",
                        "impact_score_dsm_future_tolerance_breach": "OPTIONAL: impact score for DSM future tolerance breach (0-100)",
                        "impact_score_dhm_delay_threshold_breach": "OPTIONAL: impact score for DHM delay threshold breach (0-100)",
                        "impact_score_dhm_latency_threshold_breach": "OPTIONAL: impact score for DHM latency threshold breach (0-100)",
                        "impact_score_dhm_future_tolerance_breach": "OPTIONAL: impact score for DHM future tolerance breach (0-100)",
                        "impact_score_mhm_metric_alert": "OPTIONAL: impact score for MHM metric alert (0-100)",
                        "impact_score_mhm_future_tolerance_breach": "OPTIONAL: impact score for MHM future tolerance breach (0-100)",
                        "impact_score_flx_inactive": "OPTIONAL: impact score for FLX inactive (0-100)",
                        "impact_score_flx_status_not_met": "OPTIONAL: impact score for FLX status not met (0-100)",
                        "impact_score_fqm_status_not_met": "OPTIONAL: impact score for FQM status not met (0-100)",
                        "impact_score_wlk_skipping_searches": "OPTIONAL: impact score for WLK skipping searches (0-100)",
                        "impact_score_wlk_execution_errors": "OPTIONAL: impact score for WLK execution errors (0-100)",
                        "impact_score_wlk_orphan_search": "OPTIONAL: impact score for WLK orphan search (0-100)",
                        "impact_score_wlk_execution_delayed": "OPTIONAL: impact score for WLK execution delayed (0-100)",
                        "impact_score_wlk_out_of_monitoring_times": "OPTIONAL: impact score for WLK out of monitoring times (0-100)",
                        "impact_score_wlk_status_not_met": "OPTIONAL: impact score for WLK status not met (0-100)",
                        "update_comment": "OPTIONAL: a comment for the update, comments are added to the audit record, if unset will be defined to: API update",
                    }
                ],
            }

            return {"payload": response, "status": 200}

        # Update comment is optional and used for audit changes
        try:
            update_comment = resp_dict.get("update_comment")
            if update_comment == "Enter your audit message" or update_comment is None:
                update_comment = "No comment was provided for this operation"
        except Exception as e:
            update_comment = "No comment was provided for this operation"

        # Get vtenant account configuration via REST call
        url = f"{request_info.server_rest_uri}/servicesNS/nobody/trackme/trackme_vtenants/{tenant_id}"
        vtenant_data = {}
        vtenant_account_found = False

        try:
            # Get current vtenant account configuration
            response = requests.get(
                url,
                headers={"Authorization": f"Splunk {request_info.system_authtoken}"},
                verify=False,
                params={"output_mode": "json"},
                timeout=600,
            )
            if response.status_code in (200, 201, 204):
                logger.info(f"successfully retrieved vtenant configuration")
                vtenant_data_json = response.json()
                vtenant_data_current = vtenant_data_json["entry"][0]["content"]
                vtenant_account_found = True
                vtenant_data = dict(vtenant_data_current)
            else:
                error_msg = f"failed to retrieve vtenant configuration, status_code={response.status_code}"
                logger.error(error_msg)

        except Exception as e:
            error_msg = f"failed to retrieve vtenant configuration, exception={str(e)}"
            logger.error(error_msg)

        if not vtenant_account_found:
            error_msg = f'tenant_id="{tenant_id}" cannot be found'
            logger.error(error_msg)
            return {
                "payload": {
                    "response": error_msg,
                },
                "status": 404,
            }

        # Extract original score configuration for audit
        orig_score_config = {}
        for field_name in vtenant_account_default.keys():
            if field_name.startswith("impact_score_"):
                orig_score_config[field_name] = vtenant_data.get(
                    field_name, vtenant_account_default[field_name]
                )

        # Update score configuration fields from request
        updated_fields = {}
        dest_score_config = orig_score_config.copy()
        for field_name in vtenant_account_default.keys():
            if field_name.startswith("impact_score_"):
                if field_name in resp_dict:
                    new_value = resp_dict[field_name]
                    # Validate value is between 0 and 100
                    try:
                        int_value = int(new_value)
                        if int_value < 0 or int_value > 100:
                            return {
                                "payload": {
                                    "response": f'Invalid value for {field_name}: {new_value}. Value must be between 0 and 100.',
                                },
                                "status": 400,
                            }
                        vtenant_data[field_name] = str(int_value)
                        dest_score_config[field_name] = str(int_value)
                        updated_fields[field_name] = {
                            "old": orig_score_config.get(field_name),
                            "new": str(int_value),
                        }
                    except ValueError:
                        return {
                            "payload": {
                                "response": f'Invalid value for {field_name}: {new_value}. Value must be an integer.',
                            },
                            "status": 400,
                        }

        # If no fields were updated, return early
        if not updated_fields:
            return {
                "payload": {
                    "response": "No score configuration fields were provided for update.",
                },
                "status": 400,
            }

        # Remove system fields before POST
        vtenant_data.pop("disabled", None)
        vtenant_data.pop("eai:acl", None)
        vtenant_data.pop("eai:appName", None)
        vtenant_data.pop("eai:userName", None)

        # Update the tenant account configuration via REST call
        try:
            logger.info(
                f'attempting to update vtenant configuration, vtenant_data="{json.dumps(vtenant_data, indent=2)}"'
            )
            response = requests.post(
                url,
                headers={
                    "Authorization": f"Splunk {request_info.system_authtoken}",
                },
                data=vtenant_data,
                verify=False,
                timeout=600,
            )
            if response.status_code in (200, 201, 204):
                logger.info(
                    f'tenant_id="{tenant_id}", updated score configuration, updated_fields="{json.dumps(updated_fields, indent=2)}"'
                )
            else:
                error_msg = f'failed to update vtenant configuration, status_code={response.status_code}, response={response.text}'
                logger.error(error_msg)
                return {
                    "payload": {
                        "response": error_msg,
                    },
                    "status": 500,
                }
        except Exception as e:
            error_msg = f'failed to update vtenant configuration, exception="{str(e)}"'
            logger.error(error_msg)
            return {
                "payload": {
                    "response": error_msg,
                },
                "status": 500,
            }

        # Create audit record
        user = request_info.user
        action = "success"
        result = f"Impact score configuration updated: {json.dumps(updated_fields, indent=2)}"

        audit_record = {
            "_time": str(time.time()),
            "timeStr": str(datetime.datetime.now()),
            "user": str(user),
            "action": action,
            "change_type": "update impact score configuration virtual tenant",
            "object": str(tenant_id),
            "object_category": "virtual_tenants",
            "object_attrs": json.dumps(orig_score_config),
            "results_summary": result,
            "results_details": json.dumps(updated_fields),
            "comment": str(update_comment),
        }

        # Log audit event
        try:
            trackme_audit_event(
                session_key=request_info.system_authtoken,
                splunkd_uri=request_info.server_rest_uri,
                tenant_id=tenant_id,
                user=request_info.user,
                action=str(action),
                change_type="update impact score configuration virtual tenant",
                object_name=str(tenant_id),
                object_category="virtual_tenants",
                object_attrs=json.dumps(orig_score_config),
                result=json.dumps(updated_fields),
                comment=str(update_comment),
            )
        except Exception as e:
            logger.error(
                f'failed to generate an audit event with exception="{str(e)}"'
            )

        return {"payload": audit_record, "status": 200}
