#!/usr/bin/env python
# coding=utf-8

__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2022-2026, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

# Standard library imports
import os
import sys
import time
from collections import OrderedDict
import ast

# Logging imports
import logging
from logging.handlers import RotatingFileHandler

# Networking imports
import urllib3

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# set logging
filehandler = RotatingFileHandler(
    "%s/var/log/splunk/trackme_stsummary_splk_dhm.log" % splunkhome,
    mode="a",
    maxBytes=10000000,
    backupCount=1,
)
formatter = logging.Formatter(
    "%(asctime)s %(levelname)s %(filename)s %(funcName)s %(lineno)d %(message)s"
)
logging.Formatter.converter = time.gmtime
filehandler.setFormatter(formatter)
log = logging.getLogger()  # root logger - Good to get it only once.
for hdlr in log.handlers[:]:  # remove the existing file handlers
    if isinstance(hdlr, logging.FileHandler):
        log.removeHandler(hdlr)
log.addHandler(filehandler)  # set the new handler
# set the log level to INFO, DEBUG as the default is ERROR
log.setLevel(logging.INFO)

# append current directory
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

# import libs
import import_declare_test

# Import Splunk libs
from splunklib.searchcommands import (
    dispatch,
    StreamingCommand,
    Configuration,
    Option,
    validators,
)

# Import trackme libs
from trackme_libs import trackme_reqinfo


@Configuration(distributed=False)
class TrackMeMergeSplkDhm(StreamingCommand):
    def stream(self, records):
        # Get request info and set logging level
        reqinfo = trackme_reqinfo(
            self._metadata.searchinfo.session_key, self._metadata.searchinfo.splunkd_uri
        )
        log.setLevel(reqinfo["logging_level"])

        # loop through the records
        for splrecord in records:
            # attempt to get the current_dict
            try:
                current_dict = ast.literal_eval(splrecord.get("splk_dhm_st_summary"))
            except Exception as e:
                raise ValueError(
                    f'could not load the expected field splk_dhm_st_summary from upstream results, exception="{str(e)}"'
                )

            # try to load time
            try:
                record_time = splrecord.get("_time")
            except Exception as e:
                record_time = time.time()

            for p_id, p_info in current_dict.items():
                yield {
                    "_time": record_time,
                    "object": splrecord.get("object"),
                    "index": p_info["idx"],
                    "sourcetype": p_info["st"],
                    "first_time": p_info["first_time"],
                    "last_time": p_info["last_time"],
                    "last_ingest_lag": p_info["last_ingest_lag"],
                    "last_event_lag": p_info["last_event_lag"],
                    "time_measure": p_info["time_measure"],
                    "last_ingest": p_info["last_ingest"],
                    "count": p_info["last_eventcount"],
                }


dispatch(TrackMeMergeSplkDhm, sys.argv, sys.stdin, sys.stdout, __name__)
