#!/usr/bin/env python
# coding=utf-8

__author__ = "TrackMe Limited"
__copyright__ = "Copyright 2023-2025, TrackMe Limited, U.K."
__credits__ = "TrackMe Limited, U.K."
__license__ = "TrackMe Limited, all rights reserved"
__version__ = "0.1.0"
__maintainer__ = "TrackMe Limited, U.K."
__email__ = "support@trackme-solutions.com"
__status__ = "PRODUCTION"

import os
import sys
import json
from collections import OrderedDict
import time
import logging
from logging.handlers import RotatingFileHandler
from urllib.parse import urlencode
import urllib3

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# splunk home
splunkhome = os.environ["SPLUNK_HOME"]

# append lib
sys.path.append(os.path.join(splunkhome, "etc", "apps", "trackme", "lib"))

# import TrackMe libs
from trackme_libs import JSONFormatter

# Import trackMe utils libs
from trackme_libs_utils import decode_unicode

# logging:
# To avoid overriding logging destination of callers, the libs will not set on purpose any logging definition
# and rely on callers themselves


def trackme_scoring_gen_metrics(tenant_id, metrics_idx, records):
    # proceed
    try:
        # Validate inputs
        if records is None:
            logging.warning(
                f'context="scoring_gen_metrics", tenant_id="{tenant_id}", records parameter is None, skipping metrics generation'
            )
            return False
        
        if not isinstance(records, (list, tuple)):
            logging.warning(
                f'context="scoring_gen_metrics", tenant_id="{tenant_id}", records parameter is not iterable (type: {type(records)}), skipping metrics generation'
            )
            return False
        
        if len(records) == 0:
            logging.debug(
                f'context="scoring_gen_metrics", tenant_id="{tenant_id}", records is empty, skipping metrics generation'
            )
            return True
        
        if metrics_idx is None:
            logging.warning(
                f'context="scoring_gen_metrics", tenant_id="{tenant_id}", metrics_idx is None, skipping metrics generation'
            )
            return False

        # Create a dedicated logger for SLA metrics
        scoring_logger = logging.getLogger("trackme.scoring.metrics")
        scoring_logger.setLevel(logging.INFO)

        # Only add the handler if it doesn't exist yet
        if not scoring_logger.handlers:
            # Set up the file handler
            filehandler = RotatingFileHandler(
                f"{splunkhome}/var/log/splunk/trackme_scoring_metrics.log",
                mode="a",
                maxBytes=100000000,
                backupCount=1,
            )
            formatter = JSONFormatter()
            logging.Formatter.converter = time.gmtime
            filehandler.setFormatter(formatter)
            scoring_logger.addHandler(filehandler)
            # Prevent propagation to root logger
            scoring_logger.propagate = False
        else:
            # Find the RotatingFileHandler among existing handlers
            filehandler = None
            for handler in scoring_logger.handlers:
                if isinstance(handler, RotatingFileHandler):
                    filehandler = handler
                    break
            
            # If no RotatingFileHandler found, create one
            if filehandler is None:
                filehandler = RotatingFileHandler(
                    f"{splunkhome}/var/log/splunk/trackme_scoring_metrics.log",
                    mode="a",
                    maxBytes=100000000,
                    backupCount=1,
                )
                formatter = JSONFormatter()
                logging.Formatter.converter = time.gmtime
                filehandler.setFormatter(formatter)
                scoring_logger.addHandler(filehandler)

        for record in records:
            scoring_logger.info(
                "Metrics - group=scoring_metrics",
                extra={
                    "target_index": metrics_idx,
                    "tenant_id": tenant_id,
                    "object": decode_unicode(record.get("object")),
                    "object_id": record.get("object_id"),
                    "object_category": record.get("object_category"),
                    "score_source": record.get("score_source"),
                    "metrics_event": json.dumps(record.get("metrics_event")),
                },
            )

        return True

    except Exception as e:
        raise Exception(str(e))
